/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <AGMLApp.h>
#include <AGMLMainDialog.h>
#include <AGMLChangePasswordDialog.h>
#include <AGMLSynchronizeCE.h>
#include <AGSyncCommon.h>
#include <AGMLUtilWinCE.h>
#include <AGUtil.h>
#include <windowsx.h>
#include <prsht.h>
#ifndef HPC
#include <aygshell.h>
#endif
#include "AGMLResources.h"

#define kBigBufferSize (1024)

#define kLocationConfigFilename TEXT("\\windows\\MALLocationConfig.dat")
#define kMALConfigFilename TEXT("\\windows\\MALConfig.dat")

#define kMALNewServerName "<New MAL Server>"

#define kMALRegKey TEXT("Software\\Mobile Application Link")
#define kMALRegValueLastSelected TEXT("UID of Last Server")

typedef struct tagMLSynchronizePage {
    int dummy;
} MLSynchronizePage, * pMLSynchronizePage;

typedef struct tagMLNetworkPage {
    AGLocationConfig * lc;
} MLNetworkPage, * pMLNetworkPage;

typedef struct tagMLServerPage {
    int dummy;
} MLServerPage, * pMLServerPage;

typedef struct tagMLPropSheetPage {
    int id;
    HINSTANCE hInstance;
    void * data;
} MLPropSheetPage, * pMLPropSheetPage;

enum {
    PAGE_ID_SYNCHRONIZE = 0,
// <Add back in for proxy>    PAGE_ID_NETWORK,
    PAGE_ID_SERVER
};
#define PAGE_ID_NETWORK (-1)

HWND g_hwndMainDialog = NULL;
static g_synchronize = FALSE;
static g_sync_in_progress = FALSE;
static g_cancel_sync = FALSE;

static pMLPropSheetPage getLocalStorage(HWND hwnd)
{
    return (pMLPropSheetPage)GetWindowLong(hwnd, GWL_USERDATA);
}

static BOOL onInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
    PROPSHEETPAGE * psp;

    /* The first time onInitDialog gets called, it's for the
    main property sheet.  So record its window handle globally. */
    if (g_hwndMainDialog == NULL)
        g_hwndMainDialog = hwnd;

    /* When the PropertySheet function creates the page,
    the dialog box procedure for the page receives a WM_INITDIALOG message.
    The lParam parameter of this message points to the PROPSHEETPAGE
    structure used to create the page. */

    psp = (PROPSHEETPAGE *)lParam;
    SetWindowLong(hwnd, GWL_USERDATA, psp->lParam);

    return TRUE;
}

static void enableSynchronizePageItems(HWND hwnd, BOOL enable)
{
}

static void enableNetworkPageItems(HWND hwnd, BOOL enable)
{
    EnableWindow(GetDlgItem(hwnd, IDC_HTTP_ADDRESS), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_HTTP_PORT), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_SOCKS_ADDRESS), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_SOCKS_PORT), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_BYPASS_LOCAL), enable);
}

static void enableServerPageItems(HWND hwnd, BOOL enable)
{
    EnableWindow(GetDlgItem(hwnd, IDC_COMBO), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_ADDRESS), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_PORT), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_USERNAME), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_CHANGE), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_ENABLED), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_CONNECT_SECURELY), enable);
    EnableWindow(GetDlgItem(hwnd, IDC_REMOVE), enable);
}

static int32 getRegUID(void)
{
    HKEY hkey = NULL;
    int32 result = -1;
    DWORD cbSize;
    DWORD type;
    DWORD dwDisposition;

    RegCreateKeyEx(HKEY_CURRENT_USER,
        kMALRegKey,
        0,
        NULL,
        0,
        0,
        NULL,
        &hkey,
        &dwDisposition);

    if (NULL != hkey) {

        cbSize = sizeof(result);
        RegQueryValueEx(hkey,
            kMALRegValueLastSelected,
            0,
            &type,
            (LPBYTE)&result,
            &cbSize);

        RegCloseKey(hkey);

    }

    return result;
}

static void setRegUID(int32 uid)
{
    HKEY hkey = NULL;
    DWORD cbSize;
    DWORD type;

    RegOpenKeyEx(HKEY_CURRENT_USER, kMALRegKey, 0, KEY_WRITE, &hkey);
    if (NULL != hkey) {

        cbSize = sizeof(uid);
        type = REG_DWORD;
        RegSetValueEx(hkey,
            kMALRegValueLastSelected,
            0,
            type,
            (LPBYTE)&uid,
            cbSize);

        RegCloseKey(hkey);

    }
}

static int32 getUIDofCurrentSelection(HWND hwnd)
{
    int index;
    HWND hComboBox;

    hComboBox = GetDlgItem(hwnd, IDC_COMBO);
    if (NULL == hComboBox)
        return -1;

    index = ComboBox_GetCurSel(hComboBox);
    if (CB_ERR != index)
        return ComboBox_GetItemData(hComboBox, index);
    else
        return -1;
}

static int32 findComboBoxData(HWND hComboBox, int32 data)
{
    int n;

    n = ComboBox_GetCount(hComboBox);
    while (n--) {

        if (data == ComboBox_GetItemData(hComboBox, n))
            return n;

    }
    return CB_ERR;
}

static void setCurrentServer(HWND hwnd)
{
    setRegUID(getUIDofCurrentSelection(hwnd));
}

static void selectCurrentServer(HWND hwnd)
{
    HWND hComboBox;
    BOOL updatedWithSomething = FALSE;
    int index;
    int32 uid;
    AGUserConfig * userConfig = NULL;
    AGServerConfig * serverConfig = NULL;

    hComboBox = GetDlgItem(hwnd, IDC_COMBO);
    if (NULL == hComboBox)
        return;

    // Get the last-selected server.  If there is none, select the
    // first in the list.
    uid = getRegUID();
    if (-1 == uid)
        index = 0;
    else {
        index = findComboBoxData(hComboBox, uid);
        if (CB_ERR == index)
            index = 0;
    }

    ComboBox_SetCurSel(hComboBox, index);
    uid = ComboBox_GetItemData(hComboBox, index);

    if (CB_ERR != uid) {

        userConfig = AGReadUserConfigFromDisk(kMALConfigFilename);

        if (NULL != userConfig)
            serverConfig = AGUserConfigGetServerByUID(userConfig,
                uid,
                AGUSERCONFIG_FIND);

    }

    if (NULL != serverConfig) {

        if (AGServerConfigIsValid(serverConfig)) {

            TCHAR buf[MAX_PATH];
            mbstowcs(buf, serverConfig->serverName, -1);
            SetDlgItemText(hwnd, IDC_ADDRESS, buf);
            SetDlgItemInt(hwnd, IDC_PORT, serverConfig->serverPort, FALSE);
            mbstowcs(buf, serverConfig->userName, -1);
            SetDlgItemText(hwnd, IDC_USERNAME, buf);

            updatedWithSomething = TRUE;

        }

        SendMessage(GetDlgItem(hwnd, IDC_ENABLED),
            BM_SETCHECK,
            (WPARAM)(!serverConfig->disabled) ? BST_CHECKED : BST_UNCHECKED,
            0);

        SendMessage(GetDlgItem(hwnd, IDC_CONNECT_SECURELY),
            BM_SETCHECK,
            (WPARAM)serverConfig->connectSecurely
                ? BST_CHECKED
                : BST_UNCHECKED,
            0);

        EnableWindow(GetDlgItem(hwnd, IDC_CONNECT_SECURELY),
            serverConfig->allowSecureConnection);

    }
    else {
        // We couldn't find the currently selected server config.
        // Do nothing.
    }

    // If for some reason we couldn't find a valid current selection
    // (either last one was deleted or it's a new one that doesn't have
    // all the new info yet), put in blank stuff.
    if (!updatedWithSomething) {

        SetDlgItemText(hwnd, IDC_ADDRESS, TEXT(""));
        SetDlgItemInt(hwnd, IDC_PORT, 80, FALSE);
        SetDlgItemText(hwnd, IDC_USERNAME, TEXT(""));

        SendMessage(GetDlgItem(hwnd, IDC_ENABLED),
            BM_SETCHECK,
            (WPARAM)BST_CHECKED,
            0);

        SendMessage(GetDlgItem(hwnd, IDC_CONNECT_SECURELY),
            BM_SETCHECK,
            (WPARAM)BST_UNCHECKED,
            0);

        EnableWindow(GetDlgItem(hwnd, IDC_CONNECT_SECURELY), FALSE);

    }

    if (NULL != userConfig)
        AGUserConfigFree(userConfig);
}

static void buildServerList(HWND hwnd,
                            pMLPropSheetPage pInfo)
{
    AGUserConfig * userConfig;
    HWND hComboBox;
    int i, n;

    hComboBox = GetDlgItem(hwnd, IDC_COMBO);
    if (NULL == hComboBox)
        return;
    ComboBox_ResetContent(hComboBox);

    userConfig = AGReadUserConfigFromDisk(kMALConfigFilename);
  
    n = AGUserConfigCount(userConfig);
    for (i = 0; i < n; ++i) {

        WCHAR smallbuf[MAX_PATH];
        AGServerConfig * serverConfig;
        int index;

        serverConfig = AGUserConfigGetServerByIndex(userConfig, i);

        mbstowcs(smallbuf,
            (serverConfig->friendlyName) ? serverConfig->friendlyName
            : serverConfig->serverName, -1);

        index = ComboBox_AddString(hComboBox, smallbuf);

        if (index != CB_ERR)
            ComboBox_SetItemData(hComboBox, index, serverConfig->uid);

    }

    enableServerPageItems(hwnd, AGUserConfigCount(userConfig) > 0);

    if (NULL != userConfig)
        AGUserConfigFree(userConfig);

}

static void updateServerPageDisplay(HWND hwnd, pMLPropSheetPage pInfo)
{
    buildServerList(hwnd, pInfo);
    selectCurrentServer(hwnd);
}

static void handleAddButton(HWND hwnd, pMLPropSheetPage pInfo)
{
    AGServerConfig * serverConfig;
    int32 uid = -1;

    serverConfig = AGServerConfigNew();
    if (NULL != serverConfig) {

        AGUserConfig * userConfig;
        HANDLE f;

        serverConfig->friendlyName = strdup(kMALNewServerName);

        // pending(miket): this will go away when we support cleartext.
        serverConfig->hashPassword = TRUE;

        userConfig =
            AGReadUserConfigFromDiskAtomically(kMALConfigFilename, &f);

        if (INVALID_HANDLE_VALUE != f) {

            if (NULL == userConfig)
                userConfig = AGUserConfigNew();

            if (NULL != userConfig) {
                uid = AGUserConfigAddServerToDevice(userConfig,
                    serverConfig);
                enableServerPageItems(hwnd, TRUE);
            }

            AGWriteUserConfigToDiskAtomically(userConfig, f);

            if (NULL != userConfig)
                AGUserConfigFree(userConfig);
        }

    }

    if (-1 != uid) {
        setRegUID(uid);
        updateServerPageDisplay(hwnd, pInfo);
    }

}

static AGServerConfig * getSelectedServerConfig(HWND hwnd,
                                                AGUserConfig * userConfig)
{
    if (NULL != userConfig)
        return AGUserConfigGetServerByUID(userConfig,
            getUIDofCurrentSelection(hwnd),
            AGUSERCONFIG_FIND);
    return NULL;
}

static void handleResetButton(HWND hwnd, pMLPropSheetPage pInfo)
{
    AGUserConfig * userConfig;
    HANDLE f;

    userConfig = AGReadUserConfigFromDiskAtomically(kMALConfigFilename, &f);

    if (INVALID_HANDLE_VALUE == f)
        return;

    if (NULL != userConfig) {

        AGServerConfig * serverConfig;

        serverConfig = getSelectedServerConfig(hwnd, userConfig);

        if (NULL != serverConfig) {

            BOOL reset = TRUE;

            // If the serverconfig was garbage anyway, don't confirm
            // reset.
            if (AGServerConfigIsValid(serverConfig)) {

                TCHAR wcsname[MAX_PATH];
                TCHAR wcsbuf[MAX_PATH];
                TCHAR wcsproto[MAX_PATH];
                TCHAR wcscaption[MAX_PATH];

                mbstowcs(wcsname, serverConfig->friendlyName, -1);

                LoadString(pInfo->hInstance,
                    IDS_RESET_CONFIRM_MESSAGE,
                    wcsproto, MAX_PATH);

                LoadString(pInfo->hInstance,
                    IDS_RESET_CONFIRM_CAPTION,
                    wcscaption, MAX_PATH);

                wsprintf(wcsbuf, wcsproto, wcsname);

                if (IDOK != MessageBox(hwnd,
                    wcsbuf,
                    wcscaption,
                    MB_OKCANCEL | MB_ICONQUESTION)) {

                    reset = FALSE;

                }

            }

            if (reset) {
                AGServerConfigResetCookie(serverConfig);
                AGServerConfigResetNonce(serverConfig);
            }

            AGWriteUserConfigToDiskAtomically(userConfig, f);

        }
  
        AGUserConfigFree(userConfig);

    }

}

static void handleRemoveButton(HWND hwnd, pMLPropSheetPage pInfo)
{
    AGUserConfig * userConfig;
    HANDLE f;

    userConfig = AGReadUserConfigFromDiskAtomically(kMALConfigFilename, &f);

    if (INVALID_HANDLE_VALUE == f)
        return;

    if (NULL != userConfig) {

        AGServerConfig * serverConfig;

        serverConfig = getSelectedServerConfig(hwnd, userConfig);

        if (NULL != serverConfig) {

            BOOL remove = TRUE;

            // If the serverconfig was garbage anyway, don't confirm
            // removal.
            if (AGServerConfigIsValid(serverConfig)) {

                TCHAR wcsname[MAX_PATH];
                TCHAR wcsbuf[MAX_PATH];
                TCHAR wcsproto[MAX_PATH];
                TCHAR wcscaption[MAX_PATH];

                mbstowcs(wcsname, serverConfig->friendlyName, -1);

                LoadString(pInfo->hInstance,
                    IDS_CONFIRM_SERVER_REMOVAL,
                    wcsproto, MAX_PATH);

                LoadString(pInfo->hInstance,
                    IDS_CONFIRM_SERVER_REMOVAL_CAPTION,
                    wcscaption, MAX_PATH);

                wsprintf(wcsbuf, wcsproto, wcsname);

                if (IDOK != MessageBox(hwnd,
                    wcsbuf,
                    wcscaption,
                    MB_OKCANCEL | MB_ICONQUESTION)) {

                    remove = FALSE;

                }

            }

            if (remove)
                AGUserConfigRemoveServer(userConfig, serverConfig);

            AGWriteUserConfigToDiskAtomically(userConfig, f);

            if (remove)
                updateServerPageDisplay(hwnd, pInfo);

        }
  
        AGUserConfigFree(userConfig);

    }

}

static void handleSelectionChange(HWND hwnd)
{
    setCurrentServer(hwnd);
    selectCurrentServer(hwnd);
}

static void handleChangePasswordButton(HWND hwnd, pMLPropSheetPage pInfo)
{
    AGUserConfig * userConfig;
    HANDLE f;

    userConfig = AGReadUserConfigFromDiskAtomically(kMALConfigFilename, &f);

    if (INVALID_HANDLE_VALUE == f)
        return;

    if (NULL != userConfig) {

        AGServerConfig * serverConfig;

        serverConfig = getSelectedServerConfig(hwnd, userConfig);

        if (NULL != serverConfig) {

            AGMLChangePasswordDialogDo(hwnd,
                pInfo->hInstance,
                serverConfig);
        }

        AGWriteUserConfigToDiskAtomically(userConfig, f);
  
        AGUserConfigFree(userConfig);

    }

}

static char * safeStrDupAndReplace(char * dest, char * src)
{
    if (NULL != dest)
        free(dest);

    if (NULL != src)
        return strdup(src);

    return NULL;
}

static void sowSynchronizePage(HWND hwnd, pMLPropSheetPage pInfo)
{
    WCHAR bigbuf[kBigBufferSize];
    AGUserConfig * userConfig;
    int i, n;

    userConfig = AGReadUserConfigFromDisk(kMALConfigFilename);
  
    n = AGUserConfigCount(userConfig);
    if (n > 0) {

        LoadString(pInfo->hInstance, IDS_USED_BY, bigbuf, kBigBufferSize);

        for (i = 0; i < n; ++i) {

            WCHAR smallbuf[MAX_PATH];
            AGServerConfig * serverConfig;

            serverConfig = AGUserConfigGetServerByIndex(userConfig, i);

            mbstowcs(smallbuf,
                (serverConfig->friendlyName) ? serverConfig->friendlyName
                : serverConfig->serverName, -1);

            _tcscat(bigbuf, smallbuf);

            if (i+1 < n)
                _tcscat(bigbuf, TEXT("; "));

        }

        _tcscat(bigbuf, TEXT("."));

    }
    else
        LoadString(pInfo->hInstance,
            IDS_USED_BY_NOBODY,
            bigbuf,
            kBigBufferSize);

    SetDlgItemText(hwnd, IDC_STATUS, bigbuf);

    if (NULL != userConfig)
        AGUserConfigFree(userConfig);
}

static void reapSynchronizePage(HWND hwnd)
{
}

static void sowNetworkPage(HWND hwnd, AGLocationConfig * lc)
{
    if (NULL != lc) {

        TCHAR buf[MAX_PATH];
        BOOL useAnyProxy;

        useAnyProxy = (lc->HTTPUseProxy || lc->SOCKSUseProxy);

        SendMessage(GetDlgItem(hwnd, IDC_PROXY),
            BM_SETCHECK,
            (WPARAM)useAnyProxy ? BST_CHECKED : BST_UNCHECKED,
            0);

        enableNetworkPageItems(hwnd, useAnyProxy);

        if (NULL != lc->HTTPName) {

            mbstowcs(buf, lc->HTTPName, -1);
            SetWindowText(GetDlgItem(hwnd, IDC_HTTP_ADDRESS), buf);
            SetDlgItemInt(hwnd, IDC_HTTP_PORT, lc->HTTPPort, FALSE);

        }

        if (NULL != lc->SOCKSName) {

            mbstowcs(buf, lc->SOCKSName, -1);
            SetWindowText(GetDlgItem(hwnd, IDC_SOCKS_ADDRESS), buf);
            SetDlgItemInt(hwnd, IDC_SOCKS_PORT, lc->SOCKSPort, FALSE);

        }

    }

    SendMessage(GetDlgItem(hwnd, IDC_BYPASS_LOCAL),
        BM_SETCHECK,
        (WPARAM)lc->bypassLocal ? BST_CHECKED : BST_UNCHECKED,
        0);

}

static void reapNetworkPage(HWND hwnd,
                            AGLocationConfig * lc)
{
    WCHAR wcsbuf[MAX_PATH];
    char mbsbuf[MAX_PATH];

    GetDlgItemText(hwnd, IDC_HTTP_ADDRESS, wcsbuf, MAX_PATH);
    wcstombs(mbsbuf, wcsbuf, -1);
    lc->HTTPName = safeStrDupAndReplace(lc->HTTPName, mbsbuf);
    lc->HTTPPort = GetDlgItemInt(hwnd, IDC_HTTP_PORT, NULL, FALSE);
    if (_tcslen(wcsbuf) > 0)
        lc->HTTPUseProxy = TRUE;
    else
        lc->HTTPUseProxy = FALSE;

    GetDlgItemText(hwnd, IDC_SOCKS_ADDRESS, wcsbuf, MAX_PATH);
    wcstombs(mbsbuf, wcsbuf, -1);
    lc->SOCKSName = safeStrDupAndReplace(lc->SOCKSName, mbsbuf);
    lc->SOCKSPort = GetDlgItemInt(hwnd, IDC_SOCKS_PORT, NULL, FALSE);
    if (_tcslen(wcsbuf) > 0)
        lc->SOCKSUseProxy = TRUE;
    else
        lc->SOCKSUseProxy = FALSE;

    if (SendMessage(GetDlgItem(hwnd, IDC_PROXY), BM_GETCHECK, 0, 0)
            != BST_CHECKED) {

        lc->HTTPUseProxy = FALSE;
        lc->SOCKSUseProxy = FALSE;

    }

    lc->bypassLocal = (SendMessage(GetDlgItem(hwnd, IDC_BYPASS_LOCAL),
        BM_GETCHECK, 0, 0) == BST_CHECKED);

}

static void sowServerPage(HWND hwnd,
                          pMLPropSheetPage pInfo)
{
    updateServerPageDisplay(hwnd, pInfo);
}

/*  Look through given userConfig list and test each serverConfig for
    validity.  If any are not valid, return TRUE.  If pareNew is true,
    then remove invalid new servers and exclude them from the final
    true/false result.
*/
static BOOL handleInvalidServerConfigs(AGUserConfig * userConfig,
                                       BOOL pareNew)
{
    int n;
    BOOL result = FALSE;

    // We have to go backwards through the list because we might delete
    // entries in it while iterating.
    n = AGUserConfigCount(userConfig);
    while (n--) {

        AGServerConfig * serverConfig;

        serverConfig = AGUserConfigGetServerByIndex(userConfig, n);

        if (NULL == serverConfig->friendlyName)
            continue;

        if (!AGServerConfigIsValid(serverConfig)) {

            if (strcmp(kMALNewServerName, serverConfig->friendlyName)) {
                result = TRUE;
                continue;
            }
            else {

                if (pareNew)
                    AGUserConfigRemoveServer(userConfig, serverConfig);
                else
                    result = TRUE;

            }
        }
    }

    return result;
}

static void removeNewInvalidServerConfigs(void)
{
    AGUserConfig * userConfig = NULL;
    HANDLE f;

    userConfig = AGReadUserConfigFromDiskAtomically(kMALConfigFilename, &f);
    if (INVALID_HANDLE_VALUE != f) {

        handleInvalidServerConfigs(userConfig, TRUE);

        AGWriteUserConfigToDiskAtomically(userConfig, f);

        if (NULL != userConfig)
            AGUserConfigFree(userConfig);

    }
}

/* Returns TRUE if we should DISALLOW moving away from this page. */
static BOOL reapServerPage(HWND hwnd)
{
    AGUserConfig * userConfig = NULL;
    HANDLE f;

    userConfig = AGReadUserConfigFromDiskAtomically(kMALConfigFilename, &f);
    if (INVALID_HANDLE_VALUE != f) {

        AGServerConfig * serverConfig;

        if (NULL != userConfig) {

            serverConfig = getSelectedServerConfig(hwnd, userConfig);

            if (NULL != serverConfig) {

                WCHAR wcsbuf[MAX_PATH];
                char mbsbuf[MAX_PATH];

                GetDlgItemText(hwnd, IDC_ADDRESS, wcsbuf, MAX_PATH);
                wcstombs(mbsbuf, wcsbuf, -1);
                serverConfig->serverName =
                    safeStrDupAndReplace(serverConfig->serverName, mbsbuf);

                serverConfig->serverPort =
                    GetDlgItemInt(hwnd, IDC_PORT, NULL, FALSE);

                GetDlgItemText(hwnd, IDC_USERNAME, wcsbuf, MAX_PATH);
                wcstombs(mbsbuf, wcsbuf, -1);
                serverConfig->userName =
                    safeStrDupAndReplace(serverConfig->userName, mbsbuf);

                serverConfig->disabled =
                    (SendMessage(GetDlgItem(hwnd, IDC_ENABLED),
                    BM_GETCHECK, 0, 0) != BST_CHECKED);

                serverConfig->connectSecurely =
                    (SendMessage(GetDlgItem(hwnd, IDC_CONNECT_SECURELY),
                    BM_GETCHECK, 0, 0) == BST_CHECKED);

            }

        }
        else
            userConfig = AGUserConfigNew();

        AGWriteUserConfigToDiskAtomically(userConfig, f);

        if (NULL != userConfig)
            AGUserConfigFree(userConfig);

    }

    removeNewInvalidServerConfigs();

    return FALSE;
}

static void activateSynchronizePage(HWND hwnd, pMLPropSheetPage pInfo)
{
    if (NULL != pInfo->data)
        MessageBox(NULL,
            TEXT("Error! (NULL != pInfo->data) at activateSynchronizePage"),
            TEXT(""),
            MB_OK);

    pInfo->data = malloc(sizeof(MLSynchronizePage));
    if (NULL != pInfo->data) {

        pMLSynchronizePage pPageInfo;

        pPageInfo = (pMLSynchronizePage)pInfo->data;

        sowSynchronizePage(hwnd, pInfo);

        if (g_synchronize)
            PostMessage(hwnd, AGML_SYNCHRONIZE, 0, 0);

    }

}

static void activateNetworkPage(HWND hwnd,
                                pMLPropSheetPage pInfo)
{
    if (NULL != pInfo->data)
        MessageBox(NULL,
            TEXT("Error! (NULL != pInfo->data) at activateNetworkPage"),
            TEXT(""),
            MB_OK);

    pInfo->data = malloc(sizeof(MLNetworkPage));
    if (NULL != pInfo->data) {

        pMLNetworkPage pPageInfo;

        pPageInfo = (pMLNetworkPage)pInfo->data;

        pPageInfo->lc = AGReadLocationConfigFromDisk(kLocationConfigFilename);

        if (NULL == pPageInfo->lc)
            pPageInfo->lc = AGLocationConfigNew();

        sowNetworkPage(hwnd, pPageInfo->lc);
    }

}

static void activateServerPage(HWND hwnd,
                               pMLPropSheetPage pInfo)
{
    if (NULL != pInfo->data)
        MessageBox(NULL,
            TEXT("Error! (NULL != pInfo->data) at activateServerPage"),
            TEXT(""),
            MB_OK);

    pInfo->data = malloc(sizeof(MLServerPage));
    if (NULL != pInfo->data) {

        pMLServerPage pPageInfo;

        pPageInfo = (pMLServerPage)pInfo->data;

        sowServerPage(hwnd, pInfo);
    }
}

static void freeSynchronizePageData(void * data)
{
    if (NULL != data)
        free(data);
}

static void freeNetworkPageData(void * data)
{
    if (NULL != data) {

        pMLNetworkPage pPageInfo;

        pPageInfo = (pMLNetworkPage)data;

        if (NULL != pPageInfo->lc)
            AGLocationConfigFree(pPageInfo->lc);

        free(data);

    }
}

static void freeServerPageData(void * data)
{
    if (NULL != data)
        free(data);
}

static void killSynchronizePage(HWND hwnd,
                                pMLPropSheetPage pInfo)
{
    reapSynchronizePage(hwnd);
    freeSynchronizePageData(pInfo->data);
    pInfo->data = NULL;
}

static void killNetworkPage(HWND hwnd,
                            pMLPropSheetPage pInfo)
{

    if (NULL != pInfo->data) {

        pMLNetworkPage pPageInfo;

        pPageInfo = (pMLNetworkPage)pInfo->data;

        if (NULL != pPageInfo->lc) {

            reapNetworkPage(hwnd, pPageInfo->lc);

            AGWriteLocationConfigToDisk(kLocationConfigFilename,
                pPageInfo->lc);

        }

    }

    freeNetworkPageData(pInfo->data);
    pInfo->data = NULL;

}

static void killServerPage(HWND hwnd,
                           pMLPropSheetPage pInfo)
{
    reapServerPage(hwnd);
    freeServerPageData(pInfo->data);
    pInfo->data = NULL;
}

static void resetSynchronizePage(HWND hwnd,
                                 pMLPropSheetPage pInfo)
{
    freeSynchronizePageData(pInfo->data);
    pInfo->data = NULL;
}

static void resetNetworkPage(HWND hwnd,
                             pMLPropSheetPage pInfo)
{
    freeNetworkPageData(pInfo->data);
    pInfo->data = NULL;
}

static void resetServerPage(HWND hwnd,
                            pMLPropSheetPage pInfo)
{
    removeNewInvalidServerConfigs();
    freeServerPageData(pInfo->data);
    pInfo->data = NULL;
}

static void handleSynchronizeButton(HWND hwnd, pMLPropSheetPage pInfo)
{
    AGUserConfig * userConfig = NULL;
    AGLocationConfig * locationConfig = NULL;

// <Add back in for proxy>    locationConfig = AGReadLocationConfigFromDisk(kLocationConfigFilename);

    userConfig = AGReadUserConfigFromDisk(kMALConfigFilename);

    if (NULL != userConfig) {

        TCHAR buttonText[64];
        g_sync_in_progress = TRUE;
        GetDlgItemText(hwnd, IDC_SYNCHRONIZE, buttonText, 64);
        SetDlgItemText(hwnd, IDC_SYNCHRONIZE, TEXT("Cancel"));
        g_cancel_sync = FALSE;

        AGMLSynchronize(pInfo->hInstance,
            hwnd,
            GetDlgItem(hwnd, IDC_STATUS),
            GetDlgItem(hwnd, IDC_PROGRESS),
            GetDlgItem(hwnd, IDC_SERVER_NAME),
            GetDlgItem(hwnd, IDC_X_OF_Y),
            userConfig,
            locationConfig,
            &g_cancel_sync);

        SetDlgItemText(hwnd, IDC_SYNCHRONIZE, buttonText);
        g_sync_in_progress = FALSE;

        AGWriteUserConfigToDisk(kMALConfigFilename, userConfig);

        sowSynchronizePage(hwnd, pInfo);

        AGUserConfigFree(userConfig);

    }

    if (NULL != locationConfig)
        AGLocationConfigFree(locationConfig);

}

static void onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    pMLPropSheetPage pInfo;
    
    pInfo = getLocalStorage(hwnd);

    switch (codeNotify) {
        case LBN_SELCHANGE:
            handleSelectionChange(hwnd);
            break;
        case CBN_DROPDOWN:
            AGMLHandleDropDown(GetDlgItem(hwnd, IDC_COMBO));
            break;
        case EN_SETFOCUS:

            // Usually we want to pop up the keyboard if the user moves
            // to an edit control.  Exception is the status field on the
            // sync property sheet.
            if (IDC_STATUS != id)
                ShowSIP(hwnd);

            break;
        case EN_KILLFOCUS:
            HideSIP(hwnd);
            break;
        case BN_CLICKED:
            switch (id) {
                case IDOK:
                    HideSIP(hwnd);
                    break;
                case IDC_SYNCHRONIZE:

                    // This button metamorphoses from cancel to sync.
                    if (g_sync_in_progress)
                        g_cancel_sync = TRUE;
                    else
                        handleSynchronizeButton(hwnd, pInfo);
                    break;
                case IDC_CHANGE:
                    handleChangePasswordButton(hwnd, pInfo);
                    break;
                case IDC_PROXY:
                    enableNetworkPageItems(hwnd,
                        SendMessage(GetDlgItem(hwnd, IDC_PROXY),
                            BM_GETCHECK,
                            0,
                            0));
                    break;
                case IDC_ADD:
                    handleAddButton(hwnd, pInfo);
                    break;
                case IDC_REMOVE:
                    handleRemoveButton(hwnd, pInfo);
                    break;
                case IDC_RESET:
                    handleResetButton(hwnd, pInfo);
                    break;

            }
            break;
        default:
            break;
    }
}

static LRESULT onNotify(HWND hwnd, NMHDR * nmh)
{
    pMLPropSheetPage pInfo;
    
    pInfo = getLocalStorage(hwnd);

    switch (nmh->code) {

        // This page is about to be activated.
        case PSN_SETACTIVE: {

            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);

            switch (pInfo->id) {

                case PAGE_ID_SYNCHRONIZE:
                    activateSynchronizePage(hwnd, pInfo);
                    break;
                case PAGE_ID_NETWORK:
                    activateNetworkPage(hwnd, pInfo);
                    break;
                case PAGE_ID_SERVER:
                    activateServerPage(hwnd, pInfo);
                    break;
                default:
                    MessageBox(NULL, TEXT("Somebody had wrong id."),
                        TEXT(""),
                        MB_OK);
            }
            break;

        }

        // User pressed ok or moved to another page.  Accept current
        // settings.
        case PSN_KILLACTIVE:

            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);

            switch (pInfo->id) {

                case PAGE_ID_SYNCHRONIZE:
                    killSynchronizePage(hwnd, pInfo);
                    break;
                case PAGE_ID_NETWORK:
                    killNetworkPage(hwnd, pInfo);
                    break;
                case PAGE_ID_SERVER:
                    killServerPage(hwnd, pInfo);
                    break;
            }
            break;

        case PSN_RESET:

            SetWindowLong(hwnd, DWL_MSGRESULT, FALSE);

            switch (pInfo->id) {

                case PAGE_ID_SYNCHRONIZE:
                    resetSynchronizePage(hwnd, pInfo);
                    break;
                case PAGE_ID_NETWORK:
                    resetNetworkPage(hwnd, pInfo);
                    break;
                case PAGE_ID_SERVER:
                    resetServerPage(hwnd, pInfo);
                    break;
            }
            break;

        default:
            return FALSE;

    }

    return TRUE;
}

static BOOL CALLBACK pspProc(HWND hwnd, UINT message,
                             WPARAM wParam, LPARAM lParam)
{
    pMLPropSheetPage pInfo;
    
    pInfo = getLocalStorage(hwnd);

    switch (message) {
        HANDLE_MSG(hwnd, WM_INITDIALOG, onInitDialog);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        case WM_NOTIFY:
            return onNotify(hwnd, (NMHDR*)lParam);
        case AGML_SYNCHRONIZE:
            PropSheet_SetCurSelByID(GetParent(hwnd), IDD_SYNCHRONIZE);
            handleSynchronizeButton(hwnd, pInfo);
            return 0;
        case AGML_SERVER_OPTIONS:
            PropSheet_SetCurSelByID(GetParent(hwnd), IDD_SERVER);
            return 0;
    }
    return 0;
}

// <Add back in for proxy> #define kNumPropSheets (3)
#define kNumPropSheets (2)
void AGMLMainDialogDo(HWND hwndParent, HINSTANCE hInstance)
{
    WCHAR titleBuf[64];
    PROPSHEETPAGE psp[kNumPropSheets];
    MLPropSheetPage pspInfo[kNumPropSheets];
    PROPSHEETHEADER psh;
    int i;

    memset(psp, 0, sizeof(psp));
    memset(pspInfo, 0, sizeof(pspInfo));
    memset(&psh, 0, sizeof(psh));

    for (i = 0; i < kNumPropSheets; ++i) {

        pspInfo[i].hInstance = hInstance;
        pspInfo[i].id = PAGE_ID_SYNCHRONIZE + i;

        psp[i].dwSize = sizeof(PROPSHEETPAGE);
        psp[i].dwFlags = PSP_DEFAULT;
        psp[i].hInstance = hInstance;
        psp[i].lParam = (LPARAM)&pspInfo[i];

    }

    i = 0;
    psp[i].pszTemplate = MAKEINTRESOURCE(IDD_SYNCHRONIZE);
    psp[i].pfnDlgProc = pspProc;
    ++i;
// <Add back in for proxy>    psp[i].pszTemplate = MAKEINTRESOURCE(IDD_NETWORK);
// <Add back in for proxy>    psp[i].pfnDlgProc = pspProc;
// <Add back in for proxy>    ++i;
    psp[i].pszTemplate = MAKEINTRESOURCE(IDD_SERVER);
    psp[i].pfnDlgProc = pspProc;
    ++i;

    psh.dwSize = sizeof(PROPSHEETHEADER);
    psh.dwFlags = PSH_PROPSHEETPAGE | PSH_NOAPPLYNOW;
    psh.hwndParent = hwndParent;
    LoadString(hInstance, IDS_DIALOG_TITLE, titleBuf, 64);
    psh.pszCaption = titleBuf;
    psh.nPages = kNumPropSheets;
    psh.nStartPage = PAGE_ID_SYNCHRONIZE;
    psh.ppsp = (LPCPROPSHEETPAGE)psp;

    PropertySheet(&psh);

    PostQuitMessage(0);

}
