/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

#include <Pilot.h> 
#include <palmmal.h>
#include <SystemMgr.h> // for SysGetROMToken()

/* sysROMTokenSnum only defined in 3.0 headers */
#ifndef sysROMTokenSnum
#define sysROMTokenSnum 'snum'
#define sysTrapHwrGetROMToken 0xA340
Err SysGetROMToken(Word cardNo, DWord token, BytePtr *dataP, WordPtr sizeP)
     SYS_TRAP(sysTrapHwrGetROMToken);
#endif

#define DEFAULT_CARD_NUM ((UInt)0)

// Currently there's some magic in PalmMalResetSyncFlags() that uses this,
// can't remember why or where we got this from.
#define agRecAttrArchived 0x08

static DWord screenGetDepths();
static DWord getPalmOSVersion();
static DWord getPalmOSVersionMajor();
static void *getSerialNumber();


int32 PalmMalOpenDB(char *name, int32 cardNum, int32 mode, 
                     uint32 *handle) {
    LocalID dbId = 0;
    int32 flag = dmModeReadWrite; //PENDING(klobad) ignore the mode

    dbId = DmFindDatabase((UInt)cardNum, name);
    if(dbId == 0) {
        return PALMMAL_DB_FIND_ERR;
    }
    *handle = (uint32)DmOpenDatabase((UInt)cardNum, dbId,
                                         flag);
    if (*handle == 0) {
        return PALMMAL_DB_OPEN_ERR;
    }
    return PALMMAL_OK;
}

int32 PalmMalCreateDB(int32 creator, int32 flags, int32 cardNum, 
                       char *name, int32 type, uint32 *handle) {

    Err err;

    err = DmCreateDatabase((UInt)cardNum, name, creator, 
                           type, ((flags & dmHdrAttrResDB) == 1));

    if ((err == dmErrAlreadyExists) || (err == 0))
        return PalmMalOpenDB(name, cardNum, dmModeReadWrite, handle);
    else
        return PALMMAL_ERR;
}

int32 PalmMalCloseDB(uint32 handle) {
    if(0 == DmCloseDatabase((DmOpenRef) handle))
        return PALMMAL_OK;
    return PALMMAL_ERR;
}

int32 PalmMalDeleteDB(char *name, int32 cardNum) {
    LocalID dbId = 0;
    Err err;

    dbId = DmFindDatabase((UInt)cardNum, name);
    if(dbId == 0) {
        return PALMMAL_OK;
    }
    
    err = DmDeleteDatabase((UInt)cardNum, dbId);
    if(0 == err)  {
        return PALMMAL_OK;
    }

    return PALMMAL_ERR;
}

int32 PalmMalGetDBRecordCount(uint32 handle, int32 *recCount) {
    *recCount = (int32)DmNumRecords((DmOpenRef)handle);
    return PALMMAL_OK;
}

int32 PalmMalPurgeDeletedRecs(uint32 handle) {
    UInt recordCount;
    UInt attr;

    recordCount = DmNumRecords((DmOpenRef)handle);
    while (recordCount-- > 0) {
        attr = 0;
        DmRecordInfo((DmOpenRef)handle, recordCount, &attr, NULL, NULL);
        if(attr & dmRecAttrDelete) 
            DmRemoveRecord((DmOpenRef)handle, recordCount);
    }
    return PALMMAL_OK;
}

int32 PalmMalResetSyncFlags(uint32 handle) {
    UInt recordCount;
    UInt attr;
    AGBool dirty = false;

    recordCount = DmNumRecords((DmOpenRef)handle);
    while (recordCount-- > 0) {
        attr = 0;
        dirty = false;
        DmRecordInfo((DmOpenRef)handle, recordCount, &attr, NULL, NULL);
        if(attr & dmRecAttrDirty) {
            attr = attr ^ dmRecAttrDirty;
            dirty = true;
        }
        //PENDING(klobad) I can't remember why we're doing this magic,
        // need to check into it further.
        if(attr & agRecAttrArchived) {
            attr = attr ^ agRecAttrArchived;
            dirty = true;
        }
        if(dirty)
            DmSetRecordInfo((DmOpenRef)handle, recordCount, &attr, NULL);
    }
    return PALMMAL_OK;
}

int32 PalmMalReadRecordByIndex(PalmMalCRawRecordInfo *rec) 
{
    VoidHand recHand;   
    UInt attr;

    recHand = DmQueryRecord((DmOpenRef)rec->fileHandle, rec->recIndex);
    if(recHand == 0)
        return PALMMAL_RECORD_NOT_FOUND;
    rec->recSize = (int32)MemHandleSize(recHand);
    rec->pBytes = MemHandleLock(recHand);
    DmRecordInfo((DmOpenRef)rec->fileHandle, rec->recIndex, &attr, 
                    &rec->recId, NULL);
    rec->attribs = attr;
    return PALMMAL_OK;
}

int32 PalmMalCloseRecord(PalmMalCRawRecordInfo *rec)
{
    if(rec->pBytes != NULL) {
       MemPtrUnlock(rec->pBytes);
       rec->pBytes = NULL;
    }
    return PALMMAL_OK;
}

int32 PalmMalReadRecordById(PalmMalCRawRecordInfo *rec) {
    UInt index;
    Err err;

    err = DmFindRecordByID((DmOpenRef)rec->fileHandle, (ULong)rec->recId, &index);
    if(0 != err)
        return PALMMAL_RECORD_NOT_FOUND;
    // PENDING(klobad) is this legal to move the recIndex var without asking?
    rec->recIndex = index;
    return PalmMalReadRecordByIndex(rec);
}

int32 PalmMalReadNextModifiedRec(PalmMalCRawRecordInfo *rec) {
    UInt recordCount;
    UInt attr;
    UInt curIndex;
    UInt index;
    ULong id;

    //PENDING(klobad) assert rec->recId is the last record we
    //sent to them. Start searching from there forward for the 
    //next modified record.
    //PENDING(klobad) how do I tell them that there is no nextRecord
    //PENDING(klobad) since the Id they are sending in is the Id I give them,
    //I get to assert that the Ids will be in order and that getting it's index
    //and incrementing will work and not accidentily skip records.
    recordCount = DmNumRecords((DmOpenRef)rec->fileHandle);
    if (rec->recId != -1 && rec->recId != 0) {
        DmFindRecordByID((DmOpenRef)rec->fileHandle, 
                            (ULong)rec->recId, &index);
        curIndex = index + 1;
    } else {
        curIndex = 0;
    }
    while (curIndex < recordCount) {
        attr = 0;
        DmRecordInfo((DmOpenRef)rec->fileHandle, curIndex, &attr, &id, NULL);
        if(attr & dmRecAttrDelete) {
            rec->recIndex = curIndex;
            rec->recId = id;
            rec->attribs = attr;
            return PALMMAL_OK;
        }
        if(attr & dmRecAttrDirty) {
            rec->recIndex = curIndex;
            rec->recId = id;
            rec->attribs = attr;
            return PalmMalReadRecordByIndex(rec);
        }
        curIndex++;
    }

    // if we fall through, then the rec->recIndex is == recordCount
    rec->recId = 0;
    rec->recIndex = 0;
    return PALMMAL_RECORD_NOT_FOUND;
}

int32 PalmMalWriteResource(PalmMalCRawRecordInfo *rec) {
    VoidHand handle = 0;
    void *record;

    handle = DmNewResource((DmOpenRef)rec->fileHandle, rec->recId, 
                            rec->recIndex, rec->recSize);
    if(handle == 0) {
        return PALMMAL_ERR;
    }

    record = MemHandleLock(handle);
    if(record == NULL)
        return PALMMAL_ERR;
    DmWrite(record, 0, rec->pBytes, rec->recSize);
    MemPtrUnlock(record);

    DmReleaseResource(handle);
    return PALMMAL_OK;
}

int32 PalmMalWriteRecord(PalmMalCRawRecordInfo *rec) {
    VoidHand handle = 0;
    void *record;
    UInt index = -1;
    UInt attr;

    // We don't write zero length records
    if(rec->recSize <= 0)
        return PALMMAL_ERR;

    // The recId is < 1 then this is definately a new record - create it
    if(rec->recId == 0 || rec->recId == -1) {
        handle = DmNewRecord((DmOpenRef)rec->fileHandle, 
                                &index, rec->recSize);
        if(handle == 0) {
            return PALMMAL_ERR;
        }
    } 

    // There is a recId > 0, so this must be a rewrite of an old record
    if(handle == 0) {
        if(0 == DmFindRecordByID((DmOpenRef)rec->fileHandle, 
                    (ULong)rec->recId, &index)) {
            // If we find the record, and it's not deleted, then resize it to
            // the new size per the request.
            DmRecordInfo((DmOpenRef)rec->fileHandle, index, &attr, NULL, NULL);
            if(!(attr & dmRecAttrDelete)) {
                handle = DmResizeRecord((DmOpenRef)rec->fileHandle, 
                                        index, rec->recSize);
                if(handle == 0)
                    return PALMMAL_ERR;             
            } else {
                // This record exists and is deleted. We need to really delete
                // it so that we can recreate it.
                DmRemoveRecord((DmOpenRef)rec->fileHandle, index);
                // This will force the following code to create the record anew
                handle = 0; 
            }
        }
    }

    // The record has a recId, but we could not find it in our database,
    // perhaps the server set the id, or perhaps this was a deleted record.
    // in either case, create the record anew, and set its recId to the 
    // requested recId
    if(handle == 0) {
        handle = DmNewRecord((DmOpenRef)rec->fileHandle, 
                                &index, rec->recSize);
        if(handle == 0) {
            // This is the final hurrah, if this doesn't work we bail completely
            return PALMMAL_ERR;
        }
        if(0 != DmSetRecordInfo ((DmOpenRef)rec->fileHandle, index,  NULL,
                                    &rec->recId)) {
            return PALMMAL_ERR; 
        }
    }

    // Just in case 
    if(handle == 0) {
        return PALMMAL_ERR;
    }

    record = MemHandleLock(handle);
    if(record == NULL)
        return PALMMAL_ERR;
    DmWrite(record, 0, rec->pBytes, rec->recSize);
    MemPtrUnlock(record);
    //PENDING(klobad) need this??
    DmReleaseRecord((DmOpenRef)rec->fileHandle, index, true);

    //PENDING(klobad) the false parm on DmReleaseRecord() doesn't seem to do it
    DmRecordInfo((DmOpenRef)rec->fileHandle, index, &attr, &rec->recId, NULL);
    if(attr & dmRecAttrDirty) {
        attr = attr ^ dmRecAttrDirty;
        DmSetRecordInfo((DmOpenRef)rec->fileHandle, index, &attr, NULL);
    }

    return PALMMAL_OK;
}

int32 PalmMalDeleteRecord(PalmMalCRawRecordInfo *rec) {
    UInt index = -1;
    uint16 attr = 0;

    if(0 == DmFindRecordByID((DmOpenRef)rec->fileHandle, 
                                (ULong)rec->recId, &index)) {
        DmRecordInfo((DmOpenRef)rec->fileHandle, index, &attr, NULL, NULL);
        if(!(attr & dmRecAttrDelete)) {
            DmDeleteRecord((DmOpenRef)rec->fileHandle, index);    
        }
    }
    return PALMMAL_OK;
}

int32 PalmMalAvailableBytes(int32 cardNum, int32 *sizeAvailable) {
    uint32 tmp = 0;
    
    MemCardInfo((UInt)cardNum, NULL, NULL, NULL, NULL, NULL, NULL, &tmp);
    // we're seeing this guy report too much space,
    // so we're going to back it off by the delta we're seeing for now.
    //PENDING(klobad) figure out why this is reporting too much space
    if(tmp > (25L * 1024L))
        tmp -= (25L * 1024L);
    else
        tmp = 0L;
        
    *sizeAvailable = tmp;
    return PALMMAL_OK;
}

int32 PalmMalGetHardwareConfig(char **osName, char **osVersion, 
                                    int32 *colorDepth, int32 *screenWidth, 
                                    int32 *screenHeight, char **serialNum)
{

    DWord romVersion = 0;
    char verStr[10];
    
    *osName = (char *)MemPtrNew(StrLen(PALM_OS_NAME)+1);
    MemMove(*osName, PALM_OS_NAME, StrLen(PALM_OS_NAME)+1);

    romVersion = getPalmOSVersion();
    
    MemSet(verStr, 10, 0);
    StrIToA(verStr, sysGetROMVerMajor(romVersion));
    StrCat(verStr, ".");
    StrIToA(verStr + StrLen(verStr), sysGetROMVerMinor(romVersion));    
    *osVersion = (char *)MemPtrNew(StrLen(verStr)+1);
    MemMove(*osVersion, verStr, StrLen(verStr)+1);
    
    if((screenGetDepths() & 0x2) != 0)
        *colorDepth = 2;
    else
        *colorDepth = 1;
    
    *screenWidth = 150;
    *screenHeight = 150;
    *serialNum = getSerialNumber();
   
    return PALMMAL_OK;
}

void PalmMalInit(PalmMalCRawRecordInfo *rRec, uint32 fileHandle,
                                uint32 recId, uint16 recIndex, uint8 attribs,
                                uint32 recSize, uint8 *pBytes)
{
    if (NULL == rRec)
        return;
    
    MemSet(rRec, sizeof(PalmMalCRawRecordInfo), 0);
    rRec->fileHandle   = fileHandle;
    rRec->recId        = recId;
    rRec->recIndex     = recIndex;
    rRec->attribs      = attribs;
    rRec->recSize      = recSize;
    rRec->pBytes       = pBytes;
}

/*
 * From PalmOS 3.0
 */
typedef enum {
    scrDisplayModeGetDefaults,
    scrDisplayModeGet,
    scrDisplayModeSetToDefaults,
    scrDisplayModeSet,
    scrDisplayModeGetSupportedDepths,
    scrDisplayModeGetSupportsColor
} ScrDisplayModeOperation;

Err ScrDisplayMode(ScrDisplayModeOperation operation,
                   DWordPtr widthP, DWordPtr heightP, DWordPtr depthP,
                   BooleanPtr enableColorP) SYS_TRAP(0xa33e);

static DWord screenGetDepths()
{
    if (getPalmOSVersionMajor() >= 3) {
        DWord supportedDepths;
        Err   err;
        
        err = ScrDisplayMode(scrDisplayModeGetSupportedDepths, NULL, NULL,
                             &supportedDepths, NULL);

        if (err == 0)
            return supportedDepths;
    }
    return 0x1;
}


static DWord getPalmOSVersion()
{
    DWord romVersion = 0;
    Err err = FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
        
    if (err != 0) /* should never happen according to Palm */
        romVersion = 0x00000001; /* 0.1 */

    return romVersion;
}

static DWord getPalmOSVersionMajor()
{
    return sysGetROMVerMajor(getPalmOSVersion());
}

/*
http://www.palm.com/devzone/faqs/Hardware.html#a32

Can I access a unique serial number for a Palm device? (4/10/98)

Flash-ROM-based Palm III devices have unique serial numbers burned into them at 
production. Previous Palm Computing Platform devices did not have unique serial 
numbers available to software, and it isn't guaranteed that all future devices 
will have them. 

If a device doesn't have a unique serial number in its ROM, then there's no 
perfectly reliable way to tell them apart; the only thing you can do is rely on 
things like creation dates for cards, user names, user IDs, etc. It's not 
guaranteed that they will be unique - but it's the only thing you have. In just 
the same way, normal desktop computers don't have any software-accessible serial 
numbers; you can't tell them apart from each other with certainty.  

If a Palm Computing Platform device has a serial number, then it is a 
displayable text buffer, currently 12 characters long, with NO null terminator. 
It is shown to the user in the Application Launcher, along with a checksum digit 
which you can use to validate input in case your users need to read the ID from 
their device and type it in or tell it to someone else.
*/
static void *getSerialNumber()
{
	//PENDING(klobad) figure out security concerns
	return NULL;
	
/*
    CharPtr bufP;
    Word    bufLen;
    void   *result = NULL;
    Word    retval;

    if (getPalmOSVersionMajor() < 3)
        return NULL;

    retval = SysGetROMToken (0, sysROMTokenSnum, (BytePtr*) &bufP, &bufLen); 
    // if there's a serial number
    if ((!retval) && (bufP) && ((Byte) *bufP != 0xFF)) {  
        result = MemPtrNew(bufLen + 1);
        MemSet(result, bufLen + 1, 0);
        MemMove(result, bufP, bufLen);
    }

    return result;
*/
}


