/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

#include <Pilot.h>
#include <SysEvtMgr.h>
#include <mobilelinkRsc.h>
#include <AGUserConfig.h>
#include <AGBufferReader.h>
#include <AGBufferWriter.h>
#include <malclient.h>
#include <AGProtectedMem.h>
#include <AGNet.h>
#include <AGSyncCommon.h>
#include <password.h>
#include <AGMD5.h>
#include <mobilelinklaunch.h>

#include <libmal.h>

/***********************************************************************
 *
 *   Global variables
 *
 ***********************************************************************/

typedef struct MLGlobalState {
    AGBool initedUserConfig;
    AGUserConfig userConfig;
    AGServerConfig *serverConfig;
    int16 currentConfigIndex;
    MobileLinkLaunchData *launchData;
    
    char* passwordSetString;
    char* passwordUnsetString;
    VoidHand passwordSetStringHandle;
    VoidHand passwordUnsetStringHandle;
} MLGlobalState;

static MLGlobalState state;

Word MALLibRefNum;

static VoidPtr GetObjectPtr(Word objectID);
static void initServerPopup();
static void setListSelection(UInt index);
static void finalizeServerPopup();
static void doModemSync();
static void reapUIValues();
static void initUserConfig();
static void finalizeUserConfig();
static void saveUserConfig();
static void setPasswordTriggerText();

static void newServer();
static void deleteServer();

static void queueAppStopEvent();

/***********************************************************************
 *
 *   Internal Constants
 *
 ***********************************************************************/

// Define the minimum OS version we support
#define ourMinVersion sysMakeROMVersion(2,0,0,sysROMStageRelease,0)
#define DEFAULT_CARD_NUM (0)

/***********************************************************************
 *
 *   Internal Functions
 *
 ***********************************************************************/

static void doModemSync()
{
    reapUIValues();
    saveUserConfig();

    SysBroadcastActionCode(MobileLinkPreSyncLaunchCode, NULL);
    MALSyncAllServers(&state.userConfig);

    saveUserConfig();
    SysBroadcastActionCode(MobileLinkPostSyncLaunchCode, NULL);
    finalizeUserConfig();

    initUserConfig();
    initServerPopup();
    setListSelection(0);
}

static char *myStrdup(char *s)
{
    char *r;
    Word len;

    if(s == NULL)
        s = "";
        
    len = StrLen(s) + 1;
    r = MemPtrNew(len);
    MemMove(r, s, len);
    return r;
}

static void
utilFldSetText(FieldPtr field, char* s)
{
    Word len = FldGetTextLength(field);

    FldDelete(field, 0, len);
    if(s != NULL)
        FldInsert(field, s, StrLen(s));
}

static void
utilFldSetNumeric(FieldPtr field, Int n)
{
    char buf[16];

    StrIToA(buf, n);

    utilFldSetText(field, buf);
}

static Int
utilFldGetNumeric(FieldPtr field)
{
    char* p = FldGetTextPtr(field);
    Int   rv = 0; /* good default? */

    if (p != NULL)
        rv = StrAToI(p);

    return rv;
}

//isn't this magic 16 a constant anywhere?
static void clearPassword(uint8 pwd[16])
{
    int i;
    for (i = 0; i < 16; i++)
    pwd[i]=0;
}

static AGBool passwordIsSet(uint8 pwd[16])
{
    int i;
    for (i = 0; i < 16; i++)
    if (pwd[i])
        return true;
    return false;
}

static void doGetPassword()
{
    char* newPassword = PwdDialogDo();
    int    newPasswordLen;
    if (newPassword != NULL){
        newPasswordLen = StrLen(newPassword);
        if (0 == newPasswordLen)
            clearPassword(state.serverConfig->password);
        else
            MALLibAGMd5(MALLibRefNum, (uint8 *)newPassword, newPasswordLen, state.serverConfig->password);
        MemPtrFree(newPassword);
        state.userConfig.dirty = TRUE;
        saveUserConfig();            
    }
}

static void setListSelection(UInt index)
{
    ControlPtr control;
    ListPtr list;
    FieldPtr field;
    AGUserConfig *userConfig;
    AGServerConfig *serverConfig;

    userConfig = &state.userConfig;

    if(index >= MALLibAGUserConfigCount(MALLibRefNum, userConfig))
        return;

    serverConfig = MALLibAGUserConfigGetServerByIndex(MALLibRefNum, userConfig, index);
    state.serverConfig = serverConfig;
    state.currentConfigIndex = index;

    list = GetObjectPtr(MainServerList);
    LstSetSelection(list, index);

    field = (FieldPtr)GetObjectPtr(MainServerNameField);
    utilFldSetText(field, serverConfig->friendlyName);

    field = (FieldPtr)GetObjectPtr(MainServerURLField);
    utilFldSetText(field, serverConfig->serverName);

    field = (FieldPtr)GetObjectPtr(MainUserNameField);
    utilFldSetText(field, serverConfig->userName);

    field = (FieldPtr)GetObjectPtr(MainPortField);
    utilFldSetNumeric(field, serverConfig->serverPort);

    control = (ControlPtr)GetObjectPtr(MainEnabledCheckbox);
    CtlSetValue(control, ! serverConfig->disabled);

    setPasswordTriggerText();    
}

static void initUserConfig()
{
    AGBufferReader r;
    LocalID id;
    DmOpenRef dbRef;
    void *ptr;
    AGUserConfig *userConfig = &state.userConfig;
    VoidHand handle;

    if(state.initedUserConfig)
        finalizeUserConfig();

    id = DmFindDatabase(DEFAULT_CARD_NUM, DEVICE_USERCONFIG_DB_NAME);
    if(id != 0) {
        dbRef = DmOpenDatabase(DEFAULT_CARD_NUM, id, dmModeReadOnly);
        handle = DmQueryRecord(dbRef, 0);
        if(handle) {
            ptr = MemHandleLock(handle);
            MALLibAGBufferReaderInit(MALLibRefNum, &r, (uint8 *)ptr);
            MALLibAGUserConfigInitAndReadData(MALLibRefNum, userConfig, (AGReader *)&r);
            MALLibAGBufferReaderFinalize(MALLibRefNum, &r);
            MemHandleUnlock(handle);
        } else {
            MALLibAGUserConfigInit(MALLibRefNum, userConfig);   
        }
        DmCloseDatabase(dbRef);
        userConfig->dirty = FALSE;
    } else {
        MALLibAGUserConfigInit(MALLibRefNum, userConfig);
    }
    state.initedUserConfig = TRUE;
    
    state.passwordSetStringHandle = DmGetResource(strRsc, PasswordSetString);
    state.passwordSetString = (char*)MemHandleLock(state.passwordSetStringHandle);
    state.passwordUnsetStringHandle = DmGetResource(strRsc, PasswordUnsetString);
    state.passwordUnsetString = (char*)MemHandleLock(state.passwordUnsetStringHandle);
}

static void finalizeUserConfig()
{
    if(state.initedUserConfig) { 
        MemHandleUnlock(state.passwordSetStringHandle); 
        DmReleaseResource(state.passwordSetStringHandle);
        state.passwordSetString = NULL;
        
        MemHandleUnlock(state.passwordUnsetStringHandle); 
        DmReleaseResource(state.passwordUnsetStringHandle);
        state.passwordUnsetString = NULL;

        MALLibAGUserConfigFinalize(MALLibRefNum, &state.userConfig);
        state.initedUserConfig = FALSE;
    }
}

static void saveUserConfig()
{
    AGBufferWriter w;
    LocalID id;
    DmOpenRef dbRef;
    VoidHand handle;
    void *ptr;
    UInt indx = 0;
    Err er;
    AGUserConfig *userConfig = &state.userConfig;
    int32 size;

    if(!state.initedUserConfig || !userConfig->dirty ) {
        return;
    }
    
    id = DmFindDatabase(DEFAULT_CARD_NUM, DEVICE_USERCONFIG_DB_NAME);
    if(id == 0) {
        er = DmCreateDatabase(DEFAULT_CARD_NUM, 
                                DEVICE_USERCONFIG_DB_NAME, 
                                DEVICE_USERCONFIG_DB_CREATOR, 
                                DEVICE_USERCONFIG_DB_TYPE, 
                                false);
        id = DmFindDatabase(DEFAULT_CARD_NUM, DEVICE_USERCONFIG_DB_NAME);
    }
    if(id != 0) {
        MALLibAGBufferWriterInit(MALLibRefNum, &w, 1024);
        MALLibAGUserConfigWriteData(MALLibRefNum, userConfig, (AGWriter *)&w);

        dbRef = DmOpenDatabase(DEFAULT_CARD_NUM, id, dmModeReadWrite);
        if(DmNumRecords(dbRef) > 0) {
            DmRemoveRecord(dbRef, 0);
        }
        size = MALLibAGBufferWriterGetBufferSize(MALLibRefNum, &w);
        handle = DmNewRecord(dbRef, &indx, size);
        ptr = MemHandleLock(handle);
        DmWrite(ptr, 0, MALLibAGBufferWriterGetBuffer(MALLibRefNum, &w), 
                            MALLibAGBufferWriterGetBufferSize(MALLibRefNum, &w));
        MALLibAGBufferWriterFinalize(MALLibRefNum, &w);
        MemHandleUnlock(handle);
        DmReleaseRecord(dbRef, indx, true);
        DmCloseDatabase(dbRef);
        userConfig->dirty = FALSE;
    }
}

static Boolean differ(char *a, char *b)
{
    if(a == NULL && b == NULL)
        return false;
    if(a != NULL && b == NULL)
        return true;
    if(a == NULL && b != NULL)
        return true;

    return (StrCompare(a, b) != 0);
}

static void reapUIValues()
{
    char *serverurl = NULL, *username = NULL, 
         *servername = NULL;
    short port = 0;
    ControlPtr control;
    FieldPtr field;
    AGUserConfig *userConfig;
    AGServerConfig *serverConfig;
    char **servers;
    AGBool disabled;
    ListPtr list;
    Word selection;

    if(MainForm != FrmGetActiveFormID())
        return;

    userConfig = &state.userConfig;
    serverConfig = state.serverConfig;
   
    field = (FieldPtr)GetObjectPtr(MainServerNameField);
    servername = FldGetTextPtr(field);
    if(differ(servername, serverConfig->friendlyName)) {
        if(serverConfig->friendlyName)
            MemPtrFree(serverConfig->friendlyName);
        serverConfig->friendlyName = myStrdup(servername);
        userConfig->dirty = TRUE;

        // Update the selection list if the name changes
        list = GetObjectPtr(MainServerList);
        servers = list->itemsText;
        selection = state.currentConfigIndex;
        MemPtrFree(servers[selection]);
        servers[selection] = myStrdup(servername);
        LstDrawList(list);

    }

    field = (FieldPtr)GetObjectPtr(MainServerURLField);
    serverurl = FldGetTextPtr(field);
    if(differ(serverurl, serverConfig->serverName)) {
        if(serverConfig->serverName)
            MemPtrFree(serverConfig->serverName);
        serverConfig->serverName = myStrdup(serverurl);
        userConfig->dirty = TRUE;
    }
    
    field = (FieldPtr)GetObjectPtr(MainUserNameField);
    username = FldGetTextPtr(field);
    if(differ(username, serverConfig->userName)) {
        if(serverConfig->userName)
            MemPtrFree(serverConfig->userName);
        serverConfig->userName = myStrdup(username);
        userConfig->dirty = TRUE;
    }

    field = (FieldPtr)GetObjectPtr(MainPortField);
    port = utilFldGetNumeric(field);
    if(port != serverConfig->serverPort) {
        serverConfig->serverPort = port;
        userConfig->dirty = TRUE;
    }

    control = (ControlPtr)GetObjectPtr(MainEnabledCheckbox);
    disabled = ! CtlGetValue(control);
    if (disabled != serverConfig->disabled) {
        serverConfig->disabled = disabled;
        userConfig->dirty = TRUE;
    }
    
}

static void initServerPopup()
{
    int32 count, i, visCount;
    AGUserConfig *userConfig = &state.userConfig;
    ListPtr list;

    count = MALLibAGUserConfigCount(MALLibRefNum, userConfig);
    if(count > 0) {
        char **servers;
        AGServerConfig *server;

        servers = (char **)MemPtrNew(sizeof(unsigned char *) * count);
        for(i = 0; i < count; i++) {
            server = MALLibAGUserConfigGetServerByIndex(MALLibRefNum, userConfig, i);
            servers[i] = myStrdup(server->friendlyName);
        }

        list = GetObjectPtr(MainServerList);
        LstSetListChoices(list, servers, count);
        LstSetSelection(list, 0);

        // make sure the list is not bigger than 14 
        // - the number of items visible on a 160 screen
        visCount = min(count, 14); 
        // make sure the list is not smaller than 1
        visCount = max(1, visCount);     
        LstSetHeight(list, visCount);
    } else {
        //disable popup
    }
}

static void setPasswordTriggerText()
{
    ControlPtr control;
    control = (ControlPtr)GetObjectPtr(MainPasswordSelTrigger);
    //Because CtlSetLabel is leaving a few pixels when the label gets longer.
    CtlEraseControl(control); 
    CtlSetLabel(control, passwordIsSet(state.serverConfig->password) 
            ? state.passwordSetString : state.passwordUnsetString); 
    CtlDrawControl(control);
}

static void finalizeServerPopup()
{
    int32 count, i;
    char **servers;
    ListPtr list;

    list = GetObjectPtr(MainServerList);
    count = LstGetNumberOfItems(list);
    servers = list->itemsText;
    LstSetListChoices(list, NULL, 0);

    for(i = 0; i < count; i++) {
        MemPtrFree(servers[i]);
    }
    if(servers)
        MemPtrFree(servers);
}

static void newServer()
{
    AGUserConfig *userConfig = &state.userConfig;
    AGServerConfig *sConfig;

    reapUIValues();

    sConfig = MALLibAGServerConfigNew(MALLibRefNum);
    sConfig->serverPort = 80;
    sConfig->friendlyName = myStrdup("untitled");
    sConfig->disabled = FALSE;
    sConfig->serverName = myStrdup("");
    
    MALLibAGUserConfigAddServer(MALLibRefNum, userConfig, sConfig);
    saveUserConfig();
    finalizeUserConfig();
    finalizeServerPopup();

    initUserConfig();
    initServerPopup();
    setListSelection(MALLibAGUserConfigCount(MALLibRefNum, userConfig) - 1);
}

static void deleteServer()
{
    AGUserConfig *userConfig = &state.userConfig;
    AGServerConfig *serverConfig;

    if(FrmAlert(DeleteConfirmAlert))
        return;
    
    serverConfig = MALLibAGUserConfigGetServerByIndex(MALLibRefNum, userConfig, state.currentConfigIndex);
    MALLibAGUserConfigRemoveServer(MALLibRefNum, userConfig, serverConfig);

    saveUserConfig();
    finalizeUserConfig();
    finalizeServerPopup();

    initUserConfig();
    initServerPopup();

    // Cannot delete the last one
    if(MALLibAGUserConfigCount(MALLibRefNum, userConfig) < 1)
        newServer();
    setListSelection(0);
}

/***********************************************************************
 *
 * FUNCTION:    RomVersionCompatible
 *
 * DESCRIPTION: This routine checks that a ROM version is meet your
 *              minimum requirement.
 *
 * PARAMETERS:  requiredVersion - minimum rom version required
 *                                (see sysFtrNumROMVersion in SystemMgr.h 
 *                                for format)
 *              launchFlags     - flags that indicate if the application 
 *                                UI is initialized.
 *
 * RETURNED:    error code or zero if rom is compatible
 *
 * REVISION HISTORY:
 *
 ***********************************************************************/
static Err RomVersionCompatible(DWord requiredVersion, Word launchFlags)
{
    DWord romVersion;

    // See if we're on in minimum required version of the ROM or later.
    FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
    if (romVersion < requiredVersion)
        {
        if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
            (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
            {
            FrmAlert (RomIncompatibleAlert);
        
            // Pilot 1.0 will continuously relaunch this app unless we switch to 
            // another safe one.
            if (romVersion < sysMakeROMVersion(2,0,0,sysROMStageRelease,0))
                AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
            }
        
        return (sysErrRomIncompatible);
        }

    return (0);
}


/***********************************************************************
 *
 * FUNCTION:    GetObjectPtr
 *
 * DESCRIPTION: This routine returns a pointer to an object in the current
 *              form.
 *
 * PARAMETERS:  formId - id of the form to display
 *
 * RETURNED:    VoidPtr
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static VoidPtr GetObjectPtr(Word objectID)
    {
    FormPtr frmP;


    frmP = FrmGetActiveForm();
    return (FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, objectID)));
}

/***********************************************************************
 *
 * FUNCTION:    MainFormInit
 *
 * DESCRIPTION: This routine initializes the MainForm form.
 *
 * PARAMETERS:  frm - pointer to the MainForm form.
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void MainFormInit(FormPtr frmP)
{
    initUserConfig();
    initServerPopup();
    setListSelection(0);
    if (state.launchData)
    {
        switch((state.launchData)->command)
        {
            case MLServerPref:
                CtlSetUsable(GetObjectPtr(MainDoneButton), TRUE);
                CtlSetUsable(GetObjectPtr(MainMobileSyncButton), FALSE);
                break;                
            case MLModemSync:
                FrmDrawForm(frmP);
                doModemSync();
                queueAppStopEvent();
                break;
            default:
                break;
        }
    }
}

/***********************************************************************
 *
 * FUNCTION:    MainFormDoCommand
 *
 * DESCRIPTION: This routine performs the menu command specified.
 *
 * PARAMETERS:  command  - menu item id
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean MainFormDoCommand(Word command)
{
    Boolean handled = false;
    FormPtr aboutForm;

    switch (command)
        {
        case MainOptionsAboutMobileLink:
            MenuEraseStatus (0);
            aboutForm = FrmInitForm(AboutForm);
            FrmDoDialog(aboutForm);
            FrmDeleteForm(aboutForm);
            handled = true;
            break;
        case MainOptionsModemSync:
            MenuEraseStatus (0);
            doModemSync();
            handled = true;
            break;
        case MainServerNewServer:
            MenuEraseStatus (0);
            newServer();
            handled = true;
            break;
        case MainServerDelete:
            MenuEraseStatus (0);
            deleteServer();
            handled = true;
            break;
        case MainServerResetConnection:
            MenuEraseStatus(0);
            if (ResetConfirmOK == FrmAlert(ResetConfirmAlert)) {
                MALLibAGServerConfigResetCookie(MALLibRefNum, state.serverConfig);
                state.userConfig.dirty = TRUE;
                saveUserConfig();
            }
            handled = true;
            break;            
        }
    return handled;
}
    

/***********************************************************************
 *
 * FUNCTION:    MainFormHandleEvent
 *
 * DESCRIPTION: This routine is the event handler for the 
 *              "MainForm" of this application.
 *
 * PARAMETERS:  eventP  - a pointer to an EventType structure
 *
 * RETURNED:    true if the event has handle and should not be passed
 *              to a higher level handler.
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean MainFormHandleEvent(EventPtr eventP)
{
    Boolean handled = false;
    FormPtr frmP;


    switch (eventP->eType) 
        {
        case menuEvent:
            return MainFormDoCommand(eventP->data.menu.itemID);

        case frmOpenEvent:
            frmP = FrmGetActiveForm();
            MainFormInit( frmP);
            FrmDrawForm ( frmP);
            handled = true;
            break;

        case ctlSelectEvent:
            if(eventP->data.ctlSelect.controlID == MainMobileSyncButton) {
                doModemSync();
                handled = true;
            }
            else if(eventP->data.ctlSelect.controlID == MainPasswordSelTrigger){
                doGetPassword();
                setPasswordTriggerText();
                handled = true;
            }
            else if(eventP->data.popSelect.controlID == MainServerPopTrigger) {
                reapUIValues();
                saveUserConfig();
                //default handling will pop up the list
            }
            else if(eventP->data.ctlSelect.controlID == MainDoneButton){
                queueAppStopEvent();
                handled = true;
            }
            break;

        case popSelectEvent:
            if(eventP->data.popSelect.listID == MainServerList) {
                setListSelection(eventP->data.popSelect.selection);
                handled = true;
            }
            break;
        default:
            break;
        
        }
    
    return handled;
}

/***********************************************************************
 *
 * FUNCTION:    AppHandleEvent
 *
 * DESCRIPTION: This routine loads form resources and set the event
 *              handler for the form loaded.
 *
 * PARAMETERS:  event  - a pointer to an EventType structure
 *
 * RETURNED:    true if the event has handle and should not be passed
 *              to a higher level handler.
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean AppHandleEvent( EventPtr eventP)
{
    Word formId;
    FormPtr frmP;


    if (eventP->eType == frmLoadEvent)
        {
        // Load the form resource.
        formId = eventP->data.frmLoad.formID;
        frmP = FrmInitForm(formId);
        FrmSetActiveForm(frmP);

        // Set the event handler for the form.  The handler of the currently
        // active form is called by FrmHandleEvent each time is receives an
        // event.
        switch (formId)
            {
            case MainForm:
                FrmSetEventHandler(frmP, MainFormHandleEvent);
                break;
            default:
//                ErrFatalDisplay("Invalid Form Load Event");
                break;

            }
        return true;
        }
    
    return false;
}


/***********************************************************************
 *
 * FUNCTION:    AppEventLoop
 *
 * DESCRIPTION: This routine is the event loop for the application.  
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void AppEventLoop(void)
{
    Word error;
    EventType event;


    do {
        EvtGetEvent(&event, evtWaitForever);
        
        
        if (! SysHandleEvent(&event))
            if (! MenuHandleEvent(0, &event, &error))
                if (! AppHandleEvent(&event))
                    FrmDispatchEvent(&event);

    } while (event.eType != appStopEvent);
}

static Err
openLibMal(Word* refNum)
{
    Err      err;
    VoidHand librHandle;

    /* look for built in libmal */
    librHandle = DmGet1Resource('libr', 0);
    if (librHandle != 0) { /* found builtin libmal */
        SysLibEntryProcPtr code;
    
        code = (SysLibEntryProcPtr)MemHandleLock(librHandle);
    
        err = SysLibInstall(code, refNum);

    } else { /* try for standalone libmal */

        err = SysLibLoad('libr', 'MBln', refNum);
    }

    if (err != 0) {
        *refNum = 0;
        FrmCustomAlert(5000, 
            "MobileLink requires libmal.prc to function properly, ", 
            "but it does not appear to be installed.", " ");
    }

    return err;
}

/***********************************************************************
 *
 * FUNCTION:     AppStart
 *
 * DESCRIPTION:  Get the current application's preferences.
 *
 * PARAMETERS:   nothing
 *
 * RETURNED:     Err value 0 if nothing went wrong
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Err AppStart(void)
{
    Err err;
    AGServerConfig *sConfig;
    AGUserConfig *userConfig = &state.userConfig;

    err = openLibMal(&MALLibRefNum);

    if (err != 0) {
    return err;
    }

    err = MALLibOpen(MALLibRefNum);
    if (err != 0) {
    return err;
    }

    MemSet(&state, sizeof(MLGlobalState), 0);
    MALLibAGProtectedMemoryInit(MALLibRefNum);

    initUserConfig();
    userConfig = &state.userConfig;
    if(MALLibAGUserConfigCount(MALLibRefNum, userConfig) < 1) {
    
        sConfig = MALLibAGServerConfigNew(MALLibRefNum);
        sConfig->serverPort = 80;
        sConfig->friendlyName = myStrdup("untitled");
    
        MALLibAGUserConfigAddServer(MALLibRefNum, userConfig, sConfig);
        saveUserConfig();
    }
    return 0;
}

static void queueAppStopEvent()
{
     EventType newEvent;
     newEvent.eType = appStopEvent;
     EvtAddEventToQueue(&newEvent);
}

/***********************************************************************
 *
 * FUNCTION:    AppStop
 *
 * DESCRIPTION: Save the current state of the application.
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void AppStop(void)
{
    reapUIValues();
    saveUserConfig();
    finalizeUserConfig();
    finalizeServerPopup();
    MALLibAGProtectedMemoryFinalize(MALLibRefNum);    
    MALLibClose(MALLibRefNum);
    SysLibRemove(MALLibRefNum);
}

/***********************************************************************
 *
 * FUNCTION:    StarterPilotMain
 *
 * DESCRIPTION: This is the main entry point for the application.
 * PARAMETERS:  cmd - word value specifying the launch code. 
 *              cmdPB - pointer to a structure that is associated with the launch code. 
 *              launchFlags -  word value providing extra information about the launch.
 *
 * RETURNED:    Result of launch
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static DWord StarterPilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    Err error;


    error = RomVersionCompatible (ourMinVersion, launchFlags);
    if (error) return (error);


    switch (cmd)
        {
        case sysAppLaunchCmdPanelCalledFromApp:
        case sysAppLaunchCmdNormalLaunch:
            error = AppStart();
            if (error) 
                return error;
            if (cmd == sysAppLaunchCmdPanelCalledFromApp)
                state.launchData = (MobileLinkLaunchData*) cmdPBP;
            FrmGotoForm(MainForm);

            AppEventLoop();
            AppStop();
            break;

        default:
            break;

        }
    
    return 0;
}


/***********************************************************************
 *
 * FUNCTION:    PilotMain
 *
 * DESCRIPTION: This is the main entry point for the application.
 *
 * PARAMETERS:  cmd - word value specifying the launch code. 
 *              cmdPB - pointer to a structure that is associated with the launch code. 
 *              launchFlags -  word value providing extra information about the launch.
 * RETURNED:    Result of launch
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
DWord PilotMain( Word cmd, Ptr cmdPBP, Word launchFlags)
{
    return StarterPilotMain(cmd, cmdPBP, launchFlags);
}


