/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <stdio.h>
#include <crtdbg.h>

#include <AGUtil.h>
#include <AGClientProcessor.h>
#include <AGProtocol.h>
#include <AGBufferReader.h>
#include <AGSyncPalm.h>
#include <AGSyncPalmResource.h>
#include <AGSyncPalmStatusDialog.h>
#include <AGPalmProtocol.h>
#include <AGUserConfig.h>
#include <AGServerConfig.h>
#include <AGSyncCommon.h>
#include <AGCommandProcessor.h>
#include <AGShlapi.h>
#include <AGDesktopInfoPalm.h>
#include <AGTimeoutMessageBox.h>
#include <AGMobileLinkSettings.h>

/* WARNING (miket):  The resources for this message box are all screwed up.
    I couldn't get multiple scripts to be included in a single project, so
    the resource.h defines are duplicated in several places, and the
    resources themselves are duplicated.  I recommend grepping for things
    like IDC_LINK to find everything. */
#include <AGLinkMessageBox.h>

typedef void *CString;
typedef void *CStringArray;
#include <syncmgr.h>
#include <condapi.h>
#include <hslog.h>
#include <basemon.h>
#define DEFAULT_CARD_NUM 0

#ifdef _DEBUG
#define debug_print(x) OutputDebugString(x)
#else
#define debug_print(x)
#endif

typedef sword (*netInitFunc)(AGNetCtx *ctx);
typedef sword (*netCloseFunc)(AGNetCtx *ctx);
typedef int32 (*netCtxSizeFunc)(void);
typedef void  (*netPreSyncHook) (AGNetCtx *ctx, 
                                AGServerConfig *sc,
                                AGLocationConfig *lc,
                                AGSyncProcessor *sp,
                                AGBool connectSecure);
typedef void  (*netPostSyncHook) (AGNetCtx *ctx, 
                                 AGServerConfig *sc,
                                 AGLocationConfig *lc,
                                 AGSyncProcessor *sp,
                                 AGBool connectSecure);

typedef struct {

    AGDeviceInfo * deviceInfo;
    AGLocationConfig * locationConfig;
    AGUserConfig * userConfig;
    AGServerConfig * currentServerConfig;
    AGClientProcessor * clientProcessor;
    AGPlatformCalls * platform;
    AGRecord * record;
    AGDBConfig * currentDb;
    AGSyncPalmStatusDialog * statusDialog;
    AGCommandProcessor *commandProcessor;
    AGBool quit;
    
    // Pilot-specific
    AGBool currentDBIsResourceType;
    BYTE pilot_rHandle;
    WORD pilot_RecIndex;
    CRawRecordInfo pilot_rRec;
    BYTE * pilot_buffer;
    int16 pilot_buffer_size;
    
    /* Secure Network Library Stuff */
    AGBool          hasseclib;
    netInitFunc     secnetinit;
    netCloseFunc    secnetclose;
    netCtxSizeFunc  secctxsize;
    netPreSyncHook  sec_presynchook;
    netPreSyncHook  sec_postsynchook;

} PalmSyncInfo;

static int32 initAndOpenDatabase(void *out, AGDBConfig *theDatabase, 
                                 int32 *errCode);
static int32 getNextModifiedRecord(void * out, AGRecord ** record, 
                                   int32 *errCode);
static int32 getNextRecord(void * out, AGRecord ** record,
                           int32 *errCode);

// Platform Specific Calls
static int32 cmdTASK(void *out, int32 *returnErrorCode, char *currentTask,
                     AGBool bufferable);
static int32 cmdITEM(void *out, int32 *returnErrorCode,
                                   int32 currentItemNumber,
                                   int32 totalItemCount,
                                   char *currentItem);
static int32 cmdDELETEDATABASE(void *out, int32 *returnErrorCode,
                                             char *dbname);
static int32 cmdOPENDATABASE(void *out, int32 *returnErrorCode,
                                           char *dbname);
static int32 cmdCLOSEDATABASE(void *out, int32 *returnErrorCode);
static int32 cmdCLEARMODS(void *out, int32 *returnErrorCode);
static int32 cmdGOODBYE(void *out, int32 *returnErrorCode,
                                      AGSyncStatus syncStatus,
                                      int32 errorCode,
                                      char *errorMessage);
static int32 cmdRECORD(void *out, int32 *returnErrorCode,
                                     int32 *newUID,
                                     int32 uid,
                                     AGRecordStatus mod,
                                     int32 recordDataLength,
                                     void *recordData,
                                     int32 platformDataLength,
                                     void *platformData);
//static int32 cmdEXPANSION(void *out, int32 *returnErrorCode,
//                                      int32 expansionCommand, 
//                                      int32 commandLength,
//                                      void *commandBytes);


/* ----------------------------------------------------------------------------
    void syncInfoFree(PalmSyncInfo * pInfo)

    Free the given PalmSyncInfo structure and everything that it knows about.

*/
void syncInfoFree(PalmSyncInfo * pInfo)
{
    if (NULL != pInfo) {

        if (NULL != pInfo->platform)
            free(pInfo->platform);

        if (NULL != pInfo->statusDialog)
            AGSyncPalmStatusDialogFree(pInfo->statusDialog);

        if (NULL != pInfo->locationConfig)
            AGLocationConfigFree(pInfo->locationConfig);

        if (NULL != pInfo->userConfig)
            AGUserConfigFree(pInfo->userConfig);

        if (NULL != pInfo->pilot_buffer)
            free(pInfo->pilot_buffer);

        if (NULL != pInfo->commandProcessor)
            AGCommandProcessorFree(pInfo->commandProcessor);

        free(pInfo);
    }
}

/* ----------------------------------------------------------------------------
    PalmSyncInfo * syncInfoNew()

    Allocate PalmSyncInfo structure and whatever else is guaranteed to be
    needed, as well.

*/
PalmSyncInfo * syncInfoNew()
{
    PalmSyncInfo * pInfo;

    /* Allocate the PalmSyncInfo structure. */
    pInfo = (PalmSyncInfo *)malloc(sizeof(PalmSyncInfo));
    if (NULL != pInfo) {

        const WORD pbs = 65535;

        bzero(pInfo, sizeof(PalmSyncInfo));

        pInfo->pilot_buffer_size    = pbs;
        pInfo->pilot_buffer         = (BYTE*)malloc(pbs);
        if (NULL == pInfo->pilot_buffer)
            goto fail;

        /* Allocate the platform calls record. */
        pInfo->platform = (AGPlatformCalls *)malloc(sizeof(AGPlatformCalls));
        bzero(pInfo->platform, sizeof(AGPlatformCalls));
        if (NULL == pInfo->platform)
            goto fail;

        return pInfo;
    }

fail:
    if (NULL != pInfo)
        syncInfoFree(pInfo);

    return NULL;
}

/* ----------------------------------------------------------------------------
    static void doWindowsMessagePump()

*/
static void doWindowsMessagePump()
{
    MSG msg;
    while (PeekMessage(&msg, 0, 0, 0, PM_REMOVE | PM_NOYIELD)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
}

static void getUserConfigFilename(char * userName,
                                  char * userConfig,
                                  char * syncUserConfig)
{
    char * filename;
    char prefsPath[MAX_PATH];
    int bufSize;

    prefsPath[0] = (char)0;

    if (NULL != userName) {

        AGDesktopInfoPalm * desktopInfo;
        char * temp;
    
        desktopInfo = AGDesktopInfoPalmNew();
        if (NULL == desktopInfo)
            return;
    
        bufSize = MAX_PATH;
        AGDesktopInfoPalmGetUserDirectory(desktopInfo,
            userName,
            prefsPath,
            &bufSize);

        /* Build the path of the config file. */
        temp = (char*)AGSyncCommonGetStringConstant(agMALSubdirectoryName,
            FALSE);
        PathAppend(prefsPath, temp);
        free(temp);

        AGDesktopInfoPalmFree(desktopInfo);

    }

    if (prefsPath[0] != '\0') {

        if (NULL != userConfig) {

            filename = (char*)AGSyncCommonGetStringConstant(
                agPreferencesFilename, FALSE);
            PathCombine(userConfig, prefsPath, filename);
            free(filename);

        }

        if (NULL != syncUserConfig) {

            filename = (char*)AGSyncCommonGetStringConstant(
                agSynchronizedPreferencesFilename, FALSE);
            PathCombine(syncUserConfig, prefsPath, filename);
            free(filename);

        }

    }

}

/* ----------------------------------------------------------------------------
    static AGBool writeUserConfigs(AGUserConfig * userConfig, char * userName)
*/
static AGBool writeUserConfigs(AGUserConfig * userConfig, char * userName)
{
    char userConfigFilename[MAX_PATH];
    char syncUserConfigFilename[MAX_PATH];

    getUserConfigFilename(userName,
        userConfigFilename,
        syncUserConfigFilename);

    /* Write out config files to disk. */
    AGWriteUserConfigToDisk(userConfigFilename, userConfig);
    AGWriteUserConfigToDisk(syncUserConfigFilename, userConfig);

    return FALSE;
}

/* ----------------------------------------------------------------------------
    static AGBool readLocationConfig(PalmSyncInfo * pInfo)

    Part of initialization:  Read locationConfig file from disk.

*/
static AGBool readLocationConfig(PalmSyncInfo * pInfo)
{
    TCHAR * stringPointer;

    stringPointer = (TCHAR*)
        AGSyncCommonGetStringConstant(agLocationConfigPath, FALSE);
    pInfo->locationConfig =
        AGReadLocationConfigFromDisk(stringPointer);
    free(stringPointer);

    return FALSE;
}

/* ----------------------------------------------------------------------------
    static AGBool setupPlatformCalls(PalmSyncInfo * pInfo)
*/
static AGBool setupPlatformCalls(PalmSyncInfo * pInfo)
{
    pInfo->platform->out = pInfo;
    pInfo->platform->nextModifiedRecordFunc =
        getNextModifiedRecord;
    pInfo->platform->nextRecordFunc =
        getNextRecord;
    pInfo->platform->openDatabaseFunc =
        initAndOpenDatabase;

    return FALSE;
}

/* ----------------------------------------------------------------------------
    static AGBool startStatusDialog(PalmSyncInfo * pInfo,
                                    HINSTANCE h,
                                    AGArray * graphic,
                                    AGBool bitmapFormat)
*/
static AGBool startStatusDialog(PalmSyncInfo * pInfo,
                                HINSTANCE h,
                                AGArray * graphic,
                                AGBool bitmapFormat)
{
    char strbuf[256];

    pInfo->statusDialog = AGSyncPalmStatusDialogNew(h,
        &pInfo->quit,
        graphic,
        250,
        bitmapFormat);
    pInfo->quit = FALSE;

    if (NULL != pInfo->statusDialog) {
        /* pending(miket): we lose server-side localization by hard-coding
        a string in here.  How can we give a universal "please wait"
        message so we don't have to distribute language-specific clients? */
        LoadString(h, IDS_CONNECTING, strbuf, 256);
        AGSyncPalmStatusDialogSetTask(pInfo->statusDialog,
            strbuf);
        return FALSE;
    }
    return TRUE;
}

/* ----------------------------------------------------------------------------
    static void setStatusDialogToConnecting(PalmSyncInfo * pInfo, HINSTANCE h)
*/
static void setStatusDialogToConnecting(PalmSyncInfo * pInfo, HINSTANCE h)
{
    char strbuf[256];
    LoadString(h, IDS_CONNECTING, strbuf, 256);
    AGSyncPalmStatusDialogSetTask(pInfo->statusDialog, strbuf);
}

/* ----------------------------------------------------------------------------
    static void readDeviceInfo(PalmSyncInfo * pInfo)
*/
static void readDeviceInfo(PalmSyncInfo * pInfo)
{
    AGDeviceInfo * devInfo;
    CSystemInfo sysInfo;
    CCardInfo cardInfo;
    int majorver, minorver;
    char osverstringmajor[3];
    char osverstringminor[2];
    char osverstring[6];
    char * minorptr;

    /* Start with clean slate. */
    devInfo = pInfo->deviceInfo;

    sysInfo.m_AllocedLen = 0x7f; /* Grrr, byte */
    sysInfo.m_ProductIdText = (BYTE*)malloc(sysInfo.m_AllocedLen);
    if (NULL != sysInfo.m_ProductIdText) {
        SyncReadSystemInfo(sysInfo);
        free(sysInfo.m_ProductIdText);
    }
    cardInfo.m_CardNo = DEFAULT_CARD_NUM;
    SyncReadSingleCardInfo(cardInfo);

    /* Find out version of operating system. */
    majorver = sysInfo.m_RomSoftVersion >> 24;
    minorver = 0xff & (sysInfo.m_RomSoftVersion >> 16);
    _itoa(majorver, osverstringmajor, 10);
    if (minorver < 10) {
        strcpy(osverstringminor, "0");
        minorptr = &osverstringminor[1];
    }
    else
        minorptr = &osverstringminor[0];
    _itoa(minorver, minorptr, 10);
    strcpy(osverstring, osverstringmajor);
    strcat(osverstring, ".");
    strcat(osverstring, osverstringminor);

    devInfo->availableBytes   = cardInfo.m_FreeRam;
    devInfo->serialNumber     = strdup("");
    devInfo->osName           = strdup("PALM_OS");
    devInfo->osVersion        = strdup(osverstring);
    devInfo->screenWidth      = 150; /* pending(miket): get right values */
    devInfo->screenHeight     = 150; /* pending(miket): get right values */
    if(majorver > 2)
       devInfo->colorDepth    = 2; /* pending(miket): get right values */
    else
       devInfo->colorDepth    = 1; /* pending(miket): get right values */
}

/* ----------------------------------------------------------------------------
    static AGBool getPalmDatabaseCreationInfo(AGDBConfig * db, uint32 * creator,
    uint32 * flags, uint32 * type)

    Given a pointer to a AGDBConfig struct, pull Palm-specific information out
    of the platform-specific data in it.

    -> AGDBConfig * db: structure from which to pull information.

    -> uint32 * creator: pointer to dword that will receive Palm creator info.
    NULL if this item isn't needed.

    -> uint32 * flags: pointer to dword that will receive Palm type info.
    NULL if this item isn't needed.

    -> uint32 * type: pointer to dword that will receive Palm type info.
    NULL if this item isn't needed.

    <- TRUE if no error; FALSE otherwise.

*/
static AGBool getPalmDatabaseCreationInfo(AGDBConfig * db, uint32 * creator,
                                          uint32 * flags, uint32 * type)
{
    debug_print("GetPalmDatabaseCreationInfo()\n");

    AGBufferReader * r = NULL;

    if (NULL == db)
        return FALSE;
    if (0 == db->platformDataLength || NULL == db->platformData)
        return FALSE;

    r = AGBufferReaderNew((uint8*)db->platformData);
    if (NULL == r)
        return FALSE;

    AGPalmReadDBConfigPlatformData((AGReader*)r, creator, type, flags);
    AGBufferReaderFree(r);
    return TRUE;
}

/* ----------------------------------------------------------------------------
    static BYTE createDatabase(AGDBConfig * db)

    Wrapper around Palm SyncCreateDB().

    -> AGDBConfig * db: pointer to AGDBConfig structure describing the database
    to create.

    <- BYTE: handle to opened database on device (non-zero if successful).

*/
static BYTE createDatabase(AGDBConfig * db)
{
    debug_print("CreateDatabase()\n");

    BYTE result = 0;
    CDbCreateDB cdb;
    DWORD creator;
    eDbFlags flags;
    BYTE cardNo = DEFAULT_CARD_NUM;
    DWORD type;
    WORD version = 0;

    if (NULL == db)
        return 0;

    getPalmDatabaseCreationInfo(db, (uint32*)&creator, 
        (uint32*)&flags, (uint32*)&type);
    cdb.m_Creator    = creator;
    cdb.m_Flags      = flags;
    cdb.m_CardNo     = cardNo;
    // copy only as much of database as will fit into the Pilot field.
    if (strlen(db->dbname) < DB_NAMELEN)
        strcpy(cdb.m_Name, db->dbname);
    else {
        debug_print("WARNING: Attempt to create database having name of \
excess length!\n");
        strncpy(cdb.m_Name, db->dbname, DB_NAMELEN - 1);
    }
    cdb.m_Type       = type;
    cdb.m_Version    = version;
    if (SYNCERR_NONE == SyncCreateDB(cdb))
        result = cdb.m_FileHandle;
    return result;
}

/* ----------------------------------------------------------------------------
    static BYTE openDatabase(PalmSyncInfo * pInfo, char * dbname,
    AGBool create)

    Wrapper around Palm SyncOpenDB().  Creates database if create is TRUE and
    open fails because of SYNCERR_FILE_NOT_FOUND.

    -> PalmSyncInfo * pInfo: pointer to our information.

    -> char * dbname: pointer to null-terminated string naming database to
    open.

    -> AGBool create: whether to create if open fails.

    <- long: SYNCERR result code.

*/
static long openDatabase(PalmSyncInfo * pInfo, char * dbname,
                         AGBool create)
{
    debug_print("OpenDatabase() ");

    long result;
    if (NULL == dbname || NULL == pInfo) {
        debug_print("\n");
        return SYNCERR_BAD_ARG;
    }
    debug_print("... opening '");
    debug_print(dbname);
    debug_print("' ... ");

    pInfo->currentDb =
        AGServerConfigGetDBConfigNamed(pInfo->currentServerConfig, dbname);
    result = SyncOpenDB(dbname, DEFAULT_CARD_NUM, pInfo->pilot_rHandle,
        eDbRead | eDbWrite);
    if (SYNCERR_FILE_NOT_FOUND == result && create)
        pInfo->pilot_rHandle = createDatabase(pInfo->currentDb);

    if (pInfo->pilot_rHandle) {
        uint32 creator, flags, type;
        if (getPalmDatabaseCreationInfo(pInfo->currentDb, &creator, &flags, 
            &type) && (flags & 0x01) != 0) {
            pInfo->currentDBIsResourceType = TRUE;
        } else
            pInfo->currentDBIsResourceType = FALSE;
        debug_print("successfully.\n");
    } else {
        debug_print("unsuccessfully.\n");
        pInfo->currentDBIsResourceType = FALSE;
        pInfo->currentDb = NULL;
    }

    return result;
}

/* ----------------------------------------------------------------------------
    static void closeDatabase(PalmSyncInfo * pInfo)

    Wrapper around Palm SyncCloseDB().

    -> PalmSyncInfo * pInfo: pointer to our information.

    <- long: SYNCERR result code.

*/
static long closeDatabase(PalmSyncInfo * pInfo)
{
    debug_print("CloseDatabase()\n");

    long result = SyncCloseDB(pInfo->pilot_rHandle);
    pInfo->pilot_rHandle = 0;
    pInfo->currentDb = NULL;
    pInfo->currentDBIsResourceType = FALSE;
    return result;
}

/* ----------------------------------------------------------------------------
    static void deleteDatabase(char * dbname)
*/
static void deleteDatabase(char * dbname)
{
    debug_print("deleteDatabase()\n");
    SyncDeleteDB(dbname, DEFAULT_CARD_NUM);
}

/* ----------------------------------------------------------------------------
    static void clearMods(BYTE dbHandle)
*/
static void clearMods(BYTE dbHandle)
{
    debug_print("clearMods()\n");
    SyncPurgeDeletedRecs(dbHandle);
    SyncResetSyncFlags(dbHandle);
}

/* ----------------------------------------------------------------------------
    static BYTE openUserConfigDatabase(void)

*/
static BYTE openUserConfigDatabase(void)
{
    long result;
    BYTE userConfigDBHandle = 0;

    result = SyncOpenDB(DEVICE_USERCONFIG_DB_NAME,
        DEFAULT_CARD_NUM, userConfigDBHandle, eDbRead | eDbWrite);
    if (SYNCERR_FILE_NOT_FOUND == result) {

        CDbCreateDB cdb;
        cdb.m_Creator    = DEVICE_USERCONFIG_DB_CREATOR;
        cdb.m_Flags      = eRecord;
        cdb.m_CardNo     = DEFAULT_CARD_NUM;
        strcpy(cdb.m_Name, DEVICE_USERCONFIG_DB_NAME);
        cdb.m_Type       = DEVICE_USERCONFIG_DB_TYPE;
        cdb.m_Version    = 0;

        if (SYNCERR_NONE == SyncCreateDB(cdb))
            userConfigDBHandle = cdb.m_FileHandle;
    }
    return userConfigDBHandle;
}

/* ----------------------------------------------------------------------------
    static void fillInRecordStruct(CRawRecordInfo * rRec, [...] )

    Wrapper to fill in Palm CRawRecordInfo struct.

    -> CRawRecordInfo * rRec: pointer to preallocated CRawRecordInfo structure.

    -> (various parameters): see Palm documentation.

    <- none.

*/
static void fillInRecordStruct(CRawRecordInfo * rRec, BYTE FileHandle,
                                DWORD RecId, WORD RecIndex, BYTE Attribs,
                                int16 CatId, int32 ConduitId, DWORD RecSize,
                                WORD TotalBytes, BYTE * pBytes)
{
    debug_print("FillInRecordStruct()\n");

    if (NULL == rRec)
        return;

    rRec->m_FileHandle   = FileHandle;
    rRec->m_RecId        = RecId;
    rRec->m_RecIndex     = RecIndex;
    rRec->m_Attribs      = Attribs;
    rRec->m_CatId        = CatId;
    rRec->m_ConduitId    = ConduitId;
    rRec->m_RecSize      = RecSize;
    rRec->m_TotalBytes   = TotalBytes;
    rRec->m_pBytes       = pBytes;
    rRec->m_dwReserved   = 0;
}

/* ----------------------------------------------------------------------------
    static int32 readDeviceUserConfig(BYTE userConfigDBHandle,
                                      AGUserConfig ** deviceUserConfig)

    Reads the device's userConfig record.  Explodes if any bad
    parameters are passed in.

    Returns Pilot's unique ID for this record.

*/
static int32 readDeviceUserConfig(BYTE userConfigDBHandle,
                                  AGUserConfig ** deviceUserConfig)
{
    CRawRecordInfo rawRec;
    const uint32 bufferSize = 0xffff;
    uint8 buffer[bufferSize];
    AGBufferReader * r = NULL;

    fillInRecordStruct(&rawRec, userConfigDBHandle, 0, 0, 0,
        0, 0, 0, bufferSize, (BYTE*)buffer);
    if (SYNCERR_NONE != SyncReadRecordByIndex(rawRec))
        return 0;
    r = AGBufferReaderNew(buffer);
    if (NULL != r) {
        *deviceUserConfig = AGUserConfigNewAndReadData((AGReader*)r);
        AGBufferReaderFree(r);
        return rawRec.m_RecId;
    }
    else
        return 0;
}

/* ----------------------------------------------------------------------------
    static void writeDeviceUserConfig(BYTE userConfigDBHandle,
                                      AGUserConfig * deviceUserConfig,
                                      uint32 recID)

    Writes the device's userConfig record.  Explodes if any bad parameters
    are passed in.

*/
    static void writeDeviceUserConfig(BYTE userConfigDBHandle,
                                      AGUserConfig * deviceUserConfig,
                                      uint32 recID)
{
    CRawRecordInfo rawRec;
    AGBufferWriter * w = NULL;
    bzero(&rawRec, sizeof(CRawRecordInfo));
    w = AGBufferWriterNew(0);
    if (NULL != w) {
        long result;
        AGUserConfigWriteData(deviceUserConfig, (AGWriter*)w);
        fillInRecordStruct(&rawRec, userConfigDBHandle, recID,
            0, eRecord, 0, 0, AGBufferWriterGetBufferSize(w),
            AGBufferWriterGetBufferSize(w), AGBufferWriterGetBuffer(w));
        result = SyncWriteRec(rawRec);
        AGBufferWriterFree(w);
    }
}

/* ----------------------------------------------------------------------------
    static AGUserConfig * getAndSyncUserConfigs(uint32 * pilotID,
        char * userName)

    Returns a synchronized UserConfig.
    
    Reads desktop and agreed record, receives device's record,
    and syncs them all.  Also returns Pilot ID of the record on the device.

*/
static AGUserConfig * getAndSyncUserConfigs(uint32 * pilotID, char * userName)
{
    char userConfigFilename[MAX_PATH];
    char syncUserConfigFilename[MAX_PATH];
    BYTE userConfigDBHandle = 0;
    AGUserConfig * agreedUserConfig = NULL;
    AGUserConfig * deviceUserConfig = NULL;
    AGUserConfig * desktopUserConfig = NULL;
    AGUserConfig * newAgreed = NULL;

    getUserConfigFilename(userName,
        userConfigFilename,
        syncUserConfigFilename);

    /* Get desktop's userConfig. */
    desktopUserConfig = AGReadUserConfigFromDisk(userConfigFilename);

    /* Get last known agreed userConfig. */
    agreedUserConfig = AGReadUserConfigFromDisk(syncUserConfigFilename);

    /* Be sure an agreed userConfig exists. */
    if (NULL == agreedUserConfig)
        agreedUserConfig = AGUserConfigNew();

    if (NULL != agreedUserConfig) {

        /* Get device's record. */
        userConfigDBHandle = openUserConfigDatabase();
        if (0 != userConfigDBHandle) {

            /* Retrieve device's idea of current userConfig. */
            *pilotID = readDeviceUserConfig(userConfigDBHandle,
                &deviceUserConfig);

            /* Done with database for now, so close it. */
            SyncCloseDB(userConfigDBHandle);

            /* Sync the three records. */
            newAgreed = AGUserConfigSynchronize(agreedUserConfig,
                deviceUserConfig, desktopUserConfig);

            /* Clean up agreed userConfig. */
            AGUserConfigFree(agreedUserConfig);

        }

    }

    /* Clean up desktop userConfig. */
    if (NULL != desktopUserConfig)
        AGUserConfigFree(desktopUserConfig);

    /* Clean up device userConfig. */
    if (NULL != deviceUserConfig)
        AGUserConfigFree(deviceUserConfig);

    return newAgreed;
}

/* ----------------------------------------------------------------------------
    static void storeDeviceUserConfig(AGUserConfig * userConfig, uint32 id)

    Write out the current desktop userConfig to the device (the desktop is the
    product of:
        
          1. The device's userConfig, which was
          2. Synced with the desktop's userConfig using the
          3. Agreed userConfig as a reference, after which
          4. We connected to the server and updated the desktop userConfig
             accordingly.)
*/
static void storeDeviceUserConfig(AGUserConfig * userConfig, uint32 id)
{
    BYTE userConfigDBHandle = openUserConfigDatabase();
    if (0 != userConfigDBHandle) {
        writeDeviceUserConfig(userConfigDBHandle,
            userConfig, id);
        SyncCloseDB(userConfigDBHandle);
    }
}

/* ----------------------------------------------------------------------------

    Prepares ourselves and device for a set of MAL commands pertaining to a
    particular server.

*/
static void doStartServer(PalmSyncInfo * pInfo,
                          AGServerConfig *sc,
                          int32 *errCode)
{
    pInfo->currentServerConfig = sc;
    if(pInfo->commandProcessor) {
        AGCommandProcessorFree(pInfo->commandProcessor);
        pInfo->commandProcessor = NULL;
    }
    pInfo->commandProcessor = AGCommandProcessorNew(sc);
    pInfo->platform->performCommandOut = pInfo->commandProcessor;
    pInfo->platform->performCommandFunc = 
                    AGCommandProcessorGetPerformFunc(pInfo->commandProcessor);

    pInfo->commandProcessor->commands.out = pInfo;

    pInfo->commandProcessor->commands.performTaskFunc = cmdTASK;
    pInfo->commandProcessor->commands.performItemFunc = cmdITEM;
    pInfo->commandProcessor->commands.performDeleteDatabaseFunc = cmdDELETEDATABASE;
    pInfo->commandProcessor->commands.performOpenDatabaseFunc = cmdOPENDATABASE;
    pInfo->commandProcessor->commands.performCloseDatabaseFunc = cmdCLOSEDATABASE;
    pInfo->commandProcessor->commands.performClearModsFunc = cmdCLEARMODS;
    pInfo->commandProcessor->commands.performGoodbyeFunc = cmdGOODBYE;
    pInfo->commandProcessor->commands.performRecordFunc = cmdRECORD;
//  pInfo->commandProcessor->commands.performExpansionFunc = cmdEXPANSION;

    
}

/* ----------------------------------------------------------------------------
    static void doEndServer(PalmSyncInfo * pInfo,
                            int32 *errCode)

    Cleanup after sequence of MAL commands.

*/
static void doEndServer(PalmSyncInfo * pInfo,
                        int32 *errCode)
{
    pInfo->currentServerConfig = NULL;
    if(pInfo->commandProcessor) {
        AGCommandProcessorFree(pInfo->commandProcessor);
        pInfo->commandProcessor = NULL;
    }
}

static int32 cmdTASK(void *out, int32 *returnErrorCode, char *currentTask, 
                     AGBool bufferable)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_TASK_CMD()\n");

    if (NULL != currentTask) {
        AGSyncPalmStatusDialogSetTask(pInfo->statusDialog, currentTask);
    }
    return AGCLIENT_CONTINUE;
}

static int32 cmdITEM(void *out, int32 *returnErrorCode,
                                   int32 currentItemNumber,
                                   int32 totalItemCount,
                                   char *currentItem)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_ITEM_CMD()\n");

    AGSyncPalmStatusDialogSetItem(pInfo->statusDialog, currentItem);
    AGSyncPalmStatusDialogSetBar(pInfo->statusDialog, currentItemNumber,
        totalItemCount);
    return AGCLIENT_CONTINUE;
}

static int32 cmdDELETEDATABASE(void *out, int32 *returnErrorCode,
                                             char *dbname)
{
    debug_print("doCmdAG_DELETEDATABASE_CMD()\n");

    if (NULL != dbname) {
        debug_print("... trying to delete database ");
        debug_print(dbname);
        debug_print(" from device.\n");
        deleteDatabase(dbname);
    }
    return AGCLIENT_CONTINUE;
}

static int32 cmdOPENDATABASE(void *out, int32 *returnErrorCode,
                                           char *dbname)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_OPENDATABASE_CMD()\n");

    if (NULL != dbname) {
        long result;
        /* We assign a result code here only for debugging.  It's ok for an
        open to fail here. */
        result = openDatabase(pInfo, dbname, TRUE);
    }
    return AGCLIENT_CONTINUE;
}

static int32 cmdCLOSEDATABASE(void *out, int32 *returnErrorCode)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_CLOSEDATABASE_CMD()\n");
    closeDatabase(pInfo);
    return AGCLIENT_CONTINUE;
}

static int32 cmdCLEARMODS(void *out, int32 *returnErrorCode)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_CLEARMODS_CMD()\n");
    clearMods(pInfo->pilot_rHandle);
    return AGCLIENT_CONTINUE;
}

static int32 cmdGOODBYE(void *out, int32 *returnErrorCode,
                        AGSyncStatus syncStatus,
                        int32 errorCode,
                        char *errorMessage)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_GOODBYE_CMD()\n");

    if (NULL != errorMessage) {
        if (0 != errorCode) {

            int result;
            char url[MAX_PATH];

            url[0] = '\0';

            result = MessageBoxWithLink(pInfo->statusDialog->hDlg,
                errorMessage,
                pInfo->currentServerConfig->friendlyName,
                TRUE,
                url,
                kTimeoutValue);

            if (IDC_LINK == result) {

                if (NULL != url
                    && PathIsURL(url)
                    && !strncmp("http://", url, 7)) {

                    ShellExecute(pInfo->statusDialog->hDlg,
                        "open",
                        url,
                        NULL,
                        NULL,
                        1);

                }

            }

        }

    }

    return AGCLIENT_CONTINUE;
}

WORD getIndexFromPlatformData(uint8 *platformData)
{
    int16 recIndex;
    AGBufferReader reader;

    if (!platformData)
        return 0;

    AGBufferReaderInit(&reader, platformData);
    AGPalmReadRecordPlatformData((AGReader *)&reader, &recIndex);
    AGBufferReaderFinalize(&reader);
    return (WORD)recIndex;
}

static int32 cmdRECORD(void *out, int32 *returnErrorCode,
                                     int32 *newUID,
                                     int32 uid,
                                     AGRecordStatus mod,
                                     int32 recordDataLength,
                                     void *recordData,
                                     int32 platformDataLength,
                                     void *platformData)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    debug_print("doCmdAG_RECORD_CMD()\n");

    long result;

    if(mod == AG_RECORD_NEW_TEMPORARY_UID) {
        //PENDING(klobad) for palm, should this be zero or -1 or something else?
        uid = 0; // reset the uid so the WriteRecord creates a new one
    }

    if (AG_RECORD_DELETED == mod) {
        fillInRecordStruct(&pInfo->pilot_rRec, pInfo->pilot_rHandle,
            uid, 0, 0, 0, 0, 0, 0, 0);
        result = SyncDeleteRec(pInfo->pilot_rRec);
    } else if (recordDataLength <= 0x0000ffff) {
        if (pInfo->currentDBIsResourceType) {
            fillInRecordStruct(&pInfo->pilot_rRec, pInfo->pilot_rHandle,
                uid, getIndexFromPlatformData((uint8 *)platformData), 
                0, 0, 0,
                recordDataLength, pInfo->pilot_buffer_size, (BYTE*)recordData);
            result = SyncWriteResourceRec(pInfo->pilot_rRec);
        } else {
            fillInRecordStruct(&pInfo->pilot_rRec, pInfo->pilot_rHandle,
                uid, 0, 0, 0, 0,
                recordDataLength, pInfo->pilot_buffer_size, (BYTE*)recordData);
            result = SyncWriteRec(pInfo->pilot_rRec);
        }
        *newUID = pInfo->pilot_rRec.m_RecId;
    }
    return AGCLIENT_CONTINUE;
}

static int32 initAndOpenDatabase(void *_pInfo, AGDBConfig *db,
                                    int32 *errCode)
{
    long result;
    PalmSyncInfo * pInfo = (PalmSyncInfo *)_pInfo;

    if (NULL == db->dbname) {
        *errCode = AGCLIENT_OPEN_ERR;
        return AGCLIENT_ERR;
    }

    result = openDatabase(pInfo, db->dbname, FALSE);
    if (SYNCERR_NONE != result) {
        if (SYNCERR_FILE_NOT_FOUND == result)
            *errCode = AGCLIENT_OPEN_ERR;
        else
            *errCode = AGCLIENT_UNKNOWN_ERR;
        return AGCLIENT_ERR;
    }

    pInfo->pilot_RecIndex = 0;
    pInfo->record = AGRecordNew(0, AG_RECORD_UNMODIFIED, 0, 0, 0, 0);
    if (NULL == pInfo->record) {
        *errCode = AGCLIENT_OPEN_ERR;
        return AGCLIENT_ERR;
    }

    return AGCLIENT_IDLE;
}

static int32 leaveGetRecord(PalmSyncInfo * pInfo, int32 result)
{
    if (NULL != pInfo->record) {

        /* Set recordData to NULL so that AGRecordFree doesn't try
        to free it (we own that buffer and will release it when the
        program ends). */
        pInfo->record->recordData = NULL;

        AGRecordFree(pInfo->record);
        pInfo->record = NULL;
    }
    return result;
}

static int32 getRecordBase(PalmSyncInfo * pInfo, AGBool modonly, 
                           AGRecord ** record, int32 * errCode)
{
    int32 result;

    pInfo->pilot_rRec.m_FileHandle  = pInfo->pilot_rHandle;
    pInfo->pilot_rRec.m_RecIndex    = pInfo->pilot_RecIndex++;
    pInfo->pilot_rRec.m_pBytes      = (BYTE*)pInfo->pilot_buffer;
    pInfo->pilot_rRec.m_TotalBytes  = pInfo->pilot_buffer_size;
    result = (modonly) ?
        SyncReadNextModifiedRec(pInfo->pilot_rRec) :
        SyncReadRecordByIndex(pInfo->pilot_rRec);
    if (SYNCERR_NONE != result) {
        closeDatabase(pInfo);
        if (SYNCERR_NOT_FOUND == result) {
            debug_print("(successfully reached end of records ...)\n");
            return leaveGetRecord(pInfo, AGCLIENT_IDLE);
        }
        else {
            *errCode = AGCLIENT_UNKNOWN_ERR;
            return leaveGetRecord(pInfo, AGCLIENT_ERR);
        }
    }
    pInfo->record = AGRecordInit(pInfo->record, pInfo->pilot_rRec.m_RecId,
        AGPalmPilotAttribsToMALMod((uint8)pInfo->pilot_rRec.m_Attribs),
        pInfo->pilot_rRec.m_RecSize, pInfo->pilot_buffer, 0, NULL);
    *record = pInfo->record;
    return AGCLIENT_CONTINUE;
}

static int32 getNextModifiedRecord(void * out, AGRecord ** record, 
                                    int32 *errCode)
{
    debug_print("GetNextModifiedRecord()\n");

    return getRecordBase((PalmSyncInfo *)out, TRUE, record, errCode);
}

static int32 getNextRecord(void * out, AGRecord ** record, int32 *errCode)
{
    debug_print("GetNextRecord()\n");

    return getRecordBase((PalmSyncInfo *)out, FALSE, record, errCode);
}

typedef struct tagThreadStruct {
    AGBool * quit;
    AGClientProcessor *processor;
} threadStruct;

static DWORD WINAPI quickClientProcessorLoop(void *out)
{
    int32 cpResult;
    AGBool cancelled = FALSE;
    threadStruct * pInfo = (threadStruct *)out;
    AGClientProcessor *processor = pInfo->processor;

    cpResult = AGCLIENT_CONTINUE;
    while (AGCLIENT_CONTINUE == cpResult) {
        cpResult = AGClientProcessorProcess(processor);
        doWindowsMessagePump();

        if (AGCLIENT_CONTINUE == cpResult && *pInfo->quit) {
            cancelled = TRUE;
            processor->finished = TRUE;
            cpResult = AGCLIENT_IDLE;
        }

    }

    /* If we terminated via an error (i.e., we didn't end because of
    an END command), signal main thread that we're done. */
    processor->finished = TRUE;

    return cpResult;
}

static DWORD startClientProcessorThread(AGClientProcessor * processor,
                                        AGBool * quit,
                                        threadStruct * threadData)
{
    DWORD threadID = 0;

    processor->threadWriter = AGBufferWriterNew(1024);
    if(processor->threadWriter) {
        processor->readIndex = processor->writeIndex = 0;
        processor->mutex = CreateMutex(NULL, FALSE, NULL);
        if(processor->mutex) {
            threadData->processor = processor;
            threadData->quit = quit;
            processor->threadHandle =
                CreateThread(NULL,
                0,
                quickClientProcessorLoop,
                (LPVOID)threadData,
                0,
                &threadID);
        }
    }
    return threadID;
}

static AGBool doClientProcessorLoop(PalmSyncInfo * pInfo, AGNetCtx *ctx)
{
    int32 dummyError;
    int32 cpResult;
    int32 syncCount;
    int32 i, n;
    AGBool cancelled = FALSE;
    HINSTANCE hInst;
    char serverNameProto[MAX_PATH];
    char statusBarProto[MAX_PATH];
    threadStruct threadData;
    DWORD threadID;

    hInst = (HINSTANCE)GetWindowLong(pInfo->statusDialog->hDlg, GWL_HINSTANCE);
    LoadString(hInst,
        IDS_CONNECTING_TO_SERVER,
        serverNameProto,
        MAX_PATH);

    LoadString(hInst,
        IDS_STATUS_BAR_SYNCING,
        statusBarProto,
        MAX_PATH);

    n = AGUserConfigCount(pInfo->userConfig);

    for (i = 0; i < n; ++i) {

        char serverNameBuf[MAX_PATH];

        AGServerConfig * sc =
            AGUserConfigGetServerByIndex(pInfo->userConfig, i);

        if (cancelled)
            continue;

        if (NULL == sc)
            continue;

        if (sc->disabled)
            continue;

        if (NULL == sc->serverName || sc->serverPort <= 0)
            continue;

        sprintf(serverNameBuf,
            serverNameProto,
            sc->serverName,
            sc->serverPort);

        AGSyncPalmStatusDialogSetTask(pInfo->statusDialog,
            serverNameBuf);

        sprintf(serverNameBuf,
            statusBarProto,
            sc->serverName);

        AGSyncPalmStatusDialogSetTitle(pInfo->statusDialog,
            serverNameBuf);

/*
        if (pInfo->hasseclib) 
            (*pInfo->sectoggle) (ctx, sc->connectSecurely);
*/            
        syncCount = 0;
        doStartServer(pInfo, sc, &dummyError);
 
        do {
            AGCommandProcessorStart(pInfo->commandProcessor);

            pInfo->deviceInfo = AGDeviceInfoNew();
            /* If this fails, we're totally dead.  Exit with fail code. */
            if(!pInfo->deviceInfo)
                return FALSE;
                
            readDeviceInfo(pInfo);
            
            pInfo->clientProcessor =
                AGClientProcessorNew(pInfo->currentServerConfig,
                                     pInfo->deviceInfo, 
                                     pInfo->locationConfig,
                                     pInfo->platform, 
                                     TRUE,
                                     ctx);
                                        
            /* If this fails, we're totally dead.  Exit with fail code. */
            if (NULL == pInfo->clientProcessor) {
                AGDeviceInfoFree(pInfo->deviceInfo);
                return FALSE;
            }

            AGClientProcessorSetBufferServerCommands(pInfo->clientProcessor,
                FALSE);

            AGClientProcessorSync(pInfo->clientProcessor);

            if (pInfo->hasseclib) {
                (*pInfo->sec_presynchook) (ctx, 
                                           pInfo->currentServerConfig,
                                           pInfo->locationConfig,
                                           &pInfo->clientProcessor->syncProcessor,
                                           sc->connectSecurely);
            }

            /* Spin up thread that will queue commands being read
            from the network. */
            threadID = startClientProcessorThread(pInfo->clientProcessor,
                &pInfo->quit,
                &threadData);

            /* If thread creation was successful, start handling commands
            in the main thread.  We need to do it this weird way because
            the PalmOS can't talk to the device in anything but the original
            thread in which the HotSync is taking place. */
            if (0 != threadID) {

                /* Call the dispatcher. */
                AGClientProcessorBeginCommandDispatcher(
                    pInfo->clientProcessor,
                    doWindowsMessagePump,
                    &pInfo->quit);

                /* Once we've returned from dispatching commands, we wait
                to make sure the second thread finished, and then get its
                result. */
                WaitForSingleObject(pInfo->clientProcessor->threadHandle,
                    INFINITE);
                cpResult = AGCLIENT_ERR;
                GetExitCodeThread(pInfo->clientProcessor->threadHandle,
                    (LPDWORD)&cpResult);
            }
            else {

                /* For some reason, we weren't able to create the second
                thread.  So just go about syncing single-threadedly. */
                cpResult = AGCLIENT_CONTINUE;
                while (AGCLIENT_CONTINUE == cpResult) {
                    cpResult = AGClientProcessorProcess(pInfo->clientProcessor);
                    doWindowsMessagePump();

                    if (AGCLIENT_CONTINUE == cpResult && pInfo->quit) {
                        cancelled = TRUE;
                        cpResult = AGCLIENT_IDLE;
                    }

                }
            }

            if (pInfo->clientProcessor->mutex) {
                CloseHandle(pInfo->clientProcessor->mutex);
                pInfo->clientProcessor->mutex = NULL;
            }
            if(pInfo->clientProcessor->threadWriter) {
                AGBufferWriterFree(pInfo->clientProcessor->threadWriter);
                pInfo->clientProcessor->threadWriter = NULL;
            }

            if(cpResult == AGCLIENT_ERR) {
                if(AGGetMsg(pInfo->clientProcessor->errStringId))
                    MessageBoxTimeout(pInfo->statusDialog->hDlg,
                        AGGetMsg(pInfo->clientProcessor->errStringId),
                        pInfo->currentServerConfig->friendlyName,
                        MB_ICONEXCLAMATION | MB_OK,
                        kTimeoutValue);
                else
                    MessageBoxTimeout(pInfo->statusDialog->hDlg,
                        "Unknown Error during processing",
                        pInfo->currentServerConfig->friendlyName,
                        MB_ICONEXCLAMATION | MB_OK,
                        kTimeoutValue);
            }

            AGClientProcessorFree(pInfo->clientProcessor);
            AGDeviceInfoFree(pInfo->deviceInfo);

        } while (!cancelled
            && AGCommandProcessorShouldSyncAgain(pInfo->commandProcessor) 
            && syncCount++ < 2);
        doEndServer(pInfo, &dummyError);

        if (pInfo->hasseclib) {
            (*pInfo->sec_postsynchook) (ctx, 
                                        pInfo->currentServerConfig,
                                        pInfo->locationConfig,
                                        &pInfo->clientProcessor->syncProcessor,
                                        sc->connectSecurely);
        }

        /* It's possible to cancel in the middle of an open database,
        so if that happened, close it. */
        if (pInfo->pilot_rHandle)
            closeDatabase(pInfo);

    }

    if (0 == n) {

        char message[MAX_PATH];
        char caption[MAX_PATH];

        LoadString(hInst,
            IDS_NO_SERVERS_MESSAGE,
            message,
            MAX_PATH);

        LoadString(hInst,
            IDS_NO_SERVERS_CAPTION,
            caption,
            MAX_PATH);

        /* Add the message into the Palm log. */
        LogAddEntry(message, slWarning, FALSE);

        /* Show the message to the user. */
        MessageBoxTimeout(pInfo->statusDialog->hDlg,
            message,
            caption,
            MB_ICONEXCLAMATION | MB_OK,
            kTimeoutValue);

    }

    return TRUE; /* success */
}

/*---------------------------------------------------------------------------*/
static void checkForSecNet(PalmSyncInfo * pInfo) 
{
    
    HINSTANCE h = LoadLibrary("secnet.dll");
    pInfo->hasseclib = 0;

    if (h) {
        pInfo->secnetinit = 
            (netInitFunc)GetProcAddress(h, "NetInit");
        pInfo->secnetclose = 
            (netCloseFunc)GetProcAddress(h, "NetClose");
        pInfo->secctxsize = 
            (netCtxSizeFunc)GetProcAddress(h, "NetGetCtxSize");
        pInfo->sec_presynchook = 
            (netPreSyncHook)GetProcAddress(h, "NetPreSyncHook");
        pInfo->sec_postsynchook = 
            (netPostSyncHook)GetProcAddress(h, "NetPreSyncHook");

        if (pInfo->sec_postsynchook && pInfo->sec_presynchook && 
            pInfo->secnetclose &&  pInfo->secnetinit &&  pInfo->secctxsize)
            pInfo->hasseclib = 1;
    }
}

/* ----------------------------------------------------------------------------
*/
static AGDeviceEntry * makeDeviceEntry(char * username,
                                       AGDeviceType deviceType,
                                       char * basePath)
{
    char * temp;
    TCHAR userDir[MAX_PATH];

    strcpy(userDir, basePath);

    /* Build the path of the config file. */
    temp = (char*)AGSyncCommonGetStringConstant(agMALSubdirectoryName,
        FALSE);
    PathAppend(userDir, temp);
    free(temp);

    /* Make sure the path exists. */
    if (AG_NO_DEVICE_TYPE != deviceType)
        CreateDirectory(userDir, NULL);

    return AGMobileLinkDeviceEntryNew(strdup(username),
        deviceType,
        strdup(userDir));
}

/* ----------------------------------------------------------------------------
*/
static void setCurrentUser(char * userName)
{
    AGDeviceEntry * dle;
    AGDesktopInfoPalm * desktopInfo;
    TCHAR tempDir[MAX_PATH];
    int   intBufSize;

    desktopInfo = AGDesktopInfoPalmNew();
    if (NULL == desktopInfo)
        return;

    intBufSize = MAX_PATH;
    AGDesktopInfoPalmGetUserDirectory(desktopInfo,
        userName,
        tempDir,
        &intBufSize);
    if (0 == strlen(tempDir))
        return;
    
    dle = makeDeviceEntry(strdup(userName),
        AG_PALM_DEVICE_TYPE,
        strdup(tempDir));

    AGMobileLinkSetCurrentDevice(dle);

    AGMobileLinkDeviceEntryFree(dle);

    AGDesktopInfoPalmFree(desktopInfo);

}

/* ----------------------------------------------------------------------------
*/
BOOL AGSyncToServer(HINSTANCE h, char * userName) 
{
    _CrtSetDbgFlag( _CRTDBG_ALLOC_MEM_DF | _CRTDBG_CHECK_ALWAYS_DF |
                    _CRTDBG_CHECK_CRT_DF |
                    _CRTDBG_DELAY_FREE_MEM_DF |
                    _CRTDBG_LEAK_CHECK_DF);
    debug_print("SyncToServer()\n");

    BOOL result;
    PalmSyncInfo * pInfo;
    uint32 pilotID;
    AGNetCtx *ctx;
    AGArray * graphics = NULL;
    AGBool bitmapFormat;

    result = FALSE;

    __try {

        pInfo = syncInfoNew();
        if (NULL == pInfo)
            __leave;

        checkForSecNet(pInfo);

        if (pInfo->hasseclib) {
            ctx = (AGNetCtx *)malloc((*pInfo->secctxsize)());
            (*pInfo->secnetinit)(ctx);
        } else {
            ctx = (AGNetCtx *)malloc(sizeof(AGNetCtx));
            AGNetInit(ctx);
        }

        setCurrentUser(userName);

        if (readLocationConfig(pInfo))
            __leave;

        if (setupPlatformCalls(pInfo))
            __leave;

        graphics = AGSyncCommonLoadGraphics(h,
            IDI_GENERIC_PROGRESS_ICON0,
            &bitmapFormat);

        if (startStatusDialog(pInfo, h, graphics, bitmapFormat))
            __leave;

        setStatusDialogToConnecting(pInfo, h);

        pInfo->userConfig = getAndSyncUserConfigs(&pilotID, userName);

        result = doClientProcessorLoop(pInfo, ctx);

        storeDeviceUserConfig(pInfo->userConfig, pilotID);

        writeUserConfigs(pInfo->userConfig, userName);
    }

    __finally {

        if (pInfo->hasseclib) {
            (*pInfo->secnetclose)(ctx);
        } else { 
            AGNetClose(ctx);
        }

        if (NULL != graphics)
            AGSyncCommonFreeGraphics(graphics, bitmapFormat);

        syncInfoFree(pInfo);
        free(ctx);

    }

    return result;
}
