/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <AGSyncCommon.h>
#ifdef _WIN32 /**** Entire file is NOOP if not on Windows. *****/

#include <stdlib.h>
#include <AGUtil.h>

const TCHAR * HKEY_MAL_CURRENT_USER_ROOT =
    TEXT("Software\\Mobile Application Link");

TCHAR * stringConstants[10] = {
    TEXT("RAPI Server DLL Name"),
    TEXT("RAPI Server Name"),
    TEXT("Location Configuration"),
    TEXT("Mobile Link Location"),
    TEXT("Preferences Filename"),
    TEXT("Synchronized Preferences Filename"),
    TEXT("MAL Subdirectory Name"),
    TEXT("Progress Bitmap Filename"),
    TEXT("Extension DLL Path"),
    TEXT("Extension DLL Name"),
};

TCHAR * integerConstants[1] = {
    TEXT("Number of Progress Bitmap Frames")
};

ExportFunc TCHAR * AGSyncCommonMakeUniqueServerString(char * serverName,
                                                      int16 serverPort)
{
    TCHAR * szResult;
    TCHAR * szColon = TEXT(":");
    TCHAR szServerPortString[6];
    TCHAR * szServerName;

#ifdef UNICODE
    _itow(serverPort, szServerPortString, 10);
    szServerName = (TCHAR*)malloc(sizeof(TCHAR) *
        (strlen(serverName) + 1));
    if (NULL == szServerName)
        return NULL;
    mbstowcs(szServerName, serverName, -1);
#else
    _itoa(serverPort, szServerPortString, 10);
    szServerName = serverName;
#endif

    szResult = (TCHAR *)malloc(sizeof(TCHAR) *
        (strlen(serverName)
        + lstrlen(szColon)
        + lstrlen(szServerPortString)
        + 1));
    if (NULL == szResult)
        return NULL;

    lstrcpy(szResult, szServerName);
    lstrcat(szResult, szColon);
    lstrcat(szResult, szServerPortString);

#ifdef UNICODE
    free(szServerName);
#endif

    return szResult;
}

/* ----------------------------------------------------------------------------
    void * AGSyncCommonGetStringConstant(agStringConstants strNum,
                                         AGBool forceWide)

    Given an enumerated constant specifying the requested string, return
    that string from the registry.  If requested, return in Unicode no matter
    whether UNICODE is defined.

*/
ExportFunc void * AGSyncCommonGetStringConstant(agStringConstants strNum,
                                                AGBool forceWide)
{
    HKEY rootKey = NULL;
    HRESULT hr;
    void * result = NULL;

    hr = RegOpenKeyEx(HKEY_CURRENT_USER, HKEY_MAL_CURRENT_USER_ROOT,
        0, KEY_EXECUTE, &rootKey);

    if (ERROR_SUCCESS == hr) {
    
        DWORD bufsize = 0;
        DWORD valueType = 0;

        hr = RegQueryValueEx(rootKey, stringConstants[strNum], 0,
            &valueType, NULL, &bufsize);

        if (ERROR_SUCCESS == hr) {

            if (forceWide)
                bufsize *=2;

            /* In bytes, not Unicode characters! */
            if (bufsize > 0)
                result = malloc(bufsize);

            if (NULL != result) {

                bzero(result, bufsize);

                if (forceWide) {

                    char * tempstr;

                    bufsize /= 2;

                    tempstr = (char *)malloc(bufsize);

                    if (NULL != tempstr) {

                        bzero(tempstr, bufsize);

                        RegQueryValueEx(rootKey,
                            stringConstants[strNum],
                            0,
                            &valueType,
                            (BYTE*)tempstr,
                            &bufsize);

                        mbstowcs((WCHAR*)result, tempstr, -1);

                        free(tempstr);

                    }

                } else {

                    RegQueryValueEx(rootKey,
                        stringConstants[strNum],
                        0,
                        &valueType,
                        (BYTE*)result,
                        &bufsize);

                }

            }

        }

        RegCloseKey(rootKey);

    }
    
    return result;

}

ExportFunc DWORD AGSyncCommonGetIntegerConstant(agIntegerConstants intNum)
{
    HKEY rootKey = NULL;
    HRESULT hr;
    DWORD result = 0;

    hr = RegOpenKeyEx(HKEY_CURRENT_USER, HKEY_MAL_CURRENT_USER_ROOT,
        0, KEY_EXECUTE, &rootKey);

    if (ERROR_SUCCESS == hr) {
    
        DWORD bufsize = sizeof(DWORD);
        DWORD valueType = 0;

        RegQueryValueEx(rootKey,
            integerConstants[intNum],
            0,
            &valueType,
            (uint8*)&result,
            &bufsize);
            
        RegCloseKey(rootKey);

    }
    
    return result;

}

/* ----------------------------------------------------------------------------
    static int32 writeDiskFunc(void * info, void * data, int32 len)

    Wrapper for AGWriter write function. Writes to file on disk.

*/
static int32 writeDiskFunc(void * info, void * data, int32 len)
{
    DWORD bytes = 0;
    WriteFile((HANDLE)info, data, len, &bytes, NULL);
    return bytes;
}

/* ----------------------------------------------------------------------------
    static int32 readDiskFunc(void * info, void * data, int32 len)

    Wrapper for AGReader read function. Reads from file on disk.

*/
static int32 readDiskFunc(void * info, void * data, int32 len)
{
    DWORD bytes = 0;
    ReadFile((HANDLE)info, data, len, &bytes, NULL);
    return bytes;
}

static AGUserConfig * createAndRead(TCHAR * filename,
                                    HANDLE * fileHandle,
                                    AGBool writeAccess)
{
    AGUserConfig * userConfig = NULL;

    *fileHandle =
        CreateFile(filename,
        GENERIC_READ | ((writeAccess) ? GENERIC_WRITE : 0),
        0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
    if (INVALID_HANDLE_VALUE != *fileHandle) {
        AGReader * reader = AGReaderNew(*fileHandle, readDiskFunc);
        userConfig = AGUserConfigNewAndReadData(reader);
        AGReaderFree(reader);
    }
    else {
        if (writeAccess)
            *fileHandle = CreateFile(filename, GENERIC_READ | GENERIC_WRITE,
            0, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0);
    }

    return userConfig;
}

/* ----------------------------------------------------------------------------
    AGUserConfig * AGReadUserConfigFromDiskAtomically(TCHAR * filename,
                                                      HANDLE * fileHandle)

    Reads userConfig from disk and returns a handle to a still-open,
    exclusive-access file.
*/
ExportFunc AGUserConfig * AGReadUserConfigFromDiskAtomically(TCHAR * filename,
                                                  HANDLE * fileHandle)
{
    return createAndRead(filename, fileHandle, TRUE);
}

#define FILE_LOCK_MUTEX L"MAL File Lock Mutex"

/* ----------------------------------------------------------------------------
    AGUserConfig * AGReadUserConfigFromDisk(TCHAR * filename)

    Reads userConfig from disk.
*/
ExportFunc AGUserConfig * AGReadUserConfigFromDisk(TCHAR * filename)
{
    HANDLE f;
    AGUserConfig * userConfig;
#ifdef _WIN32_WCE
    HANDLE hFileLock;
#endif /* _WIN32_WCE */

#ifdef _WIN32_WCE
    hFileLock = CreateMutex(NULL, FALSE, FILE_LOCK_MUTEX);
    WaitForSingleObject(hFileLock, 10000);
#endif /* _WIN32_WCE */

    userConfig = createAndRead(filename, &f, FALSE);

    if (INVALID_HANDLE_VALUE != f)
        CloseHandle(f);

#ifdef _WIN32_WCE
    ReleaseMutex(hFileLock);
#endif /* _WIN32_WCE */

    return userConfig;
}

/* ----------------------------------------------------------------------------
    void AGWriteUserConfigToDiskAtomically(AGUserConfig * userConfig,
                                           HANDLE fileHandle)

    Writes userConfig to the supplied open file, and then closes the file.

*/
ExportFunc void AGWriteUserConfigToDiskAtomically(AGUserConfig * userConfig,
                                       HANDLE fileHandle)

{
    if (INVALID_HANDLE_VALUE != fileHandle) {
        AGWriter * writer = AGWriterNew(fileHandle, writeDiskFunc);
        if (NULL != userConfig) {
            SetFilePointer(fileHandle, 0, NULL, FILE_BEGIN);
            AGUserConfigWriteData(userConfig, writer);
        }
        AGWriterFree(writer);
        SetEndOfFile(fileHandle);
        CloseHandle(fileHandle);
    }
}

/* ----------------------------------------------------------------------------
    void AGWriteUserConfigToDisk(TCHAR * filename, AGUserConfig * userConfig)

    Writes userConfig to disk.

*/
ExportFunc void AGWriteUserConfigToDisk(TCHAR * filename, AGUserConfig * userConfig)
{
    HANDLE f;
#ifdef _WIN32_WCE
    HANDLE hFileLock;
#endif /* _WIN32_WCE */

#ifdef _WIN32_WCE
    hFileLock = CreateMutex(NULL, FALSE, FILE_LOCK_MUTEX);
    WaitForSingleObject(hFileLock, 10000);
#endif /* _WIN32_WCE */

    f = CreateFile(filename, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS,
        FILE_ATTRIBUTE_NORMAL, 0);
    AGWriteUserConfigToDiskAtomically(userConfig, f);

#ifdef _WIN32_WCE
    ReleaseMutex(hFileLock);
#endif /* _WIN32_WCE */

}

static AGLocationConfig * lcCreateAndRead(TCHAR * filename,
                                          HANDLE * fileHandle,
                                          AGBool writeAccess)
{
    AGLocationConfig * locationConfig = NULL;

    *fileHandle =
        CreateFile(filename,
        GENERIC_READ | ((writeAccess) ? GENERIC_WRITE : 0), 0, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
    if (INVALID_HANDLE_VALUE != *fileHandle) {
        AGReader * reader = AGReaderNew(*fileHandle, readDiskFunc);
        locationConfig = AGLocationConfigNewAndReadData(reader);
        AGReaderFree(reader);
    }
    else {
        if (writeAccess)
            *fileHandle = CreateFile(filename, GENERIC_READ | GENERIC_WRITE,
            0, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0);
    }

    return locationConfig;
}

/* ----------------------------------------------------------------------------
    AGLocationConfig * AGReadLocationConfigFromDiskAtomically(TCHAR * filename,
                                                      HANDLE * fileHandle)

    Reads locationConfig from disk and returns a handle to a still-open,
    exclusive-access file.
*/
ExportFunc AGLocationConfig * AGReadLocationConfigFromDiskAtomically(TCHAR * filename,
                                                          HANDLE * fileHandle)
{
    return lcCreateAndRead(filename, fileHandle, TRUE);
}

/* ----------------------------------------------------------------------------
    AGLocationConfig * AGReadLocationConfigFromDisk(TCHAR * filename)

    Reads locationConfig from disk.
*/
ExportFunc AGLocationConfig * AGReadLocationConfigFromDisk(TCHAR * filename)
{
    HANDLE f;
    AGLocationConfig * locationConfig;

    locationConfig = lcCreateAndRead(filename, &f, FALSE);

    if (INVALID_HANDLE_VALUE != f)
        CloseHandle(f);

    return locationConfig;
}

/* ----------------------------------------------------------------------------
    void AGWriteLocationConfigToDiskAtomically(TCHAR * filename,
                                           AGLocationConfig * locationConfig,
                                           HANDLE fileHandle)

    Writes locationConfig to the supplied open file, and then closes the file.

*/
ExportFunc void AGWriteLocationConfigToDiskAtomically(TCHAR * filename,
                                           AGLocationConfig * locationConfig,
                                           HANDLE fileHandle)

{
    if (INVALID_HANDLE_VALUE != fileHandle) {
        AGWriter * writer = AGWriterNew(fileHandle, writeDiskFunc);
        if (NULL != locationConfig) {
            SetFilePointer(fileHandle, 0, NULL, FILE_BEGIN);
            AGLocationConfigWriteData(locationConfig, writer);
        }
        AGWriterFree(writer);
        SetEndOfFile(fileHandle);
        CloseHandle(fileHandle);
    }
}

/* ----------------------------------------------------------------------------
    void AGWriteLocationConfigToDisk(TCHAR * filename,
        AGLocationConfig * locationConfig)

    Writes locationConfig to disk.

*/
ExportFunc void AGWriteLocationConfigToDisk(TCHAR * filename,
                                 AGLocationConfig * locationConfig)
{
    HANDLE f;

    f = CreateFile(filename, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS,
        FILE_ATTRIBUTE_NORMAL, 0);
    AGWriteLocationConfigToDiskAtomically(filename, locationConfig, f);
}

/* ----------------------------------------------------------------------------
*/
ExportFunc AGArray * AGSyncCommonLoadGraphics(HINSTANCE h,
                                              int genericIDBase,
                                              AGBool * filesFound)
{
    AGArray * result;
    AGBool localFilesFound = FALSE;

    result = AGArrayNew(AGUnownedPointerElements, 0);

    if (NULL != result) {

        LPTSTR prototype = NULL;
        int n;
        n = AGSyncCommonGetIntegerConstant(agProgressBitmapFrameCount);

        if (n > 0) {

            prototype = (LPTSTR)
                AGSyncCommonGetStringConstant(agProgressBitmapFilename,
                FALSE);

        }

        if (NULL != prototype) {

            int i;

            for (i = 0; i < n; i++) {

#ifndef _WIN32_WCE
                TCHAR filename[MAX_PATH];

                sprintf(filename, prototype, i);

                AGArrayAppend(result,
                    LoadImage(NULL,
                        filename,
                        IMAGE_BITMAP,
                        0,
                        0,
                        LR_LOADFROMFILE |
                        LR_LOADMAP3DCOLORS |
                        LR_LOADTRANSPARENT));
#endif
            }

            free(prototype);

            localFilesFound = TRUE;

        } else {

            int i;
            int n = 4;

            for (i = 0; i < n; i++) {

                /* LoadIcon is supposedly outdated, but I couldn't get
                LoadImage to load with the right transparencies, so I
                went back to the old way. */
                AGArrayAppend(result,
                    LoadIcon(h, MAKEINTRESOURCE(genericIDBase + i)));

            }
        }

    }

    if (NULL != filesFound)
        *filesFound = localFilesFound;

    return result;
}

/* ----------------------------------------------------------------------------
*/
ExportFunc void AGSyncCommonFreeGraphics(AGArray * array,
                                         AGBool bitmaps)
{
    while (AGArrayCount(array)) {

        if (bitmaps)
            DeleteObject((HBITMAP)AGArrayElementAt(array, 0));
        else
            DestroyIcon((HICON)AGArrayElementAt(array, 0));

        AGArrayRemoveAt(array, 0);

    }

}

#endif
