/*
 *  Copyright (C) 2006  MakeHuman Project
 *
 *  This program is free software; you  can  redistribute  it  and/or
 *  modify  it  under  the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either  version  2  of
 *  the License, or (at your option) any later version.
 *
 *  This  program  is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the  implied  warranty  of
 *  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foun-
 *  dation, Inc., 59 Temple Place, Suite 330, Boston,  MA  02111-1307
 *  USA
 *  
 *  File   : util.cpp
 *  Project: MakeHuman <info@makehuman.org>, http://www.makehuman.org/
 *  App    : makehuman
 *
 *  For individual developers look into the AUTHORS file.
 *   
 */

#include "util.h"
#include <animorph/DirectoryList.h>
#include <animorph/RIBExporter.h>
#include <mhgui/Console.h>
#include "ConsoleCommands.h"
#include "Global.h"

#ifdef _WIN32
    #include <windows.h>
    #include <winbase.h>
#elif defined(__APPLE__) && defined(__MACH__)
    #include "MACFileUtils.h"
#endif

#ifndef _WIN32
    // Not available on WIN32 'cos still not tested there!
    #include "FileTools.h"
#endif

#ifdef _WIN32
    static const string getEXEDir();
#endif



#if defined(__APPLE__) && defined(__MACH__)
/* ========================================================================== */
/**
 */
/* ========================================================================== */
class AQSISEnvSetter
{
public:
    AQSISEnvSetter(const string& inAQSISBasePath)
    : mBasePath(inAQSISBasePath)
    {
        setEnvs();
    }

private:
    void setEnvs();

private:
    string mBasePath;

/*
 export AQSIS_BASE_PATH=/Applications/Graphics/MacAqsis.0.9.1 
 export AQSIS_SHADERS_PATH=$AQSIS_BASE_PATH/aqsis/share/shaders 
 export AQSIS_DISPLAYS_PATH=$AQSIS_BASE_PATH/aqsis/share/displays 
 export AQSIS_ARCHIVES_PATH=$AQSIS_BASE_PATH/aqsis/share/archives 
 export AQSIS_TEXTURES_PATH=$AQSIS_BASE_PATH/aqsis/share/textures 
 export AQSIS_DSO_LIBS=$AQSIS_BASE_PATH/aqsis/share/dso 
 export AQSIS_CONFIG_PATH=$AQSIS_BASE_PATH/etc 
 export AQSIS_CONFIG=$AQSIS_CONFIG_PATH/.aqsisrc 
 export PATH=$PATH:$AQSIS_BASE_PATH/bin
*/
}; // class AQSISEnvSetter

/* @rerurn true on success, false otherwise */
static bool setEnv(const char *inEnvName, const string& inValue)
{
    return (::setenv(inEnvName, inValue.c_str(), 1) == 0);
}


/* ========================================================================== */
/**
 */
/* ========================================================================== */
void AQSISEnvSetter::setEnvs()
{
    setEnv("AQSIS_BASE_PATH",     mBasePath);
    setEnv("AQSIS_SHADERS_PATH",  mBasePath + "/share/aqsis/shaders");
    setEnv("AQSIS_DISPLAYS_PATH", mBasePath + "/share/aqsis/displays");
    setEnv("AQSIS_ARCHIVES_PATH", mBasePath + "/share/aqsis/archives");
    setEnv("AQSIS_TEXTURES_PATH", mBasePath + "/share/aqsis/textures");
    setEnv("AQSIS_DSO_LIBS",      mBasePath + "/share/aqsis/dso");
    setEnv("AQSIS_CONFIG_PATH",   mBasePath + "/etc");
    setEnv("AQSIS_CONFIG",        mBasePath + "/etc/.aqsisrc");
}

    const string aqsis_base_path         ("/Applications/Graphics/MacAqsis.0.9.1/");
    const string aqsis_bin_path          (aqsis_base_path + "bin/");
    const string pixie_base_path         ("/Applications/Graphics/Pixie/");
    const string pixie_bin_path          (pixie_base_path + "bin/");
#else
    const string aqsis_base_path         ("");
    const string aqsis_bin_path          ("");
    const string pixie_base_path         ("");
    const string pixie_bin_path          ("");
#endif // defined(__APPLE__) && defined(__MACH__)


const static string aqsis_version         (aqsis_bin_path + "aqsis -version");
const static string pixie_version         (pixie_bin_path + "rndr -h");
const static string aqsis_compiler        (aqsis_bin_path + "teqser -swrap=periodic -twrap=periodic -filter=gaussian -swidth=1 -twidth=1");
const static string pixie_compiler        (pixie_bin_path + "texmake");
const static string aqsis_shader_compiler (aqsis_bin_path + "aqsl -o");
const static string pixie_shader_compiler (pixie_bin_path + "sdrc -o");
const static string aqsis_render          (aqsis_bin_path + "aqsis");
const static string pixie_render          (pixie_bin_path + "rndr");


const static string render_template  ("out_render.rib.template");
const static string preview_template ("out_render_preview.rib.template");
const static string toon_template    ("out_render_toon.rib.template");
const static string render_rib       ("out_render.rib");
const static string preview_rib      ("out_render_preview.rib");
const static string toon_rib         ("out_render_toon.rib");

const static string aqsis_textures ("aqsis");
const static string pixie_textures ("pixie");

// TODO: Need to test the "/" vs. "\" problematic! Targets...

using namespace std;
//using namespace mhgui;

static const string kEmptyString;

const string searchTextureFile (const string& texture_file)
{
  return searchFile (getTexturesAlternatives(texture_file));
}

const string searchShaderFile (const string& shader_file)
{
  return searchFile (getShadersAlternatives(shader_file));
}

const string searchBackgroundFile   (const string& data_file)
{
  return searchFile (getBackgroundsAlternatives (data_file));
}

const string searchDataFile   (const string& data_file)
{
  string s(searchFile (getDataAlternatives (data_file)));
  if (s.empty())
  {
     fprintf(stderr, "*** WARNING in searchDataFile() : attempt to load file '%s'" \
                     "' but this could not found in data/!\n", data_file.c_str());
  }
  return s;
}

const string searchPixmapFile (const string& pixmap_file)
{
  string s(searchFile (getPixmapsAlternatives (pixmap_file)));
  if (s.empty())
  {
     fprintf(stderr, "*** WARNING in searchPixmapFile() : attempt to load file '%s'" \
                     "' but this could not found in pixmaps/!\n", pixmap_file.c_str());
  }
  return s;
}

const string searchDataDir   (const string& data_dir)
{
  return searchDir (getDataAlternatives (data_dir));
}

const string searchPixmapDir (const string& pixmap_dir)
{
  return searchDir (getPixmapsAlternatives (pixmap_dir));
}

const StringVector getTexturesAlternatives (const string& texture)
{
  StringVector name_vector;
  
  name_vector.push_back (getTexturesPath() + texture);
  
  return name_vector;
}

const StringVector getShadersAlternatives (const string& shader)
{
  StringVector name_vector;

  name_vector.push_back (getShadersPath() + shader);

  return name_vector;
}

const StringVector getPixmapsAlternatives (const string& pixmap)
{
  StringVector name_vector;

#if defined(__APPLE__) && defined(__MACH__)
  name_vector.push_back ("../Resources/pixmaps/" + pixmap);
#else
  name_vector.push_back ("pixmaps/" + pixmap);
  name_vector.push_back ("../pixmaps/" + pixmap);
#endif

#ifdef _WIN32
  name_vector.push_back (getEXEDir ());
#else             
#ifdef PACKAGE_PIXMAPS_DIR
  name_vector.push_back (string (PACKAGE_PIXMAPS_DIR) + "/" + pixmap);
#endif
#endif
  return name_vector;
}

const StringVector getDataAlternatives (const string& data)
{
  StringVector name_vector;

#if defined(__APPLE__) && defined(__MACH__)
  name_vector.push_back ("../Resources/data/" + data);
#else
  name_vector.push_back ("data/" + data);
  name_vector.push_back ("../data/" + data);
#endif

#ifdef _WIN32
  name_vector.push_back (getEXEDir ());
#else
#ifdef PACKAGE_PIXMAPS_DIR
  name_vector.push_back (string (PACKAGE_DATA_DIR) + "/" + data);
#endif
#endif
  return name_vector;
}

const StringVector getBackgroundsAlternatives (const string& data)
{
  StringVector name_vector;

#if defined(__APPLE__) && defined(__MACH__)
  name_vector.push_back ("../Resources/backgrounds/" + data);
#else
  name_vector.push_back ("backgrounds/" + data);
  name_vector.push_back ("../backgrounds/" + data);
#endif

#ifdef _WIN32
  name_vector.push_back (getEXEDir ());
#else
#ifdef PACKAGE_BACKGROUNDS_DIR
  name_vector.push_back (string (PACKAGE_BACKGROUNDS_DIR) + "/" + data);
#endif
#endif
  return name_vector;
}

const string searchFile (const StringVector &name_vector)
{
  struct stat buf;

  for (unsigned int i = 0; i < name_vector.size (); i++)
  {
    const string &try_name = name_vector[i];
    int result;

    result = stat (try_name.c_str (), &buf);

    if (result != 0)
    {
      continue;
    }
    else
    {
      if (buf.st_mode & S_IFREG)
        return try_name;
    }
  }

  return kEmptyString;
}

const string searchDir (const StringVector &name_vector)
{
  struct stat buf;

  for (unsigned int i = 0; i < name_vector.size (); i++)
  {
    const string &try_name = name_vector[i];
    int result;

    result = stat (try_name.c_str (), &buf);

    if (result != 0)
    {
      continue;
    }
    else
    {
      if (buf.st_mode & S_IFDIR)
        return try_name;
    }
  }
  return kEmptyString;
}

#ifdef _WIN32
static const string getEXEDir ()
{
  char path[PATH_MAX];
  char *pos;

  GetModuleFileName (NULL, path, sizeof (path));
  pos = strrchr (path, '\\');

  if (pos)
    *pos = '\0';

  return path;
}
#endif

const string getHomeDir ()
{
#if defined(__APPLE__) && defined(__MACH__)
    return getCurrentUserHomeFolderPath() + "/"; // call of MACFileUtils
#endif

#ifndef _WIN32  
  uid_t uid;
  struct passwd *pass;

  uid = getuid();
  pass = getpwuid (uid);

  return string(string(pass->pw_dir)+"/");
#else // _WIN32

  HINSTANCE hinstLib;
  MYPROC ProcAdd;
  BOOL fFreeResult, fRunTimeLinkSuccess = FALSE;
  TCHAR szPath[MAX_PATH];

  /* Get a handle to the DLL module. */
  hinstLib = LoadLibrary("shfolder");

  /* If the handle is valid, try to get the function address. */
  if (hinstLib != NULL)
  {
    ProcAdd = (MYPROC) GetProcAddress(hinstLib, "SHGetFolderPathA");


    /* If the function address is valid, call the function. */
    if (NULL != ProcAdd)
    {
      fRunTimeLinkSuccess = TRUE;
      ProcAdd (NULL,
               CSIDL_PERSONAL, /* Test CSIDL_FLAG_CREATE !! */
               NULL,
               0,
               szPath);
    }

    /* Free the DLL module. */
    fFreeResult = FreeLibrary(hinstLib);
  }

  /* If unable to call the DLL function, use an alternative. */
  if (! fRunTimeLinkSuccess)
  {
    /* later use getWindir?? or something else as fallback */
    return string ("c:\\");
  }

  return string (string(szPath) + "\\");
#endif // _WIN32
}

/* Return the Users Working directory where the load and save dialog should be
   point to. On Linux and Window this will be the same as getHomeDir() but
   on a OS X Box this will point to the Users Desktop because it is not common
   to store files which are supposed for further processing on the users home 
   dir. */
const string getUserWorkDir()
{
#if defined(__APPLE__) && defined(__MACH__)
    return getUsersDocumentsFolderPath() + "/";
#else
    return getHomeDir();
#endif
}


const string getMyObjPath()
{
//#if ((defined(__APPLE__) && defined(__MACH__)) || defined(_WIN32))
    return string(getUserWorkDir() + "makehuman/myobjs/");
/*#else
    // To Do : Really special path handling for Linux required?!
    return string(searchDataDir("myobjs") + "/");
#endif*/
}

const string getMyPosesPath()
{
//#if ((defined(__APPLE__) && defined(__MACH__)) || defined(_WIN32))
    return string(getUserWorkDir() + "makehuman/myposes/ok/");
/*#else
    // To Do : Really special path handling for Linux required?!
    return string(searchDataDir("myposes") + "/ok/");
#endif*/
}

const string getMyPosesBasePath()
{
//#if ((defined(__APPLE__) && defined(__MACH__)) || defined(_WIN32))
    return string(getUserWorkDir() + "makehuman/myposes/");
/*#else
    // To Do : Really special path handling for Linux required?!
    return string(searchDataDir("myposes") + "/");
#endif*/
}

const string getMyBodysettingsPath()
{
//#if ((defined(__APPLE__) && defined(__MACH__)) || defined(_WIN32))
    return string(getUserWorkDir() + "makehuman/mybs/ok/");
/*#else 
    // To Do : Really special path handling for Linux required?!
    return string(searchDataDir("mybs") + "/ok/");
#endif*/
}

const string getMyBodysettingsBasePath()
{
//#if ((defined(__APPLE__) && defined(__MACH__)) || defined(_WIN32))
    return string(getUserWorkDir() + "makehuman/mybs/");
/*#else
    // To Do : Really special path handling for Linux required?!
    return string(searchDataDir("mybs") + "/");
#endif*/
}

const string getRenderingPath()
{
  return string(getUserWorkDir() + "makehuman/rendering/");
}

const string getTexturesPath()
{
  return string(getUserWorkDir() + "makehuman/textures/");
}

const string getShadersPath()
{
  return string(getUserWorkDir() + "makehuman/shaders/");
}

static void createDirWhenNotExists(const string& inPath)
{
#if defined(_WIN32)
  DIR *dirHd;

  // Does the given dir still exists?
  if((dirHd = opendir(inPath.c_str())) != NULL)
  {
      // Yes, then release the handle again and return
      ::closedir(dirHd);
      return;
  }
  ::mkdir(inPath.c_str());   // create it otherwise (does not exists)!

// On Linux and OS X specific code benefit from the FileTools
#else
   FileTools::makeDirHier(inPath);
#endif
}

void createWorkingDirs()
{
  createDirWhenNotExists(getUserWorkDir() + "makehuman/");
  createDirWhenNotExists(getMyObjPath());
  createDirWhenNotExists(getMyPosesBasePath());
  createDirWhenNotExists(getMyPosesPath());
  createDirWhenNotExists(getMyBodysettingsBasePath());
  createDirWhenNotExists(getMyBodysettingsPath());
  createDirWhenNotExists(getRenderingPath());
  createDirWhenNotExists(getShadersPath());
  createDirWhenNotExists(getTexturesPath());
  createDirWhenNotExists(getTexturesPath() + "aqsis");
  createDirWhenNotExists(getTexturesPath() + "pixie");
}

bool callRender(const string& cmd, Window &mainWindow)
{
  if(std::system(cmd.c_str()) != 0)
  {
    cout << "Error: " << cmd << endl;
    mainWindow.getConsole()->open();
    mainWindow.getConsole()->printMessage(kConsoleMessage_RenderError);
    return false;
  }
  else
  {
    return true;
  }
}

bool compileShaders(const string& shader_compiler, Window &mainWindow)
{
  vector<string> shaders;

  DirectoryList dir_list;
  string shaders_root_path (searchDataDir("rib_data") + "/shaders_data/");
  dir_list.setRootPath (shaders_root_path);
  dir_list.setRecursive (0);
  dir_list.setFileFilter (".sl");

  const StringList &str_list(dir_list.getDirectoryList ());

  bool must_compile = false;
  string compiled_extension = (shader_compiler == pixie_shader_compiler ? ".sdr" : ".slx");
  string shaders_dir = getShadersPath();

  for (StringList::const_iterator sl_it = str_list.begin ();
       sl_it != str_list.end ();
       sl_it++)
  {
    const string &file(*sl_it);
    string shader_name (file);

    shader_name.erase (0, shaders_root_path.length()+1);
    shader_name.erase (shader_name.length() - 3, shader_name.length());

    string s(searchShaderFile(shaders_dir + "/" + shader_name + compiled_extension));

    if (s.empty())
    {
      must_compile = true;
      shaders.push_back(shader_name);
    }
  }

  if(must_compile)
  {
    for (vector<string>::iterator s_it = shaders.begin ();
         s_it != shaders.end ();
         s_it++)
    {
      string cmd (shader_compiler + " \"" + getShadersPath() + (*s_it) + compiled_extension +
                  "\" \"" + searchDataDir("rib_data") + "/shaders_data/" + (*s_it) + ".sl\"");

      if(std::system(cmd.c_str()) != 0)
      {
        cout << "Error: " << cmd << endl;

        mainWindow.getConsole()->open();
        mainWindow.getConsole()->printMessage(kConsoleMessage_ShaderCompileError);
        return false;
      }
    }
  }

  return true;
}

bool compileTextures(const string& textures_dir, const string& texture_compiler, Window &mainWindow)
{
  vector<string> textures;

  DirectoryList dir_list;
  string texture_root_path (searchDataDir("rib_data") + "/textures_data/");
  dir_list.setRootPath (texture_root_path);
  dir_list.setRecursive (0);
  dir_list.setFileFilter (".tif");

  const StringList &str_list(dir_list.getDirectoryList ());

  bool must_compile = false;
  for (StringList::const_iterator sl_it = str_list.begin ();
       sl_it != str_list.end ();
       sl_it++)
  {
    const string &file(*sl_it);
    string texture_name (file);

    texture_name.erase (0, texture_root_path.length()+1);
    texture_name.erase (texture_name.length() - 4, texture_name.length());

    string s(searchTextureFile(textures_dir + "/" + texture_name + ".tx"));

    if (s.empty())
    {
      must_compile = true;
      textures.push_back(texture_name);
    }
  }

  if(must_compile)
  {
    for (vector<string>::iterator s_it = textures.begin ();
         s_it != textures.end ();
         s_it++)
    {
      string cmd (texture_compiler + " \"" + searchDataDir("rib_data") + "/textures_data/" +
                       (*s_it) + ".tif\" \"" + getTexturesPath() + textures_dir + "/" + (*s_it) + ".tx\"");

      if(std::system(cmd.c_str()) != 0)
      {
        cout << "Error: " << cmd << endl;

        mainWindow.getConsole()->open();
        mainWindow.getConsole()->printMessage(kConsoleMessage_TextureCompileError);
        return false;
      }
    }
  }

  return true;
}

void rendering(Window &mainWindow, const RenderType type)
{
  Global &global = Global::instance ();

  Mesh *mesh = global.getMesh ();
  assert(mesh);
  Camera *camera = global.getCamera();
  assert (camera);

  bool error = false;
  string texture_compiler;
  string render;
  string textures_dir;
  string shader_compiler;

  RIBExporter ribExporter (*mesh);

  if(std::system(aqsis_version.c_str()) == 0)
  {

#if defined(__APPLE__) && defined(__MACH__)
    AQSISEnvSetter aqsisEnv(aqsis_base_path);
#endif // defined(__APPLE__) && defined(__MACH__)

    render = aqsis_render;
    if(type == NORMAL || type == TOON)
    {
      texture_compiler = aqsis_compiler;
      textures_dir = aqsis_textures;
      shader_compiler = aqsis_shader_compiler;
    }
  }
  else if(std::system(pixie_version.c_str()) == 0)
  {
    render = pixie_render;
    if(type == NORMAL || type == TOON)
    {
      texture_compiler = pixie_compiler;
      textures_dir = pixie_textures;
      shader_compiler = pixie_shader_compiler;
    }
  }
  else
  {
    cout << "Unable to find a render engine" << endl;
    mainWindow.getConsole()->open();
    mainWindow.getConsole()->printMessage(kConsoleMessage_RenderEngineNotFound);
    error = true;
  }

  if(!error)
  {
    if(type == PREVIEW || compileTextures(textures_dir, texture_compiler, mainWindow))
    {
      if(type == PREVIEW || compileShaders(shader_compiler, mainWindow))
      {
        // export mesh
        ribExporter.exportFile (getRenderingPath() + "Base.rib");

        // create render RIB
        list <StringPair> replaceList;

        Vector3f position = camera->getPosition();
        float angleX = camera->getAngleX();
        float angleY = camera->getAngleY();

        ostringstream oss;

        StringPair strMeshPath     ("${MESH_PATH}", getRenderingPath());
        if(type == NORMAL || type == TOON)
        {
          StringPair strShadersPath  ("${SHADERS_PATH}", getShadersPath());
          StringPair strTexturesPath ("${TEXTURES_PATH}", getTexturesPath() + textures_dir);
          
          replaceList.push_back (strShadersPath);
          replaceList.push_back (strTexturesPath);
        }
        
        oss << -position.x;
        StringPair strTranslateX   ("${TRANSLATE_X}", oss.str());

        oss.str("");
        oss << -position.z;
        StringPair strTranslateY   ("${TRANSLATE_Y}", oss.str());

        oss.str("");
        oss << position.y;
        StringPair strTranslateZ   ("${TRANSLATE_Z}", oss.str());

        oss.str("");
        oss << rad2deg(M_PI/2 + angleX);
        StringPair strRotateX      ("${ROTATE_X}", oss.str());

        oss.str("");
        oss << rad2deg(-angleY);
        StringPair strRotateZ      ("${ROTATE_Z}", oss.str());

        replaceList.push_back (strMeshPath);
        replaceList.push_back (strTranslateX);
        replaceList.push_back (strTranslateY);
        replaceList.push_back (strTranslateZ);
        replaceList.push_back (strRotateX);
        replaceList.push_back (strRotateZ);

        string used_template;
        string used_rib;
        
        switch(type)
        {
          case NORMAL:
            used_template = render_template;
            used_rib = render_rib;
            break;
          case PREVIEW:
            used_template = preview_template;
            used_rib = preview_rib;
            break;
          case TOON:
            used_template = toon_template;
            used_rib = toon_rib;
            break;
        }

        ribExporter.exportFile (searchDataDir("rib_data"), used_template,
                                    getRenderingPath() + used_rib, replaceList);

        // render
        const string cmd = render + " \"" + getRenderingPath() + used_rib + "\"";
        callRender(cmd, mainWindow);
      }
    }
  }
}



