/*
 *  Copyright (C) 2006  MakeHuman Project
 *
 *  This program is free software; you  can  redistribute  it  and/or
 *  modify  it  under  the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either  version  2  of
 *  the License, or (at your option) any later version.
 *
 *  This  program  is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the  implied  warranty  of
 *  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foun-
 *  dation, Inc., 59 Temple Place, Suite 330, Boston,  MA  02111-1307
 *  USA
 *  
 *  File   : makehuman.cpp
 *  Project: MakeHuman <info@makehuman.org>, http://www.makehuman.org/
 *  App    : makehuman
 *
 *  For individual developers look into the AUTHORS file.
 *   
 */

// TODO: create makehuman.h
#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#if defined(__APPLE__) && defined(__MACH__)
  #include <GLUT/glut.h>
#else
  #ifdef USE_FREEGLUT     
    #include <GL/freeglut.h>
  #else
    #include <GL/glut.h>
  #endif
#endif
#include <iostream>

//mhh...maybe some of these includes are retundant
#include <mhgui/Panel.h>
#include <mhgui/Widget.h>
#include <mhgui/Image.h>
#include <mhgui/ImageSlider.h>
#include <mhgui/Window.h>
#include <mhgui/Console.h>
#include <mhgui/Size.h>
#include <mhgui/Point.h>
#include <mhgui/CGUtilities.h>
#include <mhgui/Camera.h>
#include <animorph/Mesh.h>
#include <animorph/util.h>
#include <animorph/Vector3.h>
#ifdef DEBUG
#include <StopClock/StopClock.h>
#endif

#include "CharacterSettingPanel.h"
#include "Global.h"
#include "ToolbarPanel.h"
#include "UtilitybarPanel.h"
#include "FooterPanel.h"
#include "ViewPanel.h"
#include "TooltipPanel.h"
#include "TargetPanel.h"
#include "BottomPanel.h"
#include "SplashPanel.h"
#include "ConsoleListener.h"
#include "ComponentID.h"
//#include "DialogPanel.h" // test
#include "util.h"

using namespace std;
using namespace Animorph;

static void renderMesh ();
static void renderMeshEdges();

// console listener
static ConsoleListener       *consoleListener;
static TooltipPanel          *tooltipPanel;
static ToolbarPanel          *toolbarPanel;
static UtilitybarPanel       *utilitybarPanel;
static FooterPanel           *footerPanel;
static ViewPanel             *viewPanel;
static BottomPanel           *bottomPanel;
static SplashPanel           *splashPanel;
static Console               *console; 
static Mesh                  *mesh;
static Camera                *camera;

bool init;  // shows the init status
int splashMotionCount;

const Color border_color   (1.0, 0.55, 0.0, 0.8);
const Color grid_color (0.35, 0.50, 0.30, 0.50);

//Display function
static void display()
{
  camera->applyMatrix ();
  
  //cgutils::drawAxis ();

  renderMesh ();
  
  if(Global::instance().getFlatShading())
    renderMeshEdges();
                            
  Window &mainWindow(Window::instance());
  mainWindow.draw ();
  
  if (Global::instance().getDrawGrid ())
    cgutils::drawGrid ( mainWindow.getSize(), 220, 70, grid_color, border_color, 50);
  
  cgutils::swapBuffers ();

  // TODO: put this in a own function
  if (!init)
  {
    //mesh->loadTargetsFactory (searchDataDir ("targets"));

    init = true;  
  }
};

//Glut callbacks
static void reshape     (int w, int h)
{
  Window &mainWindow(Window::instance());
  mainWindow.reshape            (Size(w,h));
  camera->reshape (w, h);
}

static void motion      (int x, int y)
{
  Window &mainWindow(Window::instance());
  splashPanel = (SplashPanel *)mainWindow.getPanel(kComponentID_SplashPanel);
  if(splashPanel != NULL && splashMotionCount++ >= 5)
  {
    mainWindow.removePanel(splashPanel);
    delete splashPanel;
    splashPanel = NULL;

    Global::instance().setDrawGrid (true);
    splashMotionCount = 0;
    cgutils::redisplay();
  }

  mainWindow.isMouseOverPanel   (Point(x,y));
}

static void special     (int key, int x, int y)
{
  Window &mainWindow(Window::instance());
  if (!mainWindow.isKeyTypePanel (key))
  {
    switch(key) 
    {
      case GLUT_KEY_UP:
        camera->move(0, 1, 0);
        break;
      case GLUT_KEY_DOWN:
        camera->move(0, -1, 0);
        break;
      case GLUT_KEY_LEFT:
        camera->move(-1, 0, 0);
        break;
      case GLUT_KEY_RIGHT:
        camera->move(1, 0, 0);
        break;
    }                               
  }  
  cgutils::redisplay ();     
}

// --- START OF UGLY TESTCODE - THIS WILL BE REMOVED IN THE FINAL RELEASE ----
#ifdef ANIM_TARGET_TRACKER_TEST

static Vector3f startVector(0.0f,  0.0f, -30.0f);
static Vector3f endVector  (0.0f,  -7.0f, -3.0f);

static float startAngleX = ( 0.0 * M_PI) / 180.0f;
static float endAngleX   = (-90.0 * M_PI) / 180.0f;

static float startAngleY = ( 0.0 * M_PI) / 180.0f;
static float endAngleY   = ( 00.0 * M_PI) / 180.0f;


static float getYForX(float x)
{
    float v = 0.5 * (::cosf(x * M_PI) +1.0f);
    return 1.0f - (v * v);
}

static void calcForStepAnimate(float inX)
{
    camera->resetPosition();
    camera->resetRotation();

    float s = getYForX(inX);

    static Vector3f currentVector;
    static float    currentAngleX;
    static float    currentAngleY;
    
    currentVector = Vector3f (startVector.x + s * (endVector.x - startVector.x),
                              startVector.y + s * (endVector.y - startVector.y),
                              startVector.z + s * (endVector.z - startVector.z));

    
    currentAngleX = startAngleX + s * (endAngleX - startAngleX);
    currentAngleY = startAngleY + s * (endAngleY - startAngleY);


    camera->rotate(currentAngleX, X_AXIS);
    camera->rotate(currentAngleY, Y_AXIS);


    camera->move(currentVector.x, currentVector.y, currentVector.z);

}


static const float timeForMorph   = 3.0; // in Seconds
static const float kTimePerRaster = 0.03f;
static int   steps  = static_cast <int> (timeForMorph / kTimePerRaster);

static int   step = -1;


static void timerTrigger(int)
{
    if (step<0)
        return;

//    printf("Triggering timer");
//    camera->rotate (-M_PI/100, X_AXIS);

    float currentTime = step * kTimePerRaster;


    calcForStepAnimate(currentTime / timeForMorph);
    cgutils::redisplay ();
    ::glutTimerFunc(static_cast <unsigned int> (kTimePerRaster*1000), timerTrigger, 0);     


    ++step;

    if (step > steps)
        step = -1;
}

static void startAnim()
{
    step = 0;
    
    startVector = camera->getPosition();
    startAngleX = camera->getAngleX();
    startAngleY = camera->getAngleY();

    ::glutTimerFunc(static_cast <unsigned int> (kTimePerRaster*1000), timerTrigger, 0);
}

static void moveCameraAnimated(const Vector3f& inEndPos, float inEndAngleX, float inEndAngleY)
{
    endAngleX = inEndAngleX;
    endAngleY = inEndAngleY;
    endVector = inEndPos;
    startAnim();
}

#endif // ANIM_TARGET_TRACKER_TEST
// --- END OF UGLY TESTCODE - THIS WILL BE REMOVED IN THE FINAL RELEASE ----

static void keyboard    (unsigned char key,     int x, int y)
{
  Window &mainWindow(Window::instance());
  if (!mainWindow.isKeyTypePanel (key))
  {
    switch (toupper (key)) {           
    case 27:  /*  Escape key  */
      exit (0);
      break;
    case '+':
      camera->move (0,0,1);
      break;
    case '-':
      camera->move (0,0,-1);
      break;
    case '8':
      camera->rotate (-M_PI/12, X_AXIS);
      break;
    case '2':
      camera->rotate (M_PI/12, X_AXIS);
      break;
    case '1':
      camera->resetRotation();
      camera->resetPosition();      
      camera->rotate (-M_PI/2, X_AXIS);
      camera->move (0,0,-75);
      break;
    case '7':
      camera->resetRotation();
      break;
    case '6':
      camera->rotate (-M_PI/12, Y_AXIS);
      break;
    case '4':
      camera->rotate (M_PI/12, Y_AXIS);
      break;
    case '3':
      camera->resetRotation();
      camera->resetPosition();      
      camera->rotate (-M_PI/2, X_AXIS);
      camera->rotate (-M_PI/2, Y_AXIS);      
      camera->move (0,0,-75);
      break;

    case '.':

// Test Code! - This will be removed in final release!
#ifdef ANIM_TARGET_TRACKER_TEST 
      moveCameraAnimated(Vector3f(0.0f,  0.0f, -75.0f), 
                                  ((-90.0f) * M_PI) / 180.0f,
                                  (   0.0f * M_PI) / 180.0f);
#else // original code
      camera->resetPosition();
      camera->move (0,0,-75);      
#endif // ANIM_TARGET_TRACKER_TEST 

      break;      
    case 'O':
      console->open();
      break;
    
// Test Code! - This will be removed in final release!
#ifdef ANIM_TARGET_TRACKER_TEST 
    case 'A':
      // The offset of 360.0 degrees is just to illustarte, that the
      // camera animation engine is capable to interpolate multiple spins...
      moveCameraAnimated(Vector3f(0.0f,  -7.2f, -7.5f), 
                                  ((-90.0f+360.0f) * M_PI) / 180.0f,
                                  ((  0.0f-360.0f)* M_PI) / 180.0f);

      break;
#endif // ANIM_TARGET_TRACKER_TEST 
    case 'T':
      rendering(mainWindow, TOON);
      break;
    default:
      break;
    }
  }
  cgutils::redisplay ();
}

static void mouse (int button, int state, int x, int y)
{
  Window &mainWindow(Window::instance());
  splashPanel = (SplashPanel *)mainWindow.getPanel(kComponentID_SplashPanel);
  if(splashPanel != NULL)
  {
    mainWindow.removePanel(splashPanel);
    delete splashPanel;
    splashPanel = NULL;

    Global::instance().setDrawGrid (true);
    splashMotionCount = 0;
    cgutils::redisplay();
  }
  
  if(console->isActive())
  {
    console->isMouseClick(Point(x,y), button, state);
    return;
  }
       
  //cerr << "mouse: " << button << endl;
  camera->mouseRotateStart (x, y);
 
  if (button == 3)
  {
    camera->move (0,0,1);
  }

  if (button == 4)
  {
    camera->move (0,0,-1);
  }

  mainWindow.isMouseClickPanel  (Point(x, y), button, state);

  cgutils::redisplay ();
}

static void activeMotion (int x, int y)
{
  if(console->isActive())
  {
    console->isMouseDragged(Point(x,y));
    return;
  }
         
  Window &mainWindow(Window::instance());
  if (!mainWindow.isMouseDraggedPanel(Point(x,y)))
  {
    camera->rotateMouse (x, y);
    cgutils::redisplay ();
  }
}

#if defined(__APPLE__) && defined(__MACH__)
int main_wrapper (int argc, char** argv)
#else
int main (int argc, char** argv)
#endif
{
  // On OS X this is already done!
#if !(defined(__APPLE__) && defined(__MACH__))
  //Glut init
  glutInit (&argc, argv);
#endif

//  atexit(shutdown);

  mhgui::Window::createSingelton(800, 600 ,"MakeHuman", Color (0,0,0));
  Window &mainWindow(Window::instance());

  //atexit (shutdown);
  splashMotionCount = 0;
  createWorkingDirs();

  tooltipPanel          = new TooltipPanel();  
  toolbarPanel          = new ToolbarPanel();
  utilitybarPanel       = new UtilitybarPanel();
  footerPanel           = new FooterPanel();
  viewPanel             = new ViewPanel();
  bottomPanel           = new BottomPanel();
  splashPanel           = new SplashPanel();
  mesh                  = new Mesh();
  camera                = new Camera();

  CharacterSettingPanel *characterSettingPanel = new CharacterSettingPanel();
  mainWindow.initWindow ();

  // TODO: put animorph mesh loading stuff in extra function

  // load mesh with factory function
  bool mesh_loaded = mesh->loadMeshFactory (searchDataFile ("base.vertices"),
                     searchDataFile ("base.faces"));
  if (!mesh_loaded)
  {
    cerr << "couldn't load mesh geometry" << endl;
    return 1;
  }


  // load material for mesh
  bool material_loaded = mesh->loadMaterialFactory (searchDataFile ("base.materials"),
                         searchDataFile ("base.colors"));
  if (!material_loaded)
  {
    cerr << "couldn't load mesh material informations" << endl;
    return 1;
  }

  // load UV texture data, currently no Factory function
  TextureVector &texturevector = mesh->getTextureVectorRef ();
  texturevector.load (searchDataFile ("base.uv"));

  // put mesh container into the Global Singleton
  Global::instance().setMesh (mesh);
  
  // put camera into the Global Singleton
  Global::instance().setCamera (camera);


  mesh->loadTargetsFactory (searchDataDir ("targets"));
  mesh->loadPoseTargetsFactory(searchDataDir ("rotations"));  
  mesh->loadCharactersFactory(searchDataDir ("bs"));
  mesh->loadCharactersFactory(searchDataDir ("mybs"));

  init = false;
   
  consoleListener = new ConsoleListener();

  //Add console
  console = new Console (FOUR_CHAR_CONST('C','O','N','1'));  
  console->setListener (consoleListener);

  console->loadPNG (searchPixmapFile ("ui/console.png"));
  console->addSplashLine ("MakeHuman operating console");
  console->addSplashLine ("version 0.1");
  console->addSplashLine ("2005-2006");
  console->addSplashLine ("- - - - - - - - - - - - - -");
  mainWindow.setConsole (console);

  //Add panels to mainwindow  
  mainWindow.addPanel (tooltipPanel);
  mainWindow.addPanel (toolbarPanel);
  mainWindow.addPanel (utilitybarPanel);
  mainWindow.addPanel (footerPanel);
  mainWindow.addPanel (viewPanel);
  mainWindow.addPanel (bottomPanel);  
  mainWindow.addPanel (characterSettingPanel);
  mainWindow.addPanel (splashPanel);

  // set initial camera position
  camera->rotate (-M_PI/2, X_AXIS);
  camera->move (0,0,-75.0f);

  // create after adding
  tooltipPanel->createWidgets ();  
  toolbarPanel->createWidgets ();
  utilitybarPanel->createWidgets ();
  footerPanel->createWidgets ();
  viewPanel->createWidgets ();
  bottomPanel->createWidgets();
  characterSettingPanel->createWidgets();
  splashPanel->createWidgets();
 
  mainWindow.loadPNG (searchPixmapFile ("ui/background.png"));

  //Activate the images textures
  //Note it's after context creation
  toolbarPanel->show_all ();
  utilitybarPanel->show_all ();
  footerPanel->show_all ();
  viewPanel->show_all();
  tooltipPanel->show_all();
  bottomPanel->show_all();
  characterSettingPanel->show_all();
  splashPanel->show_all();

  
  //Glut callbacks
  mainWindow.setDisplayCallback(display);
  mainWindow.setReshapeCallback(reshape);
  mainWindow.setPassiveMotionCallback(motion);
  mainWindow.setKeyboardCallback(keyboard);
  mainWindow.setMouseCallback(mouse);
  mainWindow.setMotionCallback(activeMotion);
  mainWindow.setSpecialCallback(special); 

  console->show ();
  mainWindow.show (); 


  mainWindow.mainLoop();

  // we never reach this point.
  return 0;
}

void renderMeshEdges()
{
  const FaceVector     &facevector    (mesh->getFaceVectorRef ());
  const VertexVector   &vertexvector  (mesh->getVertexVectorRef ());
  
  const Color edges_color (0.4, 0.3, 0.3, 0.5);
  
  FaceVector::const_iterator i; 
  
  cgutils::enableBlend ();
  cgutils::enableLineSmoothing ();
  glDisable (GL_LIGHTING);
    
  for (i = facevector.begin(); i != facevector.end(); ++i)
  {
    const Face &face(*i);

    size_t j;

    ::glBegin (GL_LINE_LOOP);
   
    // Set the color for these vertices
   ::glColor4fv  (edges_color.getAsOpenGLVector());
   
    for (j = 0; j != face.getSize(); ++j)
    {
      const Vertex &vertex = vertexvector[face.getVertexAtIndex(j)];
      ::glNormal3fv (face.no.getAsOpenGLVector());
      ::glVertex3fv ((vertex.co + (face.no * 0.005)).getAsOpenGLVector());
    }
    ::glEnd ();    
  } //   for (i = facevector.begin(); i != facevector.end(); ++i)  
  
  glEnable (GL_LIGHTING);
  cgutils::disableLineSmoothing ();
  cgutils::disableBlend();  
}

void renderMesh ()
{
  const MaterialVector &materialvector(mesh->getMaterialVectorRef ());
  const FaceVector     &facevector    (mesh->getFaceVectorRef ());
  const VertexVector   &vertexvector  (mesh->getVertexVectorRef ());

  FaceVector::const_iterator i; 

  for (i = facevector.begin(); i != facevector.end(); ++i)
  {
    const Face &face(*i);

    size_t j;

    ::glBegin (GL_POLYGON);

    int material_index = face.getMaterialIndex ();
    if (material_index != -1)
    {
      const Material &material(materialvector[material_index]);
      const Color    &color   (material.getRGBCol ());

      // Set the color for these vertices
      ::glColor4fv  (color.getAsOpenGLVector());

    }
    for (j = 0; j != face.getSize(); ++j)
    {
      const Vertex &vertex = vertexvector[face.getVertexAtIndex(j)];
      
      //::glNormal3fv (face.no.getAsOpenGLVector());
      ::glNormal3fv (vertex.no.getAsOpenGLVector());  
      ::glVertex3fv (vertex.co.getAsOpenGLVector());
    }
    ::glEnd ();    
  } //   for (i = facevector.begin(); i != facevector.end(); ++i)
}

