/* imagetypes.c:
 *
 * this contains things which reference the global ImageTypes array
 *
 * jim frost 09.27.89
 *
 * Copyright 1989, 1991 Jim Frost.
 * See included file "copyright.h" for complete copyright information.
 */

#include "copyright.h"
#include "image.h"
#include "imagetypes.h"
#include <errno.h>

/* some of these are order-dependent
 */

static struct imagetypes {
  int    (*identifier)(); /* print out image info if this kind of image */
  Image *(*loader)();     /* load image if this kind of image */
  char  *name;            /* name of this image format */
} ImageTypes[] = {
  { fbmIdent,       fbmLoad,       "FBM Image" },
  { sunRasterIdent, sunRasterLoad, "Sun Rasterfile" },
  { cmuwmIdent,     cmuwmLoad,     "CMU WM Raster" },
  { pbmIdent,       pbmLoad,       "Portable Bit Map (PBM, PGM, PPM)" },
  { facesIdent,     facesLoad,     "Faces Project" },
#ifdef USE_GIF
  { gifIdent,       gifLoad,       "GIF Image" },
#endif
#ifdef USE_PNG
  { pngIdent,       pngLoad,       "PNG Image" },
#endif
  { rleIdent,       rleLoad,       "Utah RLE Image" },
  { jpegIdent,      jpegLoad,      "JFIF-style JPEG Image" },
  { xwdIdent,       xwdLoad,       "X Window Dump" },
  { mcidasIdent,    mcidasLoad,    "McIDAS areafile" },
  { g3Ident,        g3Load,        "G3 FAX Image" },
  { pcxIdent,       pcxLoad,       "PC Paintbrush Image" },
  { imgIdent,       imgLoad,       "GEM Bit Image" },
  { macIdent,       macLoad,       "MacPaint Image" },
  { xpixmapIdent,   xpixmapLoad,   "X Pixmap" },
  { xbitmapIdent,   xbitmapLoad,   "X Bitmap" },
  { NULL,           NULL,          NULL }
};
/* SUPPRESS 560 */

extern int errno;
extern int findImage(char *, char *);

/* load a named image
 */

Image *loadImage(name, verbose)
     char         *name;
     unsigned int  verbose;
{ char   fullname[BUFSIZ];
  Image *image;
  int    a;

  if (findImage(name, fullname) < 0) {
    if (errno == ENOENT)
      fprintf(stderr, "%s: image not found\n", name);
    else
      perror(fullname);
    return(NULL);
  }
  for (a= 0; ImageTypes[a].loader; a++)
    if ((image = ImageTypes[a].loader(fullname, name, verbose))) {
      zreset(NULL);
      return(image);
    }
  fprintf(stderr, "%s: unknown or unsupported image type\n", fullname);
  zreset(NULL);
  return(NULL);
}

/* identify what kind of image a named image is
 */

void identifyImage(name)
     char *name;
{ char fullname[BUFSIZ];
  int  a;

  if (findImage(name, fullname) < 0) {
    if (errno == ENOENT)
      fprintf(stderr, "%s: image not found\n", name);
    else
      perror(fullname);
    return;
  }
  for (a= 0; ImageTypes[a].identifier; a++) {
    if (ImageTypes[a].identifier(fullname, name)) {
      zreset(NULL);
      return;
    }
}
  zreset(NULL);
  fprintf(stderr, "%s: unknown or unsupported image type\n", fullname);
}

/* tell user what image types we support
 */

void supportedImageTypes()
{ int a;

  fprintf(stderr, "Image types supported:\n");
  for (a= 0; ImageTypes[a].name; a++)
    fprintf(stderr, "  %s\n", ImageTypes[a].name);
}

void goodImage(image, func)
     Image *image;
     char  *func;
{
  if (!image) {
    fprintf(stderr, "%s: nil image\n", func);
    cleanup(-1);
  }
  switch (image->type) {
  case IBITMAP:
  case IRGB:
  case ITRUE:
    break;
  default:
    fprintf(stderr, "%s: bad destination image\n", func);
    cleanup(-1);
  }
}
