// -*- C++ -*-
/**
 * \file package.h
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Angus Leeming
 *
 * Full author contact details are available in file CREDITS.
 *
 * A store of the paths to the various different directoies used
 * by LyX. These paths differ markedly from one OS to another,
 * following the local Windows, MacOS X or Posix conventions.
 */
#ifndef LYX_PACHAGE_H
#define LYX_PACHAGE_H

#include <string>

namespace lyx {
namespace support {

class Package;

/** When run in-place <build-dir>/src/lyx is one level up from
 *  the <build-dir> whilst <build-dir>/src/tex2lyx/tex2lyx is
 *  two levels up.
 */
enum exe_build_dir_to_top_build_dir {
	top_build_dir_is_one_level_up,
	top_build_dir_is_two_levels_up
};


/** Initialise package() with the command line data.
 *  This data is exactly as it was passed in the argv[] array.
 *
 *  @param command_line_arg0 argv[0], the name of the LyX executable
 *  as passed on the command line.
 *
 *  @param command_line_system_support_dir, the LyX system dir,
 *  as specified on the command line with "-sysdir <path to dir>".
 *
 *  @param command_line_user_support_dir, the LyX user dir,
 *  as specified on the command line with "-userdir <path to dir>".
 */
void init_package(std::string const & command_line_arg0,
		  std::string const & command_line_system_support_dir,
		  std::string const & command_line_user_support_dir,
		  exe_build_dir_to_top_build_dir);

/** Accessor to the global data.
 *  Asserts that init_package() has been called first.
 */
Package const & package();

class Package {
public:
	/// Default constructor does not lead to the paths being set.
	Package();

	/** Called by init_package, above.
	 *  All paths will be initialized.
	 */
	Package(std::string const & command_line_arg0,
		std::string const & command_line_system_support_dir,
		std::string const & command_line_user_support_dir,
		exe_build_dir_to_top_build_dir);

	/** The directory containing the LyX executable.
	 */
	std::string const & binary_dir() const;

	/** The top of the LyX source code tree.
	 *  Used by the GTK frontend when searching for .glade files.
	 */
	std::string const & top_srcdir() const;

	/** The path to the system-level support files
	 *  we're actually going to use.
	 */
	std::string const & system_support() const;

	/** The path to the autogenerated support files
	 *  when running in-place.
	 */
	std::string const & build_support() const;

	/** The path to the user-level support files.
	 */
	std::string const & user_support() const;

	/** The user_support directory was set explicitly using either
	 *  the -userdir command line switch or
	 *  the LYX_USERDIR_14x environment variable.
	 */
	bool explicit_user_support() const;

	/** The path to the locale directory.
	 */
	std::string const & locale_dir() const;

	/** The default document directory.
	 *  Can be reset by LyXRC.
	 */
	std::string & document_dir() const;

	/** The path to the temporary directory.
	 *  (Eg /tmp on *nix.)
	 *  Can be reset by LyXRC.
	 */
	std::string & temp_dir() const;

	/** Used when setting the user_support directory.
	 *  Used also when expanding "~/" or contracting to "~/". (filetools.C)
	 *  Used by the XForms file dialog.
	 *  Used in emergencyWrite (bufferlist.C) as one possible location
	 *  for the dump.
	 */
	std::string const & home_dir() const;

private:
	std::string binary_dir_;
	std::string system_support_dir_;
	std::string build_support_dir_;
	std::string user_support_dir_;
	std::string locale_dir_;
	mutable std::string document_dir_;
	mutable std::string temp_dir_;
	std::string home_dir_;
	bool explicit_user_support_dir_;
};


inline
Package::Package() {}

inline
std::string const & Package::binary_dir() const
{
	return binary_dir_;
}

inline
std::string const & Package::system_support() const
{
	return system_support_dir_;
}

inline
std::string const & Package::build_support() const
{
	return build_support_dir_;
}

inline
std::string const & Package::user_support() const
{
	return user_support_dir_;
}

inline
bool Package::explicit_user_support() const
{
	return explicit_user_support_dir_;
}

inline
std::string const & Package::locale_dir() const
{
	return locale_dir_;
}

inline
std::string & Package::document_dir() const
{
	return document_dir_;
}

inline
std::string & Package::temp_dir() const
{
	return temp_dir_;
}

inline
std::string const & Package::home_dir() const
{
	return home_dir_;
}

} // namespace support
} // namespace lyx

#endif // LYX_PACHAGE_H
