/* This file is part of
 * ====================================================== 
 * 
 *           LyX, The Document Processor
 * 	 
 *           Copyright 1995 Matthias Ettrich
 *           Copyright 1995-2000 The LyX Team.
 *
 * ====================================================== */

#include <config.h>
#include <cstdlib>
#include <cctype>
#include <algorithm>

#include "layout.h"
#include "lyxparagraph.h"
#include "lyxtext.h"
#include "support/textutils.h"
#include "insets/insetbib.h"
#include "insets/insettext.h"
#include "lyx_gui_misc.h"
#include "gettext.h"
#include "bufferparams.h"
#include "buffer.h"
#include "minibuffer.h"
#include "debug.h"
#include "lyxrc.h"
#include "LyXView.h"
#include "lyxrow.h"
#include "Painter.h"
#include "tracer.h"
#include "font.h"
#include "encoding.h"
#include "lyxscreen.h"
#include "bufferview_funcs.h"

using std::max;
using std::min;
using std::endl;
using std::pair;

static const int LYX_PAPER_MARGIN = 20;

extern int bibitemMaxWidth(BufferView *, LyXFont const &);


int LyXText::workWidth(BufferView * bview) const
{
	if (inset_owner) {
		return inset_owner->textWidth(bview->painter());
	}
	return bview->workWidth();
}


unsigned char LyXText::TransformChar(unsigned char c, LyXParagraph * par,
			LyXParagraph::size_type pos) const
{
	if (!Encodings::is_arabic(c))
		if (lyxrc.font_norm_type == LyXRC::ISO_8859_6_8 && isdigit(c))
			return c + (0xb0 - '0');
		else
			return c;

	unsigned char const prev_char = pos > 0 ? par->GetChar(pos-1) : ' ';
	unsigned char next_char = ' ';
	for (LyXParagraph::size_type i = pos+1; i < par->Last(); ++i)
		if (!Encodings::IsComposeChar_arabic(par->GetChar(i))) {
			next_char = par->GetChar(i);
			break;
		}

	if (Encodings::is_arabic(next_char)) {
		if (Encodings::is_arabic(prev_char))
			return Encodings::TransformChar(c, Encodings::FORM_MEDIAL);
		else
			return Encodings::TransformChar(c, Encodings::FORM_INITIAL);
	} else {
		if (Encodings::is_arabic(prev_char))
			return Encodings::TransformChar(c, Encodings::FORM_FINAL);
		else
			return Encodings::TransformChar(c, Encodings::FORM_ISOLATED);
	}
}

// This is the comments that some of the warnings below refers to.
// There are some issues in this file and I don't think they are
// really related to the FIX_DOUBLE_SPACE patch. I'd rather think that
// this is a problem that has been here almost from day one and that a
// larger userbase with differenct access patters triggers the bad
// behaviour. (segfaults.) What I think happen is: In several places
// we store the paragraph in the current cursor and then moves the
// cursor. This movement of the cursor will delete paragraph at the
// old position if it is now empty. This will make the temporary
// pointer to the old cursor paragraph invalid and dangerous to use.
// And is some cases this will trigger a segfault. I have marked some
// of the cases where this happens with a warning, but I am sure there
// are others in this file and in text2.C. There is also a note in
// Delete() that you should read. In Delete I store the paragraph->id
// instead of a pointer to the paragraph. I am pretty sure this faulty
// use of temporary pointers to paragraphs that might have gotten
// invalidated (through a cursor movement) before they are used, are
// the cause of the strange crashes we get reported often.
//
// It is very tiresom to change this code, especially when it is as
// hard to read as it is. Help to fix all the cases where this is done
// would be greately appreciated.
//
// Lgb

int LyXText::SingleWidth(BufferView * bview, LyXParagraph * par,
			 LyXParagraph::size_type pos) const
{
	char const c = par->GetChar(pos);
	return SingleWidth(bview, par, pos, c);
}


int LyXText::SingleWidth(BufferView * bview, LyXParagraph * par,
			 LyXParagraph::size_type pos, char c) const
{
	LyXFont font = GetFont(bview->buffer(), par, pos);

	// The most common case is handled first (Asger)
	if (IsPrintable(c)) {
		if (font.language()->RightToLeft()) {
			if (font.language()->lang() == "arabic" &&
			    (lyxrc.font_norm_type == LyXRC::ISO_8859_6_8 ||
			     lyxrc.font_norm_type == LyXRC::ISO_10646_1))
				if (Encodings::IsComposeChar_arabic(c))
					return 0;
				else
					c = TransformChar(c, par, pos);
			else if (font.language()->lang() == "hebrew" &&
				 Encodings::IsComposeChar_hebrew(c))
				return 0;
		}
		return lyxfont::width(c, font);

	} else if (IsHfillChar(c)) {
		return 3;	/* Because of the representation
				 * as vertical lines */
#ifndef NEW_INSETS
	} else if (c == LyXParagraph::META_FOOTNOTE ||
		   c == LyXParagraph::META_MARGIN ||
		   c == LyXParagraph::META_FIG ||
		   c == LyXParagraph::META_TAB ||
		   c == LyXParagraph::META_WIDE_FIG ||
		   c == LyXParagraph::META_WIDE_TAB ||
		   c == LyXParagraph::META_ALGORITHM) {
		string fs;
		switch (c) {
		case LyXParagraph::META_MARGIN:
			fs = "margin";
			break;
		case LyXParagraph::META_FIG:
			fs = "fig";
			break;
		case LyXParagraph::META_TAB:
			fs = "tab";
			break;
		case LyXParagraph::META_ALGORITHM:
			fs = "alg";
			break;
		case LyXParagraph::META_WIDE_FIG:
			fs = "wide-fig";
			break;
		case LyXParagraph::META_WIDE_TAB:
			fs = "wide-tab";
			break;
		case LyXParagraph::META_FOOTNOTE:
			fs = "foot";
			break;
		}
		font.decSize();
		font.decSize();
		return lyxfont::width(fs, font);
#endif
	} else if (c == LyXParagraph::META_INSET) {
		Inset * tmpinset = par->GetInset(pos);
		if (tmpinset) {
			tmpinset->update(bview, font);
			return tmpinset->width(bview, font);
		} else
			return 0;

	} else if (IsSeparatorChar(c))
		c = ' ';
	else if (IsNewlineChar(c))
		c = 'n';
	return lyxfont::width(c, font);
}


// Returns the paragraph position of the last character in the specified row
LyXParagraph::size_type LyXText::RowLast(Row const * row) const
{
	if (row->next() == 0)
		return row->par()->Last() - 1;
	else if (row->next()->par() != row->par()) 
		return row->par()->Last() - 1;
	else 
		return row->next()->pos() - 1;
}


LyXParagraph::size_type LyXText::RowLastPrintable(Row const * row) const
{
	LyXParagraph::size_type const last = RowLast(row);
	if (last >= row->pos()
	    && row->next()
	    && row->next()->par() == row->par()
	    && row->par()->IsSeparator(last))
		return last - 1;
	else
		return last;
}


void LyXText::ComputeBidiTables(Buffer const * buf, Row * row) const
{
	bidi_same_direction = true;
	if (!lyxrc.rtl_support) {
		bidi_start = -1;
		return;
	}

	bidi_start = row->pos();
	bidi_end = RowLastPrintable(row);

	if (bidi_start > bidi_end) {
		bidi_start = -1;
		return;
	}

	if (bidi_end + 2 - bidi_start >
	    static_cast<LyXParagraph::size_type>(log2vis_list.size())) {
		LyXParagraph::size_type new_size = 
			(bidi_end + 2 - bidi_start < 500) ?
			500 : 2 * (bidi_end + 2 - bidi_start);
		log2vis_list.resize(new_size);
		vis2log_list.resize(new_size);
		bidi_levels.resize(new_size);
	}

	vis2log_list[bidi_end + 1 - bidi_start] = -1;
	log2vis_list[bidi_end + 1 - bidi_start] = -1;

	LyXParagraph::size_type	stack[2];
	bool const rtl_par =
		row->par()->getParLanguage(buf->params)->RightToLeft();
	int level = 0;
	bool rtl = false;
	bool rtl0 = false;
	LyXParagraph::size_type const main_body =
		BeginningOfMainBody(buf, row->par());

	for (LyXParagraph::size_type lpos = bidi_start;
	     lpos <= bidi_end; ++lpos) {
		bool is_space = row->par()->IsLineSeparator(lpos);
		LyXParagraph::size_type const pos =
			(is_space && lpos + 1 <= bidi_end &&
			 !row->par()->IsLineSeparator(lpos + 1) &&
			 !row->par()->IsNewline(lpos + 1))
			? lpos + 1 : lpos;
		LyXFont font = row->par()->GetFontSettings(buf->params, pos);
		if (pos != lpos && 0 < lpos && rtl0 && font.isRightToLeft() &&
		    font.number() == LyXFont::ON &&
		    row->par()->GetFontSettings(buf->params, lpos-1).number()
		    == LyXFont::ON) {
			font = row->par()->GetFontSettings(buf->params, lpos);
			is_space = false;
		}


		bool new_rtl = font.isVisibleRightToLeft();
		bool new_rtl0 = font.isRightToLeft();
		int new_level;

		if (lpos == main_body - 1
		    && row->pos() < main_body - 1
		    && is_space) {
			new_level = (rtl_par) ? 1 : 0;
			new_rtl = new_rtl0 = rtl_par;
		} else if (new_rtl0)
			new_level = (new_rtl) ? 1 : 2;
		else
			new_level = (rtl_par) ? 2 : 0;

		if (is_space && new_level >= level) {
			new_level = level;
			new_rtl = rtl;
			new_rtl0 = rtl0;
		}

		int new_level2 = new_level;

		if (level == new_level && rtl0 != new_rtl0) {
			--new_level2;
			log2vis_list[lpos - bidi_start] = (rtl) ? 1 : -1;
		} else if (level < new_level) {
			log2vis_list[lpos - bidi_start] =  (rtl) ? -1 : 1;
			if (new_level > rtl_par)
				bidi_same_direction = false;
		} else
			log2vis_list[lpos - bidi_start] = (new_rtl) ? -1 : 1;
		rtl = new_rtl;
		rtl0 = new_rtl0;
		bidi_levels[lpos - bidi_start] = new_level;

		while (level > new_level2) {
			LyXParagraph::size_type old_lpos =
				stack[--level];
			int delta = lpos - old_lpos - 1;
			if (level % 2)
				delta = -delta;
			log2vis_list[lpos - bidi_start] += delta;
			log2vis_list[old_lpos - bidi_start] += delta;
		}
		while (level < new_level)
			stack[level++] = lpos;
	}

	while (level > 0) {
		LyXParagraph::size_type const old_lpos = stack[--level];
		int delta = bidi_end - old_lpos;
		if (level % 2)
			delta = -delta;
		log2vis_list[old_lpos - bidi_start] += delta;
	}

	LyXParagraph::size_type vpos = bidi_start - 1;
	for (LyXParagraph::size_type lpos = bidi_start;
	     lpos <= bidi_end; ++lpos) {
		vpos += log2vis_list[lpos - bidi_start];
		vis2log_list[vpos - bidi_start] = lpos;
		log2vis_list[lpos - bidi_start] = vpos;
	}
}


// This method requires a previous call to ComputeBidiTables()
bool LyXText::IsBoundary(Buffer const * buf, LyXParagraph * par,
			 LyXParagraph::size_type pos) const
{
	if (!lyxrc.rtl_support || pos == 0)
		return false;

	if (!bidi_InRange(pos - 1)) {
		/// This can happen if pos is the first char of a row.
		/// Returning false in this case is incorrect!
		return false;
	}

	bool const rtl = bidi_level(pos - 1) % 2;
	bool rtl2 = bidi_InRange(pos)
		? bidi_level(pos) % 2
		: par->isRightToLeftPar(buf->params);
	return rtl != rtl2;
}


bool LyXText::IsBoundary(Buffer const * buf, LyXParagraph * par,
			 LyXParagraph::size_type pos,
			 LyXFont const & font) const
{
	if (!lyxrc.rtl_support)
		return false;    // This is just for speedup

	bool const rtl = font.isVisibleRightToLeft();
	bool rtl2 = bidi_InRange(pos)
		? bidi_level(pos) % 2
		: par->isRightToLeftPar(buf->params);
	return rtl != rtl2;
}


void LyXText::draw(BufferView * bview, Row const * row,
		   LyXParagraph::size_type & vpos,
		   int offset, float & x, bool cleared)
{
	Painter & pain = bview->painter();
	
	LyXParagraph::size_type pos = vis2log(vpos);
	char c = row->par()->GetChar(pos);
	float tmpx = x;

	if (IsNewlineChar(c)) {
		++vpos;
		// Draw end-of-line marker
		LyXFont const font = GetFont(bview->buffer(), row->par(), pos);
		int const wid = lyxfont::width('n', font);
		int const asc = lyxfont::maxAscent(font);
		int const y = offset + row->baseline();
		int xp[3], yp[3];
		
		if (bidi_level(pos) % 2 == 0) {
			xp[0] = int(x + wid * 0.375);
			yp[0] = int(y - 0.875 * asc * 0.75);
			
			xp[1] = int(x);
			yp[1] = int(y - 0.500 * asc * 0.75);
			
			xp[2] = int(x + wid * 0.375);
			yp[2] = int(y - 0.125 * asc * 0.75);
			
			pain.lines(xp, yp, 3, LColor::eolmarker);
			
			xp[0] = int(x);
			yp[0] = int(y - 0.500 * asc * 0.75);
			
			xp[1] = int(x + wid);
			yp[1] = int(y - 0.500 * asc * 0.75);
			
			xp[2] = int(x + wid);
			yp[2] = int(y - asc * 0.75);
			
			pain.lines(xp, yp, 3, LColor::eolmarker);
		} else {
			xp[0] = int(x + wid * 0.625);
			yp[0] = int(y - 0.875 * asc * 0.75);
			
			xp[1] = int(x + wid);
			yp[1] = int(y - 0.500 * asc * 0.75);
			
			xp[2] = int(x + wid * 0.625);
			yp[2] = int(y - 0.125 * asc * 0.75);
			
			pain.lines(xp, yp, 3, LColor::eolmarker);
			
			xp[0] = int(x + wid);
			yp[0] = int(y - 0.500 * asc * 0.75);
			
			xp[1] = int(x);
			yp[1] = int(y - 0.500 * asc * 0.75);
			
			xp[2] = int(x);
			yp[2] = int(y - asc * 0.75);
			
			pain.lines(xp, yp, 3, LColor::eolmarker);
		}
		x += wid;
		return;
	}

	LyXFont font = GetFont(bview->buffer(), row->par(), pos);
	LyXFont font2 = font;
#ifndef NEW_INSETS
	if (c == LyXParagraph::META_FOOTNOTE
	    || c == LyXParagraph::META_MARGIN
	    || c == LyXParagraph::META_FIG
	    || c == LyXParagraph::META_TAB
	    || c == LyXParagraph::META_WIDE_FIG
	    || c == LyXParagraph::META_WIDE_TAB
	    || c == LyXParagraph::META_ALGORITHM) {
		string fs;
		switch (c) {
		case LyXParagraph::META_MARGIN:
			fs = "margin";
			break;
		case LyXParagraph::META_FIG:
			fs = "fig";
			break;
		case LyXParagraph::META_TAB:
			fs = "tab";
			break;
		case LyXParagraph::META_ALGORITHM:
			fs = "alg";
			break;
		case LyXParagraph::META_WIDE_FIG:
			fs = "wide-fig";
			break;
		case LyXParagraph::META_WIDE_TAB:
			fs = "wide-tab";
			break;
		case LyXParagraph::META_FOOTNOTE:
			fs = "foot";
			break;
		}
		font.decSize();
		font.decSize();
	  
		// calculate the position of the footnotemark
		int const y = (row->baseline() - lyxfont::maxAscent(font2) 
			       + lyxfont::maxAscent(font));
	  
		font.setColor(LColor::footnote);

		// draw it and set new x position
		
		pain.text(int(x), offset + y, fs, font);
		x += lyxfont::width(fs, font);
		pain.line(int(tmpx), offset + row->baseline(),
			  int(x), offset + row->baseline(),
			  LColor::footnote);

		++vpos;
		return;
	} else
#endif
		if (c == LyXParagraph::META_INSET) {
		Inset * tmpinset = row->par()->GetInset(pos);
		if (tmpinset) {
//			tmpinset->update(bview, font, false);
			tmpinset->draw(bview, font, offset+row->baseline(), x,
				       cleared);
#ifdef SEEMS_TO_BE_NOT_NEEDED
			if (status == CHANGED_IN_DRAW) {
				UpdateInset(bview, tmpinset);
				status = CHANGED_IN_DRAW;
			}
#endif
		}
		++vpos;

		if (lyxrc.mark_foreign_language &&
		    font.language() != bview->buffer()->params.language) {
			int const y = offset + row->height() - 1;
			pain.line(int(tmpx), y, int(x), y,
				  LColor::language);
		}

		return;
	}

	/* usual characters, no insets */

	// Collect character that we can draw in one command

	// This is dirty, but fast. Notice that it will never be too small.
	// For the record, I'll note that Microsoft Word has a limit
	// of 768 here. We have none :-) (Asger)
	// Ok. I am the first to admit that the use of std::string will be
	// a tiny bit slower than using a POD char array. However, I claim
	// that this slowdown is so small that it is close to inperceptive.
	// So IMHO we should go with the easier and clearer implementation.
	// And even if 1024 is a large number here it might overflow, string
	// will only overflow if the machine is out of memory...
	static string textstring;
	textstring = c;
	++vpos;

	LyXParagraph::size_type const last = RowLastPrintable(row);

	if (font.language()->lang() == "hebrew") {
		if (Encodings::IsComposeChar_hebrew(c)) {
			int const width = lyxfont::width(c, font2);
			int dx = 0;
			for (LyXParagraph::size_type i = pos-1; i >= 0; --i) {
				c = row->par()->GetChar(i);
				if (!Encodings::IsComposeChar_hebrew(c)) {
					if (IsPrintableNonspace(c)) {
						int const width2 =
							SingleWidth(bview,
								    row->par(),
								    i, c);
						dx = (c == '' || c == '') // dalet / resh
							? width2 - width : (width2 - width) / 2;
					}
					break;
				}
			}
			// Draw nikud
			pain.text(int(x) + dx, offset + row->baseline(),
				  textstring, font);
		} else {
			while (vpos <= last &&
			       (pos = vis2log(vpos)) >= 0
			       && IsPrintableNonspace(c = row->par()->GetChar(pos))
			       && !Encodings::IsComposeChar_hebrew(c)
			       && font2 == GetFont(bview->buffer(), row->par(), pos)) {
				textstring += c;
				++vpos;
			}
			// Draw text and set the new x position
			pain.text(int(x), offset + row->baseline(),
				  textstring, font);
			x += lyxfont::width(textstring, font);
		}
	} else if (font.language()->lang() == "arabic" &&
		   (lyxrc.font_norm_type == LyXRC::ISO_8859_6_8 ||
		    lyxrc.font_norm_type == LyXRC::ISO_10646_1)) {
		if (Encodings::IsComposeChar_arabic(c)) {
			c = TransformChar(c, row->par(), pos);
			textstring = c;
			int const width = lyxfont::width(c, font2);
			int dx = 0;
			for (LyXParagraph::size_type i = pos-1; i >= 0; --i) {
				c = row->par()->GetChar(i);
				if (!Encodings::IsComposeChar_arabic(c)) {
					if (IsPrintableNonspace(c)) {
						int const width2 =
							SingleWidth(bview,
								    row->par(),
								    i, c);
						dx = (width2 - width) / 2;
					}
					break;
				}
			}
			// Draw nikud
			pain.text(int(x) + dx, offset + row->baseline(), 
				  textstring, font);
		} else {
			textstring = TransformChar(c, row->par(), pos);
			while (vpos <= last &&
			       (pos = vis2log(vpos)) >= 0
			       && IsPrintableNonspace(c = row->par()->GetChar(pos))
			       && !Encodings::IsComposeChar_arabic(c)
			       && font2 == GetFont(bview->buffer(), row->par(), pos)) {
				c = TransformChar(c, row->par(), pos);
				textstring += c;
				++vpos;
			}
			// Draw text and set the new x position
			pain.text(int(x), offset + row->baseline(),
				  textstring, font);
			x += lyxfont::width(textstring, font);
		}
	} else {
		while (vpos <= last &&
		       (pos = vis2log(vpos)) >= 0
		       && IsPrintableNonspace(c = row->par()->GetChar(pos))
		       && font2 == GetFont(bview->buffer(), row->par(), pos)) {
			textstring += c;
			++vpos;
		}
		// Draw text and set the new x position
		pain.text(int(x), offset + row->baseline(), textstring, font);
		x += lyxfont::width(textstring, font);
	}

	if (lyxrc.mark_foreign_language &&
	    font.language() != bview->buffer()->params.language) {
		int const y = offset + row->height() - 1;
		pain.line(int(tmpx), y, int(x), y,
			  LColor::language);
	}

	// If we want ulem.sty support, drawing
	// routines should go here. (Asger)
	// Why shouldn't LyXFont::drawText handle it internally?
}


// Returns the left beginning of the text. 
// This information cannot be taken from the layouts-objekt, because in 
// LaTeX the beginning of the text fits in some cases (for example sections)
// exactly the label-width.
int LyXText::LeftMargin(BufferView * bview, Row const * row) const
{
	LyXLayout const & layout =
		textclasslist.Style(bview->buffer()->params.textclass,
				    row->par()->GetLayout());
	
	string parindent = layout.parindent; 

	int x = LYX_PAPER_MARGIN;
	
	x += lyxfont::signedWidth(textclasslist
				  .TextClass(bview->buffer()->params.textclass)
				  .leftmargin(),
				  textclasslist
				  .TextClass(bview->buffer()->params.textclass)
				  .defaultfont());
#ifndef NEW_INSETS
	if (row->par()->footnoteflag == LyXParagraph::OPEN_FOOTNOTE)  {
		LyXFont font(LyXFont::ALL_SANE);
		font.setSize(LyXFont::SIZE_SMALL);
		x += lyxfont::width("Mwide-figM", font)
			+ LYX_PAPER_MARGIN/2;
	}
#endif
	// this is the way, LyX handles the LaTeX-Environments.
	// I have had this idea very late, so it seems to be a
	// later added hack and this is true
	if (!row->par()->GetDepth()) {
		if (!row->par()->GetLayout()) {
			// find the previous same level paragraph
#ifndef NEW_INSETS
			if (row->par()->FirstPhysicalPar()->Previous()) {
#else
			if (row->par()->Previous()) {
#endif
				LyXParagraph * newpar = row->par()
					->DepthHook(row->par()->GetDepth());
				if (newpar &&
				    textclasslist.Style(bview->buffer()->params.textclass,
							newpar->GetLayout())
				    .nextnoindent)
					parindent.erase();
			}
		}
	} else {
		// find the next level paragraph
		
		LyXParagraph * newpar =
			row->par()->DepthHook(row->par()->GetDepth()-1);
		
		// make a corresponding row. Needed to call LeftMargin()
		
		// check wether it is a sufficent paragraph 
		if (newpar
#ifndef NEW_INSETS
		    && newpar->footnoteflag == row->par()->footnoteflag
#endif
		    && textclasslist
		        .Style(bview->buffer()->params.textclass, 
			       newpar->GetLayout()).isEnvironment()) {
			Row dummyrow;
			dummyrow.par(newpar);
			dummyrow.pos(newpar->Last());
			x = LeftMargin(bview, &dummyrow);
		} else {
			// this is no longer an error, because this function
			// is used to clear impossible depths after changing
			// a layout. Since there is always a redo,
			// LeftMargin() is always called
#ifndef NEW_INSETS
			row->par()->FirstPhysicalPar()->depth = 0;
#else
			row->par()->depth = 0;
#endif
		}
		
		if (newpar && !row->par()->GetLayout()) {
#ifndef NEW_INSETS
			if (newpar->FirstPhysicalPar()->noindent)
#else
			if (newpar->noindent)
#endif
				parindent.erase();
			else
				parindent = textclasslist
					.Style(bview->buffer()->params.textclass, 
					       newpar->GetLayout()).parindent;
		}
		
	}
	
	LyXFont const labelfont = GetFont(bview->buffer(), row->par(), -2);
	switch (layout.margintype) {
	case MARGIN_DYNAMIC:
		if (!layout.leftmargin.empty()) {
			x += lyxfont::signedWidth(layout.leftmargin,
						  textclasslist
						  .TextClass(bview->buffer()->params.
							     textclass)
						  .defaultfont());
		}
		if (!row->par()->GetLabelstring().empty()) {
			x += lyxfont::signedWidth(layout.labelindent,
						  labelfont);
			x += lyxfont::width(row->par()->GetLabelstring(),
					    labelfont);
			x += lyxfont::width(layout.labelsep, labelfont);
		}
		break;
	case MARGIN_MANUAL:
		x += lyxfont::signedWidth(layout.labelindent, labelfont);
		if (row->pos() >= BeginningOfMainBody(bview->buffer(), row->par())) {
			if (!row->par()->GetLabelWidthString().empty()) {
				x += lyxfont::width(row->par()->GetLabelWidthString(),
					       labelfont);
				x += lyxfont::width(layout.labelsep, labelfont);
			}
		}
		break;
	case MARGIN_STATIC:
		x += lyxfont::signedWidth(layout.leftmargin, textclasslist.TextClass(bview->buffer()->params.textclass).defaultfont()) * 4
			/ (row->par()->GetDepth() + 4);
		break;
	case MARGIN_FIRST_DYNAMIC:
		if (layout.labeltype == LABEL_MANUAL) {
			if (row->pos() >= BeginningOfMainBody(bview->buffer(), row->par())) {
				x += lyxfont::signedWidth(layout.leftmargin,
							  labelfont);
			} else {
				x += lyxfont::signedWidth(layout.labelindent,
							  labelfont);
			}
		} else if (row->pos()
			   // Special case to fix problems with
			   // theorems (JMarc)
			   || (layout.labeltype == LABEL_STATIC
			       && layout.latextype == LATEX_ENVIRONMENT
			       && ! row->par()->IsFirstInSequence())) {
			x += lyxfont::signedWidth(layout.leftmargin,
						  labelfont);
		} else if (layout.labeltype != LABEL_TOP_ENVIRONMENT
			   && layout.labeltype != LABEL_BIBLIO
			   && layout.labeltype !=
			   LABEL_CENTERED_TOP_ENVIRONMENT) {
			x += lyxfont::signedWidth(layout.labelindent,
						  labelfont);
			x += lyxfont::width(layout.labelsep, labelfont);
			x += lyxfont::width(row->par()->GetLabelstring(),
					    labelfont);
		} 
		break;
		
	case MARGIN_RIGHT_ADDRESS_BOX:
	{
		// ok, a terrible hack. The left margin depends on the widest
		// row in this paragraph. Do not care about footnotes, they
		// are *NOT* allowed in the LaTeX realisation of this layout.
		
		// find the first row of this paragraph
		Row const * tmprow = row;
		while (tmprow->previous()
		       && tmprow->previous()->par() == row->par())
			tmprow = tmprow->previous();
		
		int minfill = tmprow->fill();
		while (tmprow->next() && tmprow->next()->par() == row->par()) {
			tmprow = tmprow->next();
			if (tmprow->fill() < minfill)
				minfill = tmprow->fill();
		}
		
		x += lyxfont::signedWidth(layout.leftmargin,
					  textclasslist
					  .TextClass(bview->buffer()->params.textclass)
					  .defaultfont());
		x += minfill;
	}
	break;
	}
	if ((workWidth(bview) > 0) &&
	    (row->par()->pextra_type == LyXParagraph::PEXTRA_INDENT)) {
		if (!row->par()->pextra_widthp.empty()) {
			x += workWidth(bview) *
				lyx::atoi(row->par()->pextra_widthp) / 100;
		} else if (!row->par()->pextra_width.empty()) {
			int xx = VSpace(row->par()->pextra_width).inPixels(bview);
			if (xx > workWidth(bview))
				xx = workWidth(bview) * 80 / 100;
			x += xx;
		} else { // should not happen
			LyXFont font(LyXFont::ALL_SANE);
			x += lyxfont::width("XXXXXX", font);
		}
	}
	
	int align; // wrong type
#ifndef NEW_INSETS
	if (row->par()->FirstPhysicalPar()->align == LYX_ALIGN_LAYOUT)
		align = layout.align;
	else
		align = row->par()->FirstPhysicalPar()->align;
#else
	if (row->par()->align == LYX_ALIGN_LAYOUT)
		align = layout.align;
	else
		align = row->par()->align;
#endif	
	// set the correct parindent
	if (row->pos() == 0) {
		if ((layout.labeltype == LABEL_NO_LABEL 
		     || layout.labeltype == LABEL_TOP_ENVIRONMENT 
		     || layout.labeltype == LABEL_CENTERED_TOP_ENVIRONMENT
		     || (layout.labeltype == LABEL_STATIC
			 && layout.latextype == LATEX_ENVIRONMENT
			 && ! row->par()->IsFirstInSequence()))
#ifndef NEW_INSETS
		    && row->par() == row->par()->FirstPhysicalPar()
#endif
		    && align == LYX_ALIGN_BLOCK
		    && !row->par()->noindent
		    && (row->par()->layout ||
			bview->buffer()->params.paragraph_separation ==
			BufferParams::PARSEP_INDENT))
			x += lyxfont::signedWidth(parindent,
						  textclasslist
						  .TextClass(bview->buffer()->params
							     .textclass)
						  .defaultfont());
		else if (layout.labeltype == LABEL_BIBLIO) {
		       	// ale970405 Right width for bibitems
			x += bibitemMaxWidth(bview,textclasslist
					     .TextClass(bview->buffer()->params
							.textclass)
					     .defaultfont());
		}
	}
	return x;
}


int LyXText::RightMargin(Buffer const * buf, Row const * row) const
{
	LyXLayout const & layout =
		textclasslist.Style(buf->params.textclass,
				    row->par()->GetLayout());
	
	int x = LYX_PAPER_MARGIN
		+ lyxfont::signedWidth(textclasslist
				       .TextClass(buf->params.textclass)
				       .rightmargin(),
				       textclasslist
				       .TextClass(buf->params.textclass)
				       .defaultfont());

#ifndef NEW_INSETS
	if (row->par()->footnoteflag == LyXParagraph::OPEN_FOOTNOTE)  {
		x += LYX_PAPER_MARGIN / 2;
	}
#endif
	// this is the way, LyX handles the LaTeX-Environments.
	// I have had this idea very late, so it seems to be a
	// later added hack and this is true
	if (row->par()->GetDepth()) {
		// find the next level paragraph
		
		LyXParagraph * newpar = row->par();
		
#ifndef NEW_INSETS
		do {
			newpar = newpar->FirstPhysicalPar()->Previous();
			if (newpar) 
				newpar = newpar->FirstPhysicalPar();
		} while (newpar && newpar->GetDepth() >= row->par()->GetDepth()
			 && newpar->footnoteflag == row->par()->footnoteflag);
#else
		do {
			newpar = newpar->Previous();
		} while (newpar
			 && newpar->GetDepth() >= row->par()->GetDepth());
#endif
		
		// make a corresponding row. Needed to call LeftMargin()
		
		// check wether it is a sufficent paragraph
		if (newpar
#ifndef NEW_INSETS
		    && newpar->footnoteflag == row->par()->footnoteflag
#endif
		    && textclasslist.Style(buf->params.textclass,
					   newpar->GetLayout())
		       .isEnvironment()) {
			Row dummyrow;
			dummyrow.par(newpar);
			dummyrow.pos(0);
			x = RightMargin(buf, &dummyrow);
		} else {
			// this is no longer an error, because this function
			// is used to clear impossible depths after changing
			// a layout. Since there is always a redo,
			// LeftMargin() is always called
#ifndef NEW_INSETS
			row->par()->FirstPhysicalPar()->depth = 0;
#else
			row->par()->depth = 0;
#endif
		}
	}
	
	//lyxerr << "rightmargin: " << layout->rightmargin << endl;
	x += lyxfont::signedWidth(layout.rightmargin, textclasslist
				  .TextClass(buf->params.textclass)
				  .defaultfont()) * 4 / (row->par()->GetDepth()
							 + 4);
	return x;
}


int LyXText::LabelEnd(BufferView * bview, Row const * row) const
{
	if (textclasslist.Style(bview->buffer()->params.textclass,
				row->par()->GetLayout()).margintype
	    == MARGIN_MANUAL) {
		Row tmprow;
		tmprow = *row;
		tmprow.pos(row->par()->Last());
		return LeftMargin(bview, &tmprow);  /* just the beginning 
						of the main body */
	} else
		return 0;  /* LabelEnd is only needed, if the  
			      layout fills a flushleft
			      label. */
}


// get the next breakpoint in a given paragraph
LyXParagraph::size_type
LyXText::NextBreakPoint(BufferView * bview, Row const * row, int width) const
{
	LyXParagraph * par = row->par();

	if (width < 0)
		return par->Last();

	LyXParagraph::size_type const pos = row->pos();


	// position of the last possible breakpoint 
	// -1 isn't a suitable value, but a flag
	LyXParagraph::size_type last_separator = -1;
	width -= RightMargin(bview->buffer(), row);
	
	LyXParagraph::size_type const main_body =
		BeginningOfMainBody(bview->buffer(), par);
	LyXLayout const & layout =
		textclasslist.Style(bview->buffer()->params.textclass, par->GetLayout());
	LyXParagraph::size_type i = pos;

	if (layout.margintype == MARGIN_RIGHT_ADDRESS_BOX) {
		/* special code for right address boxes, only newlines count */
		while (i < par->Last()) {
			if (par->IsNewline(i)) {
				last_separator = i;
				i = par->Last() - 1; // this means break
				//x = width;
			} else if (par->GetChar(i) == LyXParagraph::META_INSET &&
				   par->GetInset(i) && par->GetInset(i)->display()){
				par->GetInset(i)->display(false);
			}
			++i;
		}
	} else {
		// Last position is an invariant
		LyXParagraph::size_type const last = 
			par->Last();
		// this is the usual handling
		int x = LeftMargin(bview, row);
		bool doitonetime = true;
		while (doitonetime || ((x < width) && (i < last))) {
			doitonetime = false;
			char const c = par->GetChar(i);
			if (IsNewlineChar(c)) {
				last_separator = i;
				x = width; // this means break
			} else if (c == LyXParagraph::META_INSET &&
				   par->GetInset(i)) {
				
				// check wether a Display() inset is
				// valid here. if not, change it to
				// non-display
				if (par->GetInset(i)->display() &&
				    (layout.isCommand() ||
				     (layout.labeltype == LABEL_MANUAL
				      && i < BeginningOfMainBody(bview->buffer(), par)))) {
					// display istn't allowd
					par->GetInset(i)->display(false);
					x += SingleWidth(bview, par, i, c);
				} else if (par->GetInset(i)->display() ||
					 par->GetInset(i)->needFullRow()) {
					// So break the line here
					if (i == pos) {
						if (pos < last-1) {
							last_separator = i;
							if (IsLineSeparatorChar(par->GetChar(i+1)))
								++last_separator;
						} else
							last_separator = last; // to avoid extra rows
					} else
						last_separator = i - 1;
					x = width;  // this means break
				} else {
#if 0
					last_separator = i;
					x += width;
#else
					x += SingleWidth(bview, par, i, c);
#endif
				}
			} else  {
				if (IsLineSeparatorChar(c))
					last_separator = i;
				x += SingleWidth(bview, par, i, c);
			}
			++i;
			if (i == main_body) {
				x += lyxfont::width(layout.labelsep,
						    GetFont(bview->buffer(), par, -2));
				if (par->IsLineSeparator(i - 1))
					x-= SingleWidth(bview, par, i - 1);
				int left_margin = LabelEnd(bview, row);
				if (x < left_margin)
					x = left_margin;
			}
		}
		// end of paragraph is always a suitable separator
		if (i == last && x < width)
			last_separator = i;
	}
	
	// well, if last_separator is still 0, the line isn't breakable. 
	// don't care and cut simply at the end
	if (last_separator < 0) {
		last_separator = i;
	}
	
	// manual labels cannot be broken in LaTeX, do not care
	if (main_body && last_separator < main_body)
		last_separator = main_body - 1;
	
	return last_separator;
}


// returns the minimum space a row needs on the screen in pixel
int LyXText::Fill(BufferView * bview, Row * row, int paper_width) const
{
	if (paper_width < 0)
		return 0;

	int w;
	// get the pure distance
	LyXParagraph::size_type const last = RowLastPrintable(row);
	
	// special handling of the right address boxes
	if (textclasslist.Style(bview->buffer()->params.textclass,
				row->par()->GetLayout()).margintype
	    == MARGIN_RIGHT_ADDRESS_BOX) {
		int const tmpfill = row->fill();
		row->fill(0); // the minfill in MarginLeft()
		w = LeftMargin(bview, row);
		row->fill(tmpfill);
	} else
		w = LeftMargin(bview, row);
	
	LyXLayout const & layout = textclasslist.Style(bview->buffer()->params.textclass,
						       row->par()->GetLayout());
	LyXParagraph::size_type const main_body = 
		BeginningOfMainBody(bview->buffer(), row->par());
	LyXParagraph::size_type i = row->pos();

	while (i <= last) {
		if (main_body > 0 && i == main_body) {
			w += lyxfont::width(layout.labelsep, GetFont(bview->buffer(), row->par(), -2));
			if (row->par()->IsLineSeparator(i - 1))
				w -= SingleWidth(bview, row->par(), i - 1);
			int left_margin = LabelEnd(bview, row);
			if (w < left_margin)
				w = left_margin;
		}
		w += SingleWidth(bview, row->par(), i);
		++i;
	}
	if (main_body > 0 && main_body > last) {
		w += lyxfont::width(layout.labelsep, GetFont(bview->buffer(), row->par(), -2));
		if (last >= 0 && row->par()->IsLineSeparator(last))
			w -= SingleWidth(bview, row->par(), last);
		int const left_margin = LabelEnd(bview, row);
		if (w < left_margin)
			w = left_margin;
	}
	
	int const fill = paper_width - w - RightMargin(bview->buffer(), row);
	return fill;
}


// returns the minimum space a manual label needs on the screen in pixel
int LyXText::LabelFill(BufferView * bview, Row const * row) const
{
	LyXParagraph::size_type last = BeginningOfMainBody(bview->buffer(), row->par()) - 1;
	// -1 because a label ends either with a space that is in the label, 
	// or with the beginning of a footnote that is outside the label.

	// I don't understand this code in depth, but sometimes "last" is
	// less than 0 and this causes a crash. This fix seems to work
	// correctly, but I bet the real error is elsewhere.  The bug is
	// triggered when you have an open footnote in a paragraph
	// environment with a manual label. (Asger)
	if (last < 0) last = 0;
	
	if (row->par()->IsLineSeparator(last)) /* a sepearator at this end 
						does not count */
		--last;
	
	int w = 0;
	int i = row->pos();
	while (i <= last) {
		w += SingleWidth(bview, row->par(), i);
		++i;
	}
	
	int fill = 0;
	if (!row->par()->labelwidthstring.empty()) {
		fill = max(lyxfont::width(row->par()->labelwidthstring,
					  GetFont(bview->buffer(), row->par(), -2)) - w,
			   0);
	}
	
	return fill;
}


// returns the number of separators in the specified row. The separator 
// on the very last column doesnt count
int LyXText::NumberOfSeparators(Buffer const * buf, Row const * row) const
{
	LyXParagraph::size_type const last = RowLast(row);
	LyXParagraph::size_type p =
		max(row->pos(), BeginningOfMainBody(buf, row->par()));
	int n = 0;
	for (; p < last; ++p) {
		if (row->par()->IsSeparator(p)) {
			++n;
		}
	}
	return n;
}


// returns the number of hfills in the specified row. The LyX-Hfill is
// a LaTeX \hfill so that the hfills at the beginning and at the end were 
// ignored. This is *MUCH* more usefull than not to ignore!
int LyXText::NumberOfHfills(Buffer const * buf, Row const * row) const
{
	LyXParagraph::size_type const last = RowLast(row);
	LyXParagraph::size_type first = row->pos();
	if (first) { /* hfill *DO* count at the beginning 
		      * of paragraphs! */
		while(first <= last && row->par()->IsHfill(first))
			++first;
	}

	first = max(first, BeginningOfMainBody(buf, row->par()));
	int n = 0;
	for (int p = first; p <= last; ++p) { // last, because the end is ignored!
		if (row->par()->IsHfill(p)) {
			++n;
		}
	}
	return n;
}


// like NumberOfHfills, but only those in the manual label!
int LyXText::NumberOfLabelHfills(Buffer const * buf, Row const * row) const
{
	LyXParagraph::size_type last = RowLast(row);
	LyXParagraph::size_type first = row->pos();
	if (first) { /* hfill *DO* count at the beginning 
		      * of paragraphs! */
		while(first < last && row->par()->IsHfill(first))
			++first;
	}

	last = min(last, BeginningOfMainBody(buf, row->par()));
	int n = 0;
	for (LyXParagraph::size_type p = first;
	     p < last; ++p) {  // last, because the end is ignored!
		if (row->par()->IsHfill(p)) {
			++n;
		}
	}
	return n;
}


// returns true, if a expansion is needed.
// Rules are given by LaTeX
bool LyXText::HfillExpansion(Buffer const * buf, Row const * row_ptr,
			     LyXParagraph::size_type pos) const
{
	// by the way, is it a hfill?
	if (!row_ptr->par()->IsHfill(pos))
		return false;
	
	// at the end of a row it does not count
	if (pos >= RowLast(row_ptr))
		return false;
	
	// at the beginning of a row it does not count, if it is not 
	// the first row of a paragaph
	if (!row_ptr->pos())
		return true;
	
	// in some labels  it does not count
	if (textclasslist.Style(buf->params.textclass,
				row_ptr->par()->GetLayout()).margintype
	    != MARGIN_MANUAL
	    && pos < BeginningOfMainBody(buf, row_ptr->par()))
		return false; 
	
	// if there is anything between the first char of the row and
	// the sepcified position that is not a newline and not a hfill,
	// the hfill will count, otherwise not
	LyXParagraph::size_type i = row_ptr->pos();
	while (i < pos && (row_ptr->par()->IsNewline(i)
			   || row_ptr->par()->IsHfill(i)))
		++i;
	
	return i != pos;
}


void LyXText::SetHeightOfRow(BufferView * bview, Row * row_ptr) const
{
    /* get the maximum ascent and the maximum descent */
   int asc = 0;
   int desc = 0;
   float layoutasc = 0;
   float layoutdesc = 0;
   float tmptop = 0;
   LyXFont tmpfont;
   Inset * tmpinset = 0;

   /* this must not happen before the currentrow for clear reasons.
      so the trick is just to set the current row onto this row */
   int unused_y;
   GetRow(row_ptr->par(), row_ptr->pos(), unused_y);

   /* ok , let us initialize the maxasc and maxdesc value. 
    * This depends in LaTeX of the font of the last character
    * in the paragraph. The hack below is necessary because
    * of the possibility of open footnotes */

   /* Correction: only the fontsize count. The other properties
      are taken from the layoutfont. Nicer on the screen :) */
#ifndef NEW_INSETS   
   LyXParagraph * par = row_ptr->par()->LastPhysicalPar();
   LyXParagraph * firstpar = row_ptr->par()->FirstPhysicalPar();
#else
   LyXParagraph * par = row_ptr->par();
   LyXParagraph * firstpar = row_ptr->par();
#endif
   
   LyXLayout const & layout = textclasslist.Style(bview->buffer()->params.textclass,
						  firstpar->GetLayout());
   
   LyXFont font = GetFont(bview->buffer(), par, par->Last() - 1);
   LyXFont::FONT_SIZE const size = font.size();
   font = GetFont(bview->buffer(), par, -1);
   font.setSize(size);

   LyXFont labelfont = GetFont(bview->buffer(), par, -2);

   float spacing_val = 1.0;
   if (!row_ptr->par()->spacing.isDefault()) {
	   spacing_val = row_ptr->par()->spacing.getValue();
   } else {
	   spacing_val = bview->buffer()->params.spacing.getValue();
   }
   //lyxerr << "spacing_val = " << spacing_val << endl;
   
   int maxasc = int(lyxfont::maxAscent(font) *
		   layout.spacing.getValue() *
		   spacing_val);
   int maxdesc = int(lyxfont::maxDescent(font) *
		    layout.spacing.getValue() *
		    spacing_val);
   int const pos_end = RowLast(row_ptr);
   int labeladdon = 0;
   int maxwidth = 0;

   // Check if any insets are larger
   for (int pos = row_ptr->pos(); pos <= pos_end; ++pos) {
	   if (row_ptr->par()->GetChar(pos) == LyXParagraph::META_INSET) {
		   tmpfont = GetFont(bview->buffer(), row_ptr->par(), pos);
		   tmpinset = row_ptr->par()->GetInset(pos);
		   if (tmpinset) {
			   tmpinset->update(bview, tmpfont);
			   asc = tmpinset->ascent(bview, tmpfont);
			   desc = tmpinset->descent(bview, tmpfont);
			   maxwidth += tmpinset->width(bview, tmpfont);
			   maxasc = max(maxasc, asc);
			   maxdesc = max(maxdesc, desc);
		   }
	   } else {
		   maxwidth += SingleWidth(bview, row_ptr->par(), pos);
	   }
   }

   // Check if any custom fonts are larger (Asger)
   // This is not completely correct, but we can live with the small,
   // cosmetic error for now.
   LyXFont::FONT_SIZE const maxsize =
	   row_ptr->par()->HighestFontInRange(row_ptr->pos(),
					      pos_end);
   if (maxsize > font.size()) {
	font.setSize(maxsize);

	asc = lyxfont::maxAscent(font);
	desc = lyxfont::maxDescent(font);
	if (asc > maxasc) 
		maxasc = asc;
	if (desc > maxdesc)
		maxdesc = desc;
   }

   // This is nicer with box insets:
   ++maxasc;
   ++maxdesc;

   row_ptr->ascent_of_text(maxasc);
   
   /* is it a top line? */ 
   if (!row_ptr->pos() && (row_ptr->par() == firstpar)) {
      
      /* some parksips VERY EASY IMPLEMENTATION */ 
      if (bview->buffer()->params.paragraph_separation ==
	  BufferParams::PARSEP_SKIP) {
	 if (layout.isParagraph()
	     && firstpar->GetDepth() == 0
	     && firstpar->Previous())
	    maxasc += bview->buffer()->params.getDefSkip().inPixels(bview);
	 else if (firstpar->Previous()
		  && textclasslist.Style(bview->buffer()->params.textclass,
			   firstpar->Previous()->GetLayout()).isParagraph()
		  && firstpar->Previous()->GetDepth() == 0)
	   // is it right to use defskip here too? (AS)
	   maxasc += bview->buffer()->params.getDefSkip().inPixels(bview);
      }
      
      /* the paper margins */ 
      if (!row_ptr->par()->previous && bv_owner)
      	 maxasc += LYX_PAPER_MARGIN;
      
      /* add the vertical spaces, that the user added */
      if (firstpar->added_space_top.kind() != VSpace::NONE)
      	 maxasc += int(firstpar->added_space_top.inPixels(bview));
      
      /* do not forget the DTP-lines! 
       * there height depends on the font of the nearest character */
      if (firstpar->line_top)
      	 maxasc += 2 * lyxfont::ascent('x', GetFont(bview->buffer(),
						    firstpar, 0));
      
      /* and now the pagebreaks */ 
      if (firstpar->pagebreak_top)
      	 maxasc += 3 * DefaultHeight();
      
      /*  this is special code for the chapter, since the label of this
       * layout is printed in an extra row */ 
      if (layout.labeltype == LABEL_COUNTER_CHAPTER
	  && bview->buffer()->params.secnumdepth >= 0) {
	      float spacing_val = 1.0;
	      if (!row_ptr->par()->spacing.isDefault()) {
		      spacing_val = row_ptr->par()->spacing.getValue();
	      } else {
		      spacing_val = bview->buffer()->params.spacing.getValue();
	      }
	      
	      labeladdon = int(lyxfont::maxDescent(labelfont) *
			       layout.spacing.getValue() *
			       spacing_val)
		      + int(lyxfont::maxAscent(labelfont) *
			    layout.spacing.getValue() *
			    spacing_val);
      }
      
      /* special code for the top label */ 
      if ((layout.labeltype == LABEL_TOP_ENVIRONMENT
	   || layout.labeltype == LABEL_BIBLIO
	   || layout.labeltype == LABEL_CENTERED_TOP_ENVIRONMENT)
	  && row_ptr->par()->IsFirstInSequence()
	  && !row_ptr->par()->GetLabelstring().empty()) {
	      float spacing_val = 1.0;
	      if (!row_ptr->par()->spacing.isDefault()) {
		      spacing_val = row_ptr->par()->spacing.getValue();
	      } else {
		      spacing_val = bview->buffer()->params.spacing.getValue();
	      }
	      
	      labeladdon = int(
		      (lyxfont::maxAscent(labelfont) *
		       layout.spacing.getValue() *
		       spacing_val)
		      +(lyxfont::maxDescent(labelfont) *
			layout.spacing.getValue() *
			spacing_val)
		      + layout.topsep * DefaultHeight()
		      + layout.labelbottomsep *  DefaultHeight());
      }
   
      /* and now the layout spaces, for example before and after a section, 
       * or between the items of a itemize or enumerate environment */ 
      
      if (!firstpar->pagebreak_top) {
	 LyXParagraph * prev = row_ptr->par()->Previous();
	 if (prev)
	    prev = row_ptr->par()->DepthHook(row_ptr->par()->GetDepth());
	 if (prev && prev->GetLayout() == firstpar->GetLayout()
	     && prev->GetDepth() == firstpar->GetDepth()
	     && prev->GetLabelWidthString() == firstpar->GetLabelWidthString())
	   {
	      layoutasc = (layout.itemsep * DefaultHeight());
	   }
	 else if (row_ptr->previous()) {
	    tmptop = layout.topsep;
	    
	    if (row_ptr->previous()->par()->GetDepth() >= row_ptr->par()->GetDepth())
	       tmptop -= textclasslist.Style(bview->buffer()->params.textclass,
					     row_ptr->previous()->par()->
					     GetLayout()).bottomsep;
	    
	    if (tmptop > 0)
	       layoutasc = (tmptop * DefaultHeight());
	 }
	 else if (row_ptr->par()->line_top){
	    tmptop = layout.topsep;
	    
	    if (tmptop > 0)
	       layoutasc = (tmptop * DefaultHeight());
	 }
	 
	 prev = row_ptr->par()->DepthHook(row_ptr->par()->GetDepth()-1);
	 if (prev)  {
	    maxasc += int(textclasslist.Style(bview->buffer()->params.textclass,
					 prev->GetLayout()).parsep * DefaultHeight());
	 }
	 else {
		if (firstpar->Previous()
		    && firstpar->Previous()->GetDepth() == 0
		    && firstpar->Previous()->GetLayout() != firstpar->GetLayout()) {
		   /* avoid parsep */ 
		}
	    else if (firstpar->Previous()){
	       maxasc += int(layout.parsep * DefaultHeight());
	    }
	 }
      }
   }
   
   /* is it a bottom line? */ 
   if (
#ifndef NEW_INSETS
	   row_ptr->par()->ParFromPos(RowLast(row_ptr) + 1) == par
#else
	   row_ptr->par() == par
#endif
       && (!row_ptr->next() || row_ptr->next()->par() != row_ptr->par())) {     
	  
	  /* the paper margins */ 
	  if (!par->next && bv_owner)
	    maxdesc += LYX_PAPER_MARGIN;
	
	  /* add the vertical spaces, that the user added */
	  if (firstpar->added_space_bottom.kind() != VSpace::NONE)
		  maxdesc += int(firstpar->added_space_bottom.inPixels(bview));
	  
	  /* do not forget the DTP-lines! 
	   * there height depends on the font of the nearest character */
	  if (firstpar->line_bottom)
		  maxdesc += 2 * lyxfont::ascent('x', GetFont(bview->buffer(),
							      par, par->Last() - 1));
	  
	  /* and now the pagebreaks */
	  if (firstpar->pagebreak_bottom)
	    maxdesc += 3 * DefaultHeight();
	  
	  /* and now the layout spaces, for example before and after a section, 
	   * or between the items of a itemize or enumerate environment */
	  if (!firstpar->pagebreak_bottom && row_ptr->par()->Next()) {
	     LyXParagraph * nextpar = row_ptr->par()->Next();
	     LyXParagraph * comparepar = row_ptr->par();
	     float usual = 0;
	     float unusual = 0;
	     
	     if (comparepar->GetDepth() > nextpar->GetDepth()) {
		usual = (textclasslist.Style(bview->buffer()->params.textclass, comparepar->GetLayout()).bottomsep * DefaultHeight());
		comparepar = comparepar->DepthHook(nextpar->GetDepth());
		if (comparepar->GetLayout()!= nextpar->GetLayout()
		    || nextpar->GetLabelWidthString() != 
		    	comparepar->GetLabelWidthString())
		  unusual = (textclasslist.Style(bview->buffer()->params.textclass, comparepar->GetLayout()).bottomsep * DefaultHeight());
		
		if (unusual > usual)
		  layoutdesc = unusual;
		else
		  layoutdesc = usual;
	     }
	     else if (comparepar->GetDepth() ==  nextpar->GetDepth()) {
		
		if (comparepar->GetLayout()!= nextpar->GetLayout()
		    || nextpar->GetLabelWidthString() != 
			comparepar->GetLabelWidthString())
		  layoutdesc = int(textclasslist.Style(bview->buffer()->params.textclass, comparepar->GetLayout()).bottomsep * DefaultHeight());
	     }
	  }
       }
   
   /* incalculate the layout spaces */ 
   maxasc += int(layoutasc * 2 / (2 + firstpar->GetDepth()));
   maxdesc += int(layoutdesc * 2 / (2 + firstpar->GetDepth()));

   /* calculate the new height of the text */ 
   height -= row_ptr->height();
   
   row_ptr->height(maxasc + maxdesc + labeladdon);
   row_ptr->baseline(maxasc + labeladdon);
   
   height += row_ptr->height();
   float x, dummy;
   PrepareToPrint(bview, row_ptr, x, dummy, dummy, dummy, false);
   row_ptr->width(int(maxwidth + x));
   if (inset_owner) {
	   Row * r = firstrow;
	   width = max(0,workWidth(bview));
	   while(r) {
		   if (r->width() > width)
			   width = r->width();
		   r = r->next();
	   }
   }
}


/* Appends the implicit specified paragraph behind the specified row,
 * start at the implicit given position */
void LyXText::AppendParagraph(BufferView * bview, Row * row) const
{
   bool not_ready = true;
   
   // The last character position of a paragraph is an invariant so we can 
   // safely get it here. (Asger)
   int const lastposition = row->par()->Last();

   do {
      // Get the next breakpoint
      int z = NextBreakPoint(bview, row, workWidth(bview));
      
      Row * tmprow = row;

      // Insert the new row
      if (z < lastposition) {
	 ++z;
	 InsertRow(row, row->par(), z);
	 row = row->next();

	 row->height(0);
      } else
	 not_ready = false;
      
      // Set the dimensions of the row
      tmprow->fill(Fill(bview, tmprow, workWidth(bview)));
      SetHeightOfRow(bview, tmprow);

   } while (not_ready);
}


void LyXText::BreakAgain(BufferView * bview, Row * row) const
{
   bool not_ready = true;
   
   do  {
      /* get the next breakpoint */
	LyXParagraph::size_type z = NextBreakPoint(bview, row, workWidth(bview));
      Row * tmprow = row;
      
      if (z < row->par()->Last()) {
	 if (!row->next() || (row->next() && row->next()->par() != row->par())) {
		 // insert a new row
	    ++z;
	    InsertRow(row, row->par(), z);
	    row = row->next();
	    row->height(0);
	 } else  {
	    row = row->next();
	    ++z;
	    if (row->pos() == z)
		    not_ready = false;     // the rest will not change
	    else {
	       row->pos(z);
	    }
	 }
      } else {
	 /* if there are some rows too much, delete them */
	 /* only if you broke the whole paragraph! */ 
	 Row * tmprow2 = row;
	 while (tmprow2->next() && tmprow2->next()->par() == row->par()) {
	    tmprow2 = tmprow2->next();
	 }
	 while (tmprow2 != row) {
	    tmprow2 = tmprow2->previous();
	    RemoveRow(tmprow2->next());
	 }
	 not_ready = false;
      }
       
      /* set the dimensions of the row */ 
      tmprow->fill(Fill(bview, tmprow, workWidth(bview)));
      SetHeightOfRow(bview, tmprow);
   } while (not_ready);
}


// this is just a little changed version of break again
void LyXText::BreakAgainOneRow(BufferView * bview, Row * row)
{
	// get the next breakpoint
	LyXParagraph::size_type z = NextBreakPoint(bview, row, workWidth(bview));
	Row * tmprow = row;
	
	if (z < row->par()->Last()) {
		if (!row->next()
		    || (row->next() && row->next()->par() != row->par())) {
			/* insert a new row */ 
			++z;
			InsertRow(row, row->par(), z);
			row = row->next();
			row->height(0);
		} else  {
			row= row->next();
			++z;
			if (row->pos() != z)
				row->pos(z);
		}
	} else {
		// if there are some rows too much, delete them
		// only if you broke the whole paragraph!
		Row * tmprow2 = row;
		while (tmprow2->next()
		       && tmprow2->next()->par() == row->par()) {
			tmprow2 = tmprow2->next();
		}
		while (tmprow2 != row) {
			tmprow2 = tmprow2->previous();
			RemoveRow(tmprow2->next());
		}
	}
	
	// set the dimensions of the row
	tmprow->fill(Fill(bview, tmprow, workWidth(bview)));
	SetHeightOfRow(bview, tmprow);
}


void LyXText::BreakParagraph(BufferView * bview, char keep_layout)
{
   LyXLayout const & layout =
	   textclasslist.Style(bview->buffer()->params.textclass,
			       cursor.par()->GetLayout());

   // this is only allowed, if the current paragraph is not empty or caption
   if ((cursor.par()->Last() <= 0
#ifndef NEW_INSETS
	&& !cursor.par()->IsDummy()
#endif
	   )
       && 
       layout.labeltype!= LABEL_SENSITIVE)
     return;

   SetUndo(bview->buffer(), Undo::INSERT,
#ifndef NEW_INSETS
	   cursor.par()->ParFromPos(cursor.pos())->previous, 
	   cursor.par()->ParFromPos(cursor.pos())->next
#else
	   cursor.par()->previous, 
	   cursor.par()->next
#endif
	   ); 

   // Always break behind a space
   //
   // It is better to erase the space (Dekel)
   if (cursor.pos() < cursor.par()->Last()
       && cursor.par()->IsLineSeparator(cursor.pos()))
	   cursor.par()->Erase(cursor.pos());
           // cursor.pos(cursor.pos() + 1);

   // break the paragraph
   if (keep_layout)
     keep_layout = 2;
   else	
     keep_layout = layout.isEnvironment();
   cursor.par()->BreakParagraph(bview->buffer()->params, cursor.pos(),
				keep_layout);

   // well this is the caption hack since one caption is really enough
   if (layout.labeltype == LABEL_SENSITIVE) {
     if (!cursor.pos())
	     // set to standard-layout
	     cursor.par()->SetLayout(bview->buffer()->params, 0);
     else
	     // set to standard-layout
	     cursor.par()->Next()->SetLayout(bview->buffer()->params, 0);
   }
   
   /* if the cursor is at the beginning of a row without prior newline, 
    * move one row up! 
    * This touches only the screen-update. Otherwise we would may have
    * an empty row on the screen */
   if (cursor.pos() && !cursor.row()->par()->IsNewline(cursor.row()->pos() - 1)
       && cursor.row()->pos() == cursor.pos()) {
	   CursorLeft(bview);
   } 
   
   status = LyXText::NEED_MORE_REFRESH;
   refresh_row = cursor.row();
   refresh_y = cursor.y() - cursor.row()->baseline();
   
   // Do not forget the special right address boxes
   if (layout.margintype == MARGIN_RIGHT_ADDRESS_BOX) {
      while (refresh_row->previous() &&
	     refresh_row->previous()->par() == refresh_row->par()) {
	      refresh_row = refresh_row->previous();
	      refresh_y -= refresh_row->height();
      }
   }
   RemoveParagraph(cursor.row());
   
   // set the dimensions of the cursor row
   cursor.row()->fill(Fill(bview, cursor.row(), workWidth(bview)));

   SetHeightOfRow(bview, cursor.row());
   
   while (cursor.par()->Next()->Last()
	  && cursor.par()->Next()->IsNewline(0))
	   cursor.par()->Next()->Erase(0);
   
   InsertParagraph(bview, cursor.par()->Next(), cursor.row());

   UpdateCounters(bview, cursor.row()->previous());
   
   /* This check is necessary. Otherwise the new empty paragraph will
    * be deleted automatically. And it is more friendly for the user! */ 
   if (cursor.pos())
	   SetCursor(bview, cursor.par()->Next(), 0);
   else
	   SetCursor(bview, cursor.par(), 0);
   
   if (cursor.row()->next())
	   BreakAgain(bview, cursor.row()->next());

   need_break_row = 0;
}


#ifndef NEW_INSETS
void LyXText::OpenFootnote(BufferView * bview)
{
   LyXParagraph * endpar,* tmppar;
   Row * row;
   
   LyXParagraph * par = cursor.par()->ParFromPos(cursor.pos());
   
   /* if there is no footnote in this paragraph, just return. */ 
   if (!par->next
       || par->next->footnoteflag != LyXParagraph::CLOSED_FOOTNOTE)
     return;
   
   /* ok, move the cursor right before the footnote */ 
   
   /* just a little faster than using CursorRight() */
   for (cursor.pos(0);
	cursor.par()->ParFromPos(cursor.pos()) != par;) {
	   cursor.pos(cursor.pos() + 1);
   }
   
   /* now the cursor is at the beginning of the physical par */
   SetCursor(bview, cursor.par(),
	     cursor.pos() + cursor.par()->ParFromPos(cursor.pos())->size());
   
   /* the cursor must be exactly before the footnote */ 
   par = cursor.par()->ParFromPos(cursor.pos());
   
   status = LyXText::NEED_MORE_REFRESH;
   refresh_row = cursor.row();
   refresh_y = cursor.y() - cursor.row()->baseline();
   
   tmppar = cursor.par();
   endpar = cursor.par()->Next();
   row = cursor.row();
   
   tmppar->OpenFootnote(cursor.pos());
   RemoveParagraph(row);
   /* set the dimensions of the cursor row */
   row->fill(Fill(bview, row, workWidth(bview)));
   SetHeightOfRow(bview, row);
   // CHECK See comment on top of text.C
   tmppar = tmppar->Next();
   
   while (tmppar != endpar) {
      if (tmppar) {
	 InsertParagraph(bview, tmppar, row);
	 while (row->next() && row->next()->par() == tmppar)
	   row = row->next();
	 tmppar = tmppar->Next();
      }
   }
   SetCursor(bview, par->next, 0);
   sel_cursor = cursor;
}
#endif


// Just a macro to make some thing easier. 
void LyXText::RedoParagraph(BufferView * bview) const
{
	ClearSelection();
	RedoParagraphs(bview, cursor, cursor.par()->Next());
	SetCursorIntern(bview, cursor.par(), cursor.pos());
}


/* insert a character, moves all the following breaks in the 
 * same Paragraph one to the right and make a rebreak */
void LyXText::InsertChar(BufferView * bview, char c)
{
	SetUndo(bview->buffer(), Undo::INSERT,
#ifndef NEW_INSETS
		cursor.par()->ParFromPos(cursor.pos())->previous, 
		cursor.par()->ParFromPos(cursor.pos())->next
#else
		cursor.par()->previous, 
		cursor.par()->next
#endif
		);

	// When the free-spacing option is set for the current layout,
	// disable the double-space checking

	bool const freeSpacing = 
		textclasslist.Style(bview->buffer()->params.textclass,
			       cursor.row()->par()->GetLayout()).free_spacing;


	if (lyxrc.auto_number) {
		static string const number_operators = "+-/*";
		static string const number_unary_operators = "+-";
		static string const number_seperators = ".,:";

		if (current_font.number() == LyXFont::ON) {
			if (!isdigit(c) && !contains(number_operators, c) &&
			    !(contains(number_seperators, c) &&
			      cursor.pos() >= 1 &&
			      cursor.pos() < cursor.par()->size() &&
			      GetFont(bview->buffer(),
				      cursor.par(),
				      cursor.pos()).number() == LyXFont::ON &&
			      GetFont(bview->buffer(),
				      cursor.par(),
				      cursor.pos()-1).number() == LyXFont::ON)
			    )
				Number(bview); // Set current_font.number to OFF
		} else if (isdigit(c) &&
			   real_current_font.isVisibleRightToLeft()) {
			Number(bview); // Set current_font.number to ON

			if (cursor.pos() > 0) {
				char const c = cursor.par()->GetChar(cursor.pos() - 1);
				if (contains(number_unary_operators, c) &&
				    (cursor.pos() == 1 ||
				     cursor.par()->IsSeparator(cursor.pos() - 2) ||
				     cursor.par()->IsNewline(cursor.pos() - 2) )
				   ) {
					SetCharFont(bview->buffer(),
						    cursor.par(),
						    cursor.pos() - 1,
						    current_font);
				} else if (contains(number_seperators, c) &&
					   cursor.pos() >= 2 &&
					   GetFont(bview->buffer(),
						   cursor.par(),
						   cursor.pos()-2).number() == LyXFont::ON) {
					SetCharFont(bview->buffer(),
						    cursor.par(),
						    cursor.pos() - 1,
						    current_font);
				}
			}
		}
	}


	/* First check, if there will be two blanks together or a blank at 
	  the beginning of a paragraph. 
	  I decided to handle blanks like normal characters, the main 
	  difference are the special checks when calculating the row.fill
	  (blank does not count at the end of a row) and the check here */ 

	// The bug is triggered when we type in a description environment:
	// The current_font is not changed when we go from label to main text
	// and it should (along with realtmpfont) when we type the space.
	// CHECK There is a bug here! (Asger)
	
	LyXFont realtmpfont = real_current_font;
	LyXFont rawtmpfont = current_font;  /* store the current font.
				     * This is because of the use
				     * of cursor movements. The moving
				     * cursor would refresh the 
				     * current font */

	// Get the font that is used to calculate the baselineskip
	LyXParagraph::size_type const lastpos = cursor.par()->Last();
	LyXFont rawparfont = cursor.par()->GetFontSettings(bview->buffer()->params,
							   lastpos - 1);

	bool jumped_over_space = false;
   
	if (!freeSpacing && IsLineSeparatorChar(c)) {
		if ((cursor.pos() > 0 
		     && cursor.par()->IsLineSeparator(cursor.pos() - 1))
		    || (cursor.pos() > 0
			&& cursor.par()->IsNewline(cursor.pos() - 1))
		    || (cursor.pos() == 0
#ifndef NEW_INSETS
			&& !(cursor.par()->Previous()
			     && cursor.par()->Previous()->footnoteflag
			     == LyXParagraph::OPEN_FOOTNOTE)
#endif
			    )) {
	   		if (cursor.pos() == 0 )
				bview->owner()->getMiniBuffer()->Set(_("You cannot insert a space at the beginning of a paragraph.  Please read the Tutorial."));
			else
				bview->owner()->getMiniBuffer()->Set(_("You cannot type two spaces this way.  Please read the Tutorial."));
			charInserted();
			return;
		}
	} else if (IsNewlineChar(c)) {
#ifndef NEW_INSETS
		if (cursor.par()->FirstPhysicalPar() == cursor.par()
#else
		if (cursor.par() == cursor.par()
#endif
		    && cursor.pos() <= BeginningOfMainBody(bview->buffer(), cursor.par())) {
			charInserted();
			return;
		}
		/* No newline at first position 
		 * of a paragraph or behind labels. 
		 * TeX does not allow that. */
		
		if (cursor.pos() < cursor.par()->Last() &&
		    cursor.par()->IsLineSeparator(cursor.pos()))
			// newline always after a blank!
			CursorRight(bview);
		cursor.row()->fill(-1);	       // to force a new break
	}
   
	// the display inset stuff
	if (cursor.row()->par()->GetChar(cursor.row()->pos()) == LyXParagraph::META_INSET
	    && cursor.row()->par()->GetInset(cursor.row()->pos())
	    && (cursor.row()->par()->GetInset(cursor.row()->pos())->display() ||
		cursor.row()->par()->GetInset(cursor.row()->pos())->needFullRow()))
		cursor.row()->fill(-1); // to force a new break  

	// get the cursor row fist
	Row * row = cursor.row();
	int y = cursor.y() - row->baseline();
	if (c != LyXParagraph::META_INSET) /* Here case LyXText::InsertInset 
					    * already insertet the character */
		cursor.par()->InsertChar(cursor.pos(), c);
	SetCharFont(bview->buffer(), cursor.par(), cursor.pos(), rawtmpfont);

	if (!jumped_over_space) {
		// refresh the positions
		Row * tmprow = row;
		while (tmprow->next() && tmprow->next()->par() == row->par()) {
			tmprow = tmprow->next();
			tmprow->pos(tmprow->pos() + 1);
		}
	}
   
	// Is there a break one row above
	if ((cursor.par()->IsLineSeparator(cursor.pos())
	     || cursor.par()->IsNewline(cursor.pos())
	     || cursor.row()->fill() == -1)
	    && row->previous() && row->previous()->par() == row->par()) {
		LyXParagraph::size_type z = NextBreakPoint(bview,
							   row->previous(),
							   workWidth(bview));
		if (z >= row->pos()) {
			row->pos(z + 1);
			
			// set the dimensions of the row above
			row->previous()->fill(Fill(bview,
						   row->previous(),
						   workWidth(bview)));

			SetHeightOfRow(bview, row->previous());
	     
			y -= row->previous()->height();
			refresh_y = y;
			refresh_row = row->previous();
			status = LyXText::NEED_MORE_REFRESH;
	     
			BreakAgainOneRow(bview, row);

			current_font = rawtmpfont;
			real_current_font = realtmpfont;
			SetCursor(bview, cursor.par(), cursor.pos() + 1,
				  false, cursor.boundary());
			// cursor MUST be in row now.
	     
			if (row->next() && row->next()->par() == row->par())
				need_break_row = row->next();
			else
				need_break_row = 0;
	     
			// check, wether the last characters font has changed. 
			if (cursor.pos() && cursor.pos() == cursor.par()->Last()
			    && rawparfont != rawtmpfont)
				RedoHeightOfParagraph(bview, cursor);
			
			charInserted();
			return;
		}
	}
   
	// recalculate the fill of the row
	if (row->fill() >= 0)  /* needed because a newline
			      * will set fill to -1. Otherwise
			      * we would not get a rebreak! */
		row->fill(Fill(bview, row, workWidth(bview)));
	if (row->fill() < 0) {
		refresh_y = y;
		refresh_row = row; 
		refresh_x = cursor.x();
		refresh_pos = cursor.pos();
		status = LyXText::NEED_MORE_REFRESH;
		BreakAgainOneRow(bview, row); 
		// will the cursor be in another row now?
		if (RowLast(row) <= cursor.pos() + 1 && row->next()) {
			if (row->next() && row->next()->par() == row->par())
				// this should always be true
				row = row->next();
			BreakAgainOneRow(bview, row);
		}
		current_font = rawtmpfont;
		real_current_font = realtmpfont;

		SetCursor(bview, cursor.par(), cursor.pos() + 1, false,
			  cursor.boundary());
		if (IsBoundary(bview->buffer(), cursor.par(), cursor.pos())
		    != cursor.boundary())
			SetCursor(bview, cursor.par(), cursor.pos(), false,
			  !cursor.boundary());
		if (row->next() && row->next()->par() == row->par())
			need_break_row = row->next();
		else
			need_break_row = 0;		
	} else {
		refresh_y = y;
		refresh_x = cursor.x();
		refresh_row = row;
		refresh_pos = cursor.pos();
		
		int const tmpheight = row->height();
		SetHeightOfRow(bview, row);
		if (tmpheight == row->height())
			status = LyXText::NEED_VERY_LITTLE_REFRESH;
		else
			status = LyXText::NEED_MORE_REFRESH;
            
		current_font = rawtmpfont;
		real_current_font = realtmpfont;
		SetCursor(bview, cursor.par(), cursor.pos() + 1, false,
			  cursor.boundary());
	}

	// check, wether the last characters font has changed.
	if (cursor.pos() && cursor.pos() == cursor.par()->Last()
	    && rawparfont != rawtmpfont) {
		RedoHeightOfParagraph(bview, cursor);
	} else {
		// now the special right address boxes
		if (textclasslist.Style(bview->buffer()->params.textclass,
				   cursor.par()->GetLayout()).margintype
		    == MARGIN_RIGHT_ADDRESS_BOX) {
			RedoDrawingOfParagraph(bview, cursor); 
		}
	}

	charInserted();
}
   

void LyXText::charInserted()
{
	// Here we could call FinishUndo for every 20 characters inserted.
	// This is from my experience how emacs does it.
	static unsigned int counter = 0;
	if (counter < 20) {
		++counter;
	} else {
		FinishUndo();
		counter = 0;
	}
}


void LyXText::PrepareToPrint(BufferView * bview,
			     Row * row, float & x,
			     float & fill_separator, 
			     float & fill_hfill,
			     float & fill_label_hfill,
			     bool bidi) const
{
	float nlh, ns;
	
	float w = row->fill();
	fill_hfill = 0;
	fill_label_hfill = 0;
	fill_separator = 0;
	fill_label_hfill = 0;

        bool const is_rtl =
		row->par()->isRightToLeftPar(bview->buffer()->params);
	if (is_rtl) {
		x = (workWidth(bview) > 0)
			? RightMargin(bview->buffer(), row) : 0;
#ifndef NEW_INSETS
		if (row->par()->footnoteflag == LyXParagraph::OPEN_FOOTNOTE) {
			LyXFont font(LyXFont::ALL_SANE);
			font.setSize(LyXFont::SIZE_SMALL);
			x += lyxfont::width("Mwide-figM", font);
		}
#endif
	} else
		x = (workWidth(bview) > 0) ? LeftMargin(bview, row) : 0;
	
	// is there a manual margin with a manual label
	if (textclasslist.Style(bview->buffer()->params.textclass,
			   row->par()->GetLayout()).margintype == MARGIN_MANUAL
	    && textclasslist.Style(bview->buffer()->params.textclass,
			      row->par()->GetLayout()).labeltype == LABEL_MANUAL) {
	       
		/* one more since labels are left aligned */ 
		nlh = NumberOfLabelHfills(bview->buffer(), row) + 1;
		if (nlh && !row->par()->GetLabelWidthString().empty()) {
			fill_label_hfill = LabelFill(bview, row) / nlh;
		}
	}
		
	// are there any hfills in the row?
	float const nh = NumberOfHfills(bview->buffer(), row);

	if (nh)
	  fill_hfill = w / nh;
	else  {
		// is it block, flushleft or flushright? 
		// set x how you need it
	int align;
#ifndef NEW_INSETS
	if (row->par()->FirstPhysicalPar()->align == LYX_ALIGN_LAYOUT)
	  align = textclasslist.Style(bview->buffer()->params.textclass, row->par()->GetLayout()).align;
	else
	  align = row->par()->FirstPhysicalPar()->align;
#else
	if (row->par()->align == LYX_ALIGN_LAYOUT)
	  align = textclasslist.Style(bview->buffer()->params.textclass, row->par()->GetLayout()).align;
	else
	  align = row->par()->align;
#endif
	   
	// center displayed insets 
	Inset * inset;
	   if (row->par()->GetChar(row->pos()) == LyXParagraph::META_INSET
	       && (inset=row->par()->GetInset(row->pos()))
	       && (inset->display())) // || (inset->scroll() < 0)))
	     align = (inset->LyxCode() == Inset::MATHMACRO_CODE)
		     ? LYX_ALIGN_BLOCK : LYX_ALIGN_CENTER;

	   switch (align) {
	    case LYX_ALIGN_BLOCK:
	      ns = NumberOfSeparators(bview->buffer(), row);
	      if (ns && row->next() && row->next()->par() == row->par() &&
		  !(row->next()->par()->IsNewline(row->next()->pos() - 1))
		  && !(row->next()->par()->GetChar(row->next()->pos()) == LyXParagraph::META_INSET
		       && row->next()->par()->GetInset(row->next()->pos())
		       && row->next()->par()->GetInset(row->next()->pos())->display())
		  )
	      	fill_separator = w / ns;
	      else if (is_rtl)
		x += w;
	      break;
	    case LYX_ALIGN_RIGHT:
	      x += w;
	      break;
	    case LYX_ALIGN_CENTER:
	      x += w / 2;
	      break;
	   }
	}
	if (!bidi)
		return;

	ComputeBidiTables(bview->buffer(), row);
	if (is_rtl) {
		LyXParagraph::size_type main_body = 
			BeginningOfMainBody(bview->buffer(), row->par());
		LyXParagraph::size_type last = RowLast(row);

		if (main_body > 0 &&
		    (main_body-1 > last || 
		     !row->par()->IsLineSeparator(main_body-1))) {
			LyXLayout const & layout =
				textclasslist.Style(bview->buffer()->params.textclass,
						    row->par()->GetLayout());
			x += lyxfont::width(layout.labelsep,
					    GetFont(bview->buffer(), row->par(), -2));
			if (main_body-1 <= last)
				x += fill_label_hfill;
		}
	}
}
      
/* important for the screen */


/* the cursor set functions have a special mechanism. When they
* realize, that you left an empty paragraph, they will delete it.
* They also delete the corresponding row */

void LyXText::CursorRightOneWord(BufferView * bview) const
{
	// treat floats, HFills and Insets as words
	LyXCursor tmpcursor = cursor;
	// CHECK See comment on top of text.C

	if (tmpcursor.pos() == tmpcursor.par()->Last()
	    && tmpcursor.par()->Next()) {
			tmpcursor.par(tmpcursor.par()->Next());
			tmpcursor.pos(0);
	} else {
		int steps = 0;

		// Skip through initial nonword stuff.
		while (tmpcursor.pos() < tmpcursor.par()->Last() &&
			! tmpcursor.par()->IsWord( tmpcursor.pos() ) ) 
		{
		  //    printf("Current pos1 %d", tmpcursor.pos()) ;
			tmpcursor.pos(tmpcursor.pos() + 1);
			++steps;
		}
		// Advance through word.
		while (tmpcursor.pos() < tmpcursor.par()->Last() &&
		        tmpcursor.par()->IsWord( tmpcursor.pos() ) )
		{
		  //     printf("Current pos2 %d", tmpcursor.pos()) ;
			tmpcursor.pos(tmpcursor.pos() + 1);
			++steps;
		}
	}
	SetCursor(bview, tmpcursor.par(), tmpcursor.pos());
}


void LyXText::CursorTab(BufferView * bview) const
{
    LyXCursor tmpcursor = cursor;
    while (tmpcursor.pos() < tmpcursor.par()->Last()
           && !tmpcursor.par()->IsNewline(tmpcursor.pos()))
        tmpcursor.pos(tmpcursor.pos() + 1);
   
    if (tmpcursor.pos() == tmpcursor.par()->Last()){
        if (tmpcursor.par()->Next()) {
            tmpcursor.par(tmpcursor.par()->Next());
            tmpcursor.pos(0);
        }
    } else
        tmpcursor.pos(tmpcursor.pos() + 1);
    SetCursor(bview, tmpcursor.par(), tmpcursor.pos());
}


/* -------> Skip initial whitespace at end of word and move cursor to *start*
            of prior word, not to end of next prior word. */

void LyXText::CursorLeftOneWord(BufferView * bview)  const
{
	// treat HFills, floats and Insets as words
	LyXCursor tmpcursor = cursor;
	while (tmpcursor.pos() 
	       && (tmpcursor.par()->IsSeparator(tmpcursor.pos() - 1) 
		   || tmpcursor.par()->IsKomma(tmpcursor.pos() - 1))
	       && !(tmpcursor.par()->IsHfill(tmpcursor.pos() - 1)
#ifndef NEW_INSETS
	            || tmpcursor.par()->IsFloat(tmpcursor.pos() - 1)
#endif
		    || tmpcursor.par()->IsInset(tmpcursor.pos() - 1)))
		tmpcursor.pos(tmpcursor.pos() - 1);

	if (tmpcursor.pos()
	    && (tmpcursor.par()->IsInset(tmpcursor.pos() - 1)
#ifndef NEW_INSETS
	        || tmpcursor.par()->IsFloat(tmpcursor.pos() - 1)
#endif
		|| tmpcursor.par()->IsHfill(tmpcursor.pos() - 1))) {
		tmpcursor.pos(tmpcursor.pos() - 1);
	} else if (!tmpcursor.pos()) {
		if (tmpcursor.par()->Previous()){
			tmpcursor.par(tmpcursor.par()->Previous());
			tmpcursor.pos(tmpcursor.par()->Last());
		}
	} else {		// Here, tmpcursor != 0 
		while (tmpcursor.pos() > 0 &&
		       tmpcursor.par()->IsWord(tmpcursor.pos()-1) )
			tmpcursor.pos(tmpcursor.pos() - 1);
	}
	SetCursor(bview, tmpcursor.par(), tmpcursor.pos());
}

/* -------> Select current word. This depends on behaviour of CursorLeftOneWord(), so it is
			patched as well. */

void LyXText::SelectWord(BufferView * bview) 
{
	// Move cursor to the beginning, when not already there.
	if (cursor.pos()
	    && !cursor.par()->IsSeparator(cursor.pos()-1)
	    && !cursor.par()->IsKomma(cursor.pos()-1) )
		CursorLeftOneWord(bview);

	// set the sel cursor
	sel_cursor = cursor;

	while (cursor.pos() < cursor.par()->Last()
	       && !cursor.par()->IsSeparator(cursor.pos())
	       && !cursor.par()->IsKomma(cursor.pos()) )
		cursor.pos(cursor.pos() + 1);
	SetCursor(bview, cursor.par(), cursor.pos() );
	
	// finally set the selection
	SetSelection();
}


/* -------> Select the word currently under the cursor when:
			1: no selection is currently set,
			2: the cursor is not at the borders of the word. */

bool LyXText::SelectWordWhenUnderCursor(BufferView * bview) 
{
	if (!selection &&
	    cursor.pos() > 0 && cursor.pos() < cursor.par()->Last()
	    && !cursor.par()->IsSeparator(cursor.pos())
	    && !cursor.par()->IsKomma(cursor.pos())
	    && !cursor.par()->IsSeparator(cursor.pos() -1)
	    && !cursor.par()->IsKomma(cursor.pos() -1)) {
		SelectWord(bview);
		return true;
	}
	return false;
}


// This function is only used by the spellchecker for NextWord().
// It doesn't handle LYX_ACCENTs and probably never will.
string const LyXText::SelectNextWord(BufferView * bview,
				     float & value) const
{
	LyXParagraph * tmppar = cursor.par();
	
	// If this is not the very first word, skip rest of
	// current word because we are probably in the middle
	// of a word if there is text here.
	if (cursor.pos() || cursor.par()->previous) {
		while (cursor.pos() < cursor.par()->Last()
		       && cursor.par()->IsLetter(cursor.pos()))
			cursor.pos(cursor.pos() + 1);
	}
	// Now, skip until we have real text (will jump paragraphs)
	while ((cursor.par()->Last() > cursor.pos()
		&& (!cursor.par()->IsLetter(cursor.pos())
		    || cursor.par()->getFont(bview->buffer()->params, cursor.pos())
		    .latex() == LyXFont::ON))
	       || (cursor.par()->Last() == cursor.pos() 
		   && cursor.par()->Next())){
		if (cursor.pos() == cursor.par()->Last()) {
			cursor.par(cursor.par()->Next());
			cursor.pos(0);
		} else
			cursor.pos(cursor.pos() + 1);
	}
  
	// Update the value if we changed paragraphs
	if (cursor.par() != tmppar){
		SetCursor(bview, cursor.par(), cursor.pos());
		value = float(cursor.y())/float(height);
	}

	// Start the selection from here
	sel_cursor = cursor;

	std::ostringstream latex;

	// and find the end of the word 
	// (optional hyphens are part of a word)
	while (cursor.pos() < cursor.par()->Last()
	       && (cursor.par()->IsLetter(cursor.pos())) 
	           || (cursor.par()->GetChar(cursor.pos()) == LyXParagraph::META_INSET
		       && cursor.par()->GetInset(cursor.pos()) != 0
		       && cursor.par()->GetInset(cursor.pos())->Latex(bview->buffer(), latex, false, false) == 0
		       && latex.str() == "\\-"
			   ))
		cursor.pos(cursor.pos() + 1);

	// Finally, we copy the word to a string and return it
	string str;
	if (sel_cursor.pos() < cursor.pos()) {
		LyXParagraph::size_type i;
		for (i = sel_cursor.pos(); i < cursor.pos(); ++i) {
			if (cursor.par()->GetChar(i) != LyXParagraph::META_INSET)
				str += cursor.par()->GetChar(i);
		}
	}
	return str;
}


// This one is also only for the spellchecker
void LyXText::SelectSelectedWord(BufferView * bview)
{
	// move cursor to the beginning
	SetCursor(bview, sel_cursor.par(), sel_cursor.pos());
	
	// set the sel cursor
	sel_cursor = cursor;

	std::ostringstream latex;
	
	// now find the end of the word
	while (cursor.pos() < cursor.par()->Last()
	       && (cursor.par()->IsLetter(cursor.pos())
	           || (cursor.par()->GetChar(cursor.pos()) == LyXParagraph::META_INSET
		       && cursor.par()->GetInset(cursor.pos()) != 0
		       && cursor.par()->GetInset(cursor.pos())->Latex(bview->buffer(), latex, false, false) == 0
		       && latex.str() == "\\-"
			   )))
		cursor.pos(cursor.pos() + 1);
	
	SetCursor(bview, cursor.par(), cursor.pos());
	
	// finally set the selection
	SetSelection();
}


/* -------> Delete from cursor up to the end of the current or next word. */
void LyXText::DeleteWordForward(BufferView * bview)
{
	if (!cursor.par()->Last())
		CursorRight(bview);
	else {
		LyXCursor tmpcursor = cursor;
		tmpcursor.row(0); //??
		selection = true; // to avoid deletion 
		CursorRightOneWord(bview);
		SetCursor(bview, tmpcursor, tmpcursor.par(), tmpcursor.pos());
		sel_cursor = cursor;
		cursor = tmpcursor;
		SetSelection(); 
		
		/* -----> Great, CutSelection() gets rid of multiple spaces. */
		CutSelection(bview);
	}
}


/* -------> Delete from cursor to start of current or prior word. */
void LyXText::DeleteWordBackward(BufferView * bview)
{
       if (!cursor.par()->Last())
	       CursorLeft(bview);
       else {
	       LyXCursor tmpcursor = cursor;
	       tmpcursor.row(0); // ??
	       selection = true; // to avoid deletion 
	       CursorLeftOneWord(bview);
	       SetCursor(bview, tmpcursor, tmpcursor.par(), tmpcursor.pos());
	       sel_cursor = cursor;
	       cursor = tmpcursor;
	       SetSelection();
	       CutSelection(bview);
       }
}


/* -------> Kill to end of line. */
void LyXText::DeleteLineForward(BufferView * bview)
{
	if (!cursor.par()->Last())
		// Paragraph is empty, so we just go to the right
		CursorRight(bview);
	else {
		LyXCursor tmpcursor = cursor;
		// We can't store the row over a regular SetCursor
		// so we set it to 0 and reset it afterwards.
		tmpcursor.row(0); //??
		selection = true; // to avoid deletion 
		CursorEnd(bview);
		SetCursor(bview, tmpcursor, tmpcursor.par(), tmpcursor.pos());
		sel_cursor = cursor;
		cursor = tmpcursor;
		SetSelection();
		// What is this test for ??? (JMarc)
		if (!selection) {
			DeleteWordForward(bview);
		} else {
			CutSelection(bview);
		}
	}
}


// Change the case of a word at cursor position. 
// This function directly manipulates LyXParagraph::text because there
// is no LyXParagraph::SetChar currently. I did what I could to ensure
// that it is correct. I guess part of it should be moved to
// LyXParagraph, but it will have to change for 1.1 anyway. At least
// it does not access outside of the allocated array as the older
// version did. (JMarc) 
void LyXText::ChangeWordCase(BufferView * bview, LyXText::TextCase action) 
{
#ifndef NEW_INSETS
	LyXParagraph * tmppar = cursor.par()->ParFromPos(cursor.pos());
#else
	LyXParagraph * tmppar = cursor.par();
#endif

	SetUndo(bview->buffer(),Undo::FINISH, tmppar->previous, tmppar->next); 

#ifndef NEW_INSETS
	LyXParagraph::size_type tmppos = 
		cursor.par()->PositionInParFromPos(cursor.pos());
#else
	LyXParagraph::size_type tmppos = cursor.pos();
#endif
	while (tmppos < tmppar->size()) {
		unsigned char c = tmppar->GetChar(tmppos);
		if (IsKommaChar(c) || IsLineSeparatorChar(c))
			break;
		if (c != LyXParagraph::META_INSET) {
			switch (action) {
			case text_lowercase:
				c = tolower(c);
				break;
			case text_capitalization:
				c = toupper(c);
				action = text_lowercase;
				break;
			case text_uppercase:
				c = toupper(c);
				break;
			}
		}
		
		//tmppar->text[tmppos] = c;
		tmppar->SetChar(tmppos, c);
		++tmppos;
	}
	CheckParagraph(bview, tmppar, tmppos);
	CursorRightOneWord(bview);
}


void LyXText::Delete(BufferView * bview)
{
	// this is a very easy implementation

	LyXCursor old_cursor = cursor;
	int const old_cur_par_id = old_cursor.par()->id();
	int const old_cur_par_prev_id = old_cursor.par()->previous ?
		old_cursor.par()->previous->id() : 0;
	
	// just move to the right
	CursorRight(bview);

	// CHECK Look at the comment here.
	// This check is not very good...
	// The CursorRightIntern calls DeleteEmptyParagrapgMechanism
	// and that can very well delete the par or par->previous in
	// old_cursor. Will a solution where we compare paragraph id's
	//work better?
	if ((cursor.par()->previous ? cursor.par()->previous->id() : 0)
	    == old_cur_par_prev_id
	    && cursor.par()->id() != old_cur_par_id)
		return; // delete-empty-paragraph-mechanism has done it

	// if you had success make a backspace
	if (old_cursor.par() != cursor.par() || old_cursor.pos() != cursor.pos()) {
		LyXCursor tmpcursor = cursor;
		cursor = old_cursor; // to make sure undo gets the right cursor position
		SetUndo(bview->buffer(), Undo::DELETE,
#ifndef NEW_INSETS
			cursor.par()->ParFromPos(cursor.pos())->previous, 
			cursor.par()->ParFromPos(cursor.pos())->next
#else
			cursor.par()->previous, 
			cursor.par()->next
#endif
			); 
		cursor = tmpcursor;
		Backspace(bview);
	}
}


void LyXText::Backspace(BufferView * bview)
{
	// Get the font that is used to calculate the baselineskip
	LyXParagraph::size_type lastpos = cursor.par()->Last();
	LyXFont rawparfont = cursor.par()->GetFontSettings(bview->buffer()->params,
							 lastpos - 1);

	if (cursor.pos() == 0) {
		// The cursor is at the beginning of a paragraph,
		// so the the backspace will collapse two paragraphs into one.
		
		// we may paste some paragraphs
      
		// is it an empty paragraph?
      
		if ((lastpos == 0
		     || (lastpos == 1 && cursor.par()->IsSeparator(0)))
#ifndef NEW_INSETS
		    && !(cursor.par()->Next() 
			 && cursor.par()->footnoteflag == LyXParagraph::NO_FOOTNOTE
			 && cursor.par()->Next()->footnoteflag == LyXParagraph::OPEN_FOOTNOTE)
#endif
			) {
			// This is an empty paragraph and we delete it just by moving the cursor one step
			// left and let the DeleteEmptyParagraphMechanism handle the actual deletion
			// of the paragraph.
			
			if (cursor.par()->previous) {
#ifndef NEW_INSETS
				LyXParagraph * tmppar = cursor.par()->previous->FirstPhysicalPar();
#else
				LyXParagraph * tmppar = cursor.par()->previous;
#endif
				if (cursor.par()->GetLayout() == tmppar->GetLayout()
#ifndef NEW_INSETS
				    && cursor.par()->footnoteflag == tmppar->footnoteflag
#endif
				    && cursor.par()->GetAlign() == tmppar->GetAlign()) {
					// Inherit botom DTD from the paragraph below.
					// (the one we are deleting)
					tmppar->line_bottom = cursor.par()->line_bottom;
					tmppar->added_space_bottom = cursor.par()->added_space_bottom;
					tmppar->pagebreak_bottom = cursor.par()->pagebreak_bottom;
				}
				
				CursorLeft(bview);
		     
				// the layout things can change the height of a row !
				int const tmpheight = cursor.row()->height();
				SetHeightOfRow(bview, cursor.row());
				if (cursor.row()->height() != tmpheight) {
					refresh_y = cursor.y() - cursor.row()->baseline();
					refresh_row = cursor.row();
					status = LyXText::NEED_MORE_REFRESH;
				}
				return;
			}
		}

#ifndef NEW_INSETS
		if (cursor.par()->ParFromPos(cursor.pos())->previous){
			SetUndo(bview->buffer(), Undo::DELETE,
				cursor.par()->ParFromPos(cursor.pos())->previous->previous,
				cursor.par()->ParFromPos(cursor.pos())->next);
		}
#else
		if (cursor.par()->previous) {
			SetUndo(bview->buffer(), Undo::DELETE,
				cursor.par()->previous->previous,
				cursor.par()->next);
		}
#endif
		
  		LyXParagraph * tmppar = cursor.par();
  		Row * tmprow = cursor.row();

		// We used to do CursorLeftIntern() here, but it is
		// not a good idea since it triggers the auto-delete
		// mechanism. So we do a CursorLeftIntern()-lite,
		// without the dreaded mechanism. (JMarc)
		if (cursor.par()->Previous()) { 
			// steps into the above paragraph.
			SetCursorIntern(bview, cursor.par()->Previous(), 
					cursor.par()->Previous()->Last(), false);
		}

		/* Pasting is not allowed, if the paragraphs have different
		   layout. I think it is a real bug of all other
		   word processors to allow it. It confuses the user.
		   Even so with a footnote paragraph and a non-footnote
		   paragraph. I will not allow pasting in this case, 
		   because the user would be confused if the footnote behaves 
		   different wether it is open or closed.
		  
		   Correction: Pasting is always allowed with standard-layout
		*/
		if (cursor.par() != tmppar
		    && (cursor.par()->GetLayout() == tmppar->GetLayout()
			|| tmppar->GetLayout() == 0 /*standard*/)
#ifndef NEW_INSETS
		    && cursor.par()->footnoteflag == tmppar->footnoteflag
#endif
		    && cursor.par()->GetAlign() == tmppar->GetAlign()) {

			RemoveParagraph(tmprow);
			RemoveRow(tmprow);
			cursor.par()->PasteParagraph(bview->buffer()->params);
			
			if (!cursor.pos() || !cursor.par()->IsSeparator(cursor.pos() - 1))
				; //cursor.par()->InsertChar(cursor.pos(), ' ');
			// strangely enough it seems that commenting out the line above removes
			// most or all of the segfaults. I will however also try to move the
			// two Remove... lines in front of the PasteParagraph too.
			else
				if (cursor.pos())
					cursor.pos(cursor.pos() - 1);
			
			status = LyXText::NEED_MORE_REFRESH;
			refresh_row = cursor.row();
			refresh_y = cursor.y() - cursor.row()->baseline();
			
			// remove the lost paragraph
			// This one is not safe, since the paragraph that the tmprow and the
			// following rows belong to has been deleted by the PasteParagraph
			// above. The question is... could this be moved in front of the
			// PasteParagraph?
			//RemoveParagraph(tmprow);
			//RemoveRow(tmprow);  
			
			// This rebuilds the rows.
			AppendParagraph(bview, cursor.row());
			UpdateCounters(bview, cursor.row());
			
			// the row may have changed, block, hfills etc.
			SetCursor(bview, cursor.par(), cursor.pos(), false);
		}
	} else {
		/* this is the code for a normal backspace, not pasting
		 * any paragraphs */ 
		SetUndo(bview->buffer(), Undo::DELETE,
#ifndef NEW_INSETS
			cursor.par()->ParFromPos(cursor.pos())->previous, 
			cursor.par()->ParFromPos(cursor.pos())->next
#else
			cursor.par()->previous, 
			cursor.par()->next
#endif
			); 
		// We used to do CursorLeftIntern() here, but it is
		// not a good idea since it triggers the auto-delete
		// mechanism. So we do a CursorLeftIntern()-lite,
		// without the dreaded mechanism. (JMarc)
		SetCursorIntern(bview, cursor.par(), cursor.pos()- 1,
				false, cursor.boundary());
		
		// some insets are undeletable here
		if (cursor.par()->GetChar(cursor.pos()) == LyXParagraph::META_INSET) {
			if (!cursor.par()->GetInset(cursor.pos())->Deletable())
				return; 
			// force complete redo when erasing display insets
			// this is a cruel method but safe..... Matthias 
			if (cursor.par()->GetInset(cursor.pos())->display() ||
			    cursor.par()->GetInset(cursor.pos())->needFullRow()) {
				cursor.par()->Erase(cursor.pos());
				RedoParagraph(bview);
				return;
			}
		}
		
		Row * row = cursor.row();
		int y = cursor.y() - row->baseline();
		LyXParagraph::size_type z;
		/* remember that a space at the end of a row doesnt count
		 * when calculating the fill */ 
		if (cursor.pos() < RowLast(row) ||
		    !cursor.par()->IsLineSeparator(cursor.pos())) {
			row->fill(row->fill() + SingleWidth(bview,
							    cursor.par(),
							    cursor.pos()));
		}
		
		/* some special code when deleting a newline. This is similar
		 * to the behavior when pasting paragraphs */ 
		if (cursor.pos() && cursor.par()->IsNewline(cursor.pos())) {
			cursor.par()->Erase(cursor.pos());
			// refresh the positions
			Row * tmprow = row;
			while (tmprow->next() && tmprow->next()->par() == row->par()) {
				tmprow = tmprow->next();
				tmprow->pos(tmprow->pos() - 1);
			}
			if (cursor.par()->IsLineSeparator(cursor.pos() - 1))
				cursor.pos(cursor.pos() - 1);
			
			if (cursor.pos() < cursor.par()->Last() && !cursor.par()->IsSeparator(cursor.pos())) {
				cursor.par()->InsertChar(cursor.pos(), ' ');
				SetCharFont(bview->buffer(), cursor.par(), 
					    cursor.pos(), current_font);
				// refresh the positions
				tmprow = row;
				while (tmprow->next() && tmprow->next()->par() == row->par()) {
					tmprow = tmprow->next();
					tmprow->pos(tmprow->pos() + 1);
				}
			}
		} else {
			cursor.par()->Erase(cursor.pos());
			
			// refresh the positions
			Row * tmprow = row;
			while (tmprow->next()
			       && tmprow->next()->par() == row->par()) {
				tmprow = tmprow->next();
				tmprow->pos(tmprow->pos() - 1);
			}

			// delete newlines at the beginning of paragraphs
			while (cursor.par()->Last() &&
			       cursor.par()->IsNewline(cursor.pos()) &&
			       cursor.pos() == BeginningOfMainBody(bview->buffer(),
								   cursor.par())) {
				cursor.par()->Erase(cursor.pos());
				// refresh the positions
				tmprow = row;
				while (tmprow->next() && 
				       tmprow->next()->par() == row->par()) {
					tmprow = tmprow->next();
					tmprow->pos(tmprow->pos() - 1);
				}
			}
		}
		
		// is there a break one row above
		if (row->previous() && row->previous()->par() == row->par()) {
			z = NextBreakPoint(bview, row->previous(),
					   workWidth(bview));
			if (z >= row->pos()) {
				row->pos(z + 1);
				
				Row * tmprow = row->previous();
				
				// maybe the current row is now empty
				if (row->pos() >= row->par()->Last()) {
					// remove it
					RemoveRow(row);
					need_break_row = 0;
				} else {
					BreakAgainOneRow(bview, row);
					if (row->next() && row->next()->par() == row->par())
						need_break_row = row->next();
					else
						need_break_row = 0;
				}
				
				// set the dimensions of the row above
				y -= tmprow->height();
				tmprow->fill(Fill(bview, tmprow,
						  workWidth(bview)));
				SetHeightOfRow(bview, tmprow);
				
				refresh_y = y;
				refresh_row = tmprow;
				status = LyXText::NEED_MORE_REFRESH;
				SetCursor(bview, cursor.par(), cursor.pos(),
					  false, cursor.boundary());
				//current_font = rawtmpfont;
				//real_current_font = realtmpfont;
				// check, whether the last character's font has changed.
				if (rawparfont !=
				    cursor.par()->GetFontSettings(bview->buffer()->params,
								  cursor.par()->Last() - 1))
					RedoHeightOfParagraph(bview, cursor);
				return;
			}
		}
		
		// break the cursor row again
		if (row->next() && row->next()->par() == row->par() &&
		    (RowLast(row) == row->par()->Last() - 1 ||
		     NextBreakPoint(bview, row, workWidth(bview)) != RowLast(row))) {
			
			/* it can happen that a paragraph loses one row
			 * without a real breakup. This is when a word
			 * is to long to be broken. Well, I don t care this 
			 * hack ;-) */ 
			if (RowLast(row) == row->par()->Last() - 1)
				RemoveRow(row->next());
			
			refresh_y = y;
			refresh_row = row;
			status = LyXText::NEED_MORE_REFRESH;
			
			BreakAgainOneRow(bview, row);
			// will the cursor be in another row now?
			if (row->next() && row->next()->par() == row->par() &&
			    RowLast(row) <= cursor.pos()) {
				row = row->next();
				BreakAgainOneRow(bview, row);
			}

			SetCursor(bview, cursor.par(), cursor.pos(), false, cursor.boundary());

			if (row->next() && row->next()->par() == row->par())
				need_break_row = row->next();
			else
				need_break_row = 0;
		} else  {
			// set the dimensions of the row
			row->fill(Fill(bview, row, workWidth(bview)));
			int const tmpheight = row->height();
			SetHeightOfRow(bview, row);
			if (tmpheight == row->height())
				status = LyXText::NEED_VERY_LITTLE_REFRESH;
			else
				status = LyXText::NEED_MORE_REFRESH;
			refresh_y = y;
			refresh_row = row;
			SetCursor(bview, cursor.par(), cursor.pos(), false, cursor.boundary());
		}
	}

	// current_font = rawtmpfont;
	// real_current_font = realtmpfont;

	if (IsBoundary(bview->buffer(), cursor.par(), cursor.pos())
	    != cursor.boundary())
		SetCursor(bview, cursor.par(), cursor.pos(), false,
			  !cursor.boundary());

	lastpos = cursor.par()->Last();
	if (cursor.pos() == lastpos)
		SetCurrentFont(bview);
	
	// check, whether the last characters font has changed.
	if (rawparfont != 
	    cursor.par()->GetFontSettings(bview->buffer()->params, lastpos - 1)) {
		RedoHeightOfParagraph(bview, cursor);
	} else {
		// now the special right address boxes
		if (textclasslist.Style(bview->buffer()->params.textclass,
					cursor.par()->GetLayout()).margintype == MARGIN_RIGHT_ADDRESS_BOX) {
			RedoDrawingOfParagraph(bview, cursor); 
		}
	}
}


void LyXText::GetVisibleRow(BufferView * bview, int y_offset, int x_offset,
			    Row * row_ptr, int y, bool cleared)
{
	/* returns a printed row */
	Painter & pain = bview->painter();
	
	bool const is_rtl =
		row_ptr->par()->isRightToLeftPar(bview->buffer()->params);
	
	LyXParagraph::size_type const last = RowLastPrintable(row_ptr);

	LyXParagraph::size_type vpos, pos;
	float x;
	float tmpx;
	int y_top;
	int y_bottom;

	LyXFont font(LyXFont::ALL_SANE);
	int maxdesc;
	if (row_ptr->height() <= 0) {
		lyxerr << "LYX_ERROR: row.height: "
		       << row_ptr->height() << endl;
		return;
	}

	float fill_separator;
	float fill_hfill;
	float fill_label_hfill;
	PrepareToPrint(bview, row_ptr, x, fill_separator,
		       fill_hfill, fill_label_hfill);
	
	if (inset_owner && (x < 0))
		x = 0;
	x += x_offset;
	
	// clear the area where we want to paint/print
	int ww = bview->workWidth();

	bool clear_area = true;
	Inset * inset = 0;

	if (!bview->screen()->forceClear() && last == row_ptr->pos()
	    && row_ptr->par()->GetChar(row_ptr->pos()) == LyXParagraph::META_INSET
	    && (inset = row_ptr->par()->GetInset(row_ptr->pos()))) {
		clear_area = inset->doClearArea();
	}
	// we don't need to clear it's already done!!!
	if (cleared) {
		clear_area = true;
	} else if (clear_area) {
#ifdef WITH_WARNINGS
#warning Should be fixed with a lyxinset::clear_width(bv, font) function! (Jug)
#warning Should we not fix this in the Painter, please have a look Lars! (Jug)
#endif
		int w = (inset_owner ? inset_owner->width(bview, font)-2 : ww);
		int h = row_ptr->height();
		int x = x_offset;
		int y = y_offset;
		if (y < 0) {
		    h += y;
		    y = 0;
		}

		pain.fillRectangle(x, y, w, h);
	} else if (inset != 0) {
		int h = row_ptr->baseline() - inset->ascent(bview, font);
		if (h > 0) {
			int const w = (inset_owner ?
				 inset_owner->width(bview, font) : ww);
			pain.fillRectangle(x_offset, y_offset, w, h);
		}
		h += inset->ascent(bview, font) + inset->descent(bview, font);
		if ((row_ptr->height() - h) > 0) {
			int const w = (inset_owner ?
				 inset_owner->width(bview, font) : ww);
			pain.fillRectangle(x_offset, y_offset + h,
					   w, row_ptr->height() - h);
		}
		if (!inset_owner && !inset->display() && !inset->needFullRow())
		{
			int const w = inset->width(bview, font) + int(x);
			pain.fillRectangle(w, y_offset, ww - w, row_ptr->height());
		}
	}
	
	if (selection) {
		int const w = (inset_owner ?
			       inset_owner->width(bview, font) : ww);
		/* selection code */
		if (bidi_same_direction) {
			if (sel_start_cursor.row() == row_ptr &&
			    sel_end_cursor.row() == row_ptr) {
				if (sel_start_cursor.x() < sel_end_cursor.x())
					pain.fillRectangle(x_offset + sel_start_cursor.x(),
							   y_offset,
							   sel_end_cursor.x() - sel_start_cursor.x(),
							   row_ptr->height(),
							   LColor::selection);
				else
					pain.fillRectangle(x_offset + sel_end_cursor.x(),
							   y_offset,
							   sel_start_cursor.x() - sel_end_cursor.x(),
							   row_ptr->height(),
							   LColor::selection);
			} else if (sel_start_cursor.row() == row_ptr) {
				if (is_rtl)
					pain.fillRectangle(x_offset, y_offset,
							   sel_start_cursor.x(),
							   row_ptr->height(),
							   LColor::selection);
				else
					pain.fillRectangle(x_offset + sel_start_cursor.x(),
							   y_offset,
							   w - sel_start_cursor.x(),
							   row_ptr->height(),
							   LColor::selection);
			} else if (sel_end_cursor.row() == row_ptr) {
				if (is_rtl)
					pain.fillRectangle(x_offset + sel_end_cursor.x(),
							   y_offset,
							   w - sel_end_cursor.x(),
							   row_ptr->height(),
							   LColor::selection);
				else
					pain.fillRectangle(x_offset, y_offset,
							   sel_end_cursor.x(),
							   row_ptr->height(),
							   LColor::selection);
			} else if (y > sel_start_cursor.y()
				   && y < sel_end_cursor.y()) {
				pain.fillRectangle(x_offset, y_offset, w,
						   row_ptr->height(),
						   LColor::selection);
			}
		} else if (sel_start_cursor.row() != row_ptr &&
			    sel_end_cursor.row() != row_ptr &&
			    y > sel_start_cursor.y()
			    && y < sel_end_cursor.y()) {
			pain.fillRectangle(x_offset, y_offset, w,
					   row_ptr->height(),
					   LColor::selection);
		} else if (sel_start_cursor.row() == row_ptr ||
			   sel_end_cursor.row() == row_ptr) {
			float tmpx = x;
			if ((sel_start_cursor.row() != row_ptr && !is_rtl) ||
			     (sel_end_cursor.row() != row_ptr && is_rtl))
				pain.fillRectangle(x_offset, y_offset,
						   int(tmpx),
						   row_ptr->height(),
						   LColor::selection);
			LyXParagraph::size_type main_body =
				BeginningOfMainBody(bview->buffer(),
						    row_ptr->par());
			
			for (vpos = row_ptr->pos(); vpos <= last; ++vpos)  {
				pos = vis2log(vpos);
				float const old_tmpx = tmpx;
				if (main_body > 0 && pos == main_body-1) {
					tmpx += fill_label_hfill +
						lyxfont::width(textclasslist.Style(bview->buffer()->params.textclass,
										   row_ptr->par()->GetLayout()).labelsep,
							       GetFont(bview->buffer(),row_ptr->par(), -2));
					if (row_ptr->par()->IsLineSeparator(main_body-1))
						tmpx -= SingleWidth(bview, row_ptr->par(), main_body-1);
				}
				if (HfillExpansion(bview->buffer(), row_ptr, pos)) {
					tmpx += SingleWidth(bview, row_ptr->par(), pos);
					if (pos >= main_body)
						tmpx += fill_hfill;
					else 
						tmpx += fill_label_hfill;
				}
				else if (row_ptr->par()->IsSeparator(pos)) {
					tmpx += SingleWidth(bview, row_ptr->par(), pos);
					if (pos >= main_body)
						tmpx += fill_separator;
				} else
					tmpx += SingleWidth(bview, row_ptr->par(), pos);
				
				if ((sel_start_cursor.row() != row_ptr ||
				      sel_start_cursor.pos() <= pos) &&
				     (sel_end_cursor.row() != row_ptr ||
				      pos < sel_end_cursor.pos()) )
					/// Here we do not use x_offset as x_offset was
					// added to x.
					pain.fillRectangle(int(old_tmpx),
							   y_offset,
							   int(tmpx - old_tmpx + 1),
							   row_ptr->height(),
							   LColor::selection);
			}

			if ((sel_start_cursor.row() != row_ptr && is_rtl) ||
			     (sel_end_cursor.row() != row_ptr && !is_rtl) )
				pain.fillRectangle(x_offset + int(tmpx),
						   y_offset,
						   int(ww - tmpx),
						   row_ptr->height(),
						   LColor::selection);
		}
	}

	int box_x = 0;
#ifndef NEW_INSETS
	if (row_ptr->par()->footnoteflag == LyXParagraph::OPEN_FOOTNOTE) {
		LyXFont font(LyXFont::ALL_SANE);
		font.setSize(LyXFont::SIZE_FOOTNOTE);
		font.setColor(LColor::footnote);
		
		box_x = LYX_PAPER_MARGIN + lyxfont::width(" wide-tab ", font);
		if (row_ptr->previous() &&
		    row_ptr->previous()->par()->footnoteflag != LyXParagraph::OPEN_FOOTNOTE){
			string fs;
			switch (row_ptr->par()->footnotekind) {
			case LyXParagraph::MARGIN:
				fs = " margin";
				break;
			case LyXParagraph::FIG:
				fs = " fig";
				break;
			case LyXParagraph::TAB:
				fs = " tab";
				break;
			case LyXParagraph::WIDE_FIG:
				fs = " wide-fig";
				break;
			case LyXParagraph::WIDE_TAB:
				fs = " wide-tab";
				break;
			case LyXParagraph::ALGORITHM:
				fs = " alg";
				break;
			case LyXParagraph::FOOTNOTE:
				fs = " foot";
				break;
			}
			
			pain.fillRectangle(LYX_PAPER_MARGIN,
					   y_offset + 1,
					   box_x - LYX_PAPER_MARGIN,
					   int(lyxfont::maxAscent(font)
					       + lyxfont::maxDescent(font)),
					   LColor::footnotebg);
			
			pain.line(LYX_PAPER_MARGIN, y_offset,
				  workWidth(bview) - LYX_PAPER_MARGIN, y_offset,
				  LColor::footnoteframe);
			
			pain.text(LYX_PAPER_MARGIN,
				  y_offset + int(lyxfont::maxAscent(font)) + 1,
				  fs, font);
			
			pain.line(LYX_PAPER_MARGIN, y_offset,
				  LYX_PAPER_MARGIN,
				  y_offset + int(lyxfont::maxAscent(font)
					       + lyxfont::maxDescent(font)),
				  LColor::footnoteframe);
			
			pain.line(LYX_PAPER_MARGIN,
				  y_offset + int(lyxfont::maxAscent(font)
					       + lyxfont::maxDescent(font)) + 1,
				  box_x,
				  y_offset + int(lyxfont::maxAscent(font)
					       + lyxfont::maxDescent(font)) + 1,
				  LColor::footnoteframe);
			
		}
		
		/* draw the open floats in a red box */
		pain.line(box_x, y_offset,
			  box_x, y_offset + row_ptr->height(),
			  LColor::footnoteframe);
		
		pain.line(workWidth(bview) - LYX_PAPER_MARGIN,
			  y_offset,
			  workWidth(bview) - LYX_PAPER_MARGIN,
			  y_offset + row_ptr->height(),
			  LColor::footnoteframe);


		// Draw appendix lines
		LyXParagraph * p = row_ptr->par()->PreviousBeforeFootnote()->FirstPhysicalPar();
		if (p->appendix){
			pain.line(1, y_offset,
				  1, y_offset + row_ptr->height(),
				  LColor::appendixline);
			pain.line(workWidth(bview) - 2, y_offset,
				  workWidth(bview) - 2,
				  y_offset + row_ptr->height(),
				  LColor::appendixline);
		}

		// Draw minipage line
		bool const minipage =
			(p->pextra_type == LyXParagraph::PEXTRA_MINIPAGE);
		if (minipage)
			pain.line(LYX_PAPER_MARGIN/5, y_offset,
				  LYX_PAPER_MARGIN/5, 
				  y_offset + row_ptr->height() - 1,
				  LColor::minipageline);

		// Draw depth lines
		int const depth = p->GetDepth();
		for (int i = 1; i <= depth; ++i) {
			int const line_x = (LYX_PAPER_MARGIN / 5) *
				(i + minipage);
			pain.line(line_x, y_offset, line_x,
				  y_offset + row_ptr->height() - 1,
				  LColor::depthbar);
		}
	} else if (row_ptr->previous() &&
		   row_ptr->previous()->par()->footnoteflag
		   == LyXParagraph::OPEN_FOOTNOTE) {
		LyXFont font(LyXFont::ALL_SANE);
		font.setSize(LyXFont::SIZE_FOOTNOTE);
		
		int const box_x = LYX_PAPER_MARGIN
			+ lyxfont::width(" wide-tab ", font);
		
		pain.line(box_x, y_offset,
			  workWidth(bview) - LYX_PAPER_MARGIN,
			  y_offset, LColor::footnote);
	}
#endif
	// Draw appendix lines
#ifndef NEW_INSETS
	LyXParagraph * firstpar = row_ptr->par()->FirstPhysicalPar();
#else
	LyXParagraph * firstpar = row_ptr->par();
#endif
	if (firstpar->appendix){
		pain.line(1, y_offset,
			  1, y_offset + row_ptr->height(),
			  LColor::appendixline);
		pain.line(ww - 2, y_offset,
			  ww - 2, y_offset + row_ptr->height(),
			  LColor::appendixline);
	}

	// Draw minipage line
	bool const minipage =
		(firstpar->pextra_type == LyXParagraph::PEXTRA_MINIPAGE);
	if (minipage)
		pain.line(LYX_PAPER_MARGIN/5 + box_x, y_offset,
			  LYX_PAPER_MARGIN/5 + box_x, 
			  y_offset + row_ptr->height() - 1,
			  LColor::minipageline);

	// Draw depth lines
	int const depth = firstpar->GetDepth();
	if (depth > 0) {
		int next_depth = 0;
		int prev_depth = 0;
		if (row_ptr->next())
#ifndef NEW_INSETS
			if (row_ptr->par()->footnoteflag ==
			    row_ptr->next()->par()->footnoteflag)
				next_depth = row_ptr->next()->par()->GetDepth();
			else if (row_ptr->par()->footnoteflag != LyXParagraph::OPEN_FOOTNOTE)
				next_depth = depth;
#else
				next_depth = row_ptr->next()->par()->GetDepth();
#endif
		if (row_ptr->previous())
#ifndef NEW_INSETS
			if (row_ptr->par()->footnoteflag ==
			    row_ptr->previous()->par()->footnoteflag)
				prev_depth = row_ptr->previous()->par()->GetDepth();
			else if (row_ptr->par()->footnoteflag != LyXParagraph::OPEN_FOOTNOTE)
				prev_depth = depth;
#else
				prev_depth = row_ptr->previous()->par()->GetDepth();
#endif

		for (int i = 1; i <= depth; ++i) {
			int const line_x = (LYX_PAPER_MARGIN / 5) *
				(i + minipage) + box_x + x_offset;
			pain.line(line_x, y_offset, line_x,
				  y_offset + row_ptr->height() - 1 - (i - next_depth - 1) * 3,
				  LColor::depthbar);
		
			if (i > prev_depth)
				pain.fillRectangle(line_x, y_offset, LYX_PAPER_MARGIN / 5, 2,
						   LColor::depthbar);
			if (i > next_depth)
				pain.fillRectangle(line_x,
						   y_offset + row_ptr->height() - 2 - (i - next_depth - 1) * 3,
						   LYX_PAPER_MARGIN / 5, 2,
						   LColor::depthbar);
		}
	}

	
	LyXLayout const & layout =
		textclasslist.Style(bview->buffer()->params.textclass,
				    row_ptr->par()->GetLayout());

	y_top = 0;
	y_bottom = row_ptr->height();
	
	/* is it a first row? */ 
	if (!row_ptr->pos() && (row_ptr->par() == firstpar)) {
		
		/* start of appendix? */
		if (row_ptr->par()->start_of_appendix){
			pain.line(1, y_offset,
				  ww - 2, y_offset,
				  LColor::appendixline);
		}
		
		/* think about the margins */ 
		if (!row_ptr->previous() && bv_owner)
			y_top += LYX_PAPER_MARGIN;
		
		if (row_ptr->par()->pagebreak_top){ /* draw a top pagebreak  */
			LyXFont pb_font;
			pb_font.setColor(LColor::pagebreak).decSize();
			int w = 0, a = 0, d = 0;
			pain.line(0, y_offset + y_top + 2*DefaultHeight(),
				  ww, 
				  y_offset + y_top + 2 * DefaultHeight(),
				  LColor::pagebreak, 
				  Painter::line_onoffdash)
				.rectText(0,
					  0,
					  _("Page Break (top)"),
					  pb_font,
					  LColor::background,
					  LColor::background, false, w, a, d);
			pain.rectText((ww - w)/2,
				      y_offset + y_top + 2 * DefaultHeight() + d,
				      _("Page Break (top)"),
				      pb_font,
				      LColor::background,
				      LColor::background);
			y_top += 3 * DefaultHeight();
		}
		
		if (row_ptr->par()->added_space_top.kind() == VSpace::VFILL) {
			/* draw a vfill top  */
			pain.line(0, y_offset + 2 + y_top,
				  LYX_PAPER_MARGIN, y_offset + 2 + y_top,
				  LColor::vfillline);
			
			pain.line(0, y_offset + y_top + 3 * DefaultHeight(),
				  LYX_PAPER_MARGIN,
				  y_offset + y_top + 3 * DefaultHeight(),
				  LColor::vfillline);
			
			pain.line(LYX_PAPER_MARGIN / 2, y_offset + 2 + y_top,
				  LYX_PAPER_MARGIN / 2,
				  y_offset + y_top + 3 * DefaultHeight(),
				  LColor::vfillline);
			
			y_top += 3 * DefaultHeight();
		}
		
		/* think about user added space */ 
		y_top += int(row_ptr->par()->added_space_top.inPixels(bview));
		
		/* think about the parskip */ 
		/* some parskips VERY EASY IMPLEMENTATION */ 
		if (bview->buffer()->params.paragraph_separation == BufferParams::PARSEP_SKIP) {
			if (layout.latextype == LATEX_PARAGRAPH
			    && firstpar->GetDepth() == 0
			    && firstpar->Previous())
				y_top += bview->buffer()->params.getDefSkip().inPixels(bview);
			else if (firstpar->Previous()
				 && textclasslist.Style(bview->buffer()->params.textclass,
							firstpar->Previous()->GetLayout()).latextype == LATEX_PARAGRAPH
				 && firstpar->Previous()->GetDepth() == 0)
				// is it right to use defskip here, too? (AS) 
				y_top += bview->buffer()->params.getDefSkip().inPixels(bview);
		}
		
		if (row_ptr->par()->line_top) {      /* draw a top line  */
			y_top +=  lyxfont::ascent('x',
						  GetFont(bview->buffer(),
							  row_ptr->par(), 0));
			int const w = (inset_owner ?
				       inset_owner->width(bview, font) : ww);
			int const xp = static_cast<int>(inset_owner ? x : 0);
			pain.line(xp, y_offset + y_top,
				  w, y_offset + y_top,
				  LColor::topline,
				  Painter::line_solid,
				  Painter::line_thick);
			
			y_top +=  lyxfont::ascent('x',GetFont(bview->buffer(),
							      row_ptr->par(), 0));
		}
		
		/* should we print a label? */ 
		if (layout.labeltype >= LABEL_STATIC
		    && (layout.labeltype != LABEL_STATIC
			|| layout.latextype != LATEX_ENVIRONMENT
			|| row_ptr->par()->IsFirstInSequence())) {
			font = GetFont(bview->buffer(), row_ptr->par(), -2);
			if (!row_ptr->par()->GetLabelstring().empty()) {
				tmpx = x;
				string const tmpstring =
					row_ptr->par()->GetLabelstring();
				
				if (layout.labeltype == LABEL_COUNTER_CHAPTER) {
					if (bview->buffer()->params.secnumdepth >= 0) {
						/* this is special code for the chapter layout. This is printed in
						 * an extra row and has a pagebreak at the top. */
						float spacing_val = 1.0;
						if (!row_ptr->par()->spacing.isDefault()) {
							spacing_val = row_ptr->par()->spacing.getValue();
						} else {
							spacing_val = bview->buffer()->params.spacing.getValue();
						}
   
						maxdesc = int(lyxfont::maxDescent(font) * layout.spacing.getValue() * spacing_val)
							+ int(layout.parsep) * DefaultHeight();
						if (is_rtl)
							tmpx = ww - LeftMargin(bview, row_ptr) - 
								lyxfont::width(tmpstring, font);
						pain.text(int(tmpx),
							  y_offset + row_ptr->baseline() - row_ptr->ascent_of_text() - maxdesc,
							  tmpstring, font);
					}
				} else {
					if (is_rtl) {
						tmpx = ww - LeftMargin(bview, row_ptr)
							+ lyxfont::width(layout.labelsep, font);
#ifndef NEW_INSETS
						if (row_ptr->par()->footnoteflag == LyXParagraph::OPEN_FOOTNOTE)  {
							LyXFont font(LyXFont::ALL_SANE);
							font.setSize(LyXFont::SIZE_SMALL);
							tmpx += lyxfont::width("Mwide-fixM", font);
						}
#endif
					} else
						tmpx = x - lyxfont::width(layout.labelsep, font)
							- lyxfont::width(tmpstring, font);

					/* draw it! */
					pain.text(int(tmpx),
						  y_offset + row_ptr->baseline(),
						  tmpstring, font);
				}
			}
			/* the labels at the top of an environment. More or less for bibliography */ 
		} else if (layout.labeltype == LABEL_TOP_ENVIRONMENT ||
			   layout.labeltype == LABEL_BIBLIO ||
			   layout.labeltype == LABEL_CENTERED_TOP_ENVIRONMENT) {
			if (row_ptr->par()->IsFirstInSequence()) {
				font = GetFont(bview->buffer(),
					       row_ptr->par(), -2);
				if (!row_ptr->par()->GetLabelstring().empty()) {
					string const tmpstring =
						row_ptr->par()->GetLabelstring();
					float spacing_val = 1.0;
					if (!row_ptr->par()->spacing.isDefault()) {
						spacing_val = row_ptr->par()->spacing.getValue();
					} else {
						spacing_val = bview->buffer()->params.spacing.getValue();
					}
   
					maxdesc = int(lyxfont::maxDescent(font) * layout.spacing.getValue() * spacing_val
						      + (layout.labelbottomsep * DefaultHeight()));
					
					tmpx = x;
					if (layout.labeltype == LABEL_CENTERED_TOP_ENVIRONMENT){
						tmpx = ( (is_rtl ? LeftMargin(bview, row_ptr) : x)
							 + ww - RightMargin(bview->buffer(), row_ptr) ) / 2; 
						tmpx -= lyxfont::width(tmpstring, font) / 2;
					} else if (is_rtl)
						tmpx = ww - LeftMargin(bview, row_ptr) - 
							lyxfont::width(tmpstring, font);
					pain.text(int(tmpx),
						  y_offset + row_ptr->baseline()
						  - row_ptr->ascent_of_text()
						  - maxdesc,
						  tmpstring, font);
				}
			}
		}
		if (layout.labeltype == LABEL_BIBLIO && row_ptr->par()->bibkey) {
			font = GetFont(bview->buffer(), row_ptr->par(), -1);
			if (is_rtl)
				tmpx = ww - LeftMargin(bview, row_ptr)
					+ lyxfont::width(layout.labelsep, font);
			else
				tmpx = x - lyxfont::width(layout.labelsep, font)
					- row_ptr->par()->bibkey->width(bview, font);
			row_ptr->par()->bibkey->draw(bview, font,
						   y_offset + row_ptr->baseline(), 
						   tmpx, clear_area);
		}
	}
	
	/* is it a last row? */
#ifndef NEW_INSETS
	LyXParagraph * par = row_ptr->par()->LastPhysicalPar();
#else
	LyXParagraph * par = row_ptr->par();
#endif
	if (
#ifndef NEW_INSETS
		row_ptr->par()->ParFromPos(last + 1) == par
#else
		row_ptr->par() == par
#endif
		&& (!row_ptr->next() || row_ptr->next()->par() != row_ptr->par()))
	{
		/* think about the margins */ 
		if (!row_ptr->next() && bv_owner)
			y_bottom -= LYX_PAPER_MARGIN;
		
		/* draw a bottom pagebreak */ 
		if (firstpar->pagebreak_bottom) {
			LyXFont pb_font;
			pb_font.setColor(LColor::pagebreak).decSize();
			int w = 0, a = 0, d = 0;
			pain.line(0,
				  y_offset + y_bottom - 2 * DefaultHeight(), 
				  ww, 
				  y_offset + y_bottom - 2 * DefaultHeight(),
				  LColor::pagebreak,
				  Painter::line_onoffdash)
				.rectText(0,
					  0,
					  _("Page Break (bottom)"),
					  pb_font,
					  LColor::background,
					  LColor::background, false, w, a, d);
			pain.rectText((ww - w) / 2,
				      y_offset + y_top + 2 * DefaultHeight() + d,
				      _("Page Break (bottom)"),
				      pb_font,
				      LColor::background,
				      LColor::background);
			y_bottom -= 3 * DefaultHeight();
		}
		
		if (firstpar->added_space_bottom.kind() == VSpace::VFILL) {
			/* draw a vfill bottom  */
			pain.line(0, y_offset + y_bottom - 3 * DefaultHeight(),
				  LYX_PAPER_MARGIN,
				  y_offset + y_bottom - 3 * DefaultHeight(),
				  LColor::vfillline);
			pain.line(0, y_offset + y_bottom - 2,
				  LYX_PAPER_MARGIN,
				  y_offset + y_bottom - 2,
				  LColor::vfillline);
			pain.line(LYX_PAPER_MARGIN / 2,
				  y_offset + y_bottom - 3 * DefaultHeight(),
				  LYX_PAPER_MARGIN / 2,
				  y_offset + y_bottom - 2,
				  LColor::vfillline);
			y_bottom -= 3 * DefaultHeight();
		}
		
		/* think about user added space */ 
		y_bottom -= int(firstpar->added_space_bottom.inPixels(bview));
		
		if (firstpar->line_bottom) {
			/* draw a bottom line */
			y_bottom -= lyxfont::ascent('x', GetFont(bview->buffer(),
								 par, par->Last() - 1));
			int const w = (inset_owner ?
				       inset_owner->width(bview, font) : ww);
			int const xp = static_cast<int>(inset_owner ? x : 0);
			pain.line(xp, y_offset + y_bottom,
				  w, y_offset + y_bottom,
				  LColor::topline, Painter::line_solid,
				  Painter::line_thick);
			y_bottom -= lyxfont::ascent('x',
						    GetFont(bview->buffer(),
							    par,
							    par->Last() - 1));
		}

		// draw an endlabel
		int const endlabel =
			row_ptr->par()->GetEndLabel(bview->buffer()->params);
		switch (endlabel) {
		case END_LABEL_BOX:
		case END_LABEL_FILLED_BOX:
		{
			LyXFont const font = GetFont(bview->buffer(),
						     row_ptr->par(), last);
			int const size = int(0.75 * lyxfont::maxAscent(font));
			int const y = (y_offset + row_ptr->baseline()) - size;
			int x = is_rtl ? LYX_PAPER_MARGIN 
				: ww - LYX_PAPER_MARGIN - size;
#ifndef NEW_INSETS
			if (row_ptr->par()->footnoteflag == LyXParagraph::OPEN_FOOTNOTE)
				if (is_rtl) {
					LyXFont font(LyXFont::ALL_SANE);
					font.setSize(LyXFont::SIZE_SMALL);
					x += lyxfont::width("Mwide-figM", font);
				} else
					x -= LYX_PAPER_MARGIN/2;
#endif
			if (row_ptr->fill() <= size)
				x += (size - row_ptr->fill() + 1) * (is_rtl ? -1 : 1);
			if (endlabel == END_LABEL_BOX) {
				pain.line(x, y, x, y + size,
					  LColor::eolmarker);
				pain.line(x + size, y, x + size , y + size,
					  LColor::eolmarker);
				pain.line(x, y, x + size, y,
					  LColor::eolmarker);
				pain.line(x, y + size, x + size, y + size,
					  LColor::eolmarker);
			} else
				pain.fillRectangle(x, y, size, size,
						   LColor::eolmarker);
			break;
		}
		case END_LABEL_STATIC:
		{
			LyXTextClass::LayoutList::size_type layout = row_ptr->par()->GetLayout();
			string const tmpstring = textclasslist.
				Style(bview->buffer()->params.textclass,
				      layout).endlabelstring();
			font = GetFont(bview->buffer(), row_ptr->par(), -2);
			int const tmpx = is_rtl ?
				int(x) - lyxfont::width(tmpstring, font)
				: ww - RightMargin(bview->buffer(), row_ptr) - row_ptr->fill();
			pain.text( tmpx, y_offset + row_ptr->baseline(), tmpstring, font);
			break;
		}
		case END_LABEL_NO_LABEL:
			break;
		}
	}
	
	/* draw the text in the pixmap */  
	
	vpos = row_ptr->pos();

	LyXParagraph::size_type main_body = 
		BeginningOfMainBody(bview->buffer(), row_ptr->par());
	if (main_body > 0 &&
	    (main_body-1 > last || 
	     !row_ptr->par()->IsLineSeparator(main_body - 1)))
		main_body = 0;
	
	while (vpos <= last)  {
		pos = vis2log(vpos);
		if (main_body > 0 && pos == main_body - 1) {
			x += fill_label_hfill
				+ lyxfont::width(layout.labelsep,
						 GetFont(bview->buffer(),
							 row_ptr->par(), -2))
				- SingleWidth(bview,
					      row_ptr->par(),
					      main_body - 1);
		}
		
		if (row_ptr->par() ->IsHfill(pos)) {
			x += 1;
			pain.line(int(x),
				  y_offset + row_ptr->baseline() - DefaultHeight() / 2,
				  int(x),
				  y_offset + row_ptr->baseline(),
				  LColor::vfillline);
			
			if (HfillExpansion(bview->buffer(),
					   row_ptr, pos)) {
				if (pos >= main_body) {
					pain.line(int(x),
						  y_offset + row_ptr->baseline() - DefaultHeight() / 4,
						  int(x + fill_hfill),
						  y_offset + row_ptr->baseline() - DefaultHeight() / 4,
						  LColor::vfillline,
						  Painter::line_onoffdash);
					x += fill_hfill;
				} else {
					pain.line(int(x),
						  y_offset + row_ptr->baseline() - DefaultHeight() / 4,
						  int(x + fill_label_hfill),
						  y_offset + row_ptr->baseline() - DefaultHeight() / 4,
						  LColor::vfillline,
						  Painter::line_onoffdash);
					
					x += fill_label_hfill;
				}
				pain.line(int(x),
					  y_offset + row_ptr->baseline() - DefaultHeight() / 2,
					  int(x),
					  y_offset + row_ptr->baseline(),
					  LColor::vfillline);
			}
			x += 2;
			++vpos;
		} else if (row_ptr->par()->IsSeparator(pos)) {
			x += SingleWidth(bview,
					 row_ptr->par(), pos);
			if (pos >= main_body)
				x += fill_separator;
			++vpos;
		} else
			draw(bview, row_ptr, vpos, y_offset, x, clear_area);
	}
}


int LyXText::DefaultHeight() const
{
	LyXFont font(LyXFont::ALL_SANE);
	return int(lyxfont::maxAscent(font) + lyxfont::maxDescent(font) * 1.5);
}

   
/* returns the column near the specified x-coordinate of the row 
* x is set to the real beginning of this column  */ 
int LyXText::GetColumnNearX(BufferView * bview, Row * row, int & x,
			    bool & boundary) const
{
	float tmpx = 0.0;
	float fill_separator, fill_hfill, fill_label_hfill;
   
	PrepareToPrint(bview, row, tmpx, fill_separator,
		       fill_hfill, fill_label_hfill);

	LyXParagraph::size_type vc = row->pos();
	LyXParagraph::size_type last = RowLastPrintable(row);
	LyXParagraph::size_type c = 0;
	LyXLayout const & layout =
		textclasslist.Style(bview->buffer()->params.textclass,
				    row->par()->GetLayout());
	bool left_side = false;

	LyXParagraph::size_type
		main_body = BeginningOfMainBody(bview->buffer(), row->par());
	float last_tmpx = tmpx;
	
	if (main_body > 0 &&
	    (main_body-1 > last || 
	     !row->par()->IsLineSeparator(main_body - 1)))
		main_body = 0;
	
	while (vc <= last && tmpx <= x) {
		c = vis2log(vc);
		last_tmpx = tmpx;
		if (main_body > 0 && c == main_body-1) {
			tmpx += fill_label_hfill +
				lyxfont::width(layout.labelsep,
					       GetFont(bview->buffer(), row->par(), -2));
			if (row->par()->IsLineSeparator(main_body - 1))
				tmpx -= SingleWidth(bview, row->par(), main_body-1);
		}
		
		if (HfillExpansion(bview->buffer(), row, c)) {
			x += SingleWidth(bview, row->par(), c);
			if (c >= main_body)
				tmpx += fill_hfill;
			else
				tmpx += fill_label_hfill;
		}
		else if (row->par()->IsSeparator(c)) {
			tmpx += SingleWidth(bview, row->par(), c);
			if (c >= main_body)
				tmpx+= fill_separator;
		} else
			tmpx += SingleWidth(bview, row->par(), c);
		++vc;
	}
	
	if ((tmpx + last_tmpx) / 2 > x) {
		tmpx = last_tmpx;
		left_side = true;
	}

	if (vc > last + 1)  // This shouldn't happen.
		vc = last + 1;

	boundary = false;
	bool const lastrow = lyxrc.rtl_support // This is not needed, but gives
		                         // some speedup if rtl_support=false
		&& (!row->next() || row->next()->par() != row->par());
	bool const rtl = (lastrow)
		? row->par()->isRightToLeftPar(bview->buffer()->params)
		: false; // If lastrow is false, we don't need to compute
	                 // the value of rtl.

	if (row->pos() > last)  // Row is empty?
		c = row->pos();
	else if (lastrow &&
		 ( ( rtl &&  left_side && vc == row->pos() && x < tmpx - 5) ||
		   (!rtl && !left_side && vc == last + 1   && x > tmpx + 5) ))
		c = last + 1;
	else if (vc == row->pos()) {
		c = vis2log(vc);
		if (bidi_level(c) % 2 == 1)
			++c;
	} else {
		c = vis2log(vc - 1);
		bool const rtl = (bidi_level(c) % 2 == 1);
		if (left_side == rtl) {
			++c;
			boundary = IsBoundary(bview->buffer(), row->par(), c);
		}
	}

	if (row->pos() <= last && c > last
	    && row->par()->IsNewline(last)) {
		if (bidi_level(last) % 2 == 0)
			tmpx -= SingleWidth(bview, row->par(), last);
		else
			tmpx += SingleWidth(bview, row->par(), last);
		c = last;
	}

	c -= row->pos();
	x = int(tmpx);
	return c;
}


#ifndef NEW_INSETS
/* turn the selection into a new environment. If there is no selection,
* create an empty environment */ 
void LyXText::InsertFootnoteEnvironment(BufferView * bview, 
					LyXParagraph::footnote_kind kind)
{
   /* no footnoteenvironment in a footnoteenvironment */ 
   if (cursor.par()->footnoteflag != LyXParagraph::NO_FOOTNOTE) {
      WriteAlert(_("Impossible operation"), 
		 _("You can't insert a float in a float!"), 
		 _("Sorry."));
     return;
   }
   /* no marginpars in minipages */
   if (kind == LyXParagraph::MARGIN 
      && cursor.par()->pextra_type == LyXParagraph::PEXTRA_MINIPAGE) {
      WriteAlert(_("Impossible operation"), 
		 _("You can't insert a marginpar in a minipage!"), 
		 _("Sorry."));
      return;
   }
   
   /* this doesnt make sense, if there is no selection */ 
   bool dummy_selection = false;
   if (!selection) {
      sel_start_cursor = cursor;       /* dummy selection  */
      sel_end_cursor = cursor;
      dummy_selection = true;
   }
   
   LyXParagraph * tmppar;

   /* a test to make sure there is not already a footnote
    * in the selection. */
   
   tmppar = sel_start_cursor.par()->ParFromPos(sel_start_cursor.pos());
   
   while (tmppar != sel_end_cursor.par()->ParFromPos(sel_end_cursor.pos()) && 
	  tmppar->footnoteflag == LyXParagraph::NO_FOOTNOTE)
     tmppar = tmppar->next;
   
   if (tmppar != sel_end_cursor.par()->ParFromPos(sel_end_cursor.pos())
       || tmppar->footnoteflag != LyXParagraph::NO_FOOTNOTE) {
      WriteAlert(_("Impossible operation"), 
		 _("Float would include float!"), 
		 _("Sorry."));
      return;
   }
   
   /* ok we have a selection. This is always between sel_start_cursor
    * and sel_end cursor */

   SetUndo(bview->buffer(), Undo::FINISH, 
	   sel_start_cursor.par()->ParFromPos(sel_start_cursor.pos())->previous, 
	   sel_end_cursor.par()->ParFromPos(sel_end_cursor.pos())->next); 
   
   if (sel_end_cursor.pos() > 0 
       && sel_end_cursor.par()->IsLineSeparator(sel_end_cursor.pos() - 1))
	   sel_end_cursor.pos(sel_end_cursor.pos() - 1);
   /* please break before a space at the end */
   if (sel_start_cursor.par() == sel_end_cursor.par()
       && sel_start_cursor.pos() > sel_end_cursor.pos())
	   sel_start_cursor.pos(sel_start_cursor.pos() - 1);

   sel_end_cursor.par()->BreakParagraphConservative(bview->buffer()->params, sel_end_cursor.pos());
   
   sel_end_cursor.par(sel_end_cursor.par()->Next());
   sel_end_cursor.pos(0);
   
   // don't forget to insert a dummy layout paragraph if necessary
   if (sel_start_cursor.par()->GetLayout() != sel_end_cursor.par()->layout){
     sel_end_cursor.par()->BreakParagraphConservative(bview->buffer()->params, 0);
     sel_end_cursor.par()->layout = LYX_DUMMY_LAYOUT;
     sel_end_cursor.par(sel_end_cursor.par()->next);
   }
   else
     sel_end_cursor.par()->layout = LYX_DUMMY_LAYOUT;

   cursor = sel_end_cursor;

   /* please break behind a space, if there is one. The space should
    * be erased too */ 
   if (sel_start_cursor.pos() > 0 
       && sel_start_cursor.par()->IsLineSeparator(sel_start_cursor.pos() - 1))
     sel_start_cursor.pos(sel_start_cursor.pos() - 1);
   if (sel_start_cursor.par()->IsLineSeparator(sel_start_cursor.pos())) {
      sel_start_cursor.par()->Erase(sel_start_cursor.pos());
   }
   
   sel_start_cursor.par()->BreakParagraphConservative(bview->buffer()->params,
						    sel_start_cursor.pos());
   tmppar = sel_start_cursor.par()->Next();
   
   if (dummy_selection) {
	   tmppar->Clear();
	   if (kind == LyXParagraph::TAB
	       || kind == LyXParagraph::FIG 
               || kind == LyXParagraph::WIDE_TAB
	       || kind == LyXParagraph::WIDE_FIG 
               || kind == LyXParagraph::ALGORITHM) {
		   pair<bool, LyXTextClass::size_type> lres =
			   textclasslist.NumberOfLayout(bview->buffer()->params.textclass,
							"Caption");
		   LyXTextClass::size_type lay;
		   if (lres.first) {
			   // layout fount
			   lay = lres.second;
		   } else {
			   // layout not found
			   lay = 0; // use default layout "Standard" (0)
		   }
		   tmppar->SetLayout(bview->buffer()->params, lay);
	   }
   } else {
     if (sel_start_cursor.pos() > 0) {
       /* the footnote-environment should begin with a standard layout.
	* Imagine you insert a footnote within an enumeration, you 
	* certainly do not want an enumerated footnote! */ 
       tmppar->Clear();
     } else {
       /* this is a exception the user would sometimes expect, I hope */
       sel_start_cursor.par()->Clear();
     }
   }
   
   while (tmppar != sel_end_cursor.par()) {
      tmppar->footnoteflag = LyXParagraph::OPEN_FOOTNOTE;
      tmppar->footnotekind = kind;
      tmppar = tmppar->Next();
   }

   RedoParagraphs(bview, sel_start_cursor, sel_end_cursor.par()->Next());
   
   SetCursor(bview, sel_start_cursor.par()->Next(), 0);

   ClearSelection();
}
#endif


// returns pointer to a specified row
Row * LyXText::GetRow(LyXParagraph * par,
		      LyXParagraph::size_type pos, int & y) const
{
	if (!firstrow)
		return 0;

	Row * tmprow = firstrow;
	y = 0;
	
	// find the first row of the specified paragraph
	while (tmprow->next() && tmprow->par() != par) {
		y += tmprow->height();
		tmprow = tmprow->next();
	}
	
	// now find the wanted row
	while (tmprow->pos() < pos
	       && tmprow->next()
	       && tmprow->next()->par() == par
	       && tmprow->next()->pos() <= pos) {
		y += tmprow->height();
		tmprow = tmprow->next();
	}
	
	return tmprow;
}
