#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <ctype.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <strings.h>

#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>

#include "lopster.h"
#include "interface.h"
#include "commands.h"
#include "callbacks.h"
#include "connection.h"
#include "support.h"
#include "global.h"
#include "chat.h"
#include "search.h"
#include "transfer.h"
#include "scheme.h"
#include "hotlist.h"
#include "log.h"
#include "dialog.h"
#include "handler.h"
#include "resume.h"
#include "statistic.h"
#include "ping.h"

#define PING_LIFE     60

const int StatusInfo[S_NUMBER] = {
  T_NONE, T_CURRENT, T_CURRENT, T_NONE,
  T_TRANSFER, T_TRANSFER, T_NONE, T_NONE,
  T_NONE, T_NONE, T_CURRENT, T_CURRENT,
  T_CURRENT, T_CURRENT, T_NONE, T_NONE,
  T_NONE, T_NONE, T_NONE, T_NONE, 
  T_NONE, T_NONE
};

const int StatusDist[4][4] = {
  { 0, 1, 0, 1},
  {-1, 0,-1, 0},
  { 0, 1, 0, 1},
  {-1, 0,-1, 0}
};

char* status_names(int status) {
  switch (status) {
  case S_INACTIVE:   return _("Inactive");
  case S_CONNECTING: return _("Connecting...");
  case S_INFO:       return _("Getting info...");
  case S_DOWNLOADING:return _("Downloading");
  case S_CANCELED:   return _("Canceled");
  case S_FINISHED:   return _("Finished!");
  case S_TIMEOUT:    return _("Timeout!");
  case S_REJECT:     return _("Rejected");
  case S_INCOMPLETE: return _("Incomplete");
  case S_UPLOADING:  return _("Uploading");
  case S_INFO1:      return _("Getting info");
  case S_INFO2:      return _("Getting info");
  case S_INFO3:      return _("Getting info");
  case S_WAITING:    return _("Waiting");
  case S_FIREWALL:   return _("Both Firewalled");
  case S_CONERROR:   return _("Connection Error");
  case S_QUEUED:     return _("Queued");
  case S_REMOTE:     return _("Remotely Queued");
  case S_UNAVAILABLE:return _("Unavailable!");
  case S_RESUME_ERR: return _("Resume Error");
  case S_DELETE:     return _("Deleted");
  case S_IO:         return _("IO error");
  default: return _("unknown state");
  }
}

GtkWidget* create_upload_popup(transfer_t* transfer) {
  GtkWidget *popup;
  GtkWidget *user_popup;
  GtkWidget *item;
  GtkAccelGroup *popup_accels;
  GtkWidget *delete_upload;
  GtkWidget *separator;
  GtkWidget *trennlinie16;
  GtkWidget *customize_list2;
  GtkCList* clist;
  int item_num;
  char item_str[1024];

  popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (popup));

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  item_num = g_list_length(clist->selection);
  
  if (transfer) {
    if (item_num > 1) sprintf(item_str, _("Delete Selected (%d)"), item_num);
    else sprintf(item_str, _("Delete Upload"));
    delete_upload = gtk_menu_item_new_with_label (item_str);
    gtk_widget_show (delete_upload);
    gtk_container_add (GTK_CONTAINER (popup), delete_upload);
    gtk_signal_connect (GTK_OBJECT (delete_upload), "activate",
			GTK_SIGNAL_FUNC (on_delete_transfer_activate),
			NULL);
    
    separator = gtk_menu_item_new ();
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);

    if (!transfer_in_progress(transfer) && (transfer->status != S_FINISHED)) {
      if (item_num > 1) sprintf(item_str, _("Allow Selected (%d)"), item_num);
      else sprintf(item_str, _("Allow Upload"));
      delete_upload = gtk_menu_item_new_with_label (item_str);
      gtk_widget_show (delete_upload);
      gtk_container_add (GTK_CONTAINER (popup), delete_upload);
      gtk_signal_connect (GTK_OBJECT (delete_upload), "activate",
			  GTK_SIGNAL_FUNC (on_allow_upload_activate),
			  NULL);
      
      separator = gtk_menu_item_new ();
      gtk_widget_show (separator);
      gtk_container_add (GTK_CONTAINER (popup), separator);
      gtk_widget_set_sensitive (separator, FALSE);
    }

    delete_upload = gtk_menu_item_new_with_label (_("Open File"));
    gtk_widget_show (delete_upload);
    gtk_container_add (GTK_CONTAINER (popup), delete_upload);
    
    gtk_signal_connect (GTK_OBJECT (delete_upload), "activate",
			GTK_SIGNAL_FUNC (on_play_file_activate),
			NULL);

    separator = gtk_menu_item_new ();
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);

    item = gtk_menu_item_new_with_label (_("User Menu"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);

    user_popup = create_user_popup(M_TRANSFER);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), user_popup);

    trennlinie16 = gtk_menu_item_new ();
    gtk_widget_show (trennlinie16);
    gtk_container_add (GTK_CONTAINER (popup), trennlinie16);
    gtk_widget_set_sensitive (trennlinie16, FALSE);
  }

  customize_list2 = gtk_menu_item_new_with_label (_("Customize List"));
  gtk_widget_show (customize_list2);
  gtk_container_add (GTK_CONTAINER (popup), customize_list2);
  gtk_signal_connect (GTK_OBJECT (customize_list2), "activate",
                      GTK_SIGNAL_FUNC (on_customize_list_activate),
                      NULL);

  return popup;
}

GtkWidget*
create_download_popup (transfer_t* transfer) {
  GtkWidget *user_popup;
  GtkWidget *popup;
  GtkAccelGroup *popup_accels;
  GtkWidget *cancel_download;
  GtkWidget *delete_download;
  GtkWidget *separator3;
  GtkWidget *retry_download;
  GtkWidget *separator4;
  GtkWidget *item;
  GtkWidget *trennlinie7;
  GtkWidget *play_file;
  GtkWidget *trennlinie16;
  GtkWidget *customize_list2;
  GtkCList* clist;
  int item_num;
  char item_str[1024];

  popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (popup));
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  item_num = g_list_length(clist->selection);

  if (transfer) {
    if (item_num > 1) sprintf(item_str, _("Cancel Selected (%d)"), item_num);
    else sprintf(item_str, _("Cancel Download"));
    cancel_download = gtk_menu_item_new_with_label (item_str);
    gtk_widget_ref (cancel_download);
    gtk_object_set_data_full (GTK_OBJECT (popup), "cancel_download", cancel_download,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (cancel_download);
    gtk_container_add (GTK_CONTAINER (popup), cancel_download);
    gtk_signal_connect (GTK_OBJECT (cancel_download), "activate",
			GTK_SIGNAL_FUNC (on_cancel_transfer_activate),
			NULL);
    
    if (item_num > 1) sprintf(item_str, _("Delete Selected (%d)"), item_num);
    else sprintf(item_str, _("Delete Download"));
    delete_download = gtk_menu_item_new_with_label (item_str);
    gtk_widget_ref (delete_download);
    gtk_object_set_data_full (GTK_OBJECT (popup), "delete_download", delete_download,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (delete_download);
    gtk_container_add (GTK_CONTAINER (popup), delete_download);
    gtk_signal_connect (GTK_OBJECT (delete_download), "activate",
			GTK_SIGNAL_FUNC (on_delete_transfer_activate),
			NULL);
    
    separator3 = gtk_menu_item_new ();
    gtk_widget_ref (separator3);
    gtk_object_set_data_full (GTK_OBJECT (popup), "separator3", separator3,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (separator3);
    gtk_container_add (GTK_CONTAINER (popup), separator3);
    gtk_widget_set_sensitive (separator3, FALSE);
    
    if (transfer->status == S_QUEUED) {
      retry_download = gtk_menu_item_new_with_label (_("Force Download"));
      gtk_widget_show (retry_download);
      gtk_container_add (GTK_CONTAINER (popup), retry_download);
      
      gtk_signal_connect (GTK_OBJECT (retry_download), "activate",
			  GTK_SIGNAL_FUNC (on_force_download_activate),
			  NULL);
    } else if (transfer_in_progress(transfer)) {
      if (item_num > 1) sprintf(item_str, _("Restart Selected (%d)"), item_num);
      else sprintf(item_str, _("Restart Download"));
      retry_download = gtk_menu_item_new_with_label (item_str);
      gtk_widget_show (retry_download);
      gtk_container_add (GTK_CONTAINER (popup), retry_download);
      
      gtk_signal_connect (GTK_OBJECT (retry_download), "activate",
			  GTK_SIGNAL_FUNC (on_retry_download_activate),
			  NULL);
    } else if (transfer->status != S_FINISHED) {
      if (transfer->is_dcc) sprintf(item_str, _("Start DCC"));
      else if (item_num > 1) sprintf(item_str, _("Retry Selected (%d)"), item_num);
      else sprintf(item_str, _("Retry Download"));
      retry_download = gtk_menu_item_new_with_label (item_str);
      gtk_widget_show (retry_download);
      gtk_container_add (GTK_CONTAINER (popup), retry_download);
      
      gtk_signal_connect (GTK_OBJECT (retry_download), "activate",
			  GTK_SIGNAL_FUNC (on_retry_download_activate),
			  NULL);
    } else {
      sprintf(item_str, _("Retry Download"));
      retry_download = gtk_menu_item_new_with_label (item_str);
      gtk_widget_show (retry_download);
      gtk_container_add (GTK_CONTAINER (popup), retry_download);
      gtk_widget_set_sensitive(retry_download, FALSE);
    }
    
    separator4 = gtk_menu_item_new ();
    gtk_widget_ref (separator4);
    gtk_object_set_data_full (GTK_OBJECT (popup), "separator4", separator4,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (separator4);
    gtk_container_add (GTK_CONTAINER (popup), separator4);
    gtk_widget_set_sensitive (separator4, FALSE);
    
    item = gtk_menu_item_new_with_label (_("User Menu"));
    gtk_widget_ref (item);
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);

    user_popup = create_user_popup(M_TRANSFER);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), user_popup);

    trennlinie7 = gtk_menu_item_new ();
    gtk_widget_ref (trennlinie7);
    gtk_object_set_data_full (GTK_OBJECT (popup), "trennlinie7", trennlinie7,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (trennlinie7);
    gtk_container_add (GTK_CONTAINER (popup), trennlinie7);
    gtk_widget_set_sensitive (trennlinie7, FALSE);
    
    play_file = gtk_menu_item_new_with_label (_("Open File"));
    gtk_widget_ref (play_file);
    gtk_object_set_data_full (GTK_OBJECT (popup), "play_file", play_file,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (play_file);
    gtk_container_add (GTK_CONTAINER (popup), play_file);
    
    gtk_signal_connect (GTK_OBJECT (play_file), "activate",
			GTK_SIGNAL_FUNC (on_play_file_activate),
			NULL);
    trennlinie16 = gtk_menu_item_new ();
    gtk_widget_ref (trennlinie16);
    gtk_object_set_data_full (GTK_OBJECT (popup), "trennlinie16", trennlinie16,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (trennlinie16);
    gtk_container_add (GTK_CONTAINER (popup), trennlinie16);
    gtk_widget_set_sensitive (trennlinie16, FALSE);
  }

  customize_list2 = gtk_menu_item_new_with_label (_("Customize List"));
  gtk_widget_ref (customize_list2);
  gtk_object_set_data_full (GTK_OBJECT (popup), "customize_list2", customize_list2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (customize_list2);
  gtk_container_add (GTK_CONTAINER (popup), customize_list2);
  gtk_signal_connect (GTK_OBJECT (customize_list2), "activate",
                      GTK_SIGNAL_FUNC (on_customize_list_activate),
                      NULL);

  return popup;
}

char* valid_download_folder(int mime) {
  if (!global.incomplete_path ||
      !directory_exists(global.incomplete_path))
    return NULL;
  if (!global.mimetype[mime].download ||
      !directory_exists(global.mimetype[mime].download))
    return NULL;
  return global.mimetype[mime].download;
}

char* rename_file(char* filename) {
  struct stat st;
  char* result;
  char* base = strdup(filename);
  char* suff;
  int cnt = 1;

  suff = strrchr(base, '.');
  if (suff) {
    *suff = 0;
    suff++;
  }
  free(filename);
  while (1) {
    if (suff) result = g_strdup_printf("%s_%d.%s", base, cnt, suff);
    else result = g_strdup_printf("%s_%d", base, cnt);
    if (stat(result, &st) < 0) break;
    free(result);
    result = NULL;
    if (cnt >= 100) break;
    cnt++;
  }
  free(base);

  return result;
}

transfer_t* transfer_new() {
  transfer_t *transfer;
  int i1;

  transfer = (transfer_t*)malloc(sizeof(transfer_t));

  transfer->longname = NULL;        //
  transfer->shortname = NULL;       //
  transfer->winname = NULL;         //
  transfer->download_dir = NULL;
  transfer->md5 = NULL;
  transfer->user = NULL;
  transfer->linespeed = 0;
  transfer->progress = 0;
  transfer->rate = 0;
  transfer->timeleft = 0;
  transfer->start_time = 0;
  transfer->status = S_INACTIVE;
  transfer->size = 0;

  transfer->file = NULL;
  transfer->type = T_UNKNOWN;
  transfer->resume_check = NULL;
  transfer->check_length = 0;
  transfer->resume = NULL;
  transfer->user_info = NULL;
  transfer->is_dcc = FALSE;
  transfer->mime_type = MIME_NONE;
  transfer->start_size = 0;

  for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
    transfer->history[i1] = transfer->start_size;
  transfer->hist_pos = 0;
  transfer->hist_cnt = 0;

  return transfer;
}

void transfer_set_md5(transfer_t* transfer, char* md5) {
  if (transfer->md5) free(transfer->md5);
  transfer->md5 = strdup(md5);
}

int transfer_in_progress(transfer_t* transfer) {
  if ((StatusInfo[transfer->status] == T_CURRENT) ||
      (StatusInfo[transfer->status] == T_TRANSFER)) return 1;
  else return 0;
}

int transfer_status_dist(int s1, int s2) {
  return StatusDist[StatusInfo[s1]][StatusInfo[s2]];
}

void transfer_status_set(socket_t* socket, int status) {
  transfer_t* transfer;
  int add;
  GtkCList* clist;
  int row;
  style_t* style;
  double val;
  char str[1024];

  if (!socket) return;
  transfer = (transfer_t*)(socket->data);
  if (!transfer) return;
  add = transfer_status_dist(transfer->status, status);
  
  // test
  if ((transfer->type == T_DOWNLOAD) && (transfer->resume))
    resume_user_flag(transfer->resume, transfer->user, status);
  // end test

#ifdef TRANSFER_DEBUG  
  printf("old/new %d/%d :%d\n", transfer->status, status, add);
#endif
  if (transfer->type == T_DOWNLOAD) {
    if (transfer->user_info) transfer->user_info->cur += add;
    global.limit.cur_downloads += add;
#ifdef TRANSFER_DEBUG  
    printf("down now %d\n", global.limit.cur_downloads);
#endif
    if (status == S_DOWNLOADING) {
      global.statistic.no_download[0]++;
      global.statistic.no_download[status]++;
    }
    if (transfer->status == S_DOWNLOADING) {
      global.statistic.no_download[S_DOWNLOADING]--;
      global.statistic.no_download[status]++;
    }
  }
  if (transfer->type == T_UPLOAD) {
    if (transfer->user_info) transfer->user_info->cur += add;
    global.limit.cur_uploads += add;
    if (transfer->size >= (long)(global.limit.large_size)*1024*1024)
      global.limit.cur_large += add;
#ifdef TRANSFER_DEBUG  
    printf("up now %d\n", global.limit.cur_uploads);
#endif
    if (status == S_UPLOADING) {
      global.statistic.no_upload[0]++;
      global.statistic.no_upload[status]++;
    }
    if (transfer->status == S_UPLOADING) {
      global.statistic.no_upload[S_UPLOADING]--;
      global.statistic.no_upload[status]++;
    }
  }
    
  if (transfer->status == S_DOWNLOADING) {
    send_command(CMD_CLIENT_DOWNLOAD_END, "");
    global.limit.cur_real_downloads--;
  }
  if (status == S_DOWNLOADING) {
    send_command(CMD_CLIENT_DOWNLOAD_START, "");
    global.limit.cur_real_downloads++;
  }
  if (transfer->status == S_UPLOADING) {
    send_command(CMD_CLIENT_UPLOAD_END, "");
    global.limit.cur_real_uploads--;
  }
  if (status == S_UPLOADING) {
    send_command(CMD_CLIENT_UPLOAD_START, "");
    global.limit.cur_real_uploads++;
  }

  if (transfer->type == T_UPLOAD) {
    if (status == S_UPLOADING)
      log("uploads", LOG_OTHER, "Uploading [%s] to [%s]%s at [%ld]\n",
	  transfer->shortname, transfer->user, 
	  transfer->is_dcc?" (DCC)":"",
	  transfer->start_size);
    else if (transfer->status == S_UPLOADING) {
      if (transfer->end_time-transfer->start_time > 0)
	val = (transfer->progress-transfer->start_size) /
	  (transfer->end_time-transfer->start_time);
      else val = 0;
      print_speed(str, val, 1);
      log("uploads", LOG_OTHER, "Ending [%s] to [%s]%s :%s: [%ld] (%s)\n",
	  (transfer->shortname)?transfer->shortname:"???",
	  (transfer->user)?transfer->user:"???",
	  transfer->is_dcc?" (DCC)":"", status_names(status), 
	  transfer->progress-transfer->start_size, str);
    }
  } else if (transfer->type == T_UNKNOWN) {
    log("unknown_transfer", LOG_OTHER, "Ending [%s] [%s]%s\n",
	transfer->shortname, transfer->user, 
	transfer->is_dcc?" (DCC)":"");
    clist = NULL;
  } else {
    if (status == S_DOWNLOADING)
      log("downloads", LOG_OTHER, "Downloading [%s] from [%s]%s at [%ld]\n",
	  transfer->shortname, transfer->user, 
	  transfer->is_dcc?" (DCC)":"",
	  transfer->start_size);
    else if (transfer->status == S_DOWNLOADING) {
      if (transfer->end_time-transfer->start_time > 0)
	val = (transfer->progress-transfer->start_size) /
	  (transfer->end_time-transfer->start_time);
      else val = 0;
      print_speed(str, val, 1);
      log("downloads", LOG_OTHER, "Ending [%s] from [%s]%s :%s: [%ld] (%s)\n",
	  (transfer->shortname)?transfer->shortname:"???",
	  (transfer->user)?transfer->user:"???", 
	  transfer->is_dcc?" (DCC)":"", status_names(status),
	  transfer->progress-transfer->start_size, str);
    }
  }

  transfer->status = status;
  transfer_update(socket, 1);

  if (transfer->type == T_UPLOAD) {
    clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  } else {
    clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  }

  row = gtk_clist_find_row_from_data(clist, socket);
  if (row < 0) return;

  if ((socket->timeout >= 0) || (transfer->status == S_QUEUED)) {
    style = style_get(global.scheme, "transfer_waiting");
    if (style) {
      gtk_clist_set_background(clist, row, style->back);
      gtk_clist_set_foreground(clist, row, style->fore);
    }
  } else  {
    if (transfer_in_progress(transfer)) {
      style = style_get(global.scheme, "transfer_running");
      if (style) {
	gtk_clist_set_background(clist, row, style->back);
	gtk_clist_set_foreground(clist, row, style->fore);
      }
    } else {
      gtk_clist_set_background(clist, row, NULL);
      gtk_clist_set_foreground(clist, row, NULL);
    }
  }
}

file_t* lib_search_transfer(transfer_t* transfer) {
  GList* dlist;
  file_t* file;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    if (!strcmp(transfer->winname, file->winname)) return file;
  }
  return NULL;
}

void transfer_connect_and_start(socket_t* socket) {
  transfer_t* transfer = (transfer_t*)(socket->data);
  
  transfer_status_set(socket, S_CONNECTING);
  if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
    send_command(CMD_CLIENT_DATA_PORT_ERROR, transfer->user);
    return;
  }
  
  socket->input = 
    gdk_input_add(socket->fd, GDK_INPUT_READ, 
		  GTK_SIGNAL_FUNC(await_conn_ack), socket);
}

int download_allowed(transfer_t* transfer) {
  int add = transfer_status_dist(transfer->status, S_INACTIVE);

  if (transfer->resume && transfer->resume->transfer &&
      transfer->resume->transfer != transfer) return 0;

  if (global.limit.max_downloads <= global.limit.cur_downloads+add)
    return 0;
  if (transfer->user_info) {
    if ((transfer->user_info->max == -1) &&
	(transfer->user_info->cur+add >= global.limit.default_downloads))
      return 0;
    if ((transfer->user_info->max >= 0) &&
	(transfer->user_info->cur+add >= transfer->user_info->max))
      return 0;
  } else {
    g_warning("no user info found [%s]", transfer->user);
  }
  return 1;
}

int upload_allowed(transfer_t* transfer) {
  struct stat st;
  int add = transfer_status_dist(transfer->status, S_INACTIVE);


  if (stat(transfer->longname, &st) >= 0) {
    transfer->size = st.st_size;
  }

  if (!is_string_in_list(global.frienduser, transfer->user)) {
    if (global.limit.max_uploads <= global.limit.cur_uploads+add)
      return 0;
    if ((global.limit.max_large <= global.limit.cur_large+add) &&
	(transfer->size > (long)global.limit.large_size*1024*1024))
      return 0;
  }

  if (transfer->user_info) {
    if ((transfer->user_info->max == -1) &&
	(transfer->user_info->cur+add >= global.limit.default_uploads))
      return 0;
    if ((transfer->user_info->max >= 0) &&
	(transfer->user_info->cur+add >= transfer->user_info->max))
      return 0;
  } else {
    g_warning("no user info found [%s]", transfer->user);
  }

  return 1;
}

void download_start(socket_t* socket, int force) {
  char* t2;
  transfer_t* transfer = (transfer_t*)(socket->data);
  resume_t* resume;

  // removing timeout if set
  if (socket->timeout >= 0) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
  }

#ifdef TRANSFER_DEBUG  
  printf("starting down [%s]\n", transfer->winname);
#endif
  transfer->hist_cnt = 0;
  socket->cnt = 0;

  resume = resume_list_search(transfer);
  if (!resume) {
    // no resume found, setting up new one
    //    printf("resume not found\n");
    resume = transfer_create_resume(transfer);
    if (!resume) {
      //      printf("could not create new resume\n");
      socket_end(socket, &(SocketStatus[S_CANCELED]));
      return;
    }
    resume_list_add(resume);
  }

  if (resume->status == R_FROZEN) {
    socket_end(socket, &(SocketStatus[S_CANCELED]));
  }
  if (!transfer_grab_resume(transfer, resume)) {
    //    printf("could not grab resume\n");
    socket_end(socket, &(SocketStatus[S_QUEUED]));
    return;
  }

  // now starting the download
  if (transfer->is_dcc) {
    if (force || (download_allowed(transfer))) {
      transfer_connect_and_start(socket);
    } else {
      socket_end(socket, &(SocketStatus[S_QUEUED]));
    }   
  } else if (force || (download_allowed(transfer))) {
    t2 = g_strdup_printf("%s \"%s\"", transfer->user, transfer->winname);
    send_command(CMD_CLIENT_DOWNLOAD, t2);
    g_free(t2);
    transfer_status_set(socket, S_WAITING);
  } else {
    socket_end(socket, &(SocketStatus[S_QUEUED]));
  }
}

void upload_start(socket_t* socket, int force) {
  char* t2;
  transfer_t* transfer = (transfer_t*)(socket->data);

  socket->cnt = 0;
#ifdef TRANSFER_DEBUG  
  printf("starting up [%s]\n", transfer->winname);
#endif
  if (transfer->is_dcc) {
    transfer_status_set(socket, S_WAITING);
  } else {
    if (upload_allowed(transfer) || force) {
      t2 = g_strdup_printf("%s \"%s\"", transfer->user,
			   transfer->winname);
      send_command(CMD_CLIENT_UPLOAD_OK, t2);

      transfer_status_set(socket, S_WAITING);
      /////////////
      //      access_new_request(transfer);
      /////////////
    } else {
      t2 = g_strdup_printf("%s \"%s\" %d", transfer->user,
			   transfer->winname, global.limit.cur_uploads);
      send_command(CMD_CLIENT_LIMIT, t2);

      //      socket_end(socket, S_REJECT);
      socket_end(socket, &(SocketStatus[S_QUEUED]));
    }
    g_free(t2);
  }
}

char* extract_last_dir(char* filename) {
  static char dir[2048];
  char* pos1;
  char* pos2;

  pos1 = strchr(filename, '/');
  pos2 = strrchr(filename, '/');
  if (!pos1) return NULL;
  if (pos1 == pos2) return NULL;
  
  pos2[0] = 0;
  pos1 = strrchr(filename, '/')+1;
  strcpy(dir, pos1);
  pos2[0] = '/';
  
  return dir;
}

char* extract_filename(char* filename) {
  char* pos1;

  pos1 = strrchr(filename, '/');
  if (!pos1) return filename;
  else return (pos1+1);
}

void convert_local_name(char* fname) {
  if (!fname) return;
  while (*fname) {
    if (*fname == '`') *fname = '_';
    if (*fname == '\'') *fname = '_';
    if (*fname == '') *fname = '_';
    fname++;
  }
}

void download_file(file_t* file, int withdir) {
  transfer_t* new_trans;
  char* dir;
  char* folder;
  socket_t* socket;
  transfer_t* transfer;

  if ((folder = valid_download_folder(file->mime_type)) == NULL) {
    download_dialog(file->shortname, file->mime_type);
    return;
  }
  socket = transfer_is_in_download(file->user, file->winname);
  if (socket) transfer = (transfer_t*)(socket->data);

  if (!socket) {
    new_trans = transfer_new();
    new_trans->longname = strdup(file->longname);
    new_trans->shortname = strdup(file->shortname);
    new_trans->winname = strdup(file->winname);
    new_trans->linespeed = file->linespeed;
    new_trans->user = strdup(file->user);
    new_trans->size = file->size;
    new_trans->type = T_DOWNLOAD;
    new_trans->mime_type = file->mime_type;
    socket = socket_new(S_TRANSFER);
    socket->data = new_trans;
    
    if (withdir) {
      dir = extract_last_dir(new_trans->longname);
      if (dir) {
	new_trans->download_dir = strdup(dir);
      } else
	new_trans->download_dir = NULL;
    } else {
      new_trans->download_dir = NULL;
    }
    
#ifdef TRANSFER_DEBUG
    printf("folder = [%s][%d]\n", 
	   (new_trans->download_dir)?(new_trans->download_dir):"(null)",
	   file->mime_type);
#endif
    transfer_insert(socket);
  } else {
    new_trans = (transfer_t*)(socket->data);
    new_trans->size = file->size;
  }
  download_start(socket, FALSE);
}

socket_t* transfer_is_in_download(char* user, char* winname) {
  GtkCList* clist;
  int i1;
  socket_t* socket;
  transfer_t* transfer;
  
  //  printf("searching [%s][%s]\n", user, winname);
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) {
      socket = NULL;
      break;
    }
    socket = (socket_t*)gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t*)(socket->data);
    //    printf("dest [%s][%s] %d\n", transfer->user, transfer->winname, transfer->status);
    if (!strcmp(transfer->winname, winname) &&
	!strcmp(transfer->user, user) &&
	(!transfer_in_progress(transfer) || (transfer->status == S_WAITING)) &&
	(transfer->status != S_FINISHED) &&
	(transfer->status != S_CANCELED) &&
	(transfer->status != S_DELETE))
      return socket;
    i1++;
  }
  
  return NULL;
}

socket_t* transfer_is_in_upload(char* user, char* winname) {
  GtkCList* clist;
  int i1;
  socket_t* socket;
  transfer_t* transfer;
  
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) return NULL;
    socket = (socket_t*)gtk_clist_get_row_data(clist, i1);
    if (!socket) {
      g_warning("socket should not be == NULL\n");
      i1++;
      continue;
    }
    transfer = (transfer_t*)(socket->data);
    if (!transfer) {
      g_warning("transfer should not be == NULL\n");
      i1++;
      continue;
    }
    if (!strcmp(transfer->winname, winname) &&
	!strcmp(transfer->user, user) &&
	(!transfer_in_progress(transfer) || (transfer->status == S_WAITING)) &&
	(transfer->status != S_FINISHED) && 
	(transfer->status != S_CANCELED) &&
	(transfer->status != S_DELETE))
      return socket;
    i1++;
  }

  return NULL;
}

/*
  thanks to Matthew Pratt <mattpratt@yahoo.com> for this code
  I made a few changes....
*/
GdkPixmap *transfer_draw_progress(GtkCList *clist, GdkPixmap *pixmap, double percent, 
				  int width, int active){
  int height;
  char text[128];
  static GdkGC *gc = NULL;
  GdkColor background = { 0, 0xbf00, 0xbf00, 0xbf00 };
  GdkWindowPrivate* temp_pix;
  style_t* style1;
  style_t* style2;
  GdkFont* font;

  style1 = style_get(global.scheme, "transfer_bar1");
  if (!style1) return NULL;
  style2 = style_get(global.scheme, "transfer_bar2");
  if (!style2) return NULL;
  
  if (!gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), &background )
      || !gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), style1->fore )
      || !gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), style1->back )
      || !gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), style2->fore )
      || !gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), style2->back )) {
    g_error("couldn't allocate colour");
  }

  height = GTK_CLIST(clist)->row_height;
  temp_pix = (GdkWindowPrivate*)pixmap;

  if (!pixmap || (temp_pix->width != width) ||
      (temp_pix->height != height)) {
    if (pixmap) {
      gdk_pixmap_unref (pixmap);
      temp_pix = NULL;
      pixmap = NULL;
    }
    pixmap = gdk_pixmap_new( global.win->window, width, 
			     GTK_CLIST(clist)->row_height, -1 );
  }
  
  if (!gc) gc = gdk_gc_new(global.win->window);

  // draw the upper/left black lines
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  gdk_draw_rectangle(pixmap, gc, TRUE, 0, 0, width, height);
  
  // draw the lower/right white lines
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->white_gc);
  gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, width-1, height-1);
  
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  // draw the background (unfilled progress)
  gdk_gc_set_foreground(gc, &background);
  gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, width - 2, height - 2);
  
  // draw the actual progress bar
  // draw the lower/right white lines
  if ((int)((width-2)*percent) > 0) {
    gdk_gc_copy(gc, GTK_WIDGET(clist)->style->white_gc);
    gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, (int)((width-2)*percent), height - 2);
  }

  // draw the upper/left black lines
  if ((int)((width-2)*percent)-1 > 0) {
    gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
    gdk_draw_rectangle(pixmap, gc, TRUE, 2, 2, (int)((width-2)*percent)-1, height - 3);
  }

  if ((int)((width-2)*percent)-2 > 0) {
    if (active) gdk_gc_set_foreground(gc, style1->back);
    else gdk_gc_set_foreground(gc, style2->back);
    gdk_draw_rectangle(pixmap, gc, TRUE, 2, 2, (int)((width-2)*percent)-2, height - 4);
  }
  
  // draw the text on top
  g_snprintf( text, sizeof(text), "%.1f%%", percent*100);
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  if (active) gdk_gc_set_foreground(gc, style1->fore);
  else gdk_gc_set_foreground(gc, style2->fore);

  font = GTK_WIDGET(clist)->style->font;
  gdk_draw_text(pixmap, font, gc, 
		(width-gdk_string_width(font, text))/2, height - font->descent,
		text, strlen(text));
  
  return pixmap;
}

int transfer_update(socket_t* socket, int full) {
  GtkCList* clist;
  int row;
  char str[200];
  double trate;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  int sec;
  transfer_t* transfer;

  if (!socket) return 1;
  transfer = (transfer_t*)(socket->data);
  if (transfer->resume) resume_list_update(transfer->resume, 0);

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  // incorrect
  row = gtk_clist_find_row_from_data(clist, socket);
  if (row < 0) {
    clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
    row = gtk_clist_find_row_from_data(clist, socket);
  }
  if (row < 0) {
    return 1;
  }

  if (full) {
    // filename
    if (transfer->type == T_DOWNLOAD) {
      if (transfer->shortname) strcpy(str, transfer->shortname);
      else strcpy(str, _("Not known yet"));
    } else if (transfer->type == T_UPLOAD) {
      if (transfer->shortname) strcpy(str, transfer->shortname);
      else strcpy(str, _("Not known yet"));
    }
    gtk_clist_set_text(clist, row, 0, str);

    // user
    if (transfer->user) strcpy(str, transfer->user);
    else strcpy(str, _("Not known yet"));
    gtk_clist_set_text(clist, row, 2, str);

    // linespeed
    strcpy(str, LineSpeed(transfer->linespeed));
    gtk_clist_set_text(clist, row, 4, str);
  }
  
  // filesize
  sprintf(str, "%ld/%ld", transfer->progress, transfer->size);
  gtk_clist_set_text(clist, row, 1, str);

  // status
  if ((transfer->status < 0) || (transfer->status >= S_NUMBER)) {
    g_warning("transfer_update: invalid status %d", 
	      transfer->status);
    return 1;
  }
  if (transfer->status == S_WAITING)
    sprintf(str, _("Waiting %d secs"), 
	    (socket->max_cnt - socket->cnt));
  else 
    strcpy(str, status_names(transfer->status));
  gtk_clist_set_text(clist, row, 3, str);

  // progress
  if ((transfer->status == S_DOWNLOADING) ||
      (transfer->status == S_FINISHED) ||
      (transfer->status == S_UPLOADING)) {
    int last_prog;
    int first_prog;

    if (transfer->size >0){
      gtk_clist_get_pixmap(clist, row, 5, &pixmap, &bitmap );
      
      pixmap = 
	transfer_draw_progress(clist, pixmap, 
			       (double)transfer->progress / 
			       (double)transfer->size, 
			       clist->column[5].width,
			       transfer->status != S_FINISHED);
      gtk_clist_set_pixmap(clist, row, 5, pixmap, NULL );
    } else {
      gtk_clist_set_pixmap(clist, row, 5, NULL, NULL );
    }
    
    transfer->hist_pos++;
    if (transfer->hist_pos >= TRANSFER_HISTORY_TIME)
      transfer->hist_pos = 0;
    transfer->history[transfer->hist_pos] = transfer->progress;
    if (transfer->hist_cnt < (TRANSFER_HISTORY_TIME/global.network.transfer_delay)-1)
      transfer->hist_cnt++;
    
    last_prog = transfer->hist_pos;
    first_prog = (last_prog+TRANSFER_HISTORY_TIME-transfer->hist_cnt)%
      TRANSFER_HISTORY_TIME;

    if (transfer->status == S_FINISHED) {
      if (transfer->end_time - transfer->start_time == 0) 
	transfer->end_time++;
      trate = (transfer->size-transfer->start_size)/
	(transfer->end_time - transfer->start_time);
    } else {
      trate = (double)(transfer->history[last_prog]-
		       transfer->history[first_prog])/
	transfer->hist_cnt/global.network.transfer_delay;
    }
#ifdef TRANSFER_DEBUG
    if (trate < 0) {
      printf("%d/%d/%d\n", first_prog, last_prog,
	     TRANSFER_HISTORY_TIME);
    }
#endif
    print_speed(str, (int)trate, 1);
    
    transfer->rate = trate;
    if (trate > 20*1024) {
      pixmap = global.pix.sgreen;
      bitmap = global.pix.sgreenb;
    } else if (trate > 4*1024) {
      pixmap = global.pix.syellow;
      bitmap = global.pix.syellowb;
    } else if (trate > 2*1024) {
      pixmap = global.pix.sred;
      bitmap = global.pix.sredb;
    } else if (trate > 0) {
      pixmap = global.pix.sgray;
      bitmap = global.pix.sgrayb;
    } else {
      pixmap = global.pix.dummy;
      bitmap = global.pix.dummyb;
    }
    gtk_clist_set_pixtext (clist, row, 6, str, 5, pixmap, bitmap);
    
    if ((transfer->status == S_FINISHED) || (trate > 0)) {
      if (transfer->status == S_FINISHED)
	sec = transfer->end_time - transfer->start_time;
      else
	sec = (int)((transfer->size-transfer->progress)/trate);
      transfer->timeleft = sec;
      
      print_time_average(str, sec);
    } else {
      sprintf(str, _("stalled"));
    }
    gtk_clist_set_text(clist, row, 7, str);
  } else {
    gtk_clist_set_text(clist, row, 6, "" );
    gtk_clist_set_text(clist, row, 7, "");
  }
  
  return 1;
}

void socket_destroy2(socket_t* socket) {
  if (!socket) return;

#ifdef TRANSFER_DEBUG
  printf("destroying socket2 [%d]\n", socket->fd);
#endif
  global.sockets = g_list_remove(global.sockets, socket);
  socket_remove_clist(socket);

  if (socket->data)
    transfer_destroy(socket->data);
  free(socket);
}

void transfer_insert(socket_t *socket) {
  GtkCList* temp;
  int row;
  transfer_t* transfer = (transfer_t*)(socket->data);

  if (transfer->type == T_DOWNLOAD) {
    transfer->user_info = detect_user_info(transfer->user, 1);
    temp = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  } else {
    transfer->user_info = detect_user_info(transfer->user, 0);
    temp = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  }

  tstr[0][0] = tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 
    tstr[5][0] = tstr[6][0] = 0;
  row = gtk_clist_append(temp, list);
  gtk_clist_set_row_data_full (temp, row, (gpointer)socket, 
			       (GtkDestroyNotify)socket_destroy2);
  transfer_update(socket, 1);
}

void print_sockets() {
  GList* dlist;
  socket_t* socket;
  int cnt = 0;

  dlist = global.sockets;
  while (dlist) {
    socket = (socket_t*)(dlist->data);
    dlist = dlist->next;
    if (
	(socket->type == S_SHARE) ||
	(socket->type == S_BROWSE) ||
	(socket->type == S_HTTP) ||
	(socket->type == S_SERVER) ||
	(socket->type == S_DATA) ||
	(socket->type == S_TRANSFER) ||
	(socket->type == S_UNKNOWN) ||
	0
	)
      cnt++;
  }
  if (cnt) printf("--%d sockets------------------------\n", cnt);

  dlist = global.sockets;
  while (dlist) {
    socket = (socket_t*)(dlist->data);
    dlist = dlist->next;
    printf("[%8s]", socket_get_name(socket));
    switch (socket->type) {
    case S_SHARE:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_BROWSE:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      printf("[%s]", (char*)(socket->data));
      break;
    case S_HTTP:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_SERVER:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_DATA:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_TRANSFER:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_UNKNOWN:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    default:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    }
    printf("\n");
  }
}

void start_queued_transfers() {
  GtkCList* clist;
  int i1;
  GList* queued;
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;
  
  queued = NULL;
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    socket = (socket_t*)gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t*)(socket->data);
    if (transfer->status == S_QUEUED) {
      queued = g_list_append(queued, socket);
    }
    i1++;
  }
  
  for (dlist = queued; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    transfer = (transfer_t*)(socket->data);
    if (download_allowed(transfer)) download_start(socket, FALSE);
  }

  g_list_free(queued);

  if (!global.options.restart_queued_uploads) return;

  queued = NULL;
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    socket = (socket_t*)gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t*)(socket->data);
    if (transfer->status == S_QUEUED) {
      queued = g_list_append(queued, socket);
    }
    i1++;
  }
  
  for (dlist = queued; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    transfer = (transfer_t*)(socket->data);
    if (upload_allowed(transfer)) upload_start(socket, FALSE);
  }

  g_list_free(queued);

}

int global_timer(gpointer data) {
  char comm[2048];
  GtkWidget* temp;
  GList* dlist;
  GList* dlist2;
  resume_t* resume;
  socket_t* socket;
  transfer_t* transfer;
  static int cnt = 0;
  search_t* search1;
  search_t* search2;
  search_t* tsearch;
  GtkCTreeNode* node;
  int m_search;
  ping_t* ping;
  user_timestamp_t* stamp;
  struct timeval tv;
  time_t tim;

  cnt++;

  if (global.status.exiting == E_SAFE) {
    if (global.limit.cur_uploads + global.limit.cur_downloads == 0)
      global.status.exiting = E_NORMAL;
  } else if (global.status.exiting == E_NORMAL) {
    global_exit();
  }  

  if (global.socket_win) {
    for (dlist = global.sockets; dlist; dlist = dlist->next) {
      socket = (socket_t*)(dlist->data);
      socket_update_clist(socket);
    }
  }

  time(&tim);

  if (global.time_display) print_topright_corner();

  gettimeofday(&tv, 0);

  dlist = global.userstamp;
  while (dlist) {
    stamp = (user_timestamp_t*)(dlist->data);
    dlist2 = dlist->next;
    if (tv.tv_sec - stamp->tv.tv_sec > 100) {
      client_message(_("Message"), _("Ping to <%s> timed out after 100 seconds!"), stamp->user);
      global.userstamp = g_list_remove(global.userstamp, stamp);
      free(stamp->user);
      free(stamp);
    }
    dlist = dlist2;
  }
  
  dlist = global.pings;
  while (dlist) {
    ping = (ping_t*)(dlist->data);
    dlist2 = dlist->next;
    ping->cnt++;
    if (ping->cnt > PING_LIFE) ping_destroy(ping);
    dlist = dlist2;
  }

  dlist = global.sockets;
  while (dlist) {
    socket = (socket_t*)(dlist->data);
    dlist2 = dlist->next;
    switch (socket->type) {
    case S_SHARE:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) socket_destroy(socket, 0);
      break;
    case S_BROWSE:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) {
	//	send_command(CMD_CLIENT_BROWSE, (char*)(socket->data));
	socket_destroy(socket, 0);
      }
      break;
    case S_HTTP:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) socket_destroy(socket, 0);
      break;
    case S_SERVER:
      if (socket->fd >= 0) {
	socket->cnt++;
	if (socket->cnt >= socket->max_cnt) {
	  socket->cnt = 0;
	  napster_disconnect("Server connection timed out");
	}
      }
      break;
    case S_DATA:
      break;
    case S_TRANSFER:
      transfer = (transfer_t*)(socket->data);
      if (!transfer) break;
      if (transfer_in_progress(transfer)) {
	socket->cnt++;
	if (socket->cnt >= socket->max_cnt)
	  socket_end(socket, &(SocketStatus[S_TIMEOUT]));
	if ((cnt%global.network.transfer_delay) == 0)
	  transfer_update(socket, 0);
      }
      break;
    case S_PING:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) {
	socket_destroy(socket, 0);
      }
      break;
    case S_UNKNOWN:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) socket_destroy(socket, 0);
      break;
    default:
      break;
    }
    dlist = dlist2;
  }    

  start_queued_transfers();

  statistic_update(&global.statistic);
  if (notebook_page_visible(7)) statistic_output(&global.statistic);

  // updating stats
  temp = lookup_widget(global.win, "label534");
  sprintf(comm, "%d/%d", global.limit.cur_downloads,
	  global.limit.cur_real_downloads);
  gtk_label_set_text(GTK_LABEL(temp), comm);
  temp = lookup_widget(global.win, "label547");
  sprintf(comm, "%d/%d/%d", global.limit.cur_uploads,
	  global.limit.cur_real_uploads,
	  global.limit.cur_large);
  gtk_label_set_text(GTK_LABEL(temp), comm);

  global.down_width.current = 
    (global.down_width.current*(cnt-1)+global.down_width.bytes)/cnt;
  global.up_width.current = 
    (global.up_width.current*(cnt-1)+global.up_width.bytes)/cnt;
  global.down_width.bytes = 0;
  global.up_width.bytes = 0;

  if (cnt%global.network.transfer_delay == 0) {
    draw_band_width(&global.down_width, 0);
    draw_band_width(&global.up_width, 0);
    cnt = 0;
    global.down_width.current = 0;
    global.up_width.current = 0;
  }

  downloads_enable();
  uploads_enable();
  
  if (global.status.connection < 2) return 1;
  if (!global.napster) return 1;
  if (SERVER->network == N_UNKNOWN) return 1;

  temp = lookup_widget(global.win, "progressbar1");
  if ((global.napster->cnt > 1) && GTK_WIDGET_VISIBLE(temp)) {
    gtk_progress_bar_update(GTK_PROGRESS_BAR(temp), 
			    (double)(global.napster->cnt)/(double)global.napster->max_cnt);
  }

  // test
  if (opennap_version(0,0)) m_search = global.limit.max_searches;
  else m_search = 1;

  if (global.status.searching < m_search) {
    search1 = search2 = NULL;
    for (dlist = global.searches; dlist; dlist = dlist->next) {
      tsearch = (search_t*)(dlist->data);
      if (tsearch->queued) {
	if (tsearch->resume) {
	  if (!search2) search2 = tsearch;
	} else {
	  if (!search1) search1 = tsearch;
	}
      }
    }
    if (search1) search_do(search1);
    else if (search2) search_do(search2);
  }

  if (!global.network.auto_resume) return 1;

  temp = lookup_widget(global.win, "resume_tree");
  node = GTK_CTREE_NODE(GTK_CLIST(temp)->row_list);
  while (node) {
    resume = gtk_ctree_node_get_row_data(GTK_CTREE(temp), node);
    node = GTK_CTREE_ROW(node)->sibling;
    if (!resume->search) {
      resume_search(resume, 1);
    } else if (resume->status != R_FROZEN) {
      if (!resume->transfer)
	resume_try_next_potential(resume);
    }
  }

  return 1;
}

FILE* open_download_file(transfer_t* transfer) {
  int cnt;
  struct stat st;
  resume_t* resume;
  int exist;

  if (!transfer->resume) {
#ifdef TRANSFER_DEBUG
    printf("ops, no resume\n");
#endif
    return NULL;
  }
  resume = transfer->resume;
  
  exist = (stat(resume->filename, &st) >= 0);
  if  (exist) {
#ifdef TRANSFER_DEBUG
    printf("resume file does exist! [%s]\n", resume->filename);
#endif
    transfer->file = fopen(resume->filename, "r");
    if (transfer->file == NULL) {
#ifdef TRANSFER_DEBUG
      printf("could not open incomplete file\n");
#endif
      return NULL;
    }
    
    cnt = st.st_size - transfer->progress;
    fseek(transfer->file, transfer->progress, SEEK_SET);
#ifdef TRANSFER_DEBUG
    printf("set offset to %ld %d\n", transfer->progress, cnt);
#endif
    transfer->resume_check = (unsigned char*)malloc(cnt*sizeof(char));
    transfer->check_length = read(fileno(transfer->file),
				  transfer->resume_check, cnt);
    //    printf("openfile: %d\n", transfer->check_length);
    if (transfer->check_length < 2) {
      printf("Warning: resume check length < 2 !!\n");
    }
    fclose(transfer->file);
    // load done
    
    transfer->file = fopen(resume->filename, "a");
  } else {
#ifdef TRANSFER_DEBUG
    printf("resume file does not exist! [%s]\n", resume->filename);
#endif
    //    resume->size = transfer->size;
    cnt = 0;
    transfer->file = fopen(resume->filename, "w");
    transfer->progress = 0;
  }

  if (transfer->shortname) free(transfer->shortname);
  transfer->shortname =
    strdup(extract_short_name(resume->filename));

  if (transfer->file == NULL) {
    printf("could not open file for downloading [%s]\n",
	   resume->filename);
  }
  return transfer->file;
}

FILE* open_upload_file(transfer_t* transfer) {
  char* filename;
  struct stat st;

  filename = transfer->longname;
  stat(filename, &st);
  transfer->file = fopen(filename, "r");

  if (transfer->file != NULL) transfer->size = st.st_size;
  return transfer->file;
}

void download_end(socket_t* socket, int mode) {
  transfer_t* transfer = (transfer_t*)(socket->data);
  resume_t* resume;
  char* command;
  char* pos;
  int to_delete;
  int to_retry;
  int mime;
  char* folder;
  struct stat st;

#ifdef TRANSFER_DEBUG
  printf("download_end, mode %d\n", mode);
#endif
  
  if (transfer->resume_check) {
    free(transfer->resume_check);
    transfer->resume_check = NULL;
    transfer->check_length = 0;
  }

  switch (mode) {
  case S_FINISHED:
    if (transfer->resume) {
      mime = get_mimetype(transfer->resume->filename);
      folder = global.mimetype[mime].download;
      
      if (transfer->download_dir) free(transfer->download_dir);

#ifdef TRANSFER_DEBUG
      if (folder) printf("Folder: [%s]\n", folder);
      if (transfer->resume->dirname) 
	printf("Transfer->Resume->Dirname: [%s]\n", transfer->resume->dirname);
#endif

      if (transfer->resume->dirname) {
	if (transfer->resume->dirname[0] == '/')
	  transfer->download_dir = strdup(transfer->resume->dirname);
	else if (folder && (*folder)) 
	  transfer->download_dir =
	    g_strdup_printf("%s/%s", folder, transfer->resume->dirname);
	else 
	  transfer->download_dir =
	    g_strdup_printf("./%s", transfer->resume->dirname);
      } else {
	if (folder && (*folder)) 
	  transfer->download_dir = strdup(folder);
	else transfer->download_dir = strdup(".");
      }
	
      create_dir(transfer->download_dir);
      command = g_strdup_printf("%s/%s", transfer->download_dir, 
				extract_filename(transfer->resume->filename));
      if (stat(command, &st) >= 0) {
	pos = rename_file(command);
      } else pos = command;
      if (strcmp(transfer->download_dir, global.incomplete_path)) {
	command = g_strdup_printf("-quiet mv -- '%s' '%s'",
				  transfer->resume->filename, pos);
	//	system(command);
	
#ifdef TRANSFER_DEBUG
	printf("%s\n", command);
#endif
	lopster_exec(command);
	g_free(command);
	free(transfer->longname);
	transfer->longname = strdup(pos);
	free(transfer->shortname);
	transfer->shortname = 
	  strdup(extract_short_name(transfer->longname));
      }
      free(pos);
    } else {
      printf("ops\n");
    }
    break;
  case S_DELETE:
    break;
  case S_QUEUED:
  case S_REMOTE:
  case S_TIMEOUT:
  case S_INCOMPLETE:
  case S_CANCELED:
  default:
    break;
  }

  transfer_status_set(socket, mode);

  resume = transfer->resume;
  // return if transfer is not linked to resume
  // (should only be if resume already deleted and transfer
  // not active)
  if (!resume) return;
  // return if resume is linked to another transfer
  if (resume->transfer && (resume->transfer != transfer)) return;
  
  to_retry = transfer_to_retry(mode);
  if (transfer->is_dcc) to_retry = -1;

  to_delete = transfer_to_delete(transfer, mode);
#ifdef TRANSFER_DEBUG
  printf("%d %d\n", to_retry, to_delete);
#endif

  transfer_ungrab_resume(transfer, mode);

  if ((to_retry == -1) && (mode != S_QUEUED) && to_delete) {
#ifdef TRANSFER_DEBUG
    printf("to delete\n");
#endif
    resume_list_remove(resume, transfer);
  }
}

void upload_end(socket_t* socket, int mode) {
#ifdef TRANSFER_DEBUG
  transfer_t* transfer = (transfer_t*)(socket->data);

  printf("upload_end, mode %d old %d\n", mode, transfer->status);
#endif
  transfer_status_set(socket, mode);
}

void transfer_end(socket_t* socket, int mode) {
  transfer_t* transfer = (transfer_t*)(socket->data);

  if (!transfer) return;
#ifdef TRANSFER_DEBUG
  printf("ending %d [%s]\n", mode, 
	 transfer->winname?transfer->winname:"_unknown_");
#endif

  time(&transfer->end_time);

  if (transfer->file) {
    fclose(transfer->file);
    transfer->file = NULL;
  }

  if (transfer->type == T_DOWNLOAD) {
    download_end(socket, mode);
  } else if (transfer->type == T_UPLOAD) {
    upload_end(socket, mode);
  }
}

gint await_conn_ack(gpointer data, gint source, 
		    GdkInputCondition condition) {
  int res;
  char c;
  socket_t* socket = (socket_t*)data;

  if (condition != GDK_INPUT_READ) {
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  }

  gdk_input_remove(socket->input);

  switch (res = recv(source, &c, 1, 0)) {
  case -1:
#ifdef TRANSFER_DEBUG
    printf("rec1 error\n");
#endif
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  case 0:
#ifdef TRANSFER_DEBUG
    printf("received nothing\n");
#endif
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  default:
    break;
  }    

#ifdef TRANSFER_DEBUG
  printf("got [%c]\n", c);
#endif
  if (c != '1') {
    socket_end(socket, &(SocketStatus[S_REJECT]));
    return 1;
  }
  
  socket->input = 
    gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		  GTK_SIGNAL_FUNC(send_transfer_info), socket);
  
  return 1;
}

gint send_transfer_info(gpointer data, gint source, 
			GdkInputCondition condition) {
  char* temp_str = NULL;
  socket_t* socket;
  transfer_t* transfer;

  socket = (socket_t*)data;
  transfer = (transfer_t*)(socket->data);

  if (condition != GDK_INPUT_WRITE) {
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  }

  gdk_input_remove(socket->input);

  if (transfer->type == T_DOWNLOAD) {
#ifdef TRANSFER_DEBUG
    printf("sending [GET]\n");
#endif
    send(source, "GET", 3, 0);
    temp_str = g_strdup_printf("%s \"%s\" %lu", 
			       SERVER->nick,
			       transfer->winname, 
			       transfer->progress);
  } else if (transfer->type == T_UPLOAD) {
    send(source, "SEND", 4, 0);
#ifdef TRANSFER_DEBUG
    printf("sending [SEND]\n");
#endif
    temp_str = g_strdup_printf("%s \"%s\" %lu", 
			       SERVER->nick,
			       transfer->winname, transfer->size);
    transfer->progress = 0;
  }
  if (temp_str) {
#ifdef TRANSFER_DEBUG
    printf("sending [%s]\n", temp_str);
#endif
    send(source, temp_str, strlen(temp_str), 0);
    g_free(temp_str);
  } else {
    g_warning("Should not happen: transfer.c");
    return 1;
  }
  
  transfer_status_set(socket, S_INFO);
  
  socket->input = 
    gdk_input_add(socket->fd, GDK_INPUT_READ, 
		  GTK_SIGNAL_FUNC(get_transfer_info), socket);
  
  return 1;
}

gint get_transfer_info(gpointer data, gint source, 
		       GdkInputCondition condition) {
  socket_t* socket;
  transfer_t* transfer;
  unsigned char buffer[1024];
  int res;
  int cnt;
  int i1;
  unsigned long size;

  socket = (socket_t*)data;
  transfer = (transfer_t*)(socket->data);

  if (condition != GDK_INPUT_READ) {
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  }

  gdk_input_remove(socket->input);
  
  // 128 bytes should do it, as we are trying to get the filessize.
  switch ((res = recv(source, buffer, 128, MSG_PEEK))) {
  case -1:
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  case 0:
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  default:
    break;
  }
  
  buffer[res] = 0;
#ifdef TRANSFER_DEBUG
  printf("read [%s] [%d]\n", buffer, res);
#endif
  cnt = 0;
  while (isdigit(buffer[cnt])) cnt++;

  if (recv(source, buffer, cnt, 0) != cnt) {
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  }
  buffer[cnt] = 0;

  if (transfer->type == T_UPLOAD) {
    transfer->progress = strtoul(buffer, NULL, 10);
#ifdef TRANSFER_DEBUG
    printf("got progress %ld\n", transfer->progress);
#endif
    for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
      transfer->history[i1] = transfer->progress;
    transfer->start_size = transfer->progress;
  } else {
    size = strtoul(buffer, NULL, 10);
    if (size == 0) {
      socket_end(socket, &(SocketStatus[S_REJECT]));
      return 1;
    } else {
      transfer->size = size;
    }
#ifdef TRANSFER_DEBUG
    printf("got size [%ld] [%d]\n", transfer->size, cnt);
#endif
  }

  time(&(transfer->start_time));
  if (transfer->type == T_UPLOAD) {
    if (!open_upload_file(transfer)) {
      send(socket->fd, "FILE NOT FOUND", 
	   strlen("FILE NOT FOUND"), 0);
      socket_end(socket, &(SocketStatus[S_IO]));
      return 1;
    }
    transfer_status_set(socket, S_UPLOADING);
    socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		    GTK_SIGNAL_FUNC(upload_push_output), socket);
  } else {
    if (!open_download_file(transfer)) {
      socket_end(socket, &(SocketStatus[S_IO]));
      return 1;
    }
    transfer_status_set(socket, S_DOWNLOADING);
    transfer->type = T_DOWNLOAD;
    socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(download_get_input), socket);
  }

  return 1;
}

gint get_upload_info(gpointer data, gint source, 
		     GdkInputCondition condition) {
  char buffer[1025];
  int cnt;
  socket_t* socket = (socket_t*)data;
  socket_t* socket2;
  char* user;
  char* filename;
  char* progress;
  file_t* file;
  transfer_t* transfer;
  struct stat st;
  char* temp_str;
  int i1;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return 1;
  }

  gdk_input_remove(socket->input);
  cnt = 0;
  switch (cnt = recv(source, buffer, 1024, 0)) {
  case -1:
    socket_destroy(socket, 0);
    return 1;
  case 0:
    socket_destroy(socket, 0);
    return 1;
  default:
    break;
  }
  
  buffer[cnt] = 0;
#ifdef TRANSFER_DEBUG
  printf("upload_info [%s]\n", buffer);
#endif
  
  user = arg(buffer, 0);
  filename = arg(NULL, 0);
  progress = arg(NULL, 0);

  if (!user || !filename || !progress) {
    socket_destroy(socket, 0);
    return 1;
  }

  if (user && is_string_in_list(global.enemyuser, user)) {
    socket_destroy(socket, 0);
    return 1;
  }

#ifdef TRANSFER_DEBUG
  printf("up [%s][%s][%s]\n", user, filename, progress);
#endif
  
  socket2 = transfer_is_in_upload(user, filename);
  if (socket2) transfer = (transfer_t*)(socket2->data);

  if (socket2) {
#ifdef TRANSFER_DEBUG
    printf("transfer: type [%d]\n", transfer->type);
#endif
    socket2->fd = socket->fd;
    socket->fd = -1;
    socket2->input = socket->input;
    socket->input = -1;
    socket2->timeout = socket->timeout;
    socket->timeout = -1;
    socket2->ip_long = socket->ip_long;
    socket2->port = socket->port;

    socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
    socket = socket2;
    socket->cnt = 0;
    transfer_status_set(socket, S_INFO);
    transfer->progress = strtoul(progress, NULL, 10);
    for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
      transfer->history[i1] = transfer->progress;
    transfer->start_size = transfer->progress;
  } else {
    log("uploads", LOG_OTHER,
	"alert (incoming) %s %s\n", user, filename);
    g_warning("transfer [%s] not found, already deleted?", filename);
    socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
    return 1;
  }
  
  file = lib_search_transfer(transfer);
  if (stat(transfer->longname, &st) < 0) {
#ifdef TRANSFER_DEBUG
    printf("**sending FILE NOT FOUND\n");
#endif
    //    send(transfer->sock->fd, "0", 1,0);
    send(socket->fd, "FILE NOT FOUND", 
	 strlen("FILE NOT FOUND"), 0);
    socket_end(socket, &(SocketStatus[S_IO]));
    return 1;
  } else if (!file && !transfer->is_dcc) {
#ifdef TRANSFER_DEBUG
    printf("**sending FILE NOT SHARED\n");
#endif
    send(socket->fd, "FILE NOT SHARED", 
	 strlen("FILE NOT SHARED"), 0);
    socket_end(socket, &(SocketStatus[S_UNAVAILABLE]));
    return 1;
  }
  transfer->size = st.st_size;
  if (file) transfer->mime_type = file->mime_type;

  temp_str = g_strdup_printf("%lu", transfer->size);
#ifdef TRANSFER_DEBUG
  printf("sending size %s\n", temp_str);
#endif
  send(socket->fd, temp_str, strlen(temp_str), 0);
  g_free(temp_str);
  
  time(&(transfer->start_time));
  if (!open_upload_file(transfer)) {
    send(socket->fd, "FILE NOT FOUND", 
	 strlen("FILE NOT FOUND"), 0);
    socket_end(socket, &(SocketStatus[S_IO]));
    return 1;
  }
  transfer_status_set(socket, S_UPLOADING);
  socket->input =
    gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		  GTK_SIGNAL_FUNC(upload_push_output), socket);
  return 1;
}

gint get_download_info(gpointer data, gint source, 
		       GdkInputCondition condition) {
  char buffer[1025];
  int cnt;
  socket_t* socket = (socket_t*)data;
  socket_t* socket2;
  char* user;
  char* filename;
  char* size;
  transfer_t* transfer;
  char* temp_str;
  unsigned long lsize;

  gdk_input_remove(socket->input);

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return 1;
  }

  cnt = 0;
  switch (cnt = recv(source, buffer, 1024, 0)) {
  case -1:
    socket_destroy(socket, 0);
    return 1;
  case 0:
    socket_destroy(socket, 0);
    return 1;
  default:
    break;
  }
  
  buffer[cnt] = 0;
#ifdef TRANSFER_DEBUG
  printf("download_info [%s]\n", buffer);
#endif
  
  user = arg(buffer, 0);
  filename = arg(NULL, 0);
  size = arg(NULL, 0);
  if (!user || !filename || !size) {
    socket_destroy(socket, 0);
    return 1;
  }
#ifdef TRANSFER_DEBUG
  printf("down [%s][%s][%s]\n", user, filename, size);
#endif
  socket2 = transfer_is_in_download(user, filename);
  if (socket2) {
    transfer = (transfer_t*)(socket2->data);
    if (transfer->resume && transfer->resume->transfer &&
	(transfer->resume->transfer != transfer)) {
      socket_end(socket, &(SocketStatus[S_QUEUED]));
      return 1;
    }
  }
  
  if (socket2) {
    transfer = (transfer_t*)(socket2->data);
    socket2->fd = socket->fd;
    socket->fd = -1;
    socket2->input = socket->input;
    socket->input = -1;
    socket2->timeout = socket->timeout;
    socket->timeout = -1;
    socket2->ip_long = socket->ip_long;
    socket2->port = socket->port;

    socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
    socket = socket2;
    socket->cnt = 0;
    transfer_status_set(socket, S_INFO);
    lsize = strtoul(size, NULL, 10);
    if (lsize == 0) {
      socket_end(socket, &(SocketStatus[S_UNAVAILABLE]));
      return 1;
    } else {
      transfer->size = lsize;
    }
  } else {    
    log("downloads", LOG_OTHER,
	"alert (incoming) %s %s\n", user, filename);
    //    g_warning("transfer [%s] not found, already deleted?", filename);
    socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
    return 1;
  }
  
  temp_str = g_strdup_printf("%lu", transfer->progress);
#ifdef TRANSFER_DEBUG
  printf("sending offset %s\n", temp_str);
#endif
  send(socket->fd, temp_str, strlen(temp_str), 0);
  g_free(temp_str);
      
  time(&(transfer->start_time));
  if (!open_download_file(transfer)) {
    socket_end(socket, &(SocketStatus[S_IO]));
    return 1;
  }

  transfer_status_set(socket, S_DOWNLOADING);
  socket->input =
    gdk_input_add(socket->fd, GDK_INPUT_READ, 
		  GTK_SIGNAL_FUNC(download_get_input), socket);
  return 1;
}

#define MAX_PUSH  1024

gint upload_push_output(gpointer data, gint source, 
			GdkInputCondition condition) {
  socket_t* socket = (socket_t*)data;
  transfer_t* transfer = (transfer_t*)(socket->data);
  char buf[MAX_PUSH + 1];
  int n, sent;
  long maxpush;

  if (condition != GDK_INPUT_WRITE) {
    socket_end(socket, &(SocketStatus[S_INCOMPLETE]));
    return 1;
  }
  if (global.up_width.bytes >= global.up_width.limit) {
    if (!global.uplimit_onlyif_down || global.limit.cur_real_downloads) {
      uploads_disable();
      return 1;
    } else {
      maxpush = MAX_PUSH;
    }
  } else {
    maxpush = global.up_width.limit-global.up_width.bytes;
    if (maxpush > MAX_PUSH) maxpush = MAX_PUSH;
  }

  if (transfer->progress < transfer->size) {
    lseek(fileno(transfer->file), transfer->progress, SEEK_SET);
    n = read(fileno(transfer->file), buf, maxpush);
    if (n <= 0) {
#ifdef TRANSFER_DEBUG
      printf("read <= 0\n");
#endif
      socket_end(socket, &(SocketStatus[S_IO]));
      return 1;
    }
    sent = send(source, buf, n, 0);
    if (sent <= 0) {
#ifdef TRANSFER_DEBUG
      printf("sent error\n");
#endif
      socket_end(socket, &(SocketStatus[S_INCOMPLETE]));
    } else {
      socket->cnt = 0;
      transfer->progress += sent;
      global.up_width.bytes += sent;
      global.statistic.total[1] += (double)sent;
    }
  } else {
    socket_end(socket, &(SocketStatus[S_FINISHED]));
  }

  return 1;
}

#define MAX_GET   2048

gint download_get_input(gpointer data, gint source, GdkInputCondition condition) {
  char buf[MAX_GET+1];
  int n, n2;
  socket_t* socket = (socket_t*)data;
  transfer_t* transfer = (transfer_t*)(socket->data);
  char* cmp_buf;
#ifdef TRANSFER_DEBUG
  int cnt;
#endif
  long maxget;

  if (condition != GDK_INPUT_READ) {
    socket_end(socket, &(SocketStatus[S_INCOMPLETE]));
    return 1;
  }

  if (global.down_width.bytes >= global.down_width.limit) {
    downloads_disable();
    return 1;
  }
  maxget = global.down_width.limit-global.down_width.bytes;
  if (maxget > MAX_GET) maxget = MAX_GET;

  //  if (global.down_width.current > global.down_width.limit) return 1;
  bzero(buf, sizeof(buf));
  n = recv(source, buf, maxget, 0);
  if (n <= 0) {
    if (transfer->progress >= transfer->size) {
      socket_end(socket, &(SocketStatus[S_FINISHED]));
    } else {
      socket_end(socket, &(SocketStatus[S_INCOMPLETE]));
    }
    return 0;
  } else if (n > 0) {
    global.statistic.total[0] += (double)n;
    global.down_width.bytes += n;
    socket->cnt = 0;
    // resume check
    if (transfer->check_length > 0) {
      if (transfer->check_length+1 > n) {
	// assuming that this is FN behavier with one bytes
	// sent before file data, just ignoring it.
#ifdef TRANSFER_DEBUG
	printf("did not read enough bytes [%p][%d][%d]\n", 
	       transfer, n, transfer->check_length);
#endif
	if (!strncasecmp(buf, "FILE NOT", 8)) {
	  buf[n] = 0;
#ifdef TRANSFER_DEBUG
	  printf("got [%s]\n", buf);
#endif
	  socket_end(socket, &(SocketStatus[S_UNAVAILABLE]));
	}
	return 0;
      }
#ifdef TRANSFER_DEBUG
      printf("checking %d bytes\n", transfer->check_length);
#endif
      if (buf[0] == 'A') {cmp_buf = buf+1; n--;}
      else cmp_buf = buf;
      
      if (memcmp(cmp_buf, transfer->resume_check, 
		 transfer->check_length)) {
#ifdef TRANSFER_DEBUG
	printf("resume check failed, ending download\n");
	for (cnt = 0; cnt < transfer->check_length; cnt++)
	  printf("%4d %4d\n", buf[cnt], transfer->resume_check[cnt]);
#endif
	socket_end(socket, &(SocketStatus[S_RESUME_ERR]));
	return 0;
      }
      if ((n2 = fwrite(cmp_buf+transfer->check_length,
		       n-transfer->check_length,
		       sizeof(char), transfer->file)) <= 0) {
	socket_end(socket, &(SocketStatus[S_IO]));
	return 0;
      }

      transfer->check_length = 0;
      free(transfer->resume_check);
      transfer->resume_check = NULL;
    } else {
      if ((transfer->progress == 0) &&
	  (buf[0] == 'A')) {
	cmp_buf = buf+1; n--;
      } else cmp_buf = buf;
      if ((n2 = fwrite(cmp_buf, n, sizeof(char), transfer->file)) <= 0) {
	socket_end(socket, &(SocketStatus[S_IO]));
	return 0;
      }
    }
    transfer->progress += n;
    if (transfer->progress >= transfer->size) {
      socket_end(socket, &(SocketStatus[S_FINISHED]));
    }
  }
  return 0;
}

void transfer_destroy(transfer_t* transfer) {
  if (!transfer) return;
#ifdef TRANSFER_DEBUG
  printf("detroying transfer %p\n", transfer);
#endif
  if (transfer->longname) free(transfer->longname);
  if (transfer->shortname) free(transfer->shortname);
  if (transfer->winname) free(transfer->winname);
  if (transfer->download_dir) free(transfer->download_dir);
  if (transfer->md5) free(transfer->md5);
  if (transfer->user) free(transfer->user);
  free(transfer);
}

/*
int transfer_timeout_cb(gpointer data) {
  transfer_t* transfer;

  transfer = (transfer_t*)data;
  if (transfer->sock->cnt >= transfer->sock->max_cnt) {
    socket_end(transfer, S_TIMEOUT);
    return 1;
  }
  transfer->sock->cnt++;
  transfer_update(transfer);

  return 1;
}
*/

int transfer_remote_timeout(gpointer data) {
  socket_t* socket;
  transfer_t* transfer;

  socket = (socket_t*)data;
  transfer = (transfer_t*)(socket->data);

  if (download_allowed(transfer)) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
    download_start(socket, FALSE);
  } else if (transfer->status != S_QUEUED) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
    socket_end(socket, &(SocketStatus[S_QUEUED]));
  }

  return 1;
}

user_info_t* search_user_info(char* user, int down) {
  GList* source;
  GList* dlist;
  user_info_t* user_info;

  if (down) source = global.download_userinfo;
  else source = global.upload_userinfo;

  for (dlist = source; dlist; dlist = dlist->next) {
    user_info = (user_info_t*)(dlist->data);
    if (!strcmp(user_info->user, user)) return user_info;
  }

  return NULL;
}

user_info_t* detect_user_info(char* user, int in_download) {
  user_info_t* user_info;

  user_info = search_user_info(user, in_download);
  if (!user_info) {
    user_info = (user_info_t*)malloc(sizeof(user_info_t));
    user_info->user = strdup(user);
    user_info->cur = 0;
    user_info->max = -1;
#ifdef WINMX_QUEUE_BEHAVIER
    user_info->client = C_UNKNOWN;
#endif
    if (in_download)
      global.download_userinfo = g_list_append(global.download_userinfo, user_info);
    else global.upload_userinfo = g_list_append(global.upload_userinfo, user_info);
  }
  
  return user_info;
}

user_info_t* change_user_info(char* user, int max, int down) {
  user_info_t* user_info;
  GtkWidget* temp;
  char* text;
  
  user_info = search_user_info(user, down);
  if (!user_info) return NULL;

  if (max < -2) {
    user_info->max = user_info->cur-1;
    if (user_info->max <= 0) user_info->max = 1;
  } else user_info->max = max;
  if (down) temp = lookup_widget(global.win, "label422");
  else temp = lookup_widget(global.win, "label731");
  text = GTK_LABEL(temp)->label;
  if (!strcmp(user_info->user, text)) {
    if (down) temp = lookup_widget(global.win, "spinbutton15");
    else temp = lookup_widget(global.win, "spinbutton49");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp),
			      (gfloat)(user_info->max));
  }
  return user_info;
}

void downloads_disable() {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer) continue;
    if (transfer->type != T_DOWNLOAD) continue;
    if (transfer->status != S_DOWNLOADING) continue;
    if (socket->input >= 0) {
      gdk_input_remove(socket->input);
      socket->input = -1;
    }
  }
}

void downloads_enable() {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer) continue;
    if (transfer->type != T_DOWNLOAD) continue;
    if (transfer->status != S_DOWNLOADING) continue;
    if (socket->input == -1) {
      socket->input =
	gdk_input_add(socket->fd, GDK_INPUT_READ, 
		      GTK_SIGNAL_FUNC(download_get_input), socket);
    }
  }
}

void uploads_disable() {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer || (transfer->type != T_UPLOAD) ||
	(transfer->status != S_UPLOADING)) continue;
    if (socket->input >= 0) {
      gdk_input_remove(socket->input);
      socket->input = -1;
    }
  }
}

void uploads_enable() {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer || (transfer->type != T_UPLOAD) ||
	(transfer->status != S_UPLOADING)) continue;
    if (socket->input == -1) {
      socket->input =
	gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		      GTK_SIGNAL_FUNC(upload_push_output), socket);
    }
  }
}

void transfer_clist_remove(socket_t* socket) {
  GtkCList* clist;
  socket_t* s;
  int i1;

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    s = (socket_t*)gtk_clist_get_row_data(clist, i1);
    if (s == socket) {
      gtk_clist_remove(clist, i1);
      return;
    }
    i1++;
  }
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    s = (socket_t*)gtk_clist_get_row_data(clist, i1);
    if (s == socket) {
      gtk_clist_remove(clist, i1);
      return;
    }
    i1++;
  }

  return;
}

int transfer_to_destroy(socket_t* socket, int mode) {
  transfer_t* transfer;

  if (socket->timeout != -1) return 0;

  transfer = (transfer_t*)(socket->data);
  if (transfer->type == T_DOWNLOAD) {
    switch (mode) {
    case S_CANCELED:
      if (global.options.dl_autoremove & REMOVE_D_CANCELED)
	return 1;
      break;
    case S_FINISHED:
      if (global.options.dl_autoremove & REMOVE_D_FINISHED)
	return 1;
      break;
    case S_TIMEOUT:
      if (global.options.dl_autoremove & REMOVE_D_TIMEOUT)
	return 1;
      break;
    case S_REJECT:
      if (global.options.dl_autoremove & REMOVE_D_REJECT)
	return 1;
      break;
    case S_INCOMPLETE:
      if (global.options.dl_autoremove & REMOVE_D_INCOMPLETE)
	return 1;
      break;
    case S_FIREWALL:
      if (global.options.dl_autoremove & REMOVE_D_FIREWALL)
	return 1;
      break;
    case S_CONERROR:
      if (global.options.dl_autoremove & REMOVE_D_CONERROR)
	return 1;
      break;
    case S_REMOTE:
      if (global.options.dl_autoremove & REMOVE_D_REMOTE)
	return 1;
      break;
    case S_UNAVAILABLE:
      if (global.options.dl_autoremove & REMOVE_D_UNAVAILABLE)
	return 1;
      break;
    case S_RESUME_ERR:
      if (global.options.dl_autoremove & REMOVE_D_RESUME_ERR)
	return 1;
      break;
    case S_IO:
      if (global.options.dl_autoremove & REMOVE_D_IO)
	return 1;
      break;
    case S_DELETE:
      return 1;
      break;
    }
    return 0;
  } else {
    switch (mode) {
    case S_FINISHED:
      if (global.options.ul_autoremove & REMOVE_U_FINISHED)
	return 1;
      break;
    case S_REJECT:
    case S_REMOTE:
    case S_TIMEOUT:
    case S_INCOMPLETE:
    case S_FIREWALL:
    case S_CONERROR:
    case S_UNAVAILABLE:
    case S_RESUME_ERR:
    case S_IO:
    case S_CANCELED:
      if (global.options.ul_autoremove & REMOVE_U_ABORTED)
	return 1;
      break;
    case S_QUEUED:
      if (global.options.restart_queued_uploads) return 0;
      if (global.options.ul_autoremove & REMOVE_U_QUEUED)
	return 1;
      break;
    case S_DELETE:
      return 1;
      break;
    }
    return 0;
  }
}

int transfer_to_retry(int mode) {
  switch (mode) {
  case S_TIMEOUT:
    if (global.options.auto_retry & RETRY_TIMEOUT) 
      return global.retry_timer[0]*1000;
    break;
  case S_REJECT:
    if (global.options.auto_retry & RETRY_REJECT) 
      return global.retry_timer[1]*1000;
    break;
  case S_INCOMPLETE:
    if (global.options.auto_retry & RETRY_INCOMPLETE) 
      return global.retry_timer[2]*1000;
    break;
  case S_CONERROR:
    if (global.options.auto_retry & RETRY_CONERROR) 
      return global.retry_timer[3]*1000;
    break;
  case S_REMOTE:
    if (global.options.auto_retry & RETRY_REMOTE) 
      return global.retry_timer[4]*1000;
    break;
  case S_UNAVAILABLE:
    if (global.options.auto_retry & RETRY_UNAVAILABLE) 
      return global.retry_timer[5]*1000;
    break;
  case S_QUEUED:
    //    return 1000;
    return -1;
    break;
  }

  return -1;
}

int transfer_to_delete(transfer_t* transfer, int mode) {
  if (mode == S_DELETE) return 1;
  if (mode == S_FINISHED) return 1;
  if (transfer->is_dcc) return 1;

  if ((transfer->progress > 0) &&
      global.options.auto_delete & DELETE_INCOMPLETE)
    return 1;
  if ((transfer->progress == 0) &&
      global.options.auto_delete & DELETE_UNSTARTED)
    return 1;

  return 0;
}

void on_toggle_dl_remove(GtkMenuItem     *menuitem,
			 gpointer         user_data) {
  int no = (int)user_data;

  global.options.dl_autoremove = 
    global.options.dl_autoremove ^ (1<<no);
}

GtkWidget* create_dl_remove_popup (int val) {
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (mode_popup), "mode_popup", mode_popup);
  mode_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (mode_popup));

  mode = gtk_check_menu_item_new_with_label (_("Canceled"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<0))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)0);

  mode = gtk_check_menu_item_new_with_label (_("Finished"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<1))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)1);

  mode = gtk_check_menu_item_new_with_label (_("Timed out"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<2))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)2);

  mode = gtk_check_menu_item_new_with_label (_("Rejected"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<3))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)3);

  mode = gtk_check_menu_item_new_with_label (_("Incomplete"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<4))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)4);

  mode = gtk_check_menu_item_new_with_label (_("Firewalled"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<5))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)5);

  mode = gtk_check_menu_item_new_with_label (_("Connection error"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<6))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)6);

  mode = gtk_check_menu_item_new_with_label (_("Remotely queued"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<7))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)7);

  mode = gtk_check_menu_item_new_with_label (_("Unavailable"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<8))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)8);

  mode = gtk_check_menu_item_new_with_label (_("Resume Error"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<9))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)9);

  mode = gtk_check_menu_item_new_with_label (_("IO error"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<10))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)10);

  return mode_popup;
}

void on_toggle_autoretry(GtkMenuItem     *menuitem,
			 gpointer         user_data) {
  int no = (int)user_data;

  global.options.auto_retry = 
    global.options.auto_retry ^ (1<<no);
}

void on_retry_configure(GtkMenuItem     *menuitem,
			gpointer         user_data) {
  GtkWidget* win;
  GtkWidget* widget;

  win = create_retry_win();
  widget = lookup_widget(win, "spinbutton43");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), (gfloat)(global.retry_timer[0]));
  widget = lookup_widget(win, "spinbutton44");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), (gfloat)(global.retry_timer[1]));
  widget = lookup_widget(win, "spinbutton45");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), (gfloat)(global.retry_timer[2]));
  widget = lookup_widget(win, "spinbutton46");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), (gfloat)(global.retry_timer[3]));
  widget = lookup_widget(win, "spinbutton47");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), (gfloat)(global.retry_timer[4]));
  widget = lookup_widget(win, "spinbutton58");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), (gfloat)(global.retry_timer[5]));

  gtk_widget_show(win);
}

GtkWidget* create_retry_popup (int val) {
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (mode_popup), "mode_popup", mode_popup);
  mode_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (mode_popup));

  mode = gtk_check_menu_item_new_with_label (_("Timed out"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<0))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)0);

  mode = gtk_check_menu_item_new_with_label (_("Rejected"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<1))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)1);

  mode = gtk_check_menu_item_new_with_label (_("Incomplete"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<2))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)2);

  mode = gtk_check_menu_item_new_with_label (_("Connection error"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<3))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)3);

  mode = gtk_check_menu_item_new_with_label (_("Remotely queued"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<4))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)4);

  mode = gtk_check_menu_item_new_with_label (_("Unavailable"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<5))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)5);

  mode = gtk_menu_item_new ();
  gtk_widget_ref (mode);
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  gtk_widget_set_sensitive (mode, FALSE);
  
  mode = gtk_menu_item_new_with_label (_("Configure"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_retry_configure,
		      NULL);

  return mode_popup;
}

void on_toggle_ul_remove(GtkMenuItem     *menuitem,
			 gpointer         user_data) {
  int no = (int)user_data;

  global.options.ul_autoremove = 
    global.options.ul_autoremove ^ (1<<no);
}

GtkWidget* create_ul_remove_popup (int val) {
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (mode_popup), "mode_popup", mode_popup);
  mode_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (mode_popup));

  mode = gtk_check_menu_item_new_with_label (_("Finished"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<0))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_ul_remove,
		      (gpointer)0);

  mode = gtk_check_menu_item_new_with_label (_("Aborted"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<1))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_ul_remove,
		      (gpointer)1);

  mode = gtk_check_menu_item_new_with_label (_("Queued"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<2))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_ul_remove,
		      (gpointer)2);

  return mode_popup;
}

void on_toggle_autodelete(GtkMenuItem     *menuitem,
			  gpointer         user_data) {
  int no = (int)user_data;

  global.options.auto_delete = 
    global.options.auto_delete ^ (1<<no);
}

GtkWidget* create_autodelete_popup (int val) {
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (mode_popup), "mode_popup", mode_popup);
  mode_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (mode_popup));

  mode = gtk_check_menu_item_new_with_label (_("Unstarted"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<0))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autodelete,
		      (gpointer)0);

  mode = gtk_check_menu_item_new_with_label (_("Incomplete"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<1))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autodelete,
		      (gpointer)1);

  return mode_popup;
}

int transfer_grab_resume(transfer_t* transfer, resume_t* resume) {
  struct stat st;
  int i1;

#ifdef TRANSFER_DEBUG
  printf("t:%p grabbing r:%p\n", transfer, resume);
#endif

  if (resume->status == R_FROZEN) return 0;

  if (resume->transfer && (transfer != resume->transfer)) {
#ifdef TRANSFER_DEBUG
    printf("already grabbed\n");
#endif
    return 0;
  }

  resume->transfer = transfer;
  transfer->resume = resume;
  resume_user_flag(resume, transfer->user, S_WAITING);
  
  if (transfer->download_dir) free(transfer->download_dir);
  if (resume->dirname) transfer->download_dir = strdup(resume->dirname);
  else transfer->download_dir = NULL;

  if (stat(resume->filename, &st) < 0) {
    transfer->progress = 0;
    transfer->start_size = 0;
  } else {
    transfer->progress = st.st_size-100;
    if (transfer->progress < 0) transfer->progress = 0;
    transfer->start_size = transfer->progress;
    for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
      transfer->history[i1] = transfer->progress;
  }
  resume->status = R_DOWNLOAD;
  resume_list_update(resume, 0);

  return 1;
}

resume_t* transfer_create_resume(transfer_t* transfer) {
  struct stat st;
  char* text;
  resume_t* resume;
  char* filename;

  text = extract_filename(transfer->longname);
  
  /* this should be unnecessary */
  /*
  if (resume_list_search(transfer)) {
    g_warning(_("File already exists already as an incomplete file"));
    return NULL;
  }
  */

  filename = g_strdup_printf("%s/%s", global.incomplete_path, text);
  convert_local_name(filename);
  if (stat(filename, &st) >= 0) {
    // file exists, checking whether there is an incomplete entry for this.
    if (resume_list_search_filename(filename)) {
      filename = rename_file(filename);
    }
    if (!filename) return NULL;
  }
  resume = (resume_t*)malloc(sizeof(resume_t));
  resume->transfer = NULL;
  resume->filename = filename;
  //  printf("download %s\n", resume->filename);
  resume->size = transfer->size;
  if (transfer->download_dir)
    resume->dirname = strdup(transfer->download_dir);
  else resume->dirname = NULL;
  resume->search = NULL;
  resume->status = R_NONE;
  resume->user = strdup(transfer->user);
  resume->org_file = strdup(transfer->winname);
  resume->online = 1;
  resume->search_string = NULL;
  return resume;
}

int transfer_search_queued(resume_t* resume) {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;
  int result = 0;

  if (!resume->transfer) return 0;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (!socket) {
      printf("socket->data = NULL\n");
      continue;
    }
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer) continue;
    if (transfer->type != T_DOWNLOAD) continue;
    if ((transfer->resume == resume) && (transfer->status == S_QUEUED)) result++;
  }
  return result;
}

void transfer_ungrab_resume(transfer_t* transfer, int mode) {
  resume_t* resume;

  if (!transfer->resume) {
    g_warning("nothing grabbed\n");
    return;
  }
  
#ifdef TRANSFER_DEBUG
  printf("t:%p ungrabbing r:%p %d\n", transfer, transfer->resume, mode);
#endif

  resume = transfer->resume;
  //  resume_user_flag(resume, transfer->user, mode);

  if (transfer_search_queued(resume) > 0)
    transfer->resume->status = R_QUEUED;
  else transfer->resume->status = R_NONE;

  resume_list_update(resume, 0);

  resume->transfer = NULL;
}

void transfer_cancel_running() {
  GList* dlist;
  socket_t *socket;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (socket->type != S_TRANSFER) continue;
    socket_end(socket, &(SocketStatus[S_CANCELED]));
  }
}

void
on_ok2_clicked                   (GtkButton       *button,
				  gpointer         user_data) {
  GtkWidget* temp;
  char* filename;
  char* nick;

  temp = GTK_WIDGET(user_data);
  if (!temp) return;

  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(temp));
  nick = (char*)gtk_object_get_data(GTK_OBJECT(temp), "nick");
  if (filename && *filename && nick) dcc_send_file(nick, filename);
  if (nick) free(nick);
  gtk_widget_destroy(temp);
}

void
on_cancel2_clicked                   (GtkButton       *button,
				  gpointer         user_data) {
  GtkWidget* temp;
  char* nick;

  temp = GTK_WIDGET(user_data);
  if (!temp) return;

  nick = (char*)gtk_object_get_data(GTK_OBJECT(temp), "nick");
  if (nick) free(nick);
  gtk_widget_destroy(temp);
}

#define CTCP_DELIM_CHAR '\001'

void dcc_send_file(char* nick, char* file) {
  char* command;
  struct stat st;
  transfer_t* new_trans;
  struct sockaddr_in localaddr;
  int len = sizeof(struct sockaddr_in);
  socket_t* socket;

  if (!nick || !file) return;
  if (stat(file, &st) < 0) {
    client_message(_("DCC"), _("File does not exist!"));
    return;
  }
  if (global.status.connection < 2) {
    client_message(_("DCC"), _("Not connected!"));
    return;
  }
  
  new_trans = transfer_new();
  new_trans->longname = strdup(file);
  new_trans->winname = strdup(file);
  convert_to_win(new_trans->winname);
  new_trans->shortname =
    strdup(extract_short_name(new_trans->longname));
  new_trans->user = strdup(nick);
  new_trans->type = T_UPLOAD;
  new_trans->is_dcc = TRUE;
  new_trans->mime_type = get_mimetype(file);
  
  send_command(CMD_CLIENT_USERSPEED, new_trans->user);
  new_trans->size = st.st_size;
  socket = socket_new(S_TRANSFER);
  socket->data = new_trans;
  
  transfer_insert(socket);
  upload_start(socket, 1);    // 1 is needless here, cause dcc is always started
  
  getsockname(global.napster->fd,
	      (struct sockaddr *)&localaddr, &len);
  command = 
    g_strdup_printf("%s %cSEND %s %lu %d \"%s\" %lu %s %d%c", 
		    nick, CTCP_DELIM_CHAR, SERVER->nick, 
		    (unsigned long)BSWAP32(localaddr.sin_addr.s_addr), 
		    global.network.port, new_trans->winname, 
		    (unsigned long)st.st_size, "checksum", 0, CTCP_DELIM_CHAR);
  send_command(CMD_CLIENT_PRIVMSG, command);
}

void dcc_select_file(char* nick) {
  GtkWidget* filesel;
  GtkWidget* temp;
  char* str;

  if (!nick) return;

  str = g_strdup_printf(_("Select file to send [%s]"), nick);
  filesel = gtk_file_selection_new(str);
  g_free(str);
  gtk_widget_show(filesel);
  gtk_object_set_data(GTK_OBJECT(filesel), "nick", strdup(nick));
  
  temp = GTK_FILE_SELECTION(filesel)->ok_button;
  if (temp)
    gtk_signal_connect (GTK_OBJECT (temp), "clicked",
			GTK_SIGNAL_FUNC (on_ok2_clicked), filesel);
  temp = GTK_FILE_SELECTION(filesel)->cancel_button;
  if (temp)
    gtk_signal_connect (GTK_OBJECT (temp), "clicked",
			GTK_SIGNAL_FUNC (on_cancel2_clicked), filesel);
}

int check_dcc_message(char *from, char *message) {
  char* nick;
  unsigned long ip_long;
  int port;
  char* filename;
  unsigned long filesize;
  char* command;
  char* checksum;
  int speed;
  transfer_t* new_trans;
  char* folder;
  socket_t* socket;

  if (!message || strlen(message)<2) return 0;

  if (*message == CTCP_DELIM_CHAR && 
      message[strlen(message)-1] == CTCP_DELIM_CHAR) {
    message++;
    message[strlen(message)-1] = 0;
    
    command = arg(message, 0);
    if (command && strncmp(command, "SEND", 4)) return 0;
    nick = arg(NULL, 0);
    ip_long = strtoul(arg(NULL, 0), NULL, 10);
    port = atoi(arg(NULL, 0));
    filename = arg(NULL, 0);
    filesize = strtoul(arg(NULL, 0), NULL, 10);
    checksum = arg(NULL, 0);
    speed = atoi(arg(NULL, 0));

    if (filesize == 0) {
      client_message("DCC", _("%s is sending a 0 byte %s, ignoring"), 
		     nick, filename);
      return 1;
    } else {
      client_message("DCC", _("%s is sending %s"), nick, filename);
    }
    if (port == 0 && global.network.port == 0) {
      client_message("Message", _("Both systems are firewalled. Unable to comply"));
      return 1;
    }

    new_trans = transfer_new();
    new_trans->winname = strdup(filename);
    new_trans->mime_type = get_mimetype(filename);
    new_trans->longname = strdup(new_trans->winname);
    convert_to_unix(new_trans->longname, 1);
    new_trans->shortname = 
      strdup(extract_short_name(new_trans->longname));

    new_trans->linespeed = speed;
    new_trans->user = strdup(nick);
    new_trans->size = filesize;
    new_trans->type = T_DOWNLOAD;
    
    if ((folder = valid_download_folder(new_trans->mime_type)) == NULL) {
      transfer_destroy(new_trans);
      download_dialog(new_trans->shortname, new_trans->mime_type);
      return 0;
    }
    
    new_trans->download_dir = strdup(folder);
    new_trans->md5 = strdup(checksum);
    new_trans->is_dcc = TRUE;
    socket = socket_new(S_TRANSFER);
    socket->data = new_trans;
    socket->port = htons(port);
    //    printf("port %d %d\n", port, socket->port);
    socket->ip_long = BSWAP32(ip_long);

    transfer_insert(socket);
    
    if (!global.network.allow_dcc) {
      client_message(NULL, _("At the moment you do not allow dcc file transfers"), nick, filename);
      client_message(NULL, _("If you want to accept the transfer, go to the transfer page and continue download"));
      send_private(nick, "DCC transfers are not allowed at the moment, maybe i will start the download manually", 1);
      return 1;
    } else {
      download_start(socket, TRUE);
    }
    
    return 1;
  }
  return 0;
}

