#include <gtk/gtk.h>
#include <sys/stat.h>
#include <dirent.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>

#include "lopster.h"
#include "connection.h"
#include "global.h"
#include "scheme.h"
#include "support.h"
#include "chat.h"

scheme_t* scheme_new(void) {
  scheme_t* scheme;

  scheme = (scheme_t*)malloc(sizeof(scheme_t));
  scheme->name = NULL;
  scheme->colors = NULL;
  scheme->server_prefix = NULL;
  scheme->client_prefix = NULL;

  return scheme;
}

scheme_t* scheme_create_default(void) {
  scheme_t* scheme = NULL;

  scheme = scheme_new();

  scheme->name = strdup("default");

  scheme->colors = g_list_append(scheme->colors, 
				 style_new("text", "#c5b593", "NULL", "-b&h-lucidatypewriter-medium-r-normal-*-*-120-*-*-m-*-iso8859-1"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("user", "#007de7", "NULL", "NULL"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("yourself", "#e5cf00", "NULL", "NULL"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("friend", "#de5a6a", "NULL", "NULL"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("whisper", "#ffee00", "#3f3f3f", "-adobe-courier-medium-r-normal-*-*-140-*-*-m-*-iso8859-1"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("join", "#009f46", "NULL", "NULL"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("part", "#009f46", "NULL", "NULL"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("error", "#ffc4b2", "NULL", "-adobe-helvetica-medium-r-normal-*-*-120-*-*-p-*-iso8859-1"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("message", "#b2ccff", "NULL", "-b&h-lucidatypewriter-medium-r-normal-*-*-120-*-*-m-*-iso8859-1"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("highlight", "#ded305", "NULL", "-b&h-lucidatypewriter-bold-r-normal-*-*-120-*-*-m-*-iso8859-1"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("transfer_running", "#ff0011", "#ffd484", "NULL"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("transfer_waiting", "#002173", "#94bfdd", "NULL"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("transfer_bar1", "#ff0300", "#ffa393", "NULL"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("transfer_bar2", "#1700ff", "#77aee5", "NULL"));
  scheme->colors = g_list_append(scheme->colors, 
				 style_new("library_unshared", "#000000", "#eeeeee", "NULL"));

  scheme->client_prefix = strdup("%K=%c=%C= ");
  scheme->server_prefix = strdup("%K%r%R ");

  scheme_save(scheme);
  return scheme;
}

void scheme_destroy(scheme_t* scheme) {
  GList* dlist;
  style_t* style;

  if (!scheme) return;
  if (scheme->name) free(scheme->name);
  for (dlist = scheme->colors; dlist; dlist = dlist->next) {
    style = (style_t*)(dlist->data);
    style_destroy(style);
  }
  g_list_free(scheme->colors);
  free(scheme->server_prefix);
  free(scheme->client_prefix);

  free(scheme);
}

scheme_t* scheme_load(char* name) {
  char filename[2048];
  char temp[2048];
  char temp2[2048];
  char* pos;
  char* sname;
  char* sfore;
  char* sback;
  char* sfont;
  FILE* file;
  scheme_t* scheme;
  style_t* style;
  
  strcpy(temp2, name);
  sprintf(filename, "%s/schemes/%s", global.lopster_home, temp2);

  ///////////////////////
  if (!scheme_is_valid(temp2)) {
    return NULL;
  }
  
  file = fopen(filename, "r");
  if (file == NULL) {
    client_message(_("Error"), _("Could not load scheme!"));
    return NULL;
  }

  scheme = scheme_new();
  scheme->name = strdup(name);

  fgets(temp, 1024, file);
  while (fgets(temp, 1024, file)) {
    if (temp[0] == '#') continue;
    temp[strlen(temp)-1] = 0;
    if (!strncasecmp(temp, "ServerPrefix", 12)) {
      pos = strchr(temp, '=')+1;
      scheme->server_prefix = strdup(pos);
      continue;
    } else if (!strncasecmp(temp, "ClientPrefix", 12)) {
      pos = strchr(temp, '=')+1;
      scheme->client_prefix = strdup(pos);
      continue;
    }
    while (1) {
      pos = strtok(temp, " \t[]");
      if (pos == NULL) {
	g_warning("Error in scheme file %s\n", filename);
	return 0;
      }
      if (strlen(pos) > 0) {
	sname = strdup(pos);
	break;
      }
    }
    while (1) {
      pos = strtok(NULL, " \t[]");
      if (pos == NULL) {
	g_warning("Error in scheme file %s\n", filename);
	return 0;
      }
      if (strlen(pos) > 0) {
	sfore = strdup(pos);
	break;
      }
    }
    while (1) {
      pos = strtok(NULL, " \t[]");
      if (pos == NULL) {
	g_warning("Error in scheme file %s\n", filename);
	return 0;
      }
      if (strlen(pos) > 0) {
	sback = strdup(pos);
	break;
      }
    }
    while (1) {
      pos = strtok(NULL, " \t[]");
      if (pos == NULL) {
	g_warning("Error in scheme file %s\n", filename);
	return 0;
      }
      if (strlen(pos) > 0) {
	sfont = strdup(pos);
	break;
      }
    }
    style = style_new(sname, sfore, sback, sfont);
    if (style)
      scheme->colors = g_list_append(scheme->colors, style);

    free(sname);
    free(sfore);
    free(sback);
    free(sfont);
  }
  
  fclose(file);

  if (!scheme->client_prefix)
    scheme->client_prefix = strdup("%K=%c=%C= ");
  if (!scheme->server_prefix)
    scheme->server_prefix = strdup("%K%r%R ");

  if (!style_search(scheme, "transfer_running")) {
    scheme->colors = g_list_append(scheme->colors, 
				   style_new("transfer_running", "#ff0011", "#ffd484", "NULL"));
  }
  if (!style_search(scheme, "transfer_waiting")) {
    scheme->colors = g_list_append(scheme->colors, 
				   style_new("transfer_waiting", "#002173", "#94bfdd", "NULL"));
  }
  if (!style_search(scheme, "transfer_bar1")) {
    scheme->colors = g_list_append(scheme->colors, 
				   style_new("transfer_bar1", "#ff0300", "#ffa393", "NULL"));
  }
  if (!style_search(scheme, "transfer_bar2")) {
    scheme->colors = g_list_append(scheme->colors, 
				   style_new("transfer_bar2", "#1700ff", "#77aee5", "NULL"));
  }
  if (!style_search(scheme, "library_unshared")) {
    scheme->colors = g_list_append(scheme->colors, 
				   style_new("library_unshared", "#000000", "#eeeeee", "NULL"));
  }
  return scheme;
}

void scheme_save(scheme_t* scheme) {
  char filename[2048];
  style_t* style;
  FILE* file;
  GList* dlist;

  if (!scheme) return;

  sprintf(filename, "%s/schemes/%s", global.lopster_home, 
	  scheme->name);
  if ((file = fopen(filename, "w")) == NULL) {
    client_message(_("Error"), _("Could not save scheme [%s]"),
		   scheme->name);
    return;
  }
    
  fprintf(file, "LSF1.1\n");
  for (dlist = scheme->colors; dlist; dlist = dlist->next) {
    style = (style_t*)(dlist->data);
    fprintf(file, "%s [%s][%s][%s]\n", style->name,
	    style->n_fore, style->n_back, style->n_font);
  }
  fprintf(file, "ServerPrefix=%s\n", scheme->server_prefix);
  fprintf(file, "ClientPrefix=%s\n", scheme->client_prefix);
  
  fclose(file);
  
  scheme_update_list(scheme->name);
}

void scheme_delete(char* name) {
  char filename[2048];

  sprintf(filename, "%s/schemes/%s", global.lopster_home, name);
  unlink(filename);
  scheme_update_list(NULL);
}

int scheme_is_valid(char* name) {
  struct stat st;
  FILE* file;
  char temp[2048];

  sprintf(temp, "%s/schemes/%s", global.lopster_home, name);

  if (stat(temp, &st) < 0) return 0;
  
  if ((file = fopen(temp, "r")) == NULL) return 0;

  fgets(temp, 1024, file);
  if (strncmp(temp, "LSF1.1", 6)) return 0;

  fclose(file);
  
  return 1;
}

void scheme_update_list(char* current) {
  char dirname[2048];
  GtkWidget* temp;
  GList* list2;
  struct dirent* entry;
  struct stat buf;
  char filename[2048];
  DIR* dir;
  char* cur = NULL;

  if (!global.options_win) return;
  
  // dont know why i have to duplicate the string here
  // original would be messed up till end of function
  if (current) cur = strdup(current);

  sprintf(dirname, "%s/schemes", global.lopster_home);

  list2 = NULL;
  if ((dir = opendir(dirname)) == NULL) {
    g_warning("could not open dir [%s]", dirname);
  } else {
    while ((entry = readdir(dir)) != NULL) {
      //      if (!strncmp(entry->d_name, ".", 1)) continue;
      //      if (!strncmp(entry->d_name, "..", 2)) continue;
      sprintf(filename, "%s/%s", dirname, entry->d_name);
      stat(filename, &buf);
      if (S_ISREG(buf.st_mode)) {
	if (scheme_is_valid(entry->d_name)) {
	  list2 = g_list_insert_sorted(list2, strdup(entry->d_name),
				       (GCompareFunc)strcasecmp);
	}
      }
    }
    closedir(dir);
  }

  temp = lookup_widget(global.options_win, "combo7");
  if (list2) {
    gtk_combo_set_popdown_strings (GTK_COMBO (temp), list2);
    // maybe freeing the data
    g_list_free (list2);
  }

  temp = lookup_widget(global.options_win, "combo_entry15");
  if (cur) {
    gtk_entry_set_text(GTK_ENTRY(temp), cur);
  }

}

void scheme_draw(scheme_t* scheme) {
  GtkWidget* temp;
  GtkWidget* win;
  char* prefix;

  win = global.options_win;
  if (!win) return;

  temp = lookup_widget(win, "text8");
  gtk_text_backward_delete(GTK_TEXT(temp), 
			   gtk_text_get_length(GTK_TEXT(temp)));
  
  text_print_colored(temp, scheme, 
		     "join", "[Testuser1 joined channel]\n");
  text_print_colored(temp, scheme, "user", "<");
  text_print_colored(temp, scheme, "yourself", "Yourself");
  text_print_colored(temp, scheme, "user", "> ");
  text_print_colored(temp, scheme, "text", "Hello!\n");
  text_print_colored(temp, scheme, 
		     "part", "[Testuser2 left channel]\n");
  text_print_colored(temp, scheme, 
		     "user", "[Testuser1 emote message]\n");
  prefix = cparse(scheme->server_prefix);
  text_print_colored(temp, scheme, "error", prefix);
  prefix = cparse("%W[");
  text_print_colored(temp, scheme, "error", prefix);
  text_print_colored(temp, scheme, "error", "Error");
  prefix = cparse("%W] ");
  text_print_colored(temp, scheme, "error", prefix);
  text_print_colored(temp, scheme, "error", "A server message\n");
  text_print_colored(temp, scheme, "whisper",
		    "<from: Testuser2> This is a private message\n");
  text_print_colored(temp, scheme, "user", "<");
  text_print_colored(temp, scheme, "friend", "Friend_User");
  text_print_colored(temp, scheme, "user", "> ");
  text_print_colored(temp, scheme, "text", "This is a message with ");
  text_print_colored(temp, scheme, "highlight", "highlighted text\n");

  prefix = cparse(scheme->client_prefix);
  text_print_colored(temp, scheme, "message", prefix);
  prefix = cparse("%W[");
  text_print_colored(temp, scheme, "message", prefix);
  text_print_colored(temp, scheme, "message", "Message");
  prefix = cparse("%W] ");
  text_print_colored(temp, scheme, "message", prefix);
  text_print_colored(temp, scheme, "message", "A client message\n");
}

void scheme_load_global(char* scheme, int force) {

  if (scheme && (strlen(scheme) > 0)) {
    global.scheme = scheme_load(scheme);
    if (global.scheme) {
      client_message(_("Message"), _("Scheme [%s] loaded!"),
		     global.scheme->name);
      return;
    }
  }

  global.scheme = scheme_load("default");
  if (!global.scheme) {
    global.scheme = scheme_create_default();
    client_message(_("Message"), _("Scheme [%s] created!"), 
		   global.scheme->name);
  } else {
    client_message(_("Message"), _("Scheme [%s] loaded!"),
		   global.scheme->name);
  }
}

style_t* style_new(char* name, char* fore, char* back, char* font) {
  style_t* style;
  GdkColor color;

  style = (style_t*)malloc(sizeof(style_t));
  style->name = strdup(name);
  style->fore = NULL;
  style->back = NULL;
  style->font = NULL;
  style->n_fore = strdup("NULL");
  style->n_back = strdup("NULL");
  style->n_font = strdup("NULL");

  if (fore) {
    if (style->n_fore) free(style->n_fore);
    style->n_fore = strdup(fore);
    if (strcmp(fore, "NULL")) {
      gdk_color_parse(fore, &color);
      style->fore = gdk_color_copy(&color);
    } else {
      style->fore = NULL;
    }
  }
  if (back) {
    if (style->n_back) free(style->n_back);
    style->n_back = strdup(back);
    if (strcmp(back, "NULL")) {
      gdk_color_parse(back, &color);
      style->back = gdk_color_copy(&color);
    } else {
      style->back = NULL;
    }
  }
  if (font) {
    if (style->n_font) free(style->n_font);
    style->n_font = strdup(font);
    if (strcmp(font, "NULL")) {
      style->font = gdk_font_load(font);
    } else {
      style->font = NULL;
    }
  }

  return style;
}

void style_destroy(style_t* style) {
  if (style->name) free(style->name);
  if (style->n_fore) free(style->n_fore);
  if (style->n_back) free(style->n_back);
  if (style->n_font) free(style->n_font);
  
  free(style);
}

void style_update(scheme_t* scheme,
		  char* name, char* fore, char* back, char* font) {
  style_t* style;
  GdkColor color;

  style = style_search(scheme, name);
  if (!style) {
    style = style_new(name, fore, back, font);
    scheme->colors = g_list_append(scheme->colors, style);
  }

  if (fore) {
    if (style->n_fore) free(style->n_fore);
    style->n_fore = strdup(fore);
    if (strcmp(fore, "NULL")) {
      gdk_color_parse(fore, &color);
      style->fore = gdk_color_copy(&color);
    } else {
      style->fore = NULL;
    }
  }
  if (back) {
    if (style->n_back) free(style->n_back);
    style->n_back = strdup(back);
    if (strcmp(back, "NULL")) {
      gdk_color_parse(back, &color);
      style->back = gdk_color_copy(&color);
    } else {
      style->back = NULL;
    }
  }
  if (font) {
    if (style->n_font) free(style->n_font);
    style->n_font = strdup(font);
    if (strcmp(font, "NULL")) {
      style->font = gdk_font_load(font);
    } else {
      style->font = NULL;
    }
  }
}

style_t* style_search(scheme_t* scheme, char* name) {
  GList* dlist;
  style_t* style;

  if (!scheme) return NULL;

  for (dlist = scheme->colors; dlist; dlist = dlist->next) {
    style = (style_t*)(dlist->data);
    if (!strcasecmp(name, style->name)) return style;
  }
  return NULL;
}

style_t* style_get(scheme_t* scheme, char* name) {
  style_t* style;
  static style_t* d_style = NULL;

  style = style_search(scheme, name);
  if (!style) {
    if (d_style == NULL) {
      d_style = (style_t*)malloc(sizeof(style_t));
      d_style->name = strdup("dummy style");
      d_style->font = NULL;
      d_style->fore = NULL;
      d_style->back = NULL;
      d_style->n_fore = strdup("NULL");
      d_style->n_back = strdup("NULL");
      d_style->n_font = strdup("NULL");
    }
    return d_style;
  }
  return style;
}

void text_print_channel(GtkWidget* widget, scheme_t* scheme,
			char* color, char* text) {
  style_t* style;

  style = style_get(scheme, color);

  gtk_text_freeze(GTK_TEXT(widget));
  gtk_text_insert(GTK_TEXT(widget), style->font, 
		  style->fore, style->back, 
		  text, strlen(text));
  gtk_text_thaw(GTK_TEXT(widget));
  
  return;
}

void text_print_channel_color(GtkWidget* widget, scheme_t* scheme,
			      char* color, GdkColor* fore, 
			      GdkColor* back, char *text) {
  style_t* style;
  GdkColor* f;
  GdkColor* b;

  style = style_get(scheme, color);

  if (fore) f = fore;
  else f = style->fore;
  if (back) b = back;
  else b = style->back;
  
  gtk_text_freeze(GTK_TEXT(widget));
  gtk_text_insert(GTK_TEXT(widget), style->font, 
		  f, b, text, strlen(text));
  gtk_text_thaw(GTK_TEXT(widget));

  return;
}

void text_print_colored(GtkWidget* widget, scheme_t* scheme,
			char* base_color, char *text) {
  GdkColor* fgcolor = NULL;
  GdkColor* bgcolor = NULL;
  char* pos2;
  int len;
  char temp_str[2048];
  int index;

  while (1) {
    pos2 = search_highlight_string(text, &len);
    if (!pos2) break;

    strncpy(temp_str, text, pos2-text);
    temp_str[pos2-text] = 0;
    text_print_channel_color(widget, scheme, base_color, 
			     fgcolor, bgcolor, temp_str);

    if ((pos2[0] == 0x03)) {
      if ((pos2[1] == 0) || (pos2[2] == 0) || (pos2[3] == 0)) return;
      if (pos2[1] == 0x0f) {
	fgcolor = NULL;
	bgcolor = NULL;
	len = 2;
      } else if (pos2[1] == ',') {
	index = ColorTable2(pos2[3]-'0');
	if (index < 0) index = -index;
	bgcolor = &global.color_table[index];
	len = 4;
      } else if (pos2[1] != '-') {
	index = ColorTable(pos2[1]-'0', pos2[2]-'0');
	if (index < 0) index = -index;
	fgcolor = &global.color_table[index];
	len = 3;
      }
    } else {
      strncpy(temp_str, pos2, len);
      temp_str[len] = 0;
      text_print_channel(widget, scheme, "highlight", temp_str);
    }
    
    text = pos2+len;
  }
  text_print_channel_color(widget, scheme,
			   base_color, fgcolor, bgcolor, text);

  return;
}

