#include <ctype.h>
#include <stdio.h>
#include <sys/time.h>
#include <time.h>
#include <sys/socket.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "callbacks.h"
#include "connection.h"
#include "handler.h"
#include "global.h"
#include "search.h"
#include "transfer.h"
#include "scheme.h"
#include "resume.h"
#include "support.h"
#include "whois.h"

GtkWidget* create_resume_popup() {
  GtkWidget *popup;
  GtkWidget *user_popup;
  GtkAccelGroup *popup_accels;
  GtkWidget *item;
  GtkWidget *separator5;
  GtkCTree* ctree;
  GtkCTreeNode* node;
  resume_t* resume;
  int flags = 0;
  int status = -1;

  ctree = GTK_CTREE(global.popup_list);
  node = gtk_ctree_node_nth(ctree, global.popup_row);
  
  if (!node) {
    flags = 0;     // no row
  } else {
    if (GTK_CTREE_ROW(node)->parent != NULL) {
      node = GTK_CTREE_NODE(GTK_CTREE_ROW(node)->parent);
      flags = 1;  // file
    } else {
      flags = 2;  // resume
    }
    resume = (resume_t*)gtk_ctree_node_get_row_data(ctree, node);
    status = resume->status;
  }

  if (flags == 0) return NULL;

  popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (popup));

  if (flags == 2) {
    item = gtk_menu_item_new_with_label (_("Open File"));
    gtk_widget_ref (item);
    gtk_object_set_data_full (GTK_OBJECT (popup), "item", item,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_resume_play_activate),
			NULL);
    separator5 = gtk_menu_item_new ();
    gtk_widget_ref (separator5);
    gtk_object_set_data_full (GTK_OBJECT (popup), "separator5", separator5,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (separator5);
    gtk_container_add (GTK_CONTAINER (popup), separator5);
    gtk_widget_set_sensitive (separator5, FALSE);

    item = gtk_menu_item_new_with_label (_("Delete Selected"));
    gtk_widget_ref (item);
    gtk_object_set_data_full (GTK_OBJECT (popup), "item", item,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_resume_delete_activate),
			NULL);
    if (status == R_FROZEN) {
      item = gtk_menu_item_new_with_label (_("Thaw Selected"));
    } else {
      item = gtk_menu_item_new_with_label (_("Freeze Selected"));
    }
    gtk_widget_ref (item);
    gtk_object_set_data_full (GTK_OBJECT (popup), "item", item,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);
    if (status == R_FROZEN) {
      gtk_signal_connect (GTK_OBJECT (item), "activate",
			  GTK_SIGNAL_FUNC (on_resume_thaw_activate),
			  NULL);
    } else {
      gtk_signal_connect (GTK_OBJECT (item), "activate",
			  GTK_SIGNAL_FUNC (on_resume_freeze_activate),
			  NULL);
    }
      
    separator5 = gtk_menu_item_new ();
    gtk_widget_show (separator5);
    gtk_container_add (GTK_CONTAINER (popup), separator5);
    gtk_widget_set_sensitive (separator5, FALSE);

    item = gtk_menu_item_new_with_label (_("Configure Entry"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_resume_configure),
			resume);
    
    if (resume->user && *(resume->user)) {
      separator5 = gtk_menu_item_new ();
      gtk_widget_show (separator5);
      gtk_container_add (GTK_CONTAINER (popup), separator5);
      gtk_widget_set_sensitive (separator5, FALSE);
      
      item = gtk_menu_item_new_with_label (_("User Menu"));
      gtk_widget_show (item);
      gtk_container_add (GTK_CONTAINER (popup), item);
      
      user_popup = create_user_popup(M_RESUME);
      gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), user_popup);
      
    }

    separator5 = gtk_menu_item_new ();
    gtk_widget_show (separator5);
    gtk_container_add (GTK_CONTAINER (popup), separator5);
    gtk_widget_set_sensitive (separator5, FALSE);

    if (resume->search) {
      item = gtk_menu_item_new_with_label (_("Search File"));
    } else {
      item = gtk_menu_item_new_with_label (_("Search Again"));
    }
    
    gtk_widget_ref (item);
    gtk_object_set_data_full (GTK_OBJECT (popup), "item", item,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_resume_search_activate),
			NULL);
  } else if (flags == 1) {
    if (status >= R_QUEUED)
      item = gtk_menu_item_new_with_label (_("Force download"));
    else
      item = gtk_menu_item_new_with_label (_("Download from User"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);
    if (status >= R_QUEUED)
      gtk_signal_connect(GTK_OBJECT (item), "activate",
			 GTK_SIGNAL_FUNC (on_download3_activate),
			 resume);
    else
      gtk_signal_connect(GTK_OBJECT (item), "activate",
			 GTK_SIGNAL_FUNC (on_download2_activate),
			 NULL);

    separator5 = gtk_menu_item_new ();
    gtk_widget_ref (separator5);
    gtk_object_set_data_full (GTK_OBJECT (popup), "separator5", separator5,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (separator5);
    gtk_container_add (GTK_CONTAINER (popup), separator5);
    gtk_widget_set_sensitive (separator5, FALSE);
    
    item = gtk_menu_item_new_with_label (_("User Menu"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);

    user_popup = create_user_popup(M_SEARCH);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), user_popup);
  }
  return popup;
}

resume_t* resume_find_search(search_t* search) {
  GList* dlist;
  resume_t* resume;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    if (resume->search == search) return resume;
  }
  return NULL;
}

void resume_insert_file(resume_t* resume, file_t* file) {
  GtkCTree* ctree;
  GtkCTreeNode* node;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;

  ctree = GTK_CTREE(lookup_widget(global.win, "resume_tree"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, resume);

  if (file->size != resume->size) file->status = S_CANCELED;
  else file->status = S_INACTIVE;
  if (!resume->user) {
    resume->user = strdup(file->user);
    resume->org_file = strdup(file->winname);
    gtk_ctree_node_set_pixtext(ctree, node, 1, resume->user, 5, 
			       global.pix.user1, global.pix.user1b);
    resume_list_save();
  }
  sprintf(tstr[0], "%s", file->winname);
  sprintf(tstr[1], "%s", file->user);
  sprintf(tstr[2], "%lu", file->size);
  sprintf(tstr[3], "%s", LineSpeed(file->linespeed));
  if (file->status == S_INACTIVE) strcpy(tstr[4], "");
  else strcpy(tstr[4], status_names(file->status));

  detect_speed_pixs(file->linespeed, &pixmap, &bitmap);
  node =
    gtk_ctree_insert_node(ctree, node, NULL, list, 5,
			  pixmap, bitmap, pixmap, bitmap,
			  TRUE, FALSE);

  gtk_ctree_node_set_row_data(ctree, node, file);
  resume_list_update(resume, 0);
}

resume_t* resume_search_file(char* filename) {
  GList* dlist;
  resume_t* resume;
  char* fname;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    fname = extract_filename(resume->filename);
    if (!strcmp(fname, filename)) return resume;
  }
  return NULL;
}

void resume_list_load(char* fname, int import) {
  FILE* fd;
  resume_t* resume;
  char line[2048];
  char* filename;
  char* size;
  char* dirname;
  char* user;
  char* org_file;
  char* search;
  char* short_file;
  int have_incomplete;

  //  printf("reading [%s][%s]\n", fname, global.incomplete_path);
  if (!global.incomplete_path || !directory_exists(global.incomplete_path)) {
    if (import) {
      client_message(_("Error"), _("Could not import incomplete list: no valid incomplete folder"));
      return;
    }
    have_incomplete = 0;
  } else have_incomplete = 1;

  if ((fd = fopen(fname, "r")) == NULL) return;

  while (fgets(line, 2048-1, fd)) {
    filename = arg(line, 0);
    short_file = extract_filename(filename);
    size = arg(NULL, 0);
    dirname = arg(NULL, 0);
    user = arg(NULL, 0);
    org_file = arg(NULL, 0);
    search = arg(NULL, 0);
    
    if (!filename || !size) continue;

    if (resume_search_file(short_file)) continue;
    resume = (resume_t*)malloc(sizeof(resume_t));
    if (have_incomplete && import)
      resume->filename = g_strdup_printf("%s/%s", global.incomplete_path, short_file);
    else resume->filename = strdup(filename);

    resume->size = strtoul(size, NULL, 10);
    resume->local_size = 0;
    
    // backward compatible
    if (dirname && *dirname) resume->dirname = strdup(dirname);
    else resume->dirname = NULL;

    // backward compatible
    if (user && *user) resume->user = strdup(user);
    else resume->user = NULL;

    // backward compatible
    if (org_file && *org_file) resume->org_file = strdup(org_file);
    else resume->org_file = NULL;

    // backward compatible
    if (search && *search) resume->search_string = strdup(search);
    else resume->search_string = NULL;

    resume->online = 0;
    resume->search = NULL;
    resume->status = R_NONE;
    resume->transfer = NULL;
    global.incomplete = g_list_append(global.incomplete, resume);

    resume_clist_add(resume);
  }
  fclose(fd);
  if (import) resume_list_save();
}

// function patched by saturn_de
void resume_list_save() {
  GList* dlist;
  resume_t* resume;
  char* fname;
  char* fname_new;
  FILE* fd;
  
  fname = g_strdup_printf("%s/incomplete.list", global.lopster_home);
  fname_new = g_strdup_printf("%s/incomplete.list.new",  global.lopster_home);

  if ((fd = fopen(fname_new, "w")) == NULL) {
    g_warning(_("Could not write [%s]\n"), fname);
    free(fname);
    free(fname_new);
    return;
  }

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    fprintf(fd, "\"%s\" %lu \"%s\" \"%s\" \"%s\" \"%s\"\n", resume->filename, 
	    resume->size, 
	    (resume->dirname)?(resume->dirname):"",
	    (resume->user)?(resume->user):"",
	    (resume->org_file)?(resume->org_file):"",
	    (resume->search_string)?(resume->search_string):"");
  }

  if(! ferror(fd) )
    rename(fname_new, fname);
  else {
    g_warning(_("Could not write [%s]\n"), fname);
  }

  fclose(fd);
  free(fname);
  free(fname_new);
}

void resume_list_add(resume_t* resume) {

  global.incomplete = g_list_append(global.incomplete, resume);
  resume_list_save();
  resume_clist_add(resume);
}

GList* resume_unlink_transfers(resume_t* resume) {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;
  GList* save = NULL;

#ifdef RESUME_DEBUG
  printf("removing r:%p\n", resume);
#endif
  // first remove resume link from unactive transfers
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (!socket) {
      printf("socket->data = NULL\n");
      continue;
    }
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer) continue;
    if (transfer->type != T_DOWNLOAD) continue;
    if (transfer->resume == resume) {
#ifdef RESUME_DEBUG
      printf("removed from  t:%p\n", transfer);
#endif
      transfer->resume = NULL;
      save = g_list_append(save, socket);
    }
  }
  
  return save;
}

void resume_list_freeze(resume_t* resume) {

  if (!resume) return;
  if (resume->status == R_FROZEN) return;
  
  resume_list_stop(resume, NULL);
  if (!g_list_find(global.incomplete, resume)) return;

  resume_remove_search(resume);
  resume->status = R_FROZEN;
  resume_list_update(resume, 0);
}

void resume_list_stop(resume_t* resume, transfer_t* transfer) {
  GList* result;
  GList* dlist;
  socket_t* socket;
  transfer_t* t2;

  if (!resume) return;
  
  result = resume_unlink_transfers(resume);

  for (dlist = result; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    t2 = (transfer_t*)(socket->data);
    // cancel all transfers
    if (t2->status == S_FINISHED) continue;
    if (t2 != transfer) socket_end(socket, &(SocketStatus[S_CANCELED]));
    // if (resume->transfer != transfer) socket_end(socket, S_CANCELED);
  }
  g_list_free(result);
}

void resume_list_thaw(resume_t* resume) {

  if (!resume) return;
  
  if (resume->status != R_FROZEN) return;
  resume_remove_search(resume);
  resume->status = R_NONE;
  resume_list_update(resume, 0);
}

void resume_list_remove(resume_t* resume, transfer_t* transfer) {
  GtkCTree* ctree;
  GtkCTreeNode* node;

  if (!resume) return;
  
  resume_list_stop(resume, transfer);

  // now checking it.
  if (!g_list_find(global.incomplete, resume)) return;

  resume_remove_search(resume);

  ctree = GTK_CTREE(lookup_widget(global.win, "resume_tree"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, resume);
  if (node) gtk_ctree_remove_node(ctree, node);

  if (resume->filename && (!transfer || (transfer->status != S_FINISHED))) {
    //    printf("delete = %s\n", resume->filename);
    unlink(resume->filename);
  }

  global.incomplete = g_list_remove(global.incomplete, resume);
  resume_list_save();
  if (resume->filename) free(resume->filename);
  if (resume->dirname) free(resume->dirname);
  if (resume->user) free(resume->user);
  if (resume->org_file) free(resume->org_file);
  free(resume);
}

void resume_clist_add(resume_t* resume) {
  GtkCTree* ctree;
  char* pos;
  GtkCTreeNode* node;
  GtkCTreeNode* node2;
  file_t* file;
  GList* dlist;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;

  if (!resume) return;

  ctree = GTK_CTREE(lookup_widget(global.win, "resume_tree"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, resume);
  if (node) return;

  pos = strrchr(resume->filename, '/');
  if (!pos) pos = resume->filename;
  else pos++;

  strcpy(tstr[0], pos);
  if (resume->user) strcpy(tstr[1], resume->user);
  else strcpy(tstr[1], _("? Unknown ?"));

  sprintf(tstr[2], "%lu", resume->size);
  
  tstr[3][0] = 0;
  tstr[4][0] = 0;

  node = gtk_ctree_insert_node(ctree, NULL, NULL, list, 5,
			       global.pix.folder, 
			       global.pix.folderb,
			       global.pix.folder_open, 
			       global.pix.folder_openb,
			       FALSE, FALSE);
  gtk_ctree_node_set_row_data(ctree, node, resume);
  if (resume->user) {
    if (resume->online)
      gtk_ctree_node_set_pixtext(ctree, node, 1, tstr[1], 5, 
				 global.pix.user1, global.pix.user1b);
    else    
      gtk_ctree_node_set_pixtext(ctree, node, 1, tstr[1], 5, 
				 global.pix.user2, global.pix.user2b);
  }

  resume_list_update(resume, 0);
  if (!resume->search) return;

  for (dlist = resume->search->results; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    
    detect_speed_pixs(file->linespeed, &pixmap, &bitmap);

    sprintf(tstr[0], "%s", file->winname);
    sprintf(tstr[1], "%s", file->user);
    sprintf(tstr[2], "%lu", file->size);
    sprintf(tstr[3], "%s", LineSpeed(file->linespeed));
    if (file->status == S_INACTIVE) strcpy(tstr[4], "");
    else strcpy(tstr[4], status_names(file->status));
    node2 =
      gtk_ctree_insert_node(ctree, node, NULL, list, 5,
			    pixmap, bitmap, pixmap, bitmap,
			    TRUE, FALSE);
    gtk_ctree_node_set_row_data(ctree, node2, file);

  }

}

void resume_list_update(resume_t* resume, int full) {
  GtkCTree* ctree;
  GtkCTreeNode* node;
  char text[1024];
  struct stat st;
  style_t* style;

  if (!resume) return;

  ctree = GTK_CTREE(lookup_widget(global.win, "resume_tree"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, resume);
  if (!node) {
    g_warning(_("Resume type not found in clist"));
    return;
  }

  if (full) {
    if (resume->user) {
      if (resume->online)
	gtk_ctree_node_set_pixtext(ctree, node, 1, resume->user, 5, 
				   global.pix.user1, global.pix.user1b);
      else    
	gtk_ctree_node_set_pixtext(ctree, node, 1, resume->user, 5, 
				   global.pix.user2, global.pix.user2b);
    } else {
      gtk_ctree_node_set_text(ctree, node, 1, _("? Unknown ?"));
    }
  }

  if (stat(resume->filename, &st) < 0) {
    resume->local_size = 0;
    strcpy(text, _("not existent"));
  } else {
    resume->local_size = st.st_size;
    sprintf(text, "%.2f %%", 100.*(double)st.st_size/(double)resume->size);
  }
  gtk_ctree_node_set_text(ctree, node, 3, text);

  if (resume->status == R_NONE) {
    if (!resume->search) {
      sprintf(text, _("Not Searched"));
    } else {
      if (resume->search->queued)
	sprintf(text, _("Search queued"));
      else
	sprintf(text, _("%d results"), g_list_length(resume->search->results));
    }
  } else if (resume->status == R_QUEUED)
    sprintf(text, _("Queued..."));
  else if (resume->status == R_DOWNLOAD)
    sprintf(text, _("Downloading..."));
  else if (resume->status == R_FROZEN)
    sprintf(text, _("Frozen..."));
  else
    sprintf(text, _("Unknown status"));

  gtk_ctree_node_set_text(ctree, node, 4, text);

  if (resume->status == R_QUEUED) {
    style = style_get(global.scheme, "transfer_waiting");
    if (style) {
      gtk_ctree_node_set_background(ctree, node, style->back);
      gtk_ctree_node_set_foreground(ctree, node, style->fore);
    }
  } else if (resume->status == R_DOWNLOAD) {
    style = style_get(global.scheme, "transfer_running");
    if (style) {
      gtk_ctree_node_set_background(ctree, node, style->back);
      gtk_ctree_node_set_foreground(ctree, node, style->fore);
    }
  } else {
    gtk_ctree_node_set_background(ctree, node, NULL);
    gtk_ctree_node_set_foreground(ctree, node, NULL);
  }
}

resume_t* resume_list_search_filename(char* filename) {
  GList* dlist;
  resume_t* resume;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    if (!strcmp(resume->filename, filename)) return resume;
  }
  return NULL;
}

resume_t* resume_list_search_file(file_t* file) {
  GList* dlist;
  resume_t* resume;
  char* pos;
  char* pos2;

  pos2 = extract_filename(file->longname);
#ifdef RESUME_DEBUG
  //  printf("resume: searching2 [%s]\n", pos2);
#endif
  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    pos = extract_filename(resume->filename);
#ifdef RESUME_DEBUG
    //    printf("resume: target2 [%s]\n", pos);
#endif
    if (!strcmp(pos2, pos) && 
	(file->size == resume->size)) return resume;
  }
  return NULL;
}

resume_t* resume_list_search(transfer_t* transfer) {
  GList* dlist;
  resume_t* resume;
  /*
  char* pos;
  char* pos2;
  */
  //  pos2 = extract_filename(transfer->longname);
  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    /*
    pos = extract_filename(resume->filename);
    if (!strcmp(pos2, pos) && 
	(transfer->size == resume->size)) return resume;
    */
    if (transfer->size == resume->size) return resume;
  }
  return NULL;
}

char* get_search_string(char* str) {
  static char result[1024];
  char* pos;
  char* old_pos;
  char save;
  char* last;
  char* str2;

  str2 = strdup(str);
  pos = str2;

  last = strrchr(str2, '.');
  *result = 0;
  while (1) {
    old_pos = pos;
    while (isalpha(*pos)) pos++;
    save = *pos;
    if (pos > old_pos+1) {
      *pos = 0;
      if (*result) strcat(result, " ");
      strcat(result, old_pos);
    }
    if (last && (pos >= last)) break;
    if (save == 0) break;
    pos++;
  }
  free(str2);
  return result;
}

void resume_create_search(resume_t* resume) {
  search_t* search;
  search_pattern_t* pattern;
  char t2[1024];
  char* pos;

  search = search_new();
  search->resume = 1;
  search->link = resume;

  pos = strdup(extract_filename(resume->filename));
  sprintf(t2, "%d %s", search->id, pos);

  pattern = search_pattern_new(t2);
  if (resume->search_string && *(resume->search_string))
    pattern->include = strdup(resume->search_string);
  else
    pattern->include = strdup(get_search_string(pos));
  free(pos);
  pattern->exclude = strdup("");
  pattern->max_results = 100;
  pattern->destination = DEST_NAPSTER;

  //  if (opennap_version(0, 37)) {
  pattern->size_lo = pattern->size_hi = resume->size;
  pattern->media_type = strdup("any");
  //  } else {
  //    pattern->size_lo = pattern->size_hi = 0;
  //    pattern->media_type = strdup("mp3");
  //  }  

  search->pattern = pattern;
  resume->search = search;
  search_queue(search);
}

void resume_search(resume_t* resume, int search) {
  
  if (!resume) return;
  if (resume->status == R_FROZEN) return;
  
  resume_remove_search(resume);
  resume_create_search(resume);
  if (resume->user && search) {
    whois_request(resume->user, WHOIS_ONLINE);
  }
}

void resume_remove_search(resume_t* resume) {
  GtkCTreeNode* node;
  GtkCTreeNode* node2;
  file_t* file;
  GtkCTree* ctree;
  
  if (!resume->search) return;

  // remove files from window
  ctree = GTK_CTREE(lookup_widget(global.win, "resume_tree"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, resume);
  while ((node2 = GTK_CTREE_ROW(node)->children) != NULL) {
    file = (file_t*)gtk_ctree_node_get_row_data(ctree, node2);
    gtk_ctree_remove_node(ctree, node2);
  }
  
  // remove files from resume
  if (resume->search) search_remove(resume->search);
  resume->search = NULL;

  if (resume->status < R_QUEUED) resume->status = R_NONE;
  resume_list_update(resume, 0);
}

void resume_try_next_potential(resume_t* resume) {
  GList* dlist;
  file_t* file;
  file_t* dfile = NULL;
  
  if (!resume->search) return;

  // finding best file
  for (dlist = resume->search->results; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    if ((file->status == S_INACTIVE) &&
	(!dfile || (file->linespeed > dfile->linespeed)))
      dfile = file;
  }
  
  if (dfile) download_file(dfile, 0);
}

void resume_user_flag(resume_t* resume, char* user, int flag) {
  GList* dlist;
  file_t* file;
  GtkCTree* ctree;
  char* text;
  GtkCTreeNode* node;

  if (!resume->search) return;

  ctree = GTK_CTREE(lookup_widget(global.win, "resume_tree"));
  
  for (dlist = resume->search->results; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    if (!strcmp(user, file->user)) {
      file->status = flag;
      node = gtk_ctree_find_by_row_data(ctree, NULL, file);
      if (file->status == S_INACTIVE) text = strdup("");
      else text = strdup(status_names(file->status));
      if (node) gtk_ctree_node_set_text(ctree, node, 4, text);
    }
  }
}

void resume_search_all() {
  resume_t* resume;
  GtkCTree* ctree;
  GtkCTreeNode* node;
  GList* users = NULL;
  GList* dlist;

  ctree = GTK_CTREE(lookup_widget(global.win, "resume_tree"));
  gtk_clist_freeze(GTK_CLIST(ctree));
  node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list);
  while (node) {
    resume = gtk_ctree_node_get_row_data(ctree, node);
    node = GTK_CTREE_ROW(node)->sibling;
    if (resume->status != R_DOWNLOAD) {
      if (resume->user && (!is_string_in_list(users, resume->user))) {
	users = g_list_append(users, resume->user);
      }
      resume_search(resume, 0);
    }
  }
  gtk_clist_thaw(GTK_CLIST(ctree));

  for (dlist = users; dlist; dlist = dlist->next) {
    whois_request(dlist->data, WHOIS_ONLINE);
  }
  g_list_free(users);
}

int resume_timer(gpointer data) {
  resume_search_all();

  return 1;
}

void resume_user_online(char* user, int link) {
  GList* dlist;
  resume_t* resume;
  GtkCTree* ctree;
  GtkCTreeNode* node;
  file_t* new_file;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    if (!resume->user) continue;
    if (strcasecmp(resume->user, user)) continue;

    resume->online = 1;

    ctree = GTK_CTREE(lookup_widget(global.win, "resume_tree"));
    node = gtk_ctree_find_by_row_data(ctree, NULL, resume);

    gtk_ctree_node_set_pixtext(ctree, node, 1, resume->user, 5, 
			       global.pix.user1, global.pix.user1b);

    if (!resume->org_file) continue;
    if (!resume->search) continue;
    //    if (resume->search->queued) continue;

    new_file = (file_t*)malloc(sizeof(file_t));
    new_file->winname = strdup(resume->org_file);
    new_file->md5 = strdup("UNKNOWN");
    new_file->size = resume->size;
    new_file->bitrate = 128;
    new_file->frequency = 44100;
    new_file->duration = 0;
    new_file->user = strdup(resume->user);
    new_file->ip = 0;
    new_file->linespeed = link;
    new_file->longname = strdup(new_file->winname);
    convert_to_unix(new_file->longname, 1);
    new_file->shortname = strdup(extract_short_name(new_file->longname));
    new_file->mime_type = get_mimetype(new_file->shortname);
    new_file->flags = 0;
    new_file->status = S_INACTIVE;
    new_file->local = 0;            //remote file
    new_file->ping = 0;
    if (!search_find_file(resume->search, new_file)) {
      resume->search->results = g_list_append(resume->search->results, new_file);
      resume_insert_file(resume, new_file);
    } else {
      destroy_file_row(new_file);
    }
  }
}

void resume_user_offline(char* user) {
  GList* dlist;
  resume_t* resume;
  GtkCTree* ctree;
  GtkCTreeNode* node;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    if (!resume->user) continue;
    if (strcasecmp(resume->user, user)) continue;

    resume->online = 0;

    ctree = GTK_CTREE(lookup_widget(global.win, "resume_tree"));
    node = gtk_ctree_find_by_row_data(ctree, NULL, resume);

    gtk_ctree_node_set_pixtext(ctree, node, 1, resume->user, 5, 
			       global.pix.user2, global.pix.user2b);
  }
}

