#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <unistd.h>
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <netdb.h>
#include <sys/utsname.h>

#include "lopster.h"
#include "connection.h"
#include "global.h"
#include "search.h"
#include "transfer.h"
#include "resume.h"
#include "interface.h"
#include "support.h"
#include "callbacks.h"
#include "browse.h"
#include "share.h"
#include "hotlist.h"
#include "commands.h"
#include "chat.h"
#include "dirselect.h"
#include "scheme.h"
#include "handler.h"
#include "resume.h"
#include "server.h"
#include "preferences.h"
#include "log.h"
#include "exec.h"

int exec_find_free_id();

exec_t* exec_new(char* name) {
  exec_t* exec;

  exec = (exec_t*)malloc(sizeof(exec_t));
  exec->input = -1;
  exec->output = O_WINDOW;
  exec->who = NULL;
  exec->buf[0] = 0;
  exec->pos = 0;
  exec->id = exec_find_free_id();
  exec->pid = 0;
  if (name) exec->command = strdup(name);
  else exec->command = NULL;

  global.execs = g_list_append(global.execs, exec);
  return exec;
}

void exec_destroy(exec_t* exec) {
  int status;

  waitpid(exec->pid, &status, WNOHANG);
  if (exec->output != O_NONE) {
    if (WIFEXITED(status))
      client_message(_("Exec"), _("Process [%d] terminated (%d)"),
		     exec->pid, WEXITSTATUS(status));
    else
      client_message(_("Exec"), _("Process [%d] terminated"), exec->pid);
  }
  if (exec->fd >= 0) close(exec->fd);
  if (strlen(exec->buf)) {
    exec_print_line(exec, exec->buf);
  }
  
  if (exec->input >= 0) gdk_input_remove(exec->input);
  if (exec->command) free(exec->command);
  
  global.execs = g_list_remove(global.execs, exec);

  free(exec);
}

char* exec_get_output(exec_t* exec) {
  static char result[1024];

  switch (exec->output) {
  case O_NONE:
    sprintf(result, "[%s]", _("No Output")); break;
  case O_WINDOW:
    sprintf(result, "[%s]", _("Window")); break;
  case O_CHANNEL:
    sprintf(result, "[%s:%s]", _("Channel"), exec->who); break;
  case O_PRIVATE:
    sprintf(result, "[%s:%s]", _("Private"), exec->who); break;
  case O_OPERATOR:
    sprintf(result, "[%s:%s]", _("Channel op"), exec->who); break;
  case O_GLOBAL:
    sprintf(result, "[%s]", _("Global")); break;
  case O_WALLOP:
    sprintf(result, "[%s]", _("Wallop")); break;
  default:
    sprintf(result, "[%s]", _("Unknown")); break;
  }

  return result;
}
void exec_list() {
  GList* dlist;
  exec_t* exec;
  char id[100];

  if (global.execs) {
    client_message(NULL, _("Lopster Process List:"));
    for (dlist = global.execs; dlist; dlist = dlist->next) {
      exec = (exec_t*)(dlist->data);
      sprintf(id, "%2d", exec->id);
      client_message(id, "[%d] %s %s", exec->pid,
		     exec_get_output(exec), exec->command);
    }
  } else {
    client_message(NULL, _("No Processes running"));
  }
}

int exec_find_free_id() {
  int id = 0;
  GList* dlist;
  exec_t* exec;

  while (id < 1000) {
    for (dlist = global.execs; dlist; dlist = dlist->next) {
      exec = (exec_t*)(dlist->data);
      if (exec->id == id) break;
    }
    if (dlist) id++;
    else return id;
  }
  return -1;
}

exec_t* exec_find_id(int id) {
  GList* dlist;
  exec_t* exec;

  for (dlist = global.execs; dlist; dlist = dlist->next) {
    exec = (exec_t*)(dlist->data);
    if (exec->id == id) return exec;
  }
  return NULL;
}

void exec_kill(int id) {
  exec_t* exec;

  exec = exec_find_id(id);
  if (!exec) {
    client_message(_("Error"), _("Could not kill proccess: not found"));
    return;
  }

  client_message(NULL, _("Killing process.."));
  kill(exec->pid, SIGTERM);
  sleep(2);
  kill(exec->pid, SIGKILL);
}

void exec_kill_all() {
  exec_t* exec;
  GList* dlist;

  if (!global.execs) return;

  for (dlist = global.execs; dlist; dlist = dlist->next) {
    exec = (exec_t*)(dlist->data);
    kill(exec->pid, SIGTERM);
  }
  sleep(2);
  for (dlist = global.execs; dlist; dlist = dlist->next) {
    exec = (exec_t*)(dlist->data);
    kill(exec->pid, SIGKILL);
  }
}

void exec_fire_up(char* command, int output, char* who) {
  int	p1[2], pid;
  int i;
  exec_t* exec;
  gchar *argv[4];

  p1[0] = p1[1] = -1;
  
  if (pipe(p1)) {
    client_message(NULL, _("Unable to start new process: %s"),
		   strerror(errno));
    if (p1[0] >= 0) close(p1[0]);
    if (p1[1] >= 0) close(p1[1]);
    return;
  }
  
  switch ((pid = fork())) {
  case -1:
    client_message(NULL, _("Couldn't start new process!"));
    break;
  case 0:
    // child
    setsid();
    setuid(getuid());
    setgid(getgid());
    
    dup2(p1[1], 1);  // copy stdout
    close(p1[0]);
    p1[0] = -1;
    for (i = 3; i < 256; i++) close(i);

    argv[0] = "sh";
    argv[1] = "-c";
    argv[2] = command;
    argv[3] = 0;
    execve("/bin/sh", argv, environ);
    _exit(-1);
    break;
  default:
    // parent
    close(p1[1]);
    p1[1] = -1;
    
    exec = exec_new(command);
    exec->output = output;
    exec->pid = pid;
    exec->fd = p1[0];
    if (who) exec->who = strdup(who);
    
    exec->input = 
      gdk_input_add(exec->fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(get_exec_input), exec);
    
    break;
  }

}

/*
  executes a command
  basic implementation by r00t
  added non blocking output read and more.
*/
void exec_command(char* data) {
  char* line;
  char* command;
  int output = O_WINDOW;
  char* who = NULL;

  line = arg(data, 1);
  if (!line) {
    exec_list();
    return;
  }
  if (!strncasecmp(line, "-out", 4)) {
    command = arg(line, 0);
    command = arg(NULL, 1);
    if (in_channel()) {
      output = O_CHANNEL;
      who = global.current_page->name;
    } else if (in_private()) {
      output = O_PRIVATE;
      who = global.current_page->name;
    } else {
      output = O_WALLOP;
    }
  } else if (!strncasecmp(line, "-msg", 4) ||
	     !strncasecmp(line, "-whisper", 8)) {
    output = O_PRIVATE;
    command = arg(line, 0);
    who = arg(NULL, 0);
    command = arg(NULL, 1);
  } else if (!strncasecmp(line, "-wallop", 5)) {
    output = O_WALLOP;
    command = arg(line, 0);
    command = arg(NULL, 1);
  } else if (!strncasecmp(line, "-chop", 3)) {
    command = arg(line, 0);
    command = arg(NULL, 1);
    if (in_channel()) {
      output = O_OPERATOR;
      who = global.current_page->name;
    } else {
      client_message(_("Error"), _("Switch to a channel"));
      return;
    }
  } else if (!strncasecmp(line, "-quiet", 6)) {
    output = O_NONE;
    command = arg(line, 0);
    command = arg(NULL, 1);
  } else if (!strncasecmp(line, "-global", 7)) {
    output = O_GLOBAL;
    command = arg(line, 0);
    command = arg(NULL, 1);
  } else if (!strncasecmp(line, "-kill", 5)) {
    command = arg(line, 0);
    command = arg(NULL, 1);
    if (!command) {
      client_message(_("Error"), _("Kill what process?"));
      return;
    }
    if (!strcasecmp(command, "all")) exec_kill_all();
    else exec_kill(atoi(command));
    return;
  } else if (*line == '-') {
    client_message(_("Error"), _("Unknown option!"));
    return;
  } else {
    output = O_WINDOW;
    command = line;
  }
  if (!command) {
    client_message(_("Error"), _("Execute what?"));
    return;
  }

  exec_fire_up(command, output, who);
}

gint get_exec_input(gpointer data, gint source, 
		    GdkInputCondition condition) {
  exec_t* exec = (exec_t*)data;
  int res;
  char* pos1;
  char* pos2;

  if (condition != GDK_INPUT_READ) {
    exec_destroy(exec);
    return 0;
  }
  switch (res = read(source, exec->buf+exec->pos, 
		     2047-exec->pos-1)) {
  case -1:
    exec_destroy(exec);
    return 0;
  case 0:
    exec_destroy(exec);
    return 0;
  default:
    break;
  }

  exec->buf[res+exec->pos] = 0;

  pos1 = exec->buf;
  while ((pos2 = strchr(pos1, '\n')) || (pos2 = strchr(pos1, '\r'))) {
    *pos2 = 0;
    exec_print_line(exec, pos1);
    pos1 = pos2+1;
  }

  strcpy(exec->buf, pos1);
  exec->pos = strlen(exec->buf);
  return 0;
}

void exec_print_line(exec_t* exec, char* line) {
  switch (exec->output) {
  case O_NONE:
    break;
  case O_CHANNEL:
    send_public(exec->who, line);
    break;
  case O_PRIVATE:
    send_private(exec->who, line, 1);
    break;
  case O_WALLOP:
    send_wallop(line);
    break;
  case O_OPERATOR:
    send_chwallop(exec->who, line);
    break;
  case O_GLOBAL:
    send_global(line);
    break;
  case O_WINDOW:
  default:
    client_message(NULL, "%s", line);
    break;
  }
}
