#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <strings.h>

#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>

#include "lopster.h"
#include "commands.h"
#include "callbacks.h"
#include "connection.h"
#include "support.h"
#include "global.h"
#include "chat.h"
#include "search.h"
#include "transfer.h"
#include "hotlist.h"
#include "log.h"
#include "dialog.h"
#include "handler.h"
#include "resume.h"

file_t* global_file2;

const int StatusInfo[S_NUMBER] = {
  T_NONE, T_CURRENT, T_CURRENT, T_TRANSFER,
  T_NONE, T_NONE, T_NONE, T_NONE,
  T_NONE, T_TRANSFER, T_CURRENT, T_CURRENT,
  T_CURRENT, T_CURRENT, T_NONE, T_NONE,
  T_NONE, T_NONE, T_NONE, T_STOPPED,
  T_NONE, T_NONE, T_NONE
};

const int StatusDist[4][4] = {
  { 0, 1, 0, 1},
  {-1, 0,-1, 0},
  { 0, 1, 0, 1},
  {-1, 0,-1, 0}
};

char* status_names(int status) {
  switch (status) {
  case 0: return _("Inactive");
  case 1: return _("Connecting...");
  case 2: return _("Getting info...");
  case 3: return _("Downloading");
  case 4: return _("Canceled");
  case 5: return _("Finished!");
  case 6: return _("Timeout!");
  case 7: return _("Rejected");
  case 8: return _("Incomplete");
  case 9: return _("Uploading");
  case 10:return _("Getting info");
  case 11: return _("Getting info");
  case 12: return _("Getting info");
  case 13: return _("Waiting");
  case 14: return _("Both Firewalled");
  case 15: return _("Connection Error");
  case 16: return _("Queued");
  case 17: return _("Remotely Queued");
  case 18: return _("Unavailable!");
  case 19: return _("Stopped");
  case 20: return _("Resume Error");
  case 21: return _("IO error");
  default: return _("unknown state");
  }
}

GtkWidget* create_upload_popup(transfer_t* transfer) {
  GtkWidget *popup;
  GtkWidget *user_popup;
  GtkWidget *item;
  GtkAccelGroup *popup_accels;
  GtkWidget *delete_upload;
  GtkWidget *separator;
  GtkWidget *trennlinie16;
  GtkWidget *customize_list2;
  GtkCList* clist;
  int item_num;
  char item_str[1024];

  popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (popup));

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  item_num = g_list_length(clist->selection);
  
  if (transfer) {
    if (item_num > 1) sprintf(item_str, _("Delete Selected (%d)"), item_num);
    else sprintf(item_str, _("Delete Upload"));
    delete_upload = gtk_menu_item_new_with_label (item_str);
    gtk_widget_ref (delete_upload);
    gtk_object_set_data_full (GTK_OBJECT (popup), "delete_upload", delete_upload,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (delete_upload);
    gtk_container_add (GTK_CONTAINER (popup), delete_upload);
    gtk_signal_connect (GTK_OBJECT (delete_upload), "activate",
			GTK_SIGNAL_FUNC (on_delete_transfer_activate),
			NULL);
    
    separator = gtk_menu_item_new ();
    gtk_widget_ref (separator);
    gtk_object_set_data_full (GTK_OBJECT (popup), "separator", separator,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);
    
    delete_upload = gtk_menu_item_new_with_label (_("Open File"));
    gtk_widget_ref (delete_upload);
    gtk_object_set_data_full (GTK_OBJECT (popup), "play_upload", delete_upload,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (delete_upload);
    gtk_container_add (GTK_CONTAINER (popup), delete_upload);
    
    gtk_signal_connect (GTK_OBJECT (delete_upload), "activate",
			GTK_SIGNAL_FUNC (on_play_file_activate),
			NULL);

    separator = gtk_menu_item_new ();
    gtk_widget_ref (separator);
    gtk_object_set_data_full (GTK_OBJECT (popup), "separator", separator,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);

    item = gtk_menu_item_new_with_label (_("User Menu"));
    gtk_widget_ref (item);
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);

    user_popup = create_user_popup(M_TRANSFER);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), user_popup);

    trennlinie16 = gtk_menu_item_new ();
    gtk_widget_ref (trennlinie16);
    gtk_object_set_data_full (GTK_OBJECT (popup), "trennlinie16", trennlinie16,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (trennlinie16);
    gtk_container_add (GTK_CONTAINER (popup), trennlinie16);
    gtk_widget_set_sensitive (trennlinie16, FALSE);
  }

  customize_list2 = gtk_menu_item_new_with_label (_("Customize List"));
  gtk_widget_ref (customize_list2);
  gtk_object_set_data_full (GTK_OBJECT (popup), "customize_list2", customize_list2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (customize_list2);
  gtk_container_add (GTK_CONTAINER (popup), customize_list2);
  gtk_signal_connect (GTK_OBJECT (customize_list2), "activate",
                      GTK_SIGNAL_FUNC (on_customize_list_activate),
                      NULL);

  return popup;
}

GtkWidget*
create_download_popup (transfer_t* transfer) {
  GtkWidget *user_popup;
  GtkWidget *popup;
  GtkAccelGroup *popup_accels;
  GtkWidget *cancel_download;
  GtkWidget *delete_download;
  GtkWidget *separator3;
  GtkWidget *retry_download;
  GtkWidget *resume_download;
  GtkWidget *separator4;
  GtkWidget *item;
  GtkWidget *trennlinie7;
  GtkWidget *play_file;
  GtkWidget *trennlinie16;
  GtkWidget *customize_list2;
  GtkCList* clist;
  int item_num;
  char item_str[1024];

  popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (popup));
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  item_num = g_list_length(clist->selection);

  if (transfer) {
    if (item_num > 1) sprintf(item_str, _("Cancel Selected (%d)"), item_num);
    else sprintf(item_str, _("Cancel Download"));
    cancel_download = gtk_menu_item_new_with_label (item_str);
    gtk_widget_ref (cancel_download);
    gtk_object_set_data_full (GTK_OBJECT (popup), "cancel_download", cancel_download,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (cancel_download);
    gtk_container_add (GTK_CONTAINER (popup), cancel_download);
    gtk_signal_connect (GTK_OBJECT (cancel_download), "activate",
			GTK_SIGNAL_FUNC (on_cancel_transfer_activate),
			NULL);
    
    if (item_num > 1) sprintf(item_str, _("Delete Selected (%d)"), item_num);
    else sprintf(item_str, _("Delete Download"));
    delete_download = gtk_menu_item_new_with_label (item_str);
    gtk_widget_ref (delete_download);
    gtk_object_set_data_full (GTK_OBJECT (popup), "delete_download", delete_download,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (delete_download);
    gtk_container_add (GTK_CONTAINER (popup), delete_download);
    gtk_signal_connect (GTK_OBJECT (delete_download), "activate",
			GTK_SIGNAL_FUNC (on_delete_transfer_activate),
			NULL);
    
    separator3 = gtk_menu_item_new ();
    gtk_widget_ref (separator3);
    gtk_object_set_data_full (GTK_OBJECT (popup), "separator3", separator3,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (separator3);
    gtk_container_add (GTK_CONTAINER (popup), separator3);
    gtk_widget_set_sensitive (separator3, FALSE);
    
    if (transfer->status == S_QUEUED) {
      retry_download = gtk_menu_item_new_with_label (_("Force Download"));
      gtk_widget_ref (retry_download);
      gtk_object_set_data_full (GTK_OBJECT (popup), "force_download", retry_download,
				(GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show (retry_download);
      gtk_container_add (GTK_CONTAINER (popup), retry_download);
      
      gtk_signal_connect (GTK_OBJECT (retry_download), "activate",
			  GTK_SIGNAL_FUNC (on_force_download_activate),
			  NULL);
    } if (transfer->status == S_STOPPED) {
      resume_download = gtk_menu_item_new_with_label (_("Continue Download"));
      gtk_widget_ref (resume_download);
      gtk_object_set_data_full (GTK_OBJECT (popup), "resume_download", resume_download,
				(GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show (resume_download);
      gtk_container_add (GTK_CONTAINER (popup), resume_download);
      gtk_signal_connect (GTK_OBJECT (resume_download), "activate",
			  GTK_SIGNAL_FUNC (on_resume_download_activate),
			  NULL);
    } else if (!transfer_in_progress(transfer) && (transfer->status != S_FINISHED)) {
      retry_download = gtk_menu_item_new_with_label (_("Retry Download"));
      gtk_widget_ref (retry_download);
      gtk_object_set_data_full (GTK_OBJECT (popup), "retry_download", retry_download,
				(GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show (retry_download);
      gtk_container_add (GTK_CONTAINER (popup), retry_download);
      
      gtk_signal_connect (GTK_OBJECT (retry_download), "activate",
			  GTK_SIGNAL_FUNC (on_retry_download_activate),
			  NULL);
    } else if (transfer->status == S_DOWNLOADING) {
      resume_download = gtk_menu_item_new_with_label (_("Suspend Download"));
      gtk_widget_ref (resume_download);
      gtk_object_set_data_full (GTK_OBJECT (popup), "stop_download", resume_download,
				(GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show (resume_download);
      gtk_container_add (GTK_CONTAINER (popup), resume_download);
      gtk_signal_connect (GTK_OBJECT (resume_download), "activate",
			  GTK_SIGNAL_FUNC (on_stop_download_activate),
			  NULL);
      //      gtk_widget_set_sensitive(resume_download, FALSE);
    } else {
      resume_download = gtk_menu_item_new_with_label (_("Suspend Download"));
      gtk_widget_ref (resume_download);
      gtk_object_set_data_full (GTK_OBJECT (popup), "stop_download", resume_download,
				(GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show (resume_download);
      gtk_container_add (GTK_CONTAINER (popup), resume_download);
      gtk_signal_connect (GTK_OBJECT (resume_download), "activate",
			  GTK_SIGNAL_FUNC (on_stop_download_activate),
			  NULL);
      gtk_widget_set_sensitive(resume_download, FALSE);
    }
    
    separator4 = gtk_menu_item_new ();
    gtk_widget_ref (separator4);
    gtk_object_set_data_full (GTK_OBJECT (popup), "separator4", separator4,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (separator4);
    gtk_container_add (GTK_CONTAINER (popup), separator4);
    gtk_widget_set_sensitive (separator4, FALSE);
    
    item = gtk_menu_item_new_with_label (_("User Menu"));
    gtk_widget_ref (item);
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (popup), item);

    user_popup = create_user_popup(M_TRANSFER);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), user_popup);

    trennlinie7 = gtk_menu_item_new ();
    gtk_widget_ref (trennlinie7);
    gtk_object_set_data_full (GTK_OBJECT (popup), "trennlinie7", trennlinie7,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (trennlinie7);
    gtk_container_add (GTK_CONTAINER (popup), trennlinie7);
    gtk_widget_set_sensitive (trennlinie7, FALSE);
    
    play_file = gtk_menu_item_new_with_label (_("Open File"));
    gtk_widget_ref (play_file);
    gtk_object_set_data_full (GTK_OBJECT (popup), "play_file", play_file,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (play_file);
    gtk_container_add (GTK_CONTAINER (popup), play_file);
    
    gtk_signal_connect (GTK_OBJECT (play_file), "activate",
			GTK_SIGNAL_FUNC (on_play_file_activate),
			NULL);
    trennlinie16 = gtk_menu_item_new ();
    gtk_widget_ref (trennlinie16);
    gtk_object_set_data_full (GTK_OBJECT (popup), "trennlinie16", trennlinie16,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (trennlinie16);
    gtk_container_add (GTK_CONTAINER (popup), trennlinie16);
    gtk_widget_set_sensitive (trennlinie16, FALSE);
  }

  customize_list2 = gtk_menu_item_new_with_label (_("Customize List"));
  gtk_widget_ref (customize_list2);
  gtk_object_set_data_full (GTK_OBJECT (popup), "customize_list2", customize_list2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (customize_list2);
  gtk_container_add (GTK_CONTAINER (popup), customize_list2);
  gtk_signal_connect (GTK_OBJECT (customize_list2), "activate",
                      GTK_SIGNAL_FUNC (on_customize_list_activate),
                      NULL);

  return popup;
}

char* valid_download_folder(int mime) {
  if (!global.incomplete_path ||
      !directory_exists(global.incomplete_path))
    return NULL;
  if (!global.mimetype[mime].download ||
      !directory_exists(global.mimetype[mime].download))
    return NULL;
  return global.mimetype[mime].download;
}

#define CTCP_DELIM_CHAR '\001'

int check_dcc_message(char *from, char *message) {
  char* nick;
  unsigned long ip_long;
  int port;
  char* filename;
  unsigned long filesize;
  char* command;
  char* checksum;
  int speed;
  transfer_t* new_trans;
  char* folder;
  socket_t* socket;

  if (!message || strlen(message)<2) return 0;

  if (*message == CTCP_DELIM_CHAR && 
      message[strlen(message)-1] == CTCP_DELIM_CHAR) {
    message++;
    message[strlen(message)-1] = 0;
    
    command = arg(message, 0);
    if (command && strncmp(command, "SEND", 4)) return 0;
    nick = arg(NULL, 0);
    ip_long = strtoul(arg(NULL, 0), NULL, 10);
    port = atoi(arg(NULL, 0));
    filename = arg(NULL, 0);
    filesize = strtoul(arg(NULL, 0), NULL, 10);
    checksum = arg(NULL, 0);
    speed = atoi(arg(NULL, 0));

    if (filesize == 0) {
      client_message("DCC", _("%s is sending a 0 byte %s, ignoring"), 
		     nick, filename);
      return 1;
    } else {
      client_message("DCC", _("%s is sending %s"), nick, filename);
    }
    if (port == 0 && global.network.port == 0) {
      client_message("Message", _("Both systems are firewalled. Unable to comply"));
      return 1;
    }

    new_trans = transfer_new();
    new_trans->winname = strdup(filename);
    new_trans->mime_type = get_mimetype(filename);
    new_trans->longname = strdup(new_trans->winname);
    convert_to_unix(new_trans->longname);
    new_trans->shortname = 
      strdup(extract_short_name(new_trans->longname));

    new_trans->linespeed = speed;
    new_trans->user = strdup(nick);
    new_trans->size = filesize;
    new_trans->type = T_DOWNLOAD;
    
    if ((folder = valid_download_folder(new_trans->mime_type)) == NULL) {
      transfer_destroy(new_trans);
      download_dialog(new_trans->shortname, new_trans->mime_type);
      return 0;
    }
    
    new_trans->download_dir = strdup(folder);
    new_trans->md5 = strdup(checksum);
    new_trans->is_dcc = TRUE;
    socket = socket_new(S_TRANSFER);
    socket->data = new_trans;
    socket->port = htons(port);
    socket->ip_long = BSWAP32(ip_long);

    transfer_insert(socket);
    
    if (!global.network.allow_dcc) {
      transfer_status_set(socket, S_STOPPED);
      client_message(NULL, _("At the moment you do not allow dcc file transfers"), nick, filename);
      client_message(NULL, _("If you want to accept the transfer, go to the transfer page and continue download"));
      send_private(nick, "DCC transfers are not allowed at the moment, maybe i will start the download manually");
      return 1;
    } else {
      download_start(socket, FALSE);
    }
    
    return 1;
  }
  return 0;
}

transfer_t* transfer_new() {
  transfer_t *transfer;
  int i1;

  transfer = (transfer_t*)malloc(sizeof(transfer_t));

  transfer->longname = NULL;        //
  transfer->shortname = NULL;       //
  transfer->winname = NULL;         //
  transfer->download_dir = NULL;
  transfer->md5 = NULL;
  transfer->user = NULL;
  transfer->linespeed = 0;
  transfer->progress = 0;
  transfer->rate = 0;
  transfer->timeleft = 0;
  transfer->start_time = 0;
  transfer->status = S_INACTIVE;
  transfer->size = 0;

  transfer->file = NULL;
  transfer->type = T_UNKNOWN;
  transfer->resume_check = NULL;
  transfer->check_length = 0;
  transfer->resume = NULL;
  transfer->user_info = NULL;
  transfer->is_dcc = FALSE;
  transfer->mime_type = MIME_NONE;

  for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
    transfer->history[i1] = 0;
  transfer->hist_pos = 0;
  transfer->hist_cnt = 0;

  return transfer;
}

void transfer_set_md5(transfer_t* transfer, char* md5) {
  if (transfer->md5) free(transfer->md5);
  transfer->md5 = strdup(md5);
}

int transfer_in_progress(transfer_t* transfer) {
  if ((StatusInfo[transfer->status] == T_CURRENT) ||
      (StatusInfo[transfer->status] == T_TRANSFER)) return 1;
  else return 0;
}
int transfer_is_active(transfer_t* transfer) {
  if (transfer_in_progress(transfer) ||
      (StatusInfo[transfer->status] == T_STOPPED)) return 1;
  else return 0;
}

int transfer_status_dist(int s1, int s2) {
  return StatusDist[StatusInfo[s1]][StatusInfo[s2]];
}

void transfer_status_set(socket_t* socket, int status) {
  transfer_t* transfer = (transfer_t*)(socket->data);
  int add = transfer_status_dist(transfer->status, status);

#ifdef TRANSFER_DEBUG  
  printf("old/new %d/%d :%d\n", transfer->status, status, add);
#endif
  if (transfer->type == T_DOWNLOAD) {
    if (transfer->user_info) transfer->user_info->cur += add;
    global.limit.cur_downloads += add;
#ifdef TRANSFER_DEBUG  
    printf("down now %d\n", global.limit.cur_downloads);
#endif
  }
  if (transfer->type == T_UPLOAD) {
    if (transfer->user_info) transfer->user_info->cur += add;
    global.limit.cur_uploads += add;
#ifdef TRANSFER_DEBUG  
    printf("up now %d\n", global.limit.cur_uploads);
#endif
  }
    
  transfer->status = status;
  transfer_update(socket, 1);

  if ((add != 0) || (transfer->status == S_DOWNLOADING) ||
      (transfer->status == S_UPLOADING)) {
    time_t cur;
    char* str_time;
    
    time(&cur);
    str_time = strdup(ctime(&cur));
    str_time[strlen(str_time)-1] = 0;
    if (transfer->type == T_UPLOAD) {
      if (transfer->status == S_WAITING)
	log("uploads", LOG_OTHER, "[%s] Trying [%s] to [%s]%s\n",
	    str_time, transfer->shortname, transfer->user, 
	    transfer->is_dcc?" (DCC)":"");
      else if (transfer->status == S_UPLOADING)
	log("uploads", LOG_OTHER, "[%s] Uploading [%s] to [%s]%s\n",
	    str_time, transfer->shortname, transfer->user, 
	    transfer->is_dcc?" (DCC)":"");
      else
	log("uploads", LOG_OTHER, "[%s] Ending [%s] to [%s]%s :%s\n",
	    str_time, (transfer->shortname)?transfer->shortname:"???",
	    (transfer->user)?transfer->user:"???",
	    transfer->is_dcc?" (DCC)":"", status_names(transfer->status));
    } else if (transfer->type == T_UNKNOWN) {
      log("unknown_transfers", LOG_OTHER, "[%s] Ending [%s] [%s]%s\n",
	  str_time, transfer->shortname, transfer->user, 
	  transfer->is_dcc?" (DCC)":"");
    } else {
      if (transfer->status == S_WAITING)
	log("downloads", LOG_OTHER, "[%s] Trying [%s] from [%s]%s\n",
	    str_time, transfer->shortname, transfer->user, 
	    transfer->is_dcc?" (DCC)":"");
      else if (transfer->status == S_DOWNLOADING)
	log("downloads", LOG_OTHER, "[%s] %s [%s] from [%s]%s\n",
	    str_time, (transfer->type==T_DOWNLOAD)?"Downloading":"Resuming",
	    transfer->shortname, transfer->user, 
	    transfer->is_dcc?" (DCC)":"");
      else
	log("downloads", LOG_OTHER, "[%s] Ending [%s] from [%s]%s :%s\n",
	    str_time, (transfer->shortname)?transfer->shortname:"???",
	    (transfer->user)?transfer->user:"???", 
	    transfer->is_dcc?" (DCC)":"", status_names(transfer->status));
    }
    free(str_time);
  }
}

file_t* lib_search_transfer(transfer_t* transfer) {
  GList* dlist;
  file_t* file;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    if (!strcmp(transfer->winname, file->winname)) return file;
  }
  return NULL;
}

void transfer_connect_and_start(socket_t* socket) {
  transfer_t* transfer = (transfer_t*)(socket->data);
  
  transfer_status_set(socket, S_CONNECTING);
  if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
    send_command(CMD_CLIENT_DATA_PORT_ERROR, transfer->user);
    return;
  }
  
  socket->input = 
    gdk_input_add(socket->fd, GDK_INPUT_READ, 
		  GTK_SIGNAL_FUNC(await_conn_ack), socket);
}

int download_allowed(transfer_t* transfer) {
  if (global.limit.max_downloads <= global.limit.cur_downloads)
    return 0;
  if (transfer->user_info) {
    if ((transfer->user_info->max >= 0) &&
	(transfer->user_info->cur >= transfer->user_info->max))
      return 0;
  } else {
    g_warning("no user info found [%s]", transfer->user);
  }
  return 1;
}

int upload_allowed(transfer_t* transfer) {
  if (global.limit.max_uploads <= global.limit.cur_uploads)
    return 0;
  if (transfer->user_info) {
    if ((transfer->user_info->max >= 0) &&
	(transfer->user_info->cur >= transfer->user_info->max))
      return 0;
  } else {
    g_warning("no user info found [%s]", transfer->user);
  }
  return 1;
}

void download_start(socket_t* socket, int force) {
  char t2[2048] = "";
  transfer_t* transfer = (transfer_t*)(socket->data);
  resume_t* resume;

  // removing timeout if set
  if (socket->timeout >= 0) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
  }

#ifdef TRANSFER_DEBUG  
  printf("starting down [%s]\n", transfer->winname);
#endif
  transfer->hist_cnt = 0;
  socket->cnt = 0;

  resume = resume_list_search(transfer);
  if (!resume) {
    // no resume found, setting up new one
    //    printf("resume not found\n");
    resume = transfer_create_resume(transfer);
    if (!resume) {
      //      printf("could not create new resume\n");
      socket_end(socket, S_CANCELED);
      return;
    }
    resume_list_add(resume);
  }
  if (!transfer_grab_resume(transfer, resume)) {
    //    printf("could not grab resume\n");
    socket_end(socket, S_QUEUED);
    return;
  }

  // now starting the download
  if (transfer->is_dcc) {
    if (force || (download_allowed(transfer))) {
      transfer_connect_and_start(socket);
    } else {
      socket_end(socket, S_QUEUED);
    }   
  } else {
    if (force || (download_allowed(transfer))) {
      strcpy(t2, transfer->user);
      strcat(t2, " \"");
      strcat(t2, transfer->winname);
      strcat(t2, "\"");
      send_command(CMD_CLIENT_DOWNLOAD, t2);
      transfer_status_set(socket, S_WAITING);
    } else {
      socket_end(socket, S_QUEUED);
    }
  }
}

void upload_start(socket_t* socket) {
  char t2[2048] = "";
  char t[100];
  transfer_t* transfer = (transfer_t*)(socket->data);

#ifdef TRANSFER_DEBUG  
  printf("starting up [%s]\n", transfer->winname);
#endif
  if (transfer->is_dcc) {
    transfer_status_set(socket, S_WAITING);
  } else {
    if (upload_allowed(transfer)) {
      strcpy(t2, transfer->user);
      strcat(t2, " \"");
      strcat(t2, transfer->winname);
      strcat(t2, "\"");
      send_command(CMD_CLIENT_UPLOAD_OK, t2);
      
      transfer_status_set(socket, S_WAITING);
    } else {
      strcpy(t2, transfer->user);
      strcat(t2, " \"");
      strcat(t2, transfer->winname);
      strcat(t2, "\"");
      sprintf(t, " %d", global.limit.cur_uploads);
      strcat(t2, t);
      send_command(CMD_CLIENT_LIMIT, t2);
      socket_end(socket, S_REJECT);
    }
  }
}

char* extract_last_dir(char* filename) {
  static char dir[1024];
  char* pos1;
  char* pos2;

  pos1 = strchr(filename, '/');
  pos2 = strrchr(filename, '/');
  if (!pos1) return NULL;
  if (pos1 == pos2) return NULL;
  
  pos2[0] = 0;
  pos1 = strrchr(filename, '/')+1;
  strcpy(dir, pos1);
  pos2[0] = '/';
  
  return dir;
}

char* extract_filename(char* filename) {
  char* pos1;

  pos1 = strrchr(filename, '/');
  if (!pos1) return filename;
  else return (++pos1);
}

void convert_local_name(char* fname) {
  if (!fname) return;
  while (*fname) {
    if (*fname == '`') *fname = '_';
    if (*fname == '\'') *fname = '_';
    if (*fname == '') *fname = '_';
    fname++;
  }
}

void download_file(file_t* file, int withdir) {
  transfer_t* new_trans;
  char* dir;
  char* folder;
  socket_t* socket;

  if ((folder = valid_download_folder(file->mime_type)) == NULL) {
    download_dialog(file->shortname, file->mime_type);
    return;
  }
  
  new_trans = transfer_new();
  new_trans->longname = strdup(file->longname);
  new_trans->shortname = strdup(file->shortname);
  new_trans->winname = strdup(file->winname);
  new_trans->linespeed = file->linespeed;
  new_trans->user = strdup(file->user);
  new_trans->size = file->size;
  new_trans->type = T_DOWNLOAD;
  new_trans->mime_type = file->mime_type;
  socket = socket_new(S_TRANSFER);
  socket->data = new_trans;

  if (withdir) {
    dir = extract_last_dir(new_trans->longname);
    if (dir) {
      new_trans->download_dir =
	g_strdup_printf("%s/%s", folder, dir);
    } else
      new_trans->download_dir = strdup(folder);
  } else {
    new_trans->download_dir = strdup(folder);
  }
  
#ifdef TRANSFER_DEBUG
  printf("folder = [%s][%d]\n", new_trans->download_dir, 
	 file->mime_type);
#endif
  transfer_insert(socket);
  download_start(socket, FALSE);
}

socket_t* transfer_is_in_download(char* user, char* winname) {
  GtkCList* clist;
  int i1;
  socket_t* socket;
  transfer_t* transfer;
  
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) {
      socket = NULL;
      break;
    }
    socket = (socket_t*)gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t*)(socket->data);
    if ((strcmp(transfer->winname, winname) == 0) &&
	(strcmp(transfer->user, user) == 0)) {
      if ((transfer->status == S_WAITING) ||
	  (transfer->status == S_REMOTE))
	return socket;
    }
    i1++;
  }

  return NULL;
}

socket_t* transfer_is_in_upload(char* user, char* winname) {
  GtkCList* clist;
  int i1;
  socket_t* socket;
  transfer_t* transfer;
  
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) return NULL;
    socket = (socket_t*)gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t*)(socket->data);
    if ((strcmp(transfer->winname, winname) == 0) &&
	(strcmp(transfer->user, user) == 0)) {
      if ((transfer->status == S_WAITING) ||
	  (transfer->status == S_QUEUED))
	return socket;
    }
    i1++;
  }

  return NULL;
}

/*
  thanks to Matthew Pratt <mattpratt@yahoo.com> for this code
  I made a few changes....
*/
GdkPixmap *transfer_draw_progress(GtkCList *clist, GdkPixmap *pixmap, double percent, 
				  int width, int active){
  int height;
  char text[128];
  static GdkGC *gc = NULL;
  GdkColor background = { 0, 0xbf00, 0xbf00, 0xbf00 };
  GdkColor foreground = { 0, 0xbbbb, 0xffee, 0xbbbb };
  GdkColor foreground2 = { 0, 0xbbbb, 0xbbbb, 0xffee };
  GdkWindowPrivate* temp_pix;
  
  if (!gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), &background )
      || !gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), &foreground )
      || !gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), &foreground2 ) ) {
    g_error("couldn't allocate colour");
  }
  
  height = GTK_CLIST(clist)->row_height;
  temp_pix = (GdkWindowPrivate*)pixmap;

  if (!pixmap || (temp_pix->width != width) ||
      (temp_pix->height != height)) {
    if (pixmap) {
      // the next line would crash the client after a few
      // column resizes
      //      free(temp_pix);
      gdk_pixmap_unref (pixmap);
      temp_pix = NULL;
      pixmap = NULL;
    }
    pixmap = gdk_pixmap_new( global.win->window, width, 
			     GTK_CLIST(clist)->row_height, -1 );
  }
  
  if (!gc) gc = gdk_gc_new(global.win->window);

  // draw the upper/left black lines
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  gdk_draw_rectangle(pixmap, gc, TRUE, 0, 0, width, height);
  
  // draw the lower/right white lines
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->white_gc);
  gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, width-1, height-1);
  
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  // draw the background (unfilled progress)
  gdk_gc_set_foreground(gc, &background);
  gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, width - 2, height - 2);
  
  // draw the actual progress bar
  // draw the lower/right white lines
  if ((int)((width-2)*percent) > 0) {
    gdk_gc_copy(gc, GTK_WIDGET(clist)->style->white_gc);
    gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, (int)((width-2)*percent), height - 2);
  }

  // draw the upper/left black lines
  if ((int)((width-2)*percent)-1 > 0) {
    gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
    gdk_draw_rectangle(pixmap, gc, TRUE, 2, 2, (int)((width-2)*percent)-1, height - 3);
  }

  if ((int)((width-2)*percent)-2 > 0) {
    if (active) gdk_gc_set_foreground(gc, &foreground);
    else gdk_gc_set_foreground(gc, &foreground2);
    gdk_draw_rectangle(pixmap, gc, TRUE, 2, 2, (int)((width-2)*percent)-2, height - 4);
  }
  
  // draw the text on top
  g_snprintf( text, sizeof(text), "%.0f%%", percent*100);
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  gdk_draw_text(pixmap, GTK_WIDGET(clist)->style->font, gc, 
		width/2 - 9, height - 3,
		text, strlen(text));
  
  return pixmap;
}

int transfer_update(socket_t* socket, int full) {
  GtkCList* clist;
  int row;
  char str[200];
  double trate;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  int hour;
  int min;
  int sec;
  transfer_t* transfer;

  if (!socket) return 1;
  transfer = (transfer_t*)(socket->data);
  if (transfer->resume) resume_list_update(transfer->resume);

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  // incorrect
  row = gtk_clist_find_row_from_data(clist, socket);
  if (row < 0) {
    clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
    row = gtk_clist_find_row_from_data(clist, socket);
  }
  if (row < 0) {
    return 1;
  }

  if (full) {
    // filename
    if (transfer->type == T_DOWNLOAD) {
      if (transfer->shortname) strcpy(str, transfer->shortname);
      else strcpy(str, _("Not known yet"));
    } else if (transfer->type == T_UPLOAD) {
      if (transfer->shortname) strcpy(str, transfer->shortname);
      else strcpy(str, _("Not known yet"));
    }
    gtk_clist_set_text(clist, row, 0, str);

    // user
    if (transfer->user) strcpy(str, transfer->user);
    else strcpy(str, _("Not known yet"));
    gtk_clist_set_text(clist, row, 2, str);

    // linespeed
    strcpy(str, LineSpeed(transfer->linespeed));
    gtk_clist_set_text(clist, row, 4, str);
  }

  // filesize
  sprintf(str, "%ld/%ld", transfer->progress, transfer->size);
  gtk_clist_set_text(clist, row, 1, str);

  // status
  if ((transfer->status < 0) || (transfer->status >= S_NUMBER)) {
    g_warning("transfer_update: invalid status %d", 
	      transfer->status);
    return 1;
  }
  if (transfer->status == S_WAITING)
    sprintf(str, _("Waiting %d secs"), 
	    (socket->max_cnt - socket->cnt));
  else 
    strcpy(str, status_names(transfer->status));
  gtk_clist_set_text(clist, row, 3, str);

  // progress
  if ((transfer->status == S_DOWNLOADING) ||
      (transfer->status == S_FINISHED) ||
      (transfer->status == S_UPLOADING)) { 
    int last_prog;
    int first_prog;

    if (transfer->size >0){
      gtk_clist_get_pixmap(clist, row, 5, &pixmap, &bitmap );
      
      pixmap = 
	transfer_draw_progress(clist, pixmap, 
			       (double)transfer->progress / 
			       (double)transfer->size, 
			       clist->column[5].width,
			       transfer->status != S_FINISHED);
      gtk_clist_set_pixmap(clist, row, 5, pixmap, NULL );
    }
    
    transfer->hist_pos++;
    if (transfer->hist_pos >= TRANSFER_HISTORY_TIME)
      transfer->hist_pos = 0;
    transfer->history[transfer->hist_pos] = transfer->progress;
    if (transfer->hist_cnt < (TRANSFER_HISTORY_TIME/global.network.transfer_delay)-1)
      transfer->hist_cnt++;
    
    last_prog = transfer->hist_pos;
    first_prog = (last_prog+TRANSFER_HISTORY_TIME-transfer->hist_cnt)%
      TRANSFER_HISTORY_TIME;

    if (transfer->status == S_FINISHED) {
      if (transfer->end_time - transfer->start_time == 0) 
	transfer->end_time++;
      trate = transfer->size/
	(transfer->end_time - transfer->start_time);
    } else {
      trate = (double)(transfer->history[last_prog]-
		       transfer->history[first_prog])/
	transfer->hist_cnt/global.network.transfer_delay;
    }
#ifdef TRANSFER_DEBUG
    if (trate < 0) {
      printf("%d/%d/%d\n", first_prog, last_prog,
	     TRANSFER_HISTORY_TIME);
    }
#endif
    print_speed(str, (int)trate, 1);
  
    if (trate > 20*1024) {
      pixmap = global.pix.sgreen;
      bitmap = global.pix.sgreenb;
    } else if (trate > 4*1024) {
      pixmap = global.pix.syellow;
      bitmap = global.pix.syellowb;
    } else if (trate > 2*1024) {
      pixmap = global.pix.sred;
      bitmap = global.pix.sredb;
    } else if (trate > 0) {
      pixmap = global.pix.sgray;
      bitmap = global.pix.sgrayb;
    } else {
      pixmap = global.pix.dummy;
      bitmap = global.pix.dummyb;
    }
    gtk_clist_set_pixtext (clist, row, 6, str, 5, pixmap, bitmap);
    
    if ((transfer->status == S_FINISHED) || (trate > 0)) {
      if (transfer->status == S_FINISHED)
	sec = transfer->end_time - transfer->start_time;
      else
	sec = (int)((transfer->size-transfer->progress)/trate);
      min = sec/60;
      sec = sec%60;
      hour = min/60;
      min = min%60;
      *str = 0;
      if (hour) {
	sprintf(str, "%s%d:%s%d h", (hour>9)?"":"0", hour,
		(min>9)?"":"0", min);
      } else if (min) {
	sprintf(str, "%s%d:%s%d m", (min>9)?"":"0", min,
		(sec>9)?"":"0", sec);
      } else {
	sprintf(str, "%s%d s", (sec>9)?"":"0", sec);
      }
    } else {
      sprintf(str, _("stalled"));
    }
    gtk_clist_set_text(clist, row, 7, str);
  }
  
  return 1;
}

void socket_destroy2(socket_t* socket) {
  if (!socket) return;

#ifdef TRANSFER_DEBUG
  printf("destroying socket2 [%d]\n", socket->fd);
#endif
  global.sockets = g_list_remove(global.sockets, socket);
  socket_remove_clist(socket);

  if (socket->data)
    transfer_destroy(socket->data);
  free(socket);
}

void transfer_insert(socket_t *socket) {
  GtkCList* temp;
  int row;
  transfer_t* transfer = (transfer_t*)(socket->data);

  if (transfer->type == T_DOWNLOAD) {
    transfer->user_info = detect_user_info(transfer->user, 1);
    temp = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  } else {
    transfer->user_info = detect_user_info(transfer->user, 0);
    temp = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  }

  tstr[0][0] = tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 
    tstr[5][0] = tstr[6][0] = 0;
  row = gtk_clist_append(temp, list);
  gtk_clist_set_row_data_full (temp, row, (gpointer)socket, 
			       (GtkDestroyNotify)socket_destroy2);
  transfer_update(socket, 1);
}

void print_sockets() {
  GList* dlist;
  socket_t* socket;
  int cnt = 0;

  dlist = global.sockets;
  while (dlist) {
    socket = (socket_t*)(dlist->data);
    dlist = dlist->next;
    if (
	(socket->type == S_SHARE) ||
	(socket->type == S_BROWSE) ||
	(socket->type == S_HTTP) ||
	(socket->type == S_SERVER) ||
	(socket->type == S_DATA) ||
	(socket->type == S_TRANSFER) ||
	(socket->type == S_UNKNOWN) ||
	0
	)
      cnt++;
  }
  if (cnt) printf("--%d sockets------------------------\n", cnt);

  dlist = global.sockets;
  while (dlist) {
    socket = (socket_t*)(dlist->data);
    dlist = dlist->next;
    printf("[%8s]", socket_get_name(socket));
    switch (socket->type) {
    case S_SHARE:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_BROWSE:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      printf("[%s]", (char*)(socket->data));
      break;
    case S_HTTP:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_SERVER:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_DATA:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_TRANSFER:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    case S_UNKNOWN:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    default:
      printf("[%2d][%3d/%3d]", socket->fd, socket->cnt,
	     socket->max_cnt);
      break;
    }
    printf("\n");
  }
}

int global_timer(gpointer data) {
  char comm[2048];
  GtkWidget* temp;
  GList* dlist;
  GList* dlist2;
  resume_t* resume;
  socket_t* socket;
  transfer_t* transfer;
  user_t* userinfo;
  static int cnt = 0;

  cnt++;

  if (global.status.exiting == E_SAFE) {
    if (global.limit.cur_uploads + global.limit.cur_downloads == 0)
      global.status.exiting = E_NORMAL;
  } else if (global.status.exiting == E_NORMAL) {
    global_exit();
  }  

  //  if (0) print_sockets();

  if (global.socket_win) {
    for (dlist = global.sockets; dlist; dlist = dlist->next) {
      socket = (socket_t*)(dlist->data);
      socket_update_clist(socket);
    }
  }

  dlist = global.sockets;
  while (dlist) {
    socket = (socket_t*)(dlist->data);
    dlist2 = dlist->next;
    switch (socket->type) {
    case S_SHARE:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) socket_destroy(socket, 0);
      break;
    case S_BROWSE:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) {
	send_command(CMD_CLIENT_BROWSE, (char*)(socket->data));
	socket_destroy(socket, 0);
      }
      break;
    case S_HTTP:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) socket_destroy(socket, 0);
      break;
    case S_SERVER:
      if (socket->fd >= 0) {
	socket->cnt++;
	if (socket->cnt >= socket->max_cnt) {
	  socket->cnt = 0;
	  napster_disconnect("Server connection timed out");
	}
      }
      break;
    case S_DATA:
      break;
    case S_TRANSFER:
      transfer = (transfer_t*)(socket->data);
      if (!transfer) break;
      if (transfer_in_progress(transfer)) {
	socket->cnt++;
	if (socket->cnt >= socket->max_cnt) 
	  socket_end(socket, S_TIMEOUT);
	if ((cnt%global.network.transfer_delay) == 0)
	  transfer_update(socket, 0);
      }
      break;
    case S_UNKNOWN:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) socket_destroy(socket, 0);
      break;
    default:
      break;
    }
    dlist = dlist2;
  }    

  // checking browses timeout
  dlist = global.browses;
  while (dlist) {
    userinfo = (user_t*)(dlist->data);
    // removing browses after 30 min
    dlist = dlist->next;
    if (userinfo->timeout > 1800) {
      hotlist_remove_user_files(userinfo);
      global.browses = g_list_remove(global.browses, userinfo);
      free(userinfo);
    } else {
      userinfo->timeout++;
    }
  }
  
  // updating stats
  temp = lookup_widget(global.win, "label534");
  sprintf(comm, "%d", global.limit.cur_downloads);
  gtk_label_set_text(GTK_LABEL(temp), comm);
  temp = lookup_widget(global.win, "label547");
  sprintf(comm, "%d", global.limit.cur_uploads);
  gtk_label_set_text(GTK_LABEL(temp), comm);

  global.down_width.current = 
    (global.down_width.current*(cnt-1)+global.down_width.bytes)/cnt;
  global.up_width.current = 
    (global.up_width.current*(cnt-1)+global.up_width.bytes)/cnt;
  global.down_width.bytes = 0;
  global.up_width.bytes = 0;

  if (cnt%global.network.transfer_delay == 0) {
    draw_band_width(&global.down_width, 0);
    draw_band_width(&global.up_width, 0);
    cnt = 0;
    global.down_width.current = 0;
    global.up_width.current = 0;
  }

  downloads_enable();
  uploads_enable();
  
  if (!global.network.auto_resume) return 1;
  if (global.status.connection < 2) return 1;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    if (resume->status < R_SEARCH) {
      resume_search(resume);
    } else if (!resume->transfer) {
      resume_try_next_potential(resume);
    }
  }

  return 1;
}

FILE* open_download_file(transfer_t* transfer) {
  int cnt;
  struct stat st;
  resume_t* resume;
  int exist;
#ifdef TRANSFER_DEBUG
  int i1;
#endif

  if (!transfer->resume) {
#ifdef TRANSFER_DEBUG
    printf("ops, no resume\n");
#endif
    return NULL;
  }
  resume = transfer->resume;
  
  exist = (stat(resume->filename, &st) >= 0);
  if  (exist) {
#ifdef TRANSFER_DEBUG
    printf("resume file does exist! [%s]\n", resume->filename);
#endif
    transfer->file = fopen(resume->filename, "r");
    if (transfer->file == NULL) {
#ifdef TRANSFER_DEBUG
      printf("could not open incomplete file\n");
#endif
      return NULL;
    }
    
    cnt = st.st_size - transfer->progress;
    fseek(transfer->file, transfer->progress, SEEK_SET);
#ifdef TRANSFER_DEBUG
    printf("set offset to %ld %d\n", transfer->progress, cnt);
#endif
    transfer->resume_check = (unsigned char*)malloc(cnt*sizeof(char));
    transfer->check_length = read(fileno(transfer->file),
				  transfer->resume_check, cnt);
    //    printf("openfile: %d\n", transfer->check_length);
    if (transfer->check_length < 2) {
      printf("Warning: resume check length < 2 !!\n");
    }
    fclose(transfer->file);
    // load done
    
    transfer->file = fopen(resume->filename, "a");
  } else {
#ifdef TRANSFER_DEBUG
    printf("resume file does not exist! [%s]\n", resume->filename);
#endif
    resume->size = transfer->size;
    /*
    if (exist) {
#ifdef TRANSFER_DEBUG
      printf("renaming [%s]\n", resume->filename);
#endif
      strcpy(temp_file, resume->filename);
      pos = strrchr(temp_file, '.');
      if (!pos) {
	suffix = temp_file + strlen(temp_file);
      } else {
	*pos = 0;
	suffix = pos+1;
      }
      i1 = 1;
      do {
	free(resume->filename);
	resume->filename =
	  g_strdup_printf("%s_%d.%s", temp_file, i1, suffix);
#ifdef TRANSFER_DEBUG
	printf("renaming to [%s]\n", resume->filename);
#endif
	if (i1 > 0xffff) return NULL;
	i1++;
      } while (stat(resume->filename, &st) >= 0);
    }
    */
    cnt = 0;
    transfer->file = fopen(resume->filename, "w");
    transfer->progress = 0;
  }

  if (transfer->shortname) free(transfer->shortname);
  transfer->shortname =
    strdup(extract_short_name(resume->filename));

  if (transfer->file == NULL) {
    printf("could not open file for downloading [%s]\n",
	   resume->filename);
  }
  return transfer->file;
}

FILE* open_upload_file(transfer_t* transfer) {
  char* filename;
  struct stat st;

  filename = transfer->longname;
  stat(filename, &st);
  transfer->file = fopen(filename, "r");

  if (transfer->file != NULL) transfer->size = st.st_size;
  return transfer->file;
}

void download_end(transfer_t* transfer, int mode) {
  resume_t* resume;
  char command[2048];
  char* pos;
  int to_delete;
  int to_retry;

#ifdef TRANSFER_DEBUG
  printf("download_end, mode %d\n", mode);
#endif
  if (transfer->status == S_DOWNLOADING) {
    send_command(CMD_CLIENT_DOWNLOAD_END, "");
  }
  
  if (transfer->resume_check) {
    free(transfer->resume_check);
    transfer->resume_check = NULL;
    transfer->check_length = 0;
  }

  switch (mode) {
  case S_FINISHED:
    if (transfer->resume) {
      pos = extract_filename(transfer->resume->filename);
      
      create_dir(transfer->download_dir);
      if (strcmp(transfer->download_dir, global.incomplete_path)) {
	sprintf(command, "mv \"%s\" \"%s\"",
		transfer->resume->filename, transfer->download_dir);
	//	printf("%s\n", command);
	system(command);
	sprintf(command, "%s/%s", transfer->download_dir, pos);
	free(transfer->longname);
	transfer->longname = strdup(command);
	free(transfer->shortname);
	transfer->shortname = 
	  strdup(extract_short_name(transfer->longname));
      }
    } else {
      printf("ops\n");
    }
    break;
  case S_DELETE:
    break;
  case S_STOPPED:
  case S_QUEUED:
  case S_REMOTE:
  case S_TIMEOUT:
  case S_INCOMPLETE:
  case S_CANCELED:
  default:
    break;
  }

  resume = transfer->resume;
  // return if transfer is not linked to resume
  // (should only be if resume already deleted and transfer
  // not active)
  if (!resume) return;
  // return if resume is linked to antoher transfer
  if (resume->transfer && (resume->transfer != transfer)) return;
  
  to_retry = transfer_to_retry(mode);
  to_delete = transfer_to_delete(transfer, mode);

  if ((to_retry >= 0) ||
      (mode == S_STOPPED)) {
    // dont really unlink resume, only update status
#ifdef TRANSFER_DEBUG
    printf("retry %d %d\n", mode, transfer->status);
#endif
    transfer_ungrab_resume(transfer, 0);
  } else {
    // unlink resume
#ifdef TRANSFER_DEBUG
    printf("no retry %d\n", mode);
#endif
    transfer_ungrab_resume(transfer, 1);
    if (to_delete) {
#ifdef TRANSFER_DEBUG
      printf("to delete\n");
#endif
      resume_list_remove(resume, TRUE);
      transfer->resume = NULL;
    }
  }

}

void upload_end(transfer_t* transfer, int mode) {

#ifdef TRANSFER_DEBUG
  printf("upload_end, mode %d old %d\n", mode, transfer->status);
#endif
  if (transfer->status == S_UPLOADING) {
    send_command(CMD_CLIENT_UPLOAD_END, "");
  }
}

void transfer_end(transfer_t* transfer, int mode) {
  if (!transfer) return;
#ifdef TRANSFER_DEBUG
  printf("ending %d [%s]\n", mode, 
	 transfer->winname?transfer->winname:"_unknown_");
#endif

  time(&transfer->end_time);

  if (transfer->file) {
    fclose(transfer->file);
    transfer->file = NULL;
  }

  if (transfer->type == T_DOWNLOAD) {
    download_end(transfer, mode);
  } else if (transfer->type == T_UPLOAD) {
    upload_end(transfer, mode);
  }
}

gint await_conn_ack(gpointer data, gint source, 
		    GdkInputCondition condition) {
  int res;
  char c;
  socket_t* socket = (socket_t*)data;

  if (condition != GDK_INPUT_READ) {
    socket_end(socket, S_CONERROR);
    return 1;
  }

  gdk_input_remove(socket->input);

  switch (res = recv(source, &c, 1, 0)) {
  case -1:
#ifdef TRANSFER_DEBUG
    printf("rec1 error\n");
#endif
    socket_end(socket, S_CONERROR);
    return 1;
  case 0:
#ifdef TRANSFER_DEBUG
    printf("received nothing\n");
#endif
    socket_end(socket, S_CONERROR);
    return 1;
  default:
    break;
  }    

#ifdef TRANSFER_DEBUG
  printf("got [%c]\n", c);
#endif
  if (c != '1') {
    socket_end(socket, S_REJECT);
    return 1;
  }
  
  socket->input = 
    gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		  GTK_SIGNAL_FUNC(send_transfer_info), socket);

  return 1;
}

gint send_transfer_info(gpointer data, gint source, 
			GdkInputCondition condition) {
  char* temp_str = NULL;
  socket_t* socket;
  transfer_t* transfer;

  socket = (socket_t*)data;
  transfer = (transfer_t*)(socket->data);

  if (condition != GDK_INPUT_WRITE) {
    socket_end(socket, S_CONERROR);
    return 1;
  }

  gdk_input_remove(socket->input);

  if (transfer->type == T_DOWNLOAD) {
#ifdef TRANSFER_DEBUG
    printf("sending [GET]\n");
#endif
    send(source, "GET", 3, 0);
    temp_str = g_strdup_printf("%s \"%s\" %lu", 
			       SERVER->nick,
			       transfer->winname, 
			       transfer->progress);
    transfer->size = 0;
  } else if (transfer->type == T_UPLOAD) {
    send(source, "SEND", 4, 0);
#ifdef TRANSFER_DEBUG
    printf("sending [SEND]\n");
#endif
    temp_str = g_strdup_printf("%s \"%s\" %lu", 
			       SERVER->nick,
			       transfer->winname, transfer->size);
    transfer->progress = 0;
  }
  if (temp_str) {
#ifdef TRANSFER_DEBUG
    printf("sending [%s]\n", temp_str);
#endif
    send(source, temp_str, strlen(temp_str), 0);
    g_free(temp_str);
  } else {
    g_warning("Should not happen: transfer.c");
    return 1;
  }
  
  transfer_status_set(socket, S_INFO);
  
  socket->input = 
    gdk_input_add(socket->fd, GDK_INPUT_READ, 
		  GTK_SIGNAL_FUNC(get_transfer_info), socket);

  return 1;
}

gint get_transfer_info(gpointer data, gint source, 
		       GdkInputCondition condition) {
  socket_t* socket;
  transfer_t* transfer;
  unsigned char buffer[1024];
  int res;
  int cnt;

  socket = (socket_t*)data;
  transfer = (transfer_t*)(socket->data);

  if (condition != GDK_INPUT_READ) {
    socket_end(socket, S_CONERROR);
    return 1;
  }

  gdk_input_remove(socket->input);
  
  switch (res = recv(source, buffer, 1024, MSG_PEEK)) {
  case -1:
    socket_end(socket, S_CONERROR);
    return 1;
  case 0:
    socket_end(socket, S_CONERROR);
    return 1;
  default:
    break;
  }
  
  buffer[res] = 0;
#ifdef TRANSFER_DEBUG
  printf("read [%s] [%d]\n", buffer, res);
#endif
  cnt = 0;
  while (isdigit(buffer[cnt])) cnt++;

  if (recv(source, buffer, cnt, 0) != cnt) {
    socket_end(socket, S_CONERROR);
    return 1;
  }
  buffer[cnt] = 0;

  if (transfer->type == T_UPLOAD) {
    transfer->progress = strtoul(buffer, NULL, 10);
#ifdef TRANSFER_DEBUG
    printf("got progress %ld\n", transfer->progress);
#endif
  } else {
    transfer->size = strtoul(buffer, NULL, 10);
    if (transfer->size == 0) {
      socket_end(socket, S_REJECT);
      return 1;
    }
#ifdef TRANSFER_DEBUG
    printf("got size %ld\n", transfer->size);
#endif
  }

  time(&(transfer->start_time));
  if (transfer->type == T_UPLOAD) {
    if (!open_upload_file(transfer)) {
      send(socket->fd, "FILE NOT FOUND", 
	   strlen("FILE NOT FOUND"), 0);
      socket_end(socket, S_IO);
      return 1;
    }
    transfer_status_set(socket, S_UPLOADING);
    send_command(CMD_CLIENT_UPLOAD_START, "");
    socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		    GTK_SIGNAL_FUNC(upload_push_output), socket);
  } else {
    if (!open_download_file(transfer)) {
      socket_end(socket, S_IO);
      return 1;
    }
    transfer_status_set(socket, S_DOWNLOADING);
    transfer->type = T_DOWNLOAD;
    send_command(CMD_CLIENT_DOWNLOAD_START, "");
    socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(download_get_input), socket);
  }

  return 1;
}

gint get_upload_info(gpointer data, gint source, 
		     GdkInputCondition condition) {
  char buffer[1024];
  int cnt;
  socket_t* socket = (socket_t*)data;
  socket_t* socket2;
  char* user;
  char* filename;
  char* progress;
  file_t* file;
  transfer_t* transfer;
  struct stat st;
  char* temp_str;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return 1;
  }

  gdk_input_remove(socket->input);
  cnt = 0;
  switch (cnt = recv(source, buffer, 1024, 0)) {
  case -1:
    socket_destroy(socket, 0);
    return 1;
  case 0:
    socket_destroy(socket, 0);
    return 1;
  default:
    break;
  }
  
  buffer[cnt] = 0;
#ifdef TRANSFER_DEBUG
  printf("upload_info [%s]\n", buffer);
#endif
  
  user = arg(buffer, 0);
  filename = arg(NULL, 0);
  progress = arg(NULL, 0);
#ifdef TRANSFER_DEBUG
  printf("up [%s][%s][%s]\n", user, filename, progress);
#endif
  socket2 = transfer_is_in_upload(user, filename);
  if (socket2) {
    transfer = (transfer_t*)(socket2->data);
#ifdef TRANSFER_DEBUG
    printf("transfer: type [%d]\n", transfer->type);
#endif
    socket2->fd = socket->fd;
    socket->fd = -1;
    socket2->input = socket->input;
    socket->input = -1;
    socket2->timeout = socket->timeout;
    socket->timeout = -1;
    socket_destroy(socket, 1);
    socket = socket2;
    transfer->status = S_INFO;
    transfer->progress = strtoul(progress, NULL, 10);
  } else {
    log("uploads", LOG_OTHER,
	"alert (incoming) %s %s\n", user, filename);
    g_warning("transfer [%s] not found, already deleted?", filename);
    socket_destroy(socket, 1);
    return 1;
  }
  
  file = lib_search_transfer(transfer);
  if (stat(transfer->longname, &st) < 0) {
#ifdef TRANSFER_DEBUG
    printf("**sending FILE NOT FOUND\n");
#endif
    //    send(transfer->sock->fd, "0", 1,0);
    send(socket->fd, "FILE NOT FOUND", 
	 strlen("FILE NOT FOUND"), 0);
    socket_end(socket, S_IO);
    return 1;
  } else if (!file && !transfer->is_dcc) {
#ifdef TRANSFER_DEBUG
    printf("**sending FILE NOT SHARED\n");
#endif
    send(socket->fd, "FILE NOT SHARED", 
	 strlen("FILE NOT SHARED"), 0);
    socket_end(socket, S_UNAVAILABLE);
    return 1;
  } else if (file) {
    transfer->size = st.st_size;
    transfer->mime_type = file->mime_type;
  } else {
    transfer->size = st.st_size;
  }

  temp_str = g_strdup_printf("%lu", transfer->size);
#ifdef TRANSFER_DEBUG
  printf("sending size %s\n", temp_str);
#endif
  send(socket->fd, temp_str, strlen(temp_str), 0);
  g_free(temp_str);
  
  time(&(transfer->start_time));
  if (!open_upload_file(transfer)) {
    send(socket->fd, "FILE NOT FOUND", 
	 strlen("FILE NOT FOUND"), 0);
    socket_end(socket, S_IO);
    return 1;
  }
  transfer_status_set(socket, S_UPLOADING);
  send_command(CMD_CLIENT_UPLOAD_START, "");
  socket->input =
    gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		  GTK_SIGNAL_FUNC(upload_push_output), socket);
  return 1;
}

gint get_download_info(gpointer data, gint source, 
		       GdkInputCondition condition) {
  char buffer[1024];
  int cnt;
  socket_t* socket = (socket_t*)data;
  socket_t* socket2;
  char* user;
  char* filename;
  char* size;
  transfer_t* transfer;
  char* temp_str;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return 1;
  }

  gdk_input_remove(socket->input);

  cnt = 0;
  switch (cnt = recv(source, buffer, 1024, 0)) {
  case -1:
    socket_destroy(socket, 0);
    return 1;
  case 0:
    socket_destroy(socket, 0);
    return 1;
  default:
    break;
  }
  
  buffer[cnt] = 0;
#ifdef TRANSFER_DEBUG
  printf("download_info [%s]\n", buffer);
#endif
  
  user = arg(buffer, 0);
  filename = arg(NULL, 0);
  size = arg(NULL, 0);
#ifdef TRANSFER_DEBUG
  printf("down [%s][%s][%s]\n", user, filename, size);
#endif
  socket2 = transfer_is_in_download(user, filename);
  if (socket2) {
    transfer = (transfer_t*)(socket2->data);
    socket2->fd = socket->fd;
    socket->fd = -1;
    socket2->input = socket->input;
    socket->input = -1;
    socket2->timeout = socket->timeout;
    socket->timeout = -1;
    socket_destroy(socket, 1);
    socket = socket2;
    transfer->status = S_INFO;
    transfer->size = strtoul(size, NULL, 10);
    if (transfer->size == 0) {
      socket_end(socket, S_UNAVAILABLE);
      return 1;
    }
  } else {    
    log("downloads", LOG_OTHER,
	"alert (incoming) %s %s\n", user, filename);
    g_warning("transfer [%s] not found, already deleted?", filename);
    socket_destroy(socket, 1);
    return 1;
  }
  
  temp_str = g_strdup_printf("%lu", transfer->progress);
#ifdef TRANSFER_DEBUG
  printf("sending offset %s\n", temp_str);
#endif
  send(socket->fd, temp_str, strlen(temp_str), 0);
  g_free(temp_str);
      
  time(&(transfer->start_time));
  if (!open_download_file(transfer)) {
    socket_end(socket, S_IO);
    return 1;
  }

  transfer_status_set(socket, S_DOWNLOADING);
  send_command(CMD_CLIENT_DOWNLOAD_START, "");
  socket->input =
    gdk_input_add(socket->fd, GDK_INPUT_READ, 
		  GTK_SIGNAL_FUNC(download_get_input), socket);
  return 1;
}

gint upload_push_output(gpointer data, gint source, 
			GdkInputCondition condition) {
  socket_t* socket = (socket_t*)data;
  transfer_t* transfer = (transfer_t*)(socket->data);
  char buf[2048 + 1];
  int n, sent;
  long maxpush;

  if (condition != GDK_INPUT_WRITE) {
    socket_end(socket, S_INCOMPLETE);
    return 1;
  }
  if (global.up_width.bytes >= global.up_width.limit) {
    uploads_disable();
    return 1;
  }
  maxpush = global.up_width.limit-global.up_width.bytes;
  if (maxpush > 1024) maxpush = 1024;


  if (transfer->progress < transfer->size) {
    lseek(fileno(transfer->file), transfer->progress, SEEK_SET);
    n = read(fileno(transfer->file), buf, maxpush);
    if (n <= 0) {
#ifdef TRANSFER_DEBUG
      printf("read <= 0\n");
#endif
      socket_end(socket, S_IO);
      return 1;
    }
    sent = send(source, buf, n, 0);
    if (sent <= 0) {
#ifdef TRANSFER_DEBUG
      printf("sent error\n");
#endif
      socket_end(socket, S_INCOMPLETE);
    } else {
      socket->cnt = 0;
      transfer->progress += sent;
      global.up_width.bytes += sent;
    }
  } else {
    socket_end(socket, S_FINISHED);
  }

  return 1;
}

gint download_get_input(gpointer data, gint source, GdkInputCondition condition) {
  char buf[2048+1];
  int n, n2;
  socket_t* socket = (socket_t*)data;
  transfer_t* transfer = (transfer_t*)(socket->data);
  char* cmp_buf;
#ifdef TRANSFER_DEBUG
  int cnt;
#endif
  long maxget;

  if (condition != GDK_INPUT_READ) {
    socket_end(socket, S_INCOMPLETE);
    return 1;
  }

  if (global.down_width.bytes >= global.down_width.limit) {
    downloads_disable();
    return 1;
  }
  maxget = global.down_width.limit-global.down_width.bytes;
  if (maxget > 2048) maxget = 2048;

  //  if (global.down_width.current > global.down_width.limit) return 1;
  bzero(buf, sizeof(buf));
  n = recv(source, buf, maxget, 0);
  if (n <= 0) {
    if (transfer->progress >= transfer->size) {
      socket_end(socket, S_FINISHED);
    } else {
      socket_end(socket, S_INCOMPLETE);
    }
    return 0;
  } else if (n > 0) {
    global.down_width.bytes += n;
    socket->cnt = 0;
    // resume check
    if (transfer->check_length > 0) {
      if (transfer->check_length+1 > n) {
	// assuming that this is FN behavier with one bytes
	// sent before file data, just ignoring it.
#ifdef TRANSFER_DEBUG
	printf("did not read enough bytes [%p][%d][%d]\n", 
	       transfer, n, transfer->check_length);
#endif
	return 0;
      }
#ifdef TRANSFER_DEBUG
      printf("checking %d bytes\n", transfer->check_length);
#endif
      if (buf[0] == 'A') {cmp_buf = buf+1; n--;}
      else cmp_buf = buf;
      
      if (memcmp(cmp_buf, transfer->resume_check, 
		 transfer->check_length)) {
#ifdef TRANSFER_DEBUG
	printf("resume check failed, ending download\n");
	for (cnt = 0; cnt < transfer->check_length; cnt++)
	  printf("%4d %4d\n", buf[cnt], transfer->resume_check[cnt]);
#endif
	socket_end(socket, S_RESUME_ERR);
	return 0;
      }
      if ((n2 = fwrite(cmp_buf+transfer->check_length,
		       n-transfer->check_length,
		       sizeof(char), transfer->file)) <= 0) {
	socket_end(socket, S_IO);
	return 0;
      }

      transfer->check_length = 0;
      free(transfer->resume_check);
      transfer->resume_check = NULL;
    } else {
      if ((transfer->progress == 0) &&
	  (buf[0] == 'A')) {
	cmp_buf = buf+1; n--;
      } else cmp_buf = buf;
      if ((n2 = fwrite(cmp_buf, n, sizeof(char), transfer->file)) <= 0) {
	socket_end(socket, S_IO);
	return 0;
      }
    }
    transfer->progress += n;
    if (transfer->progress >= transfer->size) {
      socket_end(socket, S_FINISHED);
    }
  }
  return 0;
}

void transfer_destroy(transfer_t* transfer) {
  if (!transfer) return;
#ifdef TRANSFER_DEBUG
  printf("detroying transfer %p\n", transfer);
#endif
  if (transfer->longname) free(transfer->longname);
  if (transfer->shortname) free(transfer->shortname);
  if (transfer->winname) free(transfer->winname);
  if (transfer->download_dir) free(transfer->download_dir);
  if (transfer->md5) free(transfer->md5);
  if (transfer->user) free(transfer->user);
  if (transfer->user_info) {
    transfer->user_info->cnt--;
    if (transfer->user_info->cnt == 0) {
      free(transfer->user_info);
    }
  }
  free(transfer);
}

/*
int transfer_timeout_cb(gpointer data) {
  transfer_t* transfer;

  transfer = (transfer_t*)data;
  if (transfer->sock->cnt >= transfer->sock->max_cnt) {
    socket_end(transfer, S_TIMEOUT);
    return 1;
  }
  transfer->sock->cnt++;
  transfer_update(transfer);

  return 1;
}
*/

int transfer_remote_timeout(gpointer data) {
  socket_t* socket;
  transfer_t* transfer;

  socket = (socket_t*)data;
  transfer = (transfer_t*)(socket->data);

  if (download_allowed(transfer)) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
    download_start(socket, FALSE);
  } else if (transfer->status != S_QUEUED) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
    socket_end(socket, S_QUEUED);
  }

  return 1;
}

user_info_t* detect_user_info(char* user, int in_download) {
  GtkCList* clist;
  int i1;
  transfer_t* transfer;
  user_info_t* user_info;
  socket_t* socket;

  if (in_download)
    clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  else
    clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    socket = (socket_t*)gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t*)(socket->data);
    if (strcmp(transfer->user, user) == 0) {
      user_info = transfer->user_info;
      user_info->cnt++;
      return user_info;
    }
    i1++;
  }

  user_info = (user_info_t*)malloc(sizeof(user_info_t));
  user_info->cur = 0;
  user_info->cnt = 1;
  if (in_download)
    user_info->max = global.limit.default_downloads;
  else
    user_info->max = global.limit.default_uploads;
  return user_info;
}

void change_user_info(char* user, int max) {
  GtkCList* clist;
  int i1;
  socket_t* socket;
  transfer_t* transfer;
  GtkWidget* temp;
  char* text;

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    socket = (socket_t*)gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t*)(socket->data);
    if (!transfer->user_info) continue;
    if (!strcmp(transfer->user, user)) {
      if (max < -1) {
	transfer->user_info->max = transfer->user_info->cur-1;
	if (transfer->user_info->max == 0)
	  transfer->user_info->max = 1;
      } else
	transfer->user_info->max = max;
      temp = lookup_widget(global.win, "label422");
      text = GTK_LABEL(temp)->label;
      if (!strcmp(transfer->user, text)) {
	temp = lookup_widget(global.win, "spinbutton15");
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp),
				  (gfloat)(transfer->user_info->max));
      }
      return;
    }
    i1++;
  }
  return;
}

void downloads_disable() {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer) continue;
    if (transfer->type != T_DOWNLOAD) continue;
    if (transfer->status != S_DOWNLOADING) continue;
    if (socket->input >= 0) {
      gdk_input_remove(socket->input);
      socket->input = -1;
    }
  }
}

void downloads_enable() {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer) continue;
    if (transfer->type != T_DOWNLOAD) continue;
    if (transfer->status != S_DOWNLOADING) continue;
    if (socket->input == -1) {
      socket->input =
	gdk_input_add(socket->fd, GDK_INPUT_READ, 
		      GTK_SIGNAL_FUNC(download_get_input), socket);
    }
  }
}

void uploads_disable() {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer || (transfer->type != T_UPLOAD) ||
	(transfer->status != S_UPLOADING)) continue;
    if (socket->input >= 0) {
      gdk_input_remove(socket->input);
      socket->input = -1;
    }
  }
}

void uploads_enable() {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t*)(dlist->data);
    if (socket->type != S_TRANSFER) continue;
    transfer = (transfer_t*)(socket->data);
    if (!transfer || (transfer->type != T_UPLOAD) ||
	(transfer->status != S_UPLOADING)) continue;
    if (socket->input == -1) {
      socket->input =
	gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		      GTK_SIGNAL_FUNC(upload_push_output), socket);
    }
  }
}

void transfer_clist_remove(socket_t* socket) {
  GtkCList* clist;
  socket_t* s;
  int i1;

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    s = (socket_t*)gtk_clist_get_row_data(clist, i1);
    if (s == socket) {
      gtk_clist_remove(clist, i1);
      return;
    }
    i1++;
  }
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    s = (socket_t*)gtk_clist_get_row_data(clist, i1);
    if (s == socket) {
      gtk_clist_remove(clist, i1);
      return;
    }
    i1++;
  }

  return;
}

int transfer_to_destroy(socket_t* socket, int mode) {
  transfer_t* transfer;

  if (socket->timeout != -1) return 0;

  transfer = (transfer_t*)(socket->data);
  if (transfer->type == T_DOWNLOAD) {
    switch (mode) {
    case S_CANCELED:
      if (global.options.dl_autoremove & REMOVE_D_CANCELED)
	return 1;
      break;
    case S_FINISHED:
      if (global.options.dl_autoremove & REMOVE_D_FINISHED)
	return 1;
      break;
    case S_TIMEOUT:
      if (global.options.dl_autoremove & REMOVE_D_TIMEOUT)
	return 1;
      break;
    case S_REJECT:
      if (global.options.dl_autoremove & REMOVE_D_REJECT)
	return 1;
      break;
    case S_INCOMPLETE:
      if (global.options.dl_autoremove & REMOVE_D_INCOMPLETE)
	return 1;
      break;
    case S_FIREWALL:
      if (global.options.dl_autoremove & REMOVE_D_FIREWALL)
	return 1;
      break;
    case S_CONERROR:
      if (global.options.dl_autoremove & REMOVE_D_CONERROR)
	return 1;
      break;
    case S_REMOTE:
      if (global.options.dl_autoremove & REMOVE_D_REMOTE)
	return 1;
      break;
    case S_UNAVAILABLE:
      if (global.options.dl_autoremove & REMOVE_D_UNAVAILABLE)
	return 1;
      break;
    case S_RESUME_ERR:
      if (global.options.dl_autoremove & REMOVE_D_RESUME_ERR)
	return 1;
      break;
    case S_DELETE:
      if (global.options.dl_autoremove & REMOVE_D_DELETE)
	return 1;
      break;
    case S_IO:
      if (global.options.dl_autoremove & REMOVE_D_IO)
	return 1;
      break;
    }
    return 0;
  } else {
    switch (mode) {
    case S_FINISHED:
      if (global.options.ul_autoremove & REMOVE_U_FINISHED)
	return 1;
      break;
    case S_TIMEOUT:
    case S_REJECT:
    case S_INCOMPLETE:
    case S_FIREWALL:
    case S_CONERROR:
    case S_REMOTE:
    case S_UNAVAILABLE:
    case S_RESUME_ERR:
    case S_DELETE:
    case S_IO:
    case S_CANCELED:
      if (global.options.ul_autoremove & REMOVE_U_ABORTED)
	return 1;
      break;
    }
    return 0;
  }
}

int transfer_to_retry(int mode) {
  switch (mode) {
  case S_TIMEOUT:
    if (global.options.auto_retry & RETRY_TIMEOUT) 
      return 10000;
    break;
  case S_REJECT:
    if (global.options.auto_retry & RETRY_REJECT) 
      return 10000;
    break;
  case S_INCOMPLETE:
    if (global.options.auto_retry & RETRY_INCOMPLETE) 
      return 10000;
    break;
  case S_CONERROR:
    if (global.options.auto_retry & RETRY_CONERROR) 
      return 10000;
    break;
  case S_REMOTE:
    if (global.options.auto_retry & RETRY_REMOTE) 
      return 60000;
    break;
  case S_QUEUED:
    return 1000;
    break;
  }

  return -1;
}

int transfer_to_delete(transfer_t* transfer, int mode) {
  if (mode == S_DELETE) return 1;
  if (mode == S_FINISHED) return 1;

  if ((transfer->progress > 0) &&
      global.options.auto_delete & DELETE_INCOMPLETE)
    return 1;
  if ((transfer->progress == 0) &&
      global.options.auto_delete & DELETE_UNSTARTED)
    return 1;

  return 0;
}

void on_toggle_dl_remove(GtkMenuItem     *menuitem,
			 gpointer         user_data) {
  int no = (int)user_data;

  global.options.dl_autoremove = 
    global.options.dl_autoremove ^ (1<<no);
}

GtkWidget* create_dl_remove_popup (int val) {
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (mode_popup), "mode_popup", mode_popup);
  mode_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (mode_popup));

  mode = gtk_check_menu_item_new_with_label (_("Canceled"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<0))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)0);

  mode = gtk_check_menu_item_new_with_label (_("Finished"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<1))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)1);

  mode = gtk_check_menu_item_new_with_label (_("Timed out"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<2))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)2);

  mode = gtk_check_menu_item_new_with_label (_("Rejected"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<3))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)3);

  mode = gtk_check_menu_item_new_with_label (_("Incomplete"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<4))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)4);

  mode = gtk_check_menu_item_new_with_label (_("Firewalled"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<5))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)5);

  mode = gtk_check_menu_item_new_with_label (_("Connection error"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<6))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)6);

  mode = gtk_check_menu_item_new_with_label (_("Remotely queued"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<7))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)7);

  mode = gtk_check_menu_item_new_with_label (_("Unavailable"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<8))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)8);

  mode = gtk_check_menu_item_new_with_label (_("Resume Error"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<9))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)9);

  mode = gtk_check_menu_item_new_with_label (_("Deleted"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<10))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)10);

  mode = gtk_check_menu_item_new_with_label (_("IO error"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<11))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_dl_remove,
		      (gpointer)11);

  return mode_popup;
}

void on_toggle_autoretry(GtkMenuItem     *menuitem,
			 gpointer         user_data) {
  int no = (int)user_data;

  global.options.auto_retry = 
    global.options.auto_retry ^ (1<<no);
}

GtkWidget* create_retry_popup (int val) {
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (mode_popup), "mode_popup", mode_popup);
  mode_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (mode_popup));

  mode = gtk_check_menu_item_new_with_label (_("Timed out"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<0))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)0);

  mode = gtk_check_menu_item_new_with_label (_("Rejected"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<1))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)1);

  mode = gtk_check_menu_item_new_with_label (_("Incomplete"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<2))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)2);

  mode = gtk_check_menu_item_new_with_label (_("Connection error"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<3))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)3);

  mode = gtk_check_menu_item_new_with_label (_("Remotely queued"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<4))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autoretry,
		      (gpointer)4);

  return mode_popup;
}

void on_toggle_ul_remove(GtkMenuItem     *menuitem,
			 gpointer         user_data) {
  int no = (int)user_data;

  global.options.ul_autoremove = 
    global.options.ul_autoremove ^ (1<<no);
}

GtkWidget* create_ul_remove_popup (int val) {
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (mode_popup), "mode_popup", mode_popup);
  mode_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (mode_popup));

  mode = gtk_check_menu_item_new_with_label (_("Finished"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<0))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_ul_remove,
		      (gpointer)0);

  mode = gtk_check_menu_item_new_with_label (_("Aborted"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<1))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_ul_remove,
		      (gpointer)1);

  return mode_popup;
}

void on_toggle_autodelete(GtkMenuItem     *menuitem,
			  gpointer         user_data) {
  int no = (int)user_data;

  global.options.auto_delete = 
    global.options.auto_delete ^ (1<<no);
}

GtkWidget* create_autodelete_popup (int val) {
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (mode_popup), "mode_popup", mode_popup);
  mode_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (mode_popup));

  mode = gtk_check_menu_item_new_with_label (_("Unstarted"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<0))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autodelete,
		      (gpointer)0);

  mode = gtk_check_menu_item_new_with_label (_("Incomplete"));
  gtk_widget_show (mode);
  gtk_container_add (GTK_CONTAINER (mode_popup), mode);
  if (val & (1<<1))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mode), TRUE);
  gtk_signal_connect (GTK_OBJECT (mode), "activate",
		      (GtkSignalFunc)on_toggle_autodelete,
		      (gpointer)1);

  return mode_popup;
}

int transfer_grab_resume(transfer_t* transfer, resume_t* resume) {
  struct stat st;
  int i1;

#ifdef TRANSFER_DEBUG
  printf("t:%p grabbing r:%p\n", transfer, resume);
#endif
  if (resume->transfer && (transfer != resume->transfer)) {
#ifdef TRANSFER_DEBUG
    printf("already grabbed\n");
#endif
    return 0;
  }

  resume->transfer = transfer;
  transfer->resume = resume;
  resume_user_flag(resume, transfer->user, FILE_DOWNLOAD);
  
  if (transfer->download_dir) free(transfer->download_dir);
  transfer->download_dir = strdup(resume->dirname);
  if (stat(resume->filename, &st) < 0) {
    transfer->progress = 0;
  } else {
    transfer->progress = st.st_size-100;
    if (transfer->progress < 0) transfer->progress = 0;
    for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
      transfer->history[i1] = transfer->progress;
  }
  resume->status = R_DOWNLOAD;
  resume_list_update(resume);

  return 1;
}

resume_t* transfer_create_resume(transfer_t* transfer) {
  char* text;
  resume_t* resume;
  char* filename;
  struct stat st;

  text = strrchr(transfer->longname, '/');
  if (!text) text = transfer->longname;
  else text++;
  
  if (resume_list_search(transfer)) {
    g_warning(_("File already exists already as an incomplete file"));
    return NULL;
  }
  filename = g_strdup_printf("%s/%s", global.incomplete_path, text);
  if (stat(filename, &st) >= 0) {
    g_warning(_("File already exists"));
    free(filename);
    return NULL;
  }

  resume = (resume_t*)malloc(sizeof(resume_t));
  resume->transfer = NULL;
  resume->filename = filename;
  convert_local_name(resume->filename);
  resume->size = transfer->size;
  resume->dirname = strdup(transfer->download_dir);
  resume->search = NULL;
  resume->status = R_NONE;

  return resume;
}

void transfer_ungrab_resume(transfer_t* transfer, int mode) {
#ifdef TRANSFER_DEBUG
  printf("t:%p ungrabbing r:%p\n", transfer, transfer->resume);
#endif
  if (!transfer->resume) {
#ifdef TRANSFER_DEBUG
    printf("nothing grabbed\n");
#endif
    return;
  }
  switch (mode) {
  case 0:
#ifdef TRANSFER_DEBUG
    printf("queueing\n");
#endif
    transfer->resume->status = R_QUEUED;
    resume_list_update(transfer->resume);
    break;
  case 1:
#ifdef TRANSFER_DEBUG
    printf("ungrabbing\n");
#endif
    if (transfer->resume->search)
      transfer->resume->status = R_SEARCH;
    else
      transfer->resume->status = R_NONE;
    resume_list_update(transfer->resume);
    resume_user_flag(transfer->resume, transfer->user, FILE_ERROR);
    transfer->resume->transfer = NULL;
  }
}
