#include <ctype.h>
#include <strings.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <stdlib.h>
#include <gtk/gtk.h>

#include "lopster.h"
#include "callbacks.h"
#include "connection.h"
#include "global.h"
#include "search.h"
#include "share.h"
#include "hotlist.h"
#include "handler.h"
#include "support.h"
#include "transfer.h"
#include "resume.h"

typedef int (*IntFunc)(char* str);

#define WORD_CHAR(c) \
        (isalnum((unsigned char)(c))||(c)=='\''||(unsigned char)(c) > 128)

search_pattern_t* search_pattern_create(char* name);
search_pattern_t* search_pattern_new(char* name);
void search_pattern_destroy(search_pattern_t* search);
void search_pattern_setup_list();
void search_pattern_update(search_pattern_t* search);
search_pattern_t* search_pattern_copy(search_pattern_t* search);

char* Destination(int id) {
  switch (id) {
  case DEST_NAPSTER: return _("Napster");
  case DEST_HOTLIST: return _("Hotlist");
  case DEST_LIBRARY: return _("Library");
  case DEST_SEARCH: return _("Results");
  default: return _("Unknown");
  }
  return "??";
}

char* get_real_mediatype(char* trans) {
  if (!strcasecmp(trans, _("mp3"))) return strdup("mp3");
  if (!strcasecmp(trans, _("audio"))) return strdup("audio");
  if (!strcasecmp(trans, _("video"))) return strdup("video");
  if (!strcasecmp(trans, _("application"))) return strdup("application");
  if (!strcasecmp(trans, _("image"))) return strdup("image");
  if (!strcasecmp(trans, _("text"))) return strdup("text");
  if (!strcasecmp(trans, _("any"))) return strdup("any");
  return strdup("mp3");
}

char* get_real_speed(char* trans) {
  if (!strcasecmp(trans, _("not specified"))) return NULL;
  if (!strcasecmp(trans, _("unknown"))) return strdup("unknown");
  else return strdup(trans);
}

char* get_real_entry(char* trans) {
  if (!strcasecmp(trans, _("not specified"))) return NULL;
  else return strdup(trans);
}

char* Destination_long(int id) {
  switch (id) {
  case DEST_NAPSTER: return _("Search on Napster");
  case DEST_HOTLIST: return _("Search in Hotlist");
  case DEST_LIBRARY: return _("Search in Library");
  case DEST_SEARCH: return _("Search in Results");
  default: return _("Unknown");
  }
  return "??";
}

int destination2int(char* dest) {
  if (!strcmp(_("Search on Napster"), dest)) return DEST_NAPSTER;
  else if (!strcmp(_("Search in Library"), dest)) return DEST_LIBRARY;
  else if (!strcmp(_("Search in Hotlist"), dest)) return DEST_HOTLIST;
  else if (!strcmp(_("Search in Results"), dest)) return DEST_SEARCH;
  else return DEST_NAPSTER;
}

char* search_arg(char* data) {
  static char* string;
  char* pos;

  if (data) string = data;
  if (!string) return NULL;
  
  while (*string && !WORD_CHAR(*string)) string++;

  pos = string;
  while (WORD_CHAR(*string)) {
    string++;
  }
  if (pos == string) {
    string = NULL;
    return NULL;
  } else {
    if (*string != 0)
      *string++ = 0;
    return pos;
  }
}

void search_create_page(search_t* search) {
  GtkWidget *scrolledwindow;
  GtkWidget *search_list;
  GtkWidget *label;
  search_pattern_t* pattern;
  GtkWidget* tab_label;
  char* str;
  GtkNotebook* notebook;
  
  pattern = search->pattern;
  if (!pattern) {
    printf("no pattern in search\n");
    return;
  }

  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolledwindow);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  search->link = (void*)scrolledwindow;
  
  search_list = gtk_clist_new (6);
  gtk_widget_show (search_list);

  gtk_clist_set_sort_column(GTK_CLIST(search_list), 0);
  gtk_clist_set_auto_sort(GTK_CLIST(search_list), TRUE);
  gtk_clist_set_compare_func(GTK_CLIST(search_list), search_compare);

  gtk_container_add (GTK_CONTAINER (scrolledwindow), search_list);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 0, 
			      global.search_width[0]);
  gtk_clist_set_column_visibility(GTK_CLIST (search_list), 0,
				  global.search_show[0]);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 1,
			      global.search_width[1]);
  gtk_clist_set_column_visibility(GTK_CLIST (search_list), 1,
				  global.search_show[1]);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 2,
			      global.search_width[2]);
  gtk_clist_set_column_visibility(GTK_CLIST (search_list), 2,
				  global.search_show[2]);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 3,
			      global.search_width[3]);
  gtk_clist_set_column_visibility(GTK_CLIST (search_list), 3,
				  global.search_show[3]);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 4,
			      global.search_width[4]);
  gtk_clist_set_column_visibility(GTK_CLIST (search_list), 4,
				  global.search_show[4]);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 5,
			      global.search_width[5]);
  gtk_clist_set_column_visibility(GTK_CLIST (search_list), 5,
				  global.search_show[5]);
  gtk_clist_set_selection_mode (GTK_CLIST (search_list), GTK_SELECTION_EXTENDED);
  gtk_clist_column_titles_show (GTK_CLIST (search_list));

  label = gtk_label_new (_("Filename"));
  gtk_widget_show (label);
  gtk_clist_set_column_widget (GTK_CLIST (search_list), 0, label);

  label = gtk_label_new (_("Size"));
  gtk_widget_show (label);
  gtk_clist_set_column_widget (GTK_CLIST (search_list), 1, label);

  label = gtk_label_new (_("Bitrate"));
  gtk_widget_show (label);
  gtk_clist_set_column_widget (GTK_CLIST (search_list), 2, label);

  label = gtk_label_new (_("Length"));
  gtk_widget_show (label);
  gtk_clist_set_column_widget (GTK_CLIST (search_list), 3, label);

  label = gtk_label_new (_("User"));
  gtk_widget_show (label);
  gtk_clist_set_column_widget (GTK_CLIST (search_list), 4, label);

  label = gtk_label_new (_("Line Speed"));
  gtk_widget_show (label);
  gtk_clist_set_column_widget (GTK_CLIST (search_list), 5, label);

  gtk_signal_connect (GTK_OBJECT (search_list), "click_column",
                      GTK_SIGNAL_FUNC (on_list_click_column),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (search_list), "button_press_event",
                      GTK_SIGNAL_FUNC (on_search_list_button_press_event),
                      NULL);

  str = g_strdup_printf("%s: %s\n%s", 
			Destination(pattern->destination),
			_(pattern->media_type),
			pattern->include);
  tab_label = gtk_label_new(str);
  g_free(str);

  gtk_object_set_data(GTK_OBJECT(search_list), "search", search);
  gtk_object_set_data(GTK_OBJECT(scrolledwindow), "clist", search_list);

  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook5"));
  gtk_notebook_append_page(notebook, scrolledwindow, tab_label);
  gtk_notebook_set_page(notebook, gtk_notebook_page_num(GTK_NOTEBOOK(notebook), 
							scrolledwindow));
}

GtkWidget* create_search_list_popup(file_t* file) {
  GtkWidget *popup;
  GtkWidget *user_popup;
  GtkWidget *item;
  GtkAccelGroup *popup_accels;
  GtkWidget *download;
  GtkWidget *download_selected;
  GtkWidget *clear_selected;
  GtkWidget *clear_users_files;
  GtkWidget *clear_all;
  GtkWidget *trennlinie13;
  GtkWidget *trennlinie17;
  GtkWidget *customize_list3;
  GtkCList* clist;
  char item_str[1024];
  int item_num;

  popup = gtk_menu_new ();
  popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (popup));
  clist = GTK_CLIST(global.popup_list);
  item_num = g_list_length(clist->selection);
  
  if (file) {
    if (file->local) {
      download = gtk_menu_item_new_with_label (_("Open file"));
      gtk_widget_show (download);
      gtk_container_add (GTK_CONTAINER (popup), download);
      gtk_signal_connect (GTK_OBJECT (download), "activate",
			  GTK_SIGNAL_FUNC (on_play_file3_activate),
			  NULL);
    } else if (item_num < 2) {
      download = gtk_menu_item_new_with_label (_("Download"));
      gtk_widget_show (download);
      gtk_container_add (GTK_CONTAINER (popup), download);
      gtk_signal_connect (GTK_OBJECT (download), "activate",
			  GTK_SIGNAL_FUNC (on_download_activate),
			  (void*)0);
      download = gtk_menu_item_new_with_label (_("Download with Foldername"));
      gtk_widget_show (download);
      gtk_container_add (GTK_CONTAINER (popup), download);
      gtk_signal_connect (GTK_OBJECT (download), "activate",
			  GTK_SIGNAL_FUNC (on_download_activate),
			  (void*)1);
    } else {
      sprintf(item_str, _("Download Selected (%d)"), item_num);
      download_selected = gtk_menu_item_new_with_label (item_str);
      gtk_widget_show (download_selected);
      gtk_container_add (GTK_CONTAINER (popup), download_selected);
      gtk_signal_connect (GTK_OBJECT (download_selected), "activate",
			  GTK_SIGNAL_FUNC (on_download_selected_activate),
			  (void*)0);
      sprintf(item_str, _("Download Selected (%d) with Foldername"), item_num);
      download_selected = gtk_menu_item_new_with_label (item_str);
      gtk_widget_show (download_selected);
      gtk_container_add (GTK_CONTAINER (popup), download_selected);
      gtk_signal_connect (GTK_OBJECT (download_selected), "activate",
			  GTK_SIGNAL_FUNC (on_download_selected_activate),
			  (void*)1);
    }
    
    trennlinie17 = gtk_menu_item_new ();
    gtk_widget_show (trennlinie17);
    gtk_container_add (GTK_CONTAINER (popup), trennlinie17);
    gtk_widget_set_sensitive (trennlinie17, FALSE);

    if (!file->local) {
      item = gtk_menu_item_new_with_label (_("User Menu"));
      gtk_widget_show (item);
      gtk_container_add (GTK_CONTAINER (popup), item);
      
      user_popup = create_user_popup(M_SEARCH);
      gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), user_popup);

      trennlinie17 = gtk_menu_item_new ();
      gtk_widget_show (trennlinie17);
      gtk_container_add (GTK_CONTAINER (popup), trennlinie17);
      gtk_widget_set_sensitive (trennlinie17, FALSE);
    }
      
    sprintf(item_str, _("Remove Selected (%d)"), item_num);
    clear_selected = gtk_menu_item_new_with_label (item_str);
    gtk_widget_show (clear_selected);
    gtk_container_add (GTK_CONTAINER (popup), clear_selected);
    gtk_signal_connect (GTK_OBJECT (clear_selected), "activate",
			GTK_SIGNAL_FUNC (on_clear_selected_activate),
			NULL);
    
    clear_users_files = gtk_menu_item_new_with_label (_("Remove Users Files"));
    gtk_widget_show (clear_users_files);
    gtk_container_add (GTK_CONTAINER (popup), clear_users_files);
    gtk_signal_connect (GTK_OBJECT (clear_users_files), "activate",
			GTK_SIGNAL_FUNC (on_clear_users_files_activate),
			NULL);
    if (file->local) 
      gtk_widget_set_sensitive(clear_users_files, FALSE);
  }

  clear_all = gtk_menu_item_new_with_label (_("Remove Search"));
  gtk_widget_show (clear_all);
  gtk_container_add (GTK_CONTAINER (popup), clear_all);
  gtk_signal_connect (GTK_OBJECT (clear_all), "activate",
		      GTK_SIGNAL_FUNC (on_clear_all_activate),
		      NULL);
    
  trennlinie13 = gtk_menu_item_new ();
  gtk_widget_show (trennlinie13);
  gtk_container_add (GTK_CONTAINER (popup), trennlinie13);
  gtk_widget_set_sensitive (trennlinie13, FALSE);
  
  clear_all = gtk_menu_item_new_with_label (_("Search again"));
  gtk_widget_show (clear_all);
  gtk_container_add (GTK_CONTAINER (popup), clear_all);
  gtk_signal_connect (GTK_OBJECT (clear_all), "activate",
		      GTK_SIGNAL_FUNC (on_search_again_activate),
		      NULL);
  trennlinie17 = gtk_menu_item_new ();
  gtk_widget_show (trennlinie17);
  gtk_container_add (GTK_CONTAINER (popup), trennlinie17);
  gtk_widget_set_sensitive (trennlinie17, FALSE);

  customize_list3 = gtk_menu_item_new_with_label (_("Customize List"));
  gtk_widget_show (customize_list3);
  gtk_container_add (GTK_CONTAINER (popup), customize_list3);
  gtk_signal_connect (GTK_OBJECT (customize_list3), "activate",
                      GTK_SIGNAL_FUNC (on_customize_list_activate),
                      NULL);

  return popup;
}

gint search_compare (GtkCList      *clist,
		     gconstpointer  ptr1,
		     gconstpointer  ptr2) {
  file_t* file1;
  file_t* file2;

  GtkCListRow *row1 = (GtkCListRow *) ptr1;
  GtkCListRow *row2 = (GtkCListRow *) ptr2;

  file1 = row1->data;
  file2 = row2->data;

  if (!file2) return (file1!=NULL);
  if (!file1) return -1;

  if (clist->sort_column == 0) {
    return strcasecmp (file1->shortname, file2->shortname);
  } else if (clist->sort_column == 1) {
    if (file1->size < file2->size) return -1;
    if (file1->size > file2->size) return 1;
    return 0;
  } else if (clist->sort_column == 2) {
    if (file1->bitrate < file2->bitrate) return -1;
    if (file1->bitrate > file2->bitrate) return 1;
    return 0;
  } else if (clist->sort_column == 3) {
    if (file1->duration < file2->duration) return -1;
    if (file1->duration > file2->duration) return 1;
    return 0;
  } else if (clist->sort_column == 4) {
    return strcasecmp (file1->user, file2->user);
  } else if (clist->sort_column == 5) {
    if (file1->linespeed < file2->linespeed) return -1;
    if (file1->linespeed > file2->linespeed) return +1;
    return 0;
  } else {
    return strcasecmp (file1->shortname, file2->shortname);
  }
}

file_t* file_create_from_search_response(char* data) {
  file_t* new_file;
  char* pos1;

  new_file = (file_t*)malloc(sizeof(file_t));

  new_file->winname = strdup(arg(data, 0));
  new_file->md5 = strdup(arg(NULL, 0));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%ld", &(new_file->size));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%d", &(new_file->bitrate));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%d", &(new_file->frequency));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%d", &(new_file->duration));

  new_file->user = strdup(arg(NULL, 0));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%lu", &(new_file->ip));

  pos1 = arg(NULL, 0);
  sscanf(pos1, "%d", &(new_file->linespeed));
  
  new_file->longname = strdup(new_file->winname);
  convert_to_unix(new_file->longname);
  pos1 = extract_short_name(new_file->longname);
  new_file->shortname = strdup(pos1);
  new_file->mime_type = get_mimetype(new_file->shortname);

  new_file->flags = 0;
  new_file->status = FILE_NONE;
  new_file->local = 0;            //remote file

  return new_file;
}

void append_field(search_pattern_t* pattern, char* data, int field) {
  int flag = 0;
  char *s1, *s2;
  char t[1024];
  IntFunc func = NULL;
  char str[1024];

  s1 = s2 = NULL;
  switch (field) {
  case 0:
    s1 = pattern->speed_lo;
    s2 = pattern->speed_hi;
    strcpy(t, "LINESPEED");
    func = (IntFunc)(speed2int);
    break;
  case 1:
    s1 = pattern->bitrate_lo;
    s2 = pattern->bitrate_hi;
    strcpy(t, "BITRATE");
    func = (IntFunc)(atoi);
    break;
  case 2:
    s1 = pattern->freq_lo;
    s2 = pattern->freq_hi;
    strcpy(t, "FREQ");
    func = (IntFunc)(atoi);
    break;
  }
  if (s1 && s2 && !strcmp(s1, s2)) {
    flag = 4;
  } else if (s2) {
    if (opennap_version(0, 30) && s1) flag = 3;
    else flag = 2;
  } else if (s1) {
    flag = 1;
  }
  if (flag & 1) {
    sprintf(str, " %s \"AT LEAST\" %d", t, func(s1));
    strcat(data, str);
  }
  if (flag & 2) {
    sprintf(str, " %s \"AT BEST\" %d", t, func(s2)) ;
    strcat(data, str);
  }
  if (flag & 4) {
    sprintf(str, " %s \"EQUAL TO\" %d", t, func(s1));
    strcat(data, str);
  }
}

void append_field2(search_pattern_t* pattern, char* data, int field) {
  long s1, s2;
  char t[1024];
  char str[1024];

  s1 = s2 = 0;
  switch (field) {
  case 0:
    s1 = pattern->duration_lo;
    s2 = pattern->duration_hi;
    strcpy(t, "DURATION");
    break;
  case 1:
    s1 = pattern->size_lo;
    s2 = pattern->size_hi;
    strcpy(t, "SIZE");
    break;
  }

  if ((s1 == s2) && (s1 > 0)) {
    sprintf(str, " %s \"EQUAL TO\" \"%ld\"", t, s1);
    strcat(data, str);
  } else {
    if (s1 > 0) {
      sprintf(str, " %s \"AT LEAST\" \"%ld\"", t, s1);
      strcat(data, str);
    }
    if (s1 > 0) {
      sprintf(str, " %s \"AT BEST\" \"%ld\"", t, s2);
      strcat(data, str);
    }
  }
}  

void napster_search(search_pattern_t* pattern, int local) {
  char data[2048];
  char str[1024];
  int opennap;
  char* excl;

  if (global.status.connection < 2) {
    g_warning("not connected!");
    return;
  }
  global.status.searching++;
  search_update_counter();
  setup_sensitive(-1);

  if (opennap_version(0, 30)) opennap = 1;
  else opennap = 0;
  
  *data = 0;
  if (strlen(pattern->include) == 0) return;
  if (opennap) {
    sprintf(str, "FILENAME CONTAINS \"%s\"", pattern->include);
    strcat(data, str);
    if (strlen(pattern->exclude) > 0) {
      strcat(data, str);
      sprintf(str, " FILENAME EXCLUDES \"%s\"", pattern->exclude);
    }
  } else {
    sprintf(str, "FILENAME CONTAINS \"%s", pattern->include);
    strcat(data, str);
    strcpy(str, pattern->exclude);
    excl = arg(str, 0);
    while (excl) {
      strcat(data, " -");
      strcat(data, excl);
      excl = arg(NULL, 0);
    }
    strcat(data, "\"");
  }

  sprintf(str, " MAX_RESULTS %d", pattern->max_results);
  strcat(data, str);
  
  append_field(pattern, data, 0);
  append_field(pattern, data, 1);
  append_field(pattern, data, 2);

  append_field2(pattern, data, 0);
  append_field2(pattern, data, 1);
  
  if (pattern->media_type &&
      strcmp(pattern->media_type, "mp3")) {
    sprintf(str, " TYPE %s", pattern->media_type);
    strcat(data, str);
  }
  
  if (local) strcat(data, " LOCAL_ONLY");
  send_command(CMD_CLIENT_SEARCH, data);
}

int search_find_free_id() {
  int id = 0;
  GList* dlist;
  search_t* search;

  while (id < 1000) {
    for (dlist = global.searches; dlist; dlist = dlist->next) {
      search = (search_t*)(dlist->data);
      if (search->id == id) break;
    }
    if (dlist) id++;
    else return id;
  }
  return -1;
}

void send_search_request(gboolean local) {
  search_pattern_t* pattern;
  GtkWidget* temp;
  search_t* search;
  char str[1024];

  pattern = search_pattern_find(_("Last Search"));
  if (pattern) search_pattern_update(pattern);
  else {
    pattern = search_pattern_create(_("Last Search"));
    global.search_pattern = 
      g_list_prepend(global.search_pattern, pattern);
  }
  search_pattern_setup_list();

  temp = lookup_widget(global.win, "combo_entry19");

  //////
  search = search_new();
  sprintf(str, "%d %s", search->id, pattern->include);
  search->pattern = search_pattern_create(str);
  search_create_page(search);
  /////

  pattern = search->pattern;
  if (pattern->destination == DEST_LIBRARY) {
    lib_search(pattern);
  } else if (pattern->destination == DEST_HOTLIST) {
    hotlist_search(pattern);
  } else if (pattern->destination == DEST_NAPSTER) {
    napster_search(pattern, local);
  } else if (pattern->destination == DEST_SEARCH) {
    search_search(pattern);
  } else {
    printf("unknown search destination");
  }
}

void search_insert_file(GtkWidget* child, file_t *file) {
  int row;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  GtkCList* clist;
  GtkWidget* temp;
  GtkNotebook* notebook;
  search_t* search;

  clist = gtk_object_get_data(GTK_OBJECT(child), "clist");
  
  strcpy(tstr[0], file->shortname);
  sprintf(tstr[1], "%.2f MB", (double)(file->size)/1024/1024);
  sprintf(tstr[2], "%d", file->bitrate);
  sprintf(tstr[3], "%d:%s%d", file->duration/60, 
	  (file->duration%60)<10?"0":"", file->duration%60);
  strcpy(tstr[4], file->user);
  if (file->local)
    strcpy(tstr[5], _("Local File"));
  else
    strcpy(tstr[5], LineSpeed(file->linespeed));
    
  strcpy(tstr[6], _("unknown"));

  row = gtk_clist_append(clist, &list[0]);
  gtk_clist_set_row_data (clist, row, (gpointer)file);

  detect_speed_pixs(file->linespeed, &pixmap, &bitmap);
  
  gtk_clist_set_pixtext (clist, row, 0, tstr[0], 5, pixmap, bitmap);

  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook5"));
  row = gtk_notebook_get_current_page(notebook);
  temp = gtk_notebook_get_nth_page(notebook, row);
  if (temp == child) {
    search = (search_t*)
      gtk_object_get_data(GTK_OBJECT(clist), "search");
    search_update_stats(search);
  }
}

void file_insert_search(file_t *file, int destination) {
  search_t* search;
  GList* dlist;
  int cnt = 0;
  file_t* file2;

  for (dlist = global.searches; dlist; dlist = dlist->next) {
    search = (search_t*)(dlist->data);
    if ((search->pattern->destination == destination) &&
	search_pattern_fits_file(search->pattern, file, 0)) {
      cnt++;
      file2 = file_dup(file);
      search->results = g_list_append(search->results, file2);
      if (search->resume) 
	resume_insert_file((resume_t*)(search->link), file2);
      else 
	search_insert_file(GTK_WIDGET(search->link), file2);
    }
  }
#ifdef SEARCH_DEBUG
  if (cnt == 0) {
    printf("could not find search for [%s]\n", file->longname);
  }
#endif
}

void search_pattern_load() {
  FILE* file;
  char filename[1024];
  char line[500];
  search_pattern_t* search;
  char* pos;
  char* pos2;

  strcpy(filename, global.lopster_home);
  strcat(filename, "/searches.save");
  if ((file = fopen(filename, "r")) == NULL) {
    return;
  }

  search = NULL;
  while (fgets(line, 400, file)) {
    (strchr(line, '\n'))[0] = 0;
    if ((line[0] == '#') || 
	(line[0] == ' ') ||
	(line[0] == 0)) {
    } else if (!strncasecmp(line, "Pattern", 7)) {
      pos = strchr(line, '[')+1;
      pos2 = strrchr(line, ']');
      pos2[0] = 0;
      search = search_pattern_new(pos);
      global.search_pattern = 
	g_list_append(global.search_pattern, search);
    } else if (search) {
      if (!strncasecmp(line, "contains", 8)) {
	pos = strchr(line, '=')+1;
	search->include = strdup(pos);
      } else if (!strncasecmp(line, "excludes", 8)) {
	pos = strchr(line, '=')+1;
	search->exclude = strdup(pos);
      } else if (!strncasecmp(line, "MaxResults", 10)) {
	pos = strchr(line, '=')+1;
	search->max_results = atoi(pos);
      } else if (!strncasecmp(line, "MediaType", 9)) {
	pos = strchr(line, '=')+1;
	if (strlen(pos) == 0)    // backward compatible
	  search->media_type = strdup("mp3");
	else search->media_type = strdup(pos);
      } else if (!strncasecmp(line, "Destination", 11)) {
	pos = strchr(line, '=')+1;
	search->destination = atoi(pos);
      } else if (!strncasecmp(line, "Bitrate", 7)) {
	pos = strchr(line, '=')+1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	if (*pos) search->bitrate_lo = strdup(pos);
	if (*pos2) search->bitrate_hi = strdup(pos2);
      } else if (!strncasecmp(line, "Frequency", 9)) {
	pos = strchr(line, '=')+1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	if (*pos) search->freq_lo = strdup(pos);
	if (*pos2) search->freq_hi = strdup(pos2);
      } else if (!strncasecmp(line, "LineSpeed", 9)) {
	pos = strchr(line, '=')+1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	if (*pos) search->speed_lo = strdup(pos);
	if (*pos2) search->speed_hi = strdup(pos2);
      } else if (!strncasecmp(line, "Size", 4)) {
	pos = strchr(line, '=')+1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	search->size_lo = strtoul(pos, NULL, 10);
	search->size_hi = strtoul(pos2, NULL, 10);
      } else if (!strncasecmp(line, "Duration", 8)) {
	pos = strchr(line, '=')+1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	search->duration_lo = atoi(pos);
	search->duration_hi = atoi(pos2);
      } else {
	g_warning(_("unknown tag in section Search Pattern [%s]\n[%s]"), 
		  search->name, line);
      }
    } else {
      g_warning("no search name is given");
    }
  }
  fclose(file);
  search_pattern_setup_list();
}

void search_pattern_save() {
  GList* dlist;
  search_pattern_t* search;
  FILE* file;
  char filename[1024];
  GtkWidget* temp;

  strcpy(filename, global.lopster_home);
  strcat(filename, "/searches.save");
  if ((file = fopen(filename, "w")) == NULL) {
    g_warning("Could not write searches");
    return;
  }

  for (dlist = global.search_pattern; dlist; dlist = dlist->next) {
    search = (search_pattern_t*)(dlist->data);
    fprintf(file, "PATTERN [%s]\n", search->name);
    fprintf(file, "Contains=%s\n", search->include);
    fprintf(file, "Excludes=%s\n", search->exclude);
    fprintf(file, "MaxResults=%d\n", search->max_results);
    fprintf(file, "MediaType=%s\n",
	    (search->media_type)?(search->media_type):"");
    fprintf(file, "Destination=%d\n", search->destination);
    fprintf(file, "Bitrate=%s:%s\n", 
	    (search->bitrate_lo)?(search->bitrate_lo):"",
	    (search->bitrate_hi)?(search->bitrate_hi):"");
    fprintf(file, "Frequency=%s:%s\n", 
	    (search->freq_lo)?(search->freq_lo):"",
	    (search->freq_hi)?(search->freq_hi):"");
    fprintf(file, "LineSpeed=%s:%s\n", 
	    (search->speed_lo)?(search->speed_lo):"",
	    (search->speed_hi)?(search->speed_hi):"");
    fprintf(file, "Size=%ld:%ld\n", search->size_lo, search->size_hi);
    fprintf(file, "Duration=%d:%d\n\n", search->duration_lo, search->duration_hi);
  }
  
  temp = lookup_widget(global.win, "combo_entry18");
  gtk_entry_set_text(GTK_ENTRY(temp), _("Last Search"));

  fclose(file);
}

void search_pattern_setup_list() {
  GtkWidget* temp;
  GList* dlist;
  search_pattern_t* search;
  GList* list2;
  search_pattern_t* current;

  temp = lookup_widget(global.win, "combo14");
  list2 = NULL;
  for (dlist = global.search_pattern; dlist; dlist = dlist->next) {
    search = (search_pattern_t*)(dlist->data);
    list2 = g_list_append (list2, search->name);
  }

  if (!list2) list2 = g_list_append (list2, "");

  gtk_combo_set_popdown_strings (GTK_COMBO (temp), list2);
  g_list_free (list2);

  current = gtk_object_get_user_data(GTK_OBJECT(temp));
  if (current) {
    temp = lookup_widget(global.win, "combo_entry18");
    gtk_entry_set_text(GTK_ENTRY(temp), current->name);
  }
  search_pattern_save();
}

search_pattern_t* search_pattern_find(char* iname) {
  GList* dlist;
  search_pattern_t* search;

  for (dlist = global.search_pattern; dlist; dlist = dlist->next) {
    search = (search_pattern_t*)(dlist->data);
    if (!strcmp(iname, search->name)) return search;
  }
  return NULL;
}

void search_pattern_get(search_pattern_t* search) {
  GtkWidget* temp;

  temp = lookup_widget(global.win, "search_artist");
  search->include = strdup(gtk_entry_get_text(GTK_ENTRY(temp)));
  
  temp = lookup_widget(global.win, "entry69");
  search->exclude = strdup(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "search_results");
  search->max_results = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp));

  temp = lookup_widget(global.win, "combo_entry17");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->media_type = 
      get_real_mediatype(gtk_entry_get_text(GTK_ENTRY(temp)));
  else search->media_type = strdup("mp3");

  temp = lookup_widget(global.win, "combo_entry19");
  search->destination = destination2int(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "entry76");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->bitrate_lo =
      get_real_entry(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "combo_bitrate2");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->bitrate_hi =
      get_real_entry(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "entry77");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->freq_lo =
      get_real_entry(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "combo_freq2");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->freq_hi =
      get_real_entry(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "entry78");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->speed_lo = 
      get_real_speed(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "combo_speed2");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->speed_hi =
      get_real_speed(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "spinbutton30");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->size_lo = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp))*1024*1024;
  else search->size_lo = 0;

  temp = lookup_widget(global.win, "spinbutton31");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->size_lo += gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp))*1024;

  temp = lookup_widget(global.win, "spinbutton28");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->size_hi = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp))*1024*1024;
  else search->size_hi = 0;

  temp = lookup_widget(global.win, "spinbutton29");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->size_hi += gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp))*1024;

  temp = lookup_widget(global.win, "spinbutton25");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->duration_lo = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp))*60;
  else search->duration_lo = 0;

  temp = lookup_widget(global.win, "spinbutton27");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->duration_lo += gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp));

  temp = lookup_widget(global.win, "spinbutton24");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->duration_hi = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp))*60;
  else search->duration_hi = 0;

  temp = lookup_widget(global.win, "spinbutton25");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->duration_hi += gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp));
}

search_pattern_t* search_pattern_new(char* name) {
  search_pattern_t* pattern;

  pattern = malloc(sizeof(*pattern));

  if (name) pattern->name = strdup(name);
  else pattern->name = NULL;
  pattern->include = NULL;
  pattern->exclude = NULL;
  pattern->media_type = NULL;
  pattern->destination = DEST_NAPSTER;
  pattern->bitrate_lo = NULL;
  pattern->bitrate_hi = NULL;
  pattern->freq_lo = NULL;
  pattern->freq_hi = NULL;
  pattern->speed_lo = NULL;
  pattern->speed_hi = NULL;
  pattern->max_results = 0;
  pattern->size_lo = 0;
  pattern->size_hi = 0;
  pattern->duration_lo = 0;
  pattern->duration_hi = 0;

  return pattern;
}

search_pattern_t* search_pattern_create(char* name) {
  search_pattern_t* search;
  
  search = search_pattern_new(name);
  search_pattern_get(search);

  return search;
}

void search_pattern_update(search_pattern_t* search) {
  if (search->include) free(search->include);
  search->include = NULL;
  if (search->exclude) free(search->exclude);
  search->exclude = NULL;
  if (search->media_type) free(search->media_type);
  search->media_type = NULL;
  if (search->bitrate_lo) free(search->bitrate_lo);
  search->bitrate_lo = NULL;
  if (search->bitrate_hi) free(search->bitrate_hi);
  search->bitrate_hi = NULL;
  if (search->freq_lo) free(search->freq_lo);
  search->freq_lo = NULL;
  if (search->freq_hi) free(search->freq_hi);
  search->freq_hi = NULL;
  if (search->speed_lo) free(search->speed_lo);
  search->speed_lo = NULL;
  if (search->speed_hi) free(search->speed_hi);
  search->speed_hi = NULL;
  
  search_pattern_get(search);
}

void search_pattern_destroy(search_pattern_t* search) {
  if (!search) return;
  if (search->name) free(search->name);
  if (search->include) free(search->include);
  if (search->exclude) free(search->exclude);
  if (search->media_type) free(search->media_type);
  if (search->bitrate_lo) free(search->bitrate_lo);
  if (search->bitrate_hi) free(search->bitrate_hi);
  if (search->freq_lo) free(search->freq_lo);
  if (search->freq_hi) free(search->freq_hi);
  if (search->speed_lo) free(search->speed_lo);
  if (search->speed_hi) free(search->speed_hi);
  free(search);
}

int search_pattern_fits_file(search_pattern_t* search, file_t* file,
			     int with_type) {
  static char* pos;
  static char* text;
  static long val1;
  static long val2;
  
  text = strdup(search->include);
  pos = search_arg(text);
  while (pos) {
    if (!strcasestr(file->longname, pos)) return 0;
    pos = search_arg(NULL);
  }
  free(text);

  text = strdup(search->exclude);
  pos = arg(text, 0);
  while (pos) {
    if (strcasestr(file->longname, pos)) return 0;
    pos = arg(NULL, 0);
  }
  free(text);

  if (with_type && search->media_type &&
      strcmp(search->media_type, "any") &&
      (mime2int_(search->media_type) != file->mime_type))
    return 0;
  
  val1 = (search->bitrate_lo)?atoi(search->bitrate_lo):0;
  val2 = (search->bitrate_hi)?atoi(search->bitrate_hi):0;
  if (val1 && (file->bitrate < val1)) return 0;
  if (val2 && (file->bitrate > val2)) return 0;

  val1 = (search->freq_lo)?atoi(search->freq_lo):0;
  val2 = (search->freq_hi)?atoi(search->freq_hi):0;
  if (val1 && (file->frequency < val1)) return 0;
  if (val2 && (file->frequency > val2)) return 0;

  val1 = search->size_lo;
  val2 = search->size_hi;
  if (val1 && (file->size < val1)) return 0;
  if (val2 && (file->size > val2)) return 0;
  
  val1 = search->duration_lo;
  val2 = search->duration_hi;
  if (val1 && (file->duration < val1)) return 0;
  if (val2 && (file->duration > val2)) return 0;
  
  if (!file->local) {
    val1 = (search->speed_lo)?speed2int(search->speed_lo):0;
    val2 = (search->speed_hi)?speed2int(search->speed_hi):10;
    if (file->linespeed < val1) return 0;
    if (file->linespeed > val2) return 0;
  }
  return 1;
}

void search_pattern_delete() {
  GtkWidget* temp;
  search_pattern_t* search;


  temp = lookup_widget(global.win, "combo14");
  search = (search_pattern_t*)gtk_object_get_user_data(GTK_OBJECT(temp));
  gtk_object_set_user_data(GTK_OBJECT(temp), NULL);

  if (!search) return;
  
  global.search_pattern = 
    g_list_remove(global.search_pattern, search);
  search_pattern_destroy(search);
  search_pattern_setup_list();
}

void search_pattern_rename_current(char* new_name) {
  GtkWidget* temp;
  search_pattern_t* search;

  temp = lookup_widget(global.win, "combo14");
  search = (search_pattern_t*)gtk_object_get_user_data(GTK_OBJECT(temp));
  if (!search) return;

  if (search->name) free(search->name);
  search->name = strdup(new_name);
  search_pattern_setup_list();
}

void search_pattern_save_current() {
  GtkWidget* temp;
  search_pattern_t* search;
  char* text;

  temp = lookup_widget(global.win, "combo_entry18");
  text = gtk_entry_get_text(GTK_ENTRY(temp));
  temp = lookup_widget(global.win, "combo14");
  search = (search_pattern_t*)gtk_object_get_user_data(GTK_OBJECT(temp));

  if (search) search_pattern_update(search);
  else if (strlen(text)>0) {
    search = search_pattern_create(text);
    global.search_pattern = 
      g_list_append(global.search_pattern, search);
  }
  gtk_object_set_user_data(GTK_OBJECT(temp), search);

  search_pattern_setup_list();
}

void search_pattern_show(search_pattern_t* pattern) {
  GtkWidget* temp;

  if (pattern) {
    temp = lookup_widget(global.win, "search_artist");
    gtk_entry_set_text(GTK_ENTRY(temp), pattern->include);
    temp = lookup_widget(global.win, "entry69");
    gtk_entry_set_text(GTK_ENTRY(temp), pattern->exclude);
    temp = lookup_widget(global.win, "search_results");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), pattern->max_results);
    temp = lookup_widget(global.win, "combo_entry17");
    gtk_entry_set_text(GTK_ENTRY(temp), _(pattern->media_type));
    temp = lookup_widget(global.win, "combo_entry19");
    gtk_entry_set_text(GTK_ENTRY(temp), Destination_long(pattern->destination));
    temp = lookup_widget(global.win, "entry76");
    gtk_entry_set_text(GTK_ENTRY(temp), 
		       (pattern->bitrate_lo)?(pattern->bitrate_lo):_("not specified"));
    temp = lookup_widget(global.win, "combo_bitrate2");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->bitrate_hi)?(pattern->bitrate_hi):_("not specified"));
    temp = lookup_widget(global.win, "entry77");
    gtk_entry_set_text(GTK_ENTRY(temp), 
		       (pattern->freq_lo)?(pattern->freq_lo):_("not specified"));
    temp = lookup_widget(global.win, "combo_freq2");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->freq_hi)?(pattern->freq_hi):_("not specified"));
    temp = lookup_widget(global.win, "entry78");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->speed_lo)?(pattern->speed_lo):_("not specified"));
    temp = lookup_widget(global.win, "combo_speed2");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->speed_hi)?(pattern->speed_hi):_("not specified"));
    temp = lookup_widget(global.win, "spinbutton30");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), pattern->size_lo/1024/1024);
    temp = lookup_widget(global.win, "spinbutton31");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), (pattern->size_lo/1024)%1024);
    temp = lookup_widget(global.win, "spinbutton28");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), pattern->size_hi/1024/1024);
    temp = lookup_widget(global.win, "spinbutton29");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), (pattern->size_lo/1024)%1024);
    temp = lookup_widget(global.win, "spinbutton25");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), pattern->duration_lo/60);
    temp = lookup_widget(global.win, "spinbutton27");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), pattern->duration_lo%60);
    temp = lookup_widget(global.win, "spinbutton24");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), pattern->duration_hi/60);
    temp = lookup_widget(global.win, "spinbutton25");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), pattern->duration_hi%60);
  }
  if (!pattern || !g_list_find(global.search_pattern, pattern)) {
    temp = lookup_widget(global.win, "button174");
    gtk_label_set_text(GTK_LABEL(GTK_BIN(temp)->child), _("Save Pattern"));
    temp = lookup_widget(global.win, "button175");
    gtk_widget_set_sensitive(temp, FALSE);
    temp = lookup_widget(global.win, "button177");
    gtk_widget_set_sensitive(temp, FALSE);
    temp = lookup_widget(global.win, "combo14");
    gtk_object_set_user_data(GTK_OBJECT(temp), NULL);
  } else {
    temp = lookup_widget(global.win, "button174");
    gtk_label_set_text(GTK_LABEL(GTK_BIN(temp)->child), _("Overwrite Pattern"));
    temp = lookup_widget(global.win, "button175");
    gtk_widget_set_sensitive(temp, TRUE);
    temp = lookup_widget(global.win, "button177");
    gtk_widget_set_sensitive(temp, TRUE);
    temp = lookup_widget(global.win, "combo14");
    gtk_object_set_user_data(GTK_OBJECT(temp), pattern);
  }
}

search_pattern_t* search_pattern_copy(search_pattern_t* search) {
  search_pattern_t* pattern;
  
  pattern = search_pattern_new(NULL);
  if (search->name) pattern->name = strdup(search->name);
  if (search->include) pattern->include = strdup(search->include);
  if (search->exclude) pattern->exclude = strdup(search->exclude);
  if (search->media_type) pattern->media_type = strdup(search->media_type);
  pattern->destination = search->destination;
  if (search->bitrate_lo) pattern->bitrate_lo = strdup(search->bitrate_lo);
  if (search->bitrate_hi) pattern->bitrate_hi = strdup(search->bitrate_hi);
  if (search->freq_lo) pattern->freq_lo = strdup(search->freq_lo);
  if (search->freq_hi) pattern->freq_hi = strdup(search->freq_hi);
  if (search->speed_lo) pattern->speed_lo = strdup(search->speed_lo);
  if (search->speed_hi) pattern->speed_hi = strdup(search->speed_hi);
  pattern->max_results = search->max_results;
  pattern->size_lo = search->size_lo;
  pattern->size_hi = search->size_hi;
  pattern->duration_lo = search->duration_lo;
  pattern->duration_hi = search->duration_hi;

  return pattern;
}

search_t* search_new() {
  search_t* search;

  search = (search_t*)malloc(sizeof(search_t));
  search->results = NULL;
  search->id = search_find_free_id();
  search->resume = 0;
  search->pattern = NULL;
  search->link = NULL;
  
  global.searches = g_list_append(global.searches, search);

  return search;
}

search_t* search_find(int id) {
  GList* dlist;
  search_t* search;

  for (dlist = global.searches; dlist; dlist = dlist->next) {
    search = (search_t*)(dlist->data);
    if (search->id == id) return search;
  }
  return NULL;
}

void search_clear(search_t* search) {
  GList* dlist;
  file_t* file;
  GtkCList* clist;

  for (dlist = search->results; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    destroy_file_row(file);
  }
  g_list_free(search->results);
  search->results = NULL;

  if (!search->resume) {
    clist = GTK_CLIST(gtk_object_get_data(GTK_OBJECT(search->link), "clist"));
    gtk_clist_clear(clist);
  }

}

void search_destroy(search_t* search) {

  if (!search) return;
  // releasing file list
  search_clear(search);
  search_pattern_destroy(search->pattern);
  free(search);
}

void search_remove(search_t* search) {
  GtkNotebook* notebook;
  int i1;
  
  global.searches = g_list_remove(global.searches, search);

  if (!search->resume) {
    notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook5"));
    i1 = gtk_notebook_page_num(notebook, GTK_WIDGET(search->link));
    search_destroy(search);
    gtk_notebook_remove_page(notebook, i1);
  } else {
    search_destroy(search);
  }
}

void
search_update_stats(search_t* search) 
{
  char t[1024];
  GtkWidget* label;
  
  label = lookup_widget(global.win, "label597");
  sprintf(t, "%d", g_list_length(search->results));
  gtk_label_set_text(GTK_LABEL(label), t);
}

void
search_update_counter() {
  GtkWidget* label;
  char t[1024];
  
  label = lookup_widget(global.win, "label598");
  sprintf(t, "%d", global.status.searching);
  gtk_label_set_text(GTK_LABEL(label), t);
}

gboolean
on_search_fields_delete             (GtkWidget       *widget,
				     GdkEvent        *event,
				     gpointer         user_data) {
  GtkWidget* win;
  GtkWidget* temp;
  GtkWidget* vbox;

  win = widget;

  if (win->window) {
    gdk_window_get_origin (win->window, 
			   &global.geometry[7].x, 
			   &global.geometry[7].y);
  }

  temp = GTK_BIN(win)->child;
  if (!temp) return FALSE;
  
  gtk_widget_ref(temp);
  gtk_container_remove(GTK_CONTAINER(win), temp);
  
  vbox = lookup_widget(global.win, "vbox5");
  gtk_box_pack_start (GTK_BOX (vbox), temp, FALSE, FALSE, 0);
  gtk_box_reorder_child(GTK_BOX (vbox), temp, 1);
  gtk_widget_unref(temp);

  widget = lookup_widget(global.win, "button115");
  gtk_widget_set_sensitive(widget, TRUE);
  widget = lookup_widget(global.win, "button218");
  gtk_widget_set_sensitive(widget, TRUE);
  global.extra_win &= (0xFF)^(1<<7);

  return FALSE;
}

GtkWidget*
search_fields_detach() {
  GtkWidget* window;
  GtkWidget* widget;

  widget = lookup_widget(global.win, "frame3");
  if (!widget) return NULL;
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width (GTK_CONTAINER (window), 5);
  gtk_object_set_data(GTK_OBJECT(global.win), "fields", window);
  gtk_window_set_policy (GTK_WINDOW (window), FALSE, TRUE, TRUE);
  gtk_window_set_title (GTK_WINDOW (window), _("Search Fields"));

  gtk_my_widget_show(window);

  gtk_widget_ref(widget);
  gtk_container_remove(GTK_CONTAINER(widget->parent), widget);
  gtk_container_add (GTK_CONTAINER (window), widget);
  gtk_widget_unref(widget);

  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
                      GTK_SIGNAL_FUNC (on_search_fields_delete),
                      NULL);
  widget = lookup_widget(global.win, "button115");
  set_button_state(widget, 1);
  gtk_widget_set_sensitive(widget, FALSE);
  widget = lookup_widget(global.win, "button218");
  gtk_widget_set_sensitive(widget, FALSE);
  global.extra_win |= (1<<7);

  if ((global.geometry[7].x >= 0) &&
      (global.geometry[7].y >= 0)) {
    gtk_window_reposition(GTK_WINDOW(window),
			  global.geometry[7].x,
			  global.geometry[7].y);
  }
  return window;
}

void search_search(search_pattern_t* pattern) {
  GList* dlist;
  GList* dlist2;
  file_t* file;
  int cnt = 0;
  search_t* search;

  for (dlist = global.searches; dlist; dlist = dlist->next) {
    search = (search_t*)(dlist->data);
    if (search->resume) continue;
    if (search->pattern->destination == DEST_SEARCH) continue;
    for (dlist2 = search->results; dlist2; dlist2 = dlist2->next) {
      file = (file_t*)(dlist2->data);
      if (search_pattern_fits_file(pattern, file, 1)) {
	file = file_dup(file);
	file_insert_search(file, DEST_SEARCH);
	cnt++;
      }
      if (cnt >= pattern->max_results) return;
    }
  }
}
