#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/time.h>
#include <string.h>
#include <strings.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include "lopster.h"
#include "support.h"
#include "connection.h"
#include "global.h"
#include "commands.h"
#include "search.h"
#include "transfer.h"
#include "handler.h"
#include "log.h"
#include "hotlist.h"
#include "browse.h"
#include "share.h"
#include "server.h"
#include "preferences.h"
#include "resume.h"
#include "chat.h"
#include "dialog.h"

#define NOT_DOTTED_QUAD ((u_long)-1)

int server_login(gpointer data, gint source, GdkInputCondition condition);
int connection_timeout(gpointer data);
void socket_remove_clist(socket_t* socket);

unsigned long resolv_address(char* address) {
  struct hostent* h;
  unsigned long addr;

  addr = inet_addr(address);
  if (addr != INADDR_NONE) return addr;

  h = gethostbyname(address);
  if (h)
    addr = ((struct in_addr*)(h->h_addr_list[0]))->s_addr;
  else addr = 0;

  return addr;
}

void connect_progress(char* text) {
  GtkStatusbar* status;
  
  if (!text) 
    text = _("User canceled");
  if (global.connect_win) {
    status = GTK_STATUSBAR(lookup_widget(global.connect_win, "connect_status"));
    gtk_statusbar_push(status, 1, text);
  } else {
    client_message(_("Connect"), text);
  }
}

void connect_success() {
  global.status.connection = 2;
  if (global.connect_win) {
    gtk_widget_hide(global.connect_win);
    gtk_widget_destroy(global.connect_win);
    global.connect_win = NULL;
  }
}

socket_t* socket_new(int type) {
  socket_t* sock;

  sock = (socket_t*)malloc(sizeof(socket_t));
  
  sock->ip_long = 0;
  sock->port = 0;
  sock->fd = -1;
  sock->input = -1;
  sock->timeout = -1;
  sock->cnt = 0;
  sock->data = NULL;
  sock->type = type;

  switch (type) {
  case S_SHARE:
  case S_BROWSE:
    sock->max_cnt = 30;
    break;
  case S_TRANSFER:
    sock->max_cnt = global.network.transfer_timeout;
    break;
  case S_SERVER:
    sock->max_cnt = 300;
    break;
  case S_HTTP:
    sock->max_cnt = 20;
    break;
  case S_DATA:
    sock->max_cnt = 0;
    break;
  default:
    sock->max_cnt = 100;
    break;
  }

  global.sockets = g_list_append(global.sockets, sock);
  
  return sock;
}

void socket_end(socket_t* socket, int data) {
  transfer_t* transfer;
  server_t* server;
  int tim;
  GtkWidget* temp;

#ifdef CONNECTION_DEBUG
  printf("ending socket [%d]\n", socket->fd);
#endif

  if (socket->fd >= 0) close(socket->fd);
  socket->fd = -1;
  if (socket->input >= 0) gdk_input_remove(socket->input);
  socket->input = -1;
  if (socket->timeout >= 0) gtk_timeout_remove(socket->timeout);
  socket->timeout = -1;
  
  switch (socket->type) {
  case S_TRANSFER:
    transfer = (transfer_t*)(socket->data);
    if (transfer) {
      if (transfer->type == T_DOWNLOAD) {
	if ((tim = transfer_to_retry(data)) >= 0) {
#ifdef CONNECTION_DEBUG
	  printf("adding timeout\n");
#endif
	  socket->timeout = 
 	    gtk_timeout_add(tim, transfer_remote_timeout, socket);
	}
      }
      transfer_end(socket->data, data);
      transfer_status_set(socket, data);
      
      if (transfer_to_destroy(socket, data))
	transfer_clist_remove(socket);
    }
    break;
  case S_HTTP:
    napigator_update_network();
    break;
  case S_SERVER:
    server = (server_t*)(socket->data);
    if (server) {
      global.status.connection = 0;
      connect_progress((char*)data);
      napster_reset();
      if (global.connect_win) {
	temp = lookup_widget(global.connect_win, "connection_ok");
	gtk_widget_set_sensitive(temp, TRUE);
      }
    }
  default:
    break;
  }

}

void socket_destroy(socket_t* socket, int data) {
  
  if (!socket) return;
  
#ifdef CONNECTION_DEBUG
  printf("destroying socket [%d]\n", socket->fd);
#endif
  socket_end(socket, data);
  global.sockets = g_list_remove(global.sockets, socket);
  socket_remove_clist(socket);

  switch (socket->type) {
  case S_TRANSFER:
    if (socket->data)
      transfer_destroy(socket->data);
    break;
  case S_BROWSE:
    if (socket->data)
      free(socket->data);
    break;
  case S_SERVER:
    if (socket->data) {
      server_destroy(socket->data);
      global.napster = NULL;
    }
    break;
  default:
    break;
  }
  
  free(socket);
}

char* socket_get_name(socket_t* socket) {
  switch (socket->type) {
  case S_SHARE: return _("Share");
  case S_BROWSE: return _("Browse");
  case S_HTTP: return _("Http");
  case S_SERVER: return _("Server");
  case S_DATA: return _("Data");
  case S_TRANSFER: return _("Transfer");
  case S_UNKNOWN: return _("Unknown");
  default: return _("Ooops");
  }
}

int connect_to_server(socket_t* socket) {
  GtkWidget* temp;

  if (global.connect_win) {
    temp = lookup_widget(global.connect_win, "connection_ok");
    gtk_widget_set_sensitive(temp, FALSE);
  }
  if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
    return 0;
  }
  global.status.connection = 1;
  socket->cnt = 0;

  socket->input =
    gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		  GTK_SIGNAL_FUNC(server_login), socket);
  return 0;
}

int auto_reconnect(gpointer data) {
  socket_t* socket;

  socket = (socket_t*)(data);
  if (socket->timeout >= 0) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
  }

  if (global.napster)
    connect_to_server(global.napster);
  
  return 1;
}

int get_best_server_read(gpointer data, 
			 gint source, 
			 GdkInputCondition condition) {
  int length;
  socket_t* socket;
  unsigned char buffer[100+1];
  char* pos;
  server_t* server;
  server_t* server2;

  socket = (socket_t*)data;
  
  if (condition != GDK_INPUT_READ) {
    napster_disconnect(_("Could not get best server"));
    return 1;
  }

  gdk_input_remove(socket->input);
  socket->input = -1;

  while ((length = recv(socket->fd, buffer, 100, 0)) <= 0) {
    if (length <= 0) {
      napster_disconnect(_("Could not get best server"));
      return 1;
    }
  }  

  buffer[length] = '\0';

  server = socket->data;

#ifdef CONNECTION_DEBUG
  printf("got server %s\n", buffer);
#endif

  pos = strtok(buffer, ":");
  if (strcmp(pos, "127.0.0.1")) {
    server2 = server_new();
    server_set_address(server2, pos);
    server_set_description(server2, _("No description"));
    pos = strtok(NULL, " ");
    server2->port = atoi(pos);
    server2->nick = strdup(server->nick);
    server2->passwd = strdup(server->passwd);
    connect_progress(_("Got best Server"));
    napster_disconnect(NULL);
    napster_connect(server2);
  } else {
    napster_disconnect(_("All Servers busy"));
  }

  return 1;
}

int connect_socket(socket_t* s, char* proto, int type) {
  struct protoent *protocol;
  struct sockaddr_in server;
  int sock;
  int res;

  protocol = getprotobyname(proto);
  if (!protocol) {
    socket_destroy(s, (int)_("Could not get protocol"));
    return 0;
  }
  sock = socket(AF_INET, type, protocol->p_proto);
  
  if (sock < 0) {
    socket_destroy(s, (int)_("Could not create socket"));
    return 0;
  }
  s->fd = sock;
  s->cnt = 0;

  server.sin_family = AF_INET;
  server.sin_addr.s_addr = s->ip_long;
  server.sin_port = s->port;
  
  if (server.sin_addr.s_addr == -1) {
    socket_destroy(s, (int)_("Invalid Address"));
    return 0;
  }
  
#ifdef CONNECTION_DEBUG
  printf("connecting to port [%d]\n", ntohs(s->port));
#endif

  fcntl(sock, F_SETFL, O_NONBLOCK);
  res = connect(sock, (struct sockaddr *) &server, sizeof(server));

  if ((res == 0) || (errno == EINPROGRESS)) return 1;
  else {
    socket_destroy(s, (int)_("Could not connect socket"));
    return 0;
  }
}

int get_best_server(socket_t* socket) {
  connect_progress(_("Get best server"));
  if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
    return 0;
  }

  // receiving best server
  socket->input =
    gdk_input_add(socket->fd, GDK_INPUT_READ, 
		  GTK_SIGNAL_FUNC(get_best_server_read), socket);
  
  return 1;
}

gint send_command_real(napster_command_t* command) {
  unsigned char *msg;
  gint sendt = 0;
  gint16 length;
  gint16 length2;
  int res;

  if (!global.status.connection) return -1;

  length = strlen(command->data);
  msg = (gchar *)malloc((length + 5)*sizeof(gchar));
#ifdef PROTOCOL_DEBUG
  log("protocol", LOG_OTHER, "S: %5d L: %4d [%s]\n", command->type, 
      strlen(command->data), command->data);
#endif

  length2 = BSWAP16(length);
  command->type = BSWAP16(command->type);

  memcpy(msg, &length2, 2);
  memcpy(&msg[2], &command->type, 2);
  memcpy(&msg[4], command->data, length);
  msg[length+4] = 0;

  global.queue = g_list_remove(global.queue, command);

  while(sendt < (length+4)) {
    res = send(global.napster->fd, msg+sendt, length+4-sendt, 0);
    if (res == -1) {
      if (errno != EAGAIN) {
	g_warning("send error, errno:%d", errno);
	free(msg);
	return -1;
      }
    } else if (res == 0) {
      g_warning("0 bytes sent");
    } else {
      sendt += res;
    }
  }
  free(msg);
  
  return 0;
}

gint napster_send(gpointer data, gint source, 
		  GdkInputCondition condition) {
  int sent = 0;
  socket_t* socket = (socket_t*)data;
  napster_command_t* command;

  if (condition != GDK_INPUT_WRITE) {
    napster_disconnect(_("Could not send data to server"));
    return 1;
  }

  if (global.queue) {
    command = (napster_command_t*)(global.queue->data);
    if (send_command_real(command) == 0) {
      global.queue = g_list_remove(global.queue, command);
      sent = 1;
    }
  } else if (global.share_queue) {
    command = (napster_command_t*)(global.share_queue->data);
    if (send_command_real(command) == 0) {
      global.share_queue = g_list_remove(global.share_queue, command);
      sent = 1;
      if (!global.share_queue) refresh_channels();
    }
  }
  
  if (!sent) {
    printf("could not send data, but do not disconnect at the moment");
  }
  if (!global.queue && !global.share_queue) {
    gtk_input_remove(socket->timeout);
    socket->timeout =
      gtk_timeout_add(100, napster_check, socket);
  }
  return 1;
}

int napster_check(gpointer data) {
  socket_t* socket = (socket_t*)data;

  if (global.queue || global.share_queue) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		    GTK_SIGNAL_FUNC(napster_send), socket);
  }
  return 1;
}

int server_get_input(gpointer data, gint source, GdkInputCondition condition) {
  char *mess;
  gint16 type;
  socket_t* socket = (socket_t*)data;

  if (condition != GDK_INPUT_READ) {
    napster_disconnect(_("Server read error"));
    return 1;
  }
  
  mess = receive_command(socket, &type);

  if (!mess) return 1;

#ifdef PROTOCOL_DEBUG
  log("protocol", LOG_OTHER, "R: %5d L: %4d [%s]\n", type, strlen(mess), mess);
#endif
  socket->cnt = 0;
  handle_command(type, mess);
  
  return 1;
}

int server_login(gpointer data, gint source, 
		 GdkInputCondition condition) {
  GtkWidget* temp;
  int new_user;
  int port;
  char* text;
  char t2[2048];
  char t[1024];
  socket_t* socket = (socket_t*)data;
  server_t* server = (server_t*)(socket->data);
  
  if (condition != GDK_INPUT_WRITE) {
    napster_disconnect(_("Could not login"));
    return 1;
  }
  gdk_input_remove(socket->input);
  socket->input = -1;

  connect_progress(_("Logging in..."));
  if (global.connect_win) {
    temp = lookup_widget(global.connect_win, "new_user");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp)))
      new_user = 1;
    else new_user = 0;
  } else new_user = 0;

  if (!global.upload_socket) port = 0;
  else port = ntohs(global.upload_socket->port);
  
  strcpy(t2, server->nick);
  strcat(t2, " ");
  strcat(t2, server->passwd);
  strcat(t2, " ");
  sprintf(t, "%d \"", port);
  strcat(t2, t);
  if (global.clientinfo) {
    strcat(t2, global.clientinfo);
  } else {
    strcat(t2, "Lopster ");
    strcat(t2, VERSION);
  }
  sprintf(t, "\" %d", global.user.linespeed);
  strcat(t2, t);

  if (new_user) {
    temp = lookup_widget(global.connect_win, "entry35");
    text = gtk_entry_get_text(GTK_ENTRY(temp));
    if (strcmp(text, server->passwd)) {
      napster_disconnect(_("Password Retype Error"));
      return -1;
    }
    send_command(CMD_CLIENT_REGISTER, server->nick);
    strcat(t2, " ");
    if (strlen(global.user.email) > 0) {
      strcat(t2, global.user.email);
    } else {
      strcat(t2, "anon@anon");
    }
    send_command(CMD_CLIENT_LOGIN_REGISTER, t2);
  } else {
    send_command(CMD_CLIENT_LOGIN, t2);
  }

  socket->timeout =
    gtk_timeout_add(100, napster_check, socket);
  
  socket->input =
    gdk_input_add(socket->fd, GDK_INPUT_READ, 
		  GTK_SIGNAL_FUNC(server_get_input), socket);
  
  return 0;
}

////////////////////////////////////////////////////////////

void napster_reset() {
  GtkWidget* temp;
  char* t;
  GList* dlist;
  resume_t* resume;
  GtkCList* clist;
  int row;
  user_t* userinfo;
  
  // delete ignored users
  dlist = g_list_first(global.ignored_users);
  while (dlist) {
    t = (char*)(dlist->data);
    free(t);
    dlist = dlist->next;
  }
  g_list_free(global.ignored_users);
  global.ignored_users = NULL;

  if (global.ignore_win) {
    temp = lookup_widget(global.ignore_win, "ignore_list");
    gtk_clist_clear(GTK_CLIST(temp));
  }
  
  // reset hotlist
  clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
  for (row = 0; row < clist->rows; row++) {
    userinfo = gtk_clist_get_row_data(clist, row);
    hotlist_remove_user_files(userinfo);
  }
  while (global.browses) {
    hotlist_remove_user_files((user_t*)(global.browses->data));
    free(global.browses->data);
    global.browses = 
      g_list_remove(global.browses, global.browses->data);
  }

  // resetting incomplete files
  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t*)(dlist->data);
    if (resume->status < R_QUEUED) {
      resume->status = R_NONE;
      resume_list_update(resume);
    }
    resume_remove_search(resume);
  }

  // reset left searches
  while (global.searches) {
    search_remove((search_t*)(global.searches->data));
  }

  clones_clear();

  // deleting operator channels
  for (dlist = global.opchannel; dlist; dlist = dlist->next)
    free(dlist->data);
  g_list_free(global.opchannel);
  global.opchannel = NULL;
	      
  // clear queues
  for (dlist = global.queue; dlist; dlist = dlist->next)
    free(dlist->data);
  g_list_free(global.queue);
  global.queue = NULL;

  for (dlist = global.share_queue; dlist; dlist = dlist->next)
    free(dlist->data);
  g_list_free(global.share_queue);
  global.share_queue = NULL;

  global.status.searching = 0;
  global.status.whois_hide = 0;
  
  global.user.status = STATUS_INACTIVE;
  global.user.level = L_USER;
  
  update_user_stats();
  
  // unsharing library
  lib_unshare_flag(FLAG_SHARED, FLAG_TO_SHARE, NULL);
  
  // clear channel list
  temp = lookup_widget(global.win, "channel_list");
  gtk_clist_clear(GTK_CLIST(temp));

  // freeing server links
  free_links(global.links);
  global.links = NULL;

  setup_sensitive(0);
  if (global.napster && SERVER) SERVER->network = N_UNKNOWN;

  // leaving sockets untouched
  // leaving chat pages untouched
}

void napster_disconnect(char* message) {
  // message == NULL is user disconnect
  
  if (!global.napster) return;
  socket_end(global.napster, (int)message);
  if ((global.napster->timeout == -1) && 
      global.network.auto_reconnect && message) {
    connect_progress(_("Activate reconnect timer"));
    global.napster->timeout = 
      gtk_timeout_add(global.network.auto_reconnect*1000,
		      auto_reconnect, global.napster);
  }
}

int napster_connect(server_t* server) {
  unsigned long addr;
  GtkWidget* temp;

  if (global.status.connection) return -1;
  global.status.connection = 1;
  if (global.connect_win) {
    temp = lookup_widget(global.connect_win, "connection_ok");
    gtk_widget_set_sensitive(temp, FALSE);
  }

  if (!global.napster) {
    global.napster = socket_new(S_SERVER);
  }
  if (!server) return -1;

  global.napster->cnt = 0;
  if (global.napster->timeout >= 0) 
    gtk_timeout_remove(global.napster->timeout);

  if (global.napster->data) server_destroy(global.napster->data);
  global.napster->data = server;

  addr = resolv_address(server->address);
  if (addr) {
    global.napster->ip_long = addr;
  } else {
    connect_progress(_("Could not resolve address"));
    return -1;
  }
  
  global.napster->port = htons(server->port);
  
  if (server->meta) {
    return get_best_server(global.napster);
  } else {
    return connect_to_server(global.napster);
  }
  
  return 0;
}

int send_command(gint16 type, char* data) {
  napster_command_t* command;
  server_t* server;

  if (!global.status.connection) return -1;
  if (!global.napster) return -1;
  server = (server_t*)(global.napster->data);

  if ((server->network == N_NAPSTER) &&
      (type >= 1000)) {
    g_warning("do not send command >= 1000 on napster server");
    return 1;
  }

  command = (napster_command_t*)malloc(sizeof(napster_command_t));
  command->type = type;
  command->data = strdup(data);

  if ((type == CMD_CLIENT_ADD_FILE) || 
      (type == CMD_CLIENT_REMOVE_FILE) || 
      (type == CMD_CLIENT_SHARE_FILE))
    global.share_queue = g_list_append(global.share_queue, command);
  else
    global.queue = g_list_append(global.queue, command);
  return 0;
}

char* receive_command(socket_t* socket, gint16 *code) {
  int res;
  static int progress = 0;
  static unsigned char header[4];
  static unsigned char *buffer = NULL;
  static int length = 0;
  static gint16 c_type;
  static gint16 c_length;

  // this function allows static variables, would not work with
  // connections to more than one server

  if (progress == 0) {
    if (length < 4) {
      res = recv(socket->fd, header+length, 4-length, 0);
      if (res > 0) {
	length += res;
      } else if (res == 0) {
	length = progress = 0;
	napster_disconnect(_("Remote Close"));
	return NULL;
      } else {
	g_warning("1 recv error\n");
	length = progress = 0;
	napster_disconnect(strerror(errno));
	return NULL;
      }      
    }
    if (length == 4) {
      length = 0;
      progress++;
      memcpy(&c_length, header, 2);
      memcpy(&c_type, &header[2], 2);

      c_length = BSWAP16(c_length);
      c_type = BSWAP16(c_type);
      if (buffer) free(buffer);
      buffer = (unsigned char*)malloc(c_length+1);
    }
    return NULL;
  } 
  
  if (progress == 1) {
    if (length < c_length) {
      res = recv(socket->fd, buffer+length, c_length-length, 0);
      
      if (res > 0) {
	length += res;
      } else if (res == 0) {
	length = progress = 0;
	napster_disconnect(_("Remote Close"));
	return NULL;
      } else {
	g_warning("2 recv error\n");
	napster_disconnect(strerror(errno));
	return NULL;
      }
    }
    if (length == c_length) {
      buffer[length] = '\0';
      *code = c_type;
      progress = 0;
      length = 0;
      return buffer;
    }
    return NULL;
  }
  return NULL;
}

gint get_con_type(gpointer data, gint source, 
		  GdkInputCondition condition) {
  char buffer[1025];
  int cnt;
  socket_t* socket = (socket_t*)data;

  if (condition != GDK_INPUT_READ) {
    return 1;
  }

  gdk_input_remove(socket->input);
  cnt = 0;
  switch (cnt = recv(source, buffer, 1024, MSG_PEEK)) {
  case -1:
    socket_destroy(socket, 0);
    return 1;
  case 0:
    socket_destroy(socket, 0);
    return 1;
  default:
    break;
  }
  
  buffer[cnt] = 0;
  if (cnt < 3) {
    socket_destroy(socket, 0);
    return 1;
  }
  
#ifdef TRANSFER_DEBUG
  printf("in [%s]\n", buffer);
#endif
  if (!strncmp(buffer, "GETLIST", 7)) {
    recv(source, buffer, 7, 0);
    //    client_message(NULL, _("Someone is browsing your files"));
    socket->type = S_SHARE;
    socket->max_cnt = 50000;
    socket->input = 
      gdk_input_add(socket->fd, GDK_INPUT_WRITE, 
		    GTK_SIGNAL_FUNC(send_browse_nick), socket);
  } else if (!strncmp(buffer, "SENDLIST", 8)) {
    recv(source, buffer, 8, 0);
    //    client_message(NULL, _("Someone is sending his shared list"));
    socket->type = S_BROWSE;
    socket->max_cnt = 20000;
    socket->input = 
      gdk_input_add(socket->fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(get_browse_nick), socket);
  } else if (!strncmp(buffer, "GET", 3)) {
    recv(source, buffer, 3, 0);
#ifdef TRANSFER_DEBUG
    printf("got GET\n");
#endif
    socket->type = S_TRANSFER;
    socket->input = 
      gdk_input_add(socket->fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(get_upload_info), socket);
  } else if (!strncmp(buffer, "SEND", 4)) {
    recv(source, buffer, 4, 0);
#ifdef TRANSFER_DEBUG
    printf("got SEND\n");
#endif
    socket->type = S_TRANSFER;
    socket->input = 
      gdk_input_add(socket->fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(get_download_info), socket);
  } else {
#ifdef TRANSFER_DEBUG
    printf("**sending INVALID REQUEST [%s]\n", buffer);
#endif
    send(source, "INVALID REQUEST", 
	 strlen("INVALID REQUEST"), 0);
  }
  return 1;
}

int handle_upload_request(gpointer data, gint source, 
			  GdkInputCondition condition) {
  
  struct sockaddr_in from;
  int len;
  socket_t* socket;

  socket = socket_new(S_UNKNOWN);

  len = sizeof(from);
  socket->fd = accept(global.upload_socket->fd, (struct sockaddr *)&from, &len);
  if (socket->fd < 0) return 1;
  fcntl(socket->fd, F_SETFL, O_NONBLOCK);
  
  socket->ip_long = from.sin_addr.s_addr;
  socket->port = from.sin_port;

  send(socket->fd, "1", 1, 0);
  
#ifdef CONNECTION_DEBUG
  printf("accepted new connection\n");
#endif
  socket->input = 
    gdk_input_add(socket->fd, GDK_INPUT_READ, 
		  GTK_SIGNAL_FUNC(get_con_type), socket);
  
  return 1;
}

void create_upload_port(int port, int send) {
  int len = 1;
  int sock;
  int protocol;
  struct sockaddr_in server;
  char str[2048];
  GList* dlist;

  if (global.upload_socket)
    socket_destroy(global.upload_socket, 1);
  global.upload_socket = NULL;

  if (send) {
    if ((port == 0) || global.network.firewall) {
      global.network.port = port;
      send_command(CMD_CLIENT_CHANGE_DATA_PORT, "0");
      client_message(_("Information"), _("Data Port was set to 0"));
      setup_preferences(P_GENERAL);
      return;
    }
  }
  
  protocol = getprotobyname("IP")->p_proto;
  sock = socket(AF_INET, SOCK_STREAM, protocol);
  
  if (sock < 0) {
    port_dialog(_("Could not create socket"));
    create_upload_port(0, 1);
    return;
  }
  
  global.upload_socket = socket_new(S_DATA);

  global.upload_socket->ip_long = htonl(INADDR_ANY);
  global.upload_socket->fd = sock;
  global.upload_socket->cnt = 0;
  
  setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &len, sizeof(len));
  setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char *)&len, sizeof(len));
  
  if (g_list_find(global.allowed_ports, (gpointer)port)) {
    global.upload_socket->port = htons(port);
    server.sin_family = AF_INET;
    server.sin_addr.s_addr = global.upload_socket->ip_long;
    server.sin_port = global.upload_socket->port;
    if (bind(sock, (struct sockaddr *)&server, sizeof(server)) >= 0)
      goto success;
    client_message(_("Error"), _("Could not setup port %d"), port);
  } else {
    client_message(_("Error"), _("Port %d is not in the list of allowed ports"), port);
  }

  send = 1;

  for (dlist = global.allowed_ports; dlist; dlist = dlist->next) {
    global.upload_socket->port = htons((int)(dlist->data));
    server.sin_port = global.upload_socket->port;

    if (bind(sock, (struct sockaddr *)&server, sizeof(server)) >= 0)
      goto success;
  }    
  goto failure;
  
 success:
  global.network.port = ntohs(global.upload_socket->port);
  client_message(_("Information"), _("Data Port was set to %d"), 
		 global.network.port);
  client_message(_("Information"), _("At most %d incoming connections are allowed at a time."), 
		 10 + global.limit.max_uploads + global.limit.max_downloads);

  listen(sock, 10 + global.limit.max_uploads + global.limit.max_downloads);
#ifdef PROTOCOL_DEBUG
  log("protocol", LOG_OTHER, _("Listen for incoming connections on port [%d]\n"), global.upload_socket->port);
#endif
  setup_preferences(P_GENERAL);

  if (send) {
    sprintf(str, "%d", htons(global.upload_socket->port));
    send_command(CMD_CLIENT_CHANGE_DATA_PORT, str);
  }

  global.upload_socket->input = 
    gdk_input_add(sock, GDK_INPUT_READ,
		  GTK_SIGNAL_FUNC(handle_upload_request), 
		  global.upload_socket);

  setup_preferences(P_GENERAL);
  return;

 failure:
  port_dialog(_("No available Port found"));
  create_upload_port(0, 1);
  return;
}

void socket_insert_clist(socket_t* socket) {
  GtkCList* clist;
  int row;

  if (!global.socket_win) return;

  strcpy(tstr[0], socket_get_name(socket));
  sprintf(tstr[1], "%s", ntoa(socket->ip_long));
  sprintf(tstr[2], "%d", ntohs(socket->port));
  sprintf(tstr[3], "%d/%d", socket->cnt, socket->max_cnt);
  if (socket->fd == -1) strcpy(tstr[4], _("inactive"));
  else sprintf(tstr[4], "%d", socket->fd);
  if (socket->timeout == -1) strcpy(tstr[5], _("inactive"));
  else sprintf(tstr[5], "%d", socket->timeout);
  clist = GTK_CLIST(lookup_widget(global.socket_win, "clist20"));
  row = gtk_clist_append(clist, list);
  gtk_clist_set_row_data (clist, row, (gpointer)socket);
}

void socket_remove_clist(socket_t* socket) {
  GtkCList* clist;
  int row;

  if (!global.socket_win) return;

  clist = GTK_CLIST(lookup_widget(global.socket_win, "clist20"));
  row = gtk_clist_find_row_from_data(clist, socket);
  if (row < 0) return;

  gtk_clist_remove(clist, row);

  clist = GTK_CLIST(lookup_widget(global.socket_win, "clist21"));
  if (gtk_object_get_data(GTK_OBJECT(clist), "socket") == socket) {
    socket_show_clist(NULL);
  }
}

void socket_update_clist(socket_t* socket) {
  GtkCList* clist;
  int row;
  char str[1024];
  
  if (!global.socket_win) return;

  clist = GTK_CLIST(lookup_widget(global.socket_win, "clist20"));
  row = gtk_clist_find_row_from_data(clist, socket);
  if (row < 0) {
    socket_insert_clist(socket);
    return;
  }

  sprintf(str, "%s", ntoa(socket->ip_long));
  gtk_clist_set_text(clist, row, 1, str);
  sprintf(str, "%d", ntohs(socket->port));
  gtk_clist_set_text(clist, row, 2, str);
  sprintf(str, "%d/%d", socket->cnt, socket->max_cnt);
  gtk_clist_set_text(clist, row, 3, str);
  if (socket->fd == -1) strcpy(str, _("inactive"));
  else sprintf(str, "%d", socket->fd);
  gtk_clist_set_text(clist, row, 4, str);
  if (socket->timeout == -1) strcpy(str, _("inactive"));
  else sprintf(str, "%d", socket->timeout);
  gtk_clist_set_text(clist, row, 5, str);

  clist = GTK_CLIST(lookup_widget(global.socket_win, "clist21"));
  if (gtk_object_get_data(GTK_OBJECT(clist), "socket") == socket)
    socket_show_clist(socket);
}

void copy_string(char* dest, const char* source) {
  if (source) strcpy(dest, source);
  else strcpy(dest, "_????_");
}

void socket_show_clist(socket_t* socket) {
  GtkCList* clist;
  server_t* server;
  transfer_t* transfer;

  if (!global.socket_win) return;
  if (!socket) return;

  clist = GTK_CLIST(lookup_widget(global.socket_win, "clist21"));
  gtk_clist_freeze(clist);

  gtk_clist_clear(clist);
  gtk_object_set_data(GTK_OBJECT(clist), "socket", socket);

  if (socket) {
    switch (socket->type) {
    case S_BROWSE:
      copy_string(tstr[0], _("Browsing"));
      copy_string(tstr[1], (char*)(socket->data));
      gtk_clist_append(clist, list);
      break;
    case S_SERVER:
      server = (server_t*)(socket->data);
      if (!server) {
	copy_string(tstr[0], _("Server"));
	copy_string(tstr[1], NULL);
	gtk_clist_append(clist, list);
	break;
      }
      copy_string(tstr[0], _("Address"));
      copy_string(tstr[1], server->address);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Description"));
      copy_string(tstr[1], server->description);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Port"));
      sprintf(tstr[1], "%d", server->port);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Nick"));
      copy_string(tstr[1], server->nick);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Metaserver"));
      copy_string(tstr[1], (server->meta)?_("Yes"):_("No"));
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Type"));
      copy_string(tstr[1], Network[server->network]);
      gtk_clist_append(clist, list);
      if (server->network == N_OPENNAP) {
	copy_string(tstr[0], _("Version"));
	sprintf(tstr[1], "%d.%d", server->major, server->minor);
	gtk_clist_append(clist, list);
      }
      break;
    case S_TRANSFER:
      transfer = (transfer_t*)(socket->data);
      if (!transfer) {
	copy_string(tstr[0], _("Transfer"));
	copy_string(tstr[1], NULL);
	gtk_clist_append(clist, list);
	break;
      }
      copy_string(tstr[0], _("Type"));
      if (transfer->type == T_DOWNLOAD)
	copy_string(tstr[1], _("Download"));
      else if (transfer->type == T_UPLOAD)
	copy_string(tstr[1], _("Upload"));
      else copy_string(tstr[1], NULL);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Winname"));
      copy_string(tstr[1], transfer->winname);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Longname"));
      copy_string(tstr[1], transfer->longname);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Shortname"));
      copy_string(tstr[1], transfer->shortname);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Download folder"));
      copy_string(tstr[1], transfer->download_dir);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("User"));
      copy_string(tstr[1], transfer->user);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Linespeed"));
      copy_string(tstr[1], LineSpeed(transfer->linespeed));
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Progress"));
      sprintf(tstr[1], "%ld/%ld [%.2f%%]", 
	      transfer->progress, transfer->size,
	      (transfer->size)?((double)(transfer->progress)*100.0/(double)(transfer->size)):.0);
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Status"));
      copy_string(tstr[1], status_names(transfer->status));
      gtk_clist_append(clist, list);
      copy_string(tstr[0], _("Dcc"));
      copy_string(tstr[1], (transfer->is_dcc)?"Yes":"No");
      gtk_clist_append(clist, list);
      if (transfer->user_info) {
	copy_string(tstr[0], _("Userinfo (cur/max/refs)"));
	sprintf(tstr[1], "%d/%d/%d", transfer->user_info->cur,
		transfer->user_info->max,
		transfer->user_info->cnt);
	gtk_clist_append(clist, list);
      }
      break;
    default:
      copy_string(tstr[0], _("No data"));
      copy_string(tstr[1], "");
      gtk_clist_append(clist, list);
      break;
    }
  }

  gtk_clist_thaw(clist);
}
