/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: network.c,v 1.24 2002/11/22 21:49:01 martine Exp $
 */

#include "config.h"
#include <gtk/gtk.h>

#include <errno.h>

#ifndef G_OS_WIN32
#include <unistd.h> /* read, etc. */
#else
#define WIN32
#include <io.h> /* for _pipe() */
#include <fcntl.h> /* for _pipe() flags */
#endif

#include <stdlib.h> /* atoi */
#include <sys/types.h>
#include <signal.h>

#include <curl/curl.h>

#include "protocol.h"
#include "util.h"
#include "conf.h"
#include "network.h"
#include "icons.h"

/* delay in ms between frames of the throbber */
#define THROBBER_FRAME_DELAY 400

/* maximal response size */
#define RESPONSE_SIZE 2048

#ifdef G_OS_WIN32
#define THREAD_STACK_SIZE 512*1024 /* 512K stack */
#define NETWORK_TIMEOUT 100*1000 /* 100-second timeout */
/* I'm not sure you use timeouts in the Unix version, if you don't,
   just set the above to INFINITE */
#endif

/* internal data for the forked curl process. */
typedef struct {
	CURL *curl;
	GString *responsedata;
#ifndef G_OS_WIN32
	int statuspipe;
#endif /* G_OS_WIN32 */
} curl_request;

/* Throbber is only used in in net_request.
 * throbber data separated here for clarity. */
typedef struct {
	int        state;
	guint      tag;
	GdkPixbuf *pb[THROBBER_COUNT];
	GtkWidget *image;
} Throbber;

/* per-netrequest window and status. */
typedef struct {
	GtkWidget *win, *label, *button, *progress;

	guint pipe_tag;
	int pipefds[2];

	char *responsestr;
	char errorbuf[CURL_ERROR_SIZE];
	int cancelled;

#ifndef G_OS_WIN32
	pid_t pid;
#else
	HANDLE h_thread;
#endif

	Throbber throbber;
} net_request;

#ifdef G_OS_WIN32
typedef struct curl_thread_params_t
{
	net_request* nr;
	NetRequest* request;
	GString* response;
} curl_thread_params;
#endif

static gboolean
throbber_cb(net_request *nr) {
	Throbber *t = &nr->throbber;
	t->state = (t->state + 1) % THROBBER_COUNT;
	gtk_image_set_from_pixbuf(GTK_IMAGE(t->image), t->pb[t->state]);
	return TRUE;
}
static void
throbber_start(net_request *nr) {
	if (nr->throbber.tag)
		return;

	nr->throbber.tag = gtk_timeout_add(THROBBER_FRAME_DELAY,
			(GSourceFunc)throbber_cb, nr);
}
static void
throbber_stop(net_request *nr) {
	if (!nr->throbber.tag)
		return;
	gtk_timeout_remove(nr->throbber.tag);
	nr->throbber.tag = 0;
}

/* errors can occur in a number of places, and each needs to be reported:
 * - curl errors
 *   (curl response code / curl error buffer)
 * - system errors, like pipe() 
 *   (return value / g_strerror(errno))
 * - network request went through, but lj messed up
 *   (lj_protocol_parse_response() == NULL / "LJ messed up")
 * - everything went ok on the network level but still failed (bad password?)
 *   (!net_result_succeeded() / net_result_get(result, "errmsg"))
 */
static void 
show_error(net_request *nr, const char *fmt, ...) {
	char buf[1024];
	va_list ap;

	va_start(ap, fmt);
	g_vsnprintf(buf, 1024, fmt, ap);
	va_end(ap);

	if (nr->win) {
		gtk_label_set_text(GTK_LABEL(nr->label), buf);

		throbber_stop(nr);
		gtk_image_set_from_stock(GTK_IMAGE(nr->throbber.image),
				GTK_STOCK_DIALOG_ERROR, GTK_ICON_SIZE_DIALOG);

		gtk_widget_hide(nr->progress);
		gtk_main();
	} else {
		g_print(_("Error: %s\n"), buf);
	}
}

static void 
cancel_cb(net_request *nr) {
	nr->cancelled = TRUE;
#ifndef G_OS_WIN32
	if (nr->pid) {
		close(nr->pipefds[0]);
		close(nr->pipefds[1]);
		kill(nr->pid, SIGKILL);
		nr->pid = 0;
	}
#else
	if (nr->h_thread) {
		TerminateThread(nr->h_thread, 1);
		nr->h_thread = NULL;
	}
#endif
	gtk_main_quit();
}

static void
destroy_cb(GtkWidget *w, net_request *nr) {
	int i;
	throbber_stop(nr);
	for (i = 0; i < THROBBER_COUNT; i++)
		g_object_unref(G_OBJECT(nr->throbber.pb[i]));
}

static void 
create_win(net_request *nr, const char *title, GtkWidget *parent) {
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *hbox;

	nr->win = gtk_window_new(GTK_WINDOW_TOPLEVEL);

	gtk_window_set_title(GTK_WINDOW(nr->win), title);
	if (parent)
		gtk_window_set_transient_for(GTK_WINDOW(nr->win), GTK_WINDOW(parent));
	gtk_window_set_decorated(GTK_WINDOW(nr->win), FALSE);
	gtk_window_set_modal(GTK_WINDOW(nr->win), TRUE);
	gtk_window_set_default_size(GTK_WINDOW(nr->win), 200, -1);
	gtk_window_set_position(GTK_WINDOW(nr->win), GTK_WIN_POS_CENTER_ON_PARENT);
	g_signal_connect(G_OBJECT(nr->win), "destroy",
			G_CALLBACK(destroy_cb), nr);

	lj_win_set_icon(nr->win);

	vbox = gtk_vbox_new(FALSE, 5); 
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);

	hbox = gtk_hbox_new(FALSE, 5); 

	icons_load_throbber(nr->throbber.pb);
	nr->throbber.image = gtk_image_new_from_pixbuf(nr->throbber.pb[0]);
	gtk_box_pack_start(GTK_BOX(hbox), nr->throbber.image, FALSE, FALSE, 0);
	throbber_start(nr);

	nr->label = gtk_label_new(title);
	gtk_label_set_line_wrap(GTK_LABEL(nr->label), TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), nr->label, TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 5);
	nr->progress = gtk_progress_bar_new();
	gtk_widget_set_size_request(nr->progress, 50, -1);
	gtk_box_pack_start(GTK_BOX(hbox), nr->progress, 
			TRUE, TRUE, 0);

	nr->button = gtk_button_new_with_label(_("  Cancel  "));
	g_signal_connect_swapped(G_OBJECT(nr->button), "clicked",
			G_CALLBACK(cancel_cb), nr);
	gtk_box_pack_end(GTK_BOX(hbox), nr->button, 
			FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show_all(frame);

	gtk_container_add(GTK_CONTAINER(nr->win), frame);

	gtk_widget_hide(nr->progress);
	gtk_widget_show(nr->win);
}

typedef enum {
	/* these used to be SP_foo, but that conflicted with windows header. */
	STATUS_NULL,
	STATUS_SUCCESS,
	STATUS_ERROR,
	STATUS_PROGRESS
} StatusPipeType;

static int
pipe_write(int pipe, StatusPipeType type, int len, void *data) {
	char t = type;
	if (write(pipe, &t, 1) < 1)
		return -1;
	if (write(pipe, &len, sizeof(int)) < sizeof(int))
		return -1;
	if (data) {
		if (write(pipe, data, len) < len)
			return -1;
	}
	return 0;
}

static size_t 
curlwrite_cb(void *ptr, size_t size, size_t nmemb, void *data) {
	curl_request *cr = data;
	double contentlength, progress = 0;

	g_string_append_len(cr->responsedata, ptr, size*nmemb);

	curl_easy_getinfo(cr->curl, 
			CURLINFO_CONTENT_LENGTH_DOWNLOAD, &contentlength);
	if (contentlength > 0)
		progress = (double)cr->responsedata->len/contentlength;

#ifndef G_OS_WIN32
	if (cr->statuspipe)
		if (pipe_write(cr->statuspipe, STATUS_PROGRESS, sizeof(double), &progress) < 0)
			return -1; /* FIXME: what does curl do
						  when the write callback returns -1?
						  hopefully reports the error,
						  just as it would with the normal
						  write callback (which I assume is fwrite).  */
#else
		/* FIXME: get the statuspipe to work in Win32 -- the problem is the
		 * pipe_cb() callback being explicitly called by curl_thread_func().
		 * Some sort of loop, possibly with Sleep() calls, may do the trick.
		 * (ijon) */
#endif

	return size*nmemb;
}

/*
 * see comment in run_curl_request().
static int
curl_progress_cb(void *cp, size_t dlt, size_t dln, size_t ult, size_t uln) {
	g_print("progress: %d %d %d %d\n", dlt, dln, ult, uln);
	return 0;
}*/

static int
run_curl_request(net_request *nr, NetRequest *request, int statuspipe, GString *result) {
	/* returns the server response to the request. */
	CURL *curl;
	CURLcode curlres;
	char urlreq[1024], proxyuserpass[1024];
	curl_request cr = {0};
	GString *requestdata;

	curl = curl_easy_init();
	if (curl == NULL) {
		g_snprintf(nr->errorbuf, 40, _("Unable to intialize CURL"));
		return -1;
	}

	curl_easy_setopt(curl, CURLOPT_VERBOSE, conf.options.netdump != 0);
	curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1);
	curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, nr->errorbuf);

	/*
	 * curl's progress function is mostly useless; we instead track writes.
	 *
	 * curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, curl_progress_cb);
	 * curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, nr);
	 */
	
	g_snprintf(urlreq, sizeof(urlreq), 
			"%s/interface/flat", conf_cur_server()->url);
	curl_easy_setopt(curl, CURLOPT_URL, urlreq);

	if (conf_cur_user()->fastserver) {
		curl_easy_setopt(curl, CURLOPT_COOKIE, "ljfastserver=1;");
	}

	if (conf.options.useproxy) {
		curl_easy_setopt(curl, CURLOPT_PROXY, conf.proxy);
		if (conf.options.useproxyauth) {
			g_snprintf(proxyuserpass, sizeof(proxyuserpass), "%s:%s", 
					conf.proxyuser, conf.proxypass);
			curl_easy_setopt(curl, CURLOPT_PROXYUSERPWD, proxyuserpass);
		}
	}

	requestdata = lj_protocol_request_to_string(request);
	curl_easy_setopt(curl, CURLOPT_POSTFIELDS, requestdata->str);
	curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, requestdata->len-1);
	curl_easy_setopt(curl, CURLOPT_POST, 1);

	curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curlwrite_cb);
	cr.curl = curl;
	cr.responsedata = result;
#ifndef G_OS_WIN32
	cr.statuspipe = statuspipe;
#endif
	curl_easy_setopt(curl, CURLOPT_FILE, &cr);
	
	curlres = curl_easy_perform(curl);
	curl_easy_cleanup(curl);

	g_string_free(requestdata, TRUE);

	if (curlres != CURLE_OK)
		return -1;

	if (conf.options.netdump) 
		fprintf(stderr, _("Response: [%s]\n"), result->str);

	return 0;
}

static void
pipe_cb(net_request *nr, gint pipe, GdkInputCondition cond) {
	char t;
	int len;
	int rlen = 0;
	double progress;

	len = read(pipe, &t, 1);
	if (len == 0) {
		show_error(nr, _("Pipe unexpectedly closed."));
		return;
	}
	if (len < 1) {
		show_error(nr, _("Error reading pipe (read: %s)."), g_strerror(errno));
		return;
	}

	if (read(pipe, &len, sizeof(int)) < sizeof(int)) {
		show_error(nr, _("Error reading pipe (read: %s)."), g_strerror(errno));
		return;
	}
	switch (t) {
		case STATUS_SUCCESS:
			nr->responsestr = g_new0(char, len+1);
			do {
				rlen += read(pipe, nr->responsestr+rlen, len-rlen);
			} while (rlen < len);
			nr->responsestr[len] = 0;
#ifndef G_OS_WIN32
			nr->pid = 0;
#else
			nr->h_thread = 0; /* disable cancellation */
#endif
			close(pipe);
			gtk_main_quit();
			break;
		case STATUS_ERROR:
			do {
				rlen += read(pipe, nr->errorbuf+rlen, len-rlen);
			} while (rlen < len);
			nr->errorbuf[len] = 0;
#ifndef G_OS_WIN32
			nr->pid = 0;
#else
			nr->h_thread = 0; /* disable cancellation */
#endif
			close(pipe);
			show_error(nr, _("Request error: %s."), nr->errorbuf);
			gtk_main_quit();
			break;
		case STATUS_PROGRESS:
			read(pipe, &progress, sizeof(double));
			if (nr->win) {
				gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(nr->progress), progress);
				gtk_widget_show(nr->progress);
			} else if (conf.options.netdump) {
				g_print(_("Progress: %.2f%%\n"), progress*100.0);
			}
			break;
		default:
			show_error(nr, _("Bad data from reading pipe (unexpected status %d)."), t);
	}
}

#ifndef G_OS_WIN32
static NetResult*
run_curl_request_fork(net_request *nr, NetRequest *request) {
	NetResult *result;

	/* fork, run the request, then pipe the data back out of the fork. */
	if (pipe(nr->pipefds) < 0) {
		show_error(nr, _("Error creating pipe (pipe(): %s)."), g_strerror(errno));
		return NULL;
	}
	nr->pid = fork();
	if (nr->pid < 0) {
		show_error(nr, _("Error forking (fork(): %s)."), g_strerror(errno));
		return NULL;
	} else if (nr->pid == 0) { /* child. */
		GString *response;

		response = g_string_sized_new(RESPONSE_SIZE);
		if (run_curl_request(nr, request, nr->pipefds[1], response) < 0) {
			pipe_write(nr->pipefds[1], STATUS_ERROR, CURL_ERROR_SIZE, nr->errorbuf);
		} else {
			pipe_write(nr->pipefds[1], STATUS_SUCCESS, response->len, response->str);
		}
		g_string_free(response, TRUE);

		close(nr->pipefds[0]);
		close(nr->pipefds[1]);

		_exit(0);
	} 
	/* otherwise, we're the parent. */
	nr->pipe_tag = gtk_input_add_full(nr->pipefds[0], GDK_INPUT_READ, 
			(GdkInputFunction)pipe_cb, NULL, nr, NULL);

	gtk_main(); /* wait for the response. */
	gtk_input_remove(nr->pipe_tag);
	nr->pipe_tag = 0;
	nr->pid = 0;

	close(nr->pipefds[0]);
	close(nr->pipefds[1]);
	if (nr->cancelled) {
		if (nr->responsestr) g_free(nr->responsestr);
		return NULL;
	}
	if (nr->responsestr == NULL)
		return NULL;
	result = lj_protocol_parse_response(nr->responsestr);
	g_free(nr->responsestr);
	return result;
}

#else /* use ijon's Win32 threaded request */

unsigned long __stdcall curl_thread_func(void* params) 
{
	curl_thread_params* p = params;
	if (run_curl_request(p->nr, p->request, p->nr->pipefds[1], p->response) < 0) {
		pipe_write(p->nr->pipefds[1], STATUS_ERROR, CURL_ERROR_SIZE, p->nr->errorbuf);
	} else {
		pipe_write(p->nr->pipefds[1], STATUS_SUCCESS, p->response->len, p->response->str);
	}
	g_string_free(p->response, TRUE);
	close(p->nr->pipefds[1]);
	pipe_cb(p->nr, p->nr->pipefds[0], GDK_INPUT_READ);
	return 0;
}

static NetResult*
run_curl_request_fork(net_request *nr, NetRequest *request) {
	NetResult *result;
	GString *response;
	DWORD thread_id;
	curl_thread_params params;
	response = g_string_sized_new(RESPONSE_SIZE);

	/* prepare pipe and params */
	if (pipe(nr->pipefds) < 0) {
		show_error(nr, "Error creating pipe (pipe(): %s).", g_strerror(errno));
		return NULL;
	}
	params.nr = nr;
	params.request = request;
	params.response = response;
	
	/* launch a thread */
	nr->h_thread = CreateThread(NULL, THREAD_STACK_SIZE,
			&curl_thread_func, &params, 0, &thread_id);
	if (nr->h_thread == NULL) {
		show_error(nr, "Error creating thread (%d).", GetLastError());
		/* If we want error descriptions, we can use FormatMessage */
		return NULL;
	}

	/* wait for response */
	/*nr->pipe_tag = gtk_input_add_full(nr->pipefds[0], GDK_INPUT_READ,
		(GdkInputFunction)pipe_cb, NULL, nr, NULL);*/
	gtk_main();

	/* clean up */
	/*gtk_input_remove(nr->pipe_tag);*/
	nr->pipe_tag = 0;
	nr->h_thread = 0;
	/*close(nr->pipefds[0]); */ /* closed by pipe_cb */
	if (nr->cancelled) {
		if (nr->responsestr) g_free(nr->responsestr);
		return NULL;
	}
	if (nr->responsestr == NULL)
		return NULL;
	result = lj_protocol_parse_response(nr->responsestr);
	g_free(nr->responsestr);
	return result;

}
#endif /* G_OS_WIN32 */

static NetResult*
run_curl_request_nofork(net_request *nr, NetRequest *request) {
	GString *response;
	NetResult *result;

	response = g_string_sized_new(2048);
	if (run_curl_request(nr, request, 0, response) < 0) {
		show_error(nr, _("Request error: %s."), nr->errorbuf);
		return NULL;
	} else {
		result = lj_protocol_parse_response(response->str);
	}
	g_string_free(response, TRUE);

	return result;
}

NetResult*
net_request_run_silent(NetRequest *request) {
	net_request actual_nr = {0}, *nr = &actual_nr;
	NetResult *result = NULL;

	if (conf.options.nofork || app.cli) {
		result = run_curl_request_nofork(nr, request);
	} else {
		result = run_curl_request_fork(nr, request);
	}
	return result;
}

NetResult*
net_request_run(GtkWidget *parent, const char *title, NetRequest *request) {
	net_request nr_actual = {0}, *nr = &nr_actual;
	NetResult *result = NULL;

	create_win(nr, title, parent);

	while (gtk_events_pending())
		gtk_main_iteration();
 
	if (conf.options.nofork) {
		result = run_curl_request_nofork(nr, request);
	} else {
		result = run_curl_request_fork(nr, request);
	}

	if (result && !net_result_succeeded(result)) {
		const char *msg = net_result_get(result, "errmsg");
		if (!msg)
			msg = _("Unable to parse server response"); /* FIXME is this right? */
		show_error(nr, _("Request failed: %s."), msg);
	}

	gtk_widget_destroy(nr->win);

	return result;
}

NetRequest*
net_request_new(const char *mode) {
	NetRequest *request = lj_protocol_request_new(mode, 
			conf_cur_user()->username, conf_cur_user()->password, conf.usejournal);

	return request;
}

NetResult*
net_request_run_cli(NetRequest *request) {
	net_request actual_nr = {0}, *nr = &actual_nr;
	NetResult *result = NULL;

	result = run_curl_request_nofork(nr, request);
	if (result && !net_result_succeeded(result)) {
		char *msg = net_result_get(result, "errmsg");
		show_error(nr, _("Request failed: %s."), msg);
	}
	return result;
}

gboolean
net_result_succeeded(NetResult *result) {
	return lj_protocol_request_succeeded(result);
}

char*
net_result_getf(NetResult* result, const char *key, ...) {
	char buf[100];
	va_list ap;

	va_start(ap, key);
	g_vsnprintf(buf, 100, key, ap);
	va_end(ap);
	return net_result_get(result, buf); 
}

int
net_result_geti(NetResult *result, const char *key) {
	char *val;
	val = net_result_get(result, key);
	if (val)
		return atoi(val);
	return 0;
}

char*
net_result_get_prefix(NetResult *result, const char *prefix, const char *key) {
	char buf[100];
	g_snprintf(buf, 100, "%s%s", prefix, key);
	return net_result_get(result, buf);
}
