/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: login.c,v 1.24 2002/11/26 09:39:16 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <gtk/gtklist.h>
#include <stdlib.h> /* atoi */

#include "conf.h"
#include "util.h"
#include "network.h"
#include "lj.h"
#include "menu.h"

#include "icons.h"
#include "login.h"
 
typedef struct {
	GtkWidget *win;
	LJWin *ljw;
	GtkWidget *mserver, *eusername, *epassword;
	GtkWidget *cusername, *cfastserver;
	GtkWidget *cruser, *crpassword;
} login_dlg;

static void populate_server_list(login_dlg *ldlg);
static void populate_user_list(login_dlg *ldlg);

static void 
username_changed(GtkWidget *w, login_dlg *ldlg) {
	const char *text;
	User *u;

	text = gtk_entry_get_text(GTK_ENTRY(w));
	u = conf_user_by_username(conf_cur_server(), text);
	if (u == NULL) {
		gtk_entry_set_text(GTK_ENTRY(ldlg->epassword), "");
		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(ldlg->cfastserver), FALSE);
		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(ldlg->cruser), FALSE);
		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(ldlg->crpassword), FALSE);
	} else {
		if (u->password)
			gtk_entry_set_text(GTK_ENTRY(ldlg->epassword), u->password);
		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(ldlg->cfastserver), u->fastserver);
		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(ldlg->cruser), u->remember_user);
		gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(ldlg->crpassword), u->remember_password);
	}
}

static void
read_moods(NetResult *result, Server *s) {
	char *value;
	int moodcount;
	int i;

	/* grab moods */
	moodcount = net_result_geti(result, "mood_count");

	for (i = 1; i < moodcount+1; i++) { /* 1-based list */
		char buf[30];
		Mood *m;

		m = g_new0(Mood, 1);

		g_snprintf(buf, 30, "mood_%d_", i);

		value = net_result_get_prefix(result, buf, "id");
		if (!value) continue;
		m->id = atoi(value);
		m->name = g_strdup(net_result_get_prefix(result, buf, "name"));

		s->moods = g_list_insert_sorted(s->moods, m, 
				(GCompareFunc)conf_nid_compare_alpha);
	}
}

static void
read_pickws(NetResult *result, User *u) {
	int pickwcount;
	int i;

	/* first, free all existing pickws. */
	if (u->pickws) {
		g_list_foreach(u->pickws, (GFunc)g_free, NULL);
		g_list_free(u->pickws);
		u->pickws = NULL;
	}

	/* grab picture keywords */
	pickwcount = net_result_geti(result, "pickw_count");

	for (i = 1; i < pickwcount+1; i++) { /* 1-based list */
		char *kw = net_result_getf(result, "pickw_%d", i);
		if (kw)
			u->pickws = g_list_append(u->pickws, g_strdup(kw));
	}
}

static void
read_friendgroups(NetResult *result, User *u) {
	int fgmax;
	int i;

	/* first, free all existing friendgroups. */
	if (u->friendgroups) {
		g_list_foreach(u->friendgroups, (GFunc)friend_group_free, NULL);
		g_list_free(u->friendgroups);
		u->friendgroups = NULL;
	}

	/* then read the new ones. */
	fgmax = net_result_geti(result, "frgrp_maxnum");

	for (i = 1; i <= fgmax; i++) {
		char *fgname = net_result_getf(result, "frgrp_%d_name", i);
		if (fgname) {
			FriendGroup *fg;
			char *pub;
			fg = g_new0(FriendGroup, 1);
			fg->id = i;
			fg->name = g_strdup(fgname);
			pub = net_result_getf(result, "frgrp_%d_public", i);
			fg->ispublic = pub && !(g_ascii_strcasecmp(pub, "1"));

			u->friendgroups = g_list_append(u->friendgroups, fg);
		}
	}
}

static void
read_usejournals(NetResult *result, User *u) {
	int i, accesscount;
	char key[30];

	/* first, free all existing usejournals. */
	if (u->usejournals) {
		g_list_foreach(u->usejournals, (GFunc)g_free, NULL);
		g_list_free(u->usejournals);
		u->usejournals = NULL;
	}

	/* then read the new ones. */
	accesscount = net_result_geti(result, "access_count");

	for (i = 1; i < accesscount+1; i++) {
		char *journal;
		g_snprintf(key, 30, "access_%d", i);
		journal = net_result_get(result, key);
		if (journal)
			u->usejournals = g_list_append(u->usejournals, g_strdup(journal));
	}
}

static User*
get_selected_user(login_dlg *ldlg) {
	char *username;
	User *u;

	username = gtk_editable_get_chars(GTK_EDITABLE(ldlg->eusername), 0, -1);

	if ((u = conf_user_by_username(conf_cur_server(), username)) == NULL) {
		/* this is a previously-unknown user. */
		u = g_new0(User, 1);
		u->username = username;
		conf_server_add_user(conf_cur_server(), u);
	} else {
		g_free(username);
	}

	string_replace(&u->password, 
			gtk_editable_get_chars(GTK_EDITABLE(ldlg->epassword), 0, -1));

	u->fastserver = 
		gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ldlg->cfastserver));

	u->remember_user = 
		gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ldlg->cruser));
	u->remember_password = 
		gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ldlg->crpassword));

	return u;
}

static NetRequest*
build_login_request(void) {
	NetRequest *request;
	request = net_request_new("login");

#ifndef G_OS_WIN32
	net_request_copys(request, "clientversion", "GTK2-LogJam/" VERSION);
#else
	net_request_copys(request, "clientversion", "WinGTK2-LogJam/" VERSION);
#endif
	net_request_seti(request, "getmoods", conf_mood_get_last_cached(conf_cur_server()));
	net_request_seti(request, "getmenus", 1);
	net_request_seti(request, "getpickws", 1);

	return request;
}

static gboolean 
login_run(login_dlg *ldlg) {
	NetRequest *request;
	NetResult *result;
	User *u;
	char *value;

	u = get_selected_user(ldlg);
	conf_cur_server()->usercur = u;

	request = build_login_request();
	result = net_request_run(ldlg->win, _("Logging in..."), request);
	net_request_free(request);

	if (!net_result_succeeded(result)) {
		conf.loginok = FALSE;
		conf_cur_server()->usercur = NULL;
		net_result_free(result);
		return FALSE;
	}

	conf.loginok = TRUE;

	string_replace(&u->fullname, g_strdup(net_result_get(result, "name")));

	u->fastserver = (net_result_geti(result, "fastserver") != 0);

	read_moods(result, conf_cur_server());
	read_pickws(result, u);
	read_friendgroups(result, u);
	read_usejournals(result, u);
	menu_update(ldlg->ljw, result);

	/* if there's a message, display it. */
	if ((value = net_result_get(result, "message")) != NULL)
		lj_messagebox(ldlg->win, _("LiveJournal Message"), value);

	net_result_free(result);
	gtk_widget_destroy(ldlg->win);
	return TRUE;
}

static gboolean
autolog_cb(login_dlg *ldlg) {
	gtk_dialog_response(GTK_DIALOG(ldlg->win), GTK_RESPONSE_OK);
	return FALSE;
}

static void
server_changed_cb(GtkOptionMenu *menu, login_dlg *ldlg) {
	GList *l;
	
	l = g_list_nth(conf.servers, gtk_option_menu_get_history(menu));
	if (l && l->data) {
		conf_cur_server() = l->data;
		populate_user_list(ldlg);
		if (conf_cur_server()->users)
			gtk_entry_set_text(GTK_ENTRY(ldlg->eusername), ((User*)conf_cur_server()->users->data)->username);
		else
			gtk_entry_set_text(GTK_ENTRY(ldlg->eusername), "");
		username_changed(ldlg->eusername, ldlg);
	}
}

static void
addedit_server_cb(GtkWidget *w, login_dlg *ldlg) {
	login_server_manager_run(ldlg->win);
	conf_cur_server() = NULL;
	conf_verify_a_server_exists();
	populate_server_list(ldlg);
}

static void
populate_server_list(login_dlg *ldlg) {
	GtkWidget *menu, *item;
	GList *l;
	Server *s;
	int i = 0, history = -1;

	menu = gtk_menu_new();
	for (l = conf.servers; l != NULL; l = l->next) {
		s = l->data;
		item = gtk_menu_item_new_with_label(s->name);
		gtk_widget_show(item);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
		if (conf_cur_server() && (g_ascii_strcasecmp(s->name, conf_cur_server()->name) == 0))
			history = i;
		i++;
	}

	item = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	item = gtk_menu_item_new_with_label("Add/Edit Servers...");
	g_signal_connect(G_OBJECT(item), "activate",
			G_CALLBACK(addedit_server_cb), ldlg);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	
	gtk_widget_show_all(menu);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(ldlg->mserver), menu);
	if (history >= 0)
		gtk_option_menu_set_history(GTK_OPTION_MENU(ldlg->mserver), history);
}

static void
populate_user_list(login_dlg *ldlg) {
	GList *l;
	User *u;
	GList *strings = NULL;

	for (l = conf_cur_server()->users; l != NULL; l = l->next) {
		u = l->data;
		strings = g_list_append(strings, u->username);
	}
	if (strings)
		gtk_combo_set_popdown_strings(GTK_COMBO(ldlg->cusername), strings);
	else
		gtk_list_clear_items(GTK_LIST(GTK_COMBO(ldlg->cusername)->list), 0, -1);
}

static void
dlg_activate(GtkWidget *w, login_dlg *ldlg) {
	gtk_dialog_response(GTK_DIALOG(ldlg->win), GTK_RESPONSE_OK);
}
static GtkWidget*
make_login_table(login_dlg *ldlg) {
	GtkWidget *table;

	table = lj_table_new(6, 2);

	ldlg->mserver = gtk_option_menu_new();
	gtk_widget_set_size_request(ldlg->mserver, 50, -1);
	populate_server_list(ldlg);
	lj_table_label_content(GTK_TABLE(table), 0,
			_("_Server:"), ldlg->mserver);
	g_signal_connect(G_OBJECT(ldlg->mserver), "changed",
			G_CALLBACK(server_changed_cb), ldlg);

	ldlg->cusername = gtk_combo_new();
	gtk_widget_set_size_request(ldlg->cusername, 50, -1);
	ldlg->eusername = GTK_COMBO(ldlg->cusername)->entry;
	gtk_combo_disable_activate(GTK_COMBO(ldlg->cusername)); 
	populate_user_list(ldlg);
	lj_table_label_content_mne(GTK_TABLE(table), 1,
			_("_User Name:"), ldlg->cusername, ldlg->eusername);

	ldlg->epassword = gtk_entry_new(); 
	gtk_widget_set_size_request(ldlg->epassword, 100, -1);
	gtk_entry_set_visibility(GTK_ENTRY(ldlg->epassword), FALSE);
	lj_table_label_content(GTK_TABLE(table), 2,
			_("_Password:"), ldlg->epassword);

	ldlg->cfastserver = gtk_check_button_new_with_mnemonic(_("Use _fast server"));
	gtk_table_attach(GTK_TABLE(table), ldlg->cfastserver,
			1, 2, 3, 4, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	ldlg->cruser = gtk_check_button_new_with_mnemonic(_("R_emember user"));
	gtk_table_attach(GTK_TABLE(table), ldlg->cruser,
			1, 2, 4, 5, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	ldlg->crpassword = gtk_check_button_new_with_mnemonic(_("Re_member password"));
	gtk_table_attach(GTK_TABLE(table), ldlg->crpassword,
			1, 2, 5, 6, GTK_EXPAND|GTK_FILL, 0, 0, 0);

	gtk_table_attach(GTK_TABLE(table), 
			gtk_image_new_from_stock("logjam-goat", GTK_ICON_SIZE_DIALOG),
			0, 1, 3, 6, GTK_EXPAND|GTK_FILL, GTK_EXPAND|GTK_FILL, 0, 0);

	g_signal_connect(G_OBJECT(ldlg->eusername), "changed",
			G_CALLBACK(username_changed), ldlg);
	if (conf_cur_user())
		gtk_entry_set_text(GTK_ENTRY(ldlg->eusername), conf_cur_user()->username);
	username_changed(ldlg->eusername, ldlg);

	/* pressing enter should be equiv. to hitting OK */
	g_signal_connect(G_OBJECT(ldlg->eusername), "activate",
			G_CALLBACK(dlg_activate), ldlg);
	g_signal_connect(G_OBJECT(ldlg->epassword), "activate",
			G_CALLBACK(dlg_activate), ldlg);

	return table;
}

static void
login_shown(login_dlg *ldlg) {
	gtk_widget_grab_focus(ldlg->eusername);
}

void
login_dlg_run(LJWin *ljw, int autolog) {
	login_dlg ldlg_actual = {0}, *ldlg = &ldlg_actual;
	GtkWidget *align, *vbox, *button;

	ldlg->ljw = ljw;
	ldlg->win = lj_dialog_new(GTK_WIDGET(ljw), _("LogJam Login"), -1, -1);

	gtk_widget_realize(ldlg->win);

	vbox = gtk_vbox_new(FALSE, 5); 

	gtk_box_pack_start(GTK_BOX(vbox), make_login_table(ldlg), TRUE, TRUE, 0);
	align = gtk_alignment_new(.5, .5, 1, 0);
	gtk_container_add(GTK_CONTAINER(align), vbox);

	lj_dialog_set_contents(ldlg->win, align);

	button = lj_dialog_add_okcancel(ldlg->win, "_Login");
	g_signal_connect_swapped(G_OBJECT(ldlg->win), "show",
			G_CALLBACK(login_shown), ldlg);

	if (autolog && !conf.options.noautologin)
		g_idle_add((GSourceFunc)autolog_cb, ldlg);

	while (lj_dialog_run(ldlg->win)) {
		if (login_run(ldlg))
			break;
	}
}

