/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: ljtypes.h,v 1.11 2002/11/28 03:09:28 martine Exp $
 */

#ifndef __LJTYPES_H__
#define __LJTYPES_H__

/* structures for representing livejournal data. 
 * logjam-specific structures are in conf.h.
 */ 
#include <time.h> /* post timestamp */
#include <glib-object.h>
#include "network.h" /* request */

/* associate numbers<->names in a two-way structure (using a GList).
 * functions for working with nameidhashes are in conf.h. */
#define NID_CONTENT int id; char *name; /* yay for no inheritance in C. */
typedef struct {
	NID_CONTENT
} NameIDHash;

typedef NameIDHash Mood;

typedef struct {
	NID_CONTENT
	gboolean ispublic;
} FriendGroup;
void friend_group_free(FriendGroup *fg);

typedef struct {
	char *username;
	char *fullname;
	char *password;
	gboolean fastserver;
	gboolean remember_user, remember_password;

	GList *pickws;
	GList *friendgroups;
	GList *usejournals;
} User;

typedef enum {
	SECURITY_PUBLIC = 0,
	SECURITY_FRIENDS,
	SECURITY_PRIVATE,
	SECURITY_CUSTOM
} SecurityType;

typedef struct {
	SecurityType type;
	unsigned int allowmask;
} Security;
/* an allowmask of 1 means friends-only. */
#define SECURITY_ALLOWMASK_FRIENDS 1

void security_append_to_request(Security *security, GHashTable *request);
void security_load_from_request(Security *security, GHashTable *request);
void security_load_from_strings(Security *security, const char *sectext, const char *allowmask);

/* ----- friend ----- */

#define FRIEND_CONN_MY   (1 << 0)
#define FRIEND_CONN_OF   (1 << 1)
#define FRIEND_CONN_BOTH (FRIEND_CONN_MY | FRIEND_CONN_OF)

typedef enum {
	FRIEND_TYPE_USER=1,
	FRIEND_TYPE_COMMUNITY
} FriendType;

typedef struct {
	char *username;
	char *fullname;
	char foreground[8], background[8];
	int conn; /* FRIEND_MY, etc */
	FriendType type; 
	guint32 groupmask;
} Friend;

/* the first bit is the "all friends" group. */
#define FRIEND_GROUP_ALLFRIENDS 1
Friend *friend_new(void);
Friend *friend_new_with(char *username, char *name, char *fg, char *bg, int conn, guint mask, char *type);
void    friend_free(Friend *f);
FriendType friend_type_from_str(char *str);
gint friend_compare_username(gconstpointer a, gconstpointer b);

/* we make friends into a boxed type so we can use them in our list view.
 * don't confuse the "type" here (a glib object type)
 * with the "type" above (the user/community distinction)! */
GType friend_get_gtype(void);
#define FRIEND_GTYPE friend_get_gtype()

typedef enum {
	COMMENTS_DEFAULT,
	COMMENTS_NOEMAIL,
	COMMENTS_DISABLE
} CommentsType;

/* ----- entry ----- */
typedef struct {
	int itemid;
	char *subject;
	char *event;

	char *mood, *music, *pickeyword;
	int preformatted;
	int backdated;
	CommentsType comments;

	struct tm time;
	Security security;
} Entry;

typedef enum {
	ENTRY_FILE_AUTODETECT,
	ENTRY_FILE_XML,
	ENTRY_FILE_RFC822,
	ENTRY_FILE_PLAIN
} EntryFileType;

Entry *entry_new(void);
void   entry_free(Entry *e);
char* entry_make_summary(Entry *entry);
void entry_set_request_fields(Entry *entry, NetRequest *request);

int entry_to_xml_file(Entry *entry, const char *filename);
gboolean entry_to_rfc822_file(Entry *entry, const char *filename, GError **err);

gboolean entry_load(Entry *entry, gchar *data, gsize len, EntryFileType type, GError **err);
Entry* entry_new_from_filename(const char *filename, EntryFileType type, GError **err);
Entry* entry_from_result(NetResult *result, int i);
gboolean entry_edit_with_usereditor(Entry *entry, GError **err);

void entry_load_metadata(Entry *entry, const char *key, const char *value);

#endif /* __LJTYPES_H__ */
