/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: friendedit.c,v 1.6 2002/09/24 07:57:50 martine Exp $
 */

#include <gtk/gtk.h>

#include <stdlib.h>
#include <string.h>

#include "friends.h"
#include "friendedit.h"
#include "network.h"
#include "conf.h"
#include "util.h"

typedef struct {
	GtkWidget *win;
	GtkWidget *eusername, *ebgcolor, *efgcolor;

	Friend *editfriend;
} FriendEditUI;

static void
update_preview(FriendEditUI *feui) {
	GdkColor fg, bg;

	gdk_color_parse(gtk_entry_get_text(GTK_ENTRY(feui->efgcolor)), &fg);
	gdk_color_parse(gtk_entry_get_text(GTK_ENTRY(feui->ebgcolor)), &bg);

	gtk_widget_modify_text(feui->eusername, GTK_STATE_NORMAL, &fg);
	gtk_widget_modify_base(feui->eusername, GTK_STATE_NORMAL, &bg);
}

static void 
color_entry_changed(GtkEntry *e, FriendEditUI *feui) {
	if (strlen(gtk_entry_get_text(e)) == 7) 
		update_preview(feui);
}

static gint
change_entry_color_dlg(FriendEditUI *feui, GtkWidget *toedit, const char *title) {
	GtkWidget *dlg;
	GtkColorSelection *csel;
	const char *curcolor;
	char new_hex[10];
	GdkColor color;

	dlg = gtk_color_selection_dialog_new(title);
	gtk_window_set_transient_for(GTK_WINDOW(dlg), GTK_WINDOW(feui->win));
	lj_win_set_icon(dlg);
	gtk_widget_hide(GTK_COLOR_SELECTION_DIALOG(dlg)->help_button);

	csel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(dlg)->colorsel);

	curcolor = gtk_entry_get_text(GTK_ENTRY(toedit));
	
	/* convert existing hex color to the color selection's color */
	if (strlen(curcolor) == 7 && curcolor[0] == '#') {
		gdk_color_parse(curcolor, &color);
		gtk_color_selection_set_current_color(csel, &color); 
	}

	if (lj_dialog_run(dlg)) {
		gtk_color_selection_get_current_color(csel, &color);

		gdkcolor_to_hex(&color, new_hex);

		gtk_entry_set_text(GTK_ENTRY(toedit), new_hex);
		gtk_widget_destroy(dlg);
	}

	return 0;
}

static void 
change_col_bg(GtkWidget *w, FriendEditUI *feui) {
	change_entry_color_dlg(feui, feui->ebgcolor, _("Select Background Color"));
}

static void 
change_col_fg(GtkWidget *w, FriendEditUI *feui) {
	change_entry_color_dlg(feui, feui->efgcolor, _("Select Foreground Color"));
}

static gboolean 
add_the_friend(FriendEditUI *feui) {
	NetRequest *request;
	NetResult *result;
	int friends_added = 0;
	gchar *username, *name;

	request = net_request_new("editfriends");

	net_request_sets(request, "editfriend_add_1_user",
			g_strdup(gtk_entry_get_text(GTK_ENTRY(feui->eusername))));
	net_request_sets(request, "editfriend_add_1_fg",
			g_strdup(gtk_entry_get_text(GTK_ENTRY(feui->efgcolor))));
	net_request_sets(request, "editfriend_add_1_bg",
			g_strdup(gtk_entry_get_text(GTK_ENTRY(feui->ebgcolor))));

	result = net_request_run(feui->win, _("Adding Friend..."), request);
	hash_destroy(request);

	if (!net_result_succeeded(result)) {
		hash_destroy(result);
		return FALSE;
	}

	friends_added = net_result_geti(result, "friends_added");
	if (friends_added != 1) {
		hash_destroy(result);
		return FALSE;
	}
	
	name = net_result_get(result, "friend_1_name");
	username = net_result_get(result, "friend_1_user");

	if (feui->editfriend == NULL || 
			strcmp(feui->editfriend->username, username) != 0) {
		/* we must create a new friend */
		feui->editfriend = friend_new();
		feui->editfriend->conn = FRIEND_CONN_MY;
	}

	string_replace(&feui->editfriend->username, g_strdup(username));
	string_replace(&feui->editfriend->fullname, g_strdup(name));
	strcpy(feui->editfriend->foreground,
			gtk_entry_get_text(GTK_ENTRY(feui->efgcolor)));
	strtoupper(feui->editfriend->foreground);
	strcpy(feui->editfriend->background,
			gtk_entry_get_text(GTK_ENTRY(feui->ebgcolor)));
	strtoupper(feui->editfriend->background);

	gtk_widget_destroy(feui->win);

	hash_destroy(result);
	return TRUE;
}

static void
entry_changed(GtkEntry *entry, GtkWidget* button) {
	gtk_widget_set_sensitive(button, 
			(strlen(gtk_entry_get_text(entry)) > 0));
}

static void
dialog_ok_cb(GtkEntry *entry, FriendEditUI* feui) {
	gtk_dialog_response(GTK_DIALOG(feui->win), GTK_RESPONSE_OK);
}

Friend*
friend_edit_dlg_run(GtkWidget *parent, gboolean edit, Friend *f) {
	FriendEditUI feui_actual = { 0 };
	FriendEditUI *feui = &feui_actual;
	GtkWidget *button;
	GtkWidget *table;

	memset(feui, 0, sizeof(FriendEditUI));
	feui->editfriend = f;

	feui->win = lj_dialog_new(parent,
			edit ?  _("Edit Friend") :
			(f ? _("Add This Friend") : _("Add a Friend")),
			-1, -1);

	table = lj_table_new(3, 3);
	   
	/* make the labels/entries */
	feui->eusername = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(feui->eusername), 18);
	if (edit) gtk_editable_set_editable(GTK_EDITABLE(feui->eusername), FALSE);
	gtk_widget_set_size_request(feui->eusername, 100, -1);
	lj_table_label_content(GTK_TABLE(table), 0, 
			_("Friend's _username:"), feui->eusername);

	feui->efgcolor = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(feui->efgcolor), 7);
	g_signal_connect(G_OBJECT(feui->efgcolor), "changed",
			G_CALLBACK(color_entry_changed), feui);
	gtk_widget_set_size_request(feui->efgcolor, 100, -1);
	lj_table_label_content(GTK_TABLE(table), 1, 
			_("_Text color:"), feui->efgcolor);

	feui->ebgcolor = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(feui->ebgcolor), 7);
	g_signal_connect(G_OBJECT(feui->ebgcolor), "changed",
			G_CALLBACK(color_entry_changed), feui);
	gtk_widget_set_size_request(feui->ebgcolor, 100, -1);
	lj_table_label_content(GTK_TABLE(table), 2,
			_("_Background color:"), feui->ebgcolor);

	/* make the color selector buttons */
	button = gtk_button_new_with_label(" ... ");
	gtk_table_attach(GTK_TABLE(table), button, 2, 3, 1, 2, GTK_FILL, 0, 2, 2);
	g_signal_connect(G_OBJECT(button), "clicked",
			G_CALLBACK(change_col_fg), feui);
	button = gtk_button_new_with_label(" ... ");
	gtk_table_attach(GTK_TABLE(table), button, 2, 3, 2, 3, GTK_FILL, 0, 2, 2);	  
	g_signal_connect(G_OBJECT(button), "clicked",
			G_CALLBACK(change_col_bg), feui);
		
	lj_dialog_set_contents(feui->win, table);

	button = lj_dialog_add_okcancel(feui->win, edit ? _("  Change  ") : _("  Add  ")),

	/* enable/disable the button based on name text */
	g_signal_connect(G_OBJECT(feui->eusername), "changed",
		G_CALLBACK(entry_changed), button);

	g_signal_connect(G_OBJECT(feui->eusername), "activate",
		G_CALLBACK(dialog_ok_cb), feui);

	/* fill in default values. */
	if (f) {
		gtk_entry_set_text(GTK_ENTRY(feui->eusername), f->username);
	} else {
		gtk_entry_set_text(GTK_ENTRY(feui->eusername), "");
		/* emit the "changed" signal, anyway. */
	}

	if (!edit) {
		gtk_entry_set_text(GTK_ENTRY(feui->efgcolor), "#000000");
		gtk_entry_set_text(GTK_ENTRY(feui->ebgcolor), "#FFFFFF");
	} else {
		gtk_entry_set_text(GTK_ENTRY(feui->efgcolor), f->foreground);
		gtk_entry_set_text(GTK_ENTRY(feui->ebgcolor), f->background);
	}

	while (lj_dialog_run(feui->win)) {
		if (add_the_friend(feui))
			return feui->editfriend;
	}
	return NULL;
}


