/*
 * $Id: login_ldap.c,v 1.3 2007/10/25 11:43:32 merdely Exp $
 * Original work by Peter Werner <peterw@ifost.org.au>
 * Maintained by Michael Erdely <merdely@openbsd.org>
 * Please visit http://sourceforge.net/projects/login-ldap/
 * Questions may be directed to <peterw@ifost.org.au> or <merdely@openbsd.org>
 * For information on commercial support please contact <gregb@ifost.org.au>
 */

#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <err.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <lber.h>
#define LDAP_DEPRECATED 1
#include <ldap.h>
#include <login_cap.h>
#include <stdarg.h>
#include <bsd_auth.h>

#include "login_ldap.h"

int	auth_ldap(char *, char *, char *);

static void
handler(int signo)
{
	_exit(1);
}

/*
 * Authenticate a user using LDAP
 */

int
main(int argc, char **argv)
{
	int c;
        char *class, *service, *username, *password;
	char backbuf[BUFSIZ];
	FILE *back = NULL;

        password = class = service = NULL;

	(void)signal(SIGQUIT, SIG_IGN);
	(void)signal(SIGINT, SIG_IGN);
	(void)signal(SIGALRM, handler);
	(void)setpriority(PRIO_PROCESS, 0, 0);

	openlog("login_ldap", LOG_ODELAY, LOG_AUTH);

        /*
         * Usage: login_xxx [-s service] [-v var] user [class]
         */
        while ((c = getopt(argc, argv, "dp:s:v:")) != -1)
                switch(c) {
                case 'd':
                        back = stdout;
			debug = 1;
                        break;
		case 'p':
			password = optarg;
			break;
                case 'v':
                        break;
                case 's':       /* service */
                        service = optarg;
                        if (strcmp(service, "login") != 0 &&
				strcmp(service, "challenge") != 0 &&
				strcmp(service, "response") != 0)
                                	errx(1, "%s not supported", service);
                        break;
                default:
                        dlog(0, "usage error 1");
                        exit(1);
                }

	argc -= optind;
	argv += optind;

        if (service == NULL)
                service = LOGIN_DEFSERVICE;
	
	switch (argc) {
	case 2:
		class = argv[1];
		/* FALLTHROUGH */
        case 1:
                username = argv[0];
                break;
        default:
                dlog(1, "usage error 2");
                exit(1);
        }


	if (debug == 0 && password != NULL) {
		dlog(0, "-p is for debugging only");
		exit(1);
	}

	if (strlen(username) >= MAXLOGNAME) {
		dlog(0, "username too long");
		exit(1);
	}

        /*
         * Filedes 3 is the back channel, where we send back results.
         */
        if (back == NULL && (back = fdopen(3, "a")) == NULL)  {
                dlog(0, "error reopening back channel");
                exit(1);
        }

	if (debug == 1 && password != NULL) {
		;
	} else if (strcmp(service, "login") == 0) {
		password = getpass("Password: ");
	} else if (strcmp(service, "response") == 0) {
		int n;

		/* 
		 * format of back channel message 
		 * 
		 * c h a l l e n g e \0 p a s s w o r d \0
		 *
		 * challenge can be NULL (so can password too
		 * i suppose).
		 */
 
		n = read(3, backbuf, sizeof(backbuf));
		if (n == -1) {
			dlog(0, "read error from back channel");
			exit(1);
		}

		/* null challenge */
		if (backbuf[0] == '\0') 
			password = backbuf + 1;
		/* skip the first string to get the password */
		else if ((password = strchr(backbuf, '\0')) != NULL) 
			password++;
		else
			dlog(0, "protocol error on back channel");
		
	} else if (strcmp(service, "challenge") == 0) {
		(void)fprintf(back, BI_SILENT "\n");
		exit(0);
	} else {
		dlog(0, "unsupported service type %s", service);	
		errx(1, "unsupported service type %s", service);
	}

	if (password == NULL || password[0] == '\0') {
		dlog(1, "Null password");
		(void)fprintf(back, BI_REJECT "\n");
		exit(1);
	}

	if (auth_ldap(username, class, password))
		(void)fprintf(back, BI_AUTH "\n");
	else 
		(void)fprintf(back, BI_REJECT "\n");

	(void) memset(password, 0, strlen(password));

	closelog();

	exit(0);
	/* NOTREACHED */
}

int
auth_ldap(char *user, char *class, char *pass)
{
	char *tmp;
	struct auth_ctx ctx;

	memset(&ctx, 0x00, sizeof(struct auth_ctx));

	ctx.lc = login_getclass(class);
	if (ctx.lc == NULL) {
		dlog(0, "couldnt find %s in login.conf", class);
		return(0);
	}

	ctx.class = class;
	ctx.user = user;

	if (!load_ssl_certs(&ctx)) {
		dlog(0, "load_ssl_certs");
		return(0);
	}

	if (!conn(&ctx)) {
		dlog(0, "ldap_open failed");
		return(0);
	}

	if (!get_bind_defaults(&ctx) || !get_misc(&ctx)) { 
		dlog(0, "error reading login.conf");
		return(0);
	}

	if (!bind_password(&ctx, ctx.binddn, ctx.bindpw)) {
		dlog(0, "bind failed: %s", ctx.lasterr);
		return(0);
	}

	dlog(1, "bind success!");

	if (!get_usearch_defaults(&ctx)) {
		dlog(0, "get user search defaults failed");
		return(0);
	}

	ctx.userdn = search(&ctx, ctx.basedn, ctx.ufilter, ctx.uscope); 
	if (ctx.userdn == NULL) {
		dlog(1, "no user!");
		return(0);
	}

	dlog(1, "userdn %s", ctx.userdn);

	if (!bind_password(&ctx, ctx.userdn, pass)) {
		dlog(0, "user bind failed, dn: %s", ctx.userdn);
		return(0);
	}

	dlog(1, "user bind success!");

	if (!get_gsearch_defaults(&ctx)) {
		dlog(0, "get gsearch defaults failed");
		return(0);
	}
	
	if (ctx.gfilter == NULL || ctx.gfilter[0] == '\0') {
		/* if we've gotten this far they're authenticated */
		dlog(1, "no group filter");
		return(1);
	}

	tmp = search(&ctx, ctx.groupdn, ctx.gfilter, ctx.gscope);
	if (tmp == NULL) {
		dlog(0, "user authenticated but failed group check: %s", ctx.gfilter);
		return(0);
	}

	dlog(1, "group filter matched!");
		
	return(1);
}
