/****************************************************************************
** $Id: directory.cpp,v 1.13 1999/12/14 02:16:25 riemer Exp $
**
** copyright            : (C) 1999-2000 by Tilo Riemer
**
** email                :  riemer@ppprs1.phy.tu-dresden.de
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
*****************************************************************************/

//#include <iostream.h>
#include <stdlib.h>
#include <unistd.h>

//----------------------------------------------------------------------------

#include <qdir.h>
#include <qfileinfo.h>
#include <qfile.h>
#include <qpixmap.h>
#include <qapplication.h>
#include <qtextstream.h>
#include <qdatetime.h>
#include <qpalette.h> 
#include <qstringlist.h>
#include <qmessagebox.h>
#include <qdialog.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qcheckbox.h> 

//----------------------------------------------------------------------------

#include "directory.h"
#include "globals.h"

//----------------------------------------------------------------------------

#include "pix.h"
#include "addedfile.xpm"
#include "conflictfile.xpm"
#include "lostfile.xpm"
#include "modifiedfile.xpm"
#include "needspatch.xpm"
#include "needsmerge.xpm"
#include "needscheckout.xpm"
#include "removedfile.xpm"
#include "unchangedfile.xpm"
#include "unchangedfiletimezoneconflict.xpm"

//----------------------------------------------------------------------------

Directory::Directory( QListViewItem * parent, const QString& fileName, QListView *fileListView)
   : QListViewItem( parent )
{
   init(fileName, fileListView);
}

//----------------------------------------------------------------------------

Directory::Directory( QListView * parent, const QString& fileName, QListView *fileListView)
   : QListViewItem( parent )
{
   init(fileName, fileListView);
}

//----------------------------------------------------------------------------

void Directory::activateItem()
{
   activate();
}

//----------------------------------------------------------------------------

void Directory::init(const QString& fileName, QListView *fileListView)
{
   m_pFileListView = fileListView;
   m_fullName = fileName;
   m_readable = QDir(m_fullName).isReadable();
   m_entries.setAutoDelete(true);
   
   if(m_fullName.length() == 1) {//root
      setText(0, m_fullName);
   } 
   else {
      setText(0, m_fullName.right(m_fullName.length() - m_fullName.findRev("/") - 1));
   }   

   if (!m_readable )
      setPixmap( 0, QPixmap( folder_locked ) );
   else
      setPixmap( 0, QPixmap( folder_closed_xpm ) );

   qApp->processEvents(100);
   if(!globalStopAction){
      analyzeDirs();
   }
}

//----------------------------------------------------------------------------

void Directory::addDir(QString newDir)
{
   if(m_fullName.compare(newDir) == 0){//I am it ;-)
      analyzeDirs();   //das wird wohl nur klappen, wenns dir leer is
      return;
   }

   Directory *myChild = (Directory*)firstChild();
   while(myChild) {
      if(newDir.find(myChild->fullName()) == 0) {
         myChild->addDir(newDir);
         return;      
      }
      myChild = (Directory*)myChild->nextSibling();
   }

   Directory *item = new Directory(this, newDir, m_pFileListView);
   if(item->haveCvsDir()) {
      item->setExpandable(true);
      m_subDirHasCvsDir = true;
   }
   else {
      delete item;
   }

   return;
}

//----------------------------------------------------------------------------

bool Directory::analyzeDirs()
{
   m_haveCvsDir = false;
   m_subDirHasCvsDir = false;

   QDir myDir(m_fullName);
   QStringList subDirList = myDir.entryList();
   QString path = m_fullName;
   QString curSubDirName;
   QFileInfo fileInfo;
   Directory *item;

   if(!subDirList.isEmpty()){
      if(m_fullName.length() > 1){//is not root
         path += "/";
      }
   
      for(unsigned int i = 0; i < subDirList.count(); i++){
         if(globalStopAction) exit;   //cancel this action

         curSubDirName = subDirList[i];
         if((QString::compare(curSubDirName, ".") == 0) || 
            (QString::compare(curSubDirName, "..") == 0))
            continue;   //is the current dir or the parent dir

         fileInfo.setFile(path + curSubDirName);

         if(fileInfo.isDir() &&
            fileInfo.permission(QFileInfo::ReadUser | QFileInfo::ExeUser))
         {//is a dir and readable
            globalStatusBar->message(path + curSubDirName);

            if(QString::compare(curSubDirName, "CVS") == 0){//have CVS dir
               m_haveCvsDir = checkCvsDir();
            }
            else {//is a dir, is readable and is not the CVS dir
//noch testen, ob genug speicher (wenn grosse dirs gelesen werden)   
               item = new Directory(this, path + curSubDirName, m_pFileListView);
               if(item->haveCvsDir()) {
                  item->setExpandable(true);
                  m_subDirHasCvsDir = true;
               }
               else {
                  delete item;
               }
            }
         }
      }
   }
   return m_haveCvsDir || m_subDirHasCvsDir;
}

//----------------------------------------------------------------------------

bool Directory::checkCvsDir()
{
   QString path = m_fullName;
   if(path.length() > 1){//is not root
         path += "/";
      }
      
   QFile f;
   QString line;
   int pos, posAt, posStartRootDir;
   
   //read Root
   f.setName(path + "CVS/Root");
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 

      line = textStream.readLine();
      f.close();
      if((posAt = line.find("@")) > -1) {//remote
         if(posAt < 2) return false;   //no space for user :-(

         //connectMethod
         QString left = line.left(posAt);
         if(left.find(":") != 0) return false;   //no ":" at position 0
         pos = left.find(":", 1);
         m_connectMethod = left.mid(1, pos - 1);
         if( (QString::compare(m_connectMethod, "ext"))
              && (QString::compare(m_connectMethod, "server"))
              && (QString::compare(m_connectMethod, "pserver")) ) 
         {//default remote access!
            m_userName = left.mid(1, posAt - 1);
            if(m_userName.length() == 0) return false;   //no user
            m_connectMethod = "";
         }
         else 
         {//ext, server or pserver         
            //user
            m_userName = left.right(posAt - pos - 1);
            if(m_userName.length() == 0) return false;   //no user
         }
      
         //host
         m_host = line.mid(posAt + 1, line.find(":", posAt) - posAt - 1);
         
         posStartRootDir = posAt + m_host.length() + 2;
      }
      else {//local
         m_userName = getenv("USER");
         m_host = "localhost";
         m_connectMethod = "local";
         posStartRootDir = 0;
      }
      
      m_rootDir = line.mid(posStartRootDir, line.length() - posStartRootDir);
   }
   else return false;

   //read Repository
   f.setName(path + "CVS/Repository");
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 
      m_repository = textStream.readLine();
      f.close();
   }
   else return false;

   //workaround for old cvs version
   if(m_repository.find(m_rootDir) == 0){//old cvs version detected
      m_repository = m_repository.mid(m_rootDir.length() + 1);
   }
   
      
   return true;
}

//----------------------------------------------------------------------------

void Directory::activate()
{
   m_pFileListView->clear();
   
   checkStatus();
}

//----------------------------------------------------------------------------

Directory* Directory::searchDirOfRepository(QString repository)
{
   if(repository.compare(m_rootDir + "/" + m_repository) == 0)
      return this;   //its my repository :-)

   bool isSubDir = false;
   if((m_repository.find(".")) == 0 && (m_repository.length() == 1)) {// . detected
      if(repository.find(m_rootDir + "/") == 0) {//is subdir
         isSubDir = true;
      }
   }
   else {
      if(repository.find(m_rootDir + "/" + m_repository) == 0) {//is subdir
         isSubDir = true;
      }
   }

   if(isSubDir) {//is subdir
      Directory *myChild = (Directory*)firstChild();
      Directory *result;
      while(myChild) {
         if(result = myChild->searchDirOfRepository(repository)) {
            return result;
         }
         else {
            myChild = (Directory*)myChild->nextSibling();
         }
      }
   }
   
   return 0;
}

//----------------------------------------------------------------------------

Directory* Directory::searchDirOfPath(QString path)
{
   if(path.compare(m_fullName) == 0)
      return this;   //its my path :-)

   if(path.find(m_fullName) == 0) {//is subdir
      Directory *myChild = (Directory*)firstChild();
      Directory *result;
      while(myChild) {
         if(result = myChild->searchDirOfPath(path)) {
            return result;
         }
         else {
            myChild = (Directory*)myChild->nextSibling();
         }
      }
   }
   
   return 0;
}

//----------------------------------------------------------------------------

void Directory::statusOfDir(QMultiLineEdit *output)
{
   if(!loginOk()) return;

   output->clear();   //spaeter konfigurierbar, ob alte ausgaben loeschen

   QPalette pal = output->palette();
   pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
   output->setPalette(pal);
//   qApp->processEvents(100);

   QString command = "";

   if( (QString::compare(m_connectMethod, "ext") == 0)
        || (QString::compare(m_connectMethod, "") == 0) )
   {//rsh access
      command = "export CVS_RSH=" + cvsRsh + "&& ";
   }
   
   command += "cd ";
   command += m_fullName;
   command += " && cvs status -v";
   command += " 2>&1";
 
   FILE* cvs = popen(command, "r");
   if(!cvs){
      //ordentliche fehlerbehandlung einbauen
      
      perror("popen");
      return;
   }

   qApp->mainWidget()->setCursor(waitCursor);

   char buf[512];
   QString repository, oldRepository;
   QString fileName;
   int state;
   oldRepository = "";
   
   Directory *dir = 0;
   
   while (fgets(buf, sizeof buf, cvs)) {
      QString str = buf;
      str.truncate(str.find("\n"));

      //jetzt "str" analysieren
      if(str.find("File:") == 0){//status output of next file begins
         fileName = str.mid(6, str.find("\t", 6) - 6);
         fileName = fileName.simplifyWhiteSpace();
         if(fileName.find("no File") == 0) {
            fileName = fileName.mid(8);
         }
      
         if(str.find("Up-to-date") > -1){
            state = up_to_date;
         }
         else 
         if(str.find("Locally Modified") > -1){
            state = modified;
         }
         else
         if(str.find("Needs Patch") > -1){
            state = needs_patch;
         }
         else
         if(str.find("Needs Merge") > -1){
            state = needs_merge;
         }
         else
         if(str.find("Needs Checkout") > -1){
            state = needs_checkout;
         }
         else 
         if(str.find("File had conflicts on merge") > -1){
            state = conflict;
         }
         else
         if(str.find("Locally Added") > -1){
            state = added;
         }
         else
         if(str.find("Locally Removed") > -1){
            state = removed;
         }
      }
      else
      if(str.find("Repository revision:") > -1) {//repository of fileName
         //repository extrahieren
         int pos;

         if((m_repository.find(".")) == 0 && (m_repository.length() == 1)) {// . detected
            pos = str.find(m_rootDir);
         }
         else {
            pos = str.find(m_rootDir + "/" + m_repository);
         }
         
         repository = str.mid(pos, str.findRev("/") - pos);

         if(repository.compare(oldRepository)) {
            //zugehoeriges dir finden
            dir = searchDirOfRepository(repository);
            oldRepository = repository;
         }

         //status eintragen
         if(dir) dir->setAndAppendEntry(fileName, state);
      }

      output->append(str);
   }
   
   pclose(cvs);

   activate();
   qApp->mainWidget()->setCursor(arrowCursor);
}

//----------------------------------------------------------------------------

void Directory::updateDir(QMultiLineEdit *output, const QString& tagParameter)
{
   if(!loginOk()) return;

   int state;
   output->clear();   //spaeter konfigurierbar, ob alte ausgaben loeschen

   QPalette pal = output->palette();
   pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
   output->setPalette(pal);
//   qApp->processEvents(100);

   QString path;
   QString oldPath = "";

   QString absPath = m_fullName;
   if(absPath.length() > 1){//is not root
      absPath += "/";
   }

   QString command = "";

   if( (QString::compare(m_connectMethod, "ext") == 0)
        || (QString::compare(m_connectMethod, "") == 0) )
   {//rsh access
      command = "export CVS_RSH=" + cvsRsh + "&& ";
   }
   
   command += "cd ";
   command += m_fullName;
   command += " && cvs update -P -d";
   if(!tagParameter.isNull()) {
      command += " " + tagParameter;
   }
   command += " 2>&1";
 
   FILE* cvs = popen(command, "r");
   if(!cvs){
      //ordentliche fehlerbehandlung einbauen
      
      perror("popen");
      return;
   }

   qApp->mainWidget()->setCursor(waitCursor);

   char buf[512];
   QString fileName;

   setAllToUpToDate();
   Directory *dir = 0;
         
   while (fgets(buf, sizeof buf, cvs)) {
      QString str = buf;
      str.truncate(str.find("\n"));
      state = -1;

      //jetzt "str" analysieren
      if((str.find("P ") == 0) || (str.find("U ") == 0)){
         state = up_to_date;
      }
      else 
      if(str.find("M ") == 0){
         state = modified;
      }
      else 
      if(str.find("C ") == 0){
         state = conflict;
      }
      else 
      if(str.find("A ") == 0){
         state = added;
      }
      else 
      if(str.find("R ") == 0){
         state = removed;
      }
      
      if(state > -1) {
         fileName = str.mid(2);
         fileName = fileName.simplifyWhiteSpace();

         //extract path
         int pos;
         if((pos = fileName.findRev("/")) > -1) {//file is located in subdir
            path = fileName.left(pos);
            fileName = fileName.mid(pos + 1);
            if(path.compare(oldPath)) {
               //zugehoeriges dir finden
               dir = searchDirOfPath(absPath + path);
               oldPath = path;
            }
            if(dir) dir->setAndAppendEntry(fileName, state);
         }
         else setAndAppendEntry(fileName, state);
      }      
      
      output->append(str);
   }
   
   pclose(cvs);

   activate();
   qApp->mainWidget()->setCursor(arrowCursor);
}

//----------------------------------------------------------------------------

void Directory::commitDir(QMultiLineEdit *output, QString comment)
{
   if(!loginOk()) return;

   int state;
   output->clear();   //spaeter konfigurierbar, ob alte ausgaben loeschen

   QPalette pal = output->palette();
   pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
   output->setPalette(pal);
   qApp->processEvents(100);

   QString path;
   QString oldPath = "";

   QString absPath = m_fullName;
   if(absPath.length() > 1){//is not root
      absPath += "/";
   }

   QString command = "";

   if( (QString::compare(m_connectMethod, "ext") == 0)
        || (QString::compare(m_connectMethod, "") == 0) )
   {//rsh access
      command = "export CVS_RSH=" + cvsRsh + "&& ";
   }
   
   command += "cd ";
   command += m_fullName;
   command += " && cvs commit -m ";
   command += "\"" + comment + "\"";
   command += " 2>&1";
 
   FILE* cvs = popen(command, "r");
   if(!cvs){
      //ordentliche fehlerbehandlung einbauen
      
      perror("popen");
      return;
   }

   qApp->mainWidget()->setCursor(waitCursor);

   char buf[512];
   QString fileName;
   
   while (fgets(buf, sizeof buf, cvs)) {
      QString str = buf;
      str.truncate(str.find("\n"));

      if(str.find("Checking in ") == 0) {//followed of filename
         fileName = str.mid(12, str.length() - 13);   //truncate ";"
      }
      else
      if(str.compare("done") == 0){//commit completed successfully
         setEntry(fileName, up_to_date);
      }

      output->append(str);
   }
   
   pclose(cvs);

   activate();
   qApp->mainWidget()->setCursor(arrowCursor);
}

//----------------------------------------------------------------------------

void Directory::anyCommandDir(QMultiLineEdit *output, QString cvsCommand)
{
   if(!loginOk()) return;

   QString command = "";

   if( (QString::compare(m_connectMethod, "ext") == 0)
        || (QString::compare(m_connectMethod, "") == 0) )
   {//rsh access
      command = "export CVS_RSH=" + cvsRsh + "&& ";
   }
   
   command += "cd ";
   command += m_fullName;
   command += " && cvs " + cvsCommand;
   command += " 2>&1";
 
   FILE* cvs = popen(command, "r");
   if(!cvs){
      perror("popen");
      return;
   }

   output->clear();   //spaeter konfigurierbar, ob alte ausgaben loeschen
   
   char buf[512];

   QPalette pal = output->palette();
   pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
   output->setPalette(pal);

   qApp->mainWidget()->setCursor(waitCursor);

   while (fgets(buf, sizeof buf, cvs)) {
      QString str = buf;
      str.truncate(str.find("\n"));
      output->append(str);
   }
   
   pclose(cvs);

   qApp->mainWidget()->setCursor(arrowCursor);
}

//----------------------------------------------------------------------------

void Directory::addFiles(QMultiLineEdit *output, QStringList *stringList)
{
   if(!loginOk()) return;

   output->clear();  
   
   for(unsigned int i = 0; i < stringList->count(); i++) {
      QString fileName = (*stringList)[i];
      fileName = fileName.simplifyWhiteSpace();
      QFileInfo info(fileName);
      if(!info.exists()) continue;
  
      bool isCurDir = false;
      if(m_fullName.compare(info.dirPath(true)) == 0) {
         isCurDir = true;
      }
      
      QString command = "";

      if( (QString::compare(m_connectMethod, "ext") == 0)
           || (QString::compare(m_connectMethod, "") == 0) )
      {//rsh access
         command = "export CVS_RSH=" + cvsRsh + "&& ";
      }
   
      command += "cd ";
      command += info.dirPath(true);
      command += " && cvs add ";
      command += info.fileName();
      command += " 2>&1";
 
      FILE* cvs = popen(command, "r");
      if(!cvs){
         perror("popen");
         continue;
      }

      char buf[512];

      QPalette pal = output->palette();
      pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
      output->setPalette(pal);

      qApp->mainWidget()->setCursor(waitCursor);

      while (fgets(buf, sizeof buf, cvs)) {
         QString str = buf;
         str.truncate(str.find("\n"));

         //jetzt "str" analysieren
         if(isCurDir) {
            if(str.find("use 'cvs commit' to add this file permanently") > 0){
               setAndAppendEntry(fileName, added);
            }
         }

         output->append(str);
      }

      pclose(cvs);

      activate();
      qApp->mainWidget()->setCursor(arrowCursor);
   
   }
}

//----------------------------------------------------------------------------

void Directory::statusOfFile(QMultiLineEdit *output, 
                             bool clearOutput)
{
   if(!loginOk()) return;

   if(clearOutput) {
      output->clear();
   }

   QPalette pal = output->palette();
   pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
   output->setPalette(pal);

   qApp->mainWidget()->setCursor(waitCursor);

   QListViewItem *myChild = m_pFileListView->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         QString fileName = myChild->text(0).simplifyWhiteSpace();

         QString command = "";
   
         if( (QString::compare(m_connectMethod, "ext") == 0)
              || (QString::compare(m_connectMethod, "") == 0) )
         {//rsh access
            command = "export CVS_RSH=" + cvsRsh + "&& ";
         }
   
         command += "cd ";
         command += m_fullName;
         command += " && cvs status -v ";
         command += fileName;
         command += " 2>&1";
 
         FILE* cvs = popen(command, "r");
         if(!cvs){
            //ordentliche fehlerbehandlung einbauen
      
            perror("popen");
            return;
         }

         char buf[512];

         int lineNb = 0;
         bool error = false;
   
         while (fgets(buf, sizeof buf, cvs)) {
            QString str = buf;
            str.truncate(str.find("\n"));

            //jetzt "str" analysieren
            if((lineNb == 0) && (str.find("===") == -1)){//error
               error = true;
            }
            if((lineNb == 1) && !error){//check status
               if(str.find("Up-to-date") > -1){
                  setEntry(fileName, up_to_date);
               }
               else 
               if(str.find("Locally Modified") > -1){
                  setEntry(fileName, modified);
               }
               else
               if(str.find("Needs Patch") > -1){
                  setEntry(fileName, needs_patch);
               }
               else
               if(str.find("Needs Merge") > -1){
                  setEntry(fileName, needs_merge);
               }
               else
               if(str.find("Needs Checkout") > -1){
                  setEntry(fileName, needs_checkout);
               }
               else
               if(str.find("File had conflicts on merge") > -1){
                  setEntry(fileName, conflict);
               }
               else
               if(str.find("Locally Added") > -1){
                  setEntry(fileName, added);
               }
               else
               if(str.find("Locally Removed") > -1){
                  setEntry(fileName, removed);
               }
            }
      
            output->append(str);
            lineNb++;
         }
   
         pclose(cvs);
      }
      myChild = myChild->nextSibling();
   }
   
   activate();
   qApp->mainWidget()->setCursor(arrowCursor);
}

//----------------------------------------------------------------------------

void Directory::updateFile(QMultiLineEdit *output, const QString& tagParameter, 
                           bool clearOutput)
{
   if(!loginOk()) return;
   qApp->processEvents(100);

   if(clearOutput) {
      output->clear();
   }

   QPalette pal = output->palette();
   pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
   output->setPalette(pal);

   qApp->mainWidget()->setCursor(waitCursor);

   QListViewItem *myChild = m_pFileListView->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         QString fileName = myChild->text(0).simplifyWhiteSpace();

         QString command = "";
   
         if( (QString::compare(m_connectMethod, "ext") == 0)
              || (QString::compare(m_connectMethod, "") == 0) )
         {//rsh access
            command = "export CVS_RSH=" + cvsRsh + "&& ";
         }
   
         command += "cd ";
         command += m_fullName;
         command += " && cvs update ";
         if(!tagParameter.isNull()) {
            command += " " + tagParameter + " ";
         }
         command += fileName;
         command += " 2>&1";
 
         FILE* cvs = popen(command, "r");
         if(!cvs){
            //ordentliche fehlerbehandlung einbauen
      
            perror("popen");
            return;
         }

         char buf[512];
         bool noLinesRead = true;
   
         while (fgets(buf, sizeof buf, cvs)) {
            noLinesRead = false;
            QString str = buf;
            str.truncate(str.find("\n"));

            //jetzt "str" analysieren
            if((str.find("P ") == 0) || (str.find("U ") == 0)){
               //pal.setColor(QColorGroup::Text, QColor(0, 155, 0));   //green
               //output->setPalette(pal);
               setEntry(fileName, up_to_date);
            }
            else 
            if(str.find("M ") == 0){
               //pal.setColor(QColorGroup::Text, QColor(255, 0, 0));   //red
               //output->setPalette(pal);
               setEntry(fileName, modified);
            }
            else 
            if(str.find("C ") == 0){
               //pal.setColor(QColorGroup::Text, QColor(255, 0, 0));   //red
               //output->setPalette(pal);
               setEntry(fileName, conflict);
            }
            else 
            if(str.find("A ") == 0){
               //pal.setColor(QColorGroup::Text, QColor(255, 0, 0));   //red
               //output->setPalette(pal);
               setEntry(fileName, added);
            }
            else 
            if(str.find("R ") == 0){
               //pal.setColor(QColorGroup::Text, QColor(255, 0, 0));   //red
               //output->setPalette(pal);
               setEntry(fileName, removed);
            }

            output->append(str);
         }

         if(noLinesRead){//up to date
            //pal.setColor(QColorGroup::Text, QColor(0, 155, 0));   //green
            //output->setPalette(pal);
            setEntry(fileName, up_to_date);
         }

   
         pclose(cvs);
      }
      myChild = myChild->nextSibling();
   }
   
   activate();
   qApp->mainWidget()->setCursor(arrowCursor);
}

//----------------------------------------------------------------------------

void Directory::commitFile(QMultiLineEdit *output, 
                           QString comment, bool clearOutput)
{
   if(!loginOk()) return;

   if(clearOutput) {
      output->clear();
   }

   QPalette pal = output->palette();
   pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
   output->setPalette(pal);

   qApp->mainWidget()->setCursor(waitCursor);

   QListViewItem *myChild = m_pFileListView->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         QString fileName = myChild->text(0).simplifyWhiteSpace();

         QString command = "";
   
         if( (QString::compare(m_connectMethod, "ext") == 0)
              || (QString::compare(m_connectMethod, "") == 0) )
         {//rsh access
            command = "export CVS_RSH=" + cvsRsh + "&& ";
         }
   
         command += "cd ";
         command += m_fullName;
         command += " && cvs commit -m ";
         command += "\"" + comment + "\" ";
         command += fileName;
         command += " 2>&1";
 
         FILE* cvs = popen(command, "r");
         if(!cvs){
            perror("popen");
            return;
         }

         char buf[512];

         while (fgets(buf, sizeof buf, cvs)) {
            QString str = buf;
            str.truncate(str.find("\n"));

            if(str.compare("done") == 0){//commit completed successfully
               setEntry(fileName, up_to_date);
            }

            output->append(str);
         }

         pclose(cvs);
      }
      myChild = myChild->nextSibling();
   }
   
   activate();
   qApp->mainWidget()->setCursor(arrowCursor);
}

//----------------------------------------------------------------------------

void Directory::anyCommandFile(QMultiLineEdit *output, 
                               QString cvsCommand, bool clearOutput)
{
   if(!loginOk()) return;

   if(clearOutput) {
      output->clear();
   }

   QPalette pal = output->palette();
   pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
   output->setPalette(pal);

   qApp->mainWidget()->setCursor(waitCursor);

   QListViewItem *myChild = m_pFileListView->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         QString fileName = myChild->text(0).simplifyWhiteSpace();

         QString command = "";
   
         if( (QString::compare(m_connectMethod, "ext") == 0)
              || (QString::compare(m_connectMethod, "") == 0) )
         {//rsh access
            command = "export CVS_RSH=" + cvsRsh + "&& ";
         }
   
         command += "cd ";
         command += m_fullName;
         command += " && cvs " + cvsCommand + " ";
         command += fileName;
         command += " 2>&1";
 
         FILE* cvs = popen(command, "r");
         if(!cvs){
            perror("popen");
            return;
         }

         char buf[512];

         while (fgets(buf, sizeof buf, cvs)) {
            QString str = buf;
            str.truncate(str.find("\n"));
            output->append(str);
         }

         pclose(cvs);
      }
      myChild = myChild->nextSibling();
   }
   
   qApp->mainWidget()->setCursor(arrowCursor);
}

//----------------------------------------------------------------------------

void Directory::removeFile(QMultiLineEdit *output, bool& showWarning, bool clearOutput)
{
   if(!loginOk()) return;

   if(clearOutput) {
      output->clear();
   }

   bool firstTime = true;
   
   qApp->mainWidget()->setCursor(waitCursor);

   QString absPath = m_fullName;
   if(absPath.length() > 1){//is not root
      absPath += "/";
   }
  
   QListViewItem *myChild = m_pFileListView->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         QString fileName = myChild->text(0).simplifyWhiteSpace();   //koennte problematisch werden bei dateinamen mit mehreren zusammenhaengenden leerzeichen

         QFile f(absPath + fileName);
         if(f.exists()) {//remove from disk
            if(firstTime && showWarning) {
               firstTime = false;
             
               //create dialog  
               QDialog *dlg = new QDialog(qApp->mainWidget(), "remove warning", true);
               dlg->setCaption(dlg->tr("Warning"));
               QBoxLayout *topLayer = new QVBoxLayout(dlg, 5);

               QLabel *label = new QLabel(QObject::tr("This command will remove file(s) from disk!"), dlg);
               topLayer->addWidget(label);
               topLayer->addSpacing(10);

               QCheckBox *checkBox = new QCheckBox(QObject::tr("Show warning next time"), dlg);               
               checkBox->setChecked(true);
               topLayer->addWidget(checkBox);
               topLayer->addSpacing(15);

               QBoxLayout *buttonLayer = new QHBoxLayout(topLayer, 5);
               buttonLayer->addStretch();
               QPushButton *ok = new QPushButton(dlg);
               ok->setText(dlg->tr("Continue"));
               ok->setDefault(true);
               buttonLayer->addWidget(ok);
               buttonLayer->addStretch();
               QPushButton *cancel = new QPushButton(dlg);
               cancel->setText(dlg->tr("Cancel"));
               buttonLayer->addWidget(cancel);
               buttonLayer->addStretch();
   
               dlg->connect(ok, SIGNAL(clicked()), dlg, SLOT(accept()));
               dlg->connect(cancel, SIGNAL(clicked()), dlg, SLOT(reject()));
               
               if(!dlg->exec()) {
                  delete dlg;
                  qApp->mainWidget()->setCursor(arrowCursor);
                  return;
               }
               showWarning = checkBox->isChecked();
               delete dlg;
            }
         

            f.remove();   //spaeter auf erfolg testen
         }
  
         QString command = "";
   
         if( (QString::compare(m_connectMethod, "ext") == 0)
              || (QString::compare(m_connectMethod, "") == 0) )
         {//rsh access
            command = "export CVS_RSH=" + cvsRsh + "&& ";
         }
   
         command += "cd ";
         command += m_fullName;
         command += " && cvs remove ";
         command += fileName;
         command += " 2>&1";
 
         FILE* cvs = popen(command, "r");
         if(!cvs){
            perror("popen");
            continue;
         }

         char buf[512];

         QPalette pal = output->palette();
         pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
         output->setPalette(pal);

         while (fgets(buf, sizeof buf, cvs)) {
            QString str = buf;
            str.truncate(str.find("\n"));
            output->append(str);
         }

         pclose(cvs);

      }
      myChild = myChild->nextSibling();
   }
   
   activate();
   qApp->mainWidget()->setCursor(arrowCursor);
}

//----------------------------------------------------------------------------

void Directory::setAllToUpToDate()
{
   //alle entries im eignen verzeichnis auf uptodate setzen
   QString path = m_fullName;
   if(path.length() > 1){//is not root
      path += "/";
   }

   QFile f;
   QString line, name;
   int posLeft;
   
   //read Entries
   f.setName(path + "CVS/Entries");
   QFileInfo fInfo(f);
   m_modTimeOfEntries = fInfo.lastModified().toString().simplifyWhiteSpace();
   
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 

      while(!textStream.atEnd()) {
         line = textStream.readLine();
         if(line.find("D") == 0) continue;   //entry is a directory

         if(line.find("/") == 0) {//entry seems to be correct
            if((posLeft = line.find("/", 1)) > 1) {
               name = line.mid(1, posLeft - 1);
            }   
            
            setAndAppendEntry(name, up_to_date);            
         }
      }
      f.close();
   }

   //jetzt die subdirs...
   Directory *myChild = (Directory*)firstChild();
   while(myChild) {
      myChild->setAllToUpToDate();
      myChild = (Directory*)myChild->nextSibling();
   }
}

//----------------------------------------------------------------------------

bool Directory::setEntry(QString fileName, int stateId)
{
   bool foundEntry = false;
   Entry *entry = m_entries.first();
   for(unsigned int i = 0; i < (m_entries.count()); i++) {
      if(entry->fileName().compare(fileName) == 0) {
         //status auslesen und umsetzen
         entry->setState(stateId);
         i = m_entries.count();
         foundEntry = true;
      }
      entry = m_entries.next();
   }
   
   return foundEntry;
}

//----------------------------------------------------------------------------

void Directory::setAndAppendEntry(QString fileName, int stateId)
{
   if(!setEntry(fileName, stateId))
      m_entries.append(new Entry(fileName, stateId));
}

//----------------------------------------------------------------------------

int Directory::alignWithEntries(QString name, int stateId)
{
   bool isCached = false;
   //is item cached?

   Entry *entry = m_entries.first();
   while(entry && !isCached) {
      if(entry->fileName().compare(name) == 0) {//datei wurde schon erfasst
         //status auslesen und umsetzen
         switch(stateId)
         {
            case modified:
               if((entry->state() == needs_patch)
                  || (entry->state() == needs_merge)) {
                  entry->setState(needs_merge);
                  stateId = needs_merge;
               }
               else{
                  if(entry->state() == added) {
                     stateId = added;
                  }
                  else
                     entry->setState(modified);
               }
               break;
            case missing:
               if(entry->state() == needs_checkout)
                  stateId = needs_checkout;
               break;
            case probably_up_to_date:
               if((entry->state() == added) || (entry->state() == modified)) {
                  entry->setState(probably_up_to_date);
               }
            case probably_up_to_date_and_timezone_incorrect:
               if((entry->state() == added) || (entry->state() == modified)) {
                  entry->setState(probably_up_to_date);
               }
            default:
               stateId = entry->state();
               break;
         }            

         isCached = true;
      }
      entry = m_entries.next();
   }
   
   if(!isCached) {
      m_entries.append(new Entry(name, stateId));
   }
   
   return stateId;
}

//----------------------------------------------------------------------------

void Directory::checkStatus(QListViewItem *item)
{
   QString path = m_fullName;
   if(path.length() > 1){//is not root
      path += "/";
   }

   QFile f;
   QString line;
   
   //read Entries
   f.setName(path + "CVS/Entries");
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 

      if(item) {//only one item
         bool foundFile = false;
         QString fileName = item->text(0).simplifyWhiteSpace();
         while(!textStream.atEnd() && !foundFile) {
            line = textStream.readLine();
            if(line.find("D") == 0) continue;   //entry is a directory

            if(line.find("/" + fileName + "/") == 0) {
               foundFile = true;
               setStatusInFileListView(item, line);
            }
         }
      }
      else {//all entries
         QFileInfo fInfo(f);
         m_modTimeOfEntries = fInfo.lastModified().toString().simplifyWhiteSpace();
         while(!textStream.atEnd()) {
            line = textStream.readLine();
            if(line.find("D") == 0) continue;   //entry is a directory

            if(line.find("/") == 0) {//entry seems to be correct
               setStatusInFileListView(0, line);
            }      
         }
      }
      f.close();
   }
}

//----------------------------------------------------------------------------

void Directory::setStatusInFileListView(QListViewItem *item, QString line)
{
   int stateId;
   QString name, rev, date, state, modDateLocal, modDateUtc;
   int posLeft, posRight;
   QPixmap *pix;
   QPixmap pixUnchanged(unchangedfile);
   QPixmap pixUnchangedTimezoneConflict(unchangedfiletimezoneconflict);
   QPixmap pixModified(modifiedfile);
   QPixmap pixNeedsPatch(needspatch);
   QPixmap pixNeedsMerge(needsmerge);
   QPixmap pixNeedsCheckout(needscheckout);
   QPixmap pixMissing(lostfile);
   QPixmap pixConflict(conflictfile);
   QPixmap pixAdded(addedfile);
   QPixmap pixRemoved(lostfile);
   QDateTime localDate;
   QString localDateString;

   QString path = m_fullName;
   if(path.length() > 1){//is not root
      path += "/";
   }

   if((posLeft = line.find("/", 1)) > 1) {
      name = line.mid(1, posLeft - 1);
      if((posRight = line.find("/", posLeft + 1)) > 1) {
         rev  = line.mid(posLeft + 1, posRight - 1 - posLeft);
         posLeft = posRight;
         if((posRight = line.find("/", posLeft + 1)) > 1) {
            date = line.mid(posLeft + 1, posRight - 1 - posLeft);

            //status?
            QFileInfo info(path + name);
            if(info.exists()){
               localDate = info.lastModified();
               localDate = localDate.addSecs(timeZoneDiffInSecs);
               localDateString = localDate.toString().simplifyWhiteSpace();
               if(localDateString.compare(date.simplifyWhiteSpace()) == 0){
                  stateId = probably_up_to_date;
               }
               else {
                  if(date.find("dummy timestamp") > -1) {
                     stateId = added;
                  }
                  else {
                     QString localMinSecs = localDateString.mid(localDateString.find(":"), 6);
                     QString reposMinSecs = date.mid(date.find(":"), 6);
               
                     if(localMinSecs.compare(reposMinSecs) == 0){
                        stateId = probably_up_to_date_and_timezone_incorrect;
                        //jetzt warnung ausgeben
                     }
                     else
                        stateId = modified;
                  }
               }
            }
            else
               stateId = missing;
               
            stateId = alignWithEntries(name, stateId);
   
            modDateLocal = "";
            modDateUtc = "";
                     
            switch(stateId)
            {
               case probably_up_to_date:
                  state = QObject::tr("seems up to date") + "  ";   //
                  pix = &pixUnchanged;
                  break;
               case probably_up_to_date_and_timezone_incorrect:
                  state = QObject::tr("seems up to date") + "  ";   //
                  //pix = &pixUnchangedTimezoneConflict;
                  pix = &pixUnchanged;
                  break;
               case up_to_date:
                  state = QObject::tr("up to date") + "  ";   //
                  pix = &pixUnchanged;
                  break;
               case modified:
                  state = QObject::tr("modified") + "  ";   //
                  pix = &pixModified;

                  //doppelten aufruf einsparen!
                  localDate = info.lastModified();
                  modDateLocal = localDate.toString();
                  localDate = localDate.addSecs(timeZoneDiffInSecs);
                  modDateUtc = localDate.toString();
                  break;
               case needs_patch:
                  state = QObject::tr("needs patch") + "  ";   //
                  pix = &pixNeedsPatch;
                  break;
               case needs_merge:
                  state = QObject::tr("needs merge") + "  ";   //
                  pix = &pixNeedsMerge;   //noch ersetzen

                  //doppelten aufruf einsparen!
                  localDate = info.lastModified();
                  modDateLocal = localDate.toString();
                  localDate = localDate.addSecs(timeZoneDiffInSecs);
                  modDateUtc = localDate.toString();
                  break;
               case needs_checkout:
                  state = QObject::tr("needs checkout") + "  ";   //
                  pix = &pixNeedsCheckout;
                  break;
               case missing:
                  state = QObject::tr("missing") + "  ";   //
                  pix = &pixMissing;
                  break;
               case conflict:
                  state = QObject::tr("conflict") + "  ";   //
                  pix = &pixConflict;
                  break;
               case added:
               //welche zeit anzeigen???
                  state = QObject::tr("added") + "  ";   //
                  pix = &pixAdded;
                  break;
               case removed:
               //welche zeit anzeigen???
                  state = QObject::tr("removed") + "  ";   //
                  pix = &pixRemoved;
                  break;
               default:
                  state = QObject::tr("unknown") + "  ";
                  pix = &pixUnchanged;
            }

            if(item){//item exists
               item->setText(0, name + "  "); 
               item->setText(1, rev + "  "); 
               item->setText(2, state); 
               item->setText(3, date + "  "); 
               item->setText(4, modDateLocal + "  "); 
               item->setText(5, modDateUtc + "  "); 
            }
            else {//create new item
               item = new QListViewItem(m_pFileListView, name + "  ", rev + "  ", state, date + "  ", modDateLocal + "  ", modDateUtc + "  ");
            }
            item->setPixmap(0, *pix);
         }
      }
   }
}

//----------------------------------------------------------------------------

bool Directory::entriesFileModified()
{
   QString path = m_fullName;
   if(path.length() > 1){//is not root
      path += "/";
   }

   QFileInfo fInfo(path + "CVS/Entries");
   if (m_modTimeOfEntries.compare(fInfo.lastModified().toString().simplifyWhiteSpace()) == 0)
      return false;
   else
      return true;
}

//----------------------------------------------------------------------------

bool Directory::loginOk(bool showMessage)
{
   if(!m_haveCvsDir) {
      QMessageBox::warning(0, QObject::tr("Warning"), QObject::tr("Is not a CVS directory."), 0);
      return false;
   }

   if(m_connectMethod.find("pserver") == -1) {
      if(showMessage) {
         QMessageBox::information(0, QObject::tr("Information"), 
                                  QObject::tr("No login necessary. Its a local CVS repository"), 0);
      }

      return true;
   }

	
	QString cvsRoot = ":" + m_connectMethod + ":" + m_userName + 
                     "@" + m_host + ":" + m_rootDir;

   if(!isInCvsPass(cvsRoot)){
      login(cvsRoot);
      qApp->processEvents(100);
      sleep(1);
      qApp->processEvents(100);
            
      if(!isInCvsPass(cvsRoot)) {
         QMessageBox::warning(qApp->mainWidget(), 
                              QObject::tr("Warning"), QObject::tr("Login failed."), 0);
         return false;
      }

      if(showMessage) {
         QMessageBox::information(qApp->mainWidget(), 
                                  QObject::tr("Information"), QObject::tr("Login successfully."), 0);
      }
      
      return true;
   }
   
   if(showMessage) {
      QMessageBox::information(qApp->mainWidget(), 
                               QObject::tr("Information"), QObject::tr("You are already logged in."), 0);
   }
      
   return true;
}

//----------------------------------------------------------------------------
                      
