// main.cc, Game of Life
// Copyright (c) 2003 Jesper Strandberg (skywarper@hotbrev.com)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include <iostream>
#include <exception>
#include <SDL.h>
#include "Video.hh"
#include "Field.hh"

int main(int argc, char **argv) {
    int width  = 100;
    int height = 100;
    int grid   = 5;
    int speed  = 100;

    if(argc == 2) {
        std::cerr << "Usage: " << argv[0] << " [width height [grid]]\n";
        return 1;
    }

    if(argc > 2) {
        width  = atoi(argv[1]);
        height = atoi(argv[2]);

        if(argc == 4) grid  = atoi(argv[3]);
    }

    try {
        SDL_Init(0);
        Video *video = Video::init((width*grid)-1, (height*grid)-1, 0, "Game of Life", SDL_HWSURFACE);
        Surface *surface = new Surface((width*grid)-1, (height*grid)-1, SDL_HWSURFACE);

        Life::Field *f = new Life::Field(width, height, grid);

        SDL_Event event;
        bool quit  = false;
        bool pause = true;
        bool draw  = false;
        bool button= false;
        Uint32 time= 0;

        while(!quit) {
            f->render(surface);
            surface->blit();

            if(!pause) {
                if(Uint32(speed) < SDL_GetTicks() - time) {
                    f->update();
                    time = SDL_GetTicks();
                }
            } else {
                f->drawGrid(video->getScreen(), video->mapRGB(64, 0, 0));
            }

            video->update();

            while(SDL_PollEvent(&event)) {
                if(event.type == SDL_QUIT)
                    quit = true;
                if(event.type == SDL_KEYDOWN) {
                    switch(event.key.keysym.sym) {
                        case SDLK_ESCAPE:
                            quit = true;
                            break;
                        case SDLK_f:
                            video->toggleFullscreen();
                            break;
                        case SDLK_c:
                            f->clear();
                            break;
                        case SDLK_SPACE:
                            pause = !pause;
                            break;
                        case SDLK_1:
                            speed = 450;
                            break;
                        case SDLK_2:
                            speed = 400;
                            break;
                        case SDLK_3:
                            speed = 350;
                            break;
                        case SDLK_4:
                            speed = 300;
                            break;
                        case SDLK_5:
                            speed = 250;
                            break;
                        case SDLK_6:
                            speed = 200;
                            break;
                        case SDLK_7:
                            speed = 150;
                            break;
                        case SDLK_8:
                            speed = 100;
                            break;
                        case SDLK_9:
                            speed = 50;
                            break;
                        case SDLK_0:
                            speed = 0;
                            break;
                        default:
                            break;
                    }
                }
                if(event.type == SDL_MOUSEBUTTONDOWN) {
                    int x = event.button.x;
                    int y = event.button.y;
                    draw   = !f->getXY(event.button.x, event.button.y);
                    button = true;

                    f->setXY(x, y, draw);
                }

                if(event.type == SDL_MOUSEBUTTONUP)
                    button = false;

                if(event.type == SDL_MOUSEMOTION && button) {
                    int x = event.motion.x;
                    int y = event.motion.y;

                    f->setXY(x, y, draw);
                }
            }
        }

        delete f;
        delete surface;
        delete video;

        SDL_Quit();
    }
    catch(std::exception &e) {
        std::cerr << "Error: " << e.what() << "\n";
        std::cerr << "Aborting." << "\n";
    }
    catch(...) {
        std::cerr << "Unknown exception thrown\n";
        std::cerr << "Aborting." << "\n";
    }

    return 0;
}

