// Video.cc, Video class for SDL
// Copyright (c) 2002 - 2003 Jesper Strandberg (skywarper@hotbrev.com)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "Video.hh"

#include <string>
#include <SDL.h>
#include "Surface.hh"
#include "Error.hh"

Video *Video::mVideo = 0;
Surface *Video::mScreen = 0;

Video::Video(int width, int height, int bpp, const std::string &caption, int flags) {
    if(SDL_InitSubSystem(SDL_INIT_VIDEO) < 0)
        throw Error(SDL_GetError());

    SDL_Surface *temp = SDL_SetVideoMode(width, height, bpp, flags);
    if(!temp)
        throw Error(SDL_GetError());
    mScreen = new Surface(temp);

    SDL_WM_SetCaption(caption.c_str(), 0);

    if(SDL_FULLSCREEN & flags)
        mFullscreen = true;
    else
        mFullscreen = false;

    mCursor = true;
}

Video *Video::init(int width, int height, int bpp, const std::string &caption, int flags) {
    if(!mVideo)
        mVideo = new Video(width, height, bpp, caption, flags);

    return mVideo;
}

void Video::update() const {
    if(SDL_Flip(mScreen->getSDLSurface()) < 0)
        throw Error(SDL_GetError());
}

void Video::setFullscreen() {
#   ifndef WIN32
        if(!mFullscreen) {
            if(SDL_WM_ToggleFullScreen(mScreen->getSDLSurface()) == 0)
                throw Error(SDL_GetError());
        }
#   endif // WIN32

#   ifdef WIN32
        if(!mFullscreen)
            SDL_SetVideoMode(mScreen->getWidth(),
                             mScreen->getHeight(),
                             mScreen->getFormat()->BitsPerPixel,
                             SDL_HWSURFACE|SDL_FULLSCREEN);
#   endif // WIN32

    mFullscreen = true;
}

void Video::setWindowed() {
#   ifndef WIN32
        if(mFullscreen) {
            if(SDL_WM_ToggleFullScreen(mScreen->getSDLSurface()) == 0)
                    throw Error(SDL_GetError());
        }
#   endif // WIN32

#   ifdef WIN32
        if(mFullscreen)
            SDL_SetVideoMode(mScreen->getWidth(),
                             mScreen->getHeight(),
                             mScreen->getFormat()->BitsPerPixel,
                             SDL_HWSURFACE);
#   endif // WIN32

    mFullscreen = false;
}

void Video::toggleFullscreen() {
#   ifndef WIN32
        if(SDL_WM_ToggleFullScreen(mScreen->getSDLSurface()) == 0)
            throw Error(SDL_GetError());
#   endif // WIN32

#   ifdef WIN32
        Uint32 flags = SDL_HWSURFACE|SDL_FULLSCREEN;
        if(mFullscreen)
            flags = SDL_HWSURFACE;

        SDL_SetVideoMode(mScreen->getWidth(),
                         mScreen->getHeight(),
                         mScreen->getFormat()->BitsPerPixel,
                         flags);
#   endif // WIN32

    mFullscreen = !mFullscreen;
}

void Video::hideCursor() {
    SDL_ShowCursor(SDL_DISABLE);
    mCursor = false;
}

void Video::showCursor() {
    SDL_ShowCursor(SDL_ENABLE);
    mCursor = true;
}

void Video::toggleCursor() {
    if(mCursor) {
        SDL_ShowCursor(SDL_DISABLE);
        mCursor = false;
    } else {
        SDL_ShowCursor(SDL_ENABLE);
        mCursor = true;
    }
}

