// Field.cc, Game of Life
// Copyright (c) 2003 Jesper Strandberg (skywarper@hotbrev.com)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "Field.hh"

#include <iostream>
#include <vector>
#include <algorithm>
#include <SDL.h>
#include "Video.hh"

Life::Field::Field(int width, int height, int grid) :
    mField(width*height),
    mWidth(width),
    mHeight(height),
    mGrid(grid)
{ }

void Life::Field::setGrid(int x, int y, bool alive) {
    while(x < 0) x += mWidth;
    while(y < 0) y += mHeight;

    mField[mWidth*(y%mHeight)+(x%mWidth)] = alive;
}

bool Life::Field::getGrid(int x, int y) {
    while(x < 0) x += mWidth;
    while(y < 0) y += mHeight;

    return mField[mWidth*(y%mHeight)+(x%mWidth)];
}

void Life::Field::setXY(int x, int y, bool alive) {
    if(x) x /= mGrid;
    if(y) y /= mGrid;
    mField[mWidth*y+x] = alive;
}

bool Life::Field::getXY(int x, int y) {
    if(x) x /= mGrid;
    if(y) y /= mGrid;
    return mField[mWidth*y+x];
}

void Life::Field::clear(bool alive) {
    std::fill(mField.begin(), mField.end(), alive);
}

void Life::Field::update() {
    std::vector<bool> temp(mWidth*mHeight);
    int count;

    for(int y = 0; y < mHeight; y++) {
        for(int x = 0; x < mWidth; x++) {
            count = 0;

            if(getGrid(x-1, y-1)) count++;
            if(getGrid(x+1, y-1)) count++;
            if(getGrid(x-1, y+1)) count++;
            if(getGrid(x+1, y+1)) count++;
            if(getGrid(x, y-1))   count++;
            if(getGrid(x, y+1))   count++;
            if(getGrid(x-1, y))   count++;
            if(getGrid(x+1, y))   count++;

            if(count == 3 && !getGrid(x, y))
                temp[mWidth*y+x] = true;
            else {
                if((count == 2 || count == 3) && getGrid(x, y))
                    temp[mWidth*y+x] = true;
                else
                    temp[mWidth*y+x] = false;
            }
        }
    }

    std::copy(temp.begin(), temp.end(), mField.begin());
}

void Life::Field::render(Surface *surface) {
    SDL_Rect dstRect;
    dstRect.w = dstRect.h = mGrid-1;

    surface->fill(0);
    
    for(int y = 0; y < mHeight; y++) {
        for(int x = 0; x < mWidth; x++) {
            if(getGrid(x, y)) {
                dstRect.x = x*mGrid;
                dstRect.y = y*mGrid;
                surface->fill(Video::mapRGB(255, 255, 255), &dstRect);
            }
        }
    }
}

void Life::Field::drawGrid(Surface *surface, Uint32 color) {
    SDL_Rect dstRect;

    dstRect.w = 1;
    dstRect.h = mHeight*mGrid;
    dstRect.y = 0;

    for(dstRect.x = mGrid-1; dstRect.x < mWidth*mGrid; dstRect.x += mGrid)
        surface->fill(color, &dstRect);

    dstRect.w = mWidth*mGrid;
    dstRect.h = 1;
    dstRect.x = 0;

    for(dstRect.y = mGrid-1; dstRect.y < mHeight*mGrid; dstRect.y += mGrid)
        surface->fill(color, &dstRect);
}

