#ifndef USER_H
#define USER_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <vector.h>
#include <qfile.h>

#include "icq-defines.h"
#include "socket.h"
#include "file.h"
#include "message.h"
#include "userfcndlg.h"


//+++++STRUCTURES+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//-----UserBasicInfo------------------------------------------------------------
struct UserBasicInfo
{
     char alias[32];
     char uin[16];
     char name[64];
     char firstname[32];
     char lastname[32];
     char email[64];
     char status[16];
     char ip[16];
     char port[6];
     char ip_port[24];
     char history[MAX_FILENAME_LEN];
     char awayMessage[MAX_MESSAGE_SIZE];
};

//-----UserExtInfo--------------------------------------------------------------
struct UserExtInfo
{
   char city[64];
   char state[64];
   char country[32];
   char phone[32];
   char age[16];
   char sex[10];
   char homepage[256];
   char about[MAX_MESSAGE_SIZE];
};


//+++++OBJECTS++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=====ICQUser==================================================================
class ICQUser
{
public:
  ICQUser(unsigned long id, char *filename);
  ICQUser(unsigned long id);
  ICQUser(unsigned long id, TCPSocket &sock);
  virtual ~ICQUser(void);
  void removeFiles(void);
  
  TCPSocket tcpSocket;
  QFile *history;
  CUserFunctionDlg *fcnDlg;

  // user info functions
  void setDefaults(unsigned long id);
  virtual void getBasicInfo(struct UserBasicInfo &us);
  virtual void getExtInfo(struct UserExtInfo &ud);
  void saveInfo(void);
  void saveBasicInfo(void);
  void saveExtInfo(void);
  bool isAway(void);
   
  // Accessors
  char *getAlias(void)         { return(m_sAlias); }
  char *getLastName(void)      { return(m_sLastName); }
  char *getFirstName(void)     { return(m_sFirstName); }
  char *getEmail(void)         { return(m_sEmail); }
  char *getHistoryFile(void)   { return(m_sHistoryFile); }
  char *getAwayMessage(void)   { return(m_sAwayMessage); }
  char *getCity(void)          { return(m_sCity); }
  char *getState(void)         { return(m_sState); }
  char *getPhoneNumber(void)   { return(m_sPhoneNumber); }
  char *getAbout(void)         { return(m_sAbout); }
  char *getHomepage(void)      { return(m_sHomepage); }
  bool getIsNew(void)         { return(m_bIsNew); }
  bool getInvisibleList(void) { return(m_bInvisibleList); }
  bool getVisibleList(void)   { return(m_bVisibleList); }
  bool getOnlineNotify(void)  { return(m_bOnlineNotify); }
  bool getSendServer(void)    { return(m_bSendServer); }
  bool getEnableSave(void)    { return(m_bEnableSave); }
  bool getAuthorization(void)  { return m_bAuthorization; }
  bool ShowAwayMsg(void)      { return m_bShowAwayMsg; }
  unsigned long getUin(void)           { return(m_nUin); }
  unsigned long getGroup(void)         { return(m_nGroup); }
  unsigned short getAge(void)           { return(m_nAge); }
  unsigned short getSexNum(void)        { return(m_nSex); }
  unsigned long getStatusFull(void)  { return m_nStatus; }
  unsigned short getCountryCode(void)  { return m_nCountryCode; }
  unsigned long getStatusFlags(void)  { return m_nStatus & ICQ_STATUS_FxFLAGS; }
  unsigned short getStatus(void);
  bool getStatusInvisible(void);
  bool getStatusWebPresence(void);
  bool getStatusHideIp(void);
  bool getStatusBirthday(void);
  bool getStatusOffline(void);
  unsigned long getSequence(bool = false);
  char *getSex(void);
  char *getCountry(void);
  const char *getStatusStr(void);
  
   // Settors
  void setEnableSave(bool s)          { m_bEnableSave = s; }
  void setSendServer(bool s)          { m_bSendServer = s; }
  void setSequence(unsigned long s)   { m_nSequence = s; }
  void setIsNew(bool s)               { m_bIsNew = s; saveInfo(); }
  void setInvisibleList(bool s)       { m_bInvisibleList = s; saveInfo(); }
  void setVisibleList(bool s)         { m_bVisibleList = s; saveInfo(); }
  void setOnlineNotify(bool s)        { m_bOnlineNotify = s; saveInfo(); }
  void setAuthorization(bool s)       { m_bAuthorization = s; saveBasicInfo(); }
  void setGroup(unsigned long s)      { m_nGroup = s; saveInfo(); }
  void setAwayMessage(const char *s)  { setStrField(m_sAwayMessage, s); }  
  void setUin(unsigned long s)        { m_nUin = s; }
  void setStatusOffline(void)         { m_nStatus |= ICQ_STATUS_OFFLINE; };
  void setAlias(const char *s)        { setStrField(m_sAlias, s); saveBasicInfo(); }
  void setEmail(const char *s)        { setStrField(m_sEmail, s); saveBasicInfo(); } 
  void setFirstName(const char *s)    { setStrField(m_sFirstName, s); saveBasicInfo(); }
  void setLastName(const char *s)     { setStrField(m_sLastName, s); saveBasicInfo(); }
  void setAge(unsigned short s)       { m_nAge = s; saveExtInfo(); }
  void setCity(const char *s)         { setStrField(m_sCity, s); saveExtInfo(); }
  void setState(const char *s)        { setStrField(m_sState, s); saveExtInfo(); }
  void setPhoneNumber(const char *s)  { setStrField(m_sPhoneNumber, s); saveExtInfo(); }
  void setHomepage(const char *s)     { setStrField(m_sHomepage, s); saveExtInfo(); }
  void setAbout(const char *s)        { setStrField(m_sAbout, s); saveExtInfo(); }
  void setSex(unsigned short s)       { m_nSex = s; saveExtInfo(); }
  void setCountry(unsigned short s)   { m_nCountryCode = s; saveExtInfo(); }; 
  void setShowAwayMsg(bool s)         { m_bShowAwayMsg = s; };

  void setStatus(unsigned long);
  void setHistoryFile(const char *, bool = true);

  // Message functions
  unsigned short getNumMessages(void)   { return(m_vcMessages.size()); }
  CUserEvent *GetEvent(unsigned short);
  void ClearEvent(unsigned short);
  void AddEvent(CUserEvent *);
  void WriteToHistory(const char *);

  // Group functions
  bool getIsInGroup(unsigned short);
  void addToGroup(unsigned short);
  void removeFromGroup(unsigned short);

  static unsigned short getNumUserEvents(void)  { return s_nNumUserEvents; };
  static void incNumUserEvents(void)  { s_nNumUserEvents++; };
  static void decNumUserEvents(void)  { s_nNumUserEvents--; };
   
protected:
   ICQUser(void) { /* ICQOwner inherited constructor - does nothing */ };
   bool loadDataFromFile(void);
   void setStrField(char *&, const char *);
   virtual void setStrNull(void);
   void openHistory(void);

   CIniFile m_fConf;
   unsigned long m_nUin,
                 m_nStatus, 
                 m_nSequence;
   unsigned short m_nGroup,
                  m_nSex,
                  m_nCountryCode, 
                  m_nAge;
   char *m_sAlias,
        *m_sFirstName,
        *m_sLastName,
        *m_sEmail,
        *m_sHistoryFile,
        *m_sAwayMessage,
        *m_sCity,
        *m_sState,
        *m_sPhoneNumber,
        *m_sHomepage,
        *m_sAbout;
   bool m_bOnline,
        m_bIsNew, 
        m_bOnlineNotify,
        m_bSendServer,
        m_bVisibleList,
        m_bInvisibleList,
        m_bEnableSave,
        m_bAuthorization,
        m_bShowAwayMsg;
   vector <class CUserEvent *> m_vcMessages;

   static unsigned short s_nNumUserEvents;
};


//=====ICQOwner=================================================================
class ICQOwner : public ICQUser
{
public:
   ICQOwner(void);
   char *getPassword(void);
   void setPassword(const char *);
   unsigned short getSearchSequence(void);
   virtual void getBasicInfo(struct UserBasicInfo &us);
   virtual void getExtInfo(struct UserExtInfo &us);
protected:
   virtual void setStrNull(void);
   char *m_sPassword;
   unsigned short m_nSearchSequence;
};


//=====CUserGroup===============================================================
class CUserGroup
{
public:
   CUserGroup(char *_sName);
   ~CUserGroup(void);

   void addUser(ICQUser *);
   void removeUser(ICQUser *);
   unsigned short getNumUsers(void)  { return m_vpcUsers.size(); };
   char *getName(void)  { return m_sName; };
   ICQUser *getUser(unsigned short);
   bool getIsOwner(void)  { return m_bIsOwner; };
   void setIsOwner(bool _bIsOwner)  { m_bIsOwner = _bIsOwner; };
   void reorder(ICQUser *_pcUser);
protected:
   void setUser(unsigned short i, ICQUser *u);
   
   char *m_sName;
   vector <ICQUser *> m_vpcUsers;
   bool m_bIsOwner;
};


//=====CUsers===================================================================

class CUsers
{
public:
   CUsers(void);

   ICQUser *addUser(ICQUser *);
   void removeUser(ICQUser *);
   void addGroup(CUserGroup *);
   unsigned short getNumGroups(void)  { return m_vpcGroups.size(); };
   CUserGroup *getGroup(unsigned short);
   ICQUser *getUserByUin(unsigned long);
   void reorder(ICQUser *_pcUser);
   void saveAllUsers(void);
   void addUserToGroup(ICQUser *_pcUser, unsigned short _nGroup);
   void removeUserFromGroup(ICQUser *_pcUser, unsigned short _nGroup);

   // Define some functions to act by default on group 0 (all users)
   unsigned short getNumUsers(void)  { return getGroup(0)->getNumUsers(); };
   ICQUser *getUser(unsigned short _nUser)
      { return (getGroup(0)->getUser(_nUser)); };

protected:
   vector <CUserGroup *> m_vpcGroups;
};


#endif
