#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>       
#include <netinet/in.h> 
#include <math.h>

#include "user.h"
#include "countrycodes.h"
#include "log.h"


//=====CUserGroup===============================================================
CUserGroup::CUserGroup(char *_sName)//, bool _bOnContactList)
{
   m_sName = new char[strlen(_sName) + 1];
   strcpy(m_sName, _sName);
   setIsOwner(false);
}

CUserGroup::~CUserGroup(void)
{
   delete[] m_sName;
   // if we are the owner of the pointers delete all the users
   if (getIsOwner()) 
      for (unsigned short i = 0; i < getNumUsers(); i++) 
         delete m_vpcUsers[i];
}


//-----CUserGroup::addUser---------------------------------------------------
void CUserGroup::addUser(ICQUser *_pcUser)
{
   m_vpcUsers.push_back(_pcUser);
   reorder(_pcUser);
}

//-----CUserGroup::removeUser---------------------------------------------------
void CUserGroup::removeUser(ICQUser *_pcUser)
{
   unsigned short uNum = 0;
   while (uNum < getNumUsers() && _pcUser != getUser(uNum)) uNum++;
   for (unsigned short i = uNum; i < getNumUsers() - 1; i++) 
      setUser(i, getUser(i + 1));
   m_vpcUsers.pop_back();
}


//-----CUserGroup::setUser------------------------------------------------------
void CUserGroup::setUser(unsigned short i, ICQUser *u)
{
   if (i < getNumUsers()) m_vpcUsers[i] = u;
}


//-----CUserGroup::getUser------------------------------------------------------
ICQUser *CUserGroup::getUser(unsigned short i)
{
   if (i >= getNumUsers()) 
      return NULL;
   else
      return m_vpcUsers[i];
}

//-----CUserGroup---------------------------------------------------------------
void CUserGroup::reorder(ICQUser *_pcUser)
{
   unsigned short nPos = 0;
   while (nPos < getNumUsers() && _pcUser != getUser(nPos)) 
      nPos++;
   if (nPos == getNumUsers()) return;
   
   // Now nPos is the position of the user to be reordered 
   // => _pcUser = getUser(nPos)
   if (nPos != 0 && _pcUser->getStatus() <= getUser(nPos - 1)->getStatus())
   {
      // Move user up the list
      unsigned short i;
      for (i = nPos; i > 0 && _pcUser->getStatus() <= getUser(i - 1)->getStatus(); i--) 
      {
         setUser(i, getUser(i - 1));
      }
      setUser(i, _pcUser);
   }
   else
   {
      // Move user down the list
      unsigned short i;
      for (i = nPos; i < (getNumUsers() - 1) && _pcUser->getStatus() > getUser(i + 1)->getStatus(); i++) 
      {
         setUser(i, getUser(i + 1));
      }
      setUser(i, _pcUser);
   }
}



//=====CUsers===================================================================
CUsers::CUsers(void)
{
   gLog.Info("-> User configuration.\n");
   
   // Set up the basic all users and new users group
   CUserGroup *g = new CUserGroup("All Users");
   g->setIsOwner(true);
   addGroup(g);
   addGroup(new CUserGroup("New Users"));
   

   // Load the group info from licq.conf
   char filename[MAX_FILENAME_LEN];
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "licq.conf");
   CIniFile licqConf(INI_FxERROR | INI_FxFATAL);
   licqConf.LoadFile(filename);
   
   unsigned short nGroups;
   licqConf.SetSection("groups");
   licqConf.ReadNum("NumOfGroups", nGroups);
   gLog.Info("   %d groups stated...", nGroups);

   char sGroupKey[16], sGroupName[32];
   for (unsigned short i = 1; i <= nGroups; i++) 
   {
      sprintf(sGroupKey, "Group%d.name", i);
      licqConf.ReadStr(sGroupKey, sGroupName);
      addGroup(new CUserGroup(sGroupName));
   }
   gLog.Info("all groups found.\n");
   licqConf.CloseFile();
   
   // Load users from users.conf
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "users.conf");
   CIniFile usersConf(INI_FxFATAL | INI_FxERROR);
   usersConf.LoadFile(filename);
   
   unsigned short nUsers;
   usersConf.SetSection("users");
   usersConf.ReadNum("NumOfUsers", nUsers);
   gLog.Info("   %d users stated...", nUsers);
   
   char sUserKey[16];
   unsigned long nUserUin;
   usersConf.SetFlags(INI_FxWARN);
   for(unsigned short i = 1; i <= nUsers; i++)
   {
      sprintf(sUserKey, "User%d", i);
      if (!usersConf.ReadNum(sUserKey, nUserUin))
      {   
         gLog.Warn("%sSkipping user %i, UIN not found.\n", L_WARNxSTR, i);
         continue;
      }
      sprintf(filename, "%s%s%li%s", BASE_DIR, CONF_DIR, nUserUin, ".uin");
      addUser(new ICQUser(nUserUin, filename));
   } 
   gLog.Info("%d users found.\n", getGroup(0)->getNumUsers());

}


//-----CUsers::addUser----------------------------------------------------------
ICQUser *CUsers::addUser(ICQUser *_pcUser)
{
   // go through all the groups, add the user to those they belong to
   // will always add them to group 0
   for (unsigned short i = 0; i < getNumGroups(); i++)
   {
      if (_pcUser->getIsInGroup(i)) getGroup(i)->addUser(_pcUser);
   }
   return _pcUser;
}


//-----CUsers::removeUser-------------------------------------------------------
void CUsers::removeUser(ICQUser *_pcUser)
{
   // Remove the user from all the groups they belong to
   for (unsigned short i = 0; i < getNumGroups(); i++) 
   {
      if (_pcUser->getIsInGroup(i)) 
         getGroup(i)->removeUser(_pcUser);
   }
   
   _pcUser->removeFiles();
   delete _pcUser;
}


//-----CUsers::addGroup---------------------------------------------------------
void CUsers::addGroup(CUserGroup *_pcGroup)
{
   m_vpcGroups.push_back(_pcGroup);
}


//-----CUsers::getGroup---------------------------------------------------------
CUserGroup *CUsers::getGroup(unsigned short i)
{
   return (i < getNumGroups() ? m_vpcGroups[i] : m_vpcGroups[0]);
}


//-----CUsers::getUserByUin-----------------------------------------------------
ICQUser *CUsers::getUserByUin(unsigned long _nUin)
{
   CUserGroup *g = getGroup(0);  // group 0 is all users
   for(unsigned short i = 0; i < g->getNumUsers(); i++) 
      if(g->getUser(i)->getUin() == _nUin) return(g->getUser(i));
   return NULL;
}


//-----CUsers::reorder----------------------------------------------------------
void CUsers::reorder(ICQUser *_pcUser)
// Call reorder for every group the user is in
{
   if (_pcUser == NULL) return;
   for (unsigned short i = 0; i < getNumGroups(); i++) 
      if (_pcUser->getIsInGroup(i))
         getGroup(i)->reorder(_pcUser);
}


//-----CUsers::saveAllUsers-----------------------------------------------------
void CUsers::saveAllUsers(void)
{
   for (unsigned short i = 0; i < getNumUsers(); i++) 
   {
      getUser(i)->saveInfo();
      getUser(i)->saveBasicInfo();
      getUser(i)->saveExtInfo();
   }
}

//-----CUsers::addUserToGroup---------------------------------------------------
void CUsers::addUserToGroup(ICQUser *_pcUser, unsigned short _nGroup)
{
   if (_nGroup >= getNumGroups() || _pcUser == NULL) return;

   _pcUser->addToGroup(_nGroup);
   getGroup(_nGroup)->addUser(_pcUser);
}
         

//-----CUsers::removeUserFromGroup---------------------------------------------------
void CUsers::removeUserFromGroup(ICQUser *_pcUser, unsigned short _nGroup)
{
   if (_nGroup >= getNumGroups() || _pcUser == NULL || _nGroup == 0) return;

   _pcUser->removeFromGroup(_nGroup);
   getGroup(_nGroup)->removeUser(_pcUser);
}


//=====CICQUser=================================================================

unsigned short ICQUser::s_nNumUserEvents = 0;

//-----ICQUser::constructor-----------------------------------------------------
ICQUser::ICQUser(unsigned long id, char *filename)
// Called when first constructing our known users 
{
   setEnableSave(false);

   setDefaults(id);
   
   m_fConf.SetFlags(INI_FxWARN);
   m_fConf.SetFileName(filename);
   if (!loadDataFromFile())
     gLog.Error("%sUnable to load user info from '%s' (see log for details).\n%sUsing default values.\n", 
                L_ERRORxSTR, filename, L_BLANKxSTR);
   m_fConf.SetFlag(INI_FxALLOWxCREATE);
   m_fConf.CloseFile();

   setStatus(ICQ_STATUS_OFFLINE);
   
   setEnableSave(true);
}  


//-----ICQUser::loadDataFromFile------------------------------------------------
bool ICQUser::loadDataFromFile(void)
{
  if (!m_fConf.ReloadFile()) return (false);
   
  // read in the fields, checking for errors each time
  char sTemp[MAX_DATA_LEN];
  bool bTemp;
  unsigned short nTemp, nNewMessages;
  m_fConf.SetSection("user");
  m_fConf.ReadStr("Alias", sTemp, "Unknown");
  setAlias(sTemp);
  m_fConf.ReadStr("FirstName", sTemp, "");
  setFirstName(sTemp);
  m_fConf.ReadStr("LastName", sTemp, "");
  setLastName(sTemp);
  m_fConf.ReadStr("EMail", sTemp, "");
  setEmail(sTemp);
  m_fConf.ReadBool("Authorization", bTemp, false);
  setAuthorization(bTemp);
  m_fConf.ReadStr("History", sTemp, "none");
  setHistoryFile(sTemp);
  m_fConf.ReadBool("OnlineNotify", bTemp, false);
  setOnlineNotify(bTemp);
  m_fConf.ReadNum("NewMessages", nTemp, 0);
  nNewMessages = nTemp;
  m_fConf.ReadBool("NewUser", bTemp, false);
  setIsNew(bTemp);
  if (!m_fConf.ReadNum("Group", nTemp)) setGroup(getIsNew() ? 1 : 0);
  else setGroup(nTemp);
  m_fConf.ReadBool("VisibleList", bTemp, false);
  setVisibleList(bTemp);
  m_fConf.ReadBool("InvisibleList", bTemp, false);
  setInvisibleList(bTemp);
  m_fConf.ReadStr("City", sTemp, "Unknown");
  setCity(sTemp);
  m_fConf.ReadStr("State", sTemp, "Unknown");
  setState(sTemp);
  m_fConf.ReadNum("Country", nTemp, 0xFFFF);
  setCountry(nTemp);
  m_fConf.ReadStr("PhoneNumber", sTemp, "");
  setPhoneNumber(sTemp);
  m_fConf.ReadNum("Age", nTemp, 0);
  setAge(nTemp);
  m_fConf.ReadNum("Sex", nTemp, 0);
  setSex(nTemp);
  m_fConf.ReadStr("Homepage", sTemp, "Unknown"); 
  setHomepage(sTemp);
  m_fConf.ReadStr("About", sTemp, "");
  setAbout(sTemp);
  
  // if any of the fields contain a newline, erase it
  if (getLastName()[0] == '\n') setLastName("(blank)");
  if (getEmail()[0] == '\n') setEmail("(blank)");
  if (getFirstName()[0] == '\n') setFirstName("(blank)");
  if (getAlias()[0] == '\n') setAlias("(blank)");
  if (getHistoryFile()[0] == '\n') setHistoryFile("none");
  if (getCity()[0] == '\n') setCity("(blank)");
  if (getState()[0] == '\n') setState("(blank)");
  if (getPhoneNumber()[0] == '\n') setPhoneNumber("(blank)");
  if (getHomepage()[0] == '\n') setHomepage("(blank)");
  if (getAbout()[0] == '\n') setAbout("(blank)");

  if (nNewMessages > 0)
  {
     m_vcMessages.push_back(new CEventSaved(nNewMessages));
     incNumUserEvents();
  }
  return (true);
}


//-----ICQUser::constructor-----------------------------------------------------
ICQUser::ICQUser(unsigned long id)
// Called when a new user is received without them actually connecting 
// through tcp
{
   setEnableSave(false);

   setDefaults(id);
   setIsNew(true);
   addToGroup(1);

   setHistoryFile("default");
   setStatus(ICQ_STATUS_OFFLINE);
   
   char filename[MAX_FILENAME_LEN];
   sprintf(filename, "%s%s%ld.uin", BASE_DIR, CONF_DIR, getUin());
   m_fConf.SetFlags(INI_FxWARN | INI_FxALLOWxCREATE);
   m_fConf.SetFileName(filename);

   setEnableSave(true);
   saveInfo();
   saveBasicInfo();
   saveExtInfo();
}  


//-----ICQUser::constructor------------------------------------------------------------------------
ICQUser::ICQUser(unsigned long id, TCPSocket &sock)
// Called to create a new user with a given socket
{
   setEnableSave(false);

   setDefaults(id);
   setIsNew(true);
   addToGroup(1);

   tcpSocket.TransferConnectionFrom(sock);
   setHistoryFile("default");
   setStatus(ICQ_STATUS_ONLINE);
   
   char filename[MAX_FILENAME_LEN];
   sprintf(filename, "%s%s%ld.uin", BASE_DIR, CONF_DIR, getUin());
   m_fConf.SetFileName(filename);
   m_fConf.SetFlags(INI_FxWARN | INI_FxALLOWxCREATE);
   
   setEnableSave(true);
   saveInfo();
   saveBasicInfo();
   saveExtInfo();
}


//-----ICQUser::destructor------------------------------------------------------
ICQUser::~ICQUser(void)
{
   while (getNumMessages() > 0) ClearEvent(0);
   if (history != NULL) delete history;
}


//-----ICQUser::removeFiles-----------------------------------------------------
void ICQUser::removeFiles(void)
{
   remove(m_fConf.FileName());
   if (history != NULL) 
   {
      history->close();
      remove(history->name());
   }
}

//-----ICQUser::setDefaults-----------------------------------------------------
void ICQUser::setDefaults(unsigned long id)
{
   // set default data for the user
   setStrNull();
   setUin(id);
   char szAlias[12];
   sprintf(szAlias, "%ld", id);
   setAlias(szAlias);
   setFirstName("(Unknown ");
   setLastName("Name)");
   setEmail("unknown@somesite.com");
   setHistoryFile("default");
   setAwayMessage("");
   setCity("(blank)");
   setState("(blank)");
   setSex(0);
   setAge(0xFFFF);
   setCountry(0xFFFF);
   setHomepage("(blank)");
   setPhoneNumber("(blank)");
   setAbout("");
   setGroup(0); //getIsNew() ? 1 : 0);
   setOnlineNotify(false);
   setAuthorization(false);
   setInvisibleList(false);
   setVisibleList(false);
   setOnlineNotify(false);
   setSendServer(false);
   setShowAwayMsg(true);
   setIsNew(false);   
   setSequence(1);
   fcnDlg = NULL;
}


//-----ICQUser::setStrNull------------------------------------------------------
void ICQUser::setStrNull(void)
{
   m_sAlias = NULL;
   m_sFirstName = NULL;
   m_sLastName = NULL;
   m_sEmail = NULL;
   m_sHistoryFile = NULL;
   m_sAwayMessage = NULL;
   m_sCity = NULL;
   m_sState = NULL;
   m_sPhoneNumber = NULL;
   m_sHomepage = NULL;
   m_sAbout = NULL;
   history = NULL;
}


//-----ICQUser::openHistory-----------------------------------------------------
void ICQUser::openHistory(void)
// open the history file 
{
   char temp[MAX_FILENAME_LEN];
   // default history filename, compare only first 7 chars in case of spaces
   if (strncmp(getHistoryFile(), "default", 7) == 0)
   {
      sprintf(temp, "%s%s%ld.history", BASE_DIR, HISTORY_DIR, getUin());
      setHistoryFile(temp, false);
      history = new QFile(getHistoryFile());
      history->open(IO_Append | IO_WriteOnly);
      sprintf(temp, "default (%s%s%ld.history)", BASE_DIR, HISTORY_DIR, getUin());
      setHistoryFile(temp, false);
   }
   // no history file
   else if (strncmp(getHistoryFile(), "none", 4) == 0)  // no history file
   {
      history = NULL;
   }
   // use given name
   else
   {
      history = new QFile(getHistoryFile());
      history->open(IO_Append | IO_WriteOnly);
   }
      
}


char *ICQUser::getCountry(void)
{
   static char t[32];
   
   if (m_nCountryCode == COUNTRY_UNSPECIFIED)
   {
      strcpy(t, "Unspecified");
      return (t);
   }
   
   const char *szCountry = GetCountryByCode(m_nCountryCode);
   if (szCountry == NULL) 
      sprintf(t, "Unknown (%d)", m_nCountryCode);
   else
      strcpy(t, szCountry);
   return (t);
}


char *ICQUser::getSex(void)
{ 
   static char sSex[10];

   if (m_nSex == 1)
      strcpy(sSex, "Female");
   else if (m_nSex == 2) 
      strcpy(sSex, "Male");
   else
      strcpy(sSex, "Unknown");

   return(sSex); 
}


bool ICQUser::getStatusOffline(void)
{ 
   unsigned short nStatusShort = (unsigned short)m_nStatus;
   return (nStatusShort == ICQ_STATUS_OFFLINE);
}

unsigned short ICQUser::getStatus(void)
// guarantees to return a unique status that switch can be run on
{
   if (getStatusOffline()) return ICQ_STATUS_OFFLINE;
   else if (m_nStatus & ICQ_STATUS_DND) return ICQ_STATUS_DND;
   else if (m_nStatus & ICQ_STATUS_OCCUPIED) return ICQ_STATUS_OCCUPIED;
   else if (m_nStatus & ICQ_STATUS_NA) return ICQ_STATUS_NA;
   else if (m_nStatus & ICQ_STATUS_AWAY) return ICQ_STATUS_AWAY;
   else if (m_nStatus & ICQ_STATUS_FREEFORCHAT) return ICQ_STATUS_FREEFORCHAT;
   else if (m_nStatus << 24 == 0x00) return ICQ_STATUS_ONLINE;
   else return (ICQ_STATUS_OFFLINE - 1);
}

bool ICQUser::getStatusInvisible(void)  
{ 
   return (getStatusOffline() ? false : m_nStatus & ICQ_STATUS_FxPRIVATE);
}

bool ICQUser::getStatusWebPresence(void)
{ 
   return (getStatusOffline() ? false : m_nStatus & ICQ_STATUS_FxWEBxPRESENCE);
}

bool ICQUser::getStatusHideIp(void)
{ 
   return (getStatusOffline() ? false : m_nStatus & ICQ_STATUS_FxHIDExIP);
}

bool ICQUser::getStatusBirthday(void)
{ 
   return (getStatusOffline() ? false : m_nStatus & ICQ_STATUS_FxBIRTHDAY);
}



unsigned long ICQUser::getSequence(bool increment = false)      
{ 
   if (increment) 
      return (m_nSequence++); 
   else
      return (m_nSequence);
}


void ICQUser::setStatus(unsigned long s)    
{ 
   m_nStatus = s;
   if (fcnDlg != NULL) fcnDlg->user_updateStatus();
}

bool ICQUser::isAway(void)
{
   return (getStatus() == ICQ_STATUS_AWAY || getStatus() == ICQ_STATUS_NA ||
           getStatus() == ICQ_STATUS_DND || getStatus() == ICQ_STATUS_OCCUPIED);
}

void ICQUser::setHistoryFile(const char *s, bool _bOpen = true)   
{
   setStrField(m_sHistoryFile, s); 
   if (_bOpen) 
   {
      if (history != NULL) delete history;
      openHistory();
   }
   saveInfo();
}


void ICQUser::setStrField(char *&d, const char *s)
{ 
   if (d != NULL) delete[] d;
   if (s == NULL) 
   {
      d = new char[1];
      d[0] = '\0';
   }
   else
   {
      d = new char[strlen(s) + 1];
      strcpy(d, s);
   }
}

const char *ICQUser::getStatusStr(void)
{
  static char sz[32];
  switch(getStatus())
  {
  case ICQ_STATUS_OFFLINE:     strcpy(sz, "Offline");  break;
  case ICQ_STATUS_ONLINE:      strcpy(sz, "Online");   break;
  case ICQ_STATUS_AWAY:        strcpy(sz, "Away");     break;
  case ICQ_STATUS_NA:          strcpy(sz, "Not Available"); break;
  case ICQ_STATUS_OCCUPIED:    strcpy(sz, "Occupied"); break;
  case ICQ_STATUS_DND:         strcpy(sz, "Do Not Disturb"); break;
  case ICQ_STATUS_FREEFORCHAT: strcpy(sz, "Free for Chat"); break;
  default:                     sprintf(sz, "(0x%04X)", getStatus()); break;
  }
  if (getStatusInvisible()) 
  {
    memmove(sz + 1, sz, strlen(sz) + 1);
    sz[0] = '(';
    strcat(sz, ")");
  }
  return (sz);
}


//-----ICQUser::getBasicInfo----------------------------------------------------
void ICQUser::getBasicInfo(struct UserBasicInfo &us)
{
   strcpy(us.alias, getAlias());
   sprintf(us.uin, "%ld", getUin());
   strcpy(us.firstname, getFirstName());
   strcpy(us.lastname, getLastName());
   sprintf(us.name, "%s %s", us.firstname, us.lastname);
   strcpy(us.email, getEmail());
   strcpy(us.status, getStatusStr());   
   strcpy(us.ip, tcpSocket.RemoteIpStr());
   if (getStatusHideIp()) 
   {
      char sStatus[32];
      strcpy(sStatus, us.ip);
      sprintf(us.ip, "(%s)", sStatus);
   }
   sprintf(us.port, "%d", tcpSocket.RemotePort());
   sprintf(us.ip_port, "%s:%s", us.ip, us.port);
   strcpy(us.history, getHistoryFile());
   strncpy(us.awayMessage, getAwayMessage(), MAX_MESSAGE_SIZE);
}


//-----ICQUser::getExtInfo---------------------------------------------------
void ICQUser::getExtInfo(struct UserExtInfo &ud)
{
   strcpy(ud.city, getCity());
   strcpy(ud.state, getState());
   strcpy(ud.country, getCountry());
   strcpy(ud.phone, getPhoneNumber());
   if (getAge() == 0 || getAge() == 0xFFFF) 
      strcpy(ud.age, "N/A");
   else
      sprintf(ud.age, "%d", getAge());
   strcpy(ud.sex, getSex());
   strcpy(ud.homepage, getHomepage());
   strcpy(ud.about, getAbout());
}


//-----ICQUser::saveBasicInfo---------------------------------------------------
void ICQUser::saveBasicInfo(void)
{
   if (!getEnableSave()) return;

   if (!m_fConf.ReloadFile())
   {
      gLog.Error("%sError opening '%s' for reading.\n%sSee log for details.\n", 
                 L_ERRORxSTR, m_fConf.FileName(),  L_BLANKxSTR);
      return; 
   }
   m_fConf.SetSection("user");
   m_fConf.WriteStr("Alias", getAlias());
   m_fConf.WriteStr("FirstName", getFirstName());
   m_fConf.WriteStr("LastName", getLastName());
   m_fConf.WriteStr("EMail", getEmail());
   m_fConf.WriteBool("Authorization", getAuthorization());
   m_fConf.WriteStr("History", getHistoryFile());
   if (!m_fConf.FlushFile())
   {
     gLog.Error("%sError opening '%s' for writing.\n%sSee log for details.\n", 
                L_ERRORxSTR, m_fConf.FileName(), L_BLANKxSTR);
     return; 
   }

   m_fConf.CloseFile();
}


//-----ICQUser::saveInfo--------------------------------------------------------
void ICQUser::saveInfo(void)
{
   if (!getEnableSave()) return;

   if (!m_fConf.ReloadFile()) 
   {
      gLog.Error("%sError opening '%s' for reading.\n%sSee log for details.\n", 
                 L_ERRORxSTR, m_fConf.FileName(), L_BLANKxSTR);
      return; 
   }
   m_fConf.SetSection("user");
   m_fConf.WriteBool("OnlineNotify", getOnlineNotify());
   m_fConf.WriteBool("InvisibleList", getInvisibleList());
   m_fConf.WriteBool("VisibleList", getVisibleList());
   m_fConf.WriteNum("Group", getGroup());
   m_fConf.WriteBool("NewUser", getIsNew());
   m_fConf.WriteNum("NewMessages", getNumMessages());
   m_fConf.FlushFile();
   if (!m_fConf.FlushFile())
   {
     gLog.Error("%sError opening '%s' for writing.\n%sSee log for details.\n", 
                L_ERRORxSTR, m_fConf.FileName(), L_BLANKxSTR);
     return; 
   }

   m_fConf.CloseFile();
}


//-----ICQUser::saveExtInfo--------------------------------------------------
void ICQUser::saveExtInfo(void)
{
   if (!getEnableSave()) return;
   
   if (!m_fConf.ReloadFile()) 
   {
      gLog.Error("%sError opening '%s' for writing.\n%sSee log for details.\n", 
                 L_ERRORxSTR, m_fConf.FileName(), L_BLANKxSTR);
      return; 
   }
   m_fConf.SetSection("user");
   m_fConf.WriteStr("Homepage", getHomepage());
   m_fConf.WriteStr("City", getCity());
   m_fConf.WriteStr("State", getState());
   m_fConf.WriteNum("Country", getCountryCode());
   m_fConf.WriteStr("PhoneNumber", getPhoneNumber());
   m_fConf.WriteNum("Age", getAge());
   m_fConf.WriteNum("Sex", getSexNum());
   m_fConf.WriteStr("About", getAbout());
   if (!m_fConf.FlushFile())
   {
     gLog.Error("%sError opening '%s' for writing.\n%sSee log for details.\n", 
                L_ERRORxSTR, m_fConf.FileName(), L_BLANKxSTR);
     return; 
   }

   m_fConf.CloseFile();

}




//-----ICQUser::AddEvent--------------------------------------------------------
void ICQUser::AddEvent(CUserEvent *e)
{ 
   m_vcMessages.push_back(e);    
   incNumUserEvents();
   saveInfo();
   if (fcnDlg != NULL) fcnDlg->user_addEvent();
}


void ICQUser::WriteToHistory(const char *_szText)
// add text to history file
{
   if (history != NULL) 
   {
      history->writeBlock(_szText, strlen(_szText));
      history->flush();
   }
}



//-----ICQUser::GetEvent--------------------------------------------------------
CUserEvent *ICQUser::GetEvent(unsigned short index)
{ 
   if (index >= getNumMessages() || getNumMessages() == 0) return (NULL);
   return (m_vcMessages[index]); 
}


//-----ICQUser::ClearEvent------------------------------------------------------
void ICQUser::ClearEvent(unsigned short index)    
{ 
   delete m_vcMessages[index];
   for (unsigned short i = index; i < m_vcMessages.size() - 1; i++) 
      m_vcMessages[i] = m_vcMessages[i + 1];
   m_vcMessages.pop_back();
   decNumUserEvents();
   saveInfo();
}


bool ICQUser::getIsInGroup(unsigned short _nGroup)
{
   if (_nGroup == 0) return true;
   return (getGroup() & (unsigned long)pow(2, _nGroup - 1));
}

void ICQUser::addToGroup(unsigned short _nGroup)
{
   if (_nGroup == 0) return;
   setGroup(getGroup() | (unsigned short)pow(2, _nGroup - 1));
}


void ICQUser::removeFromGroup(unsigned short _nGroup)
{
   if (_nGroup == 0) return;
   setGroup(getGroup() & (0xFFFF - (unsigned short)pow(2, _nGroup - 1)));
}


//=====ICQOwner=================================================================

//-----ICQOwner::constructor----------------------------------------------------
ICQOwner::ICQOwner(void)
{
   gLog.Info("-> Owner configuration.\n");
   unsigned long nTemp;
   char sTemp[16];
   char filename[MAX_FILENAME_LEN];

   setEnableSave(false);
   setDefaults(0);
   setStatus(ICQ_STATUS_OFFLINE);
   
   // Get data from the config file
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "owner.uin");
   m_fConf.SetFlags(INI_FxERROR | INI_FxFATAL);
   m_fConf.SetFileName(filename);
   loadDataFromFile();
   //m_fConf.SetSection("user");
   m_fConf.ReadNum("Uin", nTemp);
   setUin(nTemp);
   m_fConf.ReadStr("Password", sTemp);
   setPassword(sTemp);
   m_fConf.CloseFile();
   m_fConf.ClearFlag(INI_FxFATAL);

   gLog.Info("   %s (%d) password scanned.\n", getAlias(), getUin());
   
   if (strlen(getPassword()) > 8) 
   {
      gLog.Error("%sPassword must be 8 characters or less.  Check %s.\n", L_ERRORxSTR, m_fConf.FileName());
      exit(1);
   }
   
   sprintf(filename, "%s%s%s", BASE_DIR, HISTORY_DIR, "owner.history");
   setHistoryFile(filename);

   setEnableSave(true);
}


//-----ICQOwner::getBasicInfo---------------------------------------------------
void ICQOwner::getBasicInfo(struct UserBasicInfo &us)
{
   ICQUser::getBasicInfo(us);
   strcpy(us.ip, tcpSocket.LocalIpStr());
   if (getStatusHideIp())
   {
      char sStatus[32];
      strcpy(sStatus, us.ip);
      sprintf(us.ip, "(%s)", sStatus);
   }
   sprintf(us.port, "%d", tcpSocket.LocalPort());
   sprintf(us.ip_port, "%s:%s", us.ip, us.port);
}


//-----ICQOwner::getExtInfo----------------------------------------------------------------------
void ICQOwner::getExtInfo(struct UserExtInfo &ud)
{
   ICQUser::getExtInfo(ud);
}

void ICQOwner::setStrNull(void)
{
   ICQUser::setStrNull();
   m_sPassword = NULL;
}


char *ICQOwner::          getPassword(void) { return(m_sPassword); }
unsigned short ICQOwner:: getSearchSequence(void) { return (m_nSearchSequence++); }
void ICQOwner::setPassword(const char *s) { setStrField(m_sPassword, s); }

