#ifndef socket_h
#define socket_h

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <qsocknot.h>

#include "buffer.h"
#include "icq-defines.h"
#include "support.h"


//=====Socket===================================================================
class Socket
{
public:
  Socket(void)  { sn = NULL; m_nDescriptor = -1; m_nError = 0; }
  QSocketNotifier *sn;
  bool Connected(void)            { return(m_nDescriptor > 0);  }
  int Descriptor(void)            { return(m_nDescriptor);      }
  int Error(void)                 { return(m_nError);           }
  const char *ErrorStr(void)      { return(strerror(m_nError)); }
protected:
  int m_nDescriptor, m_nError;
  void OpenSocket(void);
  void CloseSocket(void);
};


//=====StdinSocket==============================================================
class StdinSocket : public Socket
{
public:
  StdinSocket(void);
};


//=====INetSocket==================================================================================
class INetSocket : public Socket
{
public:
  INetSocket(void);
  virtual ~INetSocket(void) { CloseConnection(); };

  unsigned long  LocalIp(void)    { return (m_sLocalAddr.sin_addr.s_addr);  }
  const char *LocalIpStr(void)    { return (inet_ntoa(*(struct in_addr *)&m_sLocalAddr.sin_addr.s_addr)); }
  unsigned long  RemoteIp(void)   { return (m_sRemoteAddr.sin_addr.s_addr); };
  const char *RemoteIpStr(void)   { return (inet_ntoa(*(struct in_addr *)&m_sRemoteAddr.sin_addr.s_addr)); }
  unsigned short LocalPort(void)  { return (ntohs(m_sLocalAddr.sin_port));  };
  unsigned short RemotePort(void) { return (ntohs(m_sRemoteAddr.sin_port)); };
  void DumpPacket(CBuffer &b, direction d);
  const char *GetIDStr(void)  { return (m_szID); }

  bool SetAddrsFromSocket(unsigned short _nFlags);
  bool SetRemoteAddr(unsigned long _nRemoteIp, unsigned short _nRemotePort);
  bool SetRemoteAddr(char *_szRemoteName, unsigned short _nRemotePort);
  bool DestinationSet(void)  { return (RemoteIp() != 0); };
  
  void ResetSocket(void);
  void CloseConnection(void);
  void ClearRecvBuffer(void)  { m_xRecvBuffer.Clear(); };
  bool RecvBufferFull(void)   { return m_xRecvBuffer.Full(); };
  CBuffer &RecvBuffer(void)   { return m_xRecvBuffer; };
  
  unsigned long GetIpByName(char *_szHostName);
  bool OpenConnection(void);
  bool StartServer(unsigned int _nPort);
  bool SendRaw(CBuffer &b);
  bool RecvRaw(void);
  
  virtual bool Send(CBuffer &b) = 0;
  virtual bool Recv(void) = 0;

protected:
  struct sockaddr_in m_sRemoteAddr, m_sLocalAddr;
  CBuffer m_xRecvBuffer;
  char m_szID[4];
  int m_nSockType;
};


//=====TCPSocket===============================================================
class TCPSocket : public INetSocket
{
public:      
  TCPSocket(void) { strcpy(m_szID, "TCP"); m_nSockType = SOCK_STREAM; }
  
  // Abstract base class overloads
  virtual bool Send(CBuffer &b)
    { return SendPacket(b); }
  virtual bool Recv(void)
    { return RecvPacket(); }

  // Functions specific to TCP
  bool SendPacket(CBuffer &b);
  bool RecvPacket(void);
  void RecvConnection(TCPSocket &newSocket);
  void TransferConnectionFrom(TCPSocket &from);
};


//=====UDPSocket===================================================================================
class UDPSocket : public INetSocket
{
public:      
  UDPSocket(void) { strcpy(m_szID, "UDP"); m_nSockType = SOCK_DGRAM; }
  
  // Abstract base class overloads
  virtual bool Send(CBuffer &b)
    { return SendRaw(b); }
  virtual bool Recv(void)
    { return RecvRaw(); }
};           


#endif
