#ifndef LOG_H
#define LOG_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdarg.h>
#include <vector.h>
#include <stdio.h>

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#include <qwidget.h>

// Info: basic information about what's going on
const unsigned short L_INFO     = 0x0001;
// Unknown: unknown packets or bytes
const unsigned short L_UNKNOWN  = 0x0002;
// Error: critical errors which should be brought to the attention of the user
const unsigned short L_ERROR    = 0x0004;
// Warn: warnings which are not critical but could be important
const unsigned short L_WARN     = 0x0008;
// Packet: packet dumps
const unsigned short L_PACKET   = 0x0010;

const unsigned short L_ALL      = L_INFO | L_UNKNOWN | L_ERROR | L_WARN | L_PACKET;
const unsigned short L_NONE     = 0;

const char L_WARNxSTR[]    = "[WRN] ";
const char L_UDPxSTR[]     = "[UDP] ";
const char L_TCPxSTR[]     = "[TCP] ";
const char L_ERRORxSTR[]   = "[ERR] ";
const char L_UNKNOWNxSTR[] = "[-?-] ";
const char L_BLANKxSTR[]   = "      ";
const char L_PACKETxSTR[]  = "[PKT] ";

const unsigned short S_STDOUT   = 1;
const unsigned short S_FILE     = 2;
const unsigned short S_WINDOW   = 3;
const unsigned short S_MSGBOX   = 4;

const unsigned short MAX_MSG_SIZE = 2048;

//-----CLogService---------------------------------------------------------------
class CLogService
{
public:
  CLogService(unsigned short _nLogTypes);
  virtual void lprintf(unsigned short _nLogType, const char *_szFormat, va_list argp) = 0;
  void SetLogTypes(unsigned short _nLogTypes);
  unsigned short ServiceType(void);
  unsigned short LogType(unsigned short _nLogType);
  void AddLogType(unsigned short _nLogType);
  void RemoveLogType(unsigned short _nLogType);
  void SetData(void *);

protected:
  unsigned short m_nLogTypes;
  unsigned short m_nServiceType;
  void *m_pData;
};


//-----StdOut-------------------------------------------------------------------
class CLogService_StdOut : public CLogService
{
public:
   CLogService_StdOut(unsigned short _nLogTypes);
   virtual void lprintf(unsigned short _nLogType, const char *_szFormat, va_list argp);
};


//-----File---------------------------------------------------------------------
class CLogService_File : public CLogService
{
public:
   CLogService_File(unsigned short _nLogTypes);
   bool SetLogFile(const char *_szFile, const char *_szFlags);
   virtual void lprintf(unsigned short _nLogType, const char *_szFormat, va_list argp);
protected:
   FILE *m_fLog;
};


//-----Window-------------------------------------------------------------------
class COutputWindow
{
public:
   virtual void wprintf(const char *_szFormat, va_list argp) = 0;
};

class CLogService_Window : public CLogService
{
public:
   CLogService_Window(unsigned short _nLogTypes);
   bool SetLogWindow(COutputWindow *_xWindow);
   virtual void lprintf(unsigned short _nLogType, const char *_szFormat, va_list argp);
protected:
   COutputWindow *m_xLogWindow;
};


//-----MsgBox-------------------------------------------------------------------
class CLogService_MsgBox : public CLogService
{
public:
   CLogService_MsgBox(unsigned short _nLogTypes);
   void SetParent(QWidget *_QtParent);
   virtual void lprintf(unsigned short _nLogType, const char *_szFormat, va_list argp);
protected:
   QWidget *QtParent;
};


//-----CLogServer---------------------------------------------------------------
class CLogServer
{
public:
  CLogServer(void);
  void AddService(CLogService *_xService);
  void AddLogTypeToService(unsigned short _nServiceType, unsigned short _nLogType);
  void RemoveLogTypeFromService(unsigned short _nServiceType, unsigned short _nLogType);
  void ModifyService(unsigned short _nServiceType, unsigned short _nLogTypes);
  void SetServiceData(unsigned short _nServiceType, void *_pData);
  const char *ErrorStr(void);

  void Info(const char *_szFormat, ...);
  void Unknown(const char *_szFormat, ...);
  void Error(const char *_szFormat, ...);
  void Warn(const char *_szFormat, ...);
  void Packet(const char *_szFormat, ...);
  
protected:
  vector <CLogService *> m_vxLogServices;
  void Log(const unsigned short _nLogType, const char *_szFormat, va_list argp);
};


// Define an external log server to be started in log.cpp
extern CLogServer gLog;


#endif
