#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <qmessagebox.h>
#include "log.h"

extern int errno;
CLogServer gLog;

//-----CLogService---------------------------------------------------------------
CLogService::CLogService(unsigned short _nLogTypes) 
{ 
   SetLogTypes(_nLogTypes); 
   SetData(NULL);
}
  
inline
void CLogService::SetLogTypes(unsigned short _nLogTypes)
{ 
   m_nLogTypes = _nLogTypes; 
}

inline
void CLogService::SetData(void *_pData)
{
  m_pData = _pData;
}

inline
unsigned short CLogService::ServiceType(void)
{ 
   return m_nServiceType; 
}

inline
unsigned short CLogService::LogType(unsigned short _nLogType)
{ 
   return m_nLogTypes & _nLogType; 
}

inline
void CLogService::AddLogType(unsigned short _nLogType)
{ 
   m_nLogTypes |= _nLogType; 
}

inline
void CLogService::RemoveLogType(unsigned short _nLogType)
{ 
   m_nLogTypes &= ~_nLogType; 
}



//-----StdOut-------------------------------------------------------------------
CLogService_StdOut::CLogService_StdOut(unsigned short _nLogTypes) 
   : CLogService(_nLogTypes)
{
   m_nServiceType = S_STDOUT;
}

inline
void CLogService_StdOut::lprintf(unsigned short _nLogType, const char *_szFormat, va_list argp)
{
   vprintf(_szFormat, argp);
   fflush(stdout);
}



//-----File---------------------------------------------------------------------
CLogService_File::CLogService_File(unsigned short _nLogTypes) 
   : CLogService(_nLogTypes)
{
   m_fLog = NULL;
   m_nServiceType = S_FILE;
}

bool CLogService_File::SetLogFile(const char *_szFile, const char *_szFlags)
{ 
   if (m_fLog != NULL) fclose (m_fLog);
   m_fLog = fopen(_szFile, _szFlags);
   return (m_fLog != NULL);
}

inline
void CLogService_File::lprintf(unsigned short _nLogType, const char *_szFormat, va_list argp)
{
   if (m_fLog == NULL) return;
   vfprintf(m_fLog, _szFormat, argp);
   fflush(m_fLog);
}


//-----Window-------------------------------------------------------------------
CLogService_Window::CLogService_Window(unsigned short _nLogTypes) 
   : CLogService(_nLogTypes)
{
   m_xLogWindow = NULL;
   m_nServiceType = S_WINDOW;
}

bool CLogService_Window::SetLogWindow(COutputWindow *_xWindow)
{ 
   m_xLogWindow = _xWindow;
   return true;
}

inline
void CLogService_Window::lprintf(unsigned short _nLogType, const char *_szFormat, va_list argp)
{
   if (m_xLogWindow == NULL) return;
   m_xLogWindow->wprintf(_szFormat, argp);
}


//-----MsgBox-------------------------------------------------------------------
CLogService_MsgBox::CLogService_MsgBox(unsigned short _nLogTypes) 
   : CLogService(_nLogTypes)
{
   QtParent = NULL;
   m_nServiceType = S_MSGBOX;
}

void CLogService_MsgBox::SetParent(QWidget *_QtParent)
{
   QtParent = _QtParent;
}

class CLicqMessageBox : public QMessageBox
{
public:
  CLicqMessageBox(const char *_szText, Icon icon) 
    : QMessageBox ("Licq", _szText, icon, Ok, 0, 0, NULL, NULL, false) 
  {}

protected:
  virtual void closeEvent(QCloseEvent *e)
  {
    QMessageBox::closeEvent(e);
    delete this;
  }
};

inline
void CLogService_MsgBox::lprintf(unsigned short _nLogType, const char *_szFormat, va_list argp)
{
  char szMsg[MAX_MSG_SIZE];
  CLicqMessageBox *b;
  vsprintf(szMsg, _szFormat, argp);
  switch (_nLogType)
  {
  case L_INFO:
  case L_UNKNOWN:
  case L_PACKET:
    b = new CLicqMessageBox(szMsg, QMessageBox::Information);
    b->exec();
    break;

  case L_WARN:
    b = new CLicqMessageBox(szMsg, QMessageBox::Warning);
    b->exec();

  case L_ERROR:
    b =  new CLicqMessageBox(szMsg, QMessageBox::Critical);
    b->exec();
    break;
  }
}


//-----CLogServer---------------------------------------------------------------
CLogServer::CLogServer(void) 
{
}

void CLogServer::AddService(CLogService *_xService)
{
   m_vxLogServices.push_back(_xService);
}


void CLogServer::AddLogTypeToService(unsigned short _nServiceType, unsigned short _nLogType)
{
   // Go through the vector setting the log types
   vector<CLogService *>::iterator iter;
   for (iter = m_vxLogServices.begin(); iter != m_vxLogServices.end(); iter++) 
   {
      if ((*iter)->ServiceType() == _nServiceType) 
         (*iter)->AddLogType(_nLogType);
   }         
}

void CLogServer::RemoveLogTypeFromService(unsigned short _nServiceType, unsigned short _nLogType)
{
   // Go through the vector setting the log types
   vector<CLogService *>::iterator iter;
   for (iter = m_vxLogServices.begin(); iter != m_vxLogServices.end(); iter++) 
   {
      if ((*iter)->ServiceType() == _nServiceType) 
         (*iter)->RemoveLogType(_nLogType);
   }         
}

void CLogServer::ModifyService(unsigned short _nServiceType, unsigned short _nLogTypes)
{
   // Go through the vector setting the log types
   vector<CLogService *>::iterator iter;
   for (iter = m_vxLogServices.begin(); iter != m_vxLogServices.end(); iter++) 
   {
      if ((*iter)->ServiceType() == _nServiceType) 
         (*iter)->SetLogTypes(_nLogTypes);
   }
}

void CLogServer::SetServiceData(unsigned short _nServiceType, void *_pData)
{
  vector<CLogService *>::iterator iter;
  for (iter = m_vxLogServices.begin(); iter != m_vxLogServices.end(); iter++) 
  {
     if ((*iter)->ServiceType() == _nServiceType)
       (*iter)->SetData(_pData);
  }  
}

void CLogServer::Info(const char *_szFormat, ...)
{
   va_list argp;
   va_start(argp, _szFormat);
   Log(L_INFO, _szFormat, argp);
   va_end(argp);
}

void CLogServer::Unknown(const char *_szFormat, ...)
{
   va_list argp;
   va_start(argp, _szFormat);
   Log(L_UNKNOWN, _szFormat, argp);
   va_end(argp);
}

void CLogServer::Error(const char *_szFormat, ...)
{
   va_list argp;
   va_start(argp, _szFormat);
   Log(L_ERROR, _szFormat, argp);
   va_end(argp);
}

void CLogServer::Warn(const char *_szFormat, ...)
{
   va_list argp;
   va_start(argp, _szFormat);
   Log(L_WARN, _szFormat, argp);
   va_end(argp);
}


void CLogServer::Packet(const char *_szFormat, ...)
{
   va_list argp;
   va_start(argp, _szFormat);
   Log(L_PACKET, _szFormat, argp);
   va_end(argp);
}


const char *CLogServer::ErrorStr(void)
{ 
   return strerror(errno); 
}
   

void CLogServer::Log(const unsigned short _nLogType, const char *_szFormat, va_list argp)
{
   vector<CLogService *>::iterator iter;
   for (iter = m_vxLogServices.begin(); iter != m_vxLogServices.end(); iter++) 
   {
      if ((*iter)->LogType(_nLogType))
        (*iter)->lprintf(_nLogType, _szFormat, argp);
   }
}

