#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <qapp.h>

#include "licqgui.h"
#include "icq-defines.h"
#include "gui-defines.h"
#include "chatdlg.h"
#include "filedlg.h"
#include "log.h"
#include "translate.h"
//#include "plugin.h"

#ifdef USE_DOCK
#include "xpm/iconOnline.xpm"
#include "xpm/iconOffline.xpm"
#include "xpm/iconAway.xpm"
#include "xpm/iconFFC.xpm"
#include "xpm/iconNA.xpm"
#include "xpm/iconDND.xpm"
#include "xpm/iconInvisible.xpm"
#include "xpm/iconOccupied.xpm"
#include "xpm/iconMsg.xpm"
#include "xpm/iconNoMsg.xpm"
#include "xpm/iconSysMsg.xpm"
#include "xpm/iconBack.xpm"
#else
#include "xpm/icon.xpm"
#include "xpm/icon-newmsg.xpm"
#endif // USE_DOCK

#include <X11/Xlib.h>  // stuff for 
#include <X11/Xutil.h> // appicon generation

#ifdef USE_SCRNSAVER
#include <X11/extensions/scrnsaver.h>
#endif // USE_SCRNSAVER


static QPixmap *ScaleWithBorder(const QPixmap &pm, int w, int h, struct Border border)
{
   QPainter p;
   
   // top left corner
   QPixmap pmTL(border.left, border.top);
   p.begin(&pmTL);
   p.drawPixmap(0, 0, pm, 0, 0, pmTL.width(), pmTL.height());
   p.end();

   // top border
   QPixmap pmT(pm.width() - border.left - border.right, border.top);
   p.begin(&pmT);
   p.drawPixmap(0, 0, pm, border.left, 0, pmT.width(), pmT.height());
   p.end();
   QImage imT( (pmT.convertToImage()).smoothScale(w - border.left - border.right, pmT.height()) );
   
   // top right corner
   QPixmap pmTR(border.right, border.top);
   p.begin(&pmTR);
   p.drawPixmap(0, 0, pm, pm.width() - border.right, 0, pmTR.width(), pmTR.height());
   p.end();

   // left border
   QPixmap pmL(border.left, pm.height() - border.top - border.bottom);
   p.begin(&pmL);
   p.drawPixmap(0, 0, pm, 0, border.top, pmL.width(), pmL.height());
   p.end();
   QImage imL( (pmL.convertToImage()).smoothScale(pmL.width(), h - border.top - border.bottom) );

   // center
   QPixmap pmC(pmT.width(), pmL.height());
   p.begin(&pmC);
   p.drawPixmap(0, 0, pm, border.left, border.top, pmC.width(), pmC.height());
   p.end();
   QImage imC( (pmC.convertToImage()).smoothScale(imT.width(), imL.height()) );

   // right border
   QPixmap pmR(border.right, pm.height() - border.top - border.bottom);
   p.begin(&pmR);
   p.drawPixmap(0, 0, pm, pm.width() - border.right, border.top, pmR.width(), pmR.height());
   p.end();
   QImage imR ( (pmR.convertToImage()).smoothScale(pmR.width(), h - border.top - border.bottom) );

   // bottom left border
   QPixmap pmBL(border.left, border.bottom);
   p.begin(&pmBL);
   p.drawPixmap(0, 0, pm, 0, pm.height() - border.bottom, pmBL.width(), pmBL.height());
   p.end();
   
   // bottom border
   QPixmap pmB(pm.width() - border.left - border.right, border.bottom);
   p.begin(&pmB);
   p.drawPixmap(0, 0, pm, border.left, pm.height() - border.bottom, pmB.width(), pmB.height());
   p.end();
   QImage imB( (pmB.convertToImage()).smoothScale(w - border.left - border.right, pmB.height()) );

   // bottom right border
   QPixmap pmBR(border.right, border.bottom);
   p.begin(&pmBR);
   p.drawPixmap(0, 0, pm, pm.width() - border.right, pm.height() - border.bottom, pmBR.width(), pmBR.height());
   p.end();

   // put the image together
   QPixmap *pmFinal = new QPixmap(w, h, pm.depth());
   p.begin(pmFinal);
   p.drawPixmap(0, 0, pmTL, 0, 0, -1, -1);
   p.drawImage(border.left, 0, imT, 0, 0, -1, -1);
   p.drawPixmap(pmFinal->width() - border.right, 0, pmTR, 0, 0, -1, -1);
   p.drawImage(0, border.top, imL, 0, 0, -1, -1);
   p.drawImage(pmFinal->width() - border.right, border.top, imR, 0, 0, -1, -1);
   p.drawPixmap(0, pmFinal->height() - border.bottom, pmBL, 0, 0, -1, -1);
   p.drawImage(border.left, pmFinal->height() - border.bottom, imB, 0, 0, -1, -1);
   p.drawPixmap(pmFinal->width() - border.right, pmFinal->height() - border.bottom, pmBR, 0, 0, -1, -1);
   p.drawImage(border.left, border.top, imC, 0, 0, -1, -1);
   p.end();

   return (pmFinal);
}


static QPixmap *MakeTransparentBg(QPixmap *_pm, const QRect &r)
{  
  QPixmap *pm = new QPixmap(r.size());
  QPainter p(pm);
  p.drawPixmap(0, 0, *_pm, r.x(), r.y(), r.width(), r.height());
  return(pm);
}



//-----CMainWindow::constructor-------------------------------------------------
CMainWindow::CMainWindow(ICQ *theServer, char *skinName, char *iconsName,
                         QWidget *parent, const char *name) : QWidget(parent, name)
{
   // set up appicon and docking, code supplied by Mark Deneed
   WId win = topLevelWidget()->winId();     // get the window
   Display *dsp = x11Display();  // get the display
   XWMHints *hints;  // hints
   XClassHint classhint;  // class hints
   classhint.res_name = "licq";  // res_name
   classhint.res_class = "Licq";  // res_class   
   XSetClassHint(dsp, win, &classhint); // set the class hints
   hints = XGetWMHints(dsp, win);  // init hints
   hints->window_group = win;  // set set the window hint
   hints->flags = WindowGroupHint;  // set the window group hint
   XSetWMHints(dsp, win, hints);  // set the window hints for WM to use.
   XFree( hints ); 

   // set the icon
   server = theServer;
#ifdef USE_KDE
   topLevelWidget()->setIconText("KLicq");
#else
   topLevelWidget()->setIconText("Licq");
#endif
#ifdef USE_DOCK
   setDockIcon(QPixmap(iconBack_xpm));
   setDockIconMsg(false, false);
   setDockIconStatus();
#else
   setIcon(QPixmap(xpmIcon));
#endif  // USE_DOCK
   
   // Create the child widgets and log services
   netWin = new OutputWin;
   CLogService_Window *w = new CLogService_Window(L_INFO | L_WARN);
   w->SetLogWindow(netWin);
   gLog.AddService(w);
   CLogService_MsgBox *m = new CLogService_MsgBox(L_ERROR);
   m->SetParent(netWin);
   gLog.AddService(m);

   // read in info from file
   char filename[MAX_FILENAME_LEN];
   
   gLog.Info("-> GUI configuration.\n");
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "licq.conf");
   CIniFile licqConf(INI_FxWARN | INI_FxFATAL);
   licqConf.LoadFile(filename);
   licqConf.ClearFlag(INI_FxFATAL);

   licqConf.SetSection("appearance");
   licqConf.ReadNum("FontSize", fontSize, 12);
   licqConf.ReadStr("FontFamily", fontFamily, "system");
   licqConf.ReadNum("FontCharSet", fontCharSetInt, 1);
   QFont::CharSet fontCharSet;
   switch (fontCharSetInt)
   {
   case 1: fontCharSet = QFont::Latin1; break;
   case 2: fontCharSet = QFont::Latin2; break;
   case 3: fontCharSet = QFont::Latin3; break;
   case 4: fontCharSet = QFont::Latin4; break;
   case 5: fontCharSet = QFont::Latin5; break;
   case 6: fontCharSet = QFont::Latin6; break;
   case 7: fontCharSet = QFont::Latin7; break;
   case 8: fontCharSet = QFont::Latin8; break;
   case 9: fontCharSet = QFont::Latin9; break;
   case 10: fontCharSet = QFont::KOI8R; break;
   default: fontCharSet = QFont::Latin1; break;
   }
   gLog.Info("   Font: %s / %d point.\n", fontFamily, fontSize);
   qApp->setFont(QFont(fontFamily, fontSize, QFont::Normal, false, fontCharSet), true);

   licqConf.ReadStr("ColorOnline", colorOnline, "blue");
   licqConf.ReadStr("ColorOffline", colorOffline, "red");
   licqConf.ReadStr("ColorAway", colorAway, "blue");
   licqConf.ReadStr("ColorNew", colorNew, "yellow");
   licqConf.ReadStr("ColorBackground", colorBackground, "grey76");
   licqConf.ReadStr("ColorGridLines", colorGridLines, "white");
   gLog.Info("   Colors: %s / %s / %s / %s.\n", colorOnline, colorOffline, colorAway, colorNew);

   licqConf.ReadBool("GridLines", gridLines, false);
   gLog.Info("   Grid lines are %s.\n", gridLines ? "enabled" : "disabled");
   licqConf.ReadBool("ShowHeader", showHeader, true);
   gLog.Info("   Column header is %s.\n", showHeader ? "enabled" : "disabled");

   licqConf.SetSection("startup");
   licqConf.ReadNum("Logon", m_nAutoLogon, 0);
   if (m_nAutoLogon > 16) m_nAutoLogon = 0;
   gLog.Info("   AutoLogon %s.\n", m_nAutoLogon == 0 ? "disabled" : "enabled");
   licqConf.ReadNum("AutoAway", autoAwayTime, 0);
   if (autoAwayTime == 0) 
      gLog.Info("   Auto-away disabled.\n");
   else
      gLog.Info("   Auto-away enabled after %d minutes.\n", autoAwayTime);
   licqConf.ReadNum("AutoNA", autoNATime, 0);
   if (autoNATime == 0) 
      gLog.Info("   Auto-not-available disabled.\n");
   else
      gLog.Info("   Auto-not-available enabled after %d minutes.\n", autoNATime);              

   licqConf.SetSection("functions");
   licqConf.ReadBool("AutoClose", autoClose, true);
   gLog.Info("   Autoclose %s.\n", autoClose ? "disabled" : "enabled");
   
   licqConf.SetSection("groups");
   unsigned short nCurrentGroup;
   licqConf.ReadNum("DefaultGroup", nCurrentGroup, 0);
   gLog.Info("   Default Group is (%d).\n",  nCurrentGroup);
   if (nCurrentGroup > 0) nCurrentGroup++;  // Group 1 is actually new users
   
   // load up position and size from file
   gLog.Info("-> Geometry configuration.\n");
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "geometry.conf");
   CIniFile geoConf(INI_FxWARN);
   if (geoConf.LoadFile(filename)) 
   {
     geoConf.SetSection("geometry"); 
     unsigned short xPos, yPos, hVal, wVal;
     geoConf.ReadNum("x", xPos, 100);
     geoConf.ReadNum("y", yPos, 100);
     geoConf.ReadNum("h", hVal, 400);
     geoConf.ReadNum("w", wVal, 150);
     gLog.Info("   Using %d x %d at (%d, %d).\n", wVal, hVal, xPos, yPos);
     setGeometry(xPos, yPos, wVal, hVal);
     geoConf.CloseFile();
   }
   else 
   {
      gLog.Info("   Using defaults: width 150, height 400 at (0, 0).\n");
      setGeometry(0, 0, 150, 400);
   }
   setMinimumSize(100, 200);
   
   // Load the icons
   licqConf.SetSection("appearance");
   licqConf.SetFlags(INI_FxERROR | INI_FxFATAL);
   if (strlen(iconsName) == 0) 
      licqConf.ReadStr("Icons", iconsName);
   loadIcons(iconsName);
   
   // Load the skin
   if (strlen(skinName) == 0) 
      licqConf.ReadStr("Skin", skinName);
   skin = new CSkin(skinName);
     
   awayMsgDlg = new AwayMsgDlg(&server->icqOwner);
   optionsDlg = NULL;
   m_bShowOffline = true;
   
   char title[128];
#ifdef USE_KDE
   sprintf(title, "KLicq (%s)", server->icqOwner.getAlias());
#else
   sprintf(title, "Licq (%s)", server->icqOwner.getAlias());
#endif
   setCaption(title);

   initMenu();

   // User List
   char colKey[16];
   unsigned short colTitle, colWidth, colAlign, numColumns;
   licqConf.ReadNum("NumColumns", numColumns);
   for (unsigned short i = 1; i <= numColumns; i++) 
   {
      sprintf(colKey, "Column%dInfo", i);
      licqConf.ReadNum(colKey, colTitle);
      sprintf(colKey, "Column%dWidth", i);
      licqConf.ReadNum(colKey, colWidth);
      sprintf(colKey, "Column%dAlign", i);
      licqConf.ReadNum(colKey, colAlign);
      colInfo.push_back(ColInfo(colTitle, colWidth, colAlign));
   }
   userView = new CUserView(mnuUser, mnuGroup, colInfo, showHeader, gridLines, this);
   userView->setPixmaps(pmOnline, pmOffline, pmAway, pmNa, pmOccupied, pmDnd, 
                        pmPrivate, pmMessage);
   userView->setColors(colorOnline, colorAway, colorOffline, colorNew, 
                       colorBackground, colorGridLines);
   
   // Group Combo Box
   cmbUserGroups = new CEComboBox(false, this);
   for (unsigned short i = 0; i < server->getUsers()->getNumGroups(); i++) 
      cmbUserGroups->insertItem(server->getUsers()->getGroup(i)->getName());
   cmbUserGroups->setCurrentItem(nCurrentGroup);
   connect(cmbUserGroups, SIGNAL(activated(int)), this, SLOT(setCurrentGroup(int)));

   // Widgets controlled completely by the skin
   btnSystem = NULL;
   lblMsg = NULL;
   lblStatus = NULL;

   applySkin();
   
   autoAwayTimer.start(10000);  // start the inactivity timer for auto away
   
   connect (&autoAwayTimer, SIGNAL(timeout()), this, SLOT(autoAway()));
   connect (server, SIGNAL(signal_updatedUsers()), this, SLOT(updateUserWin()));
   connect (server, SIGNAL(signal_updatedStatus()), this, SLOT(updateStatus()));
   connect (server, SIGNAL(signal_doneOwnerFcn(bool, unsigned short)), 
            this, SLOT(done(bool, unsigned short)));
   connect (server, SIGNAL(signal_eventResult(ICQUser *, CUserEvent *, bool, const char *, unsigned long)), 
            this, SLOT(eventResult(ICQUser *, CUserEvent *, bool, const char *, unsigned long)));
   connect (userView, SIGNAL(doubleClicked(QListViewItem *)), 
            this, SLOT(callDefaultICQFunction()));
   
   inMiniMode = false;
   updateStatus();
   setCurrentGroup(nCurrentGroup);
   manualAway = 0;
   
   show();
   userView->maxLastColumn();

   // automatically logon if requested in conf file
   if (m_nAutoLogon > 0) 
   {
      if (m_nAutoLogon >= 10) 
        mnuStatus->setItemChecked(MNUxITEM_STATUSxINVISIBLE, true);         
      if (m_nAutoLogon > 10 && m_nAutoLogon < 17)
        changeStatus(m_nAutoLogon - 11);
      else if (m_nAutoLogon > 0 && m_nAutoLogon < 7)
        changeStatus(m_nAutoLogon - 1);
   }
}


//-----applySkin----------------------------------------------------------------
void CMainWindow::applySkin(void)
{
  // Set the background pixmap and mask
  if (skin->frame.pixmap != NULL) 
  {      
     pmBorder = new QPixmap(skin->frame.pixmap);
     if (pmBorder->isNull()) 
     {
       gLog.Error("%sError loading background pixmap (%s).\n", L_ERRORxSTR, skin->frame.pixmap);
        delete skin->frame.pixmap;
        skin->frame.pixmap = NULL;
     }
  }
  if (skin->frame.mask != NULL) 
  {
     pmMask = new QPixmap(skin->frame.mask);
     if (pmMask->isNull()) 
     {
       gLog.Error("%sError loading background mask (%s).\n", L_ERRORxSTR, skin->frame.mask);
        delete skin->frame.mask;
        skin->frame.mask = NULL;
     }
  }   
  
  // Group Combo Box
  cmbUserGroups->setNamedBgColor(skin->cmbGroups.color.bg);
  cmbUserGroups->setNamedFgColor(skin->cmbGroups.color.fg);
  
  // System Button
  if (btnSystem != NULL) delete btnSystem;
  if (!skin->frame.hasMenuBar) 
  {
     if (skin->btnSys.pixmapUpNoFocus == NULL) 
     {
        btnSystem = new CEButton(skin->btnSys.caption == NULL ? "System" : skin->btnSys.caption, this);
     }
     else
     { 
         btnSystem = new CEButton(new QPixmap(skin->btnSys.pixmapUpFocus), 
                                  new QPixmap(skin->btnSys.pixmapUpNoFocus), 
                                  new QPixmap(skin->btnSys.pixmapDown),
                                  this);
     }
     connect(btnSystem, SIGNAL(clicked()), this, SLOT(popupSystemMenu()));
     btnSystem->setNamedFgColor(skin->btnSys.color.fg);
     btnSystem->setNamedBgColor(skin->btnSys.color.bg);
  }
  
  // Message Label
  if (lblMsg != NULL) delete lblMsg;
  lblMsg = new CELabel(skin->lblMsg.transparent, NULL, this);
  lblMsg->setFrameStyle(skin->lblMsg.frameStyle);
  lblMsg->setMargin(skin->lblMsg.margin);
  lblMsg->setNamedFgColor(skin->lblMsg.color.fg);
  lblMsg->setNamedBgColor(skin->lblMsg.color.bg);
  if (skin->lblMsg.transparent)
  {
    QPixmap *p = MakeTransparentBg(pmBorder, skin->borderToRect(&skin->lblMsg, pmBorder));
    lblMsg->setBackgroundPixmap(*p);
    delete p;
  }
  else if (skin->lblMsg.pixmap != NULL) 
    lblMsg->setBackgroundPixmap(QPixmap(skin->lblMsg.pixmap));
  connect(lblMsg, SIGNAL(doubleClicked()), this, SLOT(callICQMsgFunction()));
  
  // Status Label
  if (lblStatus != NULL) delete lblStatus;
  lblStatus = new CELabel(skin->lblStatus.transparent, mnuStatus, this);
  lblStatus->setFrameStyle(skin->lblStatus.frameStyle);
  lblStatus->setMargin(skin->lblStatus.margin);
  lblStatus->setNamedFgColor(skin->lblStatus.color.fg);  
  lblStatus->setNamedBgColor(skin->lblStatus.color.bg);
  if (skin->lblStatus.transparent)
  {
    QPixmap *p = MakeTransparentBg(pmBorder, skin->borderToRect(&skin->lblStatus, pmBorder));
    lblStatus->setBackgroundPixmap(*p);
    delete p;
  }
  else if (skin->lblStatus.pixmap != NULL) 
    lblStatus->setBackgroundPixmap(QPixmap(skin->lblStatus.pixmap));
  connect(lblStatus, SIGNAL(doubleClicked()), awayMsgDlg, SLOT(show()));

  resizeEvent(NULL);
}


//-----CMainWindow::destructor--------------------------------------------------
CMainWindow::~CMainWindow(void)
{
   // save window position and size
   char buf[MAX_FILENAME_LEN];
   sprintf(buf, "%s%s%s", BASE_DIR, CONF_DIR, "geometry.conf");
   QFile geoConf(buf);
   geoConf.open(IO_WriteOnly);
   sprintf(buf, "[geometry]\nx = %d\ny = %d\nw = %d\nh = %d\n", x(), y(), width(), height());
   geoConf.writeBlock(buf, strlen(buf));
   geoConf.close();
}


//-----CMainWindow::resizeEvent------------------------------------------------------------------------
void CMainWindow::resizeEvent (QResizeEvent *)
{
   userView->setGeometry(skin->frame.border.left, skin->frame.border.top, 
                         width() - skin->frameWidth(), height() - skin->frameHeight());
   userView->maxLastColumn();

   cmbUserGroups->setGeometry(skin->borderToRect(&skin->cmbGroups, this));
   if (!skin->frame.hasMenuBar) 
      btnSystem->setGeometry(skin->borderToRect(&skin->btnSys, this));
   lblMsg->setGeometry(skin->borderToRect(&skin->lblMsg, this));
   lblStatus->setGeometry(skin->borderToRect(&skin->lblStatus, this));

   // Resize the background pixmap and mask
   QPixmap *p;
   if (skin->frame.pixmap != NULL) 
   {
      p = ScaleWithBorder(*pmBorder, width(), height(), skin->frame.border);
      setBackgroundPixmap(*p);
      delete p;
   }
   if (skin->frame.mask != NULL)
   {
      p = ScaleWithBorder(*pmMask, width(), height(), skin->frame.border);
      bmMask = *p;
      setMask(bmMask);
      delete p;
   }
}


//-----CMainWindow::mouseEvent----------------------------------------------
void CMainWindow::mousePressEvent(QMouseEvent *m) 
{
   mouseX = m->x();
   mouseY = m->y();
   /*if (mouseY > height() - skin->frame.border.bottom)
   {
      userView->hide();
      cmbUserGroups->hide();
      if (!skin->frame.hasMenuBar) btnSystem->hide();
      lblMsg->hide();
      lblStatus->hide();
   } */
}

void CMainWindow::mouseMoveEvent(QMouseEvent *m)
{
   int deltaX = m->x() - mouseX;
   int deltaY = m->y() - mouseY;
   
   /*if (mouseY > height() - skin->frame.border.bottom) 
   {
      resize(width() + deltaX, height() + deltaY);
      mouseX += deltaX;
      mouseY += deltaY;
   }
   else*/
      move(x() + deltaX, y() + deltaY);
}

void CMainWindow::mouseReleaseEvent(QMouseEvent *m) 
{
   /*if (mouseY > height() - skin->frame.border.bottom)
   {
      userView->show();
      cmbUserGroups->show();
      if (!skin->frame.hasMenuBar) btnSystem->show();
      lblMsg->show();
      lblStatus->show();
   }*/
}



//-----CMainWindow::updateUserWin-----------------------------------------------
void CMainWindow::updateUserWin()
{
  unsigned short i;
  
  // set the pixmap and color for each user and add them to the view
  if (m_cCurrentGroup == NULL) 
  {
    gLog.Error("%sInternal error: Something is broken with the current group.\n%sUnable to refresh contact list.\n", 
               L_ERRORxSTR, L_BLANKxSTR);
    return;
  }

  userView->setUpdatesEnabled(false);
  userView->clear();
  for (i = 0; i < m_cCurrentGroup->getNumUsers(); i++)
  {
    if (!m_bShowOffline && m_cCurrentGroup->getUser(i)->getStatusOffline()) break;
    (void) new CUserViewItem(m_cCurrentGroup->getUser(i), i, userView);
  }
  userView->setUpdatesEnabled(true);
  userView->repaint();
  
  unsigned short nNumUserEvents = ICQUser::getNumUserEvents() -
                                  server->icqOwner.getNumMessages();
  if (server->icqOwner.getNumMessages() > 0) 
  {
     char msgsText[16];
     sprintf (msgsText, "SysMsg");
     lblMsg->setText(msgsText);      
  }
  else if (nNumUserEvents > 0)
  {
     char msgsText[16];
     sprintf (msgsText, "%d msg%c", nNumUserEvents, (nNumUserEvents == 1 ? ' ' : 's'));
     lblMsg->setText(msgsText);
  }
  else 
  {
     lblMsg->setText("No msgs");
  }
  lblMsg->update();
  setDockIconMsg(nNumUserEvents > 0, server->icqOwner.getNumMessages() > 0);
}


//-----CMainWindow::setDockIconStatus-------------------------------------------
void CMainWindow::setDockIconStatus(void)
{
#ifdef USE_DOCK   
   QPixmap m;
   switch (server->icqOwner.getStatus())
   {
   case ICQ_STATUS_ONLINE:  m = QPixmap(iconOnline_xpm); break;
   case ICQ_STATUS_AWAY: m = QPixmap(iconAway_xpm); break;
   case ICQ_STATUS_NA: m = QPixmap(iconNA_xpm); break;
   case ICQ_STATUS_OCCUPIED:  m = QPixmap(iconOccupied_xpm); break;
   case ICQ_STATUS_DND:  m = QPixmap(iconDND_xpm); break;
   case ICQ_STATUS_FREEFORCHAT:  m = QPixmap(iconFFC_xpm); break;
   case ICQ_STATUS_OFFLINE: m = QPixmap(iconOffline_xpm); break;
   } 
   if (server->icqOwner.getStatusInvisible()) m = QPixmap(iconInvisible_xpm);
   
   setDockIcon(m);
#endif // USE_DOCK
}


//-----CMainWindow::setDockIconMsg----------------------------------------------
void CMainWindow::setDockIconMsg(bool bNewMsg, bool bSysMsg)
{
#ifdef USE_DOCK
   QPixmap m;
   if (bNewMsg) 
      m = QPixmap(iconMsg_xpm);
   else
      m = QPixmap(iconNoMsg_xpm);
   setDockIcon(m);
   if (bSysMsg) setDockIcon(QPixmap(iconSysMsg_xpm));
#else
   if (bNewMsg)
      setIcon(QPixmap(xpmIconNewMsg));
   else
      setIcon(QPixmap(xpmIcon));
#endif // USE_DOCK
}


//-----CMainWindow::setDockIcon-------------------------------------------------
void CMainWindow::setDockIcon(const QPixmap &mask)
{
   if (mask.isNull()) 
   {
     gLog.Error("%sInternal error: Unable to set icon.\n", L_ERRORxSTR);
     return;
   }
   if (icon() == NULL)
   {
      setIcon(mask);
   }
   else
   {
      QPixmap p(*icon());
      QPainter painter(&p);
      painter.drawPixmap(QPoint(0, 0), mask);
      painter.end();
      setIcon(p);
   }
}


//-----CMainWindow::setCurrentGroup---------------------------------------------
void CMainWindow::setCurrentGroup(int index)
{
   m_cCurrentGroup = server->getUsers()->getGroup(index);
   updateUserWin();
}


//-----CMainWindow::updateStatus------------------------------------------------
void CMainWindow::updateStatus()
{  
   char *theColor = colorOffline;
   char sStatus[32];   
   switch (server->icqOwner.getStatus())
   {
   case ICQ_STATUS_ONLINE:  strcpy(sStatus, "Online"); theColor = colorOnline; break;
   case ICQ_STATUS_AWAY: strcpy(sStatus, "Away"); theColor = colorAway;  break;
   case ICQ_STATUS_NA: strcpy(sStatus, "Not Available"); theColor = colorAway; break;
   case ICQ_STATUS_OCCUPIED: strcpy(sStatus, "Occupied"); theColor = colorAway; break;
   case ICQ_STATUS_DND: strcpy(sStatus, "Do Not Disturb"); theColor = colorAway; break;
   case ICQ_STATUS_FREEFORCHAT: strcpy(sStatus, "Free for Chat"); theColor = colorOnline; break;
   case ICQ_STATUS_OFFLINE: strcpy(sStatus, "Offline"); theColor = colorOffline; break;
   default: 
      gLog.Error("%sInternal error: CMainWindow::updateStatus(): bad status value %d.\n",
                 L_ERRORxSTR, server->icqOwner.getStatus());
      strcpy(sStatus, "Unknown"); 
      theColor = colorAway;
      break;
   } 
   if (server->icqOwner.getStatusInvisible())
   {
      char sStatusInv[32];
      strcpy(sStatusInv, sStatus);
      sprintf(sStatus, "(%s)", sStatusInv);
   }

   lblStatus->setText(sStatus);
   lblStatus->update();

   // set the color if it isn't set by the skin
   if (skin->lblStatus.color.fg == NULL) lblStatus->setNamedFgColor(theColor);

#ifdef USE_DOCK
   setDockIconStatus();
#endif // USE_DOCK
}

//----CMainWindow::changeStatusManual------------------------------------------------------------------
void CMainWindow::changeStatusManual(int index)
{
   if (index != MNUxITEM_STATUSxINVISIBLE) manualAway = index;
   if (index == 1 || index == 2 || index == 3 || index == 4)
      awayMsgDlg->show();
   changeStatus(index);
}


//----CMainWindow::changeStatus-------------------------------------------------
void CMainWindow::changeStatus(int index)
{
   unsigned long newStatus = ICQ_STATUS_OFFLINE;
   
   switch(index)
   {
   case 0: newStatus = ICQ_STATUS_ONLINE; break;
   case 1: newStatus = ICQ_STATUS_AWAY; break;
   case 2: newStatus = ICQ_STATUS_NA; break;
   case 3: newStatus = ICQ_STATUS_OCCUPIED; break;
   case 4: newStatus = ICQ_STATUS_DND; break;
   case 5: newStatus = ICQ_STATUS_FREEFORCHAT; break;
   case 6: newStatus = ICQ_STATUS_OFFLINE; 
      server->icqLogoff(false);
      return;
   case MNUxITEM_STATUSxINVISIBLE:  // toggle invisible status
      mnuStatus->setItemChecked(MNUxITEM_STATUSxINVISIBLE, 
                                !mnuStatus->isItemChecked(MNUxITEM_STATUSxINVISIBLE));
      if (server->icqOwner.getStatusOffline()) return;
      if (mnuStatus->isItemChecked(MNUxITEM_STATUSxINVISIBLE))
         newStatus = server->icqOwner.getStatusFull() | ICQ_STATUS_FxPRIVATE; 
      else
         newStatus = server->icqOwner.getStatusFull() & (~ICQ_STATUS_FxPRIVATE);
      break;
   default: 
      gLog.Error("%sInternal error: CMainWindow::changeStatus(): bad index value %d.\n",
                       L_ERRORxSTR, index);
   }
   
   // we may have been offline and gone online with invisible toggled
   if (mnuStatus->isItemChecked(MNUxITEM_STATUSxINVISIBLE))
      newStatus |= ICQ_STATUS_FxPRIVATE;
   
   // maintain the current extended status flags (we aren't changing any of
   // them in this function so it's ok)
   newStatus |= server->icqOwner.getStatusFlags();
   
   // disable combo box, flip pixmap...
   lblStatus->setEnabled(false);
   
   // call the right function
   if (server->icqOwner.getStatusOffline())
   {
      if (!server->icqLogon(newStatus))  // unable to connect to mirabilis server
         lblStatus->setEnabled(true);  // errors will be reported by the ICQ subsystem
   }
   else 
      server->icqSetStatus(newStatus);
}


//-----CMainWindow::callDefaultICQFunction-------------------------------------------------------------
void CMainWindow::callDefaultICQFunction()      { callICQFunction(-1, true);   }
void CMainWindow::callICQOwnerFunction()        { callICQFunction((server->icqOwner.getNumMessages() > 0 ? 0 : 3), false);  }

void CMainWindow::callICQMsgFunction()
{
  // Do system messages first
  if (server->icqOwner.getNumMessages() > 0)
  {
    callICQOwnerFunction();
    return;
  }

  // Do nothing if there are no events pending
  if (ICQUser::getNumUserEvents() == 0) return;

  // We go through backwards to (almost) get the oldest messages first
  unsigned short i = server->getNumUsers(); 
  while (i--)
  {
    if (server->getUser(i)->getNumMessages() > 0)
    {
      callICQFunction(0, true, server->getUser(i));
      return;
    }
  }
}

void CMainWindow::callICQUserFunction(int fcn)  
{
   if (fcn <= MNUxITEM_VIEWxHISTORY) 
      callICQFunction(fcn, true); 
   else if (fcn == MNUxITEM_ONLINExNOTIFY) 
   {
      ICQUser *u = userView->getSelectedUser();
      if (u == NULL) return;
      u->setOnlineNotify(!u->getOnlineNotify());
   }
   else if (fcn == MNUxITEM_INVISIBLExLIST) 
   {
      ICQUser *u = userView->getSelectedUser();
      if (u == NULL) return;
      u->setInvisibleList(!u->getInvisibleList());
      server->icqSendInvisibleList(true);
   }
   else if (fcn == MNUxITEM_VISIBLExLIST) 
   {
      ICQUser *u = userView->getSelectedUser();
      if (u == NULL) return;
      u->setVisibleList(!u->getVisibleList());
      server->icqSendVisibleList(true);
   }
}     


//-----CMainWindow::callICQFunction--------------------------------------------------------------------
void CMainWindow::callICQFunction(int fcn, bool isUser, ICQUser *user = NULL)
{
   if (!isUser) 
      user = &server->icqOwner;
   else if (user == NULL)
      user = userView->getSelectedUser();
   
   if (user != NULL) 
   {
      // set default function to read or send depending on whether or not 
      // there are new messages
      if (fcn < 0) fcn = (user->getNumMessages() == 0 ? 1 : 0);  
      if (user->fcnDlg == NULL) 
      {
         user->fcnDlg = new ICQFunctions(server, user, autoClose);
         connect ((ICQFunctions *)(user->fcnDlg), SIGNAL(signal_updatedUser()), this, SLOT(updateUserWin()));
         ((ICQFunctions *)(user->fcnDlg))->setupTabs(fcn);
         connect ((ICQFunctions *)(user->fcnDlg), SIGNAL(signal_finished(ICQUser *)), this, SLOT(killICQFunction(ICQUser *)));
      }
      //else raise it to the top
   }   
}

void CMainWindow::killICQFunction(ICQUser *u)
{
   if (u->fcnDlg != NULL) 
   {
      delete (ICQFunctions *)(u->fcnDlg);
      u->fcnDlg = NULL;
   }
}


//-----CMainWindow::done-------------------------------------------------------------------------------
void CMainWindow::done(bool isOk, unsigned short cmd)
{
   updateStatus();
   switch (cmd) 
   {
   case ICQ_CMDxSND_LOGON: 
      lblStatus->setEnabled(true);
      if (!isOk) gLog.Error("%sError logging on.\n", L_ERRORxSTR);
      break;
   case ICQ_CMDxSND_LOGOFF: 
      lblStatus->setEnabled(true);
      if (!isOk) gLog.Error("%sError logging of.\n", L_ERRORxSTR);
      break;
   case ICQ_CMDxSND_SETxSTATUS: 
      lblStatus->setEnabled(true);
      if (!isOk) gLog.Error("%sError changing status.\n", L_ERRORxSTR);
      break;
   case ICQ_CMDxSND_UPDATExBASIC:
      if (!isOk) gLog.Error("%sError updating personal info.\n", L_ERRORxSTR);
      break;
   case ICQ_CMDxSND_USERxLIST:
      updateUserWin();
      if (!isOk) gLog.Error("%sError retrieving contact list.\n", L_ERRORxSTR);
      break;
   case ICQ_CMDxSND_INVISIBLExLIST:
      if (!isOk) gLog.Error("%sError sending invisible list.\n", L_ERRORxSTR);
      break;
   case ICQ_CMDxSND_VISIBLExLIST:
      if (!isOk) gLog.Error("%sError sending visible list.\n", L_ERRORxSTR);
      break;
   case ICQ_CMDxSND_SYSxMSGxREQ:
      if (!isOk) gLog.Error("%sError retrieving system messages.\n", L_ERRORxSTR);
      break;
   case ICQ_CMDxSND_PING:
      //if (!isOk) QMessageBox::critical( this, "Licq", "Error pinging Mirabilis, see Network Window for details.", QMessageBox::Ok, 0);
      break;
   case ICQ_CMDxSND_SYSxMSGxDONExACK:
      if (!isOk) gLog.Error("%sError acknowledging system messages.\n", L_ERRORxSTR);
      break;
   case ICQ_CMDxSND_USERxADD:
      if (!isOk) gLog.Error("%sError informing server of new user.\n", L_ERRORxSTR);
      break;   
   case ICQ_CMDxSND_AUTHORIZE:
      if (!isOk) 
        gLog.Error("%sError sending autorization.\n", L_ERRORxSTR);
      else 
      {
        gLog.AddLogTypeToService(L_INFO, S_MSGBOX);
        gLog.Info("%sAuthorization granted.\n", L_UDPxSTR);
        gLog.RemoveLogTypeFromService(L_INFO, S_MSGBOX);
      }
      break;   
   default:
      break;
   }
}



//-----CMainWindow::removeUser--------------------------------------------------
void CMainWindow::removeUserFromList()
{
   ICQUser *u = userView->getSelectedUser();
   if (u == NULL) return;
   char warning[256];
   sprintf(warning, "Are you sure you want to remove\n%s (%ld)\nfrom your contact list?", 
           u->getAlias(), u->getUin());
   if(QueryUser(this, warning, "Ok", "Cancel"))
      server->usrRemove(u);
}

void CMainWindow::removeUserFromGroup()
{
   if (m_cCurrentGroup == server->getUsers()->getGroup(0)) 
      removeUserFromList();
   else
   {
      ICQUser *u = userView->getSelectedUser();
      if (u == NULL) return;
      char warning[256];
      sprintf(warning, "Are you sure you want to remove\n%s (%ld)\nfrom the '%s' group?", 
              u->getAlias(), u->getUin(), m_cCurrentGroup->getName());
      if (QueryUser(this, warning, "Ok", "Cancel"))
      {
         server->getUsers()->removeUserFromGroup(u, cmbUserGroups->currentItem());
         updateUserWin();
      }
   }
}


//-----CMainWindow::saveAllUsers------------------------------------------------
void CMainWindow::saveAllUsers(void)
{
   server->getUsers()->saveAllUsers();
}

//-----CMainWindow::addUserToGroup----------------------------------------------
void CMainWindow::addUserToGroup(int _nGroup)
{
   server->getUsers()->addUserToGroup(userView->getSelectedUser(), _nGroup);
}


//-----CMainWindow::showOptions------------------------------------------------------------------------
void CMainWindow::showOptions()
{
   if (optionsDlg != NULL) return;
   optionsDlg = new OptionsDlg;

   optionsDlg->edtFontFamily->setText(fontFamily);
   optionsDlg->spnFontSize->setValue(fontSize);
   optionsDlg->cmbFontCharSet->setCurrentItem(fontCharSetInt - 1);

   optionsDlg->edtColorOnline->setText(colorOnline);
   optionsDlg->edtColorAway->setText(colorAway);
   optionsDlg->edtColorOffline->setText(colorOffline);
   optionsDlg->edtColorNew->setText(colorNew);
   optionsDlg->edtColorBackground->setText(colorBackground);
   optionsDlg->edtColorGridLines->setText(colorGridLines);
   
   optionsDlg->chkGridLines->setChecked(gridLines);
   optionsDlg->chkHeader->setChecked(showHeader);
   optionsDlg->chkAutoClose->setChecked(autoClose);

   optionsDlg->spnDefServerPort->setValue(server->getDefaultRemotePort());
   optionsDlg->spnTcpServerPort->setValue(server->getTcpServerPort());
   optionsDlg->spnMaxUsersPerPacket->setValue(server->getMaxUsersPerPacket());

   optionsDlg->cmbServers->clear();
   unsigned short i;
   for (i = 0; i < server->icqServers.numServers(); i++)
      optionsDlg->cmbServers->insertItem(server->icqServers.servers[i]->name());
   
   optionsDlg->spnAutoAway->setValue(autoAwayTime);
   optionsDlg->spnAutoNa->setValue(autoNATime);
   optionsDlg->cmbAutoLogon->setCurrentItem(m_nAutoLogon >= 10 ? m_nAutoLogon - 10 : m_nAutoLogon);
   optionsDlg->chkAutoLogonInvisible->setChecked(m_nAutoLogon > 10);

   // plugins tab
   //optionsDlg->edtErrorLog->setText(server->getErrorLogName());
   optionsDlg->edtUrlViewer->setText(server->getUrlViewer() == NULL ? "none" : server->getUrlViewer());
   const char *pc = gTranslator.getMapName();
   if (strcmp(pc, "none") == 0)
   {
     optionsDlg->cmbTrans->setCurrentItem(0);
   }
   else
   {
     unsigned short n = 1, c = optionsDlg->cmbTrans->count();
     while (n < c && strcmp(pc, optionsDlg->cmbTrans->text(n)) != 0) n++;
     if (n == c)
     {
       gLog.Error("%sError: Unable to find current translation map '%s' in translation directory.\n", 
                  L_ERRORxSTR, pc);
       optionsDlg->cmbTrans->setEnabled(false);
     }
     else
       optionsDlg->cmbTrans->setCurrentItem(n);
   }

   // set up the columns stuff
   for (i = 0; i < colInfo.size(); i++) 
   {
      optionsDlg->chkColEnabled[i]->setChecked(true);
      optionsDlg->cmbColInfo[i]->setCurrentItem(colInfo[i].info - 1);
      optionsDlg->spnColWidth[i]->setValue(userView->columnWidth(i + 1));
      optionsDlg->cmbColAlign[i]->setCurrentItem(colInfo[i].align);
      optionsDlg->chkColEnabled[i]->setEnabled(true);
      optionsDlg->cmbColInfo[i]->setEnabled(true);
      optionsDlg->spnColWidth[i]->setEnabled(true);
      optionsDlg->cmbColAlign[i]->setEnabled(true);
   }
   if (i < 4) 
   {
      optionsDlg->chkColEnabled[i]->setChecked(false);
      optionsDlg->cmbColInfo[i]->setEnabled(false);
      optionsDlg->spnColWidth[i]->setEnabled(false);
      optionsDlg->cmbColAlign[i]->setEnabled(false);
      i++;
   }
   for (; i < 4; i++) 
   {
      optionsDlg->chkColEnabled[i]->setChecked(false);
      optionsDlg->chkColEnabled[i]->setEnabled(false);
      optionsDlg->cmbColInfo[i]->setEnabled(false);
      optionsDlg->spnColWidth[i]->setEnabled(false);
      optionsDlg->cmbColAlign[i]->setEnabled(false);
   } 
   
   // set up the sound stuff
   optionsDlg->cmbSndType->setCurrentItem(server->getSoundEnabled());
   optionsDlg->edtSndPlayer->setText(server->getSoundPlayer());
   optionsDlg->edtSndMsg->setText(server->getSoundMsg());
   optionsDlg->edtSndUrl->setText(server->getSoundUrl());
   optionsDlg->edtSndChat->setText(server->getSoundChat());
   optionsDlg->edtSndFile->setText(server->getSoundFile());
   optionsDlg->edtSndNotify->setText(server->getSoundNotify());

   connect (optionsDlg, SIGNAL(signal_done()), this, SLOT(doneOptions()));
   optionsDlg->show();
}


//-----CMainWindow::getOptions-------------------------------------------------------------------------
void CMainWindow::doneOptions()
{
  // get all options from the options diagLog
  if (optionsDlg->ResultCode() == OptionsDlg::Cancel)//QDialog::Rejected)
  {
    delete optionsDlg;
    optionsDlg = NULL;
    return;
  }
  else if (optionsDlg->ResultCode() == OptionsDlg::Ok)
  {
    // Hide the dialog to avoid flicker, it will be destroyed at the end
    optionsDlg->hide();
  }
  
  strncpy(fontFamily, optionsDlg->edtFontFamily->text(), 32);
  fontSize = optionsDlg->spnFontSize->value();
  fontCharSetInt = optionsDlg->cmbFontCharSet->currentItem() + 1;
  QFont::CharSet fontCharSet;
  switch (fontCharSetInt)
  {
  case 1: fontCharSet = QFont::Latin1; break;
  case 2: fontCharSet = QFont::Latin2; break;
  case 3: fontCharSet = QFont::Latin3; break;
  case 4: fontCharSet = QFont::Latin4; break;
  case 5: fontCharSet = QFont::Latin5; break;
  case 6: fontCharSet = QFont::Latin6; break;
  case 7: fontCharSet = QFont::Latin7; break;
  case 8: fontCharSet = QFont::Latin8; break;
  case 9: fontCharSet = QFont::Latin9; break;
  case 10: fontCharSet = QFont::KOI8R; break;
  default: fontCharSet = QFont::Latin1; break;
  }
  qApp->setFont(QFont(fontFamily, fontSize, QFont::Normal, false, fontCharSet), true);

  strncpy(colorOnline, optionsDlg->edtColorOnline->text(), 32);
  strncpy(colorOffline, optionsDlg->edtColorOffline->text(), 32);
  strncpy(colorAway, optionsDlg->edtColorAway->text(), 32);
  strncpy(colorNew, optionsDlg->edtColorNew->text(), 32);
  strncpy(colorBackground, optionsDlg->edtColorBackground->text(), 32);
  strncpy(colorGridLines, optionsDlg->edtColorGridLines->text(), 32);

  gridLines = optionsDlg->chkGridLines->isChecked();
  showHeader = optionsDlg->chkHeader->isChecked();
  autoClose = optionsDlg->chkAutoClose->isChecked();

  server->setDefaultRemotePort(optionsDlg->spnDefServerPort->value());
  server->setTcpServerPort(optionsDlg->spnTcpServerPort->value());
  server->setMaxUsersPerPacket(optionsDlg->spnMaxUsersPerPacket->value());
  
  // Plugin tab
  //server->setErrorLogName((const char *)optionsDlg->edtErrorLog->text());
  server->setUrlViewer((const char *)optionsDlg->edtUrlViewer->text());
  if (optionsDlg->cmbTrans->isEnabled())
  {
    if (optionsDlg->cmbTrans->currentItem() == 0)
    {
      gTranslator.setDefaultTranslationMap();
    }
    else
    {
      char szTransFileName[MAX_FILENAME_LEN];
      sprintf(szTransFileName, "%s%s%s", BASE_DIR, TRANSLATION_DIR, optionsDlg->cmbTrans->currentText());
      gTranslator.setTranslationMap(szTransFileName);
    }
  }
  
  //optionsDlg->cmbServers->clear();
  //unsigned short i;
  //for (i = 0; i < server->icqServers.numServers(); i++)
  //   optionsDlg->cmbServers->insertItem(server->icqServers.servers[i]->name());
  
  autoAwayTime = optionsDlg->spnAutoAway->value();
  autoNATime = optionsDlg->spnAutoNa->value();
  m_nAutoLogon = optionsDlg->cmbAutoLogon->currentItem() 
                 + (optionsDlg->chkAutoLogonInvisible->isChecked() ? 10 : 0);

  // set up the columns stuff
  unsigned short i, j = colInfo.size();
  for (i = 0; i < j; i++) colInfo.pop_back();  // erase the old array
  i = 0;
  while (optionsDlg->chkColEnabled[i]->isChecked()) 
  {
     colInfo.push_back(ColInfo(optionsDlg->cmbColInfo[i]->currentItem() + 1,
                               optionsDlg->spnColWidth[i]->value(),
                               optionsDlg->cmbColAlign[i]->currentItem()));
     i++;
  }
  delete userView;
  userView = new CUserView(mnuUser, mnuGroup, colInfo, showHeader, gridLines, this);
  userView->setPixmaps(pmOnline, pmOffline, pmAway, pmNa, pmOccupied, pmDnd, 
                       pmPrivate, pmMessage);
  userView->setColors(colorOnline, colorAway, colorOffline, colorNew, 
                      colorBackground, colorGridLines);

  connect (userView, SIGNAL(doubleClicked(QListViewItem *)),
           this, SLOT(callDefaultICQFunction()));
  updateUserWin();
  userView->setGeometry(skin->frame.border.left, skin->frame.border.top, 
                        width() - skin->frameWidth(), height() - skin->frameHeight());
  userView->maxLastColumn();
  userView->show();
  
  // set up the sound stuff   
  server->setSoundEnabled(optionsDlg->cmbSndType->currentItem());
  server->setSounds(optionsDlg->edtSndPlayer->text(), optionsDlg->edtSndMsg->text(),
                    optionsDlg->edtSndUrl->text(), optionsDlg->edtSndChat->text(),
                    optionsDlg->edtSndFile->text(), optionsDlg->edtSndNotify->text());

  // Delete the dialog if the code was Ok
  if (optionsDlg->ResultCode() == OptionsDlg::Ok)
  {
    delete optionsDlg;
    optionsDlg = NULL;
  }
}


//-----CMainWindow::saveOptions------------------------------------------------------------------------
void CMainWindow::saveOptions()
{
   // save all options
   char filename[MAX_FILENAME_LEN];
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "licq.conf");
   CIniFile licqConf(INI_FxERROR | INI_FxALLOWxCREATE);
   if (!licqConf.LoadFile(filename)) return; 
      
   licqConf.SetSection("startup");
   licqConf.WriteNum("Logon", m_nAutoLogon);
   licqConf.WriteNum("AutoAway", autoAwayTime);
   licqConf.WriteNum("AutoNA", autoNATime);

   licqConf.SetSection("functions");
   licqConf.WriteBool("AutoClose", autoClose);

   licqConf.SetSection("network");
   licqConf.WriteNum("DefaultServerPort", server->getDefaultRemotePort());
   licqConf.WriteNum("TCPServerPort", server->getTcpServerPort());
   licqConf.WriteNum("MaxUsersPerPacket", server->getMaxUsersPerPacket());
   
   // Plugin tab
   //licqConf.WriteStr("Errors", server->getErrorLogName());
   licqConf.WriteStr("UrlViewer", server->getUrlViewer() == NULL ? "none" : server->getUrlViewer());
   const char *pc = strrchr(gTranslator.getMapName(), '/');
   if (pc != NULL)
     pc++;
   else
     pc = gTranslator.getMapName();
   licqConf.WriteStr("Translation", pc);
   
   //optionsDlg->cmbServers->clear();
   //unsigned short i;
   //for (i = 0; i < server->icqServers.numServers(); i++)
   //   optionsDlg->cmbServers->insertItem(server->icqServers.servers[i]->name());
   
   licqConf.SetSection("appearance");
   licqConf.WriteNum("FontSize", fontSize);
   licqConf.WriteStr("FontFamily", fontFamily);
   licqConf.WriteNum("FontCharSet", fontCharSetInt);
   licqConf.WriteStr("ColorOnline", colorOnline);
   licqConf.WriteStr("ColorOffline", colorOffline);
   licqConf.WriteStr("ColorAway", colorAway);
   licqConf.WriteStr("ColorNew", colorNew);
   licqConf.WriteStr("ColorBackground", colorBackground);
   licqConf.WriteStr("ColorGridLines", colorGridLines);
   licqConf.WriteBool("GridLines", gridLines);
   licqConf.WriteBool("ShowHeader", showHeader);
   
   // save the column info
   licqConf.WriteNum("NumColumns", (unsigned short)colInfo.size());
   char colKey[32];
   for (unsigned short i = 1; i <= colInfo.size(); i++) 
   {
      sprintf(colKey, "Column%dInfo", i);
      licqConf.WriteNum(colKey, colInfo[i - 1].info);
      sprintf(colKey, "Column%dWidth", i);
      licqConf.WriteNum(colKey, colInfo[i - 1].width);
      sprintf(colKey, "Column%dAlign", i);
      licqConf.WriteNum(colKey, colInfo[i - 1].align);
   }
      
   // save the sound stuff
   licqConf.SetSection("sound");
   licqConf.WriteNum("Enable", server->getSoundEnabled());
   licqConf.WriteStr("Player", server->getSoundPlayer());
   licqConf.WriteStr("Message", server->getSoundMsg());
   licqConf.WriteStr("Url", server->getSoundUrl());
   licqConf.WriteStr("Chat", server->getSoundChat());
   licqConf.WriteStr("File", server->getSoundFile());
   licqConf.WriteStr("OnlineNotify", server->getSoundNotify());

   licqConf.FlushFile();
}


//-----CMainWindow::aboutBox---------------------------------------------------------------------------
void CMainWindow::aboutBox()
{
   char about[1024];
   sprintf (about, "Licq version %s.\n\n%s (%ld)\n%d contacts.", VERSION, 
            server->icqOwner.getAlias(), server->icqOwner.getUin(), 
            server->getNumUsers()); 
   QueryUser(this, about, "Ok", NULL);
}


//-----CMainWindow::changeDebug------------------------------------------------------------------------
void CMainWindow::changeDebug(int _nLevel)
{
  if (_nLevel == MNUxITEM_DEBUGxALL)
  {
    gLog.ModifyService(S_STDOUT, L_ALL);
    for (int i = 0; i < 5; i++)
      mnuDebug->setItemChecked(i, true);
    return;
  }

  if (_nLevel == MNUxITEM_DEBUGxNONE)
  {
    gLog.ModifyService(S_STDOUT, L_NONE);
    for (int i = 0; i < 5; i++)
      mnuDebug->setItemChecked(i, false);
    return;
  }

  // First see if we are setting on or off the value 
  if (mnuDebug->isItemChecked(_nLevel))
  {
    gLog.RemoveLogTypeFromService(S_STDOUT, 1 << _nLevel);
    mnuDebug->setItemChecked(_nLevel, false);
  }
  else
  {
    gLog.AddLogTypeToService(S_STDOUT, 1 << _nLevel);
    mnuDebug->setItemChecked(_nLevel, true);
  }
}


//-----CMainWindow::slot_plugin-------------------------------------------------
void CMainWindow::slot_plugin(int _nPlugin)
{
  ICQUser *u = userView->getSelectedUser();
  if (u == NULL) return;
/*    
  CPlugin *p = gPluginManager.Plugin(_nPlugin);
  if (p == NULL) return;
  p->SetFields(u);

  // Set the user fields

  // Run the command
  int nSystemResult;
  switch(p->WinType())
  {
  case PluginWinGui:
  {
    p->SetBackgroundTask();
    nSystemResult = system(p->FullCommand());
    break;
  }
  case PluginWinTerm:
  {
    char szCmd[strlen(p->FullCommand() + strlen(server->Term()) + 4];
    sprintf(szCmd, "%s %s &", server->Term(), p->FullCommand());
    nSystemResult = system(szCmd);
    break;
  }
  case PluginWinLicq:
  {
    FILE *f = popen(p->FullCommand(), "r");
    nSystemResult = (f == NULL ? -1 : 0)
    // read until process is finished
    //pclose(f);
    break;
  }
  } // switch
*/
}


//-----CMainWindow::nextServer--------------------------------------------------
void CMainWindow::nextServer()
{
  // Error reporting is done at the server level 
  server->nextServer();
}


//-----ToggleShowOffline--------------------------------------------------------
void CMainWindow::ToggleShowOffline()
{
  m_bShowOffline = !m_bShowOffline;
  //mnuSystem->setItemChecked(MNUxITEM_SHOWxOFFLINE, m_bShowOffline);
  updateUserWin();
}


//-----CMainWindow::miniMode---------------------------------------------------------------------------
void CMainWindow::miniMode()
{
   static unsigned short oldH = 0;
   
   if (inMiniMode) 
   {
      userView->show();
      setMaximumHeight(2048);
      resize(width(), oldH);
      setMinimumHeight(100);
   }
   else
   {
      userView->hide();
      oldH = height();
      unsigned short newH = skin->frame.border.top + skin->frame.border.bottom;
      setMinimumHeight(newH);
      resize(width(), newH);
      setMaximumHeight(newH);
   }
   inMiniMode = !inMiniMode;
}


//-----CMainWindow::autoAway-----------------------------------------------------------
void CMainWindow::autoAway()
{
   Time idleTime = 0;  /* millisecs since last input event */
   
   unsigned short status = server->icqOwner.getStatus();
   // manualAway 1 is Away and 2 is NA (from the popup menu)
   if ((manualAway == 1 || manualAway == 2) || ( status != ICQ_STATUS_NA && status != ICQ_STATUS_ONLINE && status != ICQ_STATUS_AWAY && status != ICQ_STATUS_FREEFORCHAT))
       return;

#ifdef USE_SCRNSAVER
   static XScreenSaverInfo *mit_info = 0;
   if (!mit_info) mit_info = XScreenSaverAllocInfo ();
   XScreenSaverQueryInfo (x11Display(), DefaultRootWindow (x11Display()), mit_info);
   idleTime = mit_info->idle;
#endif // USE_SCRNSAVER

   if (((unsigned long)idleTime > (unsigned long)autoNATime * 60000 && autoNATime != 0) && (status != ICQ_STATUS_NA))
      changeStatus(2);   // set na mode
   else if (((unsigned long)idleTime > (unsigned long)autoAwayTime * 60000 && autoAwayTime != 0) && (status != ICQ_STATUS_AWAY) && (status != ICQ_STATUS_NA))
      changeStatus(1);  // go into away mode
   else if ((status == ICQ_STATUS_AWAY || status == ICQ_STATUS_NA) && ((unsigned long)idleTime < (unsigned long)autoAwayTime * 60000 && autoAwayTime != 0)) 
      changeStatus(0); // wake back up

}


//-----CMainWindow::eventResult------------------------------------------------------------------------
void CMainWindow::eventResult(ICQUser *u, CUserEvent *e, bool _bAccepted, 
                              const char *_szResponse, unsigned long port)
{
   // This would suck and should never happen
   if (e == NULL) 
   {
      gLog.Error("%sInternal error: CMainWindow::eventResult(): CUserEvent is NULL!\n", L_ERRORxSTR);
      return;
   }

   if (!_bAccepted) 
   {
      char result[128];
      sprintf(result, "%s%s with %s refused:\n%s%s", L_TCPxSTR, e->Description(), 
              u->getAlias(), L_BLANKxSTR, _szResponse);
      gLog.AddLogTypeToService(L_INFO, S_MSGBOX);
      gLog.Info("%s\n", result);
      gLog.RemoveLogTypeFromService(L_INFO, S_MSGBOX);
      return;
   }

   // Request was accepted
   switch (e->SubCommand()) 
   {
   case ICQ_CMDxSUB_CHAT:
   {
      ChatDlg *chatDlg = new ChatDlg(u, server->icqOwner.getAlias(), false, port);
      chatDlg->show();
      break;
   }
   case ICQ_CMDxSUB_FILE:
   {
      CFileDlg *fileDlg = new CFileDlg(u, server->icqOwner.getAlias(),
                                       ((CEventFile *)e)->Filename(), 
                                       ((CEventFile *)e)->FileSize(),  
                                       false, port);
      fileDlg->show(); 
      break;
   }
   default:
      break;
   }
}



void CMainWindow::popupSystemMenu()
{
   mnuSystem->popup(mapToGlobal(QPoint(btnSystem->x(), btnSystem->y())));
}


//-----CMainWindow::loadIcons---------------------------------------------------
void CMainWindow::loadIcons(char *_sIconSet)
// load the pixmaps
{   
   char sFilename[MAX_FILENAME_LEN], 
        sFilepath[MAX_FILENAME_LEN], 
        sIconPath[MAX_FILENAME_LEN];
   
   sprintf(sFilename, "%s%sicons.%s/%s.icons", BASE_DIR, DATA_DIR, _sIconSet, 
           _sIconSet);
   CIniFile fIconsConf(INI_FxERROR | INI_FxFATAL);
   fIconsConf.LoadFile(sFilename);
   sprintf(sIconPath, "%s%sicons.%s/", BASE_DIR, DATA_DIR, _sIconSet);
   fIconsConf.SetFlags(INI_FxWARN);

   fIconsConf.SetSection("icons");
   fIconsConf.ReadStr("Online", sFilename, "");
   sprintf(sFilepath, "%s%s", sIconPath, sFilename);
   pmOnline = new QPixmap(sFilepath);
   if (pmOnline->isNull()) 
     gLog.Error("%sUnable to open 'online' pixmap '%s'.\n%sUsing blank.\n", 
               L_ERRORxSTR, sFilepath, L_BLANKxSTR);
   
   fIconsConf.ReadStr("Offline", sFilename, "");
   sprintf(sFilepath, "%s%s", sIconPath, sFilename);   
   pmOffline = new QPixmap(sFilepath);
   if (pmOffline->isNull()) 
     gLog.Error("%sUnable to open 'offline' pixmap '%s'.\n%sUsing blank.\n", 
               L_ERRORxSTR, sFilepath, L_BLANKxSTR);
   
   fIconsConf.ReadStr("Away", sFilename, "");
   sprintf(sFilepath, "%s%s", sIconPath, sFilename);   
   pmAway = new QPixmap(sFilepath);
   if (pmAway->isNull()) 
     gLog.Error("%sUnable to open 'away' pixmap '%s'.\n%sUsing blank.\n", 
               L_ERRORxSTR, sFilepath, L_BLANKxSTR);
   
   fIconsConf.ReadStr("NA", sFilename, "");
   sprintf(sFilepath, "%s%s", sIconPath, sFilename);   
   pmNa = new QPixmap(sFilepath);
   if (pmNa->isNull())   
     gLog.Error("%sUnable to open 'N/A' pixmap '%s'.\n%sUsing blank.\n", 
                L_ERRORxSTR, sFilepath, L_BLANKxSTR);
   
   fIconsConf.ReadStr("Occupied", sFilename, "");
   sprintf(sFilepath, "%s%s", sIconPath, sFilename);   
   pmOccupied = new QPixmap(sFilepath);
   if (pmOccupied->isNull()) 
     gLog.Error("%sUnable to open 'occupied' pixmap '%s'.\n%sUsing blank.\n", 
               L_ERRORxSTR, sFilepath, L_BLANKxSTR);
   
   fIconsConf.ReadStr("DND", sFilename, "");
   sprintf(sFilepath, "%s%s", sIconPath, sFilename);   
   pmDnd = new QPixmap(sFilepath);
   if (pmDnd->isNull()) 
     gLog.Error("%sUnable to open 'DND' pixmap '%s'.\n%sUsing blank.\n", 
               L_ERRORxSTR, sFilepath, L_BLANKxSTR);

   fIconsConf.ReadStr("Private", sFilename, "");
   sprintf(sFilepath, "%s%s", sIconPath, sFilename);   
   pmPrivate = new QPixmap(sFilepath);
   if (pmPrivate->isNull())
     gLog.Error("%sUnable to open 'private' pixmap '%s'.\n%sUsing blank.\n", 
               L_ERRORxSTR, sFilepath, L_BLANKxSTR);
   
   fIconsConf.ReadStr("Message", sFilename, "");
   sprintf(sFilepath, "%s%s", sIconPath, sFilename);   
   pmMessage = new QPixmap(sFilepath);
   if (pmMessage->isNull())
     gLog.Error("%sUnable to open 'message' pixmap '%s'.\n%sUsing blank.\n", 
               L_ERRORxSTR, sFilepath, L_BLANKxSTR);
}


//-----CMainWindow::initMenu----------------------------------------------------
void CMainWindow::initMenu(void)
{   
   int mnuId;
   mnuStatus = new QPopupMenu(NULL);
   mnuId = mnuStatus->insertItem("&Online");
   mnuStatus->setAccel(ALT + Key_O,mnuId);
   mnuId = mnuStatus->insertItem("&Away");
   mnuStatus->setAccel(ALT + Key_A,mnuId);
   mnuId = mnuStatus->insertItem("&Not Available");
   mnuStatus->setAccel(ALT + Key_N,mnuId);
   mnuId = mnuStatus->insertItem("O&ccupied");
   mnuStatus->setAccel(ALT + Key_C,mnuId);
   mnuId = mnuStatus->insertItem("&Do Not Disturb");
   mnuStatus->setAccel(ALT + Key_D,mnuId);
   mnuId = mnuStatus->insertItem("Free for C&hat");
   mnuStatus->setAccel(ALT + Key_H,mnuId);
   mnuId = mnuStatus->insertItem("O&ffline");
   mnuStatus->setAccel(ALT + Key_F,mnuId);
   mnuStatus->insertSeparator();
   mnuId = mnuStatus->insertItem("&Invisible");
   mnuStatus->setAccel(ALT + Key_I,mnuId);
   connect(mnuStatus, SIGNAL(activated(int)), this, SLOT(changeStatusManual(int)));
   
   mnuUserAdm = new QPopupMenu(NULL);
   mnuUserAdm->insertItem("&Update Contact List", server, SLOT(icqUpdateContactList()));
   mnuUserAdm->insertItem("Update All Users (broken)", server, SLOT(updateAllUsers()));
   mnuUserAdm->insertItem("&Redraw User Window", this, SLOT(updateUserWin()));
   mnuUserAdm->insertItem("&Save All Users", this, SLOT(saveAllUsers()));
   mnuUserAdm->insertSeparator();
   mnuUserAdm->insertItem("&Add User", this, SLOT(showAddUserDlg()));
   mnuUserAdm->insertItem("S&earch for User", this, SLOT(showSearchUserDlg()));
   mnuUserAdm->insertItem("A&uthorize User", this, SLOT(showAuthUserDlg()));
   
   mnuDebug = new QPopupMenu(NULL);
   mnuDebug->insertItem("Status Info");
   mnuDebug->insertItem("Unknown Packets");
   mnuDebug->insertItem("Errors");
   mnuDebug->insertItem("Warnings");
   mnuDebug->insertItem("Packets");
   mnuDebug->insertSeparator();
   mnuDebug->insertItem("Set All");
   mnuDebug->insertItem("Clear All");
   mnuDebug->setCheckable(true);
   // This hack only works assuming that the initial debug level has been set
   // to the stdout log server log types
   for (int i = 0; i < 5; i++)
     mnuDebug->setItemChecked(i, DEBUG_LEVEL & (1 << i));
   connect(mnuDebug, SIGNAL(activated(int)), this, SLOT(changeDebug(int)));
      
   mnuSystem = new QPopupMenu(NULL);
   mnuSystem->setCheckable(true);
   mnuSystem->insertItem(server->icqOwner.getAlias(), this, SLOT(callICQOwnerFunction()));
   mnuSystem->insertItem("User Functions", mnuUserAdm);
   mnuSystem->insertItem("&Status", mnuStatus);
   mnuSystem->insertItem("Set &Auto Response", awayMsgDlg, SLOT(show()));
   mnuSystem->insertSeparator();
   mnuSystem->insertItem("Show &Network Window", netWin, SLOT(show()));
   mnuSystem->insertItem("Toggle &Mini Mode", this, SLOT(miniMode()));
   mnuSystem->insertItem("Toggle Offline &Users", this, SLOT(ToggleShowOffline()));
   mnuSystem->insertItem("&Options", this, SLOT(showOptions()));
   mnuSystem->insertItem("Sa&ve Options", this, SLOT(saveOptions()));
   mnuSystem->insertItem("Debug Level", mnuDebug);
   mnuSystem->insertSeparator();
   mnuSystem->insertItem("Next &Server", this, SLOT(nextServer()));
   mnuSystem->insertSeparator();
   mnuSystem->insertItem("&About", this, SLOT(aboutBox()));
   mnuSystem->insertItem("E&xit", qApp, SLOT(quit()));
   // Doesn't work because Qt sucks
   //mnuSystem->setItemChecked(MNUxITEM_SHOWxOFFLINE, m_bShowOffline);

   if (skin->frame.hasMenuBar) 
   {
#ifdef USE_KDE
      menu = new KMenuBar(this);
#else
      menu = new QMenuBar(this);
#endif
      menu->setFrameStyle(QFrame::Panel | QFrame::Raised);
      menu->insertItem(skin->btnSys.caption == NULL ? "&System" : skin->btnSys.caption, mnuSystem);
   }
   
   mnuRemove = new QPopupMenu(NULL);
   mnuRemove->insertItem("From Current Group", this, SLOT(removeUserFromGroup()));
   mnuRemove->insertItem("From Contact List", this, SLOT(removeUserFromList()));

   mnuGroup = new QPopupMenu(NULL);
   for (unsigned short i = 0; i < server->getUsers()->getNumGroups(); i++) 
      mnuGroup->insertItem(server->getUsers()->getGroup(i)->getName());
   connect(mnuGroup, SIGNAL(activated(int)), this, SLOT(addUserToGroup(int)));
/*   
   mnuPlugins = new QPopupMenu(NULL);
   for (unsigned short i = 0; i < gPluginManager.NumPlugins(); i++)
   {
     mnuPlugins->insertItem(gPluginManager.Plugin(i)->Name());
   }
   connect(mnuPlugins, SIGNAL(activated(int)), this, SLOT(slot_plugin(int)));
*/
   mnuUser = new QPopupMenu(NULL);
   mnuUser->setCheckable(true);
   mnuUser->insertItem("&Read Event");
   mnuUser->insertItem("&Send Message");
   mnuUser->insertItem("Send &Url");
   mnuUser->insertItem("&Chat Request");
   mnuUser->insertItem("&File Transfer");
   mnuUser->insertItem("Check &Away Message");
   mnuUser->insertItem("User &Info");
   mnuUser->insertItem("User &Detail");
   mnuUser->insertItem("View &History");
   mnuUser->insertSeparator();
   mnuUser->insertItem("Online Notify");
   mnuUser->insertItem("Invisible List");
   mnuUser->insertItem("Visible List");
   mnuUser->insertSeparator();
   mnuUser->insertItem("Remove", mnuRemove);
   mnuUser->insertItem("Add To Group", mnuGroup);
   mnuUser->insertSeparator();
//   mnuUser->insertItem("&Plug-ins", mnuPlugins);
   connect (mnuUser, SIGNAL(activated(int)), this, SLOT(callICQUserFunction(int)));
}


void CMainWindow::showSearchUserDlg(void)
{
   SearchUserDlg *searchUserDlg = new SearchUserDlg(server);
   searchUserDlg->show();
}


void CMainWindow::showAddUserDlg(void)
{
   AddUserDlg *addUserDlg = new AddUserDlg(server);
   connect (addUserDlg, SIGNAL(signal_updatedUsers()), this, SLOT(updateUserWin()));
   addUserDlg->show();
}


void CMainWindow::showAuthUserDlg(void)
{
   AuthUserDlg *authUserDlg = new AuthUserDlg(server);
   authUserDlg->show();
}



   
