#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include "time-fix.h"

#include "icq.h"
#include "icq-udp.h"
#include "icq-tcp.h"
#include "log.h"
//#include "plugin.h"
#include "translate.h"


//-----ICQ::constructor----------------------------------------------------------------------------
ICQ::ICQ(void) : m_vbTcpPorts(10)
{
   char szFilename[MAX_FILENAME_LEN];

   m_bAutoReconnect = true;
   m_nAllowUpdateUsers = 0;
   m_bTmpSoundEnabled = true;

   sprintf(szFilename, "%s%s%s", BASE_DIR, CONF_DIR, "licq.conf");
   CIniFile licqConf(INI_FxERROR | INI_FxFATAL);
   licqConf.LoadFile(szFilename);
   licqConf.ClearFlag(INI_FxFATAL | INI_FxERROR);
   licqConf.SetFlag(INI_FxWARN);
   
   // -----Network configuration-----
   gLog.Info("-> Network configuration.\n");

   unsigned short numRemoteServers, remoteServerPort;
   char remoteServerID[32], remotePortID[32], remoteServerName[64];
   
   licqConf.SetSection("network");
   licqConf.ReadNum("NumOfServers", numRemoteServers, 1);
   gLog.Info("   %d server(s).", numRemoteServers);
   
   // Check for a default server port - if it's not there, we set it to the internal default
   if (!licqConf.ReadNum("DefaultServerPort", m_nDefaultRemotePort, DEFAULT_SERVER_PORT))
     gLog.Info("Using internal default server port (%d).\n", DEFAULT_SERVER_PORT);
   else
     gLog.Info("   Default server port set to %d.\n", m_nDefaultRemotePort);
   

   // read in all the servers
   for(int i = 0; i < numRemoteServers; i++)
   {
      gLog.Info("   Reading in server %d: ", i + 1);
      sprintf(remoteServerID, "Server%d", i + 1);
      sprintf(remotePortID, "ServerPort%d", i + 1);
      if (!licqConf.ReadStr(remoteServerID, remoteServerName))
      {
         gLog.Info("Skipping server.\n");
         continue;
      } 
      gLog.Info("%s:", remoteServerName);
      licqConf.ClearFlag(INI_FxWARN);
      licqConf.ReadNum(remotePortID, remoteServerPort, getDefaultRemotePort());
      licqConf.SetFlag(INI_FxWARN);
      gLog.Info("%d.\n", remoteServerPort);
      icqServers.addServer(remoteServerName, remoteServerPort);
   }

   licqConf.ReadNum("TCPServerPort", m_nTcpServerPort, 0);
   gLog.Info("   TCP server port ");
   if (m_nTcpServerPort == 0) gLog.Info("will be assigned by the system.\n");
   else gLog.Info("set to %d.\n", m_nTcpServerPort);
   
   licqConf.ReadNum("MaxUsersPerPacket", m_nMaxUsersPerPacket, 125);
   
   // Error log file
   licqConf.ReadStr("Errors", szFilename, "none");
   if (strcmp(szFilename, "none") == 0)
   {
      gLog.Info("   No error log will be kept.\n");
   }
   else
   {
      char errorFileNameFull[256];
      sprintf(errorFileNameFull, "%s%s", BASE_DIR, szFilename);
      gLog.Info("   Opening %s as error log.\n", errorFileNameFull);
      CLogService_File *l = new CLogService_File(L_ERROR | L_UNKNOWN);
      if (!l->SetLogFile(errorFileNameFull, "a"))
      {
         gLog.Error("%sUnable to open %s as error log:\n%s%s.\n", 
                    L_ERRORxSTR, errorFileNameFull, L_BLANKxSTR, gLog.ErrorStr());
         delete l;
      }
      else
         gLog.AddService(l);
   }
   
   // Loading translation table from file
   licqConf.ReadStr("Translation", szFilename, "none");
   if (strncmp(szFilename, "none", 4) == 0)
   {
      gLog.Info("   Will not translate.\n");
   }
   else
   {
      char TranslationTableFileNameFull[MAX_FILENAME_LEN];
      sprintf(TranslationTableFileNameFull, "%s%s%s", BASE_DIR, TRANSLATION_DIR, szFilename);
      gLog.Info("   Opening %s as translation table.\n", TranslationTableFileNameFull);
      gTranslator.setTranslationMap (TranslationTableFileNameFull);
   }

   // Url viewer
   m_szUrlViewer = NULL;
   licqConf.ReadStr("UrlViewer", szFilename, "none");
   m_szUrlViewer = new char[strlen(szFilename) + 1];
   strcpy(m_szUrlViewer, szFilename);
   gLog.Info("   %s will be used to view URLs.\n", 
             getUrlViewer() == NULL ? "Nothing" : m_szUrlViewer);

   
   // -----Sound configuration-----
   char soundMessageFile[MAX_FILENAME_LEN], soundPlayerFile[MAX_FILENAME_LEN],
        soundUrlFile[MAX_FILENAME_LEN], soundChatFile[MAX_FILENAME_LEN],
        soundFileFile[MAX_FILENAME_LEN], soundNotifyFile[MAX_FILENAME_LEN];
   
   gLog.Info("-> Sound configuration.\n");
   licqConf.SetSection("sound");
   licqConf.ReadNum("Enable", m_nSoundEnabled, 0);
   gLog.Info("   Sound is %s.\n", m_nSoundEnabled == 0 ? "disabled" : "enabled");
   m_sSoundPlayer = m_sSoundMsg = m_sSoundUrl = m_sSoundChat = m_sSoundFile = m_sSoundNotify = NULL;
   licqConf.ReadStr("Player", soundPlayerFile, "play");
   licqConf.ReadStr("Message", soundMessageFile);
   licqConf.ReadStr("Url", soundUrlFile);
   licqConf.ReadStr("Chat", soundChatFile);
   licqConf.ReadStr("File", soundFileFile);
   licqConf.ReadStr("OnlineNotify", soundNotifyFile);
   setSounds(soundPlayerFile, soundMessageFile, soundUrlFile, soundChatFile, soundFileFile, soundNotifyFile);
   
   gLog.Info("   User defined sound player is \"%s <sound file>\"\n", getSoundPlayer());
   if (getSoundEnabled() == 1) gLog.Info("   Sound will be played using user defined player.\n");
   else if (getSoundEnabled() == 2) gLog.Info("   Sound will be played using pc speaker.\n");
/*   
   // Plugins
   sprintf(szFilename, "%s%s", BASE_DIR, PLUGIN_DIR);
   gPluginManager.LoadPlugins(szFilename);
*/
   // Miscellaneous startup stuff
   icqServers.setServer(1);    // set the initial UDP remote server (opened in logon)
   gLog.Info("   Starting TCP server...");
   if (!icqOwner.tcpSocket.StartServer(m_nTcpServerPort))    // start up the TCP server
   { 
      gLog.Info("unable to allocate port!\n");
      gLog.Error("%sUnable to allocate TCP port for local server (%s)!\n", 
                 L_ERRORxSTR, icqOwner.tcpSocket.ErrorStr());
      exit(1); 
   }
   gLog.Info("done.\n");
      
   // set up final variables and connections
   for (unsigned short i = 0; i < 10; i++) m_vbTcpPorts[i] = false;
   CPacket::setIcqOwner(&icqOwner);
   connect (icqOwner.tcpSocket.sn, SIGNAL(activated(int)), this, SLOT(recvNewTcp(int)));
   connect (&pingTimer, SIGNAL(timeout()), this, SLOT(ping()));   
}


//-----ICQ::destructor------------------------------------------------------------------------------
ICQ::~ICQ(void)
{
   // udpServer might not exist at this point. Think about it.
   icqOwner.tcpSocket.CloseConnection();
   if (!icqOwner.getStatusOffline()) icqLogoff(false);
}


//-----ICQ::simple stuff---------------------------------------------------------------------------
void ICQ::getBasicInfo(struct UserBasicInfo &us)  { icqOwner.getBasicInfo(us); }
void ICQ::getExtInfo(struct UserExtInfo &ud)  { icqOwner.getExtInfo(ud); }
int ICQ::getNumUsers(void)  { return(m_cUsers.getNumUsers()); }
void ICQ::setAwayMessage(char *m)  { icqOwner.setAwayMessage(m); }
char *ICQ::getAwayMessage(void)  { return(icqOwner.getAwayMessage()); }
void ICQ::setSoundEnabled(unsigned short s)  { m_nSoundEnabled = s; }
unsigned short ICQ::getSoundEnabled(void)  { return(m_nSoundEnabled); }
unsigned short ICQ::getTcpServerPort(void) { return(m_nTcpServerPort); }
unsigned short ICQ::getDefaultRemotePort(void)  { return(m_nDefaultRemotePort); }
bool ICQ::getTcpPort(unsigned short i)  { return (m_vbTcpPorts[i]); }
void ICQ::setTcpPort(unsigned short i, bool b) { m_vbTcpPorts[i] = b; }
char *ICQ::getSoundPlayer(void)  { return (m_sSoundPlayer); }
char *ICQ::getSoundMsg(void)     { return (m_sSoundMsg); }
char *ICQ::getSoundUrl(void)     { return (m_sSoundUrl); }
char *ICQ::getSoundChat(void)    { return (m_sSoundChat); }
char *ICQ::getSoundFile(void)    { return (m_sSoundFile); }
char *ICQ::getSoundNotify(void)  { return (m_sSoundNotify); }


ICQUser *ICQ::getUser(unsigned short i) 
{ 
  if (i < getNumUsers())
    return (m_cUsers.getUser(i));
  else
    return (NULL);
}

ICQUser *ICQ::getLastUser(void)
{
  return getUser(getNumUsers() - 1);
}

const char *ICQ::getUrlViewer(void)
{
  if (strcmp(m_szUrlViewer, "none") == 0)
    return (NULL);
  else 
    return (m_szUrlViewer);
}

void ICQ::setUrlViewer(const char *s)  
{ 
  if (m_szUrlViewer != NULL) delete m_szUrlViewer;       
  m_szUrlViewer = new char[strlen(s) + 1];
  strcpy(m_szUrlViewer, s);
}


//-----ICQ::saveUsers------------------------------------------------------------------------------
void ICQ::usrSave(void)
/* Write the user list to the conf/users.conf file, thus adding any new users and updating
   the order they appear in */
{
   char filename[128];
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "users.conf");
   FILE *usersConf = fopen(filename, "w");
   fprintf(usersConf, "[users]\nNumOfUsers = %d\n", getNumUsers());
   unsigned short j = 1;  // counter to keep track of which total user we are at
   for (unsigned short i = 0; i < getNumUsers(); i++) 
      fprintf(usersConf, "User%d = %ld\n", j++, getUser(i)->getUin());
   fclose(usersConf);
}


//-----ICQ::usrAdd--------------------------------------------------------------
void ICQ::usrAdd(unsigned long id, TCPSocket &sock)  
{ 
   ICQUser *cNewUser = m_cUsers.addUser(new ICQUser(id, sock));
   connect (cNewUser->tcpSocket.sn, SIGNAL(activated(int)), this, SLOT(recvTcp(int)));   
   icqAddUser(cNewUser);
}


//-----ICQ::usrAdd--------------------------------------------------------------
void ICQ::usrAdd(unsigned long id, char *filename)
{ 
   if (filename == NULL) 
   {
      ICQUser *cNewUser = m_cUsers.addUser(new ICQUser(id));
      icqAddUser(cNewUser);
   }
   else
      m_cUsers.addUser(new ICQUser(id, filename));
}


//-----ICQ::usrRemove-----------------------------------------------------------
void ICQ::usrRemove(ICQUser *u)
{
   m_cUsers.removeUser(u);
   usrSave();
   emit signal_updatedUsers();
}


//-----ICQ::eventStart----------------------------------------------------------
ICQEvent *ICQ::eventStart(INetSocket &sock, CPacket &packet, bool _bNeedAck,
                          unsigned long _nDestinationUin = 0, CUserEvent *e = NULL)
{
   if (!sock.Connected()) // don't send if not connected
   {
      if (e != NULL) delete e;
      return(NULL);
   }
   
   if (_bNeedAck) 
   {
      ICQEvent *thisEvent = new ICQEvent(&sock, packet, _nDestinationUin, e);
      icqEvents.push_back(thisEvent);
      connect (thisEvent, SIGNAL(signal_error(bool, int, int)), this, SLOT(eventDone(bool, int, int)));
      if (!thisEvent->start()) 
      {
         eventCancel(thisEvent);
         return(NULL);
      }
      return (thisEvent);
   }
   else
   {
      CBuffer buffer(packet.getBuffer());
      sock.Send(buffer);
      if (e != NULL) delete e;
      return (NULL);
   }
}


//-----ICQ::eventDone-----------------------------------------------------------
void ICQ::eventDone(bool gotAck, int sockfd, int theSeq)
{    
   unsigned short i;
   for (i = 0; i < icqEvents.size(); i++) if (icqEvents[i]->isEvent(sockfd, theSeq)) break;
   // If we didn't find a relevant event, ignore ack
   if (i == icqEvents.size()) 
   {
      gLog.Warn("%sReceived ack for non-local packet.\n", L_WARNxSTR);
      return;
   }

   // extract the useful information from the event
   unsigned short cmd = icqEvents[i]->getCommand();                  
   icqEvents[i]->stop();
   
   switch (cmd) 
   {
   case ICQ_CMDxTCP_START:  // fall through
   case ICQ_CMDxSND_THRUxSERVER: 
   {
      // Extract some more useful information
      CUserEvent *e = icqEvents[i]->UserEvent();
      unsigned short c = icqEvents[i]->getSubCommand();
      ICQUser *u = getUserByUIN(icqEvents[i]->getDestinationUin());
      // Write the event to the history file
      if (e != NULL)
         e->AddToHistory(&icqOwner, u, D_SENDER);
      // Let the world know if this was a file or chat ack
      if (c == ICQ_CMDxSUB_FILE || c == ICQ_CMDxSUB_CHAT) 
         emit signal_eventResult(u, e, m_bEventAccepted, m_szEventMessage, m_nEventPort);
      // Let the world know we are done with this event
      emit signal_doneUserFcn(gotAck, icqEvents[i]);
      break;
   }
   case ICQ_CMDxSND_USERxGETDETAILS: if (!gotAck) emit signal_doneUserExtInfo(false, icqEvents[i]->getDestinationUin()); break;
   case ICQ_CMDxSND_USERxGETINFO: if (!gotAck) emit signal_doneUserBasicInfo(false, icqEvents[i]->getDestinationUin()); break;
   case ICQ_CMDxSND_SETxSTATUS: 
      icqOwner.setStatus(m_nDesiredStatus);
      emit signal_doneOwnerFcn(gotAck, cmd);
      break;
   case ICQ_CMDxSND_PING: if (!gotAck) emit signal_doneOwnerFcn(false, cmd); break;
   case ICQ_CMDxSND_USERxADD: if (!gotAck) emit signal_doneOwnerFcn(false, cmd); break;
   case ICQ_CMDxSND_AUTHORIZE: emit signal_doneOwnerFcn(gotAck, cmd); break;
   case ICQ_CMDxSND_LOGON: 
      if (!gotAck) 
      {
         icqLogoff(false);
         emit signal_doneOwnerFcn(false, cmd); 
      }
      break;
   case ICQ_CMDxSND_USERxLIST: if (!gotAck) emit signal_doneOwnerFcn(false, cmd); break;
   case ICQ_CMDxSND_SYSxMSGxREQ: if (!gotAck) emit signal_doneOwnerFcn(false, cmd); break;
   case ICQ_CMDxSND_SYSxMSGxDONExACK: if (!gotAck) emit signal_doneOwnerFcn(false, cmd); break;
   default: break;
   }
   
   if (icqEvents.size() > 0)   // possibly all events have been deleted in gLogoff
   {
      delete icqEvents[i];                          // delete the event
      icqEvents[i] = icqEvents[icqEvents.size() - 1];     // move the last event to the position of the old one
      icqEvents.pop_back();                         // clear the last event
   }
   

   // gLogoff if server timed out in anything except first gLogon packet
   if (!gotAck && cmd != ICQ_CMDxTCP_START && cmd != ICQ_CMDxSND_LOGON) 
      icqLogoff(m_bAutoReconnect);
}


//-----ICQ::eventCancel---------------------------------------------------------
void ICQ::eventCancel(ICQEvent *&e)
{    
   unsigned short i;
   for (i = 0; i < icqEvents.size(); i++) if (icqEvents[i] == e) break;
   // If we didn't find a relevant event, ignore cancel
   if (i == icqEvents.size()) 
   {
      gLog.Warn("%sReceived cancel request for unknown packet.\n", L_WARNxSTR);
      return;  
   }

   icqEvents[i]->stop();
   if (icqEvents[i]->getSubCommand() == ICQ_CMDxSUB_CHAT) chatCancel(getUserByUIN(icqEvents[i]->getDestinationUin()), icqEvents[i]->getSequence(), true);
   if (icqEvents[i]->getSubCommand() == ICQ_CMDxSUB_FILE) fileCancel(getUserByUIN(icqEvents[i]->getDestinationUin()), icqEvents[i]->getSequence(), true);

   delete icqEvents[i];                          
   icqEvents[i] = icqEvents[icqEvents.size() - 1];     
   icqEvents.pop_back();
   // Make sure the pointer no longer points to the deleted memory
   e = NULL;  
}


//-----ICQ::getUserByUIN---------------------------------------------------------------------------
ICQUser *ICQ::getUserByUIN(unsigned long uin)
{
   if (icqOwner.getUin() == uin) return (&icqOwner);
   return(m_cUsers.getUserByUin(uin));
}


//-----ICQ::getIsKnownUser------------------------------------------------------
bool ICQ::getIsKnownUser(unsigned long id)
{
   if (icqOwner.getUin() == id) return (true);
   for(unsigned short i = 0; i < getNumUsers(); i++) 
      if(getUser(i)->getUin() == id) return(true);
   return (false);
}


            
//-----ICQ::setSounds------------------------------------------------------------------------------
void ICQ::setSounds(const char *p, const char *m, const char *u, const char *c, const char *f, const char *n)
{
   if (m_sSoundPlayer != NULL) delete[] m_sSoundPlayer;
   m_sSoundPlayer = new char[strlen(p) + 1];
   strcpy(m_sSoundPlayer, p);
   if (m_sSoundMsg != NULL) delete[] m_sSoundMsg;
   m_sSoundMsg = new char[strlen(m) + 1];
   strcpy(m_sSoundMsg, m);
   if (m_sSoundUrl != NULL) delete[] m_sSoundUrl;
   m_sSoundUrl = new char[strlen(u) + 1];
   strcpy(m_sSoundUrl, u);
   if (m_sSoundChat != NULL) delete[] m_sSoundChat;
   m_sSoundChat = new char[strlen(c) + 1];
   strcpy(m_sSoundChat, c);
   if (m_sSoundChat != NULL) delete[] m_sSoundFile;
   m_sSoundFile = new char[strlen(f) + 1];
   strcpy(m_sSoundFile, f);
   if (m_sSoundNotify != NULL) delete[] m_sSoundNotify;
   m_sSoundNotify = new char[strlen(n) + 1];
   strcpy(m_sSoundNotify, n);   
}


//-----ICQ::playSound-----------------------------------------------------------
void ICQ::playSound(char *theSound)
{
   // Check if sound is disabled for this particular event
   if (!m_bTmpSoundEnabled)
   {
      m_bTmpSoundEnabled = true;
      return;
   }
   
   // Check if globally sound should be played
   if (icqOwner.getStatus() & ICQ_STATUS_OCCUPIED || icqOwner.getStatus() & ICQ_STATUS_DND) 
         return;

   switch (getSoundEnabled()) 
   {
   case 0:
   {
      break;
   }
   case 1:
   {
      char soundCmd[strlen(getSoundPlayer()) + strlen(theSound) + 4];
      sprintf(soundCmd, "%s %s &", getSoundPlayer(), theSound);
      system(soundCmd);
      break;
   }
   case 2: 
   {
      printf("\a");
      fflush(stdout);
      break;
   }
   }
}


//-----ICQ::updateAllUsers-------------------------------------------------------------------------
void ICQ::updateAllUsers()
{
   for (unsigned short i = 0; i < getNumUsers(); i++) 
   {
      icqGetUserBasicInfo(getUser(i));
      icqGetUserExtInfo(getUser(i));
   }
}


//-----ICQ::ParseFE-------------------------------------------------------------
void ICQ::ParseFE(char *szBuffer, char ***szSubStr, int nMaxSubStr)
{
   char *pcEnd = szBuffer, *pcStart;
   unsigned short i = 0;
   bool bDone = false;
   //for (i = 0; i < nMaxSubStr; i++) szSubStr[i] = NULL;
   memset(*szSubStr, 0, nMaxSubStr * sizeof(char *));
   
   while (!bDone && i <= nMaxSubStr)
   {
      pcStart = pcEnd;
      while (*pcEnd != '\0' && (unsigned char)*pcEnd != (unsigned char)0xFE) pcEnd++;
      if (*pcEnd == '\0') 
         bDone = true;
      else // we are at an FE boundary
         *pcEnd++ = '\0';
      (*szSubStr)[i++] = pcStart;
   }
}

