/* libwmf (convert/wmf2fig.c): library for wmf conversion
   Copyright (C) 2000 - various; see CREDITS, ChangeLog, and sources

   The libwmf Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The libwmf Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the libwmf Library; see the file COPYING.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "wmfdefs.h"

#include "libwmf/fig.h"

#define A4_Width  596
#define A4_Height 842

#define Default_Margin 60

typedef struct
{	int    argc;
	char** argv;

	char** auto_files;
	char*  wmf_filename;
	char*  fig_filename;

	wmf_fig_t options;

} PlotData;

int  wmf2fig_draw (PlotData*);

void wmf2fig_init (PlotData*,int,char**);
void wmf2fig_help (PlotData*);
int  wmf2fig_args (PlotData*);
int  wmf2fig_auto (PlotData*);
int  wmf2fig_file (PlotData*);

int  explicit_wmf_error (char*,wmf_error_t);

int wmf2fig_draw (PlotData* pdata)
{	int status = 0;

	unsigned int wmf_width;
	unsigned int wmf_height;

	unsigned int page_width;
	unsigned int page_height;

	unsigned long flags;

	int page_margin;

	float ratio_wmf;
	float ratio_page;

	static char* Default_Creator = "wmf2fig";

	wmf_error_t err;

	wmf_fig_t* ddata = 0;

	wmfAPI* API = 0;

	wmfAPI_Options api_options;

	flags = 0;

	flags |= WMF_OPT_FUNCTION;
	api_options.function = wmf_fig_function;

	flags |= WMF_OPT_ARGS;
	api_options.argc = pdata->argc;
	api_options.argv = pdata->argv;
#ifndef DEBUG
	flags |= WMF_OPT_IGNORE_NONFATAL;
#endif
	err = wmf_api_create (&API,flags,&api_options);
	status = explicit_wmf_error ("wmf_api_create",err);

	if (status)
	{	if (API) wmf_api_destroy (API);
		return (status);
	}

	err = wmf_file_open (API,pdata->wmf_filename);
	status = explicit_wmf_error ("wmf_file_open",err);

	if (status)
	{	wmf_api_destroy (API);
		return (status);
	}

	err = wmf_scan (API,0,&(pdata->options.bbox));
	status = explicit_wmf_error ("wmf_scan",err);

	if (status)
	{	wmf_api_destroy (API);
		return (status);
	}

/* Okay, got this far, everything seems cool.
 */
	ddata = WMF_FIG_GetData (API);

	ddata->out = pdata->options.out;

	if (pdata->options.Title) ddata->Title = pdata->options.Title;
	else                      ddata->Title = pdata->wmf_filename;

	if (pdata->options.Creator) ddata->Creator = pdata->options.Creator;
	else                        ddata->Creator = Default_Creator;

	if (pdata->options.Date) ddata->Date = pdata->options.Date;

	if (pdata->options.For) ddata->For = pdata->options.For;

	if (pdata->options.figunit) ddata->figunit = pdata->options.figunit;
	else                        ddata->figunit = "1200";
	
	if (pdata->options.page_format) ddata->page_format = pdata->options.page_format;
	else                        ddata->page_format = "A4";

	ddata->ddec = pdata->options.ddec;

	ddata->bmp_no = pdata->options.bmp_no;

	ddata->bmp_filename = pdata->wmf_filename;

	ddata->bbox = pdata->options.bbox;

	ddata->fig_width  = pdata->options.fig_width; 
	ddata->fig_height = pdata->options.fig_height;

	ddata->flags = pdata->options.flags;

	if (ddata->flags & WMF_FIG_STYLE_PS)
	{	
		wmf_width  = (unsigned int) ceil (ddata->bbox.BR.x - ddata->bbox.TL.x);
		wmf_height = (unsigned int) ceil (ddata->bbox.BR.y - ddata->bbox.TL.y);

		if ((wmf_width == 0) || (wmf_height == 0))
		{	fprintf (stderr,"image `%s' has no size!\n",pdata->wmf_filename);
			wmf_api_destroy (API);
			return (1);
		}

	}

	if (status == 0)
	{
		err = wmf_play (API,0,&(pdata->options.bbox));
		status = explicit_wmf_error ("wmf_play",err);
	}

	wmf_api_destroy (API);

	return (status);
}

void wmf2fig_init (PlotData* pdata,int argc,char** argv)
{	pdata->argc = argc;
	pdata->argv = argv;

	pdata->auto_files = 0;
	pdata->wmf_filename = 0;
	pdata->fig_filename = 0;

	pdata->options.out = 0;

	pdata->options.Title = 0;
	pdata->options.Creator = 0;
	pdata->options.Date = 0;
	pdata->options.For = 0;

	pdata->options.fig_width  = 0;
	pdata->options.fig_height = 0;

	pdata->options.page_format = 0;
	/*
	pdata->options.page_width  = A4_Width;
	pdata->options.page_height = A4_Height;
	*/
	pdata->options.flags = 0;

	pdata->options.figunit = 0;
	pdata->options.ddec = 1;
	pdata->options.bmp_no = 1;
}

void wmf2fig_help (PlotData* pdata)
{	fputs ("\
Usage: wmf2fig [OPTION]... [-o <file.eps>] <file.wmf>\n\
  or:  wmf2fig [OPTION]... --auto <file1.wmf> [<file2.wmf> ...]\n\
Convert metafile image to XFig format.\n\
\n\
  --figunit=<int> the FIG unit as fraction of inch, default: 1200\n\
  --page=<page>   where <page> is one of (default A4):\n\
                  A[01234] B5 Letter Legal Ledger Tabloid.\n\
  --landscape     switch to landscape view.\n\
  --portrait      switch to portrait view (default).\n\
  --flat          render at one depth value\n\
  --title=<str>   comment # Title\n\
  --creator=<str> comment # Creator\n\
  --date=<str>    comment # Date\n\
  --for=<str>     comment # For\n\
  --version       display version info and exit.\n\
  --help          display this help and exit.\n\
  --wmf-help      display wmf-related help and exit.\n\
\n\
Report bugs to wvware-devel@lists.sourceforge.net.\n",stdout);
}

int wmf2fig_args (PlotData* pdata)
{	int status = 0;
	int arg = 0;

	int    argc = pdata->argc;
	char** argv = pdata->argv;

	char* page = 0;

	while ((++arg) < argc)
	{	if (strcmp (argv[arg],"--help") == 0)
		{	wmf2fig_help (pdata);
			status = argc; /* i.e., not an error but don't continue */
			break;
		}

		if (strcmp (argv[arg],"--wmf-help") == 0)
		{	fputs (wmf_help (),stdout);
			status = argc; /* i.e., not an error but don't continue */
			break;
		}

		if (strcmp (argv[arg],"--version") == 0)
		{	fprintf (stdout,"%s: version %s\n",PACKAGE,VERSION);
			status = argc; /* i.e., not an error but don't continue */
			break;
		}

		if (strncmp (argv[arg],"--figunit=",10) == 0)
		{	pdata->options.figunit = argv[arg] + 10;
			continue;
		}

		if (strncmp (argv[arg],"--page=",7) == 0)
		{	page = argv[arg] + 7;
			if (strcmp (page,"A4") == 0) pdata->options.page_format  = "A4";
			if (strcmp (page,"A3") == 0) pdata->options.page_format  = "A3";
			if (strcmp (page,"A2") == 0) pdata->options.page_format  = "A2";
			if (strcmp (page,"A1") == 0) pdata->options.page_format  = "A1";
			if (strcmp (page,"A0") == 0) pdata->options.page_format  = "A0";
			if (strcmp (page,"B5") == 0) pdata->options.page_format  = "B5";
			if (strcmp (page,"Letter")  == 0) pdata->options.page_format  = "Letter";
			if (strcmp (page,"Legal")   == 0) pdata->options.page_format  = "Legal";
			if (strcmp (page,"Legal")   == 0) pdata->options.page_format  = "Ledger";
			if (strcmp (page,"Tabloid") == 0) pdata->options.page_format  = "Tabloid";
			if (pdata->options.page_format == 0)
			{	fprintf (stderr,"wmf2fig: page `%s' not recognized.\n",page);
				status = arg;
				break;
			}
			continue;
		}

		if (strcmp (argv[arg],"--landscape") == 0)
		{	pdata->options.flags |= WMF_FIG_LANDSCAPE;
			continue;
		}
		if (strcmp (argv[arg],"--portrait") == 0)
		{	pdata->options.flags &= ~WMF_FIG_LANDSCAPE;
			continue;
		}

		if (strcmp (argv[arg],"--flat") == 0)
		{	pdata->options.ddec = 0;
			continue;
		}

		if (strncmp (argv[arg],"--title=",8) == 0)
		{	pdata->options.Title = argv[arg] + 8;
			continue;
		}
		if (strncmp (argv[arg],"--creator=",10) == 0)
		{	pdata->options.Creator = argv[arg] + 10;
			continue;
		}
		if (strncmp (argv[arg],"--date=",7) == 0)
		{	pdata->options.Date = argv[arg] + 7;
			continue;
		}
		if (strncmp (argv[arg],"--for=",6) == 0)
		{	pdata->options.For = argv[arg] + 6;
			continue;
		}

		if (strcmp (argv[arg],"--auto") == 0)
		{	pdata->auto_files = argv + arg + 1;
			break;
		}
		if (strcmp (argv[arg],"-o") == 0)
		{	if ((++arg) < argc)
			{	pdata->fig_filename = argv[arg];
				continue;
			}
			fprintf (stderr,"usage: `wmf2fig -o <file.fig> <file.wmf>'.\n");
			fprintf (stderr,"Try `%s --help' for more information.\n",argv[0]);
			status = arg;
			break;
		}
		if (strncmp (argv[arg],"--wmf-",6) == 0)
		{	continue;
		}
		if (argv[arg][0] != '-')
		{	pdata->wmf_filename = argv[arg];
			continue;
		}

		fprintf (stderr,"option `%s' not recognized.\n",argv[arg]);
		fprintf (stderr,"Try `%s --help' for more information.\n",argv[0]);
		status = arg;
		break;
	}

	if (status == 0)
	{	if ((pdata->auto_files == 0) && (pdata->wmf_filename == 0))
		{	fprintf (stderr,"No input file specified!\n");
			fprintf (stderr,"Try `%s --help' for more information.\n",argv[0]);
			status = argc;
		}
	}

	return (status);
}

int wmf2fig_auto (PlotData* pdata)
{	int status = 0;

	while (1)
	{	pdata->wmf_filename = (*(pdata->auto_files));

		if (pdata->wmf_filename == 0) break;

		if (strcmp (pdata->wmf_filename+strlen(pdata->wmf_filename)-4,".wmf"))
		{	fprintf (stderr,"%s: expected suffix `.wmf'. ",pdata->wmf_filename);
			fprintf (stderr,"skipping...\n");
			status++;
		}
		else if ((pdata->fig_filename = malloc (strlen(pdata->wmf_filename)+1)) == 0)
		{	fprintf (stderr,"mem_alloc_err: skipping %s...\n",pdata->wmf_filename);
			status++;
		}
		else
		{	strcpy (pdata->fig_filename,pdata->wmf_filename);
			strcpy (pdata->fig_filename+strlen(pdata->fig_filename)-3,"fig");
			if (wmf2fig_file (pdata)) status++;
			free (pdata->fig_filename);
		}
		
		pdata->auto_files++;
	}

	return (status);
}

int wmf2fig_file (PlotData* pdata)
{	int status = 0;

	pdata->options.out = stdout;
	if (pdata->fig_filename)
	{	if ((pdata->options.out = fopen (pdata->fig_filename,"w")) == 0)
		{	fprintf (stderr,"unable to write to `%s'. ",pdata->fig_filename);
			fprintf (stderr,"skipping...\n");
			status = 1;
			return (status);
		}
	}

	status = wmf2fig_draw (pdata);

	if (pdata->options.out != stdout) fclose (pdata->options.out);

	return (status);
}

int main (int argc,char** argv)
{	int status = 0;

	PlotData PData;

	wmf2fig_init (&PData,argc,argv);

	status = wmf2fig_args (&PData);

	if (status) return (status);

	if (PData.auto_files) status = wmf2fig_auto (&PData);
	else                  status = wmf2fig_file (&PData);

	return (status);
}

int explicit_wmf_error (char* str,wmf_error_t err)
{	int status = 0;

	switch (err)
	{
	case wmf_E_None:
#ifdef DEBUG
		fprintf (stderr,"%s returned with wmf_E_None.\n",str);
#endif
		status = 0;
	break;

	case wmf_E_InsMem:
#ifdef DEBUG
		fprintf (stderr,"%s returned with wmf_E_InsMem.\n",str);
#endif
		status = 1;
	break;

	case wmf_E_BadFile:
#ifdef DEBUG
		fprintf (stderr,"%s returned with wmf_E_BadFile.\n",str);
#endif
		status = 1;
	break;

	case wmf_E_BadFormat:
#ifdef DEBUG
		fprintf (stderr,"%s returned with wmf_E_BadFormat.\n",str);
#endif
		status = 1;
	break;

	case wmf_E_EOF:
#ifdef DEBUG
		fprintf (stderr,"%s returned with wmf_E_EOF.\n",str);
#endif
		status = 1;
	break;

	case wmf_E_DeviceError:
#ifdef DEBUG
		fprintf (stderr,"%s returned with wmf_E_DeviceError.\n",str);
#endif
		status = 1;
	break;

	case wmf_E_Glitch:
#ifdef DEBUG
		fprintf (stderr,"%s returned with wmf_E_Glitch.\n",str);
#endif
		status = 1;
	break;

	case wmf_E_Assert:
#ifdef DEBUG
		fprintf (stderr,"%s returned with wmf_E_Assert.\n",str);
#endif
		status = 1;
	break;

	default:
#ifdef DEBUG
		fprintf (stderr,"%s returned unexpected value.\n",str);
#endif
		status = 1;
	break;
	}

	return (status);
}
