/* libwmf (<libwmf/types.h>): library for wmf conversion
   Copyright (C) 2000 - various; see CREDITS, ChangeLog, and sources

   The libwmf Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The libwmf Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the libwmf Library; see the file COPYING.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */


#ifndef LIBWMF_TYPES_H
#define LIBWMF_TYPES_H

#include <stdio.h>
#include <sys/types.h>

#include <freetype/freetype.h>

#include <libwmf/fund.h>

#ifdef __cplusplus
extern "C" {
#endif

/* API Enumeration defs
 */

typedef enum _wmf_error_t
{	wmf_E_None = 0,
	wmf_E_InsMem,
	wmf_E_BadFile,
	wmf_E_BadFormat,
	wmf_E_EOF,
	wmf_E_DeviceError,
	wmf_E_Glitch,
	wmf_E_Assert,
	wmf_E_UserExit
} wmf_error_t;

/* User redefinable functions for input-stream access:
 * wmfRead: returns unsigned char cast to int, or EOF
 * wmfSeek: returns (-1) on error, else 0
 * wmfTell: returns (-1) on error, else pos
 */
typedef int  (*wmfRead) (void*);
typedef int  (*wmfSeek) (void*,long);
typedef long (*wmfTell) (void*);

/* User definable function; should return non-zero only if premature
 * exit is required for whatever reason...
 */
typedef int (*wmfStatus) (void*,float);

/* API types
 */

typedef struct _wmfD_Coord             wmfD_Coord;
typedef struct _wmfD_Rect              wmfD_Rect;

typedef struct _wmfMapping             wmfMapping;
typedef struct _wmfFontMap             wmfFontMap;

typedef struct _wmfHead                wmfHead; /* Bit nonsensical having these three at all ?? */
typedef struct _wmfMetaHeader          wmfMetaHeader,*wmfFile;
typedef struct _wmfPlaceableMetaHeader wmfPlaceableMetaHeader;

typedef struct _wmfAPI_Options         wmfAPI_Options;
typedef struct _wmfAPI                 wmfAPI;

/* API Structure defs
 */

struct _wmfD_Coord
{	float x;
	float y;
};

struct _wmfD_Rect
{	wmfD_Coord TL;
	wmfD_Coord BR;
};

struct _wmfMapping
{	char* name;
	char* mapping;

	/* I had been hoping to keep FT out of this file, but
	 * it seems easier just to use the FT encoding defs
	 * rather than create some kind of wrapper...
	 */
	FT_Encoding encoding;
};

struct _wmfFontMap
{	char* name;       /* wmf font name */

	char* normal;     /* postscript font names */
	char* italic;
	char* bold;
	char* bolditalic;
};

struct _wmfPlaceableMetaHeader
{	U32 Key;      /* Magic number (always 9AC6CDD7h) */
	U16 Handle;   /* Metafile HANDLE number (always 0) */
	S16 Left;     /* Left coordinate in metafile units */
	S16 Top;      /* Top coordinate in metafile units */
	S16 Right;    /* Right coordinate in metafile units */
	S16 Bottom;   /* Bottom coordinate in metafile units */
	U16 Inch;     /* Number of metafile units per inch */
	U32 Reserved; /* Reserved (always 0) */
	U16 Checksum; /* Checksum value for previous 10 U16s */
};

struct _wmfHead
{	U16 FileType;      /* Type of metafile (0=memory, 1=disk) */
	U16 HeaderSize;    /* Size of header in U16S (always 9) */
	U16 Version;       /* Version of Microsoft Windows used */
	U32 FileSize;      /* Total size of the metafile in U16s */
	U16 NumOfObjects;  /* Number of objects in the file */
	U32 MaxRecordSize; /* The size of largest record in U16s */
	U16 NumOfParams;   /* Not Used (always 0) */
};

struct _wmfMetaHeader
{	wmfHead*                wmfheader;
	wmfPlaceableMetaHeader* pmh;

	FILE* filein;

	long pos;

	int placeable;
};

struct _wmfAPI_Options
{	void* context;

	void* (*malloc)  (void* context,size_t size);
	void* (*realloc) (void* context,void* mem,size_t size);
	void  (*free)    (void* context,void* mem);

	int    argc;
	char** argv;

	char** fontdirs; /* NULL-terminated list of directories to search for font files */

	struct
	{	wmfFontMap* wmf; /* {0,*}-terminated list: wmf-font-name -> ps-font-name */
		wmfMapping* sub; /* {0,*}-terminated list: wmf-font-name substring equiv */
		wmfMapping* ps;  /* {0,*}-terminated list: ps-font-name -> pfb-file-name */
	} font;

	char* sys_fontmap_file;
	char* xtra_fontmap_file;

	void (*function) (wmfAPI*);

	char*  module;
	char** dirs;

	FILE* debug_out;
	FILE* error_out;
};

struct _wmfAPI
{	wmf_error_t err; /* current state of API; wmf_E_None, hopefully... */

	wmfHead                Head;  /* structures containing meta file general properties... */
	wmfPlaceableMetaHeader PlaceableMetaHeader;
	wmfMetaHeader          MetaHeader;
	wmfFile                File;

	FILE* debug_out; /* Output streams for debugger & error reports... */
	FILE* error_out;

	void* user_data; /* These are hooks for data to hang on to... */

	void* device_data;
	void* player_data;
	void* buffer_data;
	void* memory_data;

	void* function_reference;

	void* font_data;

	char** fonts; /* NULL-terminated list of fonts loaded during wmf_scan () */

	void* color_data;

	struct /* Input stream functions... */
	{	wmfRead read;
		wmfSeek seek;
		wmfTell tell;
	} bbuf;

	struct
	{	void* context;

		wmfStatus function; /* return non-zero if premature exit desired */
	} status;

	unsigned long flags; /* General flags... */
};

/* Macro-wrappers for stream functions:
 * (int)  WMF_READ((wmfAPI*) API)                 - returns unsigned char cast to int, or EOF
 * (int)  WMF_SEEK((wmfAPI*) API,(long) position) - returns (-1) on error, else 0
 * (long) WMF_TELL((wmfAPI*) API)                 - returns (-1) on error, else current position
 */
#define WMF_READ(Z)   ((Z)->bbuf.read ((Z)->buffer_data))
#define WMF_SEEK(Z,P) ((Z)->bbuf.seek ((Z)->buffer_data,P))
#define WMF_TELL(Z)   ((Z)->bbuf.tell ((Z)->buffer_data))

#ifdef __cplusplus
}
#endif

#endif /* ! LIBWMF_TYPES_H */
