/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "terminal.h"
#include "resources.h"

namespace uta {


Terminal::Terminal(Widget* parent, int x, int y, int w, int h,
		     const Font *font, bool readonly):
  MultiLineEdit(parent,x,y,w,h,font,readonly),
  scrollTo_(65535),
  currentLine_(65535)
{
  if(textFont_->getHeight())
    allLines_ = height()/textFont_->getHeight();  
  else
    allLines_ = 1;

  if(RES_AVAILABLE("terminal_color"))
    Widget::setColor(COLOR_RES("terminal_color"));
  if(RES_AVAILABLE("terminal_surface"))
    Widget::setBackground(SURFACE_RES("terminal_surface"),false);
}

  
Terminal::~Terminal()
{
}

  
  
unsigned 
Terminal::linesCount() const
{
  return allLines_;
}
 
  
unsigned 
Terminal::currentLine() const
{
  return currentLine_;
}
void
Terminal::create()
{
  if(textChanged_)
    {  
      /* recalculate the number of lines to be able to display
       * in case we have been resized
       */
      unsigned int max_lines = height()/textFont_->getHeight();
      
      /* layout the text ...
       * MultiLineEdit::breakText will create array of lines 
       * containing max maxLinesToStore_ + max_lines
       */
      std::vector<std::string> all_lines;
      MultiLineEdit::breakText(text_, all_lines, maxLinesToStore_ + max_lines);
      
      /* how many lines at all ?
       */
      allLines_ = all_lines.size() - max_lines;

      /* now check for offset
	 offset_ = 0, means same behavior like MultiLineEdit. 
      */
      std::vector<std::string> lines;
      if(all_lines.size() > max_lines)
	{
	  /* we cannot display all text lines (some are hidden)
	   */
	  if(scrollTo_ != currentLine_)
	    {
	      /* if there is a request to scroll ...
	       * limit it to the possible maximum
	       */
	      if(scrollTo_ > all_lines.size() - max_lines)
		scrollTo_ = all_lines.size() - max_lines; 
	    }
	  else
	    {
	      /* if there is no request update the 
	       * value to reflect the new situation
	       */
	      scrollTo_ = all_lines.size() - max_lines; 
	    }

	  /* if scroll request update current pointer 
	   * and emit signal
	   */
	  if(scrollTo_ != currentLine_)
	    {
	      currentLine_ = scrollTo_;
	      scrolled.emit();
	    }
	  /* copy the lines we want to display into lines */
	  for(unsigned n=currentLine_; n < currentLine_+max_lines; n++)
	    lines.push_back(all_lines[n]);
	}
      else
	{
	  /* all lines fit on the display .. so just show all ;) */
	  if(currentLine_)
	    {
	      currentLine_ = 0;
	    }
	  lines = all_lines;
	}
      
      debugN(17, { std::vector<std::string>::iterator itr = lines.begin();
      while(itr != lines.end())
	std::cerr << *itr++ << std::endl;
      std::cerr << "----------------------------";
      });

      /* let superclass render the text */
      MultiLineEdit::renderTextLines(lines);
    }

  if(!justCursor_ || textChanged_)
    {  
      Widget::create();

      int yoff = 0;
      int xoff = 0;
      int line_height = textFont_->getHeight();
      int offset = 0;
      const Font* font;

      linkAreas_.clear();
      int xsave;

      for(unsigned n=0; n < textLines_.size(); n++)
	{
	  xoff = 0;
	  xsave = 0;
	  TextLine::iterator itr = textLines_[n].begin();
	  while(itr != textLines_[n].end())
	    {
	      xsave = xoff;
	      if(itr->image > 0 && itr->image < images_.size())
		if(images_[itr->image])
		  {
		    const Surface* img = images_[itr->image];
		    offset= line_height-img->height();
		    Rect dest(xoff, yoff+offset,
			      img->width(), img->height());
		    img->blit(surface_, dest);
		    xoff += img->width();
		  }

	      if(!itr->text.empty())
		{
		  if(itr->font > 0 && itr->font < fonts_.size())
		    font = fonts_[itr->font];
		  else 
		    font = textFont_;
		  
		  /* Font::blitString() checks for a valid palette and 
		   * falls back to the default palette if none is given
		   */		  		  
		  offset = line_height - font->getHeight();
		  if(itr->palette > 0 && itr->palette < markup_.size())
		    xoff += font->blitString(itr->text, surface_, 
					     Point(xoff, yoff+offset),
					     markup_[itr->palette]);
		  else
		    xoff += font->blitString(itr->text, surface_, 
					     Point(xoff, yoff+offset),fontPal_);
		}
	      
	      if(itr->link > 0)
		{
		  LinkArea link;
		  link.rect = Rect(xsave, yoff+offset, 
				   xoff-xsave, line_height-offset);
		  link.id = itr->link;
		  linkAreas_.push_back(link);
	      
		}
	      if(xoff > surface_->width() || yoff > surface_->height())
		break;
	      itr++;
	    }
	  yoff += textFont_->getHeight();
	}

      //move the cursor

      if(xoff+textCursor_.width() > surface_->width())
	xoff = surface_->width() - textCursor_.width();
      
      dirtyRects_.push_back(globalCoord(textCursor_));
      textCursor_.warp(Point(xoff + 1, yoff-textCursor_.height())) ;

      //reset flags
      justCursor_ = false;
      textChanged_ = false; 
    }
  
  if(!readOnly_)
    {
      if(justCursor_)
	{
	  dirtyRects_.clear();
	  dirtyRects_.push_back(globalCoord(textCursor_));
	  justCursor_ = false;
	}

      //draw the cursor ...
      if(drawCursor_) 
	drawCursor();
      else
	deleteCursor();
    }
}

void Terminal::scrollTo(int line)
{
  if(line < 0)
    return;

  if((unsigned)line != currentLine_)
    {
      scrollTo_ = line;
      textChanged_ = true;
      needsUpdate_ = true;
    }
}

void Terminal::scrollUp(int lines)
{
  if(lines <= 0 || scrollTo_ == 0)
    return;

  if(currentLine_ < (unsigned)lines)
    scrollTo_ = 0;
  else
    scrollTo_ = currentLine_ - lines;

  textChanged_ = true;
  needsUpdate_ = true;
}
  
void Terminal::scrollDown(int lines)
{  
  if(lines <= 0)
    return;
  scrollTo_ = currentLine_ + lines;
  textChanged_ = true;
  needsUpdate_ = true;
}
 
void Terminal::scroll2Top()
{
  if(scrollTo_ != 0)
    {
      scrollTo_ = 0;
      textChanged_ = true;
      needsUpdate_ = true;
    }
}  

void Terminal::scroll2Bottom()
{
  if(scrollTo_ != allLines_)
    {
      scrollTo_ = allLines_;
      textChanged_ = true;
      needsUpdate_ = true;
    }
}
 
}

