/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux <klaux@student.uni-kl.de>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "mixer.h"
#include "resources.h"
#include "debug.h"

namespace uta {

//undef this to reduce CPU by a 5%
#define HIFI 

Mixer* Mixer::standardMixer = 0;

Mixer::Mixer() :
  music_(0)
{
  if(!standardMixer)
    standardMixer = this;

#ifdef HAVE_MIXER
  audio_open_ = 0;
  
  int chunksize;

  audio_format_ = MIX_DEFAULT_FORMAT;
  audio_channels_ = MIX_DEFAULT_CHANNELS;

#if defined(HIFI)  
  audio_rate_ = 44100;
  chunksize = 2048;
#else
  audio_rate_ = MIX_DEFAULT_FREQUENCY;
  chunksize = 1024;
#endif

  /* Open the audio device */
  if (Mix_OpenAudio(audio_rate_, audio_format_, 
		    audio_channels_, chunksize) < 0) 
    {
      debugN(17,std::cerr << "Couldn't open audio. " << std::endl;);
    } 
  else 
    {
      Mix_QuerySpec(&audio_rate_, &audio_format_, &audio_channels_);
      
      debugN(17,
	     std::cerr << "Opened audio at "<< audio_rate_ << " Hz  ";
	     std::cerr << (audio_format_&0xFF) << "bit  ";
	     if(audio_channels_ > 1)
	     std::cerr << " stereo"; 
	     else
	     std::cerr << " mono" ;
	     std::cerr << std::endl;  
	     );
      
      audio_open_ = 1;
    }
#endif
}




Mixer::~Mixer()
{

  standardMixer = 0;


#ifdef HAVE_MIXER
  if ( audio_open_ ) 
    {
      debugN(17,std::cerr<<"Mixer shutting down audio...";);
      Mix_CloseAudio();
      audio_open_ = 0;
      debugN(17,std::cerr<<"[OK]"<<std::endl;);
    }
#endif
}

void Mixer::setMusic(const char* data)
{
  if(!audio_open_)
    return;

  bool playing = false;

  if(music_)
    playing = music_->playing();
  
  if(playing)
    music_->stop();

  const Music* music = MUSIC_RES(data);
  if(music)
    music_ = music;

  if(playing)
    music_->play(0);
}

void Mixer::startMusic()
{
  if(!audio_open_)
    return;

  if(music_)
    music_->play(0);
}

void Mixer::stopMusic()
{
  if(!audio_open_)
    return;

  if(music_ != 0)
    music_->stop();
}

void Mixer::playSample(const char* data)
{
  if(!audio_open_)
    return;

  const Sample *res =SAMPLE_RES(data);
  if(res)
    res->play(); 
}


int 
Mixer::setVolume(int data)
{
  if(!audio_open_)
    return 0;

#ifdef HAVE_MIXER
  //set volume for all channels
  return  Mix_Volume(-1, data);
#endif
}

int 
Mixer::setMusicVolume(int data)
{
  if(!audio_open_)
    return 0;

#ifdef HAVE_MIXER
  //set volume for all channels
  return  Mix_VolumeMusic(data);
#endif
}

void
Mixer::pauseMusic()
{
  if(!audio_open_)
    return ;
#ifdef HAVE_MIXER
  Mix_PauseMusic();
#endif
}

void
Mixer::resumeMusic()
{
  if(!audio_open_)
    return ;
#ifdef HAVE_MIXER
  Mix_ResumeMusic();
#endif
}
void
Mixer::rewindMusic()
{
  if(!audio_open_)
    return ;
#ifdef HAVE_MIXER
  Mix_RewindMusic();
#endif
}

void
Mixer::mixer_callback(void* inst, Uint8* data, int len)
{
  Mixer* static_this = (Mixer*)inst;

  static_this->data.emit(data, len);
}

#ifdef HAVE_MIXER
void 
Mixer::enableHook(bool enable)
{
  if(!audio_open_)
    return ;
  if(enable)
    {
      Mix_SetPostMix(&Mixer::mixer_callback, (void*)this);
    }
  else
    {
      Mix_SetPostMix(NULL, NULL);
    }
}

#else
void
Mixer::enableHook(bool)
{
}
#endif

}
