/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux <klaux@student.uni-kl.de>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "listbox.h"
#include "debug.h"
#include "resources.h"

namespace uta {

//the width is fixed, so that the buttons fit nicely ...
ListBox::ListBox(Widget* parent, int x, int y, int w, int h) :
  Widget(parent, x, y, w, h),
  index_(0),
  lock_(false)
{

  if(RES_AVAILABLE("listbox_color"))
    Widget::setColor(COLOR_RES("listbox_color"));
  if(RES_AVAILABLE("listbox_surface"))
    Widget::setBackground(SURFACE_RES("listbox_surface"),false);
  if(RES_AVAILABLE("listbox_selected_color"))
    selectedCol_ = COLOR_RES("listbox_selected_color");
  else
    selectedCol_ = white;

  scrollUp_ = new PushButton(this, 
			      width()-SCROLLERWIDTH, 0,
			      SCROLLERWIDTH, SCROLLERHEIGHT);

  scrollUp_->setUpPic(SURFACE_RES("arrow_up"));
  scrollUp_->setDownPic(SURFACE_RES("arrow_up_pressed"));
  //  scrollUp_->setRepeat(true);

  scrollDown_ = new PushButton(this, 
				width()-SCROLLERWIDTH, 
				height()-SCROLLERHEIGHT-height()%ELEMENTHEIGHT,
				SCROLLERWIDTH, SCROLLERHEIGHT);

  scrollDown_->setUpPic(SURFACE_RES("arrow_down"));
  scrollDown_->setDownPic(SURFACE_RES("arrow_down_pressed"));
  //scrollDown_->setRepeat(true);

  scrollUp_->pressed.connect(slot(this, &ListBox::scrollUp));
  scrollDown_->pressed.connect(slot(this, &ListBox::scrollDown));

  Button* button;

  //create the element buttons
  for(int j=0; j < height() / ELEMENTHEIGHT; j++)
    {
      button = new Button(this, 
			  0,j*ELEMENTHEIGHT,
			  width()-SCROLLERWIDTH, ELEMENTHEIGHT,j);
      if(RES_AVAILABLE("list_element_color"))
	button->setColor(COLOR_RES("list_element_color"));

      button->pressed1.connect(slot(this, &ListBox::selected));
      //      button->released1.connect(slot(this, &ListBox::unselected));
            
      buttons_.push_back(button);
    }


  //select first button
  selected_ = 0;
  setSelected(selected_);

  updateList();

  //disable the scrollers
  scrollUp_->disable();
  scrollDown_->disable();
}

ListBox::~ListBox()
{
}

void
ListBox::setSelectedColor(const Color& color)
{ 
  selectedCol_ = color;
  if(selected_-index_ < buttons_.size())
    {
      buttons_[selected_-index_]->setColor(selectedCol_);
      buttons_[selected_-index_]->setTextColor(Widget::color_, selectedCol_);
    }

}

void
ListBox::setColor(const Color& color)
{ 
  Widget::setColor(color);
  for (size_t i=0; i< buttons_.size(); i++) 
    {
      buttons_[i]->setColor(color);
      if(selected_-index_ == i)
	{
	  buttons_[selected_-index_]->setTextColor(color, selectedCol_);
	}
    }
}

void
ListBox::setDisabledColor(const Color& color)
{
  Widget::setDisabledColor(color);
  for (size_t i=0; i< buttons_.size(); i++)
    buttons_[i]->setDisabledColor(color);
}

void
ListBox::setDisabledAlpha(unsigned char alpha)
{
  Widget::setDisabledAlpha(alpha);
  for (size_t i=0; i< buttons_.size(); i++)
    buttons_[i]->setDisabledAlpha(alpha);
}


void
ListBox::updateList()
{
  
  for(unsigned int i=0; i < buttons_.size(); i++)
    if(i+index_ < elements_.size())
      buttons_[i]->setText(elements_[i+index_].c_str());
    else
      buttons_[i]->setText("");

  
  //have we reached the top ?
  if(index_ == 0)
    scrollUp_->disable();
  else
    scrollUp_->enable();


  unsigned int max;

  if(elements_.size() < buttons_.size())
    max = 0;
  else
    max = elements_.size() - buttons_.size();
  
  //have we reached the end ?
  if(index_ >= max)
    scrollDown_->disable();
  else
    scrollDown_->enable();

}

std::string 
ListBox::getSelection() const
{
  std::string selection;
  if(selected_ < elements_.size())
    selection = elements_[selected_];
  else
    selection = "";
  
  return selection;
}

void ListBox::setSelected(int data)
{
  
  if(((data-index_)>=0) && (data-index_ < buttons_.size()))
    {
      Button* b = buttons_[data-index_];
      b->press();
      b->setColor(selectedCol_);
      b->setTextColor(Widget::color_, selectedCol_);
    }
  else
    {
      selected(data-index_);
    }
}
void ListBox::addElement(const std::string& data)
{
  elements_.push_back(data);
  updateList();

}

void ListBox::addElements(const std::vector<std::string>& data)
{
  for(unsigned int i=0; i < data.size(); i++)
    elements_.push_back(data[i]);
  updateList();

}

void ListBox::clear()
{
  elements_.clear();
  updateList();
}

void ListBox::scrollUp()
{
  if(index_ > 0)
    {

      if(((selected_-index_)>=0) && (selected_-index_) < buttons_.size())
	{
	  //we lock, so the signal send by the button is ignored by unselected()
	  //unselected() clears this lock
	  lock_ = true;
	  Button* b = buttons_[selected_-index_];
	  //	  b->release();
	  b->setColor(Widget::color_);
	  b->setTextColor(selectedCol_, Widget::color_);
	}

      index_ --;

 
      if(((selected_-index_)>=0) && (selected_-index_) < buttons_.size())
	{
	  //the same for selected()
	  lock_ = true;  
	  Button* b = buttons_[selected_-index_];
	  //	  b->press();
	  b->setColor(selectedCol_);
	  b->setTextColor(Widget::color_, selectedCol_);
	}

      updateList();
   }
}


void ListBox::scrollDown()
{

  unsigned int max;

  if(elements_.size() < buttons_.size())
    max = 0;
  else
    max = elements_.size() - buttons_.size();

  if(index_ < max)
    {
      
     if(((selected_-index_)>=0) && (selected_-index_) < buttons_.size())
       {
	 //we lock, so the signal send by the button is ignored by unselected()
	 //unselected() clears this lock
	 lock_ = true;
	 Button* b = buttons_[selected_-index_];
	 //	 b->release();
	 b->setColor(Widget::color_);
	 b->setTextColor(selectedCol_, Widget::color_);
       }

      index_ ++;
      
      if(((selected_-index_)>=0) && (selected_-index_) < buttons_.size())
	{
	  //the same for selected()
	  lock_ = true;

	  Button* b = buttons_[selected_-index_];
	  //	  b->press();
	  b->setColor(selectedCol_);
	  b->setTextColor(Widget::color_, selectedCol_);
	}

      updateList();
    }
}



void 
ListBox::selected(int buttonNr)
{
  
  debugN(17,std::cerr << "ListBox: selected(" << buttonNr << ")" << std::endl;);

  if(!lock_)
    {
      lock_ = true;
      
      if(selected_ != (buttonNr+index_))
	{
	  if(((selected_-index_)>=0) && (selected_-index_) < buttons_.size())
	    {
	      Button* b = buttons_[selected_-index_];
	      //	      b->release();
	      b->setColor(Widget::color_);
	      b->setTextColor(selectedCol_, Widget::color_);
	    }

	  selected_ = buttonNr+index_;
	  
	  Button* b = buttons_[buttonNr];
	  b->setColor(selectedCol_);
	  b->setTextColor(Widget::color_, selectedCol_);

	  if(buttonNr+index_ < elements_.size())
	    {
	      debugN(17,std::cerr << "ListBox: selected: " << elements_[buttonNr+index_] << std::endl;);
	      selectionChanged(elements_[selected_].c_str());
	      selection(selected_);
	    }
	  else
	    {
	      debugN(17,std::cerr << "ListBox: selected empty slot"<< std::endl;);
	      selectionChanged("");
	      selection(-1);
	    }
	}
    }

  lock_ = false;
}


void ListBox::unselected(int buttonNr)
{

  //buttonNr is the button number

  debugN(17,std::cerr << "ListBox: unselected(" << buttonNr << ")" << std::endl;);
  
  //revert this action
  //use mutex to avoid endless loops, if toggle emits any signal
  if(!lock_)
    {
      lock_ = true;

      Button* b = buttons_[buttonNr];
      //      b->press();
      b->setColor(selectedCol_);
      b->setTextColor(Widget::color_, selectedCol_);
    }
  lock_ = false;

}




}

