/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "slider.h"
#include "resources.h"

namespace uta {

Slider::Slider(Widget *parent, int x, int y, int w, int h, int orient) :
  Widget(parent, x, y, w, h),
  buttonSurface_(NULL),
  orientation_(orient),
  value_(0.0),
  delta_(10),
  min_(0),
  max_(100),
  moving_(false)
{
  
  if(orientation_ == Slider::HORIZONTAL)
    {
      int _w = height();
      button_ = Rect(0,0,_w,height());
      delta_per_pixel_ = (max_-min_)/(float)(width()-button_.width());
    }
  else
    {
      int _h = width();
      button_ = Rect(0,height()-_h,width(),_h);     
      delta_per_pixel_ = (max_-min_)/(float)(height()-button_.height());
    }

  if(RES_AVAILABLE("slider_button_color"))
    buttonColor_ = COLOR_RES("slider_button_color");
  if(RES_AVAILABLE("slider_button_surface"))
    buttonSurface_ = SURFACE_RES("slider_button_surface");  

  if(RES_AVAILABLE("slider_color"))
    Widget::setColor(COLOR_RES("slider_color"));
  if(RES_AVAILABLE("slider_surface"))
    Widget::setBackground(SURFACE_RES("slider_surface"),false);
}

Slider::~Slider()
{
}

bool 
Slider::processEvent(const Event* event)
{
  bool handled = false;
  
  if(mouseFocus_)
    {
      if(event->type() == Event::BUTTON)
	{
	  const ButtonEvent* butEvent = (const ButtonEvent*)event;
	  
	  if(butEvent->buttons() == ButtonEvent::BUTTON_LEFT)
	    {
	      if( butEvent->state() == ButtonEvent::PRESSED &&
		  globalCoord(button_).contains(butEvent->position()) )
		{
		  position_ = butEvent->position();
		  moving_ = true;
		  
		}
	      
	      if(moving_ && butEvent->state() == ButtonEvent::RELEASED )
		{
		  moving_ = false;
		}
	      handled = true;
	    }
	}
      else if(moving_ && event->type() == Event::MOUSE)
	{
	  Point new_pos = ((MouseEvent*)event)->position();
	  if(orientation_ == Slider::HORIZONTAL)
	    value_ += (new_pos.x - position_.x)*delta_per_pixel_;
	  else
	    value_ += (new_pos.y - position_.y)*delta_per_pixel_;
	  
	  //ensure value_ is still in valid range
	  if(value_ < min_)
	    {
	      value_ = min_;
	    }
	  if(value_ > max_)
	    {
	      value_ = max_;
	    }
	  needsUpdate_ = true;
	  valueChanged.emit((int)(value_+0.45));
	  position_ = new_pos;
	}
    }
  else if(moving_)
    {
      moving_ = false;
    }

  return handled;

}

void 
Slider::create()
{
  Widget::create();

  int pos = (int)(value_ / delta_per_pixel_);
    
  if(orientation_ == Slider::HORIZONTAL)
    button_.warp(Point(pos,0));
  else
    button_.warp(Point(0,pos));
   
  if(buttonColor_ != Color(0,0,0,0))
    surface_->fill(button_, buttonColor_);

  if(buttonSurface_ && !buttonSurface_->empty())
    buttonSurface_->scaledBlit(surface_, button_);
}

void
Slider::setButtonColor(const Color& col)
{
  buttonColor_ = col;
  needsUpdate_ = true;
}

void
Slider::setButtonSurface(const Surface* surface)
{
  buttonSurface_ = surface;
  needsUpdate_ = true;
}

void 
Slider::setRange(int min, int max)
{
  if(min < max)
    {
      min_ = min;
      max_ = max;
    }
  else
    {
      max_ = min;
      min_ = max;
    }

  //ensure value_ is still in valid range
  if(value_ < min_)
    {
      value_ = min_;
      valueChanged.emit((int)(value_+0.5));
    }
  if(value_ > max_)
    {
      value_ = max_;
      valueChanged.emit((int)(value_+0.5));
    }

  if(orientation_ == Slider::HORIZONTAL)
    delta_per_pixel_ = (max_-min_)/(float)(width()-button_.width());
  else
    delta_per_pixel_ = (max_-min_)/(float)(height()-button_.height());

  needsUpdate_ = true;
}

void Slider::setDelta(int delta)
{
  delta_ = delta;
  needsUpdate_ = true;
}

void
Slider::increase()
{
  if(value_ <= max_ - delta_)
    {
      value_ += delta_;
      if(value_ > max_)
	{
	  value_ = max_;
	  valueChanged.emit((int)(value_));
	}
      else
	valueChanged.emit((int)(value_+0.5));
      needsUpdate_ = true;
    }
}

void
Slider::decrease()
{
  if(value_ >= min_ + delta_)
    {
      value_ -= delta_;
      if(value_ < min_)
	{
	  value_ = min_;
	  valueChanged.emit((int)(value_));
	}
      else
	valueChanged.emit((int)(value_+0.5));
      needsUpdate_ = true;
    }
}

void
Slider::setValue(int value)
{
  float old_value_ = value_;

  if(value > max_)
    value_ = max_;
  else if(value < min_)
    value_ = min_;
  else 
    value_ = value;
  
  if(value_ != old_value_)
    valueChanged.emit((int)(value_+0.5));
  needsUpdate_ = true;
}

} // namespace uta

