/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "multilineedit.h"
#include <ctype.h>
#include "debug.h"
#include "resources.h"

namespace uta {

MultiLineEdit::MultiLineEdit(Widget* parent, int x, int y, int w, int h, 
			       const Font *font, bool readonly):
  LineEdit(parent, x, y, w, h, font),
  wordWrap_(true),
  readOnly_(readonly),
  maxLinesToStore_(100)
{
 
  enterPressed.connect(slot(this, &MultiLineEdit::newline));

  if(RES_AVAILABLE("multilineedit_color"))
    Widget::setColor(COLOR_RES("multilineedit_color"));
  if(RES_AVAILABLE("multilineedit_surface"))
    Widget::setBackground(SURFACE_RES("multilineedit_surface"),false);
    
}

MultiLineEdit::~MultiLineEdit()
{
  for(unsigned n=0; n < renderedLines_.size(); n++)
   delete renderedLines_[n];

}

void
MultiLineEdit::breakText(string& text, vector<string>& lines, 
			 unsigned maxLines)
{
  unsigned pos = 0;
  unsigned start = 0;
  unsigned len = 0;
  unsigned linestart = 0;
  int x=0;
  vector<unsigned> linebreaks;

  Rect dest;
  bool breakLine = false;
  int size_x = 0;
  int size_y = textFont_->getHeight();
  
  lines.clear();
  
  pos=0;
  while(pos < text.size())
    {
      start = pos;
      breakLine = false;
      
      if(wordWrap_)
	{
	  //dont break words
	  while(pos<text.size())
	    {
	      if(isspace(text[pos]))
		break;
	      pos++;
	      size_x+= textFont_->getSize(text[pos]).x;
	    }
	  
	  if(pos<text.size())
	    {
	      if(text[pos] == '\n')
		{
		  breakLine = true;
		  len = pos-start;
		  pos++;
		}
	      else
		{
		  pos++;
		  size_x+= textFont_->getSize(text[pos]).x;
		  len = pos-start;
		}
	    }
	  else
	    {
	      len = pos-start;
	    }
	}
      else
	{
	  pos++;
	  size_x += textFont_->getSize(text[pos]).x;	    
	}
      
      if(!breakLine && (x + size_x >= width()))
	{
	  x = 0;
	  size_y += textFont_->getHeight();
	  
	  if(linestart != text.size())
	    {
	      if(start != linestart)
		{
		  //if the word is not longer than the line....
		  linebreaks.push_back(linestart);
		  lines.push_back(text.substr(linestart,start-linestart));
		  linestart = pos = start;
		}
	      else
		{
		  //the word is longer than the line, thus only the front part
		  //will be visible
		  linebreaks.push_back(linestart);
		  lines.push_back(text.substr(linestart,pos-linestart-1));
		  linestart = start = pos;
		}
	    }
	} 
      
      x += size_x;
      
      if(breakLine)
	{
	  x = 0;
	  size_y += textFont_->getHeight();
	  
	  if(linestart != text.size())
	    {
	      linebreaks.push_back(linestart);
	      lines.push_back(text.substr(linestart,pos-linestart-1));
	      linestart = start = pos;
	    }
	}
      
      size_x = 0;
	  
    }  //while pos < text.size()
  
  //add an empty line of last char is a '\n'
  if(breakLine)
    lines.push_back("");
  
  //do not forget the last line (if it is not empty)
  if(linestart != pos)
    lines.push_back(text.substr(linestart,pos-linestart));
  
  if(lines.size() > maxLines)
    {
      text.erase(0,linebreaks[lines.size()-maxLines]);
    }

}

void
MultiLineEdit::renderTextLines(const vector<string>& lines)
{
  unsigned int max_lines = height()/textFont_->getHeight();

  for(unsigned n=0; n < renderedLines_.size(); n++)
    if(renderedLines_[n])
      delete renderedLines_[n];
  
  unsigned start = (lines.size() <= max_lines)?0:lines.size()-max_lines;
  renderedLines_.resize(lines.size()-start);
  for(unsigned n = start; n < lines.size(); n++)
    {
      if(lines[n].empty())
	renderedLines_[n-start] = NULL;
      else
	{
	  renderedLines_[n-start] = textFont_->getString(lines[n]);
	  if(fontPal_.size() != 0)
	    renderedLines_[n-start]->setPalette(fontPal_);
	}
    }
  
}

void
MultiLineEdit::create()
{
  if(textChanged_)
    {
      unsigned int max_lines = height()/textFont_->getHeight();
      vector<string> lines;
      breakText(text_, lines, maxLinesToStore_ + max_lines);
      renderTextLines(lines);
    }

  if(!justCursor_ || textChanged_)
    {  
      Widget::create();

      Rect dest;
      for(unsigned n=0; n < renderedLines_.size(); n++)
	{
	  if(renderedLines_[n])
	    {
	      dest=Rect(0,textFont_->getHeight()*n,
			renderedLines_[n]->width(), 
			renderedLines_[n]->height());
	      if(fontPal_.size() != 0)
		renderedLines_[n]->setPalette(fontPal_);
	      renderedLines_[n]->blit(surface_,dest);
	    }
	}
      
      int x = 0;
      int y = textFont_->getHeight();
      //move the cursor

      if(renderedLines_.size() > 0)
	{
	  if(renderedLines_[renderedLines_.size()-1])
	    x = renderedLines_[renderedLines_.size()-1]->width();
	  y = textFont_->getHeight()*(renderedLines_.size());
	}

      dirtyRects_.push_back(globalCoord(textCursor_));
      textCursor_.warp(Point(x + 1, y-textCursor_.height())) ;

      //reset flags
      justCursor_ = false;
      textChanged_ = false; 
    }
  
  if(!readOnly_)
    {
      if(justCursor_)
	{
	  dirtyRects_.clear();
	  dirtyRects_.push_back(globalCoord(textCursor_));
	  justCursor_ = false;
	}

      //draw the cursor ...
      if(drawCursor_) 
	drawCursor();
      else
	deleteCursor();
    }
  
}
  
void
MultiLineEdit::setWrapping(bool flag)
{
  if(flag != wordWrap_)
    {
      textChanged_ = true;
      needsUpdate_ = true;
      wordWrap_ = flag;
    }
}
  

void MultiLineEdit::addText(const char* data)
{
  if(!data)
    return;
  text_ += string(data);
  //we need to be reblitted !
  textChanged_ = true;
  needsUpdate_ = true;
}

void MultiLineEdit::newline(const char*)
{
  text_ += string("\n");
  
  //we need to be reblitted !
  textChanged_ = true;
  needsUpdate_ = true;
}

bool
MultiLineEdit::processEvent(const Event *event)
{ 
  if(readOnly_)
    return false;

  return LineEdit::processEvent(event);
  
}

void
MultiLineEdit::timer()
{

  if(readOnly_)
    return;

  LineEdit::timer();

}

}

