/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "font.h"
#include "painter.h"
#include <assert.h>
#include "debug.h" 
#include "resources.h"

namespace uta {

//static members
TT_Engine Font::engine;
int Font::engineLock = 0;


Color Font::defaultFontColor = white;
Color Font::defaultFadeColor = Color(255,255,255,128);

Font::Font(const string& fontfilename, int ptsize):
  ptsize_(ptsize),
  height_(-1),
  surfaceTable_(NULL)
{
  if(RES_AVAILABLE("text_color"))
    fontColor_ = COLOR_RES("text_color");
  else
    fontColor_ = Font::defaultFontColor;
  

  if(RES_AVAILABLE("text_background"))
    backColor_ = COLOR_RES("text_background");
  else
    backColor_ = Font::defaultFadeColor;

  create(fontfilename, ptsize, fontColor_, backColor_);
  initialized_ = true;  
}

Font::Font(const string& fontfilename, int ptsize, 
	   const Color &fontCol, const Color &backCol) :
  ptsize_(ptsize),
  height_(-1),
  fontColor_(fontCol),
  backColor_(backCol),
  surfaceTable_(NULL)
{ 
  create(fontfilename, ptsize, fontCol, backCol);
  initialized_ = true;
}

Font::Font() :
  surfaceTable_(0)
{ 
  initialized_ = false;
}


Font::Font(const Font& font)
{
  surfaceTable_ = 0;
  initialized_ = font.initialized_;
  height_ = font.height_;
  ptsize_ = font.ptsize_;
  
  if(initialized_)
    {
      int i;
      
      palette_ = new Color[5];
      for(i=0; i<5; i++)
	palette_[i]= font.palette_[i];

      surfaceTable_ = new Surface*[128];
      for(i=0; i < 128; i++)
	surfaceTable_[i] = new Surface(*font.surfaceTable_[i]);

    } 
}

Font::Font(const Surface* img)
{
  if(!img)
    {
      initialized_ = false;
      return;
    }
  
  initialized_ = true;
  height_ = img->height();
  ptsize_ = img->height();

  fontColor_ = Color(255,255,255,0);
  backColor_ = Color(255,255,255,255);
  
  palette_ = new Color[5];
  createPalette();

  int w = img->width()/128;
  
  surfaceTable_ = new Surface*[128];

  Rect src(0,0,w,height_);
  Rect dst(0,0,w,height_);
  vector<Color> palette;
  //set palette of surface
  for(int ind=0; ind<5; ind++)
    palette.push_back(palette_[ind]);

  for(int i=0; i < 128; i++)
    {
      surfaceTable_[i] = new Surface(w,height_,Pixelformat::IND8);
      surfaceTable_[i]->setPalette(palette);
      surfaceTable_[i]->setTransColor(black);
      surfaceTable_[i]->setTransparency(true);

      src.warp(Point(i*w,0));
      img->blit(surfaceTable_[i],dst,src);
    } 
}

void
Font::create(const string& fontfilename, int ptsize, 
	      const Color &fontCol, const Color &backCol )
{

  //sanity checks:
  if(fontfilename.empty())
    return;
  if(ptsize <= 4)
    {
      cerr << "Font: only fontsizes > 4 allowed. (";
      cerr << ptsize << " was given)" <<  endl;
      return;
    }
  if(ptsize > 256)
    {
      cerr << "Font: only fontsizes <= 256 allowed. (";
      cerr << ptsize << " was given)" <<  endl;
      return;
    }

  ptsize_ = ptsize;
  height_ = -1;
  fontColor_ = fontCol;
  backColor_ = backCol;
  surfaceTable_ = 0;

  TT_Error  error;
  palette_ = new Color[5];

  
  createPalette();

  
 
  error  = TT_Init_FreeType(&Font::engine);
  if(error != 0)
    {
      cerr << "Font: unable to initialize TrueType Engine"<< endl;
      assert(false);
    }

  //open font
  error = TT_Open_Face(Font::engine, fontfilename.c_str(), &face_);
  if(error != 0)
    {
      cerr << "Font: unable to open font \""<<fontfilename<<"\""<< endl;

      // free the stuff we create sofar...
      //  debugN(17,cerr << "Font: deleting palette_ " << endl;);
      if(palette_)
	delete[] palette_;
      //  debugN(17,cerr << "Font: freeing engine." << endl;);
      TT_Done_FreeType(Font::engine);

      return;
    }

  //search unicode mapping
  error = TT_Get_Face_Properties(face_, &properties_);
  int i=0;
  while( i < properties_.num_CharMaps)
    {
      TT_Get_CharMap_ID(face_, (TT_UShort)i, &platform_, &encoding_);
      if ((platform_ == 3 && encoding_ == 1) ||
	  (platform_ == 0 && encoding_ == 0) ) 
	{
	  break;
	}
      i++;
    }
  
  if(i == properties_.num_CharMaps)
    {
      cerr << "Font: Warning: font \"" << fontfilename 
	   << "\" has no unicode mapping. Text may be scrambled." << endl;
      //valid fallback ?
      i=0;
    }
  
  //load unicode-mapped-charmap
  TT_Get_CharMap(face_, i, &cmap_);


  //Glyph
  error=TT_New_Glyph(face_, &glyph_);
  if (error) 
    {
      cerr << "Font: unable to creat glyph." << endl;
      //  debugN(17,cerr << "Font: deleting palette_ " << endl;);
      if(palette_)
	delete[] palette_;
      //  debugN(17,cerr << "Font: TT_Done_Glyph()" << endl;);
      TT_Done_Glyph(glyph_);
      //  debugN(17,cerr << "Font: TT_Close_Face()" << endl;);
      TT_Close_Face(face_);
      //  debugN(17,cerr << "Font: freeing engine." << endl;);
      TT_Done_FreeType(Font::engine);
      
      return;
    }  
 
  //create instance
  error = TT_New_Instance(face_, &instance_);
  if(error != 0)
    {
      cerr << "Font: unable to create instance." << endl;
      //  debugN(17,cerr << "Font: deleting palette_ " << endl;);
      if(palette_)
	delete[] palette_;
      //  debugN(17,cerr << "Font: TT_Done_Instance()" << endl;);
      TT_Done_Instance(instance_);
      //  debugN(17,cerr << "Font: TT_Done_Glyph()" << endl;);
      TT_Done_Glyph(glyph_);
      //  debugN(17,cerr << "Font: TT_Close_Face()" << endl;);
      TT_Close_Face(face_);
      //  debugN(17,cerr << "Font: freeing engine." << endl;);
      TT_Done_FreeType(Font::engine);
      
      return;
    }
  
  //Resolution
  error=TT_Set_Instance_Resolutions(instance_, 72, 72);
  if (error != 0) 
    {
      cerr << "Font: unable to set resolution." << endl;
      
            //  debugN(17,cerr << "Font: deleting palette_ " << endl;);
      if(palette_)
	delete[] palette_;
      //  debugN(17,cerr << "Font: TT_Done_Instance()" << endl;);
      TT_Done_Instance(instance_);
      //  debugN(17,cerr << "Font: TT_Done_Glyph()" << endl;);
      TT_Done_Glyph(glyph_);
      //  debugN(17,cerr << "Font: TT_Close_Face()" << endl;);
      TT_Close_Face(face_);
      //  debugN(17,cerr << "Font: freeing engine." << endl;);
      TT_Done_FreeType(Font::engine);
      
      return;
    }      

  //Charsize
  error=TT_Set_Instance_CharSize(instance_, ptsize_*64);
  if (error != 0) 
    {
      cerr << "Font: unable to set fontsize." << endl;
            //  debugN(17,cerr << "Font: deleting palette_ " << endl;);
      if(palette_)
	delete[] palette_;
      //  debugN(17,cerr << "Font: TT_Done_Instance()" << endl;);
      TT_Done_Instance(instance_);
      //  debugN(17,cerr << "Font: TT_Done_Glyph()" << endl;);
      TT_Done_Glyph(glyph_);
      //  debugN(17,cerr << "Font: TT_Close_Face()" << endl;);
      TT_Close_Face(face_);
     
      //  debugN(17,cerr << "Font: freeing engine." << endl;);
      TT_Done_FreeType(Font::engine);
      
      return;
    }
  
  /* Create a scratch pixmap for this font */
  error = TT_Get_Instance_Metrics(instance_, &imetrics_);
  if(error != 0)
    {
      cerr << "Font: problems determining the font metrics.";
      cerr << "Font may be unreadable." << endl;
    }
  rastermap_.rows = imetrics_.y_ppem + 32;
  rastermap_.width = imetrics_.x_ppem + 32;
  rastermap_.flow = TT_Flow_Up;
  rastermap_.cols = (rastermap_.width+3) & -4;
  rastermap_.size = rastermap_.rows*rastermap_.cols;
  rastermap_.bitmap = malloc((int)rastermap_.size);
  if ( rastermap_.bitmap == 0 ) 
    {
      cerr << "Font: out of memory" << endl;
            //  debugN(17,cerr << "Font: deleting palette_ " << endl;);
      if(palette_)
	delete[] palette_;
      //  debugN(17,cerr << "Font: TT_Done_Instance()" << endl;);
      TT_Done_Instance(instance_);
      //  debugN(17,cerr << "Font: TT_Done_Glyph()" << endl;);
      TT_Done_Glyph(glyph_);
      //  debugN(17,cerr << "Font: TT_Close_Face()" << endl;);
      TT_Close_Face(face_); 
      //  debugN(17,cerr << "Font: freeing engine." << endl;);
      TT_Done_FreeType(Font::engine);
      return;
    }  

  createPalette();

  
  createFont();

  

  //  debugN(17,cerr << "Font: TT_Done_Instance()" << endl;);
  TT_Done_Instance(instance_);
  //  debugN(17,cerr << "Font: TT_Done_Glyph()" << endl;);
  TT_Done_Glyph(glyph_);
  //  debugN(17,cerr << "Font: TT_Close_Face()" << endl;);
  TT_Close_Face(face_);
  
  //  debugN(17,cerr << "Font: freeing rastermap." << endl;);
  if(rastermap_.bitmap)
    free(rastermap_.bitmap);

  //  debugN(17,cerr << "Font: freeing engine." << endl;);
  TT_Done_FreeType(Font::engine);
    
  initialized_ = true;

}


Font::~Font()
{

  if(initialized_)
    {
      
      //  debugN(17,cerr << "Font: deleting palette_ " << endl;);
      if(palette_)
	delete[] palette_;
      //      debugN(17,cerr << "Font: deleting bitmapTable_ " << endl;);
      if(surfaceTable_)
	for(int i=0; i < 128; i++)
	  delete surfaceTable_[i];
      
      if(surfaceTable_ != 0)
	delete [] surfaceTable_;

    }

}


void
Font::createPalette()
{

  int rdiff = fontColor_.r - backColor_.r;
  int gdiff = fontColor_.g - backColor_.g;
  int bdiff = fontColor_.b - backColor_.b;
  int adiff = fontColor_.a - backColor_.a;

  int r = backColor_.r;
  int g = backColor_.g;
  int b = backColor_.b;
  int a = backColor_.a;

  palette_[0] = transparent;

  for(unsigned char num=1; num < 5; num++)
    palette_[num]= Color(r + num * rdiff/4,
			 g + num * gdiff/4,
			 b + num * bdiff/4,
			 a + num * adiff/4);


  
}


Surface*
Font::renderChar(unsigned char c)
{

  int xoffset, yoffset;
  int miny = 0;
  int maxy = 0;  
  int minx = 0;
  int maxx = 0;

  Surface *surface;

  TT_UShort mapped;
  TT_Error error;
  TT_Glyph_Metrics metric;  

  //mapp unsigned char to font index
  mapped = TT_Char_Index(cmap_, (TT_UShort)c);

  //get glyph
  error = TT_Load_Glyph(instance_, glyph_, mapped,
			TTLOAD_DEFAULT);
  if(error != 0)
    {
      cerr << "Font: no Glyph available for '" << c << "'. " << endl;
    }

  //get the size of the character
  TT_Get_Glyph_Metrics(glyph_, &metric); 

  // the bounding box
  miny = metric.bbox.yMin;
  maxy = metric.bbox.yMax;
  minx = metric.bbox.xMin;
  maxx = metric.bbox.xMax;
 
  //if some "descending" characters like "g" or "y" get clipped
  //the 1.25 could be increased....
  // it is in fact just a rough guess....
  int h = (int)(imetrics_.y_ppem * 1.25);
  
  // dont use the width of the bounding box, use the advance metric
  // in order to allow alliterations and stuff
  int w = (metric.advance+32)  / 64;
  

  unsigned int descent = 0;

  //does character go below baseline
  if(miny < 0)
    {
      descent = abs(miny)/64;
//       cerr << (char) c << ":" << descent << "     ";
    }

  //in order to be on the safe side
  if((w > 0) && ( h >0 ))
    {

        
      //create target surface
      surface = new Surface(w , h, Pixelformat::IND8);

      //create painter
      Painter painter = surface;

  

      vector<Color> palette;
      //set palette of surface
      for(int ind=0; ind<5; ind++)
	palette.push_back(palette_[ind]);

  
      surface->setPalette(palette);
      surface->setTransColor(black);
      surface->setTransparency(true);

      //clear scratch surface
      memset(rastermap_.bitmap, 0, rastermap_.size);
      
      xoffset = abs(minx);
      yoffset = abs(miny);  
      
      TT_Get_Glyph_Pixmap(glyph_, &rastermap_, xoffset, yoffset);
      
      if(w > rastermap_.cols)
	w = rastermap_.cols;
      if(h > rastermap_.rows)
	h = rastermap_.rows;
      
      char *srcbuf;
      
      for(int y=0 ; y < imetrics_.y_ppem ; y++)
	{
	  srcbuf = (char*)rastermap_.bitmap+(imetrics_.y_ppem-1-y)*rastermap_.cols;
	  for(int x=0; x < w; x++)
	    {
	      srcbuf++;

	      //use setPixel to allow surface to have any pixelformat...
	      painter.setPixel(Point(x,y+descent), palette_[*srcbuf]);
	      
	      // dumo the character to stderr
	      //  	  if(*srcbuf != 0)
	      //  	    cerr << "#";
	      //  	  else
	      //  	    cerr << ".";
	      
	    }
	  //       cerr << endl;
	}

  
    }
  else
    {
      //if something went wrong return an empty surface:
      surface = new Surface();
    }

  
  return surface;
}


void
Font::createFont()
{
  if(surfaceTable_ != 0)
    delete [] surfaceTable_;

  surfaceTable_ = new Surface*[128];

  for(unsigned char c = 0; c < 128; c++)
    {
      surfaceTable_[c] = renderChar(c);
      //"calculate" maximum height for this font
      if(surfaceTable_[c]->height() > height_)
	height_ = surfaceTable_[c]->height();
    }
}

Surface *
Font::getChar(unsigned char c) const
{
  if(surfaceTable_)
    {
      Surface* surf = new Surface(*getCharPtr(c));
      surf->setTransColor(black);
      surf->setTransparency(true);
      return surf;
    }
  return new Surface();
}

Surface*
Font::getCharPtr(unsigned char c) const
{
  unsigned char  c_;

  //" converting ascii to unicode "
  if(c > 127)
    c_ = ' ';
  else
    c_ = c;

  return surfaceTable_[c_];
}

Point
Font::getSize(unsigned char c) const
{
  if(surfaceTable_ == 0)
    return Point(0,0);
  else
    return Point(getCharPtr(c)->width(), getCharPtr(c)->height());
}

  
void 
Font::blitString(const string& txt, Surface* target, const Point& p) const
{
  if(!target)
    return;
  
  int xoffset = p.x;
  int yoffset;

  Rect dest;

  for(unsigned i=0; i<txt.size(); i++)
    {

      yoffset = p.y + getHeight() - getCharPtr(txt[i])->height();
      //get bitmaps bounding box
      dest = 
	Rect(0,0,getCharPtr(txt[i])->width(), getCharPtr(txt[i])->height());
      //move destination
      dest.translate(xoffset, yoffset);
      //blit
      getCharPtr(txt[i])->blit(target, dest);
      //
      xoffset += getCharPtr(txt[i])->width();
    }
}

Surface *
Font::getString(const string &txt) const
{

  if(surfaceTable_ == 0)
    return new Surface();

  if(txt.empty())
    {
      return new Surface();
    }

  
  int h=0;
  int w=0;
//   int xoffset = 0;
//   int yoffset = 0;

  unsigned int i;
  unsigned char c;

  for(i=0; i<txt.size(); i++)
    {
      c = txt[i];

      assert(getCharPtr(c) != 0);
    
      if(getCharPtr(c)->height() > h)
	h = getCharPtr(c)->height();
      
      w += getCharPtr(c)->width();
	
    }

  //debugN(17,cerr<<"Font: stringsurface is "<<w<<"x"<< h<< " pixels."<<endl;);

  //failsafe:
  if(w==0)
    w=2;
  if(h==0)
    h=2;

  //create target surface  
  Surface *textSurface = new Surface(w,h,Pixelformat::IND8);

  vector<Color> palette;
  //set palette of surface
  for(int ind=0; ind<5; ind++)
    palette.push_back(palette_[ind]);
  textSurface->setPalette(palette);

  textSurface->setTransColor(black);
  textSurface->setTransparency(true);

  textSurface->fill(black);

  blitString(txt,textSurface,Point(0,0));

  return textSurface;
}
		     

  
}

