/*
 * FreeBSD USB support
 *
 * Derived from Linux version by Richard Tobin.
 *
 * This library is covered by the LGPL.
 */

/*
 * Note: I don't have a clue what I'm doing.  I just looked at the
 * man pages and source to try and find things that did the same as
 * the Linux version. -- Richard
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h> 
#include <fcntl.h> 
#include <errno.h> 
#include <assert.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#include <dev/usb/usb.h>

#include "usbi.h"

static int ensure_ep_open(usb_dev_handle *dev, int ep, int mode);

#define MAX_CONTROLLERS 10

/* This records the file descriptors for the endpoints.  It doesn't seem
   to work to re-open them for each read (as well as being inefficient). */

struct freebsd_usb_dev_handle_info {
    int ep_fd[USB_MAX_ENDPOINTS];
};

int usb_os_open(usb_dev_handle *dev)
{
    int i;
    struct freebsd_usb_dev_handle_info *info;

    info = malloc(sizeof(*info));
    if (!info)
	USB_ERROR(-ENOMEM);
    dev->impl_info = info;

    dev->fd = open(dev->device->filename, O_RDWR);
    if (dev->fd < 0) 
    {
	dev->fd = open(dev->device->filename, O_RDONLY);
	if (dev->fd < 0)
	{
	    free(info);
	    USB_ERROR_STR(errno, "failed to open %s: %s",
			  dev->device->filename, strerror(errno));
	}
    }

    /* Mark the endpoints as not yet open */
    for(i=0; i<USB_MAX_ENDPOINTS; i++)
	info->ep_fd[i] = -1;

    return 0;
}

int usb_os_close(usb_dev_handle *dev)
{
    struct freebsd_usb_dev_handle_info *info = dev->impl_info;
    int i;

    /* Close any open endpoints */

    for(i=0; i<USB_MAX_ENDPOINTS; i++)
	if(info->ep_fd[i] >= 0)
	{
	    fprintf(stderr, "closing endpoint %d\n", info->ep_fd[i]);
	    close(info->ep_fd[i]);
	}

    free(info);

    if (dev->fd <= 0)
	return 0;

    if (close(dev->fd) == -1)
	/* Failing trying to close a file really isn't an error, so return 0 */
	USB_ERROR_STR(0, "tried to close device fd %d: %s", dev->fd,
		      strerror(errno));

    return 0;
}

int usb_set_configuration(usb_dev_handle *dev, int configuration)
{
    int ret;

    ret = ioctl(dev->fd, USB_SET_CONFIG, &configuration);
    if (ret < 0)
	USB_ERROR_STR(ret, "could not set config %d: %s", configuration,
		      strerror(errno));

    dev->config = configuration;

    return 0;
}

int usb_claim_interface(usb_dev_handle *dev, int interface)
{
    /* FreeBSD doesn't have the corresponding ioctl.  It seems to
       be sufficient to open the relevant endpoints as needed. */

    dev->interface = interface;

    return 0;
}

int usb_release_interface(usb_dev_handle *dev, int interface)
{
    /* See above */
    return 0;
}

int usb_set_altinterface(usb_dev_handle *dev, int alternate)
{
    int ret;
    struct usb_alt_interface intf;

    if (dev->interface < 0)
	USB_ERROR(-EINVAL);

    intf.interface_index = dev->interface;
    intf.alt_no = alternate;

    ret = ioctl(dev->fd, USB_SET_ALTINTERFACE, &intf);
    if (ret < 0)
	USB_ERROR_STR(ret, "could not set alt intf %d/%d: %s",
		      dev->interface, alternate, strerror(errno));

    dev->altsetting = alternate;

    return 0;
}

static int ensure_ep_open(usb_dev_handle *dev, int ep, int mode)
{
    struct freebsd_usb_dev_handle_info *info = dev->impl_info;
    int fd;
    char buf[20];

    if(info->ep_fd[ep] < 0)
    {
	sprintf(buf, "%s.%d", dev->device->filename, ep);
	fd = open(buf, O_RDONLY);
	if(fd < 0)
	    USB_ERROR_STR(fd, "can't open %s for bulk read: %s\n",
			  buf, strerror(errno));
	info->ep_fd[ep] = fd;
    }

    return info->ep_fd[ep];
}

int usb_bulk_write(usb_dev_handle *dev, int ep, char *bytes, int size,
	int timeout)
{
    int fd, ret, sent = 0;

    ep &= 0x7f;			/* XXX why is the high bit set??? */

    fd = ensure_ep_open(dev, ep, O_WRONLY);
    if(fd < 0)
	return fd;

    ret = ioctl(fd, USB_SET_TIMEOUT, &timeout);
    if (ret < 0)
	USB_ERROR_STR(ret, "error setting timeout: %s",
		      strerror(errno));
    do {
	ret = write(fd, bytes+sent, size-sent);
	if(ret < 0)
	    USB_ERROR_STR(ret, "error writing to bulk endpoint %s.%d: %s",
			  dev->device->filename, ep, strerror(errno));
	sent += ret;
    } while(ret > 0 && sent < size);

    return sent;
}

int usb_bulk_read(usb_dev_handle *dev, int ep, char *bytes, int size,
	int timeout)
{
    int fd, ret, retrieved = 0;

    ep &= 0x7f;			/* XXX why is the high bit set??? */

    fd = ensure_ep_open(dev, ep, O_RDONLY);
    if(fd < 0)
	return fd;

    ret = ioctl(fd, USB_SET_TIMEOUT, &timeout);
    if (ret < 0)
	USB_ERROR_STR(ret, "error setting timeout: %s",
		      strerror(errno));
    do
    {
	ret = read(fd, bytes+retrieved, size-retrieved);
	if(ret < 0)
	    USB_ERROR_STR(ret, "error reading from bulk endpoint %s.%d: %s",
			  dev->device->filename, ep, strerror(errno));
	retrieved += ret;
    } while(ret > 0 && retrieved < size);

    return retrieved;
}

int usb_control_msg(usb_dev_handle *dev, int requesttype, int request,
	int value, int index, char *bytes, int size, int timeout)
{
    struct usb_ctl_request req;
    int ret;

    fprintf(stderr, "usb_control_msg %d %d %d %d %p %d %d\n",
	    requesttype, request, value, index, bytes, size, timeout);

    req.request.bmRequestType = requesttype;
    req.request.bRequest = request;
    USETW(req.request.wValue, value);
    USETW(req.request.wIndex, index);
    USETW(req.request.wLength, size);

    req.data = bytes;
    req.flags = 0;

    ret = ioctl(dev->fd, USB_SET_TIMEOUT, &timeout);
    if (ret < 0)
	USB_ERROR_STR(ret, "error setting timeout: %s",
		      strerror(errno));

    ret = ioctl(dev->fd, USB_DO_REQUEST, &req);
    if (ret < 0)
	USB_ERROR_STR(ret, "error sending control message: %s",
		      strerror(errno));

    return 0;
}

int usb_find_devices_on_bus(struct usb_bus *bus)
{
    int cfd, dfd;
    int device;

    cfd = open(bus->dirname, O_RDONLY);
    if(cfd < 0)
	USB_ERROR_STR(errno, "couldn't open(%s): %s", bus->dirname,
		      strerror(errno));

    for(device = 1; device < USB_MAX_DEVICES; device++)
    {
	struct usb_device_info di;
	struct usb_device *dev;
	char buf[20];

	di.addr = device;
	if(ioctl(cfd, USB_DEVICEINFO, &di) < 0)
	    continue;

#if 0
	{
	    int j;
	    printf("something on %d\n", device);
	    printf("vendor %s product %s\n", di.vendor, di.product);
	    for(j=0; j<4; j++)
		printf("name %s\n", di.devnames[j]);
	}
#endif

	/* There's a device; is it one we should mess with? */

	if(strncmp(di.devnames[0], "ugen", 4) != 0)
	    /* best not to play with things we don't understand */
	    continue;

	/* Open its control endpoint */

	sprintf(buf, "/dev/%s", di.devnames[0]);
	dfd = open(buf, O_RDONLY);
	if(dfd < 0)
	{
	    fprintf(stderr, "couldn't open device %s: %s\n",
		    di.devnames[0], strerror(errno));
	    continue;
	}

	dev = malloc(sizeof(*dev));
	if (!dev)
	    USB_ERROR(-ENOMEM);

	memset((void *)dev, 0, sizeof(*dev));

	dev->bus = bus;

	strcpy(dev->filename, buf);

	if(ioctl(dfd, USB_GET_DEVICE_DESC, &dev->descriptor) < 0)
	    USB_ERROR_STR(-errno, "couldn't get device descriptor for %s: %s",
			  buf, strerror(errno));

	close(dfd);

	if (bus->devices) {
	    dev->next = bus->devices;
	    dev->next->prev = dev;
	}
	bus->devices = dev;

	if (usb_debug >= 2)
	    fprintf(stderr, "usb_find_devices_on_bus: Found %s on %s\n",
		    dev->filename, bus->dirname);
    }

    close(cfd);

    return 0;
}

int usb_find_busses(void)
{
    int controller;
    int fd;
    char buf[20];

    for(controller = 0; controller < MAX_CONTROLLERS; controller++)
    {
	struct usb_bus *bus;

	sprintf(buf, "/dev/usb%d", controller);
	fd = open(buf, O_RDONLY);
	if(fd < 0)
	{
	    if(errno != ENXIO && errno != ENOENT)
		fprintf(stderr, "usb_find_busses: can't open %s: %s\n",
			buf, strerror(errno));
	    continue;
	}

	close(fd);

	bus = malloc(sizeof(*bus));
	if (!bus)
	    USB_ERROR(-ENOMEM);

	memset((void *)bus, 0, sizeof(*bus));

	strcpy(bus->dirname, buf);

	if (usb_busses) {
	    bus->next = usb_busses;
	    bus->next->prev = bus;
	}
	usb_busses = bus;

	if (usb_debug >= 2)
	    fprintf(stderr, "usb_find_busses: Found %s\n", bus->dirname);
    }

    return 0;
}

void usb_os_init(void)
{
    /* nothing */
}

int usb_resetep (usb_dev_handle *dev, unsigned int ep)
{
    /* Not yet done, because I haven't needed it. */

    fprintf(stderr, "usb_resetep  called, exiting\n");
    exit(1);
}

int usb_clear_halt (usb_dev_handle *dev, unsigned int ep)
{
    /* Not yet done, because I haven't needed it. */

    fprintf(stderr, "usb_clear_halt  called, exiting\n");
    exit(1);
}

int usb_reset (usb_dev_handle *dev)
{
    /* Not yet done, because I haven't needed it. */

    fprintf(stderr, "usb_reset  called, exiting\n");
    exit(1);
}

