/*

Copyright (C) 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>

#include <libnd.h>
#include <protocols/tcp/libnd_tcp.h>
#include <protocols/udp/libnd_udp.h>

#include <libnd_demux.h>
#include <libnd_demux_progress.h>

#ifndef RLIMIT_OFILE
#define RLIMIT_OFILE RLIMIT_NOFILE
#endif

#define LND_CONN_IDLE_LIM 3600
#define LND_DEMUX_QUEUE_MAX 10000

static LND_Protocol *ip;
static LND_Protocol *tcp;
static LND_Protocol *udp;
static mode_t        mode_640 = S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP;

struct demux_packet_queue
{
  LND_Packet *q;
  LND_Packet *q_end;
};

/* For debugging purposes when running out of fds
static guint
demux_get_fdcount(void)
{
  char procdir[MAXPATHLEN];
  struct dirent *dirent;
  DIR *dir;
  guint fd_count = 0;

  g_snprintf(procdir, MAXPATHLEN, "/proc/%u/fd", getpid());
  
  if (! (dir = opendir(procdir)))
    {
      printf("Couldn't open %s\n", procdir);
      return 0;
    }
  
  while ( (dirent = readdir(dir)))
    fd_count++;
  
  closedir(dir);
  
  return fd_count - 2;
} */


static char *
demux_create_dir(LND_Demux *dm, LND_ConnID *conn)
{
  char dir[MAXPATHLEN];
  char src[32], dst[32];
  struct in_addr ip_src, ip_dst;

  if (!conn)
    {
      D(("Invalid input\n"));
      return NULL;
    }

  libnd_conn_get_src(conn, &ip_src, NULL);
  libnd_conn_get_dst(conn, &ip_dst, NULL);

  g_snprintf(src, 32, "%s", inet_ntoa(ip_src));
  g_snprintf(dst, 32, "%s", inet_ntoa(ip_dst));

  g_snprintf(dir, MAXPATHLEN, "%s/%u", dm->output_dir, conn->proto);
  if (mkdir(dir, mode_640) < 0)
    {
      if (errno != EEXIST)
	{
	  D(("Could not create output directory %s: %s\n", dir, strerror(errno)));
	  return NULL;
	}
    }

  g_snprintf(dir, MAXPATHLEN, "%s/%u/%s", dm->output_dir, conn->proto, src);
  if (mkdir(dir, mode_640) < 0)
    {
      if (errno != EEXIST)
	{
	  D(("Could not create output directory %s: %s\n", dir, strerror(errno)));
	  return NULL;
	}
    }

  g_snprintf(dir, MAXPATHLEN, "%s/%u/%s/%s", dm->output_dir, conn->proto, src, dst);
  if (mkdir(dir, mode_640) < 0)
    {
      if (errno != EEXIST)
	{
	  D(("Could not create output directory %s: %s\n", dir, strerror(errno)));
	  return NULL;
	}
    }
  
  return g_strdup(dir);
}

static const char *
demux_get_conn_dumper_name(LND_ConnID *conn)
{
  static char name[MAXPATHLEN];

  switch (conn->proto)
    {
    case IPPROTO_TCP:
    case IPPROTO_UDP:
      {
	guint16 sport, dport;

	libnd_conn_get_src(conn, NULL, &sport);
	libnd_conn_get_dst(conn, NULL, &dport);
	
	g_snprintf(name, MAXPATHLEN, "%lu.%lu-%u-%u.trace",
		   conn->start_ts.tv_sec, conn->start_ts.tv_usec,
		   ntohs(sport), ntohs(dport));
      }
      break;

    default:
      g_snprintf(name, MAXPATHLEN, "%lu.%lu.trace",
		 conn->start_ts.tv_sec, conn->start_ts.tv_usec);
    }


  return name;
}


static void
demux_conn_queue_add_packet(LND_Demux *dm, LND_ConnID *conn)
{
  struct demux_packet_queue *queue;
  LND_Packet *p_copy;

  if (! (queue = libnd_conn_data_get(conn, "queue")))
    {
      if (! (queue = g_new0(struct demux_packet_queue, 1)))
	{
	  D(("Out of memory.\n"));
	  return;
	}

      libnd_conn_data_set(conn, "queue", queue);
    }

  if (! (p_copy = libnd_packet_duplicate(dm->current)))
    {
      D(("Out of memory.\n"));
      return;
    }
  
  if (! queue->q_end)
    {
      queue->q = p_copy;
      queue->q_end = p_copy;
    }
  else
    {
      queue->q_end->next = p_copy;
      p_copy->prev = queue->q_end;
      queue->q_end = p_copy;
    }
}

static void
demux_conn_queue_flush(LND_Demux *dm, LND_ConnID *conn, struct demux_packet_queue *queue)
{
  char dumper_name[MAXPATHLEN];
  pcap_dumper_t *dumper;
  char *dir;
  LND_Packet *packet, *p_next;

  if ( (packet = queue->q))
    {    
      dir = libnd_conn_data_get(conn, "dir");
      D_ASSERT_PTR(dir);
      
      g_snprintf(dumper_name, MAXPATHLEN, "%s/%s",
		 dir, demux_get_conn_dumper_name(conn));
      
      if (! (dumper = pcapnav_dump_open(dm->pcap, dumper_name,
					PCAPNAV_DUMP_APPEND_FAST)))
	{
	  D(("Cannot open dump: %s\n", pcap_geterr(dm->pcap)));
      return;
	}
      
      while (packet)
	{
	  pcap_dump((u_char *) dumper, &packet->ph, packet->data);
	  p_next = packet->next;
	  libnd_packet_free(packet);
	  packet = p_next;
	}

      pcap_dump_close(dumper);
    }

  g_free(queue);
}


static gboolean
demux_conn_cleanup_cb(LND_ConnID *conn, LND_Demux *dm)
{
  struct demux_packet_queue *queue;
  char *dir;

  if ( (queue = libnd_conn_data_remove(conn, "queue")))
    demux_conn_queue_flush(dm, conn, queue);

  /* Check if connection is now dead, and if so,
   * remove it from the connection table.
   */
  if (libnd_conn_is_dead(conn, dm->current))
    {
      LND_ConnID *conn_result;
      
      if ( (dir = libnd_conn_data_remove(conn, "dir")))
	g_free(dir);
      
      conn_result = libnd_conn_table_remove(dm->conns, conn);
     
      /* There is a possibilty of more than one connection in 
       * the table matching this connection -- to prevent segfaults
       * make sure we only nuke the one we mean.
       */
      if (conn_result == conn)
	libnd_conn_free(conn);
    }
  
  return TRUE;
}

struct fat_table
{
  LND_Demux *dm;
  struct bpf_timeval tv;
  LND_ConnID  *conn;
};

/*
static gboolean
demux_show_age(LND_ConnID *conn, struct fat_table *fat)
{
  struct bpf_timeval tv_diff;

  pcapnav_timeval_sub(&fat->dm->current->ph.ts, &conn->latest_ts, &tv_diff);

  if (pcapnav_timeval_cmp(&tv_diff, &fat->tv) > 0)
    {
      fat->tv = tv_diff;
      fat->conn = conn;
    }

  return TRUE;
}
*/
static void
demux_conn_table_cleanup(LND_Demux *dm)
{
  /*
  struct fat_table fat;
  char connstr[256];
  */
  if (! dm)
    return;

  libnd_conn_table_set_policy(dm->conns, LND_CONN_TABLE_INCLUDE_DEAD);
  /* printf("Size before: %u\n", libnd_conn_table_size(dm->conns)); */
  libnd_conn_table_foreach(dm->conns,
			   (LND_ConnFunc) demux_conn_cleanup_cb, dm);
  /* printf("Size after: %u\n", libnd_conn_table_size(dm->conns)); */

  /* Age check 

  fat.dm = dm;
  fat.tv.tv_sec =0;
  fat.tv.tv_usec =0;
  fat.conn = NULL;

  libnd_conn_table_foreach(dm->conns,
			   (LND_ConnFunc) demux_show_age, &fat);
  libnd_conn_to_string(fat.conn, connstr, 256);
  printf("Oldest conn: %s, %u.%u\n", connstr, fat.tv.tv_sec, fat.tv.tv_usec);
  */

  libnd_conn_table_set_policy(dm->conns, LND_CONN_TABLE_IGNORE_DEAD);
}


static void
demux_dump_non_ip(LND_Demux *dm, LND_Packet *packet)
{
  if (!dm || !dm->pcap || !dm->others_file)
    {
      D(("Invalid input.\n"));
      return;
    }
  
  if (! dm->others_dumper)
    {
      if (! (dm->others_dumper = pcapnav_dump_open(dm->pcap, dm->others_file,
						   PCAPNAV_DUMP_APPEND_FAST)))
	{
	  D(("Cannot create output for non-IP traffic: %s\n", pcap_geterr(dm->pcap)));
	  return;
	}
    }
  
  pcap_dump((u_char *) dm->others_dumper, &packet->ph, packet->data);
}


static gboolean
dm_demux_cb(LND_TraceSet *set, LND_Trace *trace, LND_Demux *dm)
{
  LND_PacketIterator  pit;
  LND_Packet *packet;
  LND_ConnID *conn;
  char *dir;
  
  libnd_demux_prog_set_file(trace->filename);
  dm->pcap = pcapnav_pcap(trace->tpm->base->pcn);
  
  /* Iterate over entire individual trace, read-only. */
  for (libnd_pit_init_mode(&pit, trace, LND_PACKET_IT_AREA_R);
       libnd_pit_get(&pit);
       libnd_pit_next(&pit), dm->iter++)
    {
      packet = libnd_pit_get(&pit);
      dm->current = packet;

      /* First check if we have connection state for this packet.
       * If not, create and register.
       */
      if (! (conn = libnd_conn_table_lookup(dm->conns, packet)))
	{
	  if (! (conn = libnd_conn_new(packet)))
	    {
	      demux_dump_non_ip(dm, packet);
	      continue;
	    }

	  libnd_conn_table_add(dm->conns, conn);

	  if (! (dir = demux_create_dir(dm, conn)))
	    {
	      D(("Error creating output directories for %p %p\n", dm, conn));
	      dm->current = NULL;
	      return FALSE;
	    }

	  libnd_conn_data_set(conn, "dir", dir);
	}
      
      /* Now, add the packet to the connection's output queue */
      demux_conn_queue_add_packet(dm, conn);

      /* Update the state of the current connection: */
      libnd_conn_update(conn, packet);

      /* Once we hit the output threshold, go through all connections
       * in the table and write packets to their corresponding dumpers.
       * At the same time, remove dead connections from the table.
       */
      if (dm->iter >= dm->iter_lim)
	{
	  demux_conn_table_cleanup(dm);
	  dm->iter = 0;
	}
    }

  /* Flush one more time at the end of the iteration to make sure
   * we write out everything.
   */
  dm->current = NULL;
  demux_conn_table_cleanup(dm);

  return TRUE;
  TOUCH(set);
}




LND_Demux *
libnd_demux_new(void)
{
  LND_Demux *dm;

  if (! (dm = g_new0(LND_Demux, 1)))
    return NULL;
  
  if (! (dm->traces = libnd_traceset_new()))
    {
      D(("out of memory"));
      g_free(dm);
      return NULL;
    }

  dm->iter = 0;
  dm->iter_lim = LND_DEMUX_QUEUE_MAX;

  dm->conns = libnd_conn_table_new(LND_CONN_TABLE_IGNORE_DEAD);
  dm->output_dir = strdup(".");
  
  return dm;
}


static gboolean
demux_free_cb(LND_ConnID *conn, void *user_data)
{
  char *dir = libnd_conn_data_remove(conn, "dir");

  if (dir)
    g_free(dir);

  return TRUE;
  TOUCH(user_data);
}


void
libnd_demux_free(LND_Demux *dm)
{
  if (! dm)
    return;

  fflush(stdout);

  libnd_traceset_free(dm->traces);

  libnd_conn_table_set_policy(dm->conns, LND_CONN_TABLE_INCLUDE_DEAD);
  libnd_conn_table_foreach(dm->conns, (LND_ConnFunc) demux_conn_cleanup_cb, dm);
  libnd_conn_table_foreach(dm->conns, demux_free_cb, NULL);
  libnd_conn_table_free(dm->conns);

  g_free(dm->output_dir);
  g_free(dm->others_file);
  
  if (dm->others_dumper)
    pcap_dump_close(dm->others_dumper);
}


gboolean
libnd_demux_set_output_dir(LND_Demux *dm, const char *output_dir)
{
  char others_file[MAXPATHLEN];

  if (! dm || !output_dir || !*output_dir)
    return FALSE;

  if (!libnd_misc_is_dir(output_dir) ||
      !libnd_misc_can_write(output_dir))
    return FALSE;

  g_free(dm->output_dir);
  dm->output_dir = g_strdup(output_dir);

  while (dm->output_dir[strlen(dm->output_dir)-1] == G_DIR_SEPARATOR)
    dm->output_dir[strlen(dm->output_dir)-1] = '\0';

  g_free(dm->others_file);
  g_snprintf(others_file, MAXPATHLEN, "%s/non_ip.trace", dm->output_dir);
  dm->others_file = g_strdup(others_file);
    
  return TRUE;
}

void
libnd_demux(LND_Demux *dm)
{
  if (!dm)
    return;

  /* Iterate over all input traces and demux flows. */
  libnd_traceset_foreach(dm->traces, (LND_TraceSetCB) dm_demux_cb, dm);
}


const char *
name(void)
{
  return "Demux";
}

const char *
author(void)
{
  return "Christian Kreibich <christian.kreibich-AT-cl.cam.ac.uk>";
}

const char *
version(void)
{
  return VERSION;
}

gboolean
init(void)
{
  if (! libnd_plugin_find("Conntrack"))
    {
      D(("Connection tracker plugin not found\n"));
      return FALSE;
    }
  
  if (! libnd_plugin_find("Trace-Set"))
    {
      D(("Trace-Set plugin not found\n"));
      return FALSE;
    }

  if (! (ip = libnd_proto_registry_find(LND_PROTO_LAYER_NET, 0x0800)))
    {
      D(("IP protocol plugin not found.\n"));
      return FALSE;
    }

  if (! (tcp = libnd_proto_registry_find(LND_PROTO_LAYER_TRANS,
					 IPPROTO_TCP)))
    {
      D(("TCP protocol plugin not found.\n"));
      return FALSE;
    }

  if (! (udp = libnd_proto_registry_find(LND_PROTO_LAYER_TRANS,
					 IPPROTO_UDP)))
    {
      D(("UDP protocol plugin not found.\n"));
      return FALSE;
    }
  
  return TRUE;
}


static void
demux_help(void)
{
  printf("Flow Demultiplexer plugin\n"
	 "USAGE: lndtool -r demux [--debug|-d] [--output-dir|-o DIR] [--progress|-p]\n"
	 "               [--names-file|-f FILE] <TRACE1> [<TRACE2> <TRACE3> ...]\n"
	 "\n"
	 "  --help, -h, -?           This message.\n"
	 "  --output-dir, -o DIR     Output directory in which to dump output. Default: cwd.\n"
	 "  --progress|-p            Displays a progress indicator at the shell.\n"
	 "  --names-file|-f FILE     Take filenames of traces from flatfile FILE.\n");
}

gboolean
run(LND_Trace *trace, LND_PluginArgs *args)
{
  int i, counter = 0;
  LND_Demux *dm;

  if (!args)
    return FALSE;

  if (args->argc == 0)
    {
      printf("Please provide one or more traces to scan.\n");
      demux_help();
      return FALSE;
    }
  
  dm = libnd_demux_new();
  
  for (i = 0; i < args->argc; i++)
    {
      if (!strcmp(args->argv[i], "-h")     ||
	  !strcmp(args->argv[i], "--help") ||
	  !strcmp(args->argv[i], "-?"))
	{
	  demux_help();
	  return FALSE;
	}
      else if (!strcmp(args->argv[i], "-o") ||
	       !strcmp(args->argv[i], "--output-dir"))
	{
	  if (++i == args->argc)
	    {
	      printf("You need to pass an output directory to the --output-dir|-o option.\n");
	      return FALSE;
	    }

	  if (! libnd_demux_set_output_dir(dm, args->argv[i]))
	    {
	      printf("Cannot use %s as output directory: %s.\n", args->argv[i], strerror(errno));
	      return FALSE;
	    }
	}
      else if (!strcmp(args->argv[i], "-f") ||
	       !strcmp(args->argv[i], "--names-file"))
	{
	  if (++i == args->argc)
	    {
	      printf("You need to pass the name of a file that contains\n"
		     "a list of trace files, one filename per line, when\n"
		     "using the --names-file|-f option.\n");
	      return FALSE;
	    }
	  
	  printf("Adding file names listed in '%s'\n", args->argv[i]);
	  
	  if (! libnd_traceset_add_trace_name_list(dm->traces, args->argv[i]))
	    printf("Could not add all traces provided in '%s'\n", args->argv[i]);
	  else
	    counter++;
	}
      else if (!strcmp(args->argv[i], "-p") ||
	       !strcmp(args->argv[i], "--progress"))
	{
	  libnd_demux_prog_observer_add(dm);
	}
      else if (args->argv[i][0] != '-') /* Often these are just mistyped args */
	{
	  if (! libnd_traceset_add_trace_name(dm->traces, args->argv[i]))
	    printf("Trace file %s not added.\n", args->argv[i]);
	  else
	    counter++;
	}
    }

  if (counter > 0)
    libnd_demux(dm);

  libnd_demux_free(dm);
  
  return TRUE;
  TOUCH(trace);
}
