/*

Copyright (C) 2003 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <protocols/ip/libnd_ip.h>
#include <protocols/tcp/libnd_tcp.h>
#include <protocols/udp/libnd_udp.h>
#include "libnd_conntrack.h"

#define LND_TCP_MSL                  60
#define LND_TCP_TIMEOUT              5*60
#define LND_TCP_TOTAL_TIMEOUT        7500 /* Bit over two hours -- no TCP survives that */
#define LND_CONN_GENERIC_TIMEOUT     5*60 /* 5 minutes */

struct lnd_tcp_conn
{
  /* Connection definition -- IP addr + port quadruple,
   * protocol, plus arbitrary data storage facility:
   */
  LND_ConnID          conn;

  /* TCP connection status in state machine: */
  LND_TCPState        state;

  /* Sequence numbers we keep track of. */
  guint32             syn;
  guint32             syn_ack;
  guint32             fin;
  guint32             fin_back;

  /* Timestamp when we enter TIME_WAIT state.
   * This is used for error timeouts (after RSTs) as well as normal
   * timeouts in TIME_WAIT.
   */
  struct bpf_timeval  wait_ts;
  
  /* The timestamp at which we enter SHUTDOWN. */
  struct bpf_timeval  timeout_ts;

  /* Various flags for our connection: */
  guchar              fin_acked:1;
  guchar              fin_back_acked:1;
  guchar              handshake_seen:1;
};

struct lnd_udp_conn
{
  /* Connection definition -- IP addr + port quadruple: */
  LND_ConnID          conn;
};

struct lnd_ip_conn
{
  /* Connection definition -- IP addr pair + proto type. Port numbers unused. */
  LND_ConnID          conn;
};


/* String table to represent states in text */
static const char *tcp_state_strings[] = {
  "ERROR",
  "LISTEN",
  "CLOSED_NORMAL",
  "RST_WAIT",
  "CLOSED_RST",
  "SYN_SENT",
  "SYN_ACK_SENT",
  "ESTABLISHED",
  "SHUTDOWN",
  "TIME_WAIT",
  "CLOSED_TIMEOUT"
};

static void
conn_init(LND_ConnID *conn, const LND_Packet *packet, struct ip *iphdr)
{
  conn->proto = iphdr->ip_p;
  conn->start_ts = packet->ph.ts;
  conn->latest_ts = packet->ph.ts;
  
  conn->ip_src = iphdr->ip_src;
  conn->ip_dst = iphdr->ip_dst;
  conn->sport  = 0; /* n/a */
  conn->dport  = 0; /* n/a */
  conn->data   = g_hash_table_new(g_str_hash, g_str_equal);
}

LND_IPConn     *
libnd_ipconn_new(const LND_Packet *packet)
{
  struct ip *iphdr;
  LND_IPConn *ipc;
  LND_ProtoData *ipdata;

  if (! (ipc = g_new0(LND_IPConn, 1)))
    return NULL;
  if (! (ipdata = libnd_packet_get_proto_data(packet, libnd_ip_get(), 0)))
    {
      g_free(ipc);
      return NULL;
    }
  
  iphdr = (struct ip*) ipdata->data;
  conn_init(&ipc->conn, packet, iphdr);
  
  return ipc;
}


void            
libnd_ipconn_free(LND_IPConn *ipc)
{
  g_hash_table_destroy(ipc->conn.data);
  g_free(ipc);
}


LND_UDPConn    *
libnd_udpconn_new(const LND_Packet *packet)
{
  struct ip *iphdr;
  struct udphdr *udphdr;
  LND_UDPConn *udpc;

  if (! (udpc = g_new0(LND_UDPConn, 1)))
    return NULL;
  if (! libnd_udp_get_headers(packet, &iphdr, &udphdr))
    {
      g_free(udpc);
      return NULL;
    }
  
  conn_init(&udpc->conn, packet, iphdr);
  udpc->conn.sport  = udphdr->uh_sport;
  udpc->conn.dport  = udphdr->uh_dport;

  return udpc;
}


void            
libnd_udpconn_free(LND_UDPConn *udpc)
{
  g_hash_table_destroy(udpc->conn.data);
  g_free(udpc);
}


LND_TCPConn    *
libnd_tcpconn_new(const LND_Packet *packet)
{
  struct ip *iphdr;
  struct tcphdr *tcphdr;
  LND_TCPConn *tcpc;

  if (! (tcpc = g_new0(LND_TCPConn, 1)))
    return NULL;
  if (! libnd_tcp_get_headers(packet, &iphdr, &tcphdr))
    {
      g_free(tcpc);
      return NULL;
    }

  conn_init(&tcpc->conn, packet, iphdr);
  tcpc->conn.sport  = tcphdr->th_sport;
  tcpc->conn.dport  = tcphdr->th_dport;  
  tcpc->state = LND_TCP_LISTEN;

  return tcpc;
}


void            
libnd_tcpconn_free(LND_TCPConn *tcpc)
{
  g_hash_table_destroy(tcpc->conn.data);
  g_free(tcpc);
}


static void
tcpconn_state_goto_shutdown(LND_TCPConn *tcpc,
			    const LND_Packet *packet,
			    struct ip *iphdr,
			    struct tcphdr *tcphdr)
{
  gboolean forward_flow = (iphdr->ip_src.s_addr == tcpc->conn.ip_src.s_addr);
  
  if (forward_flow)
    tcpc->fin = ntohl(tcphdr->th_seq) + libnd_tcp_get_payload_length(iphdr, tcphdr);
  else
    tcpc->fin_back = ntohl(tcphdr->th_seq) + libnd_tcp_get_payload_length(iphdr, tcphdr);
  
  tcpc->state = LND_TCP_SHUTDOWN;
  tcpc->timeout_ts = packet->ph.ts;
}


gboolean
libnd_tcpconn_is_timeout(LND_TCPConn *tcpc, const LND_Packet *packet)
{
  LND_ConnID *conn = (LND_ConnID *) tcpc;
  struct bpf_timeval diff;

  if (!tcpc || !packet)
    return FALSE;

  pcapnav_timeval_sub(&packet->ph.ts, &conn->latest_ts, &diff);
	
  if (diff.tv_sec >= LND_TCP_TOTAL_TIMEOUT)
    {
      tcpc->state = LND_TCP_CLOSED_TIMEOUT;
      return TRUE;
    }

  pcapnav_timeval_sub(&packet->ph.ts, &tcpc->timeout_ts, &diff);

  if (tcpc->state == LND_TCP_SYN_SENT)
    {      
      if (diff.tv_sec >= LND_TCP_TIMEOUT)
	{
	  tcpc->state = LND_TCP_CLOSED_TIMEOUT;
	  return TRUE;
	}
    }
  
  if (tcpc->state == LND_TCP_SHUTDOWN)
    {
      if (diff.tv_sec >= LND_TCP_TIMEOUT)
	{
	  tcpc->state = LND_TCP_CLOSED_TIMEOUT;
	  return TRUE;
	}
    }
    
  pcapnav_timeval_sub(&packet->ph.ts, &tcpc->wait_ts, &diff);

  if (tcpc->state == LND_TCP_TIME_WAIT)
    {        
      if (diff.tv_sec >= 2 * LND_TCP_MSL)
	{
	  tcpc->state = LND_TCP_CLOSED_NORMAL;
	  return TRUE;
	}
    }
  
  if (tcpc->state == LND_TCP_RST_WAIT)
    {
      if (diff.tv_sec >= LND_TCP_MSL)
	{
	  tcpc->state = LND_TCP_CLOSED_RST;
	  return TRUE;
	}

      libnd_tcpconn_update(tcpc, packet);

      if (tcpc->state == LND_TCP_CLOSED_RST)
	return TRUE;
    }

  return FALSE;
}


void
libnd_tcpconn_update_time(LND_TCPConn *tcpc, const struct bpf_timeval *tv)
{
  struct bpf_timeval diff;
  LND_ConnID *conn = (LND_ConnID *) tcpc;

  if (!tcpc || !tv)
    return;

  /* First of all -- no matter what state we're in,
   * time this connection out if we haven't seen anything
   * for more than two hours. Normally after two hours
   * the keepalive option would request ACKs, so we take
   * silence after that time as an indication that we've
   * missed traffic or something else is wrong.
   */
  pcapnav_timeval_sub(tv, &conn->latest_ts, &diff);
	
  if (diff.tv_sec >= LND_TCP_TOTAL_TIMEOUT)
    {
      tcpc->state = LND_TCP_CLOSED_TIMEOUT;
      return;
    }
      
  switch (tcpc->state)
    {
    case LND_TCP_SYN_SENT:
      pcapnav_timeval_sub(tv, &tcpc->timeout_ts, &diff);
      
      if (diff.tv_sec >= LND_TCP_TIMEOUT)
	tcpc->state = LND_TCP_CLOSED_TIMEOUT;
      break;

    case LND_TCP_SYN_ACK_SENT:
      pcapnav_timeval_sub(tv, &tcpc->timeout_ts, &diff);
      
      if (diff.tv_sec >= LND_TCP_TIMEOUT)
	tcpc->state = LND_TCP_CLOSED_TIMEOUT;
      break;
      
    case LND_TCP_SHUTDOWN:
      pcapnav_timeval_sub(tv, &tcpc->timeout_ts, &diff);
      
      if (diff.tv_sec >= LND_TCP_TIMEOUT)
	tcpc->state = LND_TCP_CLOSED_TIMEOUT;
      break;

    case LND_TCP_TIME_WAIT:
      pcapnav_timeval_sub(tv, &tcpc->wait_ts, &diff);
      
      if (diff.tv_sec >= 2 * LND_TCP_MSL)
	tcpc->state = LND_TCP_CLOSED_NORMAL;	
      break;

    case LND_TCP_RST_WAIT:
      pcapnav_timeval_sub(tv, &tcpc->wait_ts, &diff);
      
      if (diff.tv_sec >= LND_TCP_MSL)
	tcpc->state = LND_TCP_CLOSED_RST;
      break;
      
    default:
      break;
    }
}


void            
libnd_tcpconn_update(LND_TCPConn *tcpc,
		     const LND_Packet *packet)
{
  struct ip *iphdr;
  struct tcphdr *tcphdr;  
  gboolean forward_flow;

  if (!tcpc || !packet)
    return;

  libnd_tcpconn_update_time(tcpc, &packet->ph.ts);
  
  if (! libnd_tcp_get_headers(packet, &iphdr, &tcphdr))
    return;
  
  forward_flow = (iphdr->ip_src.s_addr == tcpc->conn.ip_src.s_addr);
  
  if ((tcphdr->th_flags & TH_RST) &&
      tcpc->state != LND_TCP_RST_WAIT)      
    {
      tcpc->state   = LND_TCP_RST_WAIT;
      tcpc->wait_ts = packet->ph.ts;
      return;
    }
  
  switch (tcpc->state)
    {
    case LND_TCP_LISTEN:
      /* If we get a SYN, we jump straigt to ESTABLISHED if
       * we also have an ACK. We assume we missed the initial SYN.
       * If there's no ACK, we start a timeout and go to SYN_SENT.
       */
      if ( (tcphdr->th_flags & TH_SYN))
	{
	  if ( (tcphdr->th_flags & TH_ACK))
	    {
	      tcpc->state = LND_TCP_ESTABLISHED;

	      if (forward_flow)
		{
		  tcpc->conn.ip_src = iphdr->ip_dst;
		  tcpc->conn.ip_dst = iphdr->ip_src;
		  tcpc->conn.sport = tcphdr->th_dport;
		  tcpc->conn.dport = tcphdr->th_sport;
		}
	    }
	  else
	    {
	      tcpc->state = LND_TCP_SYN_SENT;
	      tcpc->syn = ntohl(tcphdr->th_seq);
	      tcpc->timeout_ts = packet->ph.ts;
	    }

	  /* End of this case */
	  break;
	}
      
      /* If we don't see the connection establishment, we
       * assume we missed it and jump directly to ESTABLISHED.
       * We also look at both port numbers, and if the current src
       * port is well-known and the dst one is ephemeral, we
       * decide to flip the flow orientation to have more consistant
       * output.
       */
      if (ntohs(tcphdr->th_sport) < 1024 && ntohs(tcphdr->th_dport) >= 1024)
	{
	  tcpc->conn.ip_src = iphdr->ip_dst;
	  tcpc->conn.ip_dst = iphdr->ip_src;
	  tcpc->conn.sport = tcphdr->th_dport;
	  tcpc->conn.dport = tcphdr->th_sport;
	}

      tcpc->state = LND_TCP_ESTABLISHED;
      break;
      
    case LND_TCP_SYN_SENT:
      /* Update our timeout */
      tcpc->timeout_ts = packet->ph.ts;

      /* If we SYN-ACK of the right byte going in opposite direction,
       * we're SYN_ACK_SENT and we still have to check for the pure ACK.
       */
      if ( (tcphdr->th_flags & TH_SYN) &&
	   (tcphdr->th_flags & TH_ACK) &&
	   (ntohl(tcphdr->th_ack) == tcpc->syn + 1) &&
	   ! forward_flow)
	{
	  tcpc->syn_ack = ntohl(tcphdr->th_seq);
	  tcpc->state = LND_TCP_SYN_ACK_SENT;
	  break;
	}

      /* And if we FIN, we attempt a connection teardown,
       * including a timeout in case this goes wrong.
       */
      if ( (tcphdr->th_flags & TH_FIN))
	tcpconn_state_goto_shutdown(tcpc, packet, iphdr, tcphdr);
      break;

    case LND_TCP_SYN_ACK_SENT: 
      /* Update our timeout */
      tcpc->timeout_ts = packet->ph.ts;

      if ( (tcphdr->th_flags == TH_ACK) &&
	   (ntohl(tcphdr->th_ack) == tcpc->syn_ack + 1))
	{
	  tcpc->state = LND_TCP_ESTABLISHED;
	  tcpc->handshake_seen = TRUE;
	}
      break;
      
    case LND_TCP_ESTABLISHED:
      /* We go from ESTABLISHED to SHUTDOWN if we see a
       * FIN and start a timeout in case the shutdown goes
       * wrong.
       */
      if ( (tcphdr->th_flags & TH_FIN))
	tcpconn_state_goto_shutdown(tcpc, packet, iphdr, tcphdr);
      break;

    case LND_TCP_SHUTDOWN:
      /* Update the timeout */
      tcpc->timeout_ts = packet->ph.ts;

      if ( (tcphdr->th_flags & TH_FIN))
	tcpconn_state_goto_shutdown(tcpc, packet, iphdr, tcphdr);

      if (forward_flow)
	{
	  if ( tcpc->fin_back &&
	       (tcphdr->th_flags & TH_ACK) &&
	       ntohl(tcphdr->th_ack) == tcpc->fin_back + 1)
	    tcpc->fin_back_acked = TRUE;
	}
      else
	{
	  if ( tcpc->fin &&
	       (tcphdr->th_flags & TH_ACK) &&
	       ntohl(tcphdr->th_ack) == tcpc->fin + 1)
	    tcpc->fin_acked = TRUE;
	}
	 
      if (tcpc->fin_acked && tcpc->fin_back_acked)
	{
	  tcpc->state = LND_TCP_TIME_WAIT;
	  tcpc->wait_ts = packet->ph.ts;
	}
      break;

    case LND_TCP_RST_WAIT:
      /* After a RST we expect nothing and wait for the
       * timeout. Any packets arriving during this period
       * are assumed to have been in flight when we saw
       * the RST, and are ignored. The only exception to this
       * are SYNs which we always assume to start a new connection
       * (this may be nonsense and removed at a later time).
       * After the timeout we end at CLOSED_RST.
       */
      if (tcphdr->th_flags & TH_SYN)
	tcpc->state = LND_TCP_CLOSED_RST;
      break;
      
    default:
      break;
    }
}


gboolean
libnd_conn_get_packet_dir(const LND_ConnID *conn, const LND_Packet *packet)
{
  struct ip* iphdr;

  if (!conn || !packet)
    return 0;

  if (! (iphdr = (struct ip*) libnd_packet_get_data(packet, libnd_ip_get(), 0)))
    return 0;
  
  if (conn->ip_src.s_addr == iphdr->ip_src.s_addr &&
      conn->ip_dst.s_addr == iphdr->ip_dst.s_addr)
    return 1;
  
  if (conn->ip_src.s_addr == iphdr->ip_dst.s_addr &&
      conn->ip_dst.s_addr == iphdr->ip_src.s_addr)
    return -1;
  
  return 0;
}


gboolean        
libnd_conn_is_dead(LND_ConnID *conn, const LND_Packet *packet)
{
  struct bpf_timeval tv_diff;

  if (!conn)
    return FALSE;

  switch (conn->proto)
    {
    case IPPROTO_TCP:
      return libnd_tcpconn_is_dead((LND_TCPConn *) conn, packet);

    default:
      if (packet)
	{
	  pcapnav_timeval_sub(&packet->ph.ts, &conn->latest_ts, &tv_diff);
	    
	  if (tv_diff.tv_sec >= LND_CONN_GENERIC_TIMEOUT)
	    /* He's dead, Jim. */
	    return TRUE;
	}
    }
  
  return FALSE;
}


LND_TCPState
libnd_tcpconn_state_get(const LND_TCPConn *tcpc)
{
  if (!tcpc)
    return LND_TCP_ERROR;

  return tcpc->state;
}


const char     *
libnd_tcpconn_state_get_string(LND_TCPState state)
{
  if (state <= LND_TCP_TIME_WAIT)
    return tcp_state_strings[state];

  return "undefined";
}


gboolean        
libnd_tcpconn_handshake_seen(const LND_TCPConn *tcpc)
{
  if (!tcpc)
    return FALSE;

  return tcpc->handshake_seen;
}


gboolean        
libnd_tcpconn_teardown_seen(const LND_TCPConn *tcpc)
{
  if (! tcpc)
    return FALSE;
  
  return ( (tcpc->state == LND_TCP_CLOSED_NORMAL) ||
	   (tcpc->state == LND_TCP_CLOSED_RST)    ||
	   (tcpc->state == LND_TCP_TIME_WAIT)     ||
	   (tcpc->state == LND_TCP_RST_WAIT));
}


gboolean
libnd_tcpconn_is_dead(LND_TCPConn *tcpc, const LND_Packet *packet)
{
  if (! tcpc)
    return FALSE;
  
  if (packet)
    libnd_tcpconn_update_time(tcpc, &packet->ph.ts);
  
  return ( (tcpc->state == LND_TCP_CLOSED_NORMAL) ||
	   (tcpc->state == LND_TCP_CLOSED_RST)    ||
	   (tcpc->state == LND_TCP_CLOSED_TIMEOUT));
}


LND_ConnID *
libnd_conn_new(const LND_Packet *packet)
{
  LND_ConnID *result = NULL;
  LND_ProtoData *proto_data;
  struct ip *iphdr;

   if (! (proto_data = libnd_packet_get_proto_data(packet, libnd_ip_get(), 0)))
     return NULL;

  iphdr = (struct ip*) proto_data->data;

  switch (iphdr->ip_p)
    {
    case IPPROTO_TCP:
      if ( (result = (LND_ConnID *) libnd_tcpconn_new(packet)))
	return result;
      break;
      
    case IPPROTO_UDP:
      if ( (result = (LND_ConnID *) libnd_udpconn_new(packet)))
	return result;
      break;
      
    default:
      break;
    }
  
  return (LND_ConnID *) libnd_ipconn_new(packet);
}


void            
libnd_conn_free(LND_ConnID *conn)
{
  if (!conn)
    return;

  switch (conn->proto)
    {
    case IPPROTO_TCP:
      libnd_tcpconn_free((LND_TCPConn *) conn);
      break;

    case IPPROTO_UDP:
      libnd_udpconn_free((LND_UDPConn *) conn);
      break;

    default:
      libnd_ipconn_free((LND_IPConn *) conn);
    }
}


void            
libnd_conn_update(LND_ConnID *conn,
		  const LND_Packet *packet)
{
  if (!conn || !packet)
    return;

  /* Update the last-touch timestamp of the connection */
  conn->latest_ts = packet->ph.ts;

  /* Now do whatever protocol-specific there is to do: */
  switch (conn->proto)
    {
    case IPPROTO_TCP:
      libnd_tcpconn_update((LND_TCPConn *) conn, packet);
      break;
      
    default:
      /* Currently nothing */
      break;
    }  
}


void
libnd_conn_get_src(const LND_ConnID *conn,
		   struct in_addr *ip_src,
		   guint16 *sport)
{
  if (!conn)
    return;

  if (ip_src)
    *ip_src = conn->ip_src;
  if (sport)
    *sport = conn->sport;
}


void
libnd_conn_get_dst(const LND_ConnID *conn,
		   struct in_addr *ip_dst,
		   guint16 *dport)
{
  if (!conn)
    return;
  
  if (ip_dst)
    *ip_dst = conn->ip_dst;
  if (dport)
    *dport = conn->dport;
}


void            
libnd_conn_to_string(const LND_ConnID *conn,
		     char *str, int strlen)
{
  char ip_dst[16];

  if (!conn || !str || strlen == 0)
    return;

  g_snprintf(ip_dst, 16, "%s", inet_ntoa(conn->ip_dst));
  g_snprintf(str, strlen, "%u[%s:%u-%s:%u]",
	     conn->proto,
	     inet_ntoa(conn->ip_src), ntohs(conn->sport),
	     ip_dst, ntohs(conn->dport));
}


void            
libnd_conn_data_set(LND_ConnID *conn,
		    const char *key,
		    void *data)
{
  if (!conn || !key || !*key)
    return;
  
  g_hash_table_insert(conn->data, (gpointer) key, (gpointer) data);
}


void *
libnd_conn_data_get(const LND_ConnID *conn,
		    const char *key)
{
  if (!conn || !key || !*key)
    {
      D(("Invalid input\n"));
      return NULL;
    }

  return g_hash_table_lookup(conn->data, key);
}


void *
libnd_conn_data_remove(LND_ConnID *conn,
		       const char *key)
{
  void *data;

  if (!conn || !key || !*key)
    return NULL;
  
  data = g_hash_table_lookup(conn->data, key);
  g_hash_table_remove(conn->data, key);

  return data;
}

const char *
name(void)
{
  return "Conntrack";
}

const char *
author(void)
{
  return "Christian Kreibich, <christian.kreibich-AT-cl.cam.ac.uk>";
}

const char *
version(void)
{
  return VERSION;
}

/* No global initialization necessary -> No init(). */


static void
conntrack_help(void)
{
  printf("TCP/UDP Connection State Tracker plugin\n"
	 "USAGE: lndtool -r conntrack <trace1> [<trace2> <trace3> ...]\n"
	 "\n"
	 "  --help, -h, -?           This message.\n");
}

gboolean
run(LND_Trace *unused, LND_PluginArgs *args)
{
  int i;
  LND_Trace *trace;
  LND_PacketIterator pit;
  LND_Packet *packet;
  LND_TCPConn *tcp_conn;
  LND_Protocol *tcp_proto, *icmp_proto;
  LND_ConnTable *ct;
  char line[MAXPATHLEN];
  gboolean new_conn;

  if (!args)
    return FALSE;

  if (args->argc == 0)
    {
      printf("Please provide one or more traces to scan.\n");
      conntrack_help();
      return FALSE;
    }

  if (! (tcp_proto = libnd_proto_registry_find(LND_PROTO_LAYER_TRANS,
					       IPPROTO_TCP)))
    {
      printf("TCP protocol plugin not found.\n");
      return FALSE;
    }

  if (! (icmp_proto = libnd_proto_registry_find(LND_PROTO_LAYER_TRANS,
						IPPROTO_ICMP)))
    {
      printf("ICMP protocol plugin not found.\n");
      return FALSE;
    }

  for (i = 0; i < args->argc; i++)
    {
      if (!strcmp(args->argv[i], "-h")     ||
	  !strcmp(args->argv[i], "--help") ||
	  !strcmp(args->argv[i], "-?"))
	{
	  conntrack_help();
	  return FALSE;
	}
      else
	{
	  printf("Scanning file %s\n", args->argv[i]);
	  
	  if (! (trace = libnd_trace_new(args->argv[i])))
	    {
	      printf("Could not open %s, skipping.\n", args->argv[i]);
	      continue;
	    }
	  
	  ct = libnd_conn_table_new(LND_CONN_TABLE_IGNORE_DEAD);
	  
	  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
	    {
	      packet = libnd_pit_get(&pit);

	      if (! libnd_tcpdump_get_packet_line(packet, line, MAXPATHLEN, FALSE))
		{
		  printf("tcpdump I/O error -- aborting.\n");
		  break;
		}

	      if (libnd_packet_has_proto(packet, tcp_proto) &&
		  ! libnd_packet_has_proto(packet, icmp_proto))
		{
		  new_conn = FALSE;
		  if (! (tcp_conn = (LND_TCPConn *) libnd_conn_table_lookup(ct, packet)))
		    {
		      new_conn = TRUE;
		      tcp_conn = libnd_tcpconn_new(packet);
		      libnd_conn_table_add(ct, (LND_ConnID *) tcp_conn);
		    }
		  
		  libnd_conn_update((LND_ConnID *) tcp_conn, packet);
		  printf("%s [%s%s]\n", line,
			 libnd_tcpconn_state_get_string(libnd_tcpconn_state_get(tcp_conn)),
			 new_conn ? "*" : "");
		}
	      else
		{
		  printf("%s\n", line);
		}
	    }
	  
	  libnd_conn_table_free(ct);
	  libnd_trace_free(trace);
	}
    }
  
  return TRUE;
  TOUCH(unused);
}

