/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd_debug.h>
#include <libnd_packet.h>
#include <libnd_macros.h>
#include <libnd_filter.h>

static gboolean
filter_dummy_init(LND_Filter *filter, LND_Trace *trace, void *filter_data)
{
  return TRUE;
  TOUCH(filter); TOUCH(trace); TOUCH(filter_data);
}

static gboolean
filter_dummy_filter(LND_Filter *filter, LND_Packet *packet, void *filter_data)
{
  return TRUE;
  TOUCH(filter); TOUCH(packet); TOUCH(filter_data);
}

static void
filter_dummy_cleanup(LND_Filter *filter, void *filter_data)
{
  return;
  TOUCH(filter); TOUCH(filter_data);
}


static void
filter_dummy_free(void *filter_data)
{
  return;
  TOUCH(filter_data);
}


LND_Filter *
libnd_filter_new(const char *name,
		 LND_FilterInitFunc filter_init,
		 LND_FilterFunc filter_func,
		 LND_FilterCleanupFunc filter_cleanup,
		 LND_FilterFreeFunc filter_data_free,
		 void *filter_data)
{
  LND_Filter *filter;

  if (!name || !*name)
    return NULL;

  if (! (filter = g_new0(LND_Filter, 1)))
    return NULL;

  filter->name                = g_strdup(name);
  filter->filter_init         = filter_dummy_init;
  filter->filter_func         = filter_dummy_filter;
  filter->filter_cleanup      = filter_dummy_cleanup;
  filter->filter_data_free    = filter_dummy_free;

  if (filter_init)
    filter->filter_init = filter_init;
  if (filter_func)
    filter->filter_func = filter_func;
  if (filter_cleanup)
    filter->filter_cleanup = filter_cleanup;
  if (filter_data_free)
    filter->filter_data_free = filter_data_free;

  filter->filter_data = filter_data;

  return filter;
}


void        
libnd_filter_set_factory(LND_Filter *filter, LND_FilterFactory *factory)
{
  if (!filter)
    return;

  filter->factory = factory;
}


void       
libnd_filter_free(LND_Filter *filter)
{
  if (!filter)
    return;

  if (filter->filter_data_free)
    filter->filter_data_free(filter->filter_data);
  
  g_free(filter->name);
  g_free(filter);
}


gboolean
libnd_filter_apply(LND_Filter *filter, LND_Packet *packet)
{
  gboolean passed;

  if (!filter || !packet || !filter->filter_func)
    {
      libnd_packet_set_filtered(packet, FALSE);
      return TRUE;
    }

  passed = filter->filter_func(filter, packet, filter->filter_data);
  libnd_packet_set_filtered(packet, !passed);

  return passed;
}


gboolean   
libnd_filter_list_apply(GList *filters, LND_Packet *packet, LND_FilterMode mode)
{
  LND_Filter *filter;
  GList     *l;
  gboolean   result;

  D_ENTER;

  if (!filters || !packet)
    {
      D(("Input error.\n"));
      libnd_packet_set_filtered(packet, FALSE);
      D_RETURN_(TRUE);
    }

  filter = (LND_Filter *) filters->data;
  D(("Applying filter '%s'\n", filter->name));
  result = filter->filter_func(filter, packet, filter->filter_data);

  for (l = g_list_next(filters); l; l = g_list_next(l))
    {
      filter = (LND_Filter *) l->data;

      D(("Applying filter '%s'\n", filter->name));

      switch (mode)
	{
	case LND_FILTER_MODE_AND:
	  result = (result && filter->filter_func(filter, packet, filter->filter_data));

	  /* We can lazy-evaluate things up in AND-mode, as we already
	   * know the result once a single test fails: */
	  if (result == FALSE)
	    {
	      libnd_packet_set_filtered(packet, TRUE);
	      D(("Setting packet %p's filter status to %i\n", packet, TRUE));
	      D_RETURN_(FALSE);
	    }
	  break;

	case LND_FILTER_MODE_OR:
	default:
	  result = (result || filter->filter_func(filter, packet, filter->filter_data));

	  /* We can lazy-evaluate in OR-mode, as we already
	   * know the result once a single test succeeds: */
	  if (result == TRUE)
	    {
	      libnd_packet_set_filtered(packet, FALSE);
	      D(("Setting packet %p's filter status to %i\n", packet, FALSE));
	      D_RETURN_(TRUE);
	    }
	}
    }

  D(("Setting packet %p's filter status to %i\n", packet, !result));
  libnd_packet_set_filtered(packet, !result);
  D_RETURN_(result);
}


void       
libnd_filter_list_init(GList *filters, LND_Trace *trace)
{
  LND_Filter *filter;
  GList     *l;

  D_ENTER;

  if (!filters || !trace)
    D_RETURN;

  for (l = filters; l; l = g_list_next(l))
    {
      filter = (LND_Filter *) l->data;
      filter->filter_init(filter, trace, filter->filter_data);
    }

  D_RETURN;
}


void       
libnd_filter_list_cleanup(GList *filters)
{
  LND_Filter *filter;
  GList     *l;

  D_ENTER;

  if (!filters)
    D_RETURN;

  for (l = filters; l; l = g_list_next(l))
    {
      filter = (LND_Filter *) l->data;
      filter->filter_cleanup(filter, filter->filter_data);
    }

  D_RETURN;
}
