/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <regex.h>

#include <libnd_tcp.h>
#include <libnd_tcb.h>
#include <libnd_tcp_prefs.h>

#define LND_TCP_TCB_KEY "tcp_tcb_key"

static LND_Protocol *tcp;

static regex_t      regex_seq;
static regex_t      regex_ack;

static LND_TCPStateMode tcp_state_mode, old_tcp_state_mode;

static gboolean
tcp_header_complete(const LND_Packet *packet, guchar *data)
{
  if (!data)
    return FALSE;

  return (data + 20 <= libnd_packet_get_end(packet));
}



/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("TCP Plugin");
}


const char *
description(void)
{
  return ("A plugin providing support for the TCP protocol.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION;
}


LND_Protocol *
init(void)
{
  tcp = libnd_proto_new("TCP", LND_PROTO_LAYER_TRANS, IPPROTO_TCP);

  tcp->init_packet         = libnd_tcp_init_packet;
  tcp->header_complete     = libnd_tcp_header_complete;
  tcp->fix_packet          = libnd_tcp_fix_packet;
  tcp->init_state          = libnd_tcp_init_state;
  tcp->update_state        = libnd_tcp_update_state;
  tcp->update_tcpdump_line = libnd_tcp_update_tcpdump_line;
  tcp->free_state          = libnd_tcp_free_state;

  /* TCP is stateful, so we need some of our functions to be called
     at the right moments: */
  tcp->is_stateful = TRUE;

  tcp_state_mode = old_tcp_state_mode = LND_TCP_STATE_UNKNOWN;

  /* Hook TCP options into preferences dialog! */
  libnd_tcp_init_prefs();

  /* We need regexes to update seq/ack numbers dynamically when
   * the user so desires. Compile them now.
   */
  if (regcomp(&regex_seq, ".* ([0-9]+):([0-9]+).*", REG_EXTENDED) < 0)
    D(("seq regex error\n"));
  
  if (regcomp(&regex_ack, ".* ack ([0-9]+) .*", REG_EXTENDED) < 0)
    D(("ack regex error\n"));
  
  return tcp;
}


/* Protocol method implementations: ------------------------------------ */

void       
libnd_tcp_init_packet(LND_Packet *packet, guchar *data, guchar *data_end)
{
  struct tcphdr *tcphdr;
  LND_Protocol *payload_proto;

  D_ENTER;

  /* Cast the data pointer into your protocol's header */
  tcphdr = (struct tcphdr *) data;

  if (!tcp_header_complete(packet, data))
    {
      libnd_raw_proto_get()->init_packet(packet, data, data_end);
      D_RETURN;
    }

  libnd_packet_add_proto_data(packet, tcp, data, data_end);  
  
  /* Check the appriopriate header field value to demultiplex
     packet initialization up to the next correct protocol: */

  if (! (payload_proto = libnd_proto_registry_find(LND_PROTO_LAYER_APP, ntohs(tcphdr->th_dport))))
    {
      D(("Using raw protocol\n"));
      payload_proto = libnd_raw_proto_get();
   }
 
  payload_proto->init_packet(packet, data + (4 * tcphdr->th_off), data_end);

  D_RETURN;
}


gboolean
libnd_tcp_header_complete(const LND_Packet *packet, guint nesting)
{
  guchar      *data;

  if (!packet)
    return FALSE;

  data = libnd_packet_get_data(packet, tcp, nesting);

  return tcp_header_complete(packet, data);
}


gboolean
libnd_tcp_fix_packet(LND_Packet *packet)
{
  struct tcphdr *tcphdr;
  guint16        correct_sum;
  
  if (!packet)
    return FALSE;
  
  if (!libnd_tcp_csum_correct(packet, &correct_sum))
    {
      tcphdr = (struct tcphdr*) libnd_packet_get_data(packet, tcp, 0);
      tcphdr->th_sum = correct_sum;
      libnd_packet_modified(packet);
      return TRUE;
    }

  return FALSE;
}


void       
libnd_tcp_init_state(LND_Trace *trace)
{
  LND_TCB *tcb;

  if (!trace)
    return;

  if (tcp_state_mode == LND_TCP_STATE_NONE ||
      tcp_state_mode == LND_TCP_STATE_ONCE)
    return;

  D(("Initializing TCP state management for %s\n", trace->filename));

  tcb = libnd_tcb_new();
  libnd_reg_set_data(trace->registry, LND_TCP_TCB_KEY, tcb);
}


void       
libnd_tcp_update_state(const LND_Packet *packet, int index)
{
  LND_TCB *tcb = NULL;
  LND_Trace *trace;

  if (tcp_state_mode == LND_TCP_STATE_NONE ||
      tcp_state_mode == LND_TCP_STATE_ONCE)
    return;

  if ( (trace = libnd_packet_get_trace(packet)))
    {
      tcb = libnd_reg_get_data(trace->registry, LND_TCP_TCB_KEY);
      D_ASSERT_PTR(tcb);
      libnd_tcb_update(tcb, packet, index);
    }
}


void       
libnd_tcp_update_tcpdump_line(const LND_Packet *packet, char *line)
{
  LND_Protocol *ip;
  LND_Trace    *trace;
  regmatch_t   regex_match[3];
  LND_TCB      *tcb;
  LND_TCBConn  *tcbc;
  gboolean     rel = FALSE, is_reverse;
  guint32      ack_no, seq_start, seq_end;
  char         line2[MAXPATHLEN];
  struct       ip *iphdr;
  struct       tcphdr *tcphdr;

  if (! (ip = libnd_tcp_get_ip()))
    return;

  if (tcp_state_mode == LND_TCP_STATE_NONE ||
      tcp_state_mode == LND_TCP_STATE_ONCE)
    return;

  trace = libnd_packet_get_trace(packet);
  tcb = libnd_reg_get_data(trace->registry, LND_TCP_TCB_KEY);
  D_ASSERT_PTR(tcb);
  tcbc = libnd_tcb_lookup(tcb, packet, &is_reverse);

  if (!tcbc)
    return;

  if (!libnd_tcp_get_headers(packet, &iphdr, &tcphdr))
    return;

  if (regexec(&regex_seq, line, 3, regex_match, 0) == 0)
    {
      line[regex_match[1].rm_so] = '\0';
      rel = libnd_tcb_conn_get_rel_seq(tcbc, iphdr, tcphdr, &seq_start, &seq_end);
      g_snprintf(line2, MAXPATHLEN, "%s%u:%u%s", line, seq_start, seq_end,
		 line + regex_match[2].rm_eo);
            
      memcpy(line, line2, MAXPATHLEN);
    }	  
  
  /* It's possible that regex_seq does not match -- pure ACK: */

  if (regexec(&regex_ack, line, 2, regex_match, 0) == 0)
    {
      line[regex_match[1].rm_so] = '\0';
      libnd_tcb_conn_get_rel_ack(tcbc, iphdr, tcphdr, rel, &ack_no);
      g_snprintf(line2, MAXPATHLEN, "%s%u%s", line, ack_no,
		 line + regex_match[1].rm_eo);

      memcpy(line, line2, MAXPATHLEN);
    }
}


void       
libnd_tcp_free_state(LND_Trace *trace)
{
  LND_TCB *tcb;

  if (!trace)
    return;

  tcb = libnd_reg_del_data(trace->registry, LND_TCP_TCB_KEY);

  if (tcb)
    libnd_tcb_free(tcb);
}


void
libnd_tcp_set_state_mode(LND_TCPStateMode mode)
{
  old_tcp_state_mode = tcp_state_mode;

  D(("New TCP state mode: %i\n", mode));
  tcp_state_mode = mode;
}


LND_TCPStateMode
libnd_tcp_get_state_mode(void)
{
  return tcp_state_mode;
}


LND_TCPStateMode
libnd_tcp_get_old_state_mode(void)
{
  return old_tcp_state_mode;
}


guint16
libnd_tcp_checksum(const LND_Packet *packet)
{
  LND_Protocol    *ip;
  struct ip*      iphdr;
  struct tcphdr  *tcphdr;
  guint           tcp_len, len;
  guint16         old, result = 0;
  guint32         sum, addl_pseudo;

  if (!packet)
    return 0;

  if (! (ip = libnd_tcp_get_ip()))
    return 0;

  if (!libnd_tcp_get_headers(packet, &iphdr, &tcphdr))
    return 0;

  old = tcphdr->th_sum;
  tcphdr->th_sum = 0;
  
  /* len is payload length, tcp_len is length of entire tcp packet: */
  tcp_len = (ntohs(iphdr->ip_len) - (iphdr->ip_hl << 2));
  len = tcp_len - tcphdr->th_off*4;
      
  if (len % 2 == 1) /* Add in pad byte. */
    sum = htons(((const u_char*) tcphdr)[tcp_len - 1] << 8);
  else
    sum = 0;
      
  sum = libnd_misc_ones_complement_checksum((u_short*) &iphdr->ip_src.s_addr, 4, sum);
  sum = libnd_misc_ones_complement_checksum((u_short*) &iphdr->ip_dst.s_addr, 4, sum);
      
  addl_pseudo = (htons(IPPROTO_TCP) << 16) | htons((u_short) tcp_len);
  
  sum = libnd_misc_ones_complement_checksum((u_short*) &addl_pseudo, 4, sum);
  sum = libnd_misc_ones_complement_checksum((u_short*) tcphdr, tcp_len, sum);
  
  tcphdr->th_sum = old;
  result = ~sum;

  return result;
}


gboolean   
libnd_tcp_csum_correct(const LND_Packet *packet, guint16 *correct_sum)
{
  struct tcphdr *tcphdr;
  guint16 old_sum, correct_sum_tmp;

  if (!packet)
    return FALSE;

  tcphdr = (struct tcphdr *) libnd_packet_get_data(packet, libnd_tcp_get(), 0);

  old_sum = tcphdr->th_sum;
  correct_sum_tmp = libnd_tcp_checksum(packet);

  if (correct_sum)
    *correct_sum = correct_sum_tmp;

  return (old_sum == correct_sum_tmp);
}


LND_Protocol *
libnd_tcp_get(void)
{
  return tcp;
}


LND_Protocol *
libnd_tcp_get_ip(void)
{
  static LND_Protocol *ip = NULL;
  
  if (!ip)
    ip = libnd_proto_registry_find(LND_PROTO_LAYER_NET, 0x0800);
  
  return ip;
}


gboolean
libnd_tcp_get_headers(const LND_Packet *packet, struct ip **iphdr, struct tcphdr **tcphdr)
{
  GList        *l;
  LND_Protocol  *ip;
  LND_ProtoData *pd, *pd_prev;

  if (!packet || !iphdr || !tcphdr)
    return FALSE;

  if (! (ip = libnd_tcp_get_ip()))
    return FALSE;

  /* Walk the proto data and find the first TCP
     (we don't support nesting TCP) with a preceding IP
  */
  
  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (LND_ProtoData *) l->data;

      if (g_list_previous(l) && pd->inst.proto == tcp)
	{
	  pd_prev = (LND_ProtoData *) g_list_previous(l)->data;

	  if (pd_prev->inst.proto == ip)
	    {
	      *iphdr  = (struct ip *) pd_prev->data;
	      *tcphdr = (struct tcphdr *) pd->data;

	      return TRUE;
	    }
	}
    }
  
  return FALSE;
}


guint
libnd_tcp_get_payload_length(struct ip *iphdr, struct tcphdr *tcphdr)
{
  guint tcp_len, payload_len;

  if (!iphdr || !tcphdr)
    return 0;

  tcp_len     = (ntohs(iphdr->ip_len) - (iphdr->ip_hl << 2));
  payload_len = tcp_len - tcphdr->th_off*4;

  return payload_len;
}
