/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd_ether.h>
#include <protocols/snap/libnd_snap.h>


static LND_Protocol *ether;


static gboolean
ether_header_complete(const LND_Packet *packet, guchar *data)
{
  if (!data)
    return FALSE;

  return (data + ETHER_HDR_LEN <= libnd_packet_get_end(packet));
}


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("Ethernet Plugin");
}


const char *
description(void)
{
  return ("A plugin providing Ethernet support.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION;
}


LND_Protocol *
init(void)
{
  ether = libnd_proto_new("Ethernet", LND_PROTO_LAYER_LINK, DLT_EN10MB);
			     		
  ether->init_packet     = libnd_ether_init_packet;
  ether->header_complete = libnd_ether_header_complete;

  /* That's all -- we don't need checksums or state maintenance
   * for simple Ethernet. The other methods got initialized
   * to dummy null operations in the constructor call above.
   */
	     
  return ether;
}


/* Protocol method implementations: ------------------------------------ */

void       
libnd_ether_init_packet(LND_Packet *packet, guchar *data, guchar *data_end)
{
  LND_Protocol *payload_proto;
  struct ether_header *eh;

  D_ENTER;

  if (!ether_header_complete(packet, data))
    {
      libnd_raw_proto_get()->init_packet(packet, data, data_end);
      D_RETURN;
    }

  libnd_packet_add_proto_data(packet, ether, data, data_end);  
  
  eh = (struct ether_header *) data;


  /* Is this an Ethernet-II frame or a Ethernet 802.3 SNAP frame?*/
  /* Check ether_type > 1500 --> Ethernet II */
  if (htons(eh->ether_type) <= ETHERMTU)
    {
      /* It's Ethernet 802.3, starting with some form of LLC,
	 pass to SNAP implementation: */

      payload_proto = libnd_proto_registry_find(LND_PROTO_LAYER_LINK, LND_PROTO_SNAP);
    }
  else 
    {
      /* It's Ethernet II: */
      payload_proto = libnd_proto_registry_find(LND_PROTO_LAYER_LINK | LND_PROTO_LAYER_NET,
						htons(eh->ether_type));
    }  

  if (! payload_proto)
    payload_proto = libnd_raw_proto_get();

  payload_proto->init_packet(packet, data + ETHER_HDR_LEN, data_end);

  D_RETURN;
}


gboolean
libnd_ether_header_complete(const LND_Packet *packet, guint nesting)
{
  guchar      *data;

  if (!packet)
    return FALSE;

  data = libnd_packet_get_data(packet, ether, nesting);

  return ether_header_complete(packet, data);
  TOUCH(nesting);
}


LND_Protocol *
libnd_ether_get(void)
{
  return ether;
}

