/*

Copyright (C) 2000 - 2007 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#include <libnd.h>

void
usage(void)
{
  char *dbug = "";

#ifdef LIBND_DEBUG
  dbug = "  --debug, -d                  Print debugging output.\n"
	 "  --ludicrous-debug, -dd       More debugging output.\n";
#endif

  printf("lndtool -- libnetdude configuration and execution tool.\n"
	 "USAGE: lndtool [OPTIONS]\n"
	 "\n"
	 "  --help, -help, -h, -?        This message.\n"
	 "%s"
	 "  --prefix                     Installation prefix.\n"
	 "  --version, -v                Prints out version info.\n"
	 "  --cflags                     Preprocessor flags needed for compilation.\n"
	 "  --libs                       Linker flags needed when linking..\n"
	 "  --plugin-dir                 Plugin installation directory.\n"
	 "  --proto-dir                  Protocol installation directory.\n"
	 "  --include-dir                Header files directory.\n"
	 "  --plugins, -p                Lists all plugins that register successfully.\n"
	 "  --run, -r PLUGINNAME PARAMS  Run plugin PLUGINNAME with PARAMS.\n"
	 "\n", dbug);
}

void
dump_lnd_proto(LND_Protocol *proto, void *user_data)
{
  if (proto->id != 1)
    printf("%-32s%s\n", proto->name,
	   (proto->plugin ? proto->plugin->version() : "Unknown"));

  return;
  TOUCH(user_data);
}

void
dump_lnd_plugin(LND_Plugin *plugin, void *user_data)
{
  printf("%-32s%s\n",
	 libnd_plugin_get_name(plugin),
	 libnd_plugin_get_version(plugin));
  
  return;
  TOUCH(user_data);
}

void
dump_plugins(void)
{
  printf("libnetdude protocol plugins:\n");
  printf("--------------------------------------------------\n");
  libnd_proto_registry_foreach_proto(dump_lnd_proto, NULL);

  printf("\nlibnetdude feature plugins:\n");
  printf("--------------------------------------------------\n");
  libnd_plugin_foreach(dump_lnd_plugin, NULL);
}

int
run_plugin(char *plugin_name, int argc, char **argv)
{
  LND_Plugin *plugin;
  LND_PluginArgs args;

  args.argc = argc;
  args.argv = argv;

  if (! (plugin = libnd_plugin_find(plugin_name)))
    {
      printf("Plugin '%s' not found.\n", plugin_name);
      exit(-1);
    }

  return libnd_plugin_run(plugin, NULL, &args);
}

int
main(int argc, char **argv)
{
  int i, do_pcapnav_debug = 0;

  for (i = 1; i < argc; i++)
    {
      if (! strcmp(argv[i], "--debug") ||
	  ! strcmp(argv[i], "-d"))
	{
	  libnd_runtime_options.debug = TRUE;
	}
      else if (! strcmp(argv[i], "--ludicrous-debug") ||
	  ! strcmp(argv[i], "-dd"))
	{
	  do_pcapnav_debug = 1;
	  libnd_runtime_options.debug = TRUE;
	}
    }

  libnd_init();

  if (do_pcapnav_debug)
    pcapnav_runtime_options.debug = 1;

  if (argc == 1)
    {
      usage();
      exit(0);
    }

  for (i = 1; i < argc; i++)
    {
      if (! strcmp(argv[i], "--debug") ||
	  ! strcmp(argv[i], "-d")      ||
          ! strcmp(argv[i], "--ludicrous-debug") ||
	  ! strcmp(argv[i], "-dd"))
	{
	  /* Do nothing -- we've handled those above. */
	}
      else if (! strcmp(argv[i], "--prefix"))
	{
	  printf("/usr/local\n");
	}
      else if (! strcmp(argv[i], "-v") ||
	       ! strcmp(argv[i], "--version"))
	{
	  printf("0.10a\n");
	}
      else if (! strcmp(argv[i], "--cflags"))
	{
	  char *includes = "";

	  if (strcmp("/usr/local/include", "/usr/include"))
	    includes = "-I/usr/local/include ";
	  printf("%s-I/usr/local/include/libnetdude/0.10a -I/usr/include/glib-1.2 -I/usr/lib/glib/include -I/n/banquet/db/kreibich/inst/include -I/n/banquet/db/kreibich/inst/include\n",
		 includes);
	}
      else if (! strcmp(argv[i], "--libs"))
	{
	  printf("-L/usr/local/lib -L/usr/lib -lglib -L/n/banquet/db/kreibich/inst/lib -lpcapnav -lpcap -lnetdude\n");
	}
      else if (! strcmp(argv[i], "--include-dir"))
	{
	  printf("/usr/local/include/libnetdude/0.10a\n");
	}
      else if (! strcmp(argv[i], "--plugin-dir"))
	{
	  printf("/usr/local/share/libnetdude/0.10a/plugins\n");
	}
      else if (! strcmp(argv[i], "--proto-dir"))
	{
	  printf("/usr/local/share/libnetdude/0.10a/protocols\n");
	}
      else if (! strcmp(argv[i], "--plugins") ||
	       ! strcmp(argv[i], "-p"))
	{
	  dump_plugins();
	  exit(0);
	}
      else if (! strcmp(argv[i], "--run") ||
	       ! strcmp(argv[i], "-r"))
	{
	  if (i + 1 == argc)
	    {
	      dump_plugins();
	      exit(0);
	    }

	  return run_plugin(argv[i+1], argc - (i+2), &argv[i+2]);
	}
      else
	{
	  usage();
	  exit(0);
	}
    }

  return 0;
}
