/*

Copyright (C) 2000 - 2006 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd_ip.h>

#include <net/if.h>
#include <netinet/if_ether.h>
#include <time.h>


static LND_Protocol *ip;


static gboolean
ip_header_complete(const LND_Packet *packet, struct ip *iphdr)
{
  if (!iphdr)
    return FALSE;

  return (((guchar*) iphdr) + 4 * iphdr->ip_hl <= libnd_packet_get_end(packet));
}


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("IPv4 Plugin");
}


const char *
description(void)
{
  return ("A plugin providing IPv4 support, including:\n"
	  " - ECN codepoints according to RFC 3168\n"
	  " - IP options according to RFCs 791 and 2113\n"
	  " - Fragmentation and Reassembly");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION;
}


LND_Protocol *
init(void)
{
  ip = libnd_proto_new("IPv4", LND_PROTO_LAYER_NET, ETHERTYPE_IP);
	
  ip->init_packet     = libnd_ip_init_packet;
  ip->header_complete = libnd_ip_header_complete;
  ip->fix_packet      = libnd_ip_fix_packet;
  ip->can_fix_proceed = libnd_ip_can_fix_proceed;
  ip->adjust_len      = libnd_ip_adjust_len;

  /* IP isn't stateful, so we don't need any of the state-related
     functions. */
	     
  return ip;
}


/* Protocol method implementations: ------------------------------------ */

guchar *
libnd_ip_init_packet(LND_Packet *packet, u_char *data, u_char *data_end)
{
  LND_Protocol *payload_proto = NULL;
  struct ip* iphdr = (struct ip*) data;
  guchar *used_data_end = NULL;

  D_ENTER;

  if (!ip_header_complete(packet, iphdr))
    {
      libnd_raw_proto_get()->init_packet(packet, data, data_end);
      D_RETURN_(data_end);
    }

  libnd_packet_add_proto_data(packet, ip, data, data + ntohs(iphdr->ip_len));
  used_data_end = MIN(data_end, data + ntohs(iphdr->ip_len));

  /* If this is a first fragment or normal packet, check contents
     and delegate initialization. */

  if ((ntohs(iphdr->ip_off) & IP_OFFMASK) == 0)
    {
      if (! (payload_proto =
	     libnd_proto_registry_find(LND_PROTO_LAYER_NET | LND_PROTO_LAYER_TRANS,
				       iphdr->ip_p)))
	payload_proto = libnd_raw_proto_get();
      
      /* Adjust the data pointers so that we only pass the really
	 present data end. But IP may be nested, so we cannot rely
	 on the ip_len field to be relevant for this packet, so
	 sanity-check against end of captured data!
      */
      
      payload_proto->init_packet(packet,
				 data + 4 * iphdr->ip_hl, used_data_end);
    }
  else
    {
      /* It's fragment data -- display in raw data displayer */
      
      libnd_raw_proto_get()->init_packet(packet,
					 data + 4 * iphdr->ip_hl, used_data_end);
    }
  
  D_RETURN_(used_data_end);
}


gboolean
libnd_ip_header_complete(const LND_Packet *packet, guint nesting)
{
  struct ip *iphdr;

  if (!packet)
    return FALSE;

  iphdr = (struct ip*) libnd_packet_get_data(packet, ip, nesting);

  return ip_header_complete(packet, iphdr);
}


guint16
libnd_ip_checksum(const struct ip* iphdr)
{
  guint16    old_sum, sum;
  struct ip *iphdr_work = (struct ip*) iphdr;

  old_sum = iphdr->ip_sum;
  iphdr_work->ip_sum = 0;
  sum = libnd_misc_in_cksum((u_short *)iphdr_work, iphdr->ip_hl << 2, 0);
  iphdr_work->ip_sum = old_sum;

  return sum;
}


static gboolean
ip_fix_cb(LND_Packet    *packet,
	  LND_ProtoData *pd,
	  void         *user_data)
{
  gboolean  *modified = (gboolean *) user_data;
  guint16    correct_sum;
  struct ip *iphdr;

  if (pd->inst.proto != ip)
    return TRUE;

  iphdr = (struct ip*) pd->data;              

  /* Recalculate IP checksum, and if it changed,
     tell Netdude that we modified the packet. */
  
  if (!libnd_ip_csum_correct(iphdr, &correct_sum))
    {
      iphdr->ip_sum = correct_sum;
      *modified = TRUE;
    }

  return TRUE;
  TOUCH(packet);
}


gboolean
libnd_ip_fix_packet(LND_Packet *packet)
{
  gboolean modified = FALSE;

  if (!packet)
    return FALSE;

  libnd_packet_foreach_proto_backward(packet, ip_fix_cb, &modified);

  if (modified)
    libnd_packet_modified(packet);

  return modified;
}

gboolean
libnd_ip_can_fix_proceed(LND_Packet *packet,
			 LND_ProtoData *pd)
{
  struct ip *iphdr;
  
  if (!packet || !pd)
    return FALSE;

  if (! (iphdr = (struct ip*) pd->data))
    return FALSE;

  /* It makes sense to go past this header only if
   * it is not a fragment.
   */
  return ((ntohs(iphdr->ip_off) & IP_MF) == 0)
    && ((ntohs(iphdr->ip_off) & IP_OFFMASK) == 0);
}

typedef struct libnd_ip_adjust_data {
  gboolean modified;
  int delta;
} LND_IPAdjustData;

static gboolean
ip_adjust_cb(LND_Packet    *packet,
	     LND_ProtoData *pd,
	     void          *user_data)
{
  LND_IPAdjustData *data = (LND_IPAdjustData *) user_data;
  struct ip *iphdr;
  int result;

  if (pd->inst.proto != ip)
    return TRUE;
  
  iphdr = (struct ip*) pd->data;
  result = ntohs(iphdr->ip_len) + data->delta;
  
  /* We can't adjust for a negative size or one larger than 65K */
  if (result < 0 || result > 0xFFFF)
    return FALSE;
  
  iphdr->ip_len = htons((guint16) result);  
  data->modified = TRUE;
  
  return TRUE;
  TOUCH(packet);
}

gboolean
libnd_ip_adjust_len(LND_Packet *packet, int delta)
{
  LND_IPAdjustData data;
  data.modified = FALSE;
  data.delta = delta;
  
  libnd_packet_foreach_proto(packet, ip_adjust_cb, &data);
  
  if (data.modified)
    libnd_packet_modified(packet);
  
  return data.modified;
}

  
inline LND_Protocol *
libnd_ip_get(void)
{
  return ip;
}


gboolean   
libnd_ip_csum_correct(const struct ip *iphdr, guint16 *correct_sum)
{
  guint16 old_sum, correct_sum_tmp;

  if (!iphdr)
    return FALSE;

  old_sum = iphdr->ip_sum;
  ((struct ip *) iphdr)->ip_sum = 0;

  correct_sum_tmp = libnd_misc_in_cksum((u_short *)iphdr, iphdr->ip_hl << 2, 0);

  ((struct ip *) iphdr)->ip_sum = old_sum;

  if (correct_sum)
    *correct_sum = correct_sum_tmp;

  return (old_sum == correct_sum_tmp);
}
