/*

Copyright (C) 2000 - 2006 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <errno.h>
#include <libnd.h>
#include "libnd_cksumfix.h"

/* The Checksum Fix plugin. It calls the fix_packet() callback for all
 * protocols found in the packets contained in the provided trace. When
 * called via the run() hook, the corrected trace can be output to a
 * separate file. run() returns the number of corrected packets on success,
 * or negative error codes if problems occurred.
 */

const char *
name(void)
{
  return "Checksum-Fix";
}

const char *
author(void)
{
  return "Christian Kreibich, <christian@whoop.org>";
}

const char *
version(void)
{
  return VERSION;
}

static void
cksumfix_help(void)
{
  printf("This plugin corrects checksums of various protocol headers, as\n"
	 "implemented by the protocol plugins installed.\n"
	 "\n"
	 "USAGE: lndtool -r cksumfix \n"
	 "\n"
	 "  --input,-i <file>        Input trace file.\n"
	 "  --output,-o <file>       Output trace file.\n");
}

int
run(LND_Trace *trace, LND_PluginArgs *args)
{
  int i;
  LND_PacketIterator pit;
  LND_Dumper *dumper = NULL;
  const char *input = NULL, *output = NULL;

  /* Check arguments, if provided */
  if (args)
    {
      for (i = 0; i < args->argc; i++)
	{
	  if (!strcmp(args->argv[i], "-h")     ||
	      !strcmp(args->argv[i], "--help") ||
	      !strcmp(args->argv[i], "-?"))
	    {
	      cksumfix_help();
	      return 0;
	    }
	  else if (!strcmp(args->argv[i], "-i") ||
		   !strcmp(args->argv[i], "--input"))
	    {
	      if (++i == args->argc)
		{
		  printf("You need to pass a trace file to the --input|-i option.\n");
		  return 1;
		}

	      input = args->argv[i];
	    }
	  else if (!strcmp(args->argv[i], "-o") ||
		   !strcmp(args->argv[i], "--output"))
	    {
	      if (++i == args->argc)
		{
		  printf("You need to pass a trace file to the --output|-o option.\n");
		  return 1;
		}
	  
	      output = args->argv[i];
	    }
	}
    }

  /* Open trace only if not provided */
  if (! trace) 
    {
      if (! input)
	{
	  fprintf(stderr, "Please provide an input trace file, using --input|-i.\n");
	  return 1;
	}

      if (! (trace = libnd_trace_new(input)))
	{
	  fprintf(stderr, "Could not open trace file '%s'\n", input);
	  return errno;
	}
    }

  /* Create a dumper for the output, if requested. */
  if (output && ! (dumper = libnd_dumper_new(libnd_trace_get_pcap_handle(trace),
					     output, NULL)))
    {
      if (input)
	libnd_trace_free(trace);
      
      fprintf(stderr, "Could not create output trace '%s'\n", output);
      return errno;
    }
  
  D(("Running Checksum Fix Plugin\n"));
  
  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      LND_Packet *packet = libnd_pit_get(&pit);
      
      libnd_packet_fix(packet);
      
      if (dumper)
	libnd_dumper_write(dumper, packet);
    }
  
  if (dumper)
    libnd_dumper_free(dumper);
  
  if (input)
    libnd_trace_free(trace);  
  
  return 0;
}


void
libnd_cksumfix_run(LND_Trace *trace, guint *tested, guint *modified)
{
  LND_PacketIterator pit;
  guint last_tested = 0, last_modified = 0;

  if (!trace)
    return;

  /* Use a packet iterator packets in the given trace -- it will
   * do The Right Thing for whatever selection mode is currently
   * active.
   */
  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {      
      last_tested++;
      
      if (libnd_packet_fix(libnd_pit_get(&pit)))
	last_modified++;
    }
  
  if (tested)
    *tested = last_tested;
  if (modified)
    *modified = last_modified;
  
  if (last_modified > 0)
    libnd_trace_set_dirty(trace, TRUE);
}
